IMPLEMENTATION MODULE SMTPCommands;

        (********************************************************)
        (*                                                      *)
        (*       Command interpreter for SMTP server            *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            27 April 1998                   *)
        (*  Last edited:        2 September 1999                *)
        (*  Status:             Basically OK                    *)
        (*                                                      *)
        (*  Improvements needed:                                *)
        (*    HELO: should check validity of sending host name  *)
        (*    Do I need to allow for more complicated           *)
        (*      forms of the mail path?                         *)
        (*                                                      *)
        (********************************************************)

(********************************************************************************)
(*                        COMPLIANCE WITH THE STANDARD                          *)
(********************************************************************************)
(*                                                                              *)
(* I'm working from the SMTP standard RFC821                                    *)
(*                                                                              *)
(* The required commmands are all implemented and tested:                       *)
(*                                                                              *)
(*      DATA, HELO, MAIL, NOOP, QUIT, RCPT, RSET                                *)
(*                                                                              *)
(* The only optional commands that are implemented are:                         *)
(*                                                                              *)
(*      EHLO, EXPN, VRFY                                                        *)
(*                                                                              *)
(********************************************************************************)

IMPORT Strings;

FROM Storage IMPORT
    (* proc *)  ALLOCATE;

FROM Sockets IMPORT
    (* type *)  Socket,
    (* proc *)  send;

FROM TaskControl IMPORT
    (* type *)  Lock,
    (* proc *)  CreateLock, Obtain, Release;

FROM Semaphores IMPORT
    (* type *)  Semaphore;

FROM InetUtilities IMPORT
    (* proc *)  ConvertCard, AddEOL, CurrentTimeToString;

FROM Names IMPORT
    (* type *)  UserName, HostName;

FROM Hosts IMPORT
    (* proc *)  HostIsLocal, OurHostName, AcceptableRelayDestination;

FROM SMTPData IMPORT
    (* type *)  ItemDescriptor,
    (* proc *)  CreateItemDescriptor, DiscardItemDescriptor,
                ResetItemDescriptor, SetClaimedSendingHost,
                AddLocalRecipient, AddRelayRecipient, AcceptMessage,
                RunFilter, DistributeMessage,
                IsValidUsername, SenderNotSpecified, NoRecipients;

FROM RelayMail IMPORT
    (* proc *)  UserAndDomain;

FROM TransLog IMPORT
    (* type *)  TransactionLogID,
    (* proc *)  LogTransaction;

(********************************************************************************)

CONST
    Nul = CHR(0);
    BadCommandLimit = 3;

TYPE
    FourChar = ARRAY [0..3] OF CHAR;
    ClientState = (Idle, LoggedIn, MustExit);

    (* The session record.  The fields are:                             *)
    (*     socket      The command socket                               *)
    (*     state       To track whether the user is currently logged in *)
    (*     desc        Information about the next item to be delivered  *)
    (*     BadCommandCount  number of unknown commands we've received   *)
    (*                      since last good command                     *)
    (*     AcceptRelayMail  TRUE iff we're willing to accept mail to    *)
    (*                      be relayed                                  *)
    (*     watchdog    Semaphore that times out if we don't kick it     *)
    (*                  now and then                                    *)

    Session = POINTER TO
                  RECORD
                      ID: TransactionLogID;
                      socket: Socket;
                      state: ClientState;
                      desc: ItemDescriptor;
                      BadCommandCount: CARDINAL;
                      AcceptRelayMail: BOOLEAN;
                      watchdog: Semaphore;
                  END (*RECORD*);

(********************************************************************************)
(*                         STARTING A NEW SESSION                               *)
(********************************************************************************)

PROCEDURE OpenSession (CommandSocket: Socket;  KeepAlive: Semaphore;
                       LogID: TransactionLogID;  MayRelay: BOOLEAN): Session;

    (* Creates a new session state record. *)

    VAR result: Session;

    BEGIN
        NEW (result);
        WITH result^ DO
            ID := LogID;
            socket := CommandSocket;
            state := Idle;
            AcceptRelayMail := MayRelay;
            watchdog := KeepAlive;
            desc := CreateItemDescriptor (socket, LogID, MayRelay);
            BadCommandCount := 0;
        END (*WITH*);
        RETURN result;
    END OpenSession;

(********************************************************************************)

PROCEDURE CloseSession (S: Session);

    (* Destroys the session state record. *)

    BEGIN
        DiscardItemDescriptor (S^.desc);
        DISPOSE (S);
    END CloseSession;

(********************************************************************************)
(*                       SENDING REPLY BACK TO CLIENT                           *)
(********************************************************************************)

PROCEDURE Reply2 (session: Session;  message1, message2: ARRAY OF CHAR);

    (* Sends all of message1, followed by message2, followed by end-of-line.    *)
    (* If the operation fails, session^.state is set to MustExit.               *)

    VAR buffer: ARRAY [0..511] OF CHAR;  length: CARDINAL;

    BEGIN
        Strings.Assign (message1, buffer);
        Strings.Append (message2, buffer);
        length := AddEOL (buffer);
        IF send (session^.socket, buffer, length, 0) = MAX(CARDINAL) THEN
            session^.state := MustExit;
        END (*IF*);
    END Reply2;

(********************************************************************************)

PROCEDURE Reply (session: Session;  message: ARRAY OF CHAR);

    (* Like Reply2, except that there is no message2. *)

    VAR buffer: ARRAY [0..511] OF CHAR;  length: CARDINAL;

    BEGIN
        Strings.Assign (message, buffer);
        length := AddEOL (buffer);
        IF send (session^.socket, buffer, length, 0) = MAX(CARDINAL) THEN
            session^.state := MustExit;
        END (*IF*);
    END Reply;

(********************************************************************************)
(*                     HANDLERS FOR SOME ERROR CONDITIONS                       *)
(********************************************************************************)

PROCEDURE NoSuchCommand (session: Session;  VAR (*IN*) Command: ARRAY OF CHAR);

    (* Command is not a recognised command. *)

    BEGIN
        Reply2 (session, "500 Unknown command ", Command);
    END NoSuchCommand;

(********************************************************************************)

PROCEDURE GarbledCommandSequence (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    (* Too many unknown commands have been received. *)

    BEGIN
        dummy[0] := dummy[0];                   (* to avoid a compiler warning *)
        Reply (session, "421 Too many bad commands, closing connection");
        session^.state := MustExit;
    END GarbledCommandSequence;

(********************************************************************************)

PROCEDURE NotLoggedIn (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    (* Command is illegal because user is not yet logged in. *)

    BEGIN
        dummy[0] := dummy[0];                   (* to avoid a compiler warning *)
        Reply (session, "503 Not logged in");
    END NotLoggedIn;

(********************************************************************************)
(*                     HANDLERS FOR THE INDIVIDUAL COMMANDS                     *)
(********************************************************************************)

PROCEDURE DATA (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    BEGIN
        dummy[0] := dummy[0];                   (* to avoid a compiler warning *)
        IF SenderNotSpecified (session^.desc) THEN
            Reply (session, "503 Sender has not been specified");
        ELSIF NoRecipients (session^.desc) THEN
            Reply (session, "503 No valid recipients");
        ELSE
            Reply (session, "354 Socket to me");
            WITH session^ DO
                IF AcceptMessage (socket, desc, watchdog) THEN

                    (* Post-reception filtering. *)

                    CASE RunFilter(desc) OF
                      | 0,1: DistributeMessage (desc);
                             Reply (session, "250 OK");
                      | 2:   ResetItemDescriptor (desc, "");
                             Reply (session, "250 OK");
                      | 3:   ResetItemDescriptor (desc, "");
                             Reply (session, "554 Mail rejected by server");
                    ELSE
                             Reply (session, "554 Server error, please report to postmaster");
                    END (*CASE*);

                ELSE
                    Reply (session, "554 Transaction failed");
                END (*IF*);

            END (*WITH*);
        END (*IF*);

    END DATA;

(********************************************************************************)

PROCEDURE EHLO (session: Session;  VAR (*IN*) name: ARRAY OF CHAR);

    VAR host: HostName;
        response: ARRAY [0..255] OF CHAR;

    BEGIN
        ResetItemDescriptor (session^.desc, "");
        SetClaimedSendingHost (session^.desc, name);
        session^.state := LoggedIn;
        OurHostName (host);
        response := "250 ";
        Strings.Append (host, response);
        Strings.Append (" teletubbies say ehlo", response);
        Reply (session, response);
    END EHLO;

(********************************************************************************)

PROCEDURE EXPN (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    BEGIN
        dummy[0] := dummy[0];                   (* to avoid a compiler warning *)
        Reply (session, "550 That information is confidential");
    END EXPN;

(********************************************************************************)

PROCEDURE HELO (session: Session;  VAR (*IN*) name: ARRAY OF CHAR);

    BEGIN
        ResetItemDescriptor (session^.desc, "");
        SetClaimedSendingHost (session^.desc, name);
        session^.state := LoggedIn;
        Reply (session, "250 OK");
    END HELO;

(********************************************************************************)

PROCEDURE MAIL (session: Session;  VAR (*IN*) from: ARRAY OF CHAR);

    VAR j: CARDINAL;

    BEGIN
        (* Delete the "FROM:" part of the command.  (There's no harm in         *)
        (* assuming it's there without checking.  If it's not there then the    *)
        (* command is faulty anyway, and it's unlikely that this fault would be *)
        (* followed by a valid sender address.)                                 *)

        j := 5;
        WHILE from[j] = ' ' DO
            INC (j);
        END (*WHILE*);
        Strings.Delete (from, 0, j);
        ResetItemDescriptor (session^.desc, from);
        Reply (session, "250 Sender accepted");
    END MAIL;

(********************************************************************************)

PROCEDURE NOOP (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    BEGIN
        dummy[0] := dummy[0];                   (* to avoid a compiler warning *)
        Reply (session, "250 OK");
    END NOOP;

(********************************************************************************)

PROCEDURE QUIT (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    BEGIN
        dummy[0] := dummy[0];                   (* to avoid a compiler warning *)
        Reply (session, "221 Closing connection");
        session^.state := MustExit;
    END QUIT;

(********************************************************************************)

PROCEDURE RCPT (session: Session;  VAR (*IN*) target: ARRAY OF CHAR);

    VAR user: UserName;
        host: HostName;
        response: ARRAY [0..255] OF CHAR;

    BEGIN
        (* Delete the "TO:" part of the command.  (There's no harm in assuming  *)
        (* it's there without checking.  If it's not there then the command is  *)
        (* faulty anyway, and it's unlikely that this fault would be followed   *)
        (* by a valid recipient address.)                                       *)

        Strings.Delete (target, 0, 3);
        WHILE target[0] = ' ' DO
            Strings.Delete (target, 0, 1);
        END (*WHILE*);
        UserAndDomain (target, user, host);

        (* Has a sender been specified? *)

        IF SenderNotSpecified (session^.desc) THEN

            Reply (session, "503 Bad sequence of commands");

        (* Is this a local user? *)

        ELSIF HostIsLocal (host) THEN

            IF AddLocalRecipient (session^.desc, user) THEN
                Reply (session, "250 mailbox OK");
            ELSE
                response := "553 unknown user ";
                Strings.Append (user, response);
                Reply (session, response);
            END (*IF*);

        (* If not, do we agree to relay it? *)

        ELSIF session^.AcceptRelayMail OR AcceptableRelayDestination(host) THEN

            AddRelayRecipient (session^.desc, target);
            Reply2 (session, "250 User not local; will forward to ",
                             target);

        ELSE
            Reply2 (session, "551 User not local; please try ",
                             target);
        END (*IF*);

    END RCPT;

(********************************************************************************)

PROCEDURE RSET (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    BEGIN
        dummy[0] := dummy[0];                   (* to avoid a compiler warning *)
        ResetItemDescriptor (session^.desc, "");
        Reply (session, "250 OK");
    END RSET;

(********************************************************************************)

PROCEDURE VRFY (session: Session;  VAR (*IN*) user: ARRAY OF CHAR);

    VAR response: ARRAY [0..255] OF CHAR;

    BEGIN
        (* Is this a local user? *)

        IF IsValidUsername (user) THEN
            response := "250 ";
            Strings.Append (user, response);
        ELSE
            response := "553 unknown user";
        END (*IF*);
        Reply (session, response);

    END VRFY;

(********************************************************************************)
(*                      THE MAIN COMMAND DISPATCHER                             *)
(********************************************************************************)

TYPE
    KeywordNumber = [0..9];
    HandlerProc = PROCEDURE (Session, VAR (*IN*) ARRAY OF CHAR);
    HandlerArray = ARRAY KeywordNumber OF HandlerProc;
    KeywordArray = ARRAY KeywordNumber OF FourChar;

CONST
    KeywordList = KeywordArray {'DATA', 'EHLO', 'EXPN', 'HELO', 'MAIL', 'NOOP',
                                'QUIT', 'RCPT', 'RSET', 'VRFY'};

CONST
    HandlerList = HandlerArray {DATA, EHLO, EXPN, HELO, MAIL, NOOP, QUIT, RCPT,
                                RSET, VRFY};

(********************************************************************************)

PROCEDURE HandleCommand (S: Session;  Command: ARRAY OF CHAR;
                                                     VAR (*OUT*) Quit: BOOLEAN);

    (* Executes one user command.  Returns with Quit=TRUE if the command is one *)
    (* that closes the session, or if the connection is lost.                   *)

    VAR k: [0..3];

    (****************************************************************************)

    PROCEDURE Compare4 (n: KeywordNumber): INTEGER;

        (* Compares the first four characters of Command with KeywordList[n].   *)
        (* Returns >0 if Command[0..3] > KeywordList[n], and so on.             *)

        VAR ch1, ch2: CHAR;

        BEGIN
            k := 0;
            LOOP
                ch1 := Command[k];  ch2 := KeywordList[n][k];
                IF ch1 > ch2 THEN RETURN +1
                ELSIF ch1 < ch2 THEN RETURN -1
                ELSIF k = 3 THEN RETURN 0
                END (*IF*);
                INC (k);
            END (*LOOP*);
        END Compare4;

    (****************************************************************************)

    VAR m: CARDINAL;  Match, QuitReceived: BOOLEAN;
        first, middle, last: CARDINAL;  test: INTEGER;
        Handler: HandlerProc;

    BEGIN
        (* Watch out for lower case. *)

        FOR k := 0 TO 3 DO
            Command[k] := CAP(Command[k]);
        END (*FOR*);

        (* Go through the keyword list to find a match with the command.  *)
        (* In this version I'm using a binary search.                     *)

        first := 0;  last := MAX(KeywordNumber);  Match := FALSE;
        LOOP
            middle := (first + last) DIV 2;
            test := Compare4 (middle);
            IF test < 0 THEN
                IF middle = 0 THEN
                    EXIT (*LOOP*);
                ELSE
                    last := middle - 1;
                END (*IF*);
            ELSIF test = 0 THEN
                Match := TRUE;  EXIT (*LOOP*);
            ELSIF test > 0 THEN
                first := middle + 1;
            END (*IF*);
            IF first > last THEN EXIT (*LOOP*) END (*IF*);
        END (*LOOP*);

        IF Match THEN
            Handler := HandlerList[middle];  S^.BadCommandCount := 0;
        ELSIF S^.BadCommandCount >= BadCommandLimit THEN
            Handler := GarbledCommandSequence;
        ELSE
            Handler := NoSuchCommand;  INC(S^.BadCommandCount);
        END (*IF*);

        (* Echo command to transaction log. *)

        LogTransaction (S^.ID, Command);

        (* If the user is not yet logged in, only EHLO and HELO and QUIT are legal. *)

        QuitReceived := Handler = QUIT;
        IF NOT QuitReceived AND (S^.state <> LoggedIn) AND (Handler <> EHLO)
                             AND (Handler <> HELO) THEN
            Handler := NotLoggedIn;
        END (*IF*);

        (* Strip out the command characters, leaving only the parameters. *)

        IF Handler = NoSuchCommand THEN m := 0 ELSE m := 4 END(*IF*);
        WHILE (m < HIGH(Command)) AND (Command[m] = " ") DO INC(m) END (*WHILE*);
        Strings.Delete (Command, 0, m);

        (* Call the handler. *);

        Handler (S, Command);
        IF (S^.state = MustExit) AND NOT QuitReceived THEN
            LogTransaction (S^.ID, "Connection lost");
        END (*IF*);
        Quit := S^.state = MustExit;

    END HandleCommand;

(********************************************************************************)

END SMTPCommands.

