IMPLEMENTATION MODULE NameLookup;

        (********************************************************)
        (*                                                      *)
        (*               Host name lookup                       *)
        (*                                                      *)
        (*   We put this service into a separate module so that *)
        (*   we can arrange to contact the nameserver in a      *)
        (*   separate thread.  This means that there will be    *)
        (*   no delays if the nameserver is down or slow.       *)
        (*   If we don't have an answer by the time it's        *)
        (*   needed, we can get by with a null name.            *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            23 March 1998                   *)
        (*  Last edited:        3 November 1999                 *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (********************************************************)

IMPORT Strings;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

FROM Sockets IMPORT
    (* type *)  AddressFamily;

FROM NetDB IMPORT
    (* type *)  HostEntPtr,
    (* proc *)  gethostbyaddr, tcp_h_errno;

FROM TaskControl IMPORT
    (* type *)  Lock,
    (* proc *)  CreateLock, Obtain, Release, CreateTask;

FROM Semaphores IMPORT
    (* type *)  Semaphore,
    (* proc *)  CreateSemaphore, Wait, Signal;

(********************************************************************************)

TYPE
    NameString = ARRAY [0..511] OF CHAR;

    ElementPointer = POINTER TO
                         RECORD
                             next: ElementPointer;
                             Address: CARDINAL;
                             HostName: NameString;
                         END (*RECORD*);

(********************************************************************************)

CONST
    Nul = CHR(0);

VAR
    (* Queue of translation requests.  New requests are added at the tail of    *)
    (* the queue.  The "nexttoconvert" field points at the first entry which is *)
    (* not yet translated.  Results may be pulled out of the queue in any       *)
    (* order.  In particular it's possible to pull out a not-yet-computed       *)
    (* result.  In that case we decide that the nameserver has not been fast    *)
    (* enough for us, and we return an empty string as the result.              *)

    Requests: RECORD
                  head, nexttoconvert, tail: ElementPointer;
                  access: Lock;
                  ConversionRequest: Semaphore;
              END (*RECORD*);

(********************************************************************************)
(*                           QUEUE MANIPULATIONS                                *)
(********************************************************************************)

PROCEDURE TakeFromQueue (key: CARDINAL;  Retain: BOOLEAN): ElementPointer;

    (* Searches the queue for an address field match, removes the corresponding *)
    (* element from the queue and returns it.  A NIL result means that no       *)
    (* match was found.  If Retain = TRUE then we return a result but don't     *)
    (* remove the queue entry.                                                  *)

    VAR previous, result: ElementPointer;

    BEGIN
        Obtain (Requests.access);
        previous := NIL;  result := Requests.head;
        WHILE (result <> NIL) AND (result^.Address <> key) DO
            previous := result;
            result := result^.next;
        END (*WHILE*);
        IF (result <> NIL) AND NOT Retain THEN
            IF result = Requests.nexttoconvert THEN
                Requests.nexttoconvert := Requests.nexttoconvert^.next;
            END (*IF*);
            IF result = Requests.tail THEN
                Requests.tail := previous;
            END (*IF*);
            IF previous = NIL THEN
                Requests.head := result^.next;
            ELSE
                previous^.next := result^.next;
            END (*IF*);
        END (*IF*);
        Release (Requests.access);
        RETURN result;
    END TakeFromQueue;

(********************************************************************************)
(*                             NAMESERVER ENQUIRIES                             *)
(********************************************************************************)

PROCEDURE TranslationTask;

    (* Runs as an autonomous task, working through the request queue and        *)
    (* translating each entry as it comes to it.                                *)

    VAR NameDetails: HostEntPtr;
        addr: CARDINAL;

    BEGIN
        LOOP
            WITH Requests DO

                (* Get the next translation request. *)

                Wait (ConversionRequest);
                Obtain (Requests.access);

                (* Watch out for cancelled requests. *)

                IF nexttoconvert <> NIL THEN
                    addr := nexttoconvert^.Address;
                    Release (Requests.access);

                    (* Perform a nameserver lookup.  (This can be slow.)  *)

                    NameDetails := gethostbyaddr (addr, SIZE(CARDINAL), AF_INET);

                    (* Put the result back into the queue.  Beware of the situation *)
                    (* where the request has been cancelled.                        *)

                    Obtain (Requests.access);
                    IF (nexttoconvert <> NIL) AND (nexttoconvert^.Address = addr) THEN
                        IF (tcp_h_errno() = 0) AND (NameDetails <> NIL)
                                          AND (NameDetails^.h_name <> NIL) THEN
                            Strings.Assign (NameDetails^.h_name^, nexttoconvert^.HostName);
                        ELSE
                            nexttoconvert^.HostName[0] := Nul;
                        END (*IF*);
                        nexttoconvert := nexttoconvert^.next;
                    END (*IF*);
                END (*IF*);

                Release (Requests.access);

            END (*WITH*);

        END (*LOOP*);

    END TranslationTask;

(********************************************************************************)
(*                           THE CLIENT INTERFACE                               *)
(********************************************************************************)

PROCEDURE StartNameLookup (IPAddress: CARDINAL);

    (* Enters a request for translating an IP address to a node name.   *)

    VAR p: ElementPointer;

    BEGIN
        NEW (p);
        WITH p^ DO
            next := NIL;
            Address := IPAddress;
            HostName[0] := Nul;
        END (*WITH*);
        WITH Requests DO
            Obtain (access);
            IF tail = NIL THEN
                head := p;
            ELSE
                tail^.next := p;
            END (*IF*);
            tail := p;
            IF nexttoconvert = NIL THEN
                nexttoconvert := p;
            END (*IF*);
            Release (access);
            Signal (ConversionRequest);
        END (*WITH*);
    END StartNameLookup;

(********************************************************************************)

PROCEDURE CancelNameLookup (IPAddress: CARDINAL);

    (* The translation requested by StartNameLookup may be aborted. *)

    VAR dummy: ElementPointer;

    BEGIN
        dummy := TakeFromQueue (IPAddress, FALSE);
        IF dummy <> NIL THEN
            DISPOSE (dummy);
        END (*IF*);
    END CancelNameLookup;

(********************************************************************************)

PROCEDURE GetName (IPAddress: CARDINAL;  VAR (*OUT*) Name: ARRAY OF CHAR;
                                                         Retain: BOOLEAN);

    (* Picks up the result of the query that was initiated by StartNameLookup.  *)
    (* If Retain is TRUE we keep the result for a possible future enquiry.      *)

    VAR result: ElementPointer;

    BEGIN
        result := TakeFromQueue (IPAddress, Retain);
        IF result = NIL THEN
            Name[0] := Nul;
        ELSE
            Strings.Assign (result^.HostName, Name);
            IF NOT Retain THEN
                DISPOSE (result);
            END (*IF*);
        END (*IF*);
    END GetName;

(********************************************************************************)
(*                              INITIALISATION                                  *)
(********************************************************************************)

BEGIN
    WITH Requests DO
        head := NIL;  nexttoconvert := NIL;  tail := NIL;
        CreateLock (access);
        CreateSemaphore (ConversionRequest, 0);
    END (*WITH*);
    CreateTask (TranslationTask, 3, "translation");
END NameLookup.

