IMPLEMENTATION MODULE Resources;

        (********************************************************)
        (*                                                      *)
        (*   String resources for FtpServer and its utilities   *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            19 March 1998                   *)
        (*  Last edited:        20 March 1998                   *)
        (*  Status:             Just started                    *)
        (*                                                      *)
        (********************************************************)


IMPORT IOChan, IOConsts, Strings, ChanConsts, SeqFile;

FROM SYSTEM IMPORT
    (* proc *)  ADR;

FROM Storage IMPORT
    (* proc *)  ALLOCATE;

FROM TextIO IMPORT
    (* proc *)  ReadRestLine, SkipLine;

(************************************************************************)

CONST
    Nul = CHR(0);
    TableSize = 101;  (* Ideally should be a prime number *)

TYPE
    TableSubscript = [0..TableSize-1];

    StringList = POINTER TO OneString;
    OneString = RECORD
                    next: StringList;
                    this: StringPointer;
                    StringNumber: CARDINAL;
                END (*RECORD*);

VAR
    (* Table of string lists. *)

    HashTable: ARRAY TableSubscript OF StringList;

    (* Default string. *)

    EmptyString: ARRAY [0..0] OF CHAR;

(************************************************************************)
(*                    MAIN EXTERNALLY CALLABLE PROCEDURE                *)
(************************************************************************)

PROCEDURE RString (N: CARDINAL): StringPointer;

    (* Returns a pointer to string number N. *)

    VAR p: StringList;

    BEGIN
        p := HashTable[N MOD TableSize];
        WHILE (p <> NIL) AND (p^.StringNumber <> N) DO
            p := p^.next;
        END (*WHILE*);
        IF p = NIL THEN
            RETURN ADR(EmptyString);
        ELSE
            RETURN p^.this;
        END (*IF*);
    END RString;

(************************************************************************)
(*           LOADING THE STRING TABLE FROM THE RESOURCE FILE            *)
(************************************************************************)

PROCEDURE AddToTable (N: CARDINAL;  entry: StringPointer);

    (* Puts 'entry' in the hash table. *)

    VAR TablePosition: TableSubscript;
        p: StringList;

    BEGIN
        TablePosition := N MOD TableSize;
        NEW (p);
        WITH p^ DO
            next := HashTable[TablePosition];
            this := entry;
            StringNumber := N;
        END (*WITH*);
        HashTable[TablePosition] := p;
    END AddToTable;

(************************************************************************)

PROCEDURE LoadStrings (filename: ARRAY OF CHAR);

    (* Loads the string table from the specified file. *)

    TYPE BufferSubscript = [0..255];
         CharSet = SET OF CHAR;

    CONST Digits = CharSet {'0'..'9'};

    VAR cid: IOChan.ChanId;  status: ChanConsts.OpenResults;
        Buffer: ARRAY BufferSubscript OF CHAR;
        k: [0..MAX(BufferSubscript)+1];
        N: CARDINAL;
        ptr: StringPointer;

    BEGIN
        SeqFile.OpenRead (cid, filename,
                               ChanConsts.read+ChanConsts.text, status);
        IF status = ChanConsts.opened THEN
            LOOP
                ReadRestLine (cid, Buffer);
                IF IOChan.ReadResult(cid) = IOConsts.endOfInput THEN
                    EXIT (*LOOP*);
                END (*IF*);
    
                (* Skip past leading blanks. *)
    
                k := 0;
                WHILE (k <= MAX(BufferSubscript)) AND (Buffer[k] = ' ') DO
                    INC (k);
                END (*WHILE*);
    
                (* Ignore any line that does not start with a number. *)
    
                IF Buffer[k] IN Digits THEN
    
                    (* Convert the number *)
    
                    N := 0;
                    REPEAT
                        N := 10*N + (ORD(Buffer[k]) - ORD('0'));
                        INC (k);
                    UNTIL (k > MAX(BufferSubscript)) OR NOT (Buffer[k] IN Digits);
    
                    (* There's supposed to be a trailing period. *)
    
                    IF (k <= MAX(BufferSubscript)) AND (Buffer[k] = '.') THEN
                        INC (k);
                    END (*IF*);
                    Strings.Delete (Buffer, 0, k);
    
                    (* Take the remainder of buffer as the string value. *)
    
                    <* storage+ *>
                    NEW (ptr, LENGTH(Buffer)+1);
                    Strings.Assign (Buffer, ptr^);
                    AddToTable (N, ptr);
    
                END (*IF*);
    
                (* Go to next line. *)
    
                SkipLine (cid);
    
            END (*LOOP*);
            SeqFile.Close (cid);

        END (*IF*);

    END LoadStrings;

(************************************************************************)
(*                           INITIALISATION                             *)
(************************************************************************)

PROCEDURE ClearTable;

    (* Initialises the string table to "empty". *)

    VAR j: TableSubscript;

    BEGIN
        FOR j := 0 TO MAX(TableSubscript) DO
            HashTable[j] := NIL;
        END (*FOR*);
    END ClearTable;

(************************************************************************)

BEGIN
    EmptyString[0] := Nul;
    ClearTable;
END Resources.

