/* 
* rmail.c
* 
* Ported to OS/2 and hacked to be useful for UUPC/Extended and IBM sendmail 
* by Jochen Friedrich <jochen@audio.ruessel.sub.org>.
* Please send bug reports for this port to me, not to the original author.
*
*/ 

/*
 * Copyright (c) 1988, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * RMAIL -- UUCP mail server.
 *
 * This program reads the From ... remote from ... lines that UUCP is so
 * fond of and turns them into something reasonable.  It then execs sendmail
 * with various options built from these lines. 
 *
 * The expected syntax is:
 *
 *	 <user> := [-a-z0-9]+
 *	 <date> := ctime format
 *	 <site> := [-a-z0-9!]+
 * <blank line> := "^\n$"
 *	 <from> := "From" <space> <user> <space> <date>
 *		  [<space> "remote from" <space> <site>]
 *    <forward> := ">" <from>
 *	    msg := <from> <forward>* <blank-line> <body>
 *
 * The output of rmail(8) compresses the <forward> lines into a single
 * from path.
 *
 * The err(3) routine is included here deliberately to make this code
 * a bit more portable.
 */

#include <sys\stat.h>
#include <process.h>
#include <ctype.h>
#include <io.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "getopt.h"
#include "pipe.h"

void err(int, const char *, ...);
void usage(void);

int main(int argc, char *argv[])
{
	extern char *optarg;
	extern int errno, optind;
	FILE *fp, *infp;
	size_t fplen, fptlen, len;
	off_t offset;
	int ch, debug, i, pid, tflag, wflag, stat;
	unsigned long pdes[2], tempfd;
	char *addrp, *domain, *p, *t, *infile, *subject;
	char *from_path, *from_sys, *from_user;
	char *args[100], buf[2048], lbuf[2048];

	infile = NULL;
	subject = NULL;
	debug = 0;
	tflag = 0;
	wflag = 0;
	infp = stdin;
	domain = "UUCP";		/* Default "domain". */
	while ((ch = getopt(argc, argv, "+s:f:F:D:Ttw")) != EOF)
		switch (ch) {
		case 'T':
			debug = 1;
			break;
		case 'D':
			domain = optarg;
			break;
		case 'f':
		case 'F':
			infile = optarg;
			infp = fopen(infile,"r");
			break;
		case 's':
			subject = optarg;
			break;
		case 't':
			tflag = 1;
			break;
		case 'w':
			wflag = 1;
			break;
		case '?':
		default:
			usage();
		}

	if (subject && !wflag) usage();
	if (tflag && wflag) usage();

	argc -= optind;
	argv += optind;
	if ((argc < 1) && (tflag==0))
		usage();

	from_path = from_sys = from_user = NULL;
	for (offset = 0;;) {

		/* Get and nul-terminate the line. */
		if (fgets(lbuf, sizeof(lbuf), infp) == NULL)
			exit (2);
		if ((p = strchr(lbuf, '\n')) == NULL)
			err(2, "line too long");
		*p = '\0';

		/* Parse lines until reach a non-"From" line. */
		if (!strnicmp(lbuf, "From ", 5))
			addrp = lbuf + 5;
		else if (!strnicmp(lbuf, ">From ", 6))
			addrp = lbuf + 6;
		else if ((offset == 0) && (tflag == 0) && (wflag == 0))
			err(2,
			    "missing or empty From line: %s", lbuf);
		else {
			*p = '\n';
			break;
		}

		if ((tflag == 0) && (wflag == 0)) {
			if (*addrp == '\0')
				err(2, "corrupted From line: %s", lbuf);

			/* Use the "remote from" if it exists. */
			for (p = addrp; (p = strchr(p + 1, 'r')) != NULL;)
				if (!strnicmp(p, "remote from ", 12)) {
					for (t = p += 12; *t && !isspace(*t); ++t);
					*t = '\0';
					if (debug)
						(void)fprintf(stderr,
						    "remote from: %s\n", p);
					break;
				}

			/* Else use the string up to the last bang. */
			if (p == NULL)
				if (*addrp == '!')
					err(2,
					    "bang starts address: %s", addrp);
				else if ((t = strrchr(addrp, '!')) != NULL) {
					*t = '\0';
					p = addrp;
					addrp = t + 1;
					if (*addrp == '\0')
						err(2,
						    "corrupted From line: %s", lbuf);
					if (debug)
						(void)fprintf(stderr, "bang: %s\n", p);
				}

			/* 'p' now points to any system string from this line. */
			if (p != NULL) {
				/* Nul terminate it as necessary. */
				for (t = p; *t && !isspace(*t); ++t);
				*t = '\0';

				/* If the first system, copy to the from_sys string. */
				if (from_sys == NULL) {
					if ((from_sys = strdup(p)) == NULL)
						err(3, NULL);
					if (debug)
						(void)fprintf(stderr,
						    "from_sys: %s\n", from_sys);
				}

				/* Concatenate to the path string. */
				len = t - p;
				if (from_path == NULL) {
					fplen = 0;
					if ((from_path = malloc(fptlen = 256)) == NULL)
						err(3, NULL);
				}
				if (fplen + len + 2 > fptlen) {
					fptlen += max(fplen + len + 2, 256);
					if ((from_path =
					    realloc(from_path, fptlen)) == NULL)
						err(3, NULL);
				}
				memmove(from_path + fplen, p, len);
				fplen += len;
				from_path[fplen++] = '!';
				from_path[fplen] = '\0';
			}

			/* Save off from user's address; the last one wins. */
			for (p = addrp; *p && !isspace(*p); ++p);
			*p = '\0';
			if (from_user != NULL)
				free(from_user);
			if ((from_user = strdup(addrp)) == NULL)
				err(3, NULL);

			if (debug) {
				if (from_path != NULL)
					(void)fprintf(stderr,
					    "from_path: %s\n", from_path);
				(void)fprintf(stderr, "from_user: %s\n", from_user);
			}

			if (offset != -1)
				offset = (off_t)ftell(infp);
		}
	}
	i = 0;
	args[i++] = "sendmail.exe";	/* Build sendmail's argument list. */

#ifdef ALL_FLAG
	args[i++] = "-oee";		/* No errors, just status. */
	args[i++] = "-odq";		/* Queue it, don't try to deliver. */
	args[i++] = "-oi";		/* Ignore '.' on a line by itself. */
#endif

	if ((tflag) || (wflag)) {
		args[i++] = "-t";
		from_user = getenv("LOGNAME");
		if (from_user) {
			(void)sprintf(buf,"-f %s",from_user);
			if ((args[i++] = strdup(buf)) == NULL)
				 err(3, NULL);
		}			
	} else {


#ifdef OM_FLAG
		if (from_sys != NULL) {		/* Set sender's host name. */
			if (strchr(from_sys, '.') == NULL)
				(void)sprintf(buf,
				    "-oMs%s.%s", from_sys, domain);
			else
				(void)sprintf(buf, "-oMs%s", from_sys);
			if ((args[i++] = strdup(buf)) == NULL)
				 err(3, NULL);
		}
					/* Set protocol used. */
		(void)sprintf(buf, "-oMr%s", domain);
		if ((args[i++] = strdup(buf)) == NULL)
			err(3, NULL);
#endif
					/* Set name of ``from'' person. */
		(void)sprintf(buf, "-f %s%s",
		    from_path ? from_path : "", from_user);
		if ((args[i++] = strdup(buf)) == NULL)
			err(3, NULL);
	}

	/*
	 * Don't copy arguments beginning with - as they will be
	 * passed to sendmail and could be interpreted as flags.
	 */
	if ((tflag == 0) && (wflag == 0)) do {
		if (*argv && **argv == '-')
			err(4, "dash precedes argument: %s", *argv);
	} while ((args[i++] = *argv++) != NULL);
	else args[i++] = NULL;
	if (debug) {
		(void)fprintf(stderr, "Sendmail arguments:\n");
		for (i = 0; args[i]; i++)
			(void)fprintf(stderr, "\t%s\n", args[i]);
	}

	if (pipe(pdes) < 0)
		err(5, NULL);
	tempfd=dup(0);
	if (pdes[0] != 0) {
		dup2(pdes[0], 0);
	}
        close(pdes[0]);
	pid=spawnvp(P_NOWAIT,"sendmail.exe", args);
	dup2(tempfd, 0);
        close(tempfd);
	if ((fp = fdopen(pdes[1], "w")) == NULL)
		err(5, NULL);

	/* Copy the file down the pipe. */
	
	if (wflag) {
		strcpy(buf,"To: %s\n");
		while (*argv) {
			if (!strcmp(*argv,"-c") && (*buf == 'T'))
				strcpy(buf,"Cc: %s\n");
			else if (!strcmp(*argv,"-b"))
				strcpy(buf,"Bcc: %s\n");
			else fprintf(fp,buf,*argv);
			argv++;
		}
		if (subject) fprintf(fp,"Subject: %s\n",subject);
		fprintf(fp, "\n");
	}
	do {
		fprintf(fp, "%s", lbuf);
	} while (fgets(lbuf, sizeof(lbuf), infp) != NULL);

	if (ferror(infp))
		err(3, "infp: %s", strerror(errno));

	if (fclose(fp))
		err(5, NULL);

	if ((_cwait(&stat, pid, WAIT_CHILD)) == -1)
		err(5, "%s", "sendmail.exe");

	if ((stat&0xff))
		err(5,
		    "%s: did not terminate normally", "sendmail.exe");

	if ((stat&0xff00))
		err(stat, "%s: terminated with %d (non-zero) status",
		    "sendmail.exe", (stat&0xff00)>>8);
	if (infile != NULL) fclose(infp);
	exit(0);
}

void usage(void)
{
	(void)fprintf(stderr, "usage: rmail [-T|-t] [-f|-F file] [-D domain] user ...\n");
	(void)fprintf(stderr, "       rmail -w [-f|-F file] [-s Subject] user ... [-c user ...] [-b user ...]\n");
	exit(4);
}

#include <stdarg.h>
void err(int eval, const char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	(void)fprintf(stderr, "rmail: ");
	(void)vfprintf(stderr, fmt, ap);
	va_end(ap);
	(void)fprintf(stderr, "\n");
	exit(eval);
}
