IMPLEMENTATION MODULE SMTPData;

        (********************************************************)
        (*                                                      *)
        (*      Part of the SMTP server - files the mail        *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            27 April 1998                   *)
        (*  Last edited:        4 November 1999                 *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (********************************************************)

FROM SYSTEM IMPORT ADDRESS, CAST, ADR;

IMPORT Strings, IOChan, IOConsts, ChanConsts, RndFile, TextIO, OS2, FileSys;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

FROM TaskControl IMPORT
    (* type *)  Lock,
    (* proc *)  CreateLock, Obtain, Release;

FROM Semaphores IMPORT
    (* type *)  Semaphore,
    (* proc *)  Signal;

FROM FDFiles IMPORT
    (* proc *)  OpenAtEnd, CloseFile, FWriteChar, FWriteString,
                FWriteCard, FWriteLn;

FROM Names IMPORT
    (* type *)  UserName, HostName, FilenameIndex, FilenameString, PathString;

FROM Hosts IMPORT
    (* proc *)  OurHostName, HostIsLocal;

FROM Sockets IMPORT
    (* const*)  AF_INET,
    (* type *)  Socket, SockAddr,
    (* proc *)  getpeername, recv;

FROM NetDB IMPORT
    (* type *)  HostEntPtr,
    (* proc *)  gethostbyaddr;

FROM MyClock IMPORT
    (* proc *)  CurrentDateAndTime;

FROM InetUtilities IMPORT
    (* proc *)  OpenINIFile, INIGet, INIGetString, INIPut, WriteCard,
                IPToString, ToLower, CurrentTimeToString;

FROM RelayMail IMPORT
    (* type *)  RelayList,
    (* proc *)  UserAndDomain, Empty, WriteRelayList,
                AddToRelayList, SendRelayMail, DeleteRelayList;

FROM TransLog IMPORT
    (* type *)  TransactionLogID,
    (* proc *)  LogTransaction;

FROM SplitScreen IMPORT
    (* proc *)  ReleaseScreen, RegainScreen;

(************************************************************************)

CONST Nul = CHR(0);  CR = CHR(13);  LF = CHR(10);

TYPE
    (* Descriptor for the mailbox of a local user.                      *)
    (*     user         username - needed for detecting duplicates      *)
    (*     DirName      name of the mailbox directory.                  *)
    (*     Skip         ignore this entry except when checking for      *)
    (*                     duplicates                                   *)

    Recipient = POINTER TO
                   RECORD
                       next: RecipientList;
                       user: UserName;
                       DirName: FilenameString;
                       Skip: BOOLEAN;
                   END (*RECORD*);

    RecipientList = Recipient;

    (* An ItemDescriptor record keeps track of the information needed   *)
    (* to send one item of mail.  The fields are                        *)
    (*   RealIPAddr        the sender's real IP address as determined   *)
    (*                        from the socket binding                   *)
    (*   RealName          the sender's hostname as determined by       *)
    (*                        nameserver lookup                         *)
    (*   HELOname          the sender's hostname as supplied in the     *)
    (*                        HELO command                              *)
    (*   TempName          name of a file where the incoming item is    *)
    (*                       stored before being distributed to all     *)
    (*                       recipients.                                *)
    (*   offset            effective starting point of the file if it   *)
    (*                       has to be relayed.                         *)
    (*   charcount         bytes received                               *)
    (*   LogID             ID used for transaction logging              *)
    (*   returnpath        path supplied by MAIL FROM:                  *)
    (*   LocalRecipients   list of local mailboxes                      *)
    (*   RelayRecipients   list of non-local destinations               *)
    (*   RelayAllowed      TRUE iff the sending host is one that is     *)
    (*                       allowed to relay through us, and allowed   *)
    (*                       access to non-public aliases.              *)

    ItemDescriptor = POINTER TO
                         RECORD
                             RealIPAddr: CARDINAL;
                             RealName: HostName;
                             HELOname: HostName;
                             LogID: TransactionLogID;
                             TempName: FilenameString;
                             offset: RndFile.FilePos;
                             charcount: CARDINAL;
                             returnpath: PathString;
                             LocalRecipients: RecipientList;
                             RelayRecipients: RelayList;
                             RelayAllowed: BOOLEAN;
                         END (*RECORD*);

VAR
    (* NextName is a string used in generating unique file names. *)

    NextName: ARRAY [0..7] OF CHAR;
    NextNameLock: Lock;

    (* MailRoot is the name of the directory holding all the user mail  *)
    (* directories.  ForwardDirName is the name of the directory that   *)
    (* holds outgoing mail that has not yet been sent.                  *)

    MailRoot, ForwardDirName: FilenameString;

    (* Script to run after receiving a mail item. *)

    FilterProg: FilenameString;

    (* Software version identifier. *)

    version: ARRAY [0..15] OF CHAR;

    (* Critical section protection on the mail item log. *)

    LogFileLock: Lock;

    (* "Enable" flag for the logging. *)

    LogSMTPItems: BOOLEAN;

(************************************************************************)
(*                      GENERAL PARAMETER SETTINGS                      *)
(************************************************************************)

PROCEDURE StoreVersion (v: ARRAY OF CHAR);

    (* Stores the version number. *)

    BEGIN
        Strings.Assign (v, version);
    END StoreVersion;

(************************************************************************)

PROCEDURE GetMailRoot (VAR (*OUT*) dirname: FilenameString);

    (* Returns the name of the mail root directory to the caller. *)

    BEGIN
        dirname := MailRoot;
    END GetMailRoot;

(************************************************************************)
(*                MAINTAINING A LIST OF LOCAL RECIPIENTS                *)
(************************************************************************)

PROCEDURE DeleteRecipientList (VAR (*INOUT*) RL: RecipientList);

    (* Discards a list of mailbox records, and sets RL to NIL. *)

    VAR next: RecipientList;

    BEGIN
        WHILE RL <> NIL DO
            next := RL^.next;
            DISPOSE (RL);
            RL := next;
        END (*WHILE*);
    END DeleteRecipientList;

(************************************************************************)
(*                    FINDING THE REAL HOST NAME                        *)
(************************************************************************)

PROCEDURE RealNameOfPeer (S: Socket;  VAR (*OUT*) name: HostName;
                                   VAR (*OUT*) IPaddr: CARDINAL): BOOLEAN;

    (* Checks the nameserver for the peer's official name. *)

    VAR peer: SockAddr;  HostInfo: HostEntPtr;  number: CARDINAL;
        found: BOOLEAN;

    BEGIN
        number := SIZE (SockAddr);
        IF getpeername (S, peer, number) THEN
            IPaddr := 0;  found := FALSE;
        ELSE
            number := peer.in_addr.addr;
            IPaddr := number;
            HostInfo := gethostbyaddr (number, SIZE(CARDINAL), AF_INET);
            IF HostInfo = NIL THEN
                IPToString (number, name);
                found := TRUE;
            ELSE
                IF HostInfo^.h_name <> NIL THEN
                    Strings.Assign (HostInfo^.h_name^, name);
                    found := TRUE;
                ELSIF (HostInfo^.h_addr_list <> NIL)
                          AND (HostInfo^.h_addr_list^[0] <> NIL) THEN
                    IPToString (HostInfo^.h_addr_list^[0]^, name);
                    found := TRUE;
                ELSE
                    IPToString (number, name);
                    found := TRUE;
                END (*IF*);
            END (*IF*);
        END (*IF*);

        IF NOT found THEN
            name := "";
        END (*IF*);

        RETURN found;

    END RealNameOfPeer;

(************************************************************************)
(*                KEEPING TRACK OF THE ITEM INFORMATION                 *)
(************************************************************************)

PROCEDURE CreateItemDescriptor (S: Socket;  ID: TransactionLogID;
                                 MayRelay: BOOLEAN): ItemDescriptor;

    (* Creates a descriptor for a new mail item.  ID is for *)
    (* transaction logging.                                 *)

    VAR result: ItemDescriptor;

    BEGIN
        NEW (result);
        WITH result^ DO
            IF NOT RealNameOfPeer (S, RealName, RealIPAddr) THEN
                 (* Can't find name from nameserver. *)
                 RealName := "";
            END (*IF*);
            LogID := ID;
            HELOname := "";
            TempName := "";
            returnpath := "";
            LocalRecipients := NIL;
            RelayRecipients := NIL;
            RelayAllowed := MayRelay;
        END (*WITH*);
        RETURN result;
    END CreateItemDescriptor;

(************************************************************************)

PROCEDURE ResetItemDescriptor (desc: ItemDescriptor;
                               ReturnPath: ARRAY OF CHAR);

    (* Discards information related to sender and receivers, deletes    *)
    (* message file if one has been created, and sets a new return path.*)

    VAR dummy: BOOLEAN;

    BEGIN
        IF desc <> NIL THEN
            IF desc^.TempName[0] <> Nul THEN
                FileSys.Remove (desc^.TempName, dummy);
                desc^.TempName[0] := Nul;
            END (*IF*);
            Strings.Assign (ReturnPath, desc^.returnpath);
            DeleteRecipientList (desc^.LocalRecipients);
            DeleteRelayList (desc^.RelayRecipients);
        END (*IF*);
    END ResetItemDescriptor;

(************************************************************************)

PROCEDURE DiscardItemDescriptor (VAR (*INOUT*) desc: ItemDescriptor);

    (* Destroys the descriptor, and deletes the message file if one     *)
    (* has been created.                                                *)

    BEGIN
        IF desc <> NIL THEN
            ResetItemDescriptor (desc, "");
            DISPOSE (desc);
        END (*IF*);
    END DiscardItemDescriptor;

(************************************************************************)

PROCEDURE SetClaimedSendingHost (desc: ItemDescriptor;
                                 VAR (*IN*) ClaimedName: ARRAY OF CHAR);

    (* ClaimedName is the sending host's name as supplied in the HELO command. *)

    BEGIN
        Strings.Assign (ClaimedName, desc^.HELOname);
    END SetClaimedSendingHost;

(************************************************************************)

PROCEDURE SenderNotSpecified (desc: ItemDescriptor): BOOLEAN;

    (* Returns TRUE iff the reverse path is empty. *)

    BEGIN
        RETURN desc^.returnpath[0] = Nul;
    END SenderNotSpecified;

(************************************************************************)

PROCEDURE NoRecipients (desc: ItemDescriptor): BOOLEAN;

    (* Returns TRUE iff the list of recipients is empty. *)

    VAR current, LocalRL: RecipientList;

    BEGIN
        LocalRL := desc^.LocalRecipients;

        (* Watch out for the special case where all local recipient     *)
        (* records are "Skip" records.  In this check we discard all    *)
        (* the leading "Skip" records, but the interior ones are left   *)
        (* to be dealt with later.                                      *)

        WHILE (LocalRL <> NIL) AND LocalRL^.Skip DO
            current := LocalRL;  LocalRL := LocalRL^.next;
            DISPOSE (current);
        END (*WHILE*);
        desc^.LocalRecipients := LocalRL;

        RETURN (LocalRL = NIL) AND Empty(desc^.RelayRecipients);

    END NoRecipients;

(************************************************************************)
(*                   UPDATING LISTS OF MAIL RECIPIENTS                  *)
(************************************************************************)

PROCEDURE AddRecipient (desc: ItemDescriptor;
                             VAR (*IN*) name: ARRAY OF CHAR);

    (* Checks whether "name" is local or non-local, and adds it to the  *)
    (* appropriate list.  Invalid local recipients are silently         *)
    (* discarded.                                                       *)

    VAR user: UserName;  domain: HostName;

    BEGIN
        UserAndDomain (name, user, domain);
        IF HostIsLocal (domain) THEN
            EVAL (AddLocalRecipient (desc, user));
        ELSE
            Strings.Assign ("<", name);
            Strings.Append (user, name);
            Strings.Append ("@", name);
            Strings.Append (domain, name);
            Strings.Append (">", name);
            AddRelayRecipient (desc, name);
        END (*IF*);
    END AddRecipient;

(************************************************************************)

PROCEDURE ExpandAlias (desc: ItemDescriptor;
                             VAR (*IN*) alias: ARRAY OF CHAR): BOOLEAN;

    (* If "alias" is a valid alias, puts the expansion onto the lists   *)
    (* in desc and returns TRUE.  Otherwise returns FALSE.  Private     *)
    (* aliases are not recognised unless desc^.RelayAllowed is TRUE.    *)

    VAR hini: OS2.HINI;  success: BOOLEAN;  j, k, size: CARDINAL;
        bufptr: POINTER TO ARRAY [0..MAX(CARDINAL) DIV 4] OF CHAR;
        name: PathString;

    BEGIN
        size := 0;  bufptr := NIL;
        hini := OpenINIFile ("weasel.ini");
        success := hini <> OS2.NULLHANDLE;
        IF success THEN
            OS2.PrfQueryProfileSize (hini, "$ALIAS", alias, size);
            IF size > 0 THEN
                ALLOCATE (bufptr, size);
                OS2.PrfQueryProfileData (hini, "$ALIAS", alias, bufptr, size);
            ELSE
                success := FALSE;
            END (*IF*);
            OS2.PrfCloseProfile (hini);
        END (*IF*);

        success := success AND (desc^.RelayAllowed OR (bufptr^[0] = CHR(1)));
        IF success THEN
            j := 1;
            LOOP
                IF (j >= size) OR (bufptr^[j] = Nul) THEN
                    EXIT (*LOOP*);
                END (*IF*);
                k := 0;
                REPEAT
                    name[k] := bufptr^[j];
                    INC (k);  INC(j);
                UNTIL name[k-1] = Nul;
                IF k > 1 THEN
                    AddRecipient (desc, name);
                END (*IF*);
            END (*LOOP*);
            DEALLOCATE (bufptr, size);
        END (*IF*);

        RETURN success;

    END ExpandAlias;

(************************************************************************)

PROCEDURE IsADuplicate (RL: RecipientList;
                             VAR (*IN*) name: ARRAY OF CHAR): BOOLEAN;

    (* Returns TRUE if name is already on the list. *)

    BEGIN
        WHILE (RL <> NIL) AND NOT Strings.Equal (RL^.user, name) DO
            RL := RL^.next;
        END (*WHILE*);
        RETURN RL <> NIL;
    END IsADuplicate;

(************************************************************************)

PROCEDURE IsValidUsername (name: ARRAY OF CHAR): BOOLEAN;

    (* Returns TRUE iff this is the name of a local mailbox. *)

    VAR hini: OS2.HINI;  result: BOOLEAN;  size: CARDINAL;

    BEGIN
        hini := OpenINIFile ("weasel.ini");
        IF hini = OS2.NULLHANDLE THEN
            result := FALSE;
        ELSE
            result := OS2.PrfQueryProfileSize (hini, name, NIL, size)
                                   AND (size <> 0);
            OS2.PrfCloseProfile (hini);
        END (*IF*);
        RETURN result;

    END IsValidUsername;

(************************************************************************)

PROCEDURE AddLocalRecipient (desc: ItemDescriptor;
                             VAR (*IN*) name: ARRAY OF CHAR): BOOLEAN;

    (* Adds one local recipient to the list of recipients. *)
    (* Returns FALSE if it's not possible.                 *)

    VAR RL: RecipientList;  exists: BOOLEAN;

    BEGIN
        ToLower (name);
        IF IsADuplicate (desc^.LocalRecipients, name) THEN
            RETURN TRUE;
        END (*IF*);

        NEW (RL);
        WITH RL^ DO
            next := desc^.LocalRecipients;
            Strings.Assign (name, user);
            DirName := MailRoot;
            Skip := FALSE;
        END (*WITH*);
        desc^.LocalRecipients := RL;

        Strings.Append (name, RL^.DirName);
        exists := IsValidUsername (name);
        IF exists THEN
            Strings.Append ('\', RL^.DirName);
        ELSE
            RL^.Skip := TRUE;
            exists := ExpandAlias (desc, name);
        END (*IF*);

        IF NOT exists THEN
            desc^.LocalRecipients := RL^.next;
            DISPOSE (RL);
        END (*IF*);

        RETURN exists;

    END AddLocalRecipient;

(************************************************************************)

PROCEDURE AddRelayRecipient (desc: ItemDescriptor;
                             VAR (*IN*) name: ARRAY OF CHAR);

    (* Adds one non-local recipient to the list of recipients. *)

    BEGIN
        AddToRelayList (desc^.RelayRecipients, name);
    END AddRelayRecipient;

(********************************************************************************)
(*                                ITEM LOGGING                                  *)
(********************************************************************************)

PROCEDURE WriteLogItem (desc: ItemDescriptor);

    (* Writes the summary for this item to the user log. *)

    VAR cid: IOChan.ChanId;  datetime: ARRAY [0..31] OF CHAR;
        p: RecipientList;

    BEGIN
        Obtain (LogFileLock);
        cid := OpenAtEnd ("SMTP.LOG");
        CurrentTimeToString (datetime);
        FWriteString (cid, datetime);
        FWriteString (cid, " ");
        FWriteString (cid, desc^.RealName);
        FWriteCard (cid, desc^.charcount, 10);
        FWriteString (cid, " ");

        (* Local recipients. *)

        p := desc^.LocalRecipients;
        WHILE p <> NIL DO
            FWriteString (cid, p^.user);
            FWriteString (cid, " ");
            p := p^.next;
        END (*WHILE*);

        (* Relay recipients. *)

        WriteRelayList (cid, desc^.RelayRecipients);

        FWriteLn (cid);
        CloseFile (cid);
        Release (LogFileLock);

    END WriteLogItem;

(************************************************************************)
(*                    CREATING A UNIQUE FILENAME                        *)
(************************************************************************)

PROCEDURE MakeUniqueName (VAR (*OUT*) name: ARRAY OF CHAR);

    (* Generates a unique 8-character string.  The argument must of     *)
    (* course be big enough to take at least 8 characters.              *)

    (********************************************************************)

    PROCEDURE Increment (N: CARDINAL);

        (* Increments NextName[N], with carry as appropriate. *)

        BEGIN
            IF NextName[N] = '9' THEN
                NextName[N] := 'A';
            ELSIF NextName[N] = 'Z' THEN
                IF N = 0 THEN
                    NextName := "00000000";
                ELSE
                    NextName[N] := '0';
                    Increment (N-1);
                END (*IF*);
            ELSE
                INC (NextName[N]);
            END (*IF*);
        END Increment;

    (********************************************************************)

    BEGIN
        Obtain (NextNameLock);
        Strings.Assign (NextName, name);
        Increment (7);
        Release (NextNameLock);
    END MakeUniqueName;

(************************************************************************)

PROCEDURE MakeNewFilename (BaseName, tail: ARRAY OF CHAR;
                       VAR (*OUT*) NewName: ARRAY OF CHAR);

    (* Creates a file name of the form BaseNamexxxtail, where xxx is    *)
    (* chosen such that a file of that name does not already exist.     *)
    (* Note that BaseName and tail can include punctuation.             *)

    VAR UName: FilenameString;

    BEGIN
        REPEAT
            MakeUniqueName (UName);
            Strings.Assign (BaseName, NewName);
            Strings.Append (UName, NewName);
            Strings.Append (tail, NewName);
        UNTIL NOT FileSys.Exists(NewName);
    END MakeNewFilename;

(************************************************************************)
(*               ACCEPTING AND FILING AN INCOMING MESSAGE               *)
(************************************************************************)

PROCEDURE NextChar (S: Socket): CHAR;

    (* Receives one character of an incoming message.  The result is    *)
    (* Nul if there was a communications failure.                       *)

    VAR ch: CHAR;  amount: CARDINAL;

    BEGIN
        REPEAT
            amount := recv (S, ch, 1, 0);
        UNTIL amount <> 0;
        IF amount = MAX(CARDINAL) THEN
            ch := Nul;
        END (*IF*);
        RETURN ch;
    END NextChar;

(************************************************************************)

PROCEDURE AcceptOneLine (S: Socket;
                             VAR (*OUT*) Buffer: ARRAY OF CHAR): CARDINAL;

    (* Receives one line of an incoming message.  Returns the number of *)
    (* characters read, or MAX(CARDINAL) for a reception failure.       *)

    CONST CR = CHR(13);  LF = CHR(10);

    VAR ch: CHAR;  success: BOOLEAN;
        pos: CARDINAL;

    BEGIN
        success := TRUE;  pos := 0;
        LOOP
            ch := NextChar (S);

            IF ch = Nul THEN

                (* Connection lost, give up. *)

                success := FALSE;  EXIT(*LOOP*);

            ELSIF ch = CR THEN

                (* Line terminator should be CRLF, but I suspect that   *)
                (* some unix software sends only LF.  To handle both    *)
                (* cases, ignore the CR and use only the LF.            *)

            ELSIF ch = LF THEN

                (* End of line. *)

                EXIT (*LOOP*);

            ELSIF pos <= HIGH(Buffer) THEN

                (* Normal case, store character.  In this version we    *)
                (* don't signal an error on line overflow.              *)

                Buffer[pos] := ch;  INC(pos);

            END (*IF*);

        END (*LOOP*);

        (* Make sure string is properly terminated. *)

        IF pos <= HIGH(Buffer) THEN
            Buffer[pos] := Nul;
        END (*IF*);
        IF success THEN
            RETURN pos;
        ELSE
            RETURN MAX(CARDINAL);
        END (*IF*);

    END AcceptOneLine;

(************************************************************************)

PROCEDURE ReceiveMessage (S: Socket;  cid: IOChan.ChanId;
                                      sem: Semaphore): CARDINAL;

    (* Receives an incoming message, stores it to a previously opened   *)
    (* file.  We periodically signal on sem to confirm that the         *)
    (* operation has not timed out.  The returned value is a character  *)
    (* count, or MAX(CARDINAL) if the transfer failed.                  *)

    TYPE TwoChar = ARRAY [0..1] OF CHAR;
    CONST CRLF = TwoChar {CR, LF};

    VAR LineBuffer: ARRAY [0..1023] OF CHAR;
        EndOfMessage: BOOLEAN;
        amount, total: CARDINAL;

    BEGIN
        EndOfMessage := FALSE;  total := 0;
        REPEAT
            Signal (sem);
            amount := AcceptOneLine (S, LineBuffer);
            IF amount = MAX(CARDINAL) THEN
                total := amount;
            ELSE
                IF LineBuffer[0] = '.' THEN
                    IF LineBuffer[1] = Nul THEN
                        EndOfMessage := TRUE;
                    ELSE
                        Strings.Delete (LineBuffer, 0, 1);
                        DEC (amount);
                    END (*IF*);
                END (*IF*);
                IF NOT EndOfMessage THEN
                    IF amount > 0 THEN
                        IOChan.RawWrite (cid, ADR(LineBuffer), amount);
                    END (*IF*);
                    Strings.Assign (CRLF, LineBuffer);
                    IOChan.RawWrite (cid, ADR(LineBuffer), 2);
                END (*IF*);
                INC (total, amount+2);
            END (*IF*);
        UNTIL EndOfMessage OR (total = MAX(CARDINAL));
        RETURN total;
    END ReceiveMessage;

(************************************************************************)

PROCEDURE AcceptMessage (S: Socket;  itemdata: ItemDescriptor;
                                                 sem: Semaphore): BOOLEAN;

    (* Receives an incoming message, stores it in a temporary file      *)
    (* whose name is recorded in itemdata.  We periodically signal on   *)
    (* sem to confirm that the reception has not timed out.             *)

    CONST MaxPosInLine = 80;

    VAR PosInLine: CARDINAL;

    (********************************************************************)

    PROCEDURE AddString (cid: IOChan.ChanId;  str: ARRAY OF CHAR);

        (* Same as FWriteString, except that we wrap to a new line      *)
        (* if PosInLine is too big.                                     *)

        VAR length: CARDINAL;

        BEGIN
            length := LENGTH(str);
            IF PosInLine + length > MaxPosInLine THEN
                FWriteLn (cid);  FWriteString (cid, ' ');
                PosInLine := 1;
            END (*IF*);
            FWriteString (cid, str);
            INC (PosInLine, length);
        END AddString;

    (********************************************************************)

    VAR success, dummy: BOOLEAN;
        cid: IOChan.ChanId;
        BaseName: FilenameString;
        result: ChanConsts.OpenResults;
        LocalRL: RecipientList;
        StringBuffer: ARRAY [0..31] OF CHAR;
        LocalHost: HostName;

    BEGIN
        success := NOT NoRecipients(itemdata);

        IF success THEN

            (* Create a temporary file in the mailbox of the first recipient, *)
            (* or in the Forward directory if there are no local recipients.  *)

            LocalRL := itemdata^.LocalRecipients;
            IF LocalRL = NIL THEN
                BaseName := ForwardDirName;
            ELSE
                BaseName := LocalRL^.DirName;
            END (*IF*);
            MakeNewFilename (BaseName, ".###", itemdata^.TempName);
            RndFile.OpenClean (cid, itemdata^.TempName,
                                   ChanConsts.write+ChanConsts.raw, result);
            success := result = ChanConsts.opened;
            IF success THEN

                (* Create a "Return-Path:" header line. *)

                PosInLine := 0;
                AddString (cid, "Return-Path: ");
                AddString (cid, itemdata^.returnpath);
                FWriteLn (cid);
                itemdata^.offset := RndFile.CurrentPos(cid);

                (* Create a "Received:" header line. *)

                PosInLine := 0;
                AddString (cid, "Received: from ");
                AddString (cid, itemdata^.HELOname);
                AddString (cid, " (" );
                AddString (cid, itemdata^.RealName);
                AddString (cid, " ");
                IPToString (itemdata^.RealIPAddr, StringBuffer);
                AddString (cid, StringBuffer);
                FWriteChar (cid, ')');  INC(PosInLine);
                AddString (cid, " by " );
                OurHostName (LocalHost);
                AddString (cid, LocalHost);
                AddString (cid, " (Weasel v");
                AddString (cid, version);
                AddString (cid, "); ");
                CurrentDateAndTime (StringBuffer);
                AddString (cid, StringBuffer);
                FWriteLn (cid);

                (* Read the new message into the temporary file. *)

                itemdata^.charcount := ReceiveMessage (S, cid, sem);
                RndFile.Close (cid);
                success := itemdata^.charcount <> MAX(CARDINAL);
                IF NOT success THEN
                    FileSys.Remove (itemdata^.TempName, dummy);
                    itemdata^.TempName[0] := Nul;
                END (*IF*);

            END (*IF*);

        END (*IF*);

        IF success AND LogSMTPItems THEN
            WriteLogItem (itemdata);
        END (*IF*);

        RETURN success;

    END AcceptMessage;

(************************************************************************)

PROCEDURE MakeRecipientListFile (desc: ItemDescriptor;
                                 VAR (*OUT*) filename: FilenameString);

    VAR success: BOOLEAN;
        cid: IOChan.ChanId;
        result: ChanConsts.OpenResults;
        p: RecipientList;

    BEGIN
        MakeNewFilename (MailRoot, ".REC", filename);
        RndFile.OpenClean (cid, filename,
                               ChanConsts.write+ChanConsts.raw, result);
        success := result = ChanConsts.opened;
        IF success THEN
            FWriteString (cid, desc^.returnpath);  FWriteLn (cid);
            p := desc^.LocalRecipients;
            WHILE p <> NIL DO
                IF NOT p^.Skip THEN
                    FWriteString (cid, p^.user);  FWriteLn (cid);
                END (*IF*);
                p := p^.next;
            END (*WHILE*);
            RndFile.Close (cid);
        END (*IF*);
        (*
        IF success THEN
            WriteString ("Recipient list is in ");
            WriteString (filename);
        ELSE
            WriteString ("Failed to open recipient list file");
        END (*IF*);
        WriteLn;
        *)
    END MakeRecipientListFile;

(************************************************************************)

PROCEDURE RebuildRecipientList (desc: ItemDescriptor;
                                filename: FilenameString);

    (* Discards the current recipients for desc^, and creates a new set *)
    (* of recipients from "filename".  The first line in that file is   *)
    (* ignored.                                                         *)

    VAR cid: IOChan.ChanId;
        result: ChanConsts.OpenResults;
        name: UserName;

    BEGIN
        DeleteRecipientList (desc^.LocalRecipients);
        DeleteRelayList (desc^.RelayRecipients);
        RndFile.OpenOld (cid, filename,
                               ChanConsts.read+ChanConsts.text, result);
        IF result = ChanConsts.opened THEN
            TextIO.SkipLine (cid);
            WHILE IOChan.ReadResult(cid) = IOConsts.allRight DO
                TextIO.ReadString (cid, name);
                IF name[0] <> Nul THEN
                    AddRecipient (desc, name);
                END (*IF*);
                TextIO.SkipLine (cid);
            END (*WHILE*);
            RndFile.Close (cid);
        END (*IF*);

    END RebuildRecipientList;

(************************************************************************)

PROCEDURE RunFilter (itemdata: ItemDescriptor): CARDINAL;

    (* This procedure is to be invoked after a mail item has been       *)
    (* received but before it has been distributed to the addressees.   *)
    (* It returns the following codes:                                  *)
    (*    0    continue processing normally, i.e. deliver mail          *)
    (*    1    like 0, but the filter has modified the list of          *)
    (*         recipients.                                              *)
    (*    2    item has now been dealt with, report success to sender   *)
    (*    3    reject the message                                       *)

    CONST ONLength = 256;

    VAR rc, result: CARDINAL;  dummy: BOOLEAN;
        pArg, pEnv: POINTER TO CHAR;
        ExitStatus: OS2.RESULTCODES;
        ArgString, EnvString, recipients: FilenameString;
        FailureObjectName: ARRAY [0..ONLength-1] OF CHAR;

    BEGIN
        IF FilterProg[0] = Nul THEN
            RETURN 0;
        END (*IF*);

        ReleaseScreen;
        MakeRecipientListFile (itemdata, recipients);
        EnvString[0] := Nul;
        ArgString := "/C ";
        Strings.Append (FilterProg, ArgString);
        Strings.Append (" ", ArgString);
        Strings.Append (itemdata^.TempName, ArgString);
        Strings.Append (" ", ArgString);
        Strings.Append (recipients, ArgString);
        rc := LENGTH(ArgString);
        IF rc <= MAX(FilenameIndex) THEN
            ArgString[rc] := Nul;  INC(rc);
        END (*IF*);
        IF rc <= MAX(FilenameIndex) THEN
            ArgString[rc] := Nul;
        END (*IF*);
        pArg := ADR(ArgString);
        pEnv := ADR(EnvString);

        (*WriteString ("Calling DosExecPgm");  WriteLn;*)
        rc := OS2.DosExecPgm (FailureObjectName, ONLength, OS2.EXEC_SYNC,
                              ADR(pArg), ADR(pEnv), ExitStatus, "CMD.EXE");
        (*
        WriteString ("Returned from DosExecPgm");  WriteLn;
        WriteString ("Back from DosExecPgm, result code is ");
        WriteCard (rc);  WriteLn;
        WriteString ("FailureObjectName is ");  WriteString (FailureObjectName);
        WriteLn;
        WriteString ("codeTerminate is ");  WriteCard (ExitStatus.codeTerminate);
        WriteLn;
        WriteString ("codeResult is ");  WriteCard (ExitStatus.codeResult);
        WriteLn;
        *)
        IF rc = 0 THEN
            result := ExitStatus.codeResult;
            IF result > 3 THEN
                result := 0;
            END (*IF*);
        ELSE
            result := 0;
        END (*IF*);
        IF result = 1 THEN
            RebuildRecipientList (itemdata, recipients);
        END (*IF*);
        FileSys.Remove (recipients, dummy);
        RegainScreen;
        RETURN result;
    END RunFilter;

(************************************************************************)

PROCEDURE DistributeMessage (itemdata: ItemDescriptor);

    (* This procedure can be called after AcceptMessage has read the    *)
    (* whole message.  Now we put it into the local mailboxes, and/or   *)
    (* relay it, depending on the recipients.                           *)

    VAR LocalRL: RecipientList;  dummy: BOOLEAN;
        current: Recipient;
        DestName: FilenameString;

    BEGIN
        LocalRL := itemdata^.LocalRecipients;
        itemdata^.LocalRecipients := NIL;

        (* Send off the copies to be forwarded. *)

        WITH itemdata^ DO
            SendRelayMail (TempName, returnpath,
                               RelayRecipients, offset, LogID);
        END (*WITH*);

        IF LocalRL = NIL THEN

            (* If no local recipients, delete the temporary file. *)

            FileSys.Remove (itemdata^.TempName, dummy);
            itemdata^.TempName[0] := Nul;

        ELSE
            (* Copy the file to the mailboxes of all local  *)
            (* recipients except the first.                 *)

            current := LocalRL^.next;
            WHILE current <> NIL DO
                IF NOT current^.Skip THEN
                    MakeNewFilename (current^.DirName, ".MSG", DestName);
                    LogTransaction (itemdata^.LogID, DestName);
                    OS2.DosCopy (itemdata^.TempName, DestName, 0);
                END (*IF*);
                current := current^.next;
            END (*WHILE*);

            (* Finally, rename the original so that it's no   *)
            (* longer a temporary file.                       *)

            MakeNewFilename (LocalRL^.DirName, ".MSG", DestName);
            LogTransaction (itemdata^.LogID, DestName);
            FileSys.Rename (itemdata^.TempName, DestName, dummy);
            itemdata^.TempName[0] := Nul;
        END (*IF*);

        DeleteRelayList (itemdata^.RelayRecipients);
        DeleteRecipientList (LocalRL);

    END DistributeMessage;

(************************************************************************)
(*                        MODULE INITIALISATION                         *)
(************************************************************************)

VAR hini: OS2.HINI;

BEGIN
    CreateLock (LogFileLock);
    LogSMTPItems := FALSE;
    hini := OpenINIFile ("weasel.ini");
    IF hini <> OS2.NULLHANDLE THEN
        IF NOT INIGetString (hini, "$SYS", "FilterProg", FilterProg) THEN
            FilterProg := "";
        END (*IF*);
        WHILE FilterProg[0] = ' ' DO
            Strings.Delete (FilterProg, 0, 1);
        END (*WHILE*);
        IF NOT INIGetString (hini, "$SYS", "MailRoot", MailRoot) THEN
            MailRoot := "\MPTN\ETC\MAIL\";
        END (*IF*);
        ForwardDirName := MailRoot;
        Strings.Append ("Forward\", ForwardDirName);
        IF NOT INIGet (hini, "$SYS", "UName", NextName) THEN
            NextName := "00000000";
        END (*IF*);
        EVAL (INIGet (hini, "$SYS", "LogSMTPItems", LogSMTPItems));
        OS2.PrfCloseProfile (hini);
    END (*IF*);
    CreateLock (NextNameLock);
FINALLY
    hini := OpenINIFile ("weasel.ini");
    IF hini <> OS2.NULLHANDLE THEN
        Obtain (NextNameLock);
        INIPut (hini, "$SYS", "UName", NextName);
        Release (NextNameLock);
        OS2.PrfCloseProfile (hini);
    END (*IF*);
END SMTPData.

