/*
 *  Name:              nslook.c
 *  Author:            nickel@cs.tu-berlin.de
 *  Version:           1.6
 *  State:             saved
 *  Last modification: Sun Mar 15 22:48:26 1992 by nickel@cs.tu-berlin.de
 */

/*
 * nslook.c  - simple Domain Name Service lookup. This is realized
 * just as a front end to gethostbyname() and gethostbyaddr().
 */

#include <sys/types.h>
#include <types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>

/* exit codes: */
#define USAGE_ERROR 15		/* distinguish this from possible */
				/* values in h_errno */
#define UNKNOWN_ERROR 16	/* because h_errno is zero sometimes */
				/* even when gethostbyname() returns NULL */

#define GUESS_ADDR_FORMAT 2
#define IP_ADDR_FORMAT 1
#define NAME_ADDR_FORMAT 0

#define SHORT_OUTPUT  0
#define LONG_OUTPUT   1
#define MEDIUM_OUTPUT 2

char *progname ;
int errors = 0 ;		/* number of errors */
int reverse = 0 ;		/* do reverse mapping */
int official = 0 ;		/* look for official name */
int aflg = GUESS_ADDR_FORMAT ;	/* how to interpret arguments */
int output = SHORT_OUTPUT ;	/* output format */
char *input = NULL ;		/* file to take input from */


extern struct hostent *gethostbyname(), *gethostbyaddr() ;

main(argc, argv)
int argc ;
char **argv ;
{
    extern int optind ;		/* from getopt */
    extern char *optarg ;	/* argument to option, set by getopt(3) */
    int opt ;			/* option character */
    char *cp, *strrchr() ;
    FILE *in ;
    char buffer[2048] ;		/* hope this won't break */

    progname = argv[0] ;
    if (cp = strrchr(progname, '/')) progname = cp + 1 ;
    while ((opt = getopt(argc, argv, "olmrf:an")) != -1) {
	switch (opt) {
	  case 'l':		/* long */
	    output = LONG_OUTPUT ;
	    break ;
	  case 'a':		/* argument is address */
	    aflg = 1 ;
	    break ;
	  case 'n':		/* argument is name */
	    aflg = 0 ;
	    break ;
	  case 'm':		/* medium output format */
	    output = MEDIUM_OUTPUT ;
	    break ;
	  case 'f':		/* read arguments from file or stdin */
	    input = optarg ;
	    break ;
	  case 'r':		/* normal and reverse mapping */
	    reverse = 1 ;
	    break ;
	  case 'o':		/* official name */
	    official = 1 ;
	    break ;
	  case '?':		/* give usage message */
	    errors++ ;
	}
    }
    if (errors ||
	optind > argc ||
	(input == NULL && optind >= argc) ||
	(official && (output != SHORT_OUTPUT || reverse))) {
	usage() ;
	exit(USAGE_ERROR) ;
    }

    if (input) {
	if (strcmp(input, "-")) {
	    if ((in = fopen(input, "r")) == NULL) {
		perror(input) ;
		exit(errno) ;
	    }
	} else {
	    in = stdin ;
	}
	while (fscanf(in, "%s", buffer) != EOF) {
	    do_lookup (buffer) ;
	}
    } else {
	while (optind < argc) {
	    do_lookup(argv[optind++]) ;
	}
    }
    return errors ;
}


do_lookup (astring)
char *astring;
{
    int local_aflg = aflg ;

    if (local_aflg == GUESS_ADDR_FORMAT) {
	int i;			/* dummy for sscanf() */
	
	/* does the argument look like an address? */
	if (4 == sscanf (astring, "%d.%d.%d.%d", &i, &i, &i, &i)) {
	    local_aflg = IP_ADDR_FORMAT ;
	} else {
	    local_aflg = NAME_ADDR_FORMAT ;
	}
    }
    
    if (local_aflg == IP_ADDR_FORMAT) {
	from_address(astring) ;
    } else {			/* aflg == NAME_ADDR_FORMAT */
	from_name(astring) ;
    }
}


from_address(astring)
char *astring ;
{
    unsigned long addr ;	/* address in host order */
    struct hostent *host ;	/* structure returned by gethostbyaddr() */
    struct hostent *rev ;

    /* parse string into address */
    if ((addr = inet_addr(astring)) == -1) {
	fprintf(stderr, "%s: %s is not a valid address\n", progname, astring) ;
	return ;
    }

    /* try to find hostentry */
    if ((host = gethostbyaddr(&addr, sizeof(addr), AF_INET)) == NULL) {
	gethost_error(progname, astring, 1) ;
    } else {
	if (official) {
	    puts(host->h_name) ;
	} else {
	    switch (output) {
	      case LONG_OUTPUT:
		print_long(host) ;
		if (reverse) {
		    printf("Reverse map:   ") ;
		    if ((rev = gethostbyname(host->h_name)) != NULL) {
			paddress(*host->h_addr_list) ;
		    } else {
			puts("NONE") ;
		    }
		}
		break ;
	      case MEDIUM_OUTPUT:
		print_medium(host) ;
		break ;
	      case SHORT_OUTPUT:
		puts(host->h_name) ;
		break ;
	    }
	}
    }
}

from_name(name)
char *name ;
{
    struct hostent *host ;
    struct hostent *rev;
    struct in_addr addr;

    /* try to find host entry */
    if ((host = gethostbyname(name)) == NULL) {
	gethost_error(progname, name, 0) ;
    } else {
	if (official) {
	    puts(host->h_name) ;
	} else {
	    switch (output) {
	      case LONG_OUTPUT:
		print_long(host) ;
		if (reverse) {
		    bcopy(host->h_addr, &addr, host->h_length);
		    rev = gethostbyaddr(&addr, host->h_length,
					host->h_addrtype);
		    printf("Reverse map:   %s\n", rev ? rev->h_name : "NONE");
		}
		break ;
	      case MEDIUM_OUTPUT:
		print_medium(host) ;
		break ;
	      case SHORT_OUTPUT:
		paddress(*host->h_addr_list) ;
		break ;
	    }
	}
    }
}

/* print host entry in long form */
print_long(host)
struct hostent *host ;
{
    char **list;

    printf("Official name: %s\n", host->h_name) ;
    for (list = host->h_aliases; *list; list++) {
	printf("Alias:         %s\n", *list) ;
    }
    for (list = host->h_addr_list; *list; list++) {
	printf("Address:       ") ;
	paddress(*list) ;
    }
}

print_medium(host)
struct hostent *host ;
{
    printf("%-32s  ", host->h_name) ;
    if (*host->h_addr_list) {
	paddress (*host->h_addr_list) ;
    } else {
	printf("???\n") ;
    }
}

/* print an internet address in network order with terminating newline */
/* (fixed for 16-bit ints by sct@po.cwru.edu, 9/26/91.) */
paddress(paddr)
unsigned long *paddr ;
{
    unsigned long horder ;

    horder = ntohl(*paddr) ;
    printf("%u.%u.%u.%u\n", (unsigned) (horder >> 24),
	   (unsigned) ((horder >> 16) & 0xff),
	   (unsigned) ((horder >> 8) & 0xff), (unsigned) (horder & 0xff));
}


usage()
{
    fprintf(stderr, "\
Usage: %s [-lmr|-o] [-n] hostname ...\n\
       %s [-lm|-o]  [-a] IP-address ...\n\
       %s [-lmran|-o] -f file\n",
	    progname, progname, progname) ;
}

/* examine error condition */
gethost_error(s, name, addr_p)
char *s, *name ;
int addr_p;
{
    extern int h_errno ;	/* error code from gethostbyxxxx() */

    if (s && *s) {
	fprintf(stderr, "%s: ", s) ;
    }

    switch (h_errno) {
      case HOST_NOT_FOUND:
	fprintf(stderr, "%s %s\n",
		(addr_p ? "unknown address" : "unknown host"), name) ;
	break ;
      case TRY_AGAIN:
	fprintf(stderr, "temporary error -- try again later\n") ;
	break ;
      case NO_RECOVERY:
	fprintf(stderr, "unrecoverable error\n") ;
	break ;
#ifdef NO_DATA
      case NO_DATA:
#else
#ifdef NO_ADDRESS
      case NO_ADDRESS:
#else
if neither NO_ADDRESS nor NO_DATA is defined, you are in trouble.
#endif /* NO_ADDRESS */
#endif /* NO_DATA  */
	fprintf(stderr, "no IP address for %s\n", name) ;
	break ;
      default:
	fprintf(stderr, "unknown error number %d\n", h_errno) ;
    }
    errors++ ;
}
