/* gate.c - simple socket gateway
 *
 * Author:  Kai Uwe Rommel <rommel@ars.de>
 * Created: Sun Sep 24 1995
 */
 
static char *rcsid =
"$Id: gate.c,v 1.1 1995/09/24 18:00:42 rommel Exp rommel $";
static char *rcsrev = "$Revision: 1.1 $";

/*
 *
 * $Log: gate.c,v $
 * Revision 1.1  1995/09/24 18:00:42  rommel
 * Initial revision
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <io.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <process.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>

#define BSD_SELECT
#include <types.h>
#include <sys/select.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#define INCL_DOS
#include <os2.h>

#include "getopt.h"

typedef struct
{
  int client;
  struct in_addr server;
  int port;
}
connection;

char localhost[256], *domain;

char *real_server, *int_server;
struct in_addr real_addr, int_addr;
int real_port, int_port;

char *port_name = "nntp";
int port_nr;

int gate;
int debug;
int private = 1;

char *logfile;

int lprintf(char *format, ...)
{
  va_list argptr;
  char buffer[1024];
  int count;
  char filename[256], datetime[32];
  FILE *log;
  time_t now;
  struct tm *tm;

  time(&now);
  tm = localtime(&now);
  count = strftime(buffer, sizeof(buffer), "%m/%d/%y-%H:%M:%S ", tm);

  va_start(argptr, format);
  count += vsprintf(buffer + count, format, argptr);
  va_end(argptr);

  if (logfile != NULL && (log = fopen(logfile, "a")) != NULL)
  {
    fprintf(log, "%s\n", buffer);
    fclose(log);
  }

  putc('\r', stdout);
  fputs(buffer, stdout);
  putc('\n', stdout);
  fflush(stdout);
 
  return count;
}

void handle_client(void *arg)
{
  struct sockaddr_in sa_server;
  int client, server;
  char buffer[4096];

  client = ((connection *) arg) -> client;

  sa_server.sin_family = AF_INET;
  sa_server.sin_port = ((connection *) arg) -> port;
  sa_server.sin_addr = ((connection *) arg) -> server;

  free(arg);

  if ((server = socket(PF_INET, SOCK_STREAM, 0)) < 0)
  {
    psock_errno("socket()");
    soclose(client);
    return;
  }

  if (connect(server, (struct sockaddr *) &sa_server, sizeof(sa_server)) < 0)
  {
    psock_errno("connect()");
    soclose(server);
    soclose(client);
    return;
  }

  for (;;)
  {
    struct timeval tv;
    fd_set fds;
    int rc, bytes;
    
    FD_ZERO(&fds);
    FD_SET(client, &fds);
    FD_SET(server, &fds);
    tv.tv_sec  = 60;
    tv.tv_usec = 0;

    if ((rc = select(FD_SETSIZE, &fds, 0, 0, &tv)) < 0)
    {
      psock_errno("select()");
      break;
    }
    else if (rc == 0)
      continue;

    if (FD_ISSET(server, &fds))
    {
      if ((bytes = recv(server, buffer, sizeof(buffer), 0)) < 0)
      {
	psock_errno("recv()");
	break;
      }
      if (bytes == 0)
	break;
      if (send(client, buffer, bytes, 0) < 0)
      {
	psock_errno("send()");
	break;
      }
      if (debug & 1)
	write(1, buffer, bytes);
    }
    else if (FD_ISSET(client, &fds))
    {
      if ((bytes = recv(client, buffer, sizeof(buffer), 0)) < 0)
      {
	psock_errno("recv()");
	break;
      }
      if (bytes == 0)
	break;
      if (send(server, buffer, bytes, 0) < 0)
      {
	psock_errno("send()");
	break;
      }
      if (debug & 2)
	write(1, buffer, bytes);
    }
  }

  soclose(client);
  soclose(server);

  lprintf("closed %d", client);
}

void wait_for_requests(void)
{
  struct sockaddr_in sa_server, sa_client;
  int client, length;
  
  if ((gate = socket(PF_INET, SOCK_STREAM, 0)) < 0)
  {
    psock_errno("socket()");
    return;
  }

  sa_server.sin_family = AF_INET;
  sa_server.sin_port = port_nr;
  sa_server.sin_addr.s_addr = INADDR_ANY;

  if (bind(gate, (struct sockaddr *) &sa_server, sizeof(sa_server)) < 0)
  {
    psock_errno("bind()");
    return;
  }

  if (listen(gate, 4) != 0)
  {
    psock_errno("listen()");
    return;
  }

  for (;;)
  {
    struct timeval tv;
    fd_set fds;
    int rc;
    struct hostent *host;
    connection *newconn;
    
    FD_ZERO(&fds);
    FD_SET(gate, &fds);
    tv.tv_sec  = 60;
    tv.tv_usec = 0;

    if ((rc = select(FD_SETSIZE, &fds, 0, 0, &tv)) < 0)
    {
      psock_errno("select()");
      soclose(gate);
      return;
    }

    if (rc == 0 || FD_ISSET(gate, &fds) == 0)
      continue;

    length = sizeof(sa_client);
    if ((client = accept(gate, (struct sockaddr *) &sa_client, &length)) == -1)
      continue;

    if ((host = gethostbyaddr((char *) &sa_client.sin_addr, 
			      sizeof(sa_client.sin_addr), AF_INET)) == NULL)
    {
      lprintf("refused on %d: %s (unknown)", 
	      port_nr, inet_ntoa(sa_client.sin_addr));
      soclose(client);
    }
    else
      if (stricmp(strchr(host->h_name, '.'), domain) != 0)
      {
	if (int_server && sa_client.sin_addr.s_addr == real_addr.s_addr)
	{
	  lprintf("accepted %d on %d: %s (external) to %s:%d", client, 
		  ntohs(port_nr), host->h_name, int_server, ntohs(int_port));

	  newconn = (connection *) malloc(sizeof(connection));
	  newconn -> client = client;
	  newconn -> server = int_addr;
	  newconn -> port = int_port;

	  _beginthread(handle_client, NULL, 0x10000, (void *) newconn);
	}
	else if (private)
	{
	  lprintf("refused on %d: %s (foreign)", ntohs(port_nr), host->h_name);
	  soclose(client);
	}
	else
	{
	  lprintf("accepted %d on %d: %s (foreign) to %s:%d", client, 
		  ntohs(port_nr), host->h_name, real_server, ntohs(real_port));

	  newconn = (connection *) malloc(sizeof(connection));
	  newconn -> client = client;
	  newconn -> server = real_addr;
	  newconn -> port = real_port;

	  _beginthread(handle_client, NULL, 0x10000, (void *) newconn);
	}
      }
      else
      {
	lprintf("accepted %d on %d: %s (local) to %s:%d", client, 
		ntohs(port_nr), host->h_name, real_server, ntohs(real_port));

	newconn = (connection *) malloc(sizeof(connection));
	newconn -> client = client;
	newconn -> server = real_addr;
	newconn -> port = real_port;

	_beginthread(handle_client, NULL, 0x10000, (void *) newconn);
      }

    fflush(stdout);
  }
}

void handler(int sig)
{
  soclose(gate);
  lprintf("process terminated", localhost);
  exit(0);
}

void main(int argc, char **argv)
{
  struct in_addr addr;
  struct hostent *host;
  struct servent *port;
  char *ptr;
  int option;

  while ((option = getopt(argc, argv, "dnl:")) !=  -1)
    switch (option) 
    {
    case 'd':
      debug++;
      break;
    case 'n':
      private = 0;
      break;
    case 'l':
      logfile = optarg;
      break;
    }

  if (argc - optind < 2)
  {
    printf("\nIP socket circuit level proxy, (C) Kai Uwe Rommel, \n", rcsrev);
    printf("\nUsage: gate [-d ...] [-n] <service> <server[:port]> [<iserver[:port]>]\n"
	   "\n\t-d\t\tincrease debug level"
	   "\n\t-n\t\tdo not check for local host (i.e. it's a public proxy)"
	   "\n\tservice\t\tport number/service name"
	   "\n\teserver\t\thost name of the real/hidden server"
	   "\n\tiserver\t\toptional host name of an internal server to which"
	   "\n\t\t\tconnections from the real server are permitted\n");
    exit(1);
  }

  if (sock_init())
  {
    psock_errno("sock_init()");
    exit(1);
  }

  addr.s_addr = htonl(gethostid());
  if ((host = gethostbyaddr((char *) &addr, sizeof(addr), AF_INET)) == NULL)
  {
    psock_errno("gethostbyaddr()");
    exit(1);
  }
  else
  {
    strcpy(localhost, host->h_name);
    
    if ((domain = strchr(localhost, '.')) == NULL)
    {
      printf("no domain defined on local host (%s)\n", localhost);
      exit(1);
    }
  }

  port_name = argv[optind];

  if (isdigit(port_name[0]))
    port_nr = htons(atoi(port_name));
  else
  {
    if ((port = getservbyname(port_name, "tcp")) == NULL)
    {
      psock_errno("getservbyname()");
      exit(1);
    }

    port_nr = port->s_port;
  }

  real_server = argv[optind + 1];
  ptr = strchr(real_server, ':');

  if (ptr == NULL)
    real_port = port_nr;
  else
  {
    *ptr++ = 0;
    real_port = htons(atoi(ptr));
  }

  if ((host = gethostbyname(real_server)) == NULL)
  {
    psock_errno("gethostbyname()");
    exit(2);
  }
  else
    real_addr = * (struct in_addr *) (host->h_addr);

  if (argc - optind > 2)
  {
    int_server = argv[optind + 2];

    ptr = strchr(int_server, ':');

    if (ptr == NULL)
      int_port = port_nr;
    else
    {
      *ptr++ = 0;
      int_port = htons(atoi(ptr));
    }

    if ((host = gethostbyname(int_server)) == NULL)
    {
      psock_errno("gethostbyname()");
      exit(2);
    }
    else
      int_addr = * (struct in_addr *) (host->h_addr);
  }

  signal(SIGINT, handler);

  lprintf("starting circuit proxy on %d to %s:%d", 
	  ntohs(port_nr), real_server, ntohs(real_port));

  wait_for_requests();
}

/* end of gate.c */
