/*

	ElectrEm (c) 2000 Thomas Harte - an Acorn Electron Emulator

	This is open software, distributed under the GPL 2, see 'Copying' for details

	6502control.cpp :
	=================

	Contains :

		the various interrupt request routines
		the memory mapped device read/write routines
		routine for loading ROMs, and resetting the CPU

	Bibliography :

		the Electron Advanced User Guide
*/
#include "6502.h"
#include "keyboard.h"
#include <zlib.h>
#include <malloc.h>
#include <string.h>

#include "tape.h"
extern C_Tape tape;

#include "wd1770.h"
extern C_WD1770 disk;

#include "elecscreen.h"
extern C_screen elec_screen;

#include "input.h"
extern C_input_api inputdrvr;


/*void C_6502ULA::TriggerNMI(void)
{
	nmi = true;
}

int C_6502ULA::SetNMI(void)
{
	nmi = true;
	return 0;
}

void C_6502ULA::ResetNMI(int id)
{
	nmi = false;
}*/

unsigned __int8 C_6502ULA::o_read(broken_word addr)
{
	unsigned __int8 ret;

	if(addr.b.h < 0xc0)
	{
		if(nowpage == 8 || nowpage == 9)
			return get_keyboard(addr);
		else
		{
			return 0;	//for cheat mode, fill in here
		}
	}
	else
		switch(addr.b.h)
		{
			default :
//				fprintf(out, "device read from %04x - surely some mistake?\n", addr.a);
			break;

			case 0xfc :
				switch(addr.b.l)
				{
					default :
//						fprintf(out, "unknown read from fc page %02x\n", addr.b.l);
					break;

					case 0xc0 :
					case 0xc4 :
					case 0xc5 :
					case 0xc6 :
					case 0xc7 :
					return disk.ReadFromRegister(addr.b.l - 0xc0);

					case 0x72 :
					return 0xff;
				}
			return 0;

			case 0xfd :
//				fprintf(out, "unknown read from fd page %02x\n", addr.b.l);
			return 0;

			case 0xfe :
				addr.b.l &= 0xf;

				switch(addr.b.l)
				{
					default :
//						fprintf(out, "unknown read from fe page %02x\n", addr.b.l);
					break;

					case 0x0 :
						ret = int_status;
						int_status &= ~(POWER_IRQ | DATA_RECEIVE_IRQ | DATA_SEND_IRQ); //clear all 'non-clearable' interrupts
						irqm = false;
						if(!(int_status & (DISPLAY_IRQ | RTC_IRQ | HIGH_TONE_IRQ)))
						{
							irq = false;
							int_status &= ~MASTER_IRQ;
						}
					return ret;

					case 0x4 :
//						fprintf(out, "polling cassette interface\n");
					return tape.tape_data;
				}
			return 0;
		}

	return 0;
}

int C_6502ULA::o_write(broken_word addr, unsigned __int8 value)
{
	int c, rc;
	unsigned __int8 *rptr, t;

	switch(addr.b.h)
	{
		case 0xfc :
			switch(addr.b.l)
			{
				default :
//					fprintf(out, "unknown write [%02x] to fc page %02x\n", value, addr.b.l);
				break;

				case 0xc0 :
				case 0xc1 :
				case 0xc2 :
				case 0xc3 :
				case 0xc4 :
				case 0xc5 :
				case 0xc6 :
				case 0xc7 :
				case 0xc8 :
				case 0xc9 :
				case 0xca :
				case 0xcb :
				case 0xcc :
				case 0xcd :
				case 0xcf :
				case 0xce :
					disk.WriteToRegister(addr.b.l - 0xc0, value);
				break;
			}
		break;

		case 0xfd :
//			fprintf(out, "unknown write to fd page %02x\n", addr.b.l);
		break;

		case 0xfe :
			addr.b.l &= 0xf;
			ulavalues[addr.b.l] = value;

			switch(addr.b.l)
			{
				default :
//					fprintf(out, "unknown write [%02x] to fe page %02x\n", value, addr.b.l);
				break;

				case 0x0 :
					int_control = value;
				break;

				case 0x3 :
					elec_screen.vid_addr.b.h = (elec_screen.vid_addr.b.h & 1) | ((value&63) << 1);
				break;

				case 0x4 :
					tape.tape_data = value;
				break;

				case 0x2 :
					elec_screen.vid_addr.b.h = (elec_screen.vid_addr.b.h & ~1) | (value >> 7);
					elec_screen.vid_addr.b.l = (value&96) << 1;
				break;

				case 0x5 :
					if(value & ~15)
					{
						if(value&HIGH_TONE_CLEAR)
							int_status &= ~HIGH_TONE_IRQ;

						if(value&RTC_CLEAR)
							int_status &= ~RTC_IRQ;

						if(value&DISPLAY_CLEAR)
							int_status &= ~DISPLAY_IRQ;

/*						if(value&0x80)
							fprintf(out, "NMI clear?\n");*/

						if(!(int_status & (DISPLAY_IRQ | RTC_IRQ | HIGH_TONE_IRQ)))
						{
							irq = false;

							if(!irqm)
								int_status &= ~MASTER_IRQ;
						}
					}

//					fprintf(out, "paging %d ... ", value&15);
					t = lastpage;
					lastpage = (~value)&8;

					if(!(value&8))
						value = (value&7) | t;

//					fprintf(out, "%d\n", value&15);

					if((value&15) == 15)
						value = 15;

					nowpage = c = value&15;
					if((rptr = o_rom_ptrs[c]) && rom_present[c])
					{
						rc = 64;
						while(rc--)
						{
							mapped_flags[128+rc] = false;
							o_r_mem[128+rc] = &rptr[rc << 8];
						}
					}
					else
					{
						rc = 64;
						while(rc--)
						{
							mapped_flags[128+rc] = true;
						}
					}
				break;

				case 0x6 : //for counter (tape or sound)
					SendToCounter(value);
				break;

				case 0x7 :
					tape.motor_on = (value&0x40) ? true : false;
					ChangeCounterMode((value >> 1)&3);
					elec_screen.ChangeGFXMode((value >> 3)&7);
					inputdrvr.SetCaps((value&0x80) ? false : true);
				break;

				case 0x8 :
				case 0x9 :
				case 0xa :
				case 0xb :
				case 0xc :
				case 0xd :
				case 0xe :
				case 0xf :
					elec_screen.SetPalette(addr.b.l, value, true);
				break;
			}
		break;
	}

	return 0;
}

void C_6502ULA::SuperReset(void)
{
	Reset();
	int_control = 0;
	int_status = 2;
	p |= I_FLAG;
}

#undef end_table
#undef insert_access
#undef setup_table

#define attempt_load(x)	\
if(str = gzopen(name, "rb"))\
{\
	gzread(str, lptr, 16384);\
\
	gzclose(str);\
	loaded = true;\
}

bool C_6502ULA::LoadRom(int num, char *name) //-1 for OS, checks for .gz automatically
{
	gzFile str;
	char *tmp, *tmpend;
	bool loaded = false;
	unsigned __int8 *lptr;

	if(num > 15)			//out of range
		return true;

	if((num == 8) || (num == 9))	//keyboard
		return true;

	if(num >= 0)
		lptr = o_rom_ptrs[num];
	else
		lptr = o_r_mem[192];

	tmp = (char *)malloc(strlen(name)+4);
	sprintf(tmp, "%s.gz", name);

	attempt_load(tmp)
	else
	attempt_load(name);

	if(!loaded) //last ditch attempt
	{
		sprintf(tmp, "%s", name);
		tmpend = tmp + strlen(tmp) -1;
		while(*tmpend != '.')
			tmpend--;
		tmpend+=2;
		sprintf(tmpend, "gz");

		attempt_load(tmp);
	}

	if(loaded && !(num&~15))
		rom_present[num] = true;

	free(tmp);

	if(num < 0)
	{
		backup_bytes[0] = o_r_mem[0xf1][0xd6];
		backup_bytes[1] = o_r_mem[0xf1][0xd7];
		backup_bytes[2] = o_r_mem[0xf0][0xcc];
		backup_bytes[3] = o_r_mem[0xf0][0xcd];
		backup_bytes[4] = o_r_mem[0xf4][0x20];
		backup_bytes[5] = o_r_mem[0xf4][0x21];
		backup_bytes[6] = o_r_mem[0xf4][0x80];
		backup_bytes[7] = o_r_mem[0xf4][0x81];
		backup_bytes[8] = o_r_mem[0xff][0xa2];
		backup_bytes[9] = o_r_mem[0xff][0xa3];
		backup_bytes[10] = o_r_mem[0xf3][0x20];
		backup_bytes[11] = o_r_mem[0xf3][0x21];
		backup_bytes[12] = o_r_mem[0xf0][0xe8];
		backup_bytes[13] = o_r_mem[0xf0][0xe9];
	}

	return !loaded;
}

#undef attempt_load
void C_6502ULA::RemoveTapeHack(void)
{
	o_r_mem[0xf1][0xd6] = backup_bytes[0];
	o_r_mem[0xf1][0xd7] = backup_bytes[1];
	o_r_mem[0xf0][0xcc] = backup_bytes[2];
	o_r_mem[0xf0][0xcd] = backup_bytes[3];
	o_r_mem[0xf4][0x20] = backup_bytes[4];
	o_r_mem[0xf4][0x21] = backup_bytes[5];
	o_r_mem[0xf4][0x80] = backup_bytes[6];
	o_r_mem[0xf4][0x81] = backup_bytes[7];
	o_r_mem[0xff][0xa2] = backup_bytes[8];
	o_r_mem[0xff][0xa3] = backup_bytes[9];
	o_r_mem[0xf3][0x20] = backup_bytes[10];
	o_r_mem[0xf3][0x21] = backup_bytes[11];
	o_r_mem[0xf0][0xe8] = backup_bytes[12];
	o_r_mem[0xf0][0xe9] = backup_bytes[13];
}

void C_6502ULA::InstallTapeHack(void)
{
/*
	F1D6 (0212) FILE :	0x00	[operations on whole files]			
	F0CC (0214) ARGS :	0x01	[Adjust file arguments]				
	F420 (0216) BGET :	0x02	[Get one byte from an open file]		
	F480 (0218) BPUT :	0x03	[Put one byte to an open file]			
	FFA2 (021A) GBPB :	0x04	[Get/put a block of bytes to/from an open file]	
	F320 (021C) FIND :	0x05	[Open/close a file for byte access]		
	F0E8 (021E) FSC :	0x06	[Filing system control various actions]
*/
	//OSFILE at &f1d6
	o_r_mem[0xf1][0xd6] = TAPE_SERVICE;
	o_r_mem[0xf1][0xd7] = 0x00;

/*	//OSARGS at &f0cc
	o_r_mem[0xf0][0xcc] = TAPE_SERVICE;
	o_r_mem[0xf0][0xcd] = 0x01;

	//OSBGET at &f420
	o_r_mem[0xf4][0x20] = TAPE_SERVICE;
	o_r_mem[0xf4][0x21] = 0x02;

	//OSBPUT at &f480
	o_r_mem[0xf4][0x80] = TAPE_SERVICE;
	o_r_mem[0xf4][0x81] = 0x03;

	//OSGBPB at &ffa2
	o_r_mem[0xff][0xa2] = TAPE_SERVICE;
	o_r_mem[0xff][0xa3] = 0x04;

	//OSFIND at &f320
	o_r_mem[0xf3][0x20] = TAPE_SERVICE;
	o_r_mem[0xf3][0x21] = 0x05;*/

	//OSFSC at &f0e8
	o_r_mem[0xf0][0xe8] = TAPE_SERVICE;
	o_r_mem[0xf0][0xe9] = 0x06;
}

bool C_6502ULA::TapeHackPresent(void)
{
	return o_r_mem[0xf1][0xd6] == TAPE_SERVICE;
}

void C_6502ULA::Reset(void)
{
	broken_word addr;

	addr.a = 0xfffc;
	o_r_memory(addr, pc.b.l);
	addr.a++;
	o_r_memory(addr, pc.b.h);

	p |= A_FLAG;

	disk.Reset();
}
