/*

	ElectrEm (c) 2000 Thomas Harte - an Acorn Electron Emulator

	This is open software, distributed under the GPL 2, see 'Copying' for details

	6502norm.cpp :
	==============

	Contains :

		the normal bit size 6502 emulation code
		a small routine for non-class members to access 6502 address space RAM

	Bibliography :

		'64doc' (by John West & Marko Mkel)
		'65C02 Reference Manual' (HTML, author unknown)
		'6502' (by _Bnu)

*/
#include "6502.h"
#include <stdio.h>
#include <stdlib.h>

#include "tape.h"
extern C_Tape tape;

#define set_type(x)					lblock = types[x]; lblock[0].place = style_flags[pc.b.h]; pc.a++; lblock[1].place = style_flags[pc.b.h];
#define set_part(addr, cycle)		lblock[cycle].place = style_flags[addr.b.h];
#define set_2part(addr, cycle)		lblock[cycle].place = lblock[cycle+1].place = style_flags[addr.b.h];
#define set_3part(addr, cycle)		lblock[cycle].place = lblock[cycle+1].place = lblock[cycle+2].place = style_flags[addr.b.h];
#define set_4part(addr, cycle)		lblock[cycle].place = lblock[cycle+1].place = lblock[cycle+2].place = lblock[cycle+3].place = style_flags[addr.b.h];


#define set_immediate()				set_type(IMMEDIATE); o_r_memory(pc, mdr); pc.a++;
#define set_implied()				set_type(IMPLIED);

#define set_absolute_read()			set_type(ABS_READ); o_r_memory(pc, mar.b.l); pc.a++; set_part(pc, 2); o_r_memory(pc, mar.b.h); pc.a++; set_part(mar, 3); o_r_memory(mar, mdr);
#define set_absolute_modify()		set_type(ABS_MODIFY); o_r_memory(pc, mar.b.l); pc.a++; set_part(pc, 2); o_r_memory(pc, mar.b.h); pc.a++; set_3part(mar, 3);
#define set_absolute_write()		set_type(ABS_WRITE); o_r_memory(pc, mar.b.l); pc.a++; set_part(pc, 2); o_r_memory(pc, mar.b.h); pc.a++; set_part(mar, 3);

#define set_zero_read()				set_type(ZERO_READ); o_r_memory(pc, mar.b.l); pc.a++; o_z_r_memory(mar.b.l, mdr);
#define set_zero_modify()			set_type(ZERO_MODIFY); o_r_memory(pc, mar.b.l); pc.a++;
#define set_zero_write()			set_type(ZERO_WRITE); o_r_memory(pc, mar.b.l); pc.a++;

#define set_zero_i_read(v)			set_type(ZIND_READ); o_r_memory(pc, mar.b.l); pc.a++; mar.b.l += v; o_z_r_memory(mar.b.l, mdr);
#define set_zero_i_modify(v)		set_type(ZIND_MODIFY); o_r_memory(pc, mar.b.l); pc.a++; mar.b.l += v;
#define set_zero_i_write(v)			set_type(ZIND_WRITE); o_r_memory(pc, mar.b.l); pc.a++; mar.b.l += v;

//absolute_i_read timing is wrong if boundary is crossed!
#define set_absolute_i_read(v)		set_type(AIND_READ_NCROSS); o_r_memory(pc, mar.b.l); pc.a++; o_r_memory(pc, mar.b.h); pc.a++; mar.a += v; o_r_memory(mar, mdr); 
#define set_absolute_i_modify(v)	set_type(AIND_MODIFY); o_r_memory(pc, mar.b.l); pc.a++; set_part(pc, 2); o_r_memory(pc, mar.b.h); pc.a++; set_4part(mar, 3); mar.a += v; 
#define set_absolute_i_write(v)		set_type(AIND_WRITE); o_r_memory(pc, mar.b.l); pc.a++; set_part(pc, 2); o_r_memory(pc, mar.b.h); pc.a++; set_2part(mar, 3); mar.a += v; 

//only used with x
#define set_b_zero_i_b_read(v)		set_type(INDI_READ); o_r_memory(pc, mdr); pc.a++; mdr += v; o_z_r_memory(mdr, mar.b.l); mdr++; o_z_r_memory(mdr, mar.b.h); o_r_memory(mar, mdr); set_part(mar, 5);
#define set_b_zero_i_b_modify(v)	set_type(INDI_MODIFY); o_r_memory(pc, mdr); pc.a++; mdr += v; o_z_r_memory(mdr, mar.b.l); mdr++; o_z_r_memory(mdr, mar.b.h); set_3part(mar, 5);
#define set_b_zero_i_b_write(v)		set_type(INDI_WRITE); o_r_memory(pc, mdr); pc.a++; mdr += v; o_z_r_memory(mdr, mar.b.l); mdr++; o_z_r_memory(mdr, mar.b.h); set_part(mar, 5);

//only used with y. read is incorrect for timing as above!
#define set_b_zero_b_i_read(v)		set_type(IIND_READ_NCROSS); o_r_memory(pc, mdr); pc.a++; o_z_r_memory(mdr, mar.b.l); mdr++; o_z_r_memory(mdr, mar.b.h); mar.a += v; o_r_memory(mar, mdr);
#define set_b_zero_b_i_modify(v)	set_type(IIND_MODIFY); o_r_memory(pc, mdr); pc.a++; o_z_r_memory(mdr, mar.b.l); mdr++; o_z_r_memory(mdr, mar.b.h); mar.a += v; set_4part(mar, 4);
#define set_b_zero_b_i_write(v)		set_type(IIND_WRITE); o_r_memory(pc, mdr); pc.a++; o_z_r_memory(mdr, mar.b.l); mdr++; o_z_r_memory(mdr, mar.b.h); mar.a += v; set_2part(mar, 4);

#define push_instr(v)				o_s_w_memory(s, v); s--; set_type(PHx);
#define pull_instr(v)				s++; o_s_r_memory(s, v); set_type(PLx);
#define push(v)						o_s_w_memory(s, v); s--;
#define pull(v)						s++; o_s_r_memory(s, v);

#define set_jsr()					set_type(JSR);
#define set_rts()					set_type(RTS);
#define set_brk()					set_type(BRK);
#define set_rti()					set_type(RTI);

#define set_jmp_indirect()			set_type(ABSOLUTE_IND);
#define set_jmp_absolute()			set_type(ABS_JMP);

#define cond_branch(v)				set_type(REL); o_r_memory(pc, temp16.b.l); pc.a++; if(v) {temp16.b.h = (temp16.b.l&128) ? 0xff : 0; pc.a += temp16.a;}

#define set_nz(v)					p = (p&~(N_FLAG | Z_FLAG)) | nz_table[v]

#define adc() 	\
	\
	if(p&D_FLAG)\
	{\
		temp16.b.l = (a&0x0f) + (mdr&0x0f) + (p&C_FLAG);\
		temp8 = a + mdr + (p&C_FLAG);\
\
		p &= ~(N_FLAG | Z_FLAG | C_FLAG | V_FLAG);\
\
		p |= nz_table[temp8]&Z_FLAG;\
\
		temp16.b.l += (temp16.b.l > 9) ? 6 : 0;\
		temp16.b.h = (a >> 4) + (mdr >> 4) + ((temp16.b.l&0x10) ? 1 : 0);\
\
		p |= (temp16.b.h&8) ? N_FLAG : 0;\
		p |= (((a^(temp16.b.h << 4)) & (a^mdr))&0x80) ? V_FLAG : 0;\
\
		temp16.b.h += (temp16.b.h > 9) ? 6 : 0;\
\
		p |= (temp16.b.h & 0x10) ? C_FLAG : 0;\
		a = (temp16.b.l&15) | (temp16.b.h << 4);\
	}\
	else\
	{\
		temp16.a = a + mdr + (p&C_FLAG);\
		p &= ~(N_FLAG | Z_FLAG | C_FLAG | V_FLAG);\
\
		p |= nz_table[temp16.b.l];\
		p |= temp16.b.h ? C_FLAG : 0;\
		p |= (((a^temp16.b.l) & (a^mdr))&0x80) ? 0 : V_FLAG;\
\
		a = temp16.b.l;\
	}

#define sbc()	\
	\
	if(p&D_FLAG)\
	{\
		temp8 = (~p)&C_FLAG;\
		temp16.a = a - mdr - temp8;\
		p &= ~(N_FLAG | Z_FLAG | C_FLAG | V_FLAG);\
\
		p |= nz_table[temp16.b.l];\
		p |= (((a^temp16.b.l) & (a^mdr))&0x80) ? V_FLAG : 0;\
\
		temp16.b.l = (a&0x0f) - (mdr&0x0f) - temp8;\
		if(temp16.b.l&0x10)\
		{\
			temp16.b.l -= 0x6;\
		}\
\
		temp16.b.h = (a >> 4) - (mdr >> 4) - ((temp16.b.l&0x10) ? 1 : 0) ;\
		if(temp16.b.h&0x10)\
		{\
			temp16.b.h -= 0x6;\
		}\
		else\
			p |= C_FLAG;\
\
		a = (temp16.b.l&0x0f) | (temp16.b.h << 4);\
	}\
	else\
	{\
		mdr = ~mdr;\
		temp16.a = a + mdr + (p&C_FLAG);\
		p &= ~(N_FLAG | Z_FLAG | C_FLAG | V_FLAG);\
\
		p |= nz_table[temp16.b.l];\
		p |= temp16.b.h ? C_FLAG : 0;\
		p |= (((a^temp16.b.l) & (a^mdr))&0x80) ? 0 : V_FLAG;\
\
		a = temp16.b.l;\
	}

#define ror(v)	\
temp8 = p << 7;\
p = (p&~(N_FLAG | Z_FLAG | C_FLAG)) | (v & 1);\
v = (v >> 1) | temp8;\
p |= nz_table[v];

#define rol(v)	\
temp8 = p&1;\
p = (p&~(N_FLAG | Z_FLAG | C_FLAG)) | (v >> 7);\
v = (v << 1) | temp8;\
p |= nz_table[v];

#define lsr(v)	\
p = (p&~(N_FLAG | Z_FLAG | C_FLAG)) | (v & 1);\
v >>= 1;\
p |= nz_table[v];

#define asl(v)	\
p = (p&~(N_FLAG | Z_FLAG | C_FLAG)) | (v >> 7);\
v <<= 1;\
p |= nz_table[v];

#define cp(r, v)	\
temp16.a = r - v;\
p = (p&~(N_FLAG | Z_FLAG | C_FLAG)) | nz_table[temp16.b.l];\
p |= temp16.b.h ? 0 : C_FLAG;

#define ora()	a |= mdr; set_nz(a);
#define and()	a &= mdr; set_nz(a);
#define eor()	a ^= mdr; set_nz(a);

void C_6502ULA::GoNormal(void)
{
	unsigned __int8 instr;
	int counter = 0;
	unsigned __int8 temp8;
	broken_word temp16;

	//2,000,000 cycles a second => 40,000 cycles a frame => ~128 cycles a scanline

	//assuming 128 cycles, gives 1,996,800 cycles a second, 99.84% the correct operating speed
	quit = false;

	while(!quit)
	{
		o_r_memory(pc, instr);

		switch(instr)
		{
			default :
				fprintf(out, "Unrecognised : %02x at %04x", instr, pc.a);
				quit = true;
			break;

			case 0x6c : //JMP (addr)
				set_jmp_indirect();

				o_r_memory(pc, mar.b.l); pc.a++;
				o_r_memory(pc, mar.b.h);

				o_r_memory(mar, pc.b.l); mar.b.l++;
				o_r_memory(mar, pc.b.h);
			break;

			case 0x4c : //JMP abs
				set_jmp_absolute();

				o_r_memory(pc, mar.b.l); pc.a++; set_part(pc, 2);
				o_r_memory(pc, mar.b.h); pc.a = mar.a;
			break;

			case 0x00 : //BRK
				set_brk();
				pc.a++;

				push(pc.b.h);
				push(pc.b.l);
				p |= B_FLAG;

				push(p);
				p |= I_FLAG;		//questionable

				temp16.a = 0xfffe;
				o_r_memory(temp16, pc.b.l);
				temp16.a++;
				o_r_memory(temp16, pc.b.h);
			break;

			case 0x20 : //JSR
				set_jsr();
				o_r_memory(pc, mar.b.l);
				pc.a++;

				push(pc.b.h);
				push(pc.b.l);
				set_part(pc, 5);

				o_r_memory(pc, mar.b.h);
				pc.a = mar.a;
			break;

			case 0x40 : //RTI
				set_rti();
				pull(p);
				pull(pc.b.l);
				pull(pc.b.h);
			break;

			case 0x60 : //RTS
				set_rts();
				pull(pc.b.l);
				pull(pc.b.h);
				set_part(pc, 5);
				pc.a++;
			break;

			case 0x08 : //PHP
				p |= B_FLAG;
				push_instr(p);
			break;

			case 0x28 : //PLP
				pull_instr(p);
				p |= A_FLAG;

				if(irq && !(p&I_FLAG))
				{
					push(pc.b.h);
					push(pc.b.l);
					p &= ~B_FLAG;
					push(p);
					p |= I_FLAG;

					temp16.a = 0xfffe;
					o_r_memory(temp16, pc.b.l);
					temp16.a++;
					o_r_memory(temp16, pc.b.h);
				}
			break;

			case 0x48 : //PHA
				push_instr(a);
			break;

			case 0x68 : //PLA
				pull_instr(a);
				set_nz(a);
			break;


			case 0x18 : //CLC
				set_implied();
				p &= ~C_FLAG;
			break;

			case 0x38 : //SEC
				set_implied();
				p |= C_FLAG;
			break;

			case 0x58 : //CLI
				set_implied();
				p &= ~I_FLAG;

				if(irq)
				{
					push(pc.b.h);
					push(pc.b.l);
					p &= ~B_FLAG;
					push(p);
					p |= I_FLAG;

					temp16.a = 0xfffe;
					o_r_memory(temp16, pc.b.l);
					temp16.a++;
					o_r_memory(temp16, pc.b.h);
				}
			break;

			case 0x78 : //SEI
				set_implied();
				p |= I_FLAG;
			break;

			case 0xb8 : //CLV
				set_implied();
				p &= ~V_FLAG;
			break;

			case 0xd8 : //CLD
				set_implied();
				p &= ~D_FLAG;
			break;

			case 0xf8 : //SED
				set_implied();
				p |= D_FLAG;
			break;


			case 0x10 : //BPL
				cond_branch(!(p&N_FLAG));
			break;

			case 0x30 : //BMI
				cond_branch(p&N_FLAG);
			break;

			case 0x50 : //BVC
				cond_branch(!(p&V_FLAG));
			break;

			case 0x70 : //BVS
				cond_branch(p&V_FLAG);
			break;

			case 0x90 : //BCC
				cond_branch(!(p&C_FLAG));
			break;

			case 0xb0 : //BCS
				cond_branch(p&C_FLAG);
			break;

			case 0xd0 : //BNE
				cond_branch(!(p&Z_FLAG));
			break;

			case 0xf0 : //BEQ
				cond_branch(p&Z_FLAG);
			break;


			case 0x09 : //ORA immediate
				set_immediate();
				ora();
			break;

			case 0x05 : //ORA zero
				set_zero_read();
				ora();
			break;

			case 0x15 : //ORA zero, x
				set_zero_i_read(x);
				ora();
			break;

			case 0x0d : //ORA absolute
				set_absolute_read();
				ora();
			break;

			case 0x1d : //ORA absolute, x
				set_absolute_i_read(x);
				ora();
			break;

			case 0x19 : //ORA absolute, y
				set_absolute_i_read(y);
				ora();
			break;

			case 0x01 : //ORA (zero, x)
				set_b_zero_i_b_read(x);
				ora();
			break;

			case 0x11 : //ORA (zero), y
				set_b_zero_b_i_read(y);
				ora();
			break;


			case 0x29 : //AND immediate
				set_immediate();
				and();
			break;

			case 0x25 : //AND zero
				set_zero_read();
				and();
			break;

			case 0x35 : //AND zero,x
				set_zero_i_read(x);
				and();
			break;

			case 0x2d : //AND absolute
				set_absolute_read();
				and();
			break;

			case 0x3d : //AND absolute, x
				set_absolute_i_read(x);
				and();
			break;

			case 0x39 : //AND absolute, y
				set_absolute_i_read(y);
				and();
			break;

			case 0x21 : //AND (zero, x)
				set_b_zero_i_b_read(x);
				and();
			break;

			case 0x31 : //AND (zero), y
				set_b_zero_b_i_read(y);
				and();
			break;


			case 0x49 : //EOR immediate
				set_immediate();
				eor();
			break;

			case 0x45 : //EOR zero
				set_zero_read();
				eor();
			break;

			case 0x55 : //EOR zero, x
				set_zero_i_read(x);
				eor();
			break;

			case 0x4d : //EOR absolute
				set_absolute_read();
				eor();
			break;

			case 0x5d : //EOR absolute, x
				set_absolute_i_read(x);
				eor();
			break;

			case 0x59 : //EOR absolute, y
				set_absolute_i_read(y);
				eor();
			break;

			case 0x41 : //EOR (zero, x)
				set_b_zero_i_b_read(x);
				eor();
			break;

			case 0x51 : //EOR (zero), y
				set_b_zero_b_i_read(y);
				eor();
			break;


			case 0x24 : //BIT zero
				set_zero_read();
				p = (p&~(192 | Z_FLAG)) | (mdr&192) | (nz_table[a&mdr]&Z_FLAG);
			break;

			case 0x2c : //BIT absolute
				set_absolute_read();
				p = (p&~(192 | Z_FLAG)) | (mdr&192) | (nz_table[a&mdr]&Z_FLAG);
			break;


			case 0x0a : //ASL A
				set_implied();
				asl(a);
			break;

			case 0x06 : //ASL zero
				set_zero_modify();
				o_z_r_memory(mar.b.l, mdr);
				asl(mdr);
				o_z_w_memory(mar.b.l, mdr);
			break;

			case 0x16 : //ASL zero, x
				set_zero_i_modify(x);
				o_z_r_memory(mar.b.l, mdr);
				asl(mdr);
				o_z_w_memory(mar.b.l, mdr);
			break;

			case 0x0e : //ASL absolute
				set_absolute_modify();
				o_r_memory(mar, mdr);
				asl(mdr);
				o_w_memory(mar, mdr);
			break;

			case 0x1e : //ASL absolute, x
				set_absolute_i_modify(x);
				o_r_memory(mar, mdr);
				asl(mdr);
				o_w_memory(mar, mdr);
			break;


			case 0x2a : //ROL A
				set_implied();
				rol(a);
			break;

			case 0x26 : //ROL zero
				set_zero_modify();
				o_z_r_memory(mar.b.l, mdr);
				rol(mdr);
				o_z_w_memory(mar.b.l, mdr);
			break;

			case 0x36 : //ROL zero, x
				set_zero_i_modify(x);
				o_z_r_memory(mar.b.l, mdr);
				rol(mdr);
				o_z_w_memory(mar.b.l, mdr);
			break;

			case 0x2e : //ROL absolute
				set_absolute_modify();
				o_r_memory(mar, mdr);
				rol(mdr);
				o_w_memory(mar, mdr);
			break;

			case 0x3e : //ROL absolute,x
				set_absolute_i_modify(x);
				o_r_memory(mar, mdr);
				rol(mdr);
				o_w_memory(mar, mdr);
			break;


			case 0x4a : //LSR A
				set_implied();
				lsr(a);
			break;

			case 0x46 : //LSR zero
				set_zero_modify();
				o_z_r_memory(mar.b.l, mdr);
				lsr(mdr);
				o_z_w_memory(mar.b.l, mdr);
			break;

			case 0x56 : //LSR zero, x
				set_zero_i_modify(x);
				o_z_r_memory(mar.b.l, mdr);
				lsr(mdr);
				o_z_w_memory(mar.b.l, mdr);
			break;

			case 0x4e : //LSR absolute
				set_absolute_modify();
				o_r_memory(mar, mdr);
				lsr(mdr);
				o_w_memory(mar, mdr);
			break;

			case 0x5e : //LSR absolute, x
				set_absolute_i_modify(x);
				o_r_memory(mar, mdr);
				lsr(mdr);
				o_w_memory(mar, mdr);
			break;


			case 0x6a : //ROR A
				set_implied();
				ror(a);
			break;

			case 0x66 : //ROR zero
				set_zero_modify();
				o_z_r_memory(mar.b.l, mdr);
				ror(mdr);
				o_z_w_memory(mar.b.l, mdr);
			break;

			case 0x76 : //ROR zero, x
				set_zero_i_modify(x);
				o_z_r_memory(mar.b.l, mdr);
				ror(mdr);
				o_z_w_memory(mar.b.l, mdr);
			break;

			case 0x6e : //ROR absolute
				set_absolute_modify();
				o_r_memory(mar, mdr);
				ror(mdr);
				o_w_memory(mar, mdr);
			break;

			case 0x7e : //ROR absolute,x
				set_absolute_i_modify(x);
				o_r_memory(mar, mdr);
				ror(mdr);
				o_w_memory(mar, mdr);
			break;


			case 0x69 : //ADC immediate
				set_immediate();
				adc();
			break;

			case 0x65 : //ADC zero
				set_zero_read();
				adc();
			break;

			case 0x75 : //ADC zero, x
				set_zero_i_read(x);
				adc();
			break;

			case 0x6d : //ADC absolute
				set_absolute_read();
				adc();
			break;

			case 0x7d : //ADC absolute,x
				set_absolute_i_read(x);
				adc();
			break;

			case 0x79 : //ADC absolute,y
				set_absolute_i_read(y);
				adc();
			break;

			case 0x61 : //ADC (zero,x)
				set_b_zero_i_b_read(x);
				adc();
			break;

			case 0x71 : //ADC (zero),y
				set_b_zero_b_i_read(y);
				adc();
			break;



			case 0xe9 : //SBC immediate
				set_immediate();
				sbc();
			break;

			case 0xe5 : //SBC zero
				set_zero_read();
				sbc();
			break;

			case 0xf5 : //SBC zero, x
				set_zero_i_read(x);
				sbc();
			break;

			case 0xed : //SBC absolute
				set_absolute_read();
				sbc();
			break;

			case 0xfd : //SBC absolute,x
				set_absolute_i_read(x);
				sbc();
			break;

			case 0xf9 : //SBC absolute,y
				set_absolute_i_read(y);
				sbc();
			break;

			case 0xe1 : //SBC (zero,x)
				set_b_zero_i_b_read(x);
				sbc();
			break;

			case 0xf1 : //SBC (zero),y
				set_b_zero_b_i_read(y);
				sbc();
			break;


			case 0xc6 : //DEC zero
				set_zero_modify();
				o_z_r_memory(mar.b.l, temp8);
				temp8--; set_nz(temp8);
				o_z_w_memory(mar.b.l, temp8);
			break;

			case 0xd6 : //DEC zero, x
				set_zero_i_modify(x);
				o_z_r_memory(mar.b.l, temp8);
				temp8--; set_nz(temp8);
				o_z_w_memory(mar.b.l, temp8);
			break;

			case 0xce : //DEC absolute
				set_absolute_modify();
				o_r_memory(mar, temp8);
				temp8--; set_nz(temp8);
				o_w_memory(mar, temp8);
			break;

			case 0xde : //DEC absolute, x
				set_absolute_i_modify(x);
				o_r_memory(mar, temp8);
				temp8--; set_nz(temp8);
				o_w_memory(mar, temp8);
			break;


			case 0x85 : //STA zero
				set_zero_write();
				o_z_w_memory(mar.b.l, a);
			break;

			case 0x95 : //STA zero,x
				set_zero_i_write(x);
				o_z_w_memory(mar.b.l, a);
			break;

			case 0x8d : //STA absolute
				set_absolute_write();
				o_w_memory(mar, a);
			break;

			case 0x9d : //STA abs, x
				set_absolute_i_write(x);
				o_w_memory(mar, a);
			break;

			case 0x99 : //STA abs,y
				set_absolute_i_write(y);
				o_w_memory(mar, a);
			break;

			case 0x81 : //STA (zero, x)
				set_b_zero_i_b_write(x);
				o_w_memory(mar, a);
			break;

			case 0x91 : //STA (zero), y
				set_b_zero_b_i_write(y);
				o_w_memory(mar, a);
			break;



			case 0xa9 : //LDA immd
				set_immediate();
				a = mdr; set_nz(a);
			break;

			case 0xa5 : //LDA zero
				set_zero_read();
				a = mdr; set_nz(a);
			break;

			case 0xb5 : //LDA zero,x
				set_zero_i_read(x);
				a = mdr; set_nz(a);
			break;

			case 0xad : //LDA absolute
				set_absolute_read();
				a = mdr; set_nz(a);
			break;

			case 0xbd : //LDA absolute, x
				set_absolute_i_read(x);
				a = mdr; set_nz(a);
			break;

			case 0xb9 : //LDA absolute, y
				set_absolute_i_read(y);
				a = mdr; set_nz(a);
			break;

			case 0xa1 : //LDA (zero, x)
				set_b_zero_i_b_read(x);
				a = mdr; set_nz(a);
			break;

			case 0xb1 : //LDA (zero), y
				set_b_zero_b_i_read(y);
				a = mdr; set_nz(a);
			break;


			case 0xa2 : //LDX immd
				set_immediate();
				x = mdr; set_nz(x);
			break;

			case 0xa6 : //LDX zero
				set_zero_read();
				x = mdr; set_nz(x);
			break;

			case 0xb6 : //LDX zero, y
				set_zero_i_read(y);
				x = mdr; set_nz(x);
			break;

			case 0xae : //LDX absolute
				set_absolute_read();
				x = mdr; set_nz(x);
			break;

			case 0xbe : //LDX absolute, y
				set_absolute_i_read(y);
				x = mdr; set_nz(x);
			break;


			case 0xa0 : //LDY immd
				set_immediate();
				y = mdr; set_nz(y);
			break;

			case 0xa4 : //LDY zero
				set_zero_read();
				y = mdr; set_nz(y);
			break;

			case 0xb4 : //LDY zero,x
				set_zero_i_read(x);
				y = mdr; set_nz(y);
			break;

			case 0xac : //LDY absolute
				set_absolute_read();
				y = mdr; set_nz(y);
			break;

			case 0xbc : //LDY absolute, x
				set_absolute_i_read(x);
				y = mdr; set_nz(y);
			break;


			case 0xe6 : //INC zero
				set_zero_modify();
				o_z_r_memory(mar.b.l, temp8);
				temp8++; set_nz(temp8);
				o_z_w_memory(mar.b.l, temp8);
			break;

			case 0xf6 : //INC zero, x
				set_zero_i_modify(x);
				o_z_r_memory(mar.b.l, temp8);
				temp8++; set_nz(temp8);
				o_z_w_memory(mar.b.l, temp8);
			break;

			case 0xee : //INC absolute
				set_absolute_modify();
				o_r_memory(mar, temp8);
				temp8++; set_nz(temp8);
				o_w_memory(mar, temp8);
			break;

			case 0xfe : //INC absolute, x
				set_absolute_i_modify(x);
				o_r_memory(mar, temp8);
				temp8++; set_nz(temp8);
				o_w_memory(mar, temp8);
			break;


			case 0xca : //DEX
				set_implied();
				x--; set_nz(x);
			break;

			case 0xe8 : //INX
				set_implied();
				x++; set_nz(x);
			break;

			case 0x88 : //DEY
				set_implied();
				y--; set_nz(y);
			break;

			case 0xc8 : //INY
				set_implied();
				y++; set_nz(y);
			break;


			case 0x9a : //TXS
				set_implied();
				s = x;
			break;

			case 0xba : //TSX
				set_implied();
				x = s; set_nz(x);
			break;

			case 0x98 : //TYA
				set_implied();
				a = y; set_nz(a);
			break;

			case 0x8a : //TXA
				set_implied();
				a = x; set_nz(a);
			break;

			case 0xa8 : //TAY
				set_implied();
				y = a; set_nz(y);
			break;

			case 0xaa : //TAX
				set_implied();
				x = a; set_nz(x);
			break;


			case 0x84 : //STY zero
				set_zero_write();
				o_z_w_memory(mar.b.l, y);
			break;

			case 0x94 : //STY zero,x
				set_zero_i_write(x);
				o_z_w_memory(mar.b.l, y);
			break;

			case 0x8c : //STY absolute
				set_absolute_write();
				o_w_memory(mar, y);
			break;


			case 0x86 : //STX zero
				set_zero_write();
				o_z_w_memory(mar.b.l, x);
			break;

			case 0x96 : //STX zero.y
				set_zero_i_write(y);
				o_z_w_memory(mar.b.l, x);
			break;

			case 0x8e : //STX absolute
				set_absolute_write();
				o_w_memory(mar, x);
			break;


			case 0xc0 : //CPY immediate
				set_immediate();
				cp(y, mdr);
			break;

			case 0xc4 : //CPY zero
				set_zero_read();
				cp(y, mdr);
			break;

			case 0xcc : //CPY absolute
				set_absolute_read();
				cp(y, mdr);
			break;


			case 0xe0 : //CPX immediate
				set_immediate();
				cp(x, mdr);
			break;

			case 0xe4 : //CPX zero
				set_zero_read();
				cp(x, mdr);
			break;

			case 0xec : //CPX absolute
				set_absolute_read();
				cp(x, mdr);
			break;


			case 0xc9 : //CMP immediate
				set_immediate();
				cp(a, mdr);
			break;

			case 0xc5 : //CMP zero
				set_zero_read();
				cp(a, mdr);
			break;

			case 0xd5 : //CMP zero, x
				set_zero_i_read(x);
				cp(a, mdr);
			break;

			case 0xcd : //CMP absolute
				set_absolute_read();
				cp(a, mdr);
			break;

			case 0xdd : //CMP absolute,x
				set_absolute_i_read(x);
				cp(a, mdr);
			break;

			case 0xd9 : //CMP absolute,y
				set_absolute_i_read(y);
				cp(a, mdr);
			break;

			case 0xc1 : //CMP (zero, x)
				set_b_zero_i_b_read(x);
				cp(a, mdr);
			break;

			case 0xd1 : //CMP (zero), y
				set_b_zero_b_i_read(y);
				cp(a, mdr);
			break;

			case 0xea : //NOP
				set_implied();
			break;

			case TAPE_SERVICE :
				set_immediate();
				tape.HandleService(a, x, y, mdr);

				set_rts();
				pull(pc.b.l);
				pull(pc.b.h);
				set_part(pc, 5);
				pc.a++;
			break;
		}

		while(lblock->place != FINISH)
		{
			counter++;
			lblock++;

			if((lblock->place == RAM) && !ram_available[counter])
				lblock--;
		}

		if(counter >= 128)
		{
			counter -= 128;
			external_keepalive();
		}

/*		if(nmi)
		{
			push(pc.b.h);
			push(pc.b.l);
			p &= ~B_FLAG;
			push(p);
//			p |= I_FLAG;

			nmi = false;				//surely not?
			temp16.a = 0xfffa;
			o_r_memory(temp16, pc.b.l);
			temp16.a++;
			o_r_memory(temp16, pc.b.h);
		}
		else
		{*/

/*			if((!(p&I_FLAG)) && (irq || irqm))
			{
				push(pc.b.h);
				push(pc.b.l);
				p &= ~B_FLAG;
				push(p);
				p |= I_FLAG;

				temp16.a = 0xfffe;
				o_r_memory(temp16, pc.b.l);
				temp16.a++;
				o_r_memory(temp16, pc.b.h);

				irqm = false;
			}
		}*/
	}
}

void C_6502ULA::DoPush(unsigned __int16 addr)
{
	broken_word tmp;
	tmp.a = addr;

	push(tmp.b.h);
	push(tmp.b.l);
}

void C_6502ULA::SetIRQ(unsigned __int8 intbit)
{
	broken_word temp16;
	if(int_control&intbit)
	{
		int_status |= intbit | MASTER_IRQ;
		irq = true;

		if(!(p&I_FLAG))
		{
			push(pc.b.h);
			push(pc.b.l);
			p &= ~B_FLAG;
			push(p);
			p |= I_FLAG;

			temp16.a = 0xfffe;
			o_r_memory(temp16, pc.b.l);
			temp16.a++;
			o_r_memory(temp16, pc.b.h);
		}
	}
}

void C_6502ULA::TriggerIRQ(unsigned __int8 intbit)
{
	broken_word temp16;

	if(int_control&intbit)
	{
		int_status |= intbit | MASTER_IRQ;
//		irqm = true;
		if(!(p&I_FLAG))
		{
			push(pc.b.h);
			push(pc.b.l);
			p &= ~B_FLAG;
			push(p);
			p |= I_FLAG;

			temp16.a = 0xfffe;
			o_r_memory(temp16, pc.b.l);
			temp16.a++;
			o_r_memory(temp16, pc.b.h);
		}
	}
}

#undef eor
#undef and
#undef ora

#undef cp
#undef asl
#undef lsr
#undef rol
#undef ror
#undef sbc
#undef adc
#undef add

#undef set_type
#undef set_part
#undef set_3part

#undef set_immediate
#undef set_implied

#undef set_absolute_read
#undef set_absolute_modify
#undef set_absolute_write

#undef set_zero_read
#undef set_zero_modify
#undef set_zero_write

#undef set_zero_i_read
#undef set_zero_i_modify
#undef set_zero_i_write

#undef set_absolute_i_read
#undef set_absolute_i_modify
#undef set_absolute_i_write

#undef set_b_zero_i_b_read
#undef set_b_zero_i_b_modify
#undef set_b_zero_i_b_write

#undef set_b_zero_b_i_read
#undef set_b_zero_b_i_modify
#undef set_b_zero_b_i_write

#undef push_instr
#undef pull_instr
#undef push
#undef pull

#undef set_jsr
#undef set_rts
#undef set_brk
#undef set_rti

#undef set_jmp_indirect
#undef set_jmp_absolute
#undef cond_branch
#undef set_nz

unsigned __int8 * C_6502ULA::GetNormalPtr(unsigned __int16 addr)
{
	broken_word tmp;

	tmp.a = addr;
	return &o_r_mem[tmp.b.h][tmp.b.l];
}

