/*

  ElectrEm (c) 2000 Thomas Harte - an Acorn Electron Emulator

  This is open software, distributed under the GPL 2, see 'Copying' for details

*/
#include "sound.h"
#include <stdio.h>

#ifdef TARGET_SVGALIB
#include <sys/ioctl.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/soundcard.h>
#endif

C_sound_api::C_sound_api(void)
{
	#ifdef TARGET_WIN32
	lpds = NULL;
	dsbuffer = NULL;
	#endif

	#ifdef TARGET_ALLEGRO
	square = NULL;
	channum = -1;
	#endif
}

C_sound_api::~C_sound_api(void)
{
	#ifdef TARGET_WIN32
	if(dsbuffer)
	{
		dsbuffer->Release();
		dsbuffer = NULL;
	}

	if(lpds)
	{
		lpds->Release();
		lpds = NULL;
	}
	#endif

	#ifdef TARGET_ALLEGRO
	if(channum >= 0)
	{
		deallocate_voice(channum);
		channum = -1;
	}

	if(square)
	{
		destroy_sample(square);
		square = NULL;
	}
	#endif
}

#ifdef TARGET_WIN32
bool C_sound_api::Setup(void)
{
	DSBUFFERDESC dsbd;
	unsigned __int8 *snd_buffer, *audio_ptr_1=NULL, *audio_ptr_2=NULL;
	DWORD audio_length_1=0, audio_length_2=0;
	int c;
	WAVEFORMATEX pcmwf;

	if(FAILED(DirectSoundCreate(NULL, &lpds, NULL)))
		return true;

	if(FAILED(lpds->SetCooperativeLevel(winfo.hWnd, DSSCL_PRIORITY)))
		return true;

	memset(&pcmwf, 0, sizeof(WAVEFORMATEX));
	pcmwf.wFormatTag = WAVE_FORMAT_PCM;
	pcmwf.nChannels = 1;
	pcmwf.nSamplesPerSec = 11025;
	pcmwf.nBlockAlign = 1;
	pcmwf.nAvgBytesPerSec = pcmwf.nSamplesPerSec * pcmwf.nBlockAlign;
	pcmwf.wBitsPerSample = 8;
	pcmwf.cbSize = 0;

	dsbd.dwSize = sizeof(DSBUFFERDESC);
	dsbd.dwFlags = DSBCAPS_CTRLFREQUENCY | DSBCAPS_CTRLVOLUME | DSBCAPS_STATIC;
	dsbd.dwBufferBytes = 2048; //I don't know why
	dsbd.lpwfxFormat = &pcmwf;
	dsbd.dwReserved = 0;

	snd_buffer = (unsigned __int8 *)malloc(dsbd.dwBufferBytes);
	c = dsbd.dwBufferBytes;
	while(c--)
	{
		snd_buffer[c] = (c&8) ? 255 : 0;
	}

	if(FAILED(lpds->CreateSoundBuffer(&dsbd, &dsbuffer, NULL)))
		return true;

	if(FAILED(dsbuffer->Lock(0, dsbd.dwBufferBytes, (void **)&audio_ptr_1, &audio_length_1, (void **)&audio_ptr_2, &audio_length_2, DSBLOCK_FROMWRITECURSOR)))
		return true;

	memcpy(audio_ptr_1, snd_buffer, audio_length_1);
	memcpy(audio_ptr_2, (snd_buffer+audio_length_1), audio_length_2);

	if(FAILED(dsbuffer->Unlock(audio_ptr_1, audio_length_1, audio_ptr_2, audio_length_2)))
		return true;

	free(snd_buffer);

	volume = MIN_VOLUME;
	vol_override = false;
	dsbuffer->SetVolume(volume);
	dsbuffer->Play(0, 0, DSBPLAY_LOOPING);		

	return false;
}
#endif

#ifdef TARGET_ALLEGRO
bool C_sound_api::Setup(void)
{
	int c;

	if(install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, NULL) != -1)
	{
		square = create_sample(8, FALSE, 11025, 2048);
		for(c = 0; c < 2048; c++)
			((unsigned char *)square->data)[c] = (c&8)? 255 : 0;

		channum = allocate_voice(square);
		voice_set_playmode(channum, PLAYMODE_LOOP);
		voice_set_volume(channum, 0);
		voice_start(channum);

		return false;
	}
	else
	{
		channum = -1;
		return true;
	}

}
#endif

#ifdef TARGET_SVGALIB
bool C_sound_api::Setup(void)
{
/*	int format, stereo;
	
	sound = false;
	if ((audio_fd = open("/dev/dsp", O_WRONLY, 0)) == -1)
	{
		perror(DEVICE_NAME);
		return true;
	}
	
	format = AFMT_U8;
	if(ioctl(audio_fd, SNDCTL_DSP_SETFMT, &format) == -1)
	{
		perror("SNDCTL_DSP_SETFMT");
		return true;
	}
	
	if(format != AFMT_U8)
	{
		//I must put something decent here!
	}
	
	stereo = 0;
	if(ioctl(audio_fd, SNDCTL_DSP_STEREO, &stereo) == -1)
	{
		perror("SNDCTL_DSP_STEREO");
		return true;
	}
	
	if(stereo)	//only stereo sound supported
	{
		//I must put something here
	}
	
	sound = true;
	return false;*/
}
#endif
void C_sound_api::SetVolume(float vol)
//is this really necessary?
{
	volume = MIN_VOLUME + (int)((MAX_VOLUME-MIN_VOLUME)*vol);
	#ifdef TARGET_WIN32
	if(!vol_override && dsbuffer)
		dsbuffer->SetVolume(volume);
	#endif
	#ifdef TARGET_ALLEGRO
	if(!vol_override && (channum >= 0))
		voice_set_volume(channum, volume);
	#endif
}

void C_sound_api::SetFrequency(int freq)
{
	freq >>= 1;

	#ifdef TARGET_WIN32
//	HRESULT res;
	if(dsbuffer)
	{
		dsbuffer->SetFrequency(freq);
/*		do
		{
			res = dsbuffer->SetFrequency(freq);
		}while(res != DS_OK);*/
	}
	#endif
	#ifdef TARGET_ALLEGRO
	if(channum >= 0)
		voice_set_frequency(channum, freq);
	#endif
	
	#ifdef TARGET_SVGALIB
/*	if(sound)
	{
		if(ioctl(audio_fd, SNDCTL_DSP_SPEED, &freq) == -1)
		{
			perror("SNDCTL_DSP_SPEED");
			sound = false;
		}
	}*/
	#endif
}

void C_sound_api::SetOn(void)
{
	vol_override = false;
	#ifdef TARGET_WIN32
	if(dsbuffer)
		dsbuffer->SetVolume(volume);
	#endif
	#ifdef TARGET_ALLEGRO
	if(channum >= 0)
		voice_set_volume(channum, volume);
	#endif
}

void C_sound_api::SetOff(void)
{
	vol_override = true;
	#ifdef TARGET_WIN32
	if(dsbuffer)
		dsbuffer->SetVolume(MIN_VOLUME);
	#endif
	#ifdef TARGET_ALLEGRO
	if(channum >= 0)
		voice_set_volume(channum, MIN_VOLUME);
	#endif
}