#include "pch.h"
#include "mbQuickstart.h"
#include "mbConfig.h"
#include "mbMisc.h"
#include "mbDiscFormat.h"
#include <vector>
#include <algorithm>

enum {
	id_base=1000,
	id_ok,
	id_games_list,
	id_add,
	id_remove,
};

//////////////////////////////////////////////////////////////////////////
// mbQuickstartResult
//
// Result of a Quickstart operation.
//
// this->file describes the selected file;
// this->allow_save indicates whether 'Save' should be allowed for this disc
// image.
mbQuickstartResult::mbQuickstartResult():
allow_save(false)
{
}

//////////////////////////////////////////////////////////////////////////
// Quickstart Dialog
//
// Lists games, allows user to select one a la MAME.
class QuickstartDialog:
public wxDialog
{
public:
	QuickstartDialog(wxWindow *parent,const std::vector<wxFileName> &dirs);
	~QuickstartDialog();

	void OnAdd(wxCommandEvent &event);
	void OnRemove(wxCommandEvent &event);
	void OnOk(wxEvent &event);
	bool GetResult(mbQuickstartResult *result) const;
	void GetDirs(std::vector<wxFileName> *dirs) const;
protected:
private:
	bool result_valid_;
	mbQuickstartResult result_;
	DECLARE_EVENT_TABLE();
	wxListCtrl *games_list_;
	std::vector<wxFileName> dirs_;

	struct Game {
		Game(const wxFileName &name);

		const wxFileName name;
		const wxString name_str;
		const wxString ext_lc;
		bool is_ds;
		wxFileName side2name;
		wxString display_name;

		//typedef int (wxCALLBACK *wxListCtrlCompare)(long item1, long item2, long sortData);
		static int wxCALLBACK ListCtrlCompare(long i1,long i2,long sort_data);
	private:
		Game(const Game &);
		Game &operator=(const Game &);
	};

	std::vector<Game *> games_;

	void GetAllGamesRecursive(const wxFileName &root);
	void RefreshGames();
	void RefreshGamesList();
	void ResetGames();
	void SetGamesListStyle(int flags);
};

//////////////////////////////////////////////////////////////////////////
//
QuickstartDialog::Game::Game(const wxFileName &name_in):
name(name_in),
name_str(name_in.GetName()),
ext_lc(name_in.GetExt().Lower()),
is_ds(false),
display_name(name_in.GetName())
{
}


//////////////////////////////////////////////////////////////////////////
// Compares two items in the games list.
//
// Their Data items are Game *.
int wxCALLBACK QuickstartDialog::Game::ListCtrlCompare(long i1,long i2,long) {
	const Game *lhs=reinterpret_cast<const Game *>(i1);
	const Game *rhs=reinterpret_cast<const Game *>(i2);

	wxASSERT(lhs&&rhs);
	return lhs->display_name.CmpNoCase(rhs->display_name);
}

//////////////////////////////////////////////////////////////////////////
//
QuickstartDialog::QuickstartDialog(wxWindow *parent,const std::vector<wxFileName> &dirs):
wxDialog(parent,-1,"Quickstart",wxDefaultPosition,wxDefaultSize,
	wxDEFAULT_DIALOG_STYLE|wxRESIZE_BORDER),
result_valid_(false),
games_list_(0),
dirs_(dirs)
{
	wxStaticBox *static_box=new wxStaticBox(this,-1,"Available &games");
	wxStaticBoxSizer *game_col=new wxStaticBoxSizer(static_box,wxVERTICAL);
	wxBoxSizer *both_col=new wxBoxSizer(wxVERTICAL);
	wxBoxSizer *buttons_row=new wxBoxSizer(wxHORIZONTAL);
	wxBoxSizer *game_buttons_row=new wxBoxSizer(wxHORIZONTAL);
	static const int border_size=5;
	
	//list and statics
	games_list_=new wxListCtrl(this,id_games_list,wxDefaultPosition,wxDefaultSize,
		wxLC_LIST|wxLC_SINGLE_SEL);//SMALL_ICON);
	games_list_->SetColumnWidth(-1,wxLIST_AUTOSIZE);
	game_col->Add(games_list_,1,wxEXPAND|wxALL,border_size);
	
	game_buttons_row->Add(new wxButton(this,id_add,"&Add dir..."),0,wxALIGN_RIGHT|wxALL,border_size);
	game_buttons_row->Add(new wxButton(this,id_remove,"&Remove dir..."),0,wxALIGN_RIGHT|wxALL,border_size);
	
	game_col->Add(game_buttons_row,0,wxALIGN_RIGHT|wxLEFT|wxRIGHT,border_size);
	
	//buttons
	wxButton *ok=new wxButton(this,id_ok,"OK");
	ok->SetDefault();
	wxButton *cancel=new wxButton(this,wxID_CANCEL,"Cancel");
	buttons_row->Add(ok,0,wxALIGN_RIGHT|wxALL,border_size);
	buttons_row->Add(cancel,0,wxALIGN_RIGHT|wxALL,border_size);
	
	//both_col has game_col on top, buttons_row below
	both_col->Add(game_col,1,wxEXPAND|wxALL,border_size);
	both_col->Add(buttons_row,0,wxALIGN_RIGHT|wxALIGN_BOTTOM,border_size);
	
	this->SetSizer(both_col);
	both_col->SetSizeHints(this);

	this->RefreshGames();
}

//////////////////////////////////////////////////////////////////////////
//
QuickstartDialog::~QuickstartDialog() {
	this->ResetGames();
}

//////////////////////////////////////////////////////////////////////////
// Adds to this->games_ all the games found in the dir tree starting at
// 'root'.
#define ISTOPCODE(C) (((C)=='0'?1:0)|((C)=='A'?2:0)|((C)=='a'?4:0))
#define ISBOTCODE(C) (((C)=='1'?1:0)|((C)=='B'?2:0)|((C)=='b'?4:0))
#define ISSEPCODE(C) (((C)=='_'?1:0)|((C)=='-'?2:0))

void QuickstartDialog::GetAllGamesRecursive(const wxFileName &root) {
	std::vector<wxFileName> dirs;
	unsigned i;
	unsigned start;
	
	wxLogDebug("Looking in \"%s\"...",root.GetFullPath().c_str());
	//Files first
	std::vector<Game *> found;
	{
		start=games_.size();
		std::vector<wxString> exts;
		std::vector<wxFileName> files;
		mbDiscFormatGetAllExtensions(&exts);
		mbGetMatchingFileNames(root,"*.*",&files);
		for(i=0;i<files.size();++i) {
			printf("%s\n",files[i].GetFullPath().c_str());
			wxString ext=files[i].GetExt().Lower();
			if(std::binary_search(exts.begin(),exts.end(),ext)) {
				found.push_back(new Game(files[i]));
			} else {
				int brk=0;
			}
		}
	}

	//sides
	std::sort(found.begin(),found.end());
	for(i=0;i<found.size();++i) {
		unsigned j=i+1;
		if(i>=found.size()||j>=found.size()||!found[i]) {
			continue;
		}
		if(found[i]->ext_lc!=found[j]->ext_lc) {
			continue;
		}
		const wxString &iname=found[i]->name_str;
		const wxString &jname=found[j]->name_str;
		unsigned ilen=found[i]->name_str.size();
		unsigned jlen=found[i]->name_str.size();
		if(ilen!=jlen||ilen<3||jlen<3) {
			continue;
		}
		if(ISSEPCODE(iname[ilen-2])==0||
			ISSEPCODE(iname[ilen-2])!=ISSEPCODE(jname[jlen-2]))
		{
			continue;
		}
		if(ISTOPCODE(iname[ilen-1])==0||
			ISTOPCODE(iname[ilen-2])!=ISBOTCODE(jname[jlen-2]))
		{
			continue;
		}

		mbDiscFormat ifmt,jfmt;
		bool iok=mbDiscFormatGetFromFileName(found[i]->name,&ifmt);
		bool jok=mbDiscFormatGetFromFileName(found[j]->name,&jfmt);
		wxASSERT(iok&&jok);
		if(ifmt.sides!=1||jfmt.sides!=1) {
			continue;
		}
		
		//found[i] is side 0
		//found[j] is side 2
		//remove found[j].

		found[i]->is_ds=true;
		found[i]->side2name=found[j]->name;
		found[i]->display_name+=" (DS)";

		delete found[j];
		found[j]=0;
	}

	//add to list
	for(i=0;i<found.size();++i) {
		if(found[i]) {
			games_.push_back(found[i]);
		}
	}

	//Directories
	mbGetDirectoryNames(root,&dirs);
	for(i=0;i<dirs.size();++i) {
		this->GetAllGamesRecursive(dirs[i]);
	}
}

//////////////////////////////////////////////////////////////////////////
// Resets the list of games.
void QuickstartDialog::ResetGames() {
	for(unsigned i=0;i<games_.size();++i) {
		delete games_[i];
	}
	games_.clear();
}

//////////////////////////////////////////////////////////////////////////
// Refreshes the list of games by calling  this->GetAllGamesRecursive for
// each of the dirs.
void QuickstartDialog::RefreshGames() {
	this->ResetGames();
	for(unsigned i=0;i<dirs_.size();++i) {
		this->GetAllGamesRecursive(dirs_[i]);
	}
	this->RefreshGamesList();
}

//////////////////////////////////////////////////////////////////////////
// Clears the list control, then adds the names of all the games to it,
// then sorts the list.
void QuickstartDialog::RefreshGamesList() {
	unsigned i;

	games_list_->ClearAll();
	for(i=0;i<games_.size();++i) {
		long index=games_list_->InsertItem(0,games_[i]->display_name);
		games_list_->SetItemData(index,long(games_[i]));
	}
	games_list_->SortItems(&Game::ListCtrlCompare,0);
}


//////////////////////////////////////////////////////////////////////////
// user clicked 'Add'
void QuickstartDialog::OnAdd(wxCommandEvent &event) {
	wxDirDialog dlg(this);
	if(dlg.ShowModal()==wxID_OK) {
		wxString str=dlg.GetPath();
		wxLogDebug("chose %s\n",str.c_str());
		wxFileName name=wxFileName::DirName(str);
		if(name.IsOk()&&std::find(dirs_.begin(),dirs_.end(),name)==dirs_.end()) {
			dirs_.push_back(name);
		}
		this->RefreshGames();
	}
}

//////////////////////////////////////////////////////////////////////////
// user clicked 'Remove'
//

//cmp function for wxIntArray::Sort
static int CmpInts(int *a,int *b) {
	return *b-*a;
}

void QuickstartDialog::OnRemove(wxCommandEvent &event) {
	wxArrayInt indices;
	wxArrayString strings;
	unsigned i;
	
	for(i=0;i<dirs_.size();++i) {
		strings.Add(dirs_[i].GetFullPath());
	}
	wxGetMultipleChoices(indices,"Select dir(s) to remove","Remove dirs",
		strings,this);
	
	//Remove the indices starting at the greatest, for obvious reasons,
	//regrettably std::vector will delete only by iterator hence the
	//palaver.
	indices.Sort(&CmpInts);
	std::vector<wxFileName>::iterator end;
	end=dirs_.end();
	for(i=indices.GetCount();i;--i) {
		int idx=indices[i-1];
		std::swap(*--end,dirs_[idx]);
	}
	dirs_.erase(end,dirs_.end());
	this->RefreshGames();
}

//////////////////////////////////////////////////////////////////////////
// clicked 'OK'
// double clicked item in list
//
// Fetch selected item, if there is one, and set up 
void QuickstartDialog::OnOk(wxEvent &event) {
	int n=games_list_->GetSelectedItemCount();

	result_valid_=false;

	//If the user double clicks, and there's nothing selected, don't stop.
	if(n==0&&event.GetEventType()==wxEVT_COMMAND_LIST_ITEM_ACTIVATED) {
		return;
	}
	if(n==1) {
		long index=games_list_->GetNextItem(-1,wxLIST_NEXT_ALL,wxLIST_STATE_SELECTED);
		if(index>=0) {
			const Game *game=reinterpret_cast<Game *>(games_list_->GetItemData(index));
			if(game) {
				result_valid_=true;
				result_.allow_save=false;
				result_.drive0=game->name;
				if(!game->is_ds) {
					result_.drive2.Clear();
				} else {
					result_.drive2=game->side2name;
				}
			}
		}
	}
	this->EndModal(wxID_OK);
}

//////////////////////////////////////////////////////////////////////////
// Get result
//
// Returns true if there's a result to return. It might be that the user
// clicked 'Ok' without having selected a game -- that counts as no result.
bool QuickstartDialog::GetResult(mbQuickstartResult *result) const {
	if(result_valid_&&result) {
		*result=result_;
	}
	return result_valid_;
}

//////////////////////////////////////////////////////////////////////////
// 
void QuickstartDialog::GetDirs(std::vector<wxFileName> *dirs) const {
	wxASSERT(dirs);
	*dirs=dirs_;
}

//////////////////////////////////////////////////////////////////////////
// wx event table
BEGIN_EVENT_TABLE(QuickstartDialog,wxDialog)
	EVT_BUTTON(id_add,QuickstartDialog::OnAdd)
	EVT_BUTTON(id_remove,QuickstartDialog::OnRemove)
	EVT_BUTTON(id_ok,QuickstartDialog::OnOk)
	EVT_LIST_ITEM_ACTIVATED(id_games_list,QuickstartDialog::OnOk)
END_EVENT_TABLE()

//////////////////////////////////////////////////////////////////////////
// Does a quick start dialog. 'parent' is the parent window for the dialog.
// 'result' points to a mbQuickstartResult to be filled in.
//
// Returns true (*result filled in) if user selected a game.
// Returns false (*result untouched) if user didn't.
//
// cfg updated with current set of dirs if used clicked OK.
bool mbQuickstart(wxWindow *parent,mbQuickstartConfig &cfg,
	mbQuickstartResult *result)
{
	QuickstartDialog dlg(parent,cfg.dirs);
	dlg.SetSize(cfg.dialog_pos);
	if(dlg.ShowModal()==wxID_OK) {
		dlg.GetDirs(&cfg.dirs);
		cfg.dialog_pos=dlg.GetRect();
		if(dlg.GetResult(result)) {
			wxASSERT(result);
			return true;
		}
	}
	return false;
}
