/**************************************************************************/
/*                                                                        */
/*            em6502 - An X Based 6502 Emulator & Teaching Aid            */
/*            ------------------------------------------------            */
/*              (c) Neil Pollard, University of Bristol 1994              */
/*                                                                        */
/*            Please read the file README for more information            */
/*                                                                        */
/**************************************************************************/
/* cpu6502.c */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <stdio.h>

#include "cpu6502.h"
#include "memory.h"
#include "window.h"
#include "em6502.h"

extern char *progname;

/* TRACE is set to >0 when the monitor window is active */
unsigned char TRACE;
/*=========================================================================*/
/* Add q to A along with CARRY */
inline void adc()
{
	int la, ha, lq, hq, lr, hr;
	register int result, q;

	q = readmem(AD);

	if (DECIMAL==1)
		{

		la = (A & 0x0f);
		ha = ((A & 0xf0) >> 4);
		lq = (q & 0x0f);
		hq = ((q & 0xf0) >> 4);

		lr = (la + lq + CARRY);

		hr = (lr > 9) ? (ha + hq + 1) : (ha + hq);

		if (hr > 9)
			CARRY=1;
		else
			CARRY=0;

		A = (((hr%10) << 4)+(lr % 10));
		}
	else	
		{
		result = (A+q+CARRY);
	
		if (((A<128) && (q<128) && ((result%256)>127)) ||
		((A>127) && (q>127) && ((result%256)<128)))
	
			OVERFLOW=1;
		else
			OVERFLOW=0;

		A = (result % 256);
	
		if (result>255) 
			CARRY=1;
		else
			CARRY=0;
		}

	flags(A);	

	nxt;
}

inline void adc_imm()
{
	int la, ha, lq, hq, lr, hr;
	register int result;

	if (DECIMAL==1)
		{
		la = (A & 0x0f);
		ha = ((A & 0xf0) >> 4);
		lq = (LB & 0x0f);
		hq = ((LB & 0xf0) >> 4);

		lr = (la + lq + CARRY);

		hr = (lr > 9) ? (ha + hq + 1) : (ha + hq);

		if (hr > 9)
			CARRY=1;
		else
			CARRY=0;

		A = (((hr%10) << 4)+(lr % 10));
		}
	else	
		{
		result = (A+LB+CARRY);
	
		if (((A<128) && (LB<128) && ((result%256)>127)) ||
		((A>127) && (LB>127) && ((result%256)<128)))
	
			OVERFLOW=1;
		else
			OVERFLOW=0;

		A = (result % 256);
	
		if (result>255) 
			CARRY=1;
		else
			CARRY=0;
		}

	flags(A);	

	nxt2;
}
/*=========================================================================*/
/* Perform bitwise AND operation between q and the accumulator */
inline void and()
{
	A = (A & (readmem(AD)));

	flags(A);
	nxt;
}

inline void and_imm()
{
	A = (A & LB);

	flags(A);
	nxt2;
}

/*=========================================================================*/
/* Shift q left one bit. Bit 7 goes into CARRY and zero goes into bit 0. */
inline void asl()
{
	register unsigned char q, res;

	q = readmem(AD);

	CARRY = ((q & 128) >> 7);
	res = (q << 1);

	writemem(AD,res);

	flags(res);
	nxt;
}

inline void asl_acc()
{
	CARRY = ((A & 128) >> 7);
	A = ((A << 1) % 256);

	flags(A);
	nxt1;
}	
/*=========================================================================*/
/* Branch if CARRY==0 */
inline void bcc()
{
	register int pc;

	if (!CARRY)
		{
		if (LB > 127)
			pc = ((PCL + (PCH << 8)) - (254-LB) );
		else
			pc = ((PCL + (PCH << 8)) + LB +2);

		PCL = (pc % 256);
		PCH = (pc / 256);
		}
	else nxt;
}	
/*=========================================================================*/
/* Branch if CARRY flag ==1 */
inline void bcs()
{
	register int pc;

	if (CARRY)
		{
		if (LB > 127)
			pc = ((PCL + (PCH << 8)) - (254-LB) );
		else
			pc = ((PCL + (PCH << 8)) + LB +2);

		PCL = (pc % 256);
		PCH = (pc / 256);
		}
	else nxt;
}
/*=========================================================================*/
/* Branch if ZERO flag ==1 */
inline void beq()
{
	register int pc;

	if (ZERO)
		{
		if (LB > 127)
			pc = ((PCL + (PCH << 8)) - (254-LB) );
		else
			pc = ((PCL + (PCH << 8))+ LB + 2);

		PCL = (pc % 256);
		PCH = (pc / 256);
		}
	else nxt;
}
/*=========================================================================*/
/* Perform A AND q. If the result is zero, ZERO flag ==1, 0 otherwise.
Bits 6 and 7 are copied to OVERFLOW and NEGATIVE flags */

inline void bit()
{
	register unsigned char q;

	q = ((readmem(AD)) & A);

	if (q & 64)
		OVERFLOW=1;
	else
		OVERFLOW=0;

	flags(q);

	nxt;
}
/*=========================================================================*/
/* Brach if NEGATIVE flag ==1 */
inline void bmi()
{
	register int pc;

	if (NEGATIVE)
		{
		if (LB > 127)
			pc = ((PCL + (PCH << 8)) - (254-LB) );
		else
			pc = ((PCL + (PCH << 8))+ LB + 2);

		PCL = (pc % 256);
		PCH = (pc / 256);
		}
	else nxt;
}
/*=========================================================================*/
/* Branch if ZERO flag ==0 */
inline void bne()
{
	register int pc;

	if (!ZERO)
		{
		if (LB > 127)
			pc = ((PCL + (PCH << 8)) - (254-LB) );
		else
			pc = ((PCL + (PCH << 8))+ LB+ 2);

		PCL = (pc % 256);
		PCH = (pc / 256);
		}
	else nxt;
}			
/*=========================================================================*/
/* Brach if NEGATIVE FLAG ==0 */
inline void bpl()
{
	register int pc;

	if (!NEGATIVE)
		{
		if (LB > 127)
			pc = ((PCL + (PCH << 8)) - (254-LB) );
		else
			pc = ((PCL + (PCH << 8))+ LB + 2);

		PCL = (pc % 256);
		PCH = (pc / 256);
		}
	else nxt;
}
/*=========================================================================*/
/* BRK */
inline void usr_brk()
{
	register unsigned char p;

	PCL += 2;

	if (PCL > 255)
		{PCH += 1; PCL = (PCL % 256);}

	writemem(256+S,PCH);
	writemem(255+S,PCL);

	BRK=1;

	p=((128*NEGATIVE)+(64*OVERFLOW)+16+(8*DECIMAL)+(4*IRQD)+(2*ZERO)+CARRY);	

	writemem(254+S,p);
	S -= 3;

	IRQD=1;

	PCL = readmem(0xfffe);
	PCH = readmem(0xffff);
}
/*=========================================================================*/
/* Branch if OVERFLOW flag ==0 */
inline void bvc()
{
	register int pc;

	if (!OVERFLOW)
		{
		if (LB > 127)
			pc = ((PCL + (PCH << 8)) - (254-LB) );
		else
			pc = ((PCL + (PCH << 8))+ LB + 2);

		PCL = (pc % 256);
		PCH = (pc / 256);
		}
	else nxt;
}
/*=========================================================================*/
/* Branch if OVERFLOW flag ==1 */
inline void bvs()
{
	register int pc;

	if (OVERFLOW)
		{
		if (LB > 127)
			pc = ((PCL + (PCH << 8)) - (254-LB) );
		else
			pc = ((PCL + (PCH << 8))+ LB + 2);

		PCL = (pc % 256);
		PCH = (pc / 256);
		}
	else nxt;
}
/*=========================================================================*/
/* Set CARRY flag to 0 */
inline void clc()
{
	CARRY=0;
	nxt;
}
/*=========================================================================*/
/* Set DECIMAL flag to 0 */
inline void cld()
{
	DECIMAL=0;
	nxt;
}
/*=========================================================================*/
/* Set IRQD flag to 0 */
inline void cli()
{
	IRQD=0;
	nxt;
}
/*=========================================================================*/
/* Set OVERFLOW flag to 0 */
inline void clv()
{
	OVERFLOW=0;
	nxt;
}
/*=========================================================================*/
/* q is subtracted from A, without storing the result and flags are set */
inline void cmp()
{
	register int q;

	q = (A - readmem(AD));
	if (q < 0) 
		{q += 256; CARRY=0;}
	else
		CARRY=1;
	flags(q);
	nxt;
}

inline void cmp_imm()
{
	LB = (A - LB);

	if (LB < 0) 
		{LB += 256; CARRY=0;}
	else
		CARRY=1;

	flags(LB);
	nxt2;
}	
/*=========================================================================*/
/* q is subtracted from X without stroing the result and flags are set */ 
inline void cpx()
{
	register int q; 

	q = (X - readmem(AD));
	if (q < 0)
		{q +=256; CARRY=0;}
	else
		CARRY=1;
	flags(q);
	nxt;
}

inline void cpx_imm()
{
	LB = (X - LB);
	if (LB < 0)
		{LB +=256; CARRY=0;}
	else
		CARRY=1;
	flags(LB);
	nxt2;
}

/*=========================================================================*/
/* q is subtracted from Y without storing the result and flags are set */
inline void cpy()
{
	register int q; 

	q = (Y - readmem(AD));
	if (q<0)
		{q +=256; CARRY=0;}
	else
		CARRY=1;
	flags(q);
	nxt;
}

inline void cpy_imm()
{
	LB = (Y - LB);
	if (LB<0)
		{LB +=256; CARRY=0;}
	else
		CARRY=1;
	flags(LB);
	nxt2;
}

/*=========================================================================*/
/* q is decreased by 1 */
inline void dec()
{
	register int q;

	q = readmem(AD);
	q = (q == 0) ? 255 : (q - 1);

	writemem(AD,q);

	flags(q);
	nxt;
}
/*=========================================================================*/
/* X is decreased by 1 */
inline void dex()
{
	X = (X == 0) ? 255 : (X-1);
	flags(X);
	nxt;
}
/*=========================================================================*/
/* Y is decreased by 1 */
inline void dey()
{
	Y = (Y == 0) ? 255 : (Y-1);
	flags(Y);
	nxt;
}
/*=========================================================================*/
/* Perform exclusive on corresponding bits of A and q, result stored in A */
inline void eor()
{
	A = (A ^ (readmem(AD)));

	flags(A);
	nxt;
}

inline void eor_imm()
{
	A = (A ^ LB);

	flags(A);
	nxt2;
}
/*=========================================================================*/
/* q is increased by 1 */
inline void inc()
{
	register int q;

	q = readmem(AD);
	q = (q == 255) ? 0 : (q+1);

	writemem(AD,q);
		 			
	flags(q);
	nxt;
}
/*=========================================================================*/
/* X is increased by 1 */
inline void inx()
{
	X = (X == 255) ? 0 : (X+1);
	flags(X);
	nxt;
}
/*=========================================================================*/
/* Y is increased by 1 */
inline void iny()
{
	Y = (Y == 255) ? 0 : (Y+1);
	flags(Y);
	nxt;
}
/*=========================================================================*/
/* Jump to an address */
inline void jmp()
{
	PCL = (AD % 256);
	PCH = (AD / 256);
}
/*=========================================================================*/
/* Jump to a subroutine, storing PC+2 on the stack */
inline void jsr()
{
	PCL += 2;

	if (PCL>255) {
		PCL = (PCL % 256); PCH += 1; }

	writemem(256+S,PCH);
	writemem(255+S,PCL);

	S -= 2;

	PCL = LB;
	PCH = HB;
}
/*=========================================================================*/
/* Copy the contents of a memory location into A */
inline void lda()
{
	A = readmem(AD);		

	flags(A);
	nxt;
}

inline void lda_imm()
{
	A = LB;		

	flags(A);
	nxt2;
}

/*=========================================================================*/
/* Copy the contents of a memory location into X */
inline void ldx()
{
	X = readmem(AD);		

	flags(X);
	nxt;
}

inline void ldx_imm()
{
	X = LB;		

	flags(X);
	nxt2;
}
/*=========================================================================*/
/* Copy the contents of a memory location into Y */
inline void ldy()
{
	Y = readmem(AD);		

	flags(Y);
	nxt;
}

inline void ldy_imm()
{
	Y = LB;		

	flags(Y);
	nxt2;
}
/*=========================================================================*/
/* Shift the contents of q left one bit, bit 0 into carry and 0 into bit 7 */
inline void lsr()
{
	register int q;

	q = readmem(AD);

	CARRY = (q & 1);
	q = ((q >> 1) % 256);

	writemem(AD,q);

	flags(q); 
	nxt;
}

inline void lsr_acc()
{
	CARRY = (A & 1);

	A = ((A >> 1) % 256);

	flags(A);
	nxt1;
}

/*=========================================================================*/
/* No operation */
inline void nop() 
{
	nxt1;
}
/*=========================================================================*/
/* Perform logical OR on the corresponding bits of A and q, result in A */
inline void ora()
{
	A = (A | (readmem(AD)));

	flags(A);
	nxt;
}

inline void ora_imm()
{
	A = (A | LB);

	flags(A);
	nxt2;
}
/*=========================================================================*/
/* Push A onto the stack */
inline void pha()
{
	writemem(256+S,A);
	S -= 1;
	nxt;
}
/*=========================================================================*/
/* Push the status byte (flags) onto the stack */
inline void php()
{
	register unsigned char p;

	p=((128*NEGATIVE)+(64*OVERFLOW)+(16*BRK)+(8*DECIMAL)+(4*IRQD)+(2*ZERO)+CARRY);

	writemem(256+S,p);
	S -= 1;
	nxt;
}
/*=========================================================================*/
/* Pop A from the stack */
inline void pla()
{
	A = readmem(257+S);

	
	S += 1;
	flags(A);
	nxt;
}
/*=========================================================================*/
/* Pop the status register (flags) from the stack */
inline void plp()
{
	register int p;

	p = readmem(257+S);

	NEGATIVE=((p & 128)/128);
	OVERFLOW=((p & 64)/64);
	BRK=((p & 16)/16);
	DECIMAL=((p & 8)/8);
	IRQD=((p & 4)/4);
	ZERO=((p & 2)/2);
	CARRY=(p & 1);

	S += 1;
	nxt;
}
/*=========================================================================*/
/* Rotate q left one bit. Bit 7 goes into CARRY, and old CARRY into bit 0 */ 
inline void rol()
{
	register unsigned char q, temp;

	q = readmem(AD);

	temp = ((q & 128) >> 7);
	q = (q << 1);
	if (CARRY == 1) q += 1;
	CARRY = temp;

	writemem(AD,q);

	flags(q); 
	nxt;
}

inline void rol_acc()
{
	register int temp;

	temp = ((A & 128) >> 7);
	A = ((A << 1) % 256);
	if (CARRY == 1) A += 1;
	CARRY = temp;

	flags(A);

	nxt1;
}

/*=========================================================================*/
/* Rotate q right by 1 bit, bit 0 goes into CARRY and old CARRY into bit 7 */
inline void ror()
{
	register unsigned char temp, q;

	q = readmem(AD);

	temp = (q & 1);
	q = (q >> 1);
	if (CARRY == 1) q += 128;
	CARRY=temp;
	
	writemem(AD,q);

	flags(q); 
	nxt;
}

inline void ror_acc()
{
	int temp;

	temp = (A & 1);
	A = ((A >> 1) % 256);
	if (CARRY == 1) A += 128;
	CARRY = temp;	

	flags(A); 
	nxt1;
}
/*=========================================================================*/
/* Return from interrupt, restoring the status register and PC */
inline void rti()
{
	register unsigned char p;	

	p = readmem(257+S);

	NEGATIVE=((p & 128)/128);
	OVERFLOW=((p & 64)/64);
	BRK=((p & 16)/16);
	DECIMAL=((p & 8)/8);
	IRQD=((p & 4)/4);
	ZERO=((p & 2)/2);
	CARRY=(p & 1);

	PCH = readmem(259+S);
	PCL = readmem(258+S);

	S += 3;
}
/*=========================================================================*/
/* Return from subroutine, restoring the PC */
inline void rts()
{
	PCH = readmem(258+S);
	PCL = readmem(257+S);
	S += 2;

	PCL += 1;
	if (PCL > 255) {PCL = (PCL % 256); PCH += 1;}
}
/*=========================================================================*/
/* Subtract q from A */ 
inline void sbc()
{
	int aa, aq;
	register int q, result;

	q = readmem(AD);
	
	if (DECIMAL==1)
		{
		aa = (((A & 0xf0) >> 4) * 10) + (A & 0x0f);
		aq = (((q & 0xf0) >> 4) * 10) + (q & 0x0f);

		result = (aa - aq); 

		if (result < 0)
			{CARRY=1; result = (100-result);}
		else
			CARRY=0;

		A = ((result%10) + ((result/10) << 4));
		}
	else
		{
		result = (A+CARRY+(255-q));

		if (((A<128) && (q<128) && ((result & 0xff)>127)) ||
		((A>127) && (q>127) && ((result & 0xff)<128)))
	
			OVERFLOW=1;
		else
			OVERFLOW=0;
	
		if (result > 255) 
			CARRY=1;
		else
			CARRY=0;	
			
		A = (result & 0xff);
		}
	flags(A);

	nxt;		
}

inline void sbc_imm()
{
	int aa, aq;
	register int  result;

	if (DECIMAL==1)
		{
		aa = (((A & 0xf0) >> 4) * 10) + (A & 0x0f);
		aq = (((LB & 0xf0) >> 4) * 10) + (LB & 0x0f);

		result = (aa - aq); 

		if (result < 0)
			{CARRY=1; result = (100-result);}
		else
			CARRY=0;

		A = ((result%10) + ((result/10) << 4));
		}
	else
		{
		result = (A+CARRY+(255-LB));

		if (((A<128) && (LB<128) && ((result & 0xff)>127)) ||
		((A>127) && (LB>127) && ((result & 0xff)<128)))
	
			OVERFLOW=1;
		else
			OVERFLOW=0;
	
		if (result > 255) 
			CARRY=1;
		else
			CARRY=0;	
			
		A = (result & 0xff);
		}
	flags(A);

	nxt2;		
}
/*=========================================================================*/
/* set CARRY flag to 1 */
inline void sec()
{
	CARRY=1;
	nxt;
}
/*=========================================================================*/
/* Set DECIMAL flag to 1 */
inline void sed()
{
	DECIMAL=1;
	nxt;
}
/*=========================================================================*/
/* Set IRQD flag to 1 */
inline void sei()
{
	IRQD=1;
	nxt;
}
/*=========================================================================*/
/* Store the contents of A in memory */
inline void sta()
{
	writemem(AD,A);
	nxt;
}
/*=========================================================================*/
/* Store the contents of X in memory */
inline void stx()
{
	writemem(AD,X);
	nxt;
}
/*=========================================================================*/
/* Store the contents of Y in memory */
inline void sty()
{
	writemem(AD,Y);
	nxt;
}
/*=========================================================================*/
/* X = A */
inline void tax()
{
	X = A;
	flags(X);
	nxt;
}
/*=========================================================================*/
/* Y = A */
inline void tay()
{
	Y = A;
	flags(Y);
	nxt;
}
/*=========================================================================*/
/* X = S (Stack Pointer) */
inline void tsx()
{
	X = S;
	flags(X);
	nxt;
}
/*=========================================================================*/
/* A = X */
inline void txa()
{
	A = X;
	flags(A);
	nxt;
}
/*=========================================================================*/
/* S (Stack Pointer) = X */
inline void txs()
{
	S = X;
	flags(S);
	nxt;
}
/*=========================================================================*/
/* A = Y */
inline void tya()
{
	A = Y;
	flags(A);
	nxt;
}
/*=========================================================================*/
inline void emu_window_events(argc, argv)
int argc;
char ** argv;

{
	int count=1;
	int bufsize=1;
	KeySym keysym;
	XComposeStatus compose;
	char keybuffer[1];

	/* Get events */
	if (XCheckMaskEvent(display, ExposureMask | KeyPressMask | 
StructureNotifyMask,&emu_report))
		{
		switch  (emu_report.type) {
	/* Window is exposed */
		case Expose:
			if (emu_report.xexpose.count != 0)
				break;

			/* update emulator window */
			draw_emu_window(emu_win, emu_gc, emu_font_info);
			break;


		/* If a key is pressed it is inserted into a buffer
		 * at 0x3e0.
		 * This has been chosen for compatibility with the BBC
		 * Operating system */

		case KeyPress:

		count = XLookupString(&emu_report, keybuffer, bufsize, &keysym, &compose);	
		
		if ((keysym == XK_Return) || (keysym == XK_Linefeed))
			{
			M[0x2e1]++;
			if (M[0x2e1]==0)
				M[0x2e1]=0xe0;
			M[(0x2ff+M[0x2e1])]=13;			
			}
		else if ((keysym == XK_BackSpace) || (keysym == XK_Delete))
			{
			M[0x2e1]++;
			if (M[0x2e1]==0)
				M[0x2e1]=0xe0;
			M[(0x2ff+M[0x2e1])]=127;
			}
#if (MONITOR == 1)
		else if ((keysym == XK_F9) && (TRACE==0))
			{
			TRACE=1;
			create_mon_window(argc, argv);
			}
#endif
		else if ((keysym >= XK_space) && (keysym <= XK_asciitilde))
			{
			M[0x2e1]++;
			if (M[0x2e1]==0)
				M[0x2e1]=0xe0;
			M[(0x2ff+M[0x2e1])]=keybuffer[0];
			}

		default: break;
		}
	}
}
/*=========================================================================*/
void cpu6502(argc,argv)
int argc;
char ** argv;
{
	int updated, counter, ad;

	int screen_update_counter=0;
	int event_timer=0;
	unsigned char current_rom;

	/* Load appropiate ROM images */
	load_images();

	IRQD = ZERO = 1;	/* Disable Interrupts */
	A = X = Y = 0;		/* Initialise registers */
	S = 0xff;		/* Stack Register */
	
	PCL = M[0xfffc];	/* Start from Reset state */
	PCH = M[0xfffd];

	BREAK_LOW = BREAK_HI = 0;	/* Initialise monitor breakpoints */

/* Main instruction loop */
while (((M[(PCL+(PCH << 8))]) != 0x60) || (S<0xff))
{

/* Check for window events every EVENT_CYCLES instruction cycles*/
if (event_timer==EVENT_CYCLES)
	{
	event_timer=0;
	emu_window_events(argc, argv);	
	}
event_timer++;

/* Update screen every SCREEN_CYCLES instruction cycles or every cycle when
 * the monitor window is active */
if ((screen_update_counter==SCREEN_CYCLES) || (TRACE))
	{

	/* Ensure that screen update occurs regardless, if the monitor
	 * window is active */

	updated = TRACE;

	/* Check whether the screen has actually been updated
	 * The drawing process is time consuming, and calling
	 * it unnecessarily wastes time.
	 *
	 * A copy of the screen memory area is stored in SCREEN[]
	 * and checked against the actual memory */

	for (counter = 0; counter < 0x400; counter++)
		{
		if (SCREEN[counter] != M[counter+0x7c00])
			{
			SCREEN[counter] = M[counter+0x7c00];
			updated = 1;
			}
		}

	/* only if the screen has changed redraw the window contents */
	if (updated)
		draw_emu_window(emu_win, emu_gc, emu_font_info);

	/* Reset the counter */
	screen_update_counter=0;
	}

/* Increase the screen update counter */
screen_update_counter++;

/* This is the paged ROM selection register.
 * If it has changed then another ROM bank should be paged into main 
 * memory. Once again, this is implemented for compatibility with the
 * BBC OS */ 

if (M[0xfe30] != current_rom)
	{
	M[0xfe30]=(M[0xfe30] & 0xf);
	current_rom=M[0xfe30];
	
	/* Page the ROM bank in at 0x8000 */
	for (counter=0; counter < 0x4000; counter++)
		M[0x8000+counter]=paged_rom[current_rom][counter];
	}

#if (FILING_SYSTEM == 1)
/* For future expansion, a filing system routine */
if ((PCL == 0xdd) && (PCH == 0xff))
	{	
	fprintf(stderr,"%s: File system called\n",progname); 
	IMP; 
	rts();
	}
#endif

#if (MONITOR == 1)
/* If the Program Counter is within limits set in the monitor window
 * then switch the monitor window on */
if ((PCL+(PCH << 8)) >= BREAK_LOW && (PCL+(PCH << 8)) <= BREAK_HI 
&& (!TRACE))
	{
	TRACE = 1;
	create_mon_window(argc,argv);
	}

/* If the monitor window is switched on then deal with events from it */
if (TRACE) 
	mon_window_events();
#endif

ad = (PCL+(PCH << 8));	/* Address to load instruction */
LB = readmem(ad+1);	/* next 2 bytes... */
HB = readmem(ad+2);

/* Fetch instruction and execute it */
switch (readmem(ad)) {

	case 0 : IMP; usr_brk(); break;
	case 1 : INDX; ora(); break;
	case 5 : ZP; ora(); break;
	case 6 : ZP; asl(); break;
	case 8 : IMP; php(); break;
	case 9 : ora_imm(); break;
	case 10 : asl_acc(); break;
	case 13 : ABS; ora(); break;
	case 14 : ABS; asl(); break;
	case 16 : REL; bpl(); break;
	case 17 : INDY; ora(); break;
	case 21 : ZPX; ora(); break;
	case 22 : ZPX; asl(); break;
	case 24 : IMP; clc(); break;
	case 25 : ABSY; ora(); break;
	case 29 : ABSX; ora(); break;
	case 30 : ABSX; asl(); break;
	case 32 : ABS; jsr(); break;
	case 33 : INDX; and(); break;
	case 36 : ZP; bit(); break;
	case 37 : ZP; and(); break;
	case 38 : ZP; rol(); break;
	case 40 : IMP; plp(); break;
	case 41 : and_imm(); break;
	case 42 : rol_acc(); break;
	case 44 : ABS; bit(); break;
	case 45 : ABS; and(); break;
	case 46 : ABS; rol(); break;
	case 48 : REL; bmi(); break;
	case 49 : INDY; and(); break;
	case 53 : ZPX; and(); break;
	case 54 : ZPX; rol(); break;
	case 56 : IMP; sec(); break;
	case 57 : ABSY; and(); break;
	case 61 : ABSX; and(); break;
	case 62 : ABSX; rol(); break;
	case 64 : IMP; rti(); break;
	case 65 : INDX; eor(); break;
	case 69 : ZP; eor(); break;
	case 70 : ZP; lsr(); break;
	case 72 : IMP; pha(); break;
	case 73 : eor_imm(); break;
	case 74 : lsr_acc(); break;
	case 76 : ABS; jmp(); break;
	case 77 : ABS; eor(); break;
	case 78 : ABS; lsr(); break;
	case 80 : REL; bvc(); break; 
	case 81 : INDY; eor(); break;
	case 85 : ZPX; eor(); break;
	case 86 : ZPX; lsr(); break;
	case 88 : IMP; cli(); break;
	case 89 : ABSY; eor(); break;
	case 93 : ABSX; eor(); break;
	case 94 : ABSX; lsr(); break;
	case 96 : IMP; rts(); break;
	case 97 : INDX; adc(); break;
	case 101 : ZP; adc(); break;
	case 102 : ZP; ror(); break;
	case 104 : IMP; pla(); break;
	case 105 : adc_imm(); break;
	case 106 : ror_acc(); break;
	case 108 : INDABS; jmp(); break;
	case 109 : ABS; adc(); break;
	case 110 : ABS; ror(); break;
	case 112 : REL; bvs(); break;
	case 113 : INDY; adc(); break;
	case 117 : ZPX; adc(); break;
	case 118 : ZPX; ror(); break;
	case 120 : IMP; sei(); break;
	case 121 : ABSY; adc(); break;
	case 124 : ABSX; jmp(); break;
	case 125 : ABSX; adc(); break;
	case 126 : ABSX; ror(); break;
	case 129 : INDX; sta(); break;
	case 132 : ZP; sty(); break;
	case 133 : ZP; sta(); break;
	case 134 : ZP; stx(); break;
	case 136 : IMP; dey(); break;
	case 138 : IMP; txa(); break;
	case 140 : ABS; sty(); break;
	case 141 : ABS; sta(); break;
	case 142 : ABS; stx(); break;
	case 144 : REL; bcc(); break;
	case 145 : INDY; sta(); break;
	case 148 : ZPX; sty(); break;
	case 149 : ZPX; sta(); break;
	case 150 : ZPY; stx(); break;
	case 152 : IMP; tya(); break;
	case 153 : ABSY; sta(); break;
	case 154 : IMP; txs(); break;
	case 157 : ABSX; sta(); break;
	case 160 : ldy_imm(); break;
	case 161 : INDX; lda(); break;
	case 162 : ldx_imm(); break;
	case 164 : ZP; ldy(); break;
	case 165 : ZP; lda(); break;
	case 166 : ZP; ldx(); break;
	case 168 : IMP; tay(); break;
	case 169 : lda_imm(); break;
	case 170 : IMP; tax(); break;
	case 172 : ABS; ldy(); break;
	case 173 : ABS; lda(); break;
	case 174 : ABS; ldx(); break;
	case 176 : REL; bcs(); break;
	case 177 : INDY; lda(); break;
	case 180 : ZPX; ldy(); break;
	case 181 : ZPX; lda(); break;
	case 182 : ZPY; ldx(); break;
	case 184 : IMP; clv(); break;
	case 185 : ABSY; lda(); break;
	case 186 : IMP; tsx(); break;
	case 188 : ABSX; ldy(); break;
	case 189 : ABSX; lda(); break;
	case 190 : ABSY; ldx(); break;
	case 192 : cpy_imm(); break;
	case 193 : INDX; cmp(); break;
	case 196 : ZP; cpy(); break;
	case 197 : ZP; cmp(); break;
	case 198 : ZP; dec(); break;
	case 200 : IMP; iny(); break;
	case 201 : cmp_imm(); break;
	case 202 : IMP; dex(); break;
	case 204 : ABS; cpy(); break;
	case 205 : ABS; cmp(); break;
	case 206 : ABS; dec(); break;
	case 208 : REL; bne(); break;
	case 209 : INDY; cmp(); break;
	case 213 : ZPX; cmp(); break;
	case 214 : ZPX; dec(); break;
	case 216 : IMP; cld(); break;
	case 217 : ABSY; cmp(); break;
	case 221 : ABSX; cmp(); break;
	case 222 : ABSX; dec(); break;
	case 224 : cpx_imm(); break;
	case 225 : INDX; sbc(); break;
	case 228 : ZP; cpx(); break;
	case 229 : ZP; sbc(); break;
	case 230 : ZP; inc(); break;
	case 232 : IMP; inx(); break;
	case 233 : sbc_imm(); break;
	case 234 : IMP; nop(); break;
	case 236 : ABS; cpx(); break;
	case 237 : ABS; sbc(); break;
	case 238 : ABS; inc(); break;
	case 240 : REL; beq(); break;
	case 241 : INDY; sbc(); break;
	case 245 : ZPX; sbc(); break;
	case 246 : ZPX; inc(); break;
	case 248 : IMP; sed(); break;
	case 249 : ABSY; sbc(); break;
	case 253 : ABSX; sbc(); break;
	case 254 : ABSX; inc();  break;
	default : IMP; nop();
	break;
	}
	}
}
/*=========================================================================*/

