/**************************************************************************/
/*                                                                        */
/*            em6502 - An X Based 6502 Emulator & Teaching Aid            */
/*            ------------------------------------------------            */
/*              (c) Neil Pollard, University of Bristol 1994              */
/*                                                                        */
/*            Please read the file README for more information            */
/*                                                                        */
/**************************************************************************/
/* monitor.c */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <string.h>
#include <stdio.h>
#include "decode.h"
#include "memory.h"
#include "window.h"

extern char *progname;

/* Main registers from cpu6502.c */
extern unsigned char TRACE;
extern int A, X, Y, PCH, PCL, LB, HB, S;
extern unsigned char NEGATIVE, OVERFLOW, BRK, DECIMAL, IRQD, ZERO, CARRY;
extern int BREAK_LOW, BREAK_HI;

/* The following are global variables specific to the monitor.
 * They are not declared locally and passed between functions due to their
 * large number and frequency of use */

/* Global stack display string */
char STACK_DISPLAY[18] = {};

/* Copies of previous values of main registers used for display */
int OLD_A, OLD_X, OLD_Y, OLD_PCL, OLD_PCH;
unsigned char OLD_NEGATIVE, OLD_OVERFLOW, OLD_BRK, OLD_DECIMAL;
unsigned char OLD_IRQD, OLD_ZERO, OLD_CARRY, OLD_S;
char OLD_STACK_DISPLAY[18];

/* Four user defined memory locations (1-4), zero is disregarded */
int memory_loc[5] = {0,0,0,0,0};

/* Text box global variables (see textbox.h)
 * these will be changed to a structure in future */
extern int text_box_pos[7];
extern char text_box_string[7][5];
extern int text_box_x[7];
extern int text_box_y[7];

/*======================================================================*/
/* Print the value of a text box on screen.
 * This is specially written for the monitor, and so is not included
 * with the other functions in textbox.c */

void print_text_box(win, gc, x, y, value)
Window win;
GC gc;
int x, y, value;
{
char text[5];

/* Display the value in hex; field width of 4 */ 
sprintf(text, "%04X", value);
XClearArea(display, win, x-4, y-14, 43, 18, 0);
XDrawString(display, win, gc, x, y, text, 4);
}

/*======================================================================*/
/* Close a text box, and store its contents in appropriate place
 * Again, written specifically for the monitor so not included in
 * textbox.c */

void close_text_box(win, gc, box_number)
Window win;
GC gc;
{
int x, y;

/* If the text box has received input, store its contents */
if (text_box_pos[box_number])
	{
	switch (box_number)
		{
		case 1 : 
	memory_loc[1]=hexstring_to_int(text_box_string[1],text_box_pos[1]); 
	break;

		case 2 :
	memory_loc[2]=hexstring_to_int(text_box_string[2],text_box_pos[2]);
	break;

		case 3 :
	memory_loc[3]=hexstring_to_int(text_box_string[3],text_box_pos[3]);
	break;

		case 4 :
	memory_loc[4]=hexstring_to_int(text_box_string[4],text_box_pos[4]);
	break;

		case 5 :
	BREAK_LOW=hexstring_to_int(text_box_string[5],text_box_pos[5]);
	break;

		case 6 :
	BREAK_HI=hexstring_to_int(text_box_string[6],text_box_pos[6]);
	break;

		}
	}

/* Remove the rectangle outlining the text box */
XClearArea(display, win, text_box_x[box_number]-5, 
		text_box_y[box_number]-15, 46, 21, 0);
}

/*======================================================================*/
/* Draw the main monitor window and all static text.
 * The register contents are redrawn elsewhere since they change on a
 * regular basis.
 * This function only needs to be called after an expose event, or when
 * the contents of a text box have changed */

void draw_mon_window(win, gc, font_info)

Window win;
GC gc;
XFontStruct *font_info;
{
/* Draw the 'before and after' register keys */

char *key1 = " A  X  Y (P) NV*BDIZC  PC  SP";
char *key2 = " Stack";

XDrawString(display, win, gc, 0, 15, key1, strlen(key1));
XDrawString(display, win, gc, 0, 165, key1, strlen(key1));
XDrawString(display, win, gc, 0, 60, key2, strlen(key2));
XDrawString(display, win, gc, 0, 215, key2, strlen(key2));

/* Some instructional text */
XDrawString(display, win, gc, 11, 330, "SPACE: next  F1: resume",23);
XDrawString(display, win, gc, 38, 345, "F2: load  F3: save",18);

/* Boxes around the different areas */
XDrawLine(display, win, gc, 270, 0, 270, 310);
XDrawLine(display, win, gc, 270, 65, 0, 65);
XDrawLine(display, win, gc, 270, 145, 0, 145);
XDrawLine(display, win, gc, 270, 220, 0, 220);
XDrawLine(display, win, gc, 270, 310, 0, 310);

/* The memory map */
XDrawRectangle(display, win, gc, 350, 15, 100, 320);
XDrawLine(display, win, gc, 350, 175, 450, 175);
XDrawLine(display, win, gc, 350, 95, 450, 95);
XDrawString(display, win, gc, 360, 50, "Operating", 9);
XDrawString(display, win, gc, 374, 65, "System", 6);
XDrawString(display, win, gc, 360, 135, "Paged ROM", 9);
XDrawString(display, win, gc, 385, 250, "RAM", 3);
XDrawString(display, win, gc, 455, 340, "0000", 4);
XDrawString(display, win, gc, 455, 180, "8000", 4);
XDrawString(display, win, gc, 455, 100, "C000", 4);
XDrawString(display, win, gc, 455, 20, "FFFF", 4);


/* General text */
XDrawString(display, win, gc, 120, 240, "Restart monitor:", 16);
XDrawString(display, win, gc, 130, 260, "between", 7);
XDrawString(display, win, gc, 166, 280, "and", 3);

/* Redraw the text boxes */
print_text_box(win, gc, 11, 240, memory_loc[1]);
print_text_box(win, gc, 11, 260, memory_loc[2]);
print_text_box(win, gc, 11, 280, memory_loc[3]);
print_text_box(win, gc, 11, 300, memory_loc[4]);
print_text_box(win, gc, 211, 260, BREAK_LOW);
print_text_box(win, gc, 211, 280, BREAK_HI);
}

/*======================================================================*/
/* Update the monitor window.
 * This needs to be called after each instruction cycle and after an
 * expose event */

void update_mon_window(win, gc, font_info)

Window win;
GC gc;
XFontStruct *font_info;
{
int pc_position, counter, relative;
unsigned char addr_mode, ad, lb, hb;
char old_registers[40];
char new_registers[40];
char temp[4];
char mnemonic[5];
char code[9], full[11];
char line1[15], line2[15];
char memstring[4][6];

/* Place the old register values in a string ready for printing */
sprintf(old_registers," %02X %02X %02X    %d%d0%d%d%d%d%d %02X%02X %02X",
	OLD_A,OLD_X,OLD_Y,
	OLD_NEGATIVE,OLD_OVERFLOW,OLD_BRK,OLD_DECIMAL,OLD_IRQD,OLD_ZERO,OLD_CARRY,
	OLD_PCH, OLD_PCL, OLD_S);

/* Place the new register values in a string */
sprintf(new_registers," %02X %02X %02X    %d%d0%d%d%d%d%d %02X%02X %02X",
	A,X,Y,
	NEGATIVE, OVERFLOW, BRK, DECIMAL, IRQD, ZERO, CARRY,
	PCH, PCL, S);

/* Print old values */
XClearArea(display, win, 0, 15, 260, 15, 0); 
XDrawString(display, win, gc, 0, 30, old_registers, strlen(old_registers));

/* Print new values */
XClearArea(display, win, 0, 165, 260, 15, 0);
XDrawString(display, win, gc, 0, 180, new_registers, strlen(new_registers));

/* Calculate position for the program counter pointer on the memory map */
pc_position=(340-((PCL+(PCH << 8))/205));

/* Draw the PC pointer */
XClearArea(display, win, 310, 0, 36, 355, 0);
XDrawString(display, win, gc, 312, pc_position, "PC->",4);

for (counter = 0; counter < 18; counter ++)
	STACK_DISPLAY[counter]='\0';

/* Place the top 7 stack values in a string (if they all exist) */
counter = 1;
while ((counter < 7) && ((counter+S) < 0x100))
	{
	sprintf(temp,"%02X ",M[counter+S+0x100]);
	strcat(STACK_DISPLAY, temp); 
	counter++;
	}

/* Print the old stack values */
XClearArea(display, win, 60, 45, 200, 15, 0);
XDrawString(display, win, gc, 60, 60,OLD_STACK_DISPLAY, strlen(OLD_STACK_DISPLAY));

/* Print the new stack values */
XClearArea(display, win, 60, 200, 200, 15, 0);
XDrawString(display, win, gc, 60, 215, STACK_DISPLAY, strlen(STACK_DISPLAY));

/* Take M[PC], M[PC+1] and M[PC+2] from memory */
ad=readmem(OLD_PCL+(OLD_PCH << 8));
lb=readmem(1+OLD_PCL+(OLD_PCH << 8));
hb=readmem(2+OLD_PCL+(OLD_PCH << 8));

/* Decode the addressing mode and instruction mnemonic */
decode(ad, &addr_mode, mnemonic);

/* Prepare a string for display accordingly */
switch (addr_mode)
	{
	case ACC :
		sprintf(code, "%02X", ad);
		sprintf(full, "A");
		break;
	case IMM :
		sprintf(code, "%02X %02X", ad, lb);
		sprintf(full, "#%02X",lb);
		break;
	case ABS :
		sprintf(code, "%02X %02X %02X", ad, lb, hb);
		sprintf(full, "%02X%02X", hb, lb);
		break;
	case ZP :
		sprintf(code, "%02X %02X",ad, lb);
		sprintf(full, "%X",lb);
		break;
	case ZPX :
		sprintf(code, "%02X %02X",ad, lb);
		sprintf(full, "%02X,X", lb);
		break;
	case ZPY :
		sprintf(code, "%02X %02X",ad, lb);
		sprintf(full, "%02X,Y",lb);
		break;
	case ABSX :
		sprintf(code, "%02X %02X %02X",ad, lb, hb);
		sprintf(full, "%02X%02X,X",hb, lb);
		break;
	case ABSY :
		sprintf(code, "%02X %02X %02X",ad, lb, hb);
		sprintf(full, "%02X%02X,Y",hb, lb);
		break;
	case IMP :
		sprintf(code, "%02X", ad);
		sprintf(full, "");
		break;
	case INDX :
		sprintf(code, "%02X %02X",ad, lb);
		sprintf(full, "(%02X,X)",lb);
		break;
	case INDY :
		sprintf(code, "%02X %02X",ad, lb);
		sprintf(full, "(%02X),Y",lb);
		break;
	case INDABS :
		sprintf(code, "%02X %02X %02X",ad, lb, hb);
		sprintf(full, "(%02X%02X)", hb, lb);
		break;
	case REL :
		if (lb > 127)
			relative = ((OLD_PCL + (OLD_PCH << 8)) - (254-lb));
		else
			relative = ((OLD_PCL + (OLD_PCH << 8))+lb+2);

		sprintf(code, "%02X %02X",ad, lb);
		sprintf(full, "%04X", relative);
		break;
	default : 
		break;
		}

/* Fetch two strings to describe the 6502 operation */
symbolic(ad, line1, line2);

/* Print the instruction in hexadecimal form */
XClearArea(display, win, 69, 73, 80, 15, 0);
XDrawString(display, win, gc, 69, 85, code, strlen(code));

/* Print the instruction in mnemonic form */
XClearArea(display, win, 60, 88, 90, 15, 0);
XDrawString(display, win, gc, 60, 100, mnemonic, strlen(mnemonic));
XDrawString(display, win, gc, 96, 100, full, strlen(full));

/* Print the description strings */
XClearArea(display, win, 60, 108, 170, 30, 0);
XDrawString(display, win, gc, 60, 120, line1, strlen(line1));
XDrawString(display, win, gc, 60, 135, line2, strlen(line2));

/* Print the contents of the four selected memory locations */
for (counter = 0; counter < 5; counter ++)
	sprintf(memstring[counter],"=%04X",readmem(memory_loc[counter]));

XClearArea(display, win, 55, 225, 45, 75, 0); 
XDrawString(display, win, gc, 55, 240, memstring[1], 5);
XDrawString(display, win, gc, 55, 260, memstring[2], 5);
XDrawString(display, win, gc, 55, 280, memstring[3], 5);
XDrawString(display, win, gc, 55, 300, memstring[4], 5);
}
/*======================================================================*/
/* Store the values of all the registers, so they can be displayed at
 * the top of the updated monitor window */

void store_values()
{
int counter;

/* Store the entire string containing the stack values to save
   recalculation later */

for (counter = 0; counter < 19; counter++)
	*(OLD_STACK_DISPLAY+counter) = *(STACK_DISPLAY+counter);

/* Store the main registers */

OLD_A = A;
OLD_X = X;
OLD_Y = Y;
OLD_NEGATIVE = NEGATIVE;
OLD_OVERFLOW = OVERFLOW;
OLD_BRK= BRK;
OLD_DECIMAL = DECIMAL;
OLD_IRQD = IRQD;
OLD_ZERO = ZERO;
OLD_CARRY = CARRY;
OLD_PCH = PCH;
OLD_PCL = PCL;
OLD_S = S;
}

/*======================================================================*/
/* The main events loop of the monitor window.
 * This is called by the cpu6502 funtion if the monitor window is active.
 */

void mon_window_events()
{
int count=1;
int bufsize=1;
KeySym keysym;
XComposeStatus compose;
char keybuffer[1];
unsigned char skip = 0;
int active_text_box = 0;
int c;

update_mon_window(mon_win, mon_gc, mon_font_info);

/* Continue processing events in the loop, until 'skip' indicates that
 * the emulator can proceed with the next m/c instruction */
while (!skip)
	{

        /* Get events */
        if (XCheckMaskEvent(display, ExposureMask | KeyPressMask | 
		ButtonPressMask | StructureNotifyMask,&mon_report))
                {
                switch  (mon_report.type) 
		{
        	/* Window is exposed */
                case Expose:
                        if (mon_report.xexpose.count != 0)
                                break;

			/* If TRACE is 1, the the window has just been
			 * opened. Store the register values, but do not
			 * wait for events until the next m/c instruction
			 * has been processed. TRACE now becomes 2 */
			if (TRACE == 1)
				{
				TRACE = 2;
				skip = 1;
				store_values();
				}

			/* draw monitor window */
                        draw_mon_window(mon_win, mon_gc, mon_font_info);
			update_mon_window(mon_win, mon_gc, mon_font_info);
			break;
				
		case ButtonPress:

		/* if a mouse button is pressed, check to see whether
		 * the pointer is over one of the text boxes */		
		for (c = 1; c < 7; c++)
			{
			if ((mon_report.xbutton.x > (text_box_x[c] - 4))  &&
			    (mon_report.xbutton.x < (text_box_x[c] + 44)) &&
			    (mon_report.xbutton.y > (text_box_y[c] - 14)) &&
			    (mon_report.xbutton.y < (text_box_y[c] + 6))) 
				{
				/* if a text box is already active close it */
				if (active_text_box)
					{
					close_text_box(mon_win, mon_gc, active_text_box);
					draw_mon_window(mon_win, mon_gc, mon_font_info);
					update_mon_window(mon_win, mon_gc, mon_font_info);
					}
				/* open the text box under the pointer */
				active_text_box = c;
				open_text_box(mon_win, mon_gc, c);
				break;
				}
			}
		break;

		/* Key pressed */
                case KeyPress:
		
		count = XLookupString(&mon_report, keybuffer, bufsize, &keysym, &compose);

		/* Space bar pressed */
		if (keysym == XK_space)
			{
			/* if a text box is active, close it */
			if (active_text_box)
				{
				close_text_box(mon_win, mon_gc, active_text_box);
				draw_mon_window(mon_win, mon_gc, mon_font_info);
				update_mon_window(mon_win, mon_gc, mon_font_info);
				}
			/* 'skip' flags that the emulator can proceed with
			 * the next instruction */
			skip = 1;
			/* store the current register values for
			 * display next time around */
			store_values();
			break;
			}

		/* F1 pressed */
		else if (keysym == XK_F1)
			{
			/* Close any open text box */
			close_text_box(mon_win, mon_gc, active_text_box);
			/* Flag the emulator to continue */
			skip = 1;
			/* Indicate that the monitor window is no
			 * longer active */
			TRACE = 0;
			/* Close the monitor window */
			XUnloadFont(display, mon_font_info->fid);
			XFreeGC(display, mon_gc);
			XUnmapWindow(display, mon_win);
			break;
			}

		else if (keysym == XK_F2)
			{
			load_snapshot();
			break;
			}

		else if (keysym == XK_F3)
			{
			save_snapshot();
			break;
			}

		/* A Numerical key. Add it to the active text box if
		 * there is one */
		else if ((keysym >= XK_0) && (keysym <= XK_9)
			&& active_text_box)
			{
			add_char(mon_win, mon_gc, active_text_box, keybuffer[0]);
			break;
			}
		
		/* A-F keys. Add the character to the active text box if
		 * there is one */
		else if ((keysym >= XK_A) && (keysym <= XK_F)
			&& active_text_box)
			{
			add_char(mon_win, mon_gc, active_text_box, keybuffer[0]);
			break;
			}

		/* lowercase a-f keys. Connvert to uppercase before
		 * insertion into a text box */
		else if ((keysym >= XK_a) && (keysym <= XK_f)
			&& active_text_box)
			{
			add_char(mon_win, mon_gc, active_text_box, (keybuffer[0]-32));
			break;
			}
		
		/* Backspace or delete. Delete previous character from
		 * an active text box */
		else if (((keysym == XK_BackSpace) || (keysym == XK_Delete))
			&& active_text_box)
			{
			del_char(mon_win, mon_gc, active_text_box);
			break;
			}

		/* Carriage return. 
		 * Input is finished in the current text box. */
		else if (((keysym == XK_Return) || (keysym == XK_Linefeed))
			&& active_text_box)
			{
			close_text_box(mon_win, mon_gc, active_text_box);
			active_text_box = 0;
			draw_mon_window(mon_win, mon_gc, mon_font_info);
			update_mon_window(mon_win, mon_gc, mon_font_info);
			}
	
		default : break;
		}
	}
}
}
/* End of main monitor events loop */
/*======================================================================*/

/* TO DO:
 * 
 * Improve save and load routines to specify filenames.
 * Upgrade snapshot idea to allow partial saving of sections of memory.
 * Replace function keys presses with buttons.
 * Allow a changeable memory map.
 */
