#ifndef _MAP_HPP
#define _MAP_HPP

#include "resource.hpp"
#include "fixed.h"

/*
 * MAX_MAP defines the maximum length of one side of the map.  The map
 * array accessible by other modules will ALWAYS be allocated as
 * map[MAX_MAP][MAX_MAP], regardless of the actual map size.
 */

#define MAX_MAP 32

/*
 * BLOCKSIZE_WORLD defines the size of a map block in world coordinates.
 * This must not be changed, as its value is assumed to be 0x10000L several
 * places.
 *
 * BLOCKSIZE_MASK can be used to mask away the map block coordinate from a
 * world coordinate, leaving just the offset within a block.
 */

#define BLOCKSIZE_WORLD (0x10000L)
#define BLOCKSIZE_MASK	(0xFFFFL)

/*
 * WallType defines the available (supported) wall types.
 * NOWALL is no wall, i.e. nothing blocking the view.
 * OPAQUE is a standard wall, where color 0 is drawn (usually black).
 * GLASS is a transparent wall, where color 0 is ignored, to reveal what
 * 		lies behind the wall.
 * MIRROR is a reflecting wall, where color 0 is the reflecting color.
 * If the SOLID bit is set, the wall can be walked through.
 */

enum WallType
	{
	NOWALL=0,
	OPAQUE=1,
	GLASS=2,
	MIRROR=3,

	SOLID=128
	};

/*
 * PlateauDir defines the possible directions of a floor/ceiling bitmap,
 * i.e. the direction which the top edge faces.
 */

enum PlateauDir
	{
	NORTH,
	EAST,
	SOUTH,
	WEST
	};

/*
 * The Wall structure defines the properties of a wall on the map,
 * including bitmap number (possibly NO_BITMAP), and the wall type
 * (possibly NOWALL).
 */

struct Wall
	{
	BitmapNo bmap;
	WallType type;

	void init(int);
	};

/*
 * Plateau defines the properties of a horizontal surface (floor or
 * ceiling) within a map square.  bmap is the bitmap number.  dir
 * is reserved for orienting the bitmap in one of the four possible
 * directions.
 */

struct Plateau
	{
	BitmapNo bmap;
	PlateauDir dir;
	};

/*
 * Square pulls together the above structures to define the properties
 * of a map square.  horz defines the northern wall, vert defines the
 * eastern wall, floor and ceil defines the floor and ceiling respectively.
 * object is reserved for pointing to the first object occupying this
 * Square.
 *
 * NOTE: The Plateau members are IGNORED in this version of Zox3D!
 * See WSLICE.CPP.
 */

struct Square
	{
	Wall horz, vert;
	Plateau floor, ceil;

	void init(int, int);
	};

/*
 * mapWidth and mapHeight are the logical dimensions of the current map.
 * (The physical ones are always [MAX_MAP][MAX_MAP].)
 */

extern int mapWidth, mapHeight;

/*
 * map[][] is an array of Squares, used by other modules.
 */

extern Square map[MAX_MAP][MAX_MAP];

/*
 * initMap() is used to initialize the map array. In this version, the
 * map layout is hardcoded into MAP.CPP.  Eventually, initMap() should
 * read the map layout from a file.
 */

void initMap();

#endif
