-----------------------------------------------------------------------
--
--  File:        vgagraph.adb
--  Description: Basic VGA graphics for GNAT/DOS
--  Rev:         0.7
--  Date:        01-feb-98
--  Author:      Jerry van Dijk
--  Mail:        jdijk@acm.org
--
--  Copyright (c) Jerry van Dijk, 1996, 1997, 1998
--  Billie Holidaystraat 28
--  2324 LK Leiden
--  THE NETHERLANDS
--  tel int + 31 71 531 4365
--
--  Permission granted to use for any purpose, provided this copyright
--  remains attached and unmodified.
--
--  THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
--  IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
--  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
--
-----------------------------------------------------------------------

with Ada.Unchecked_Deallocation;
with DJGPP_Library; use DJGPP_Library;

package body VGA_Graphics is

-------------------
--  SERVICE PART --
-------------------

   -----------------
   --  Constants  --
   -----------------

   --  VGA screen constants
   Video_Base_address           : constant Unsigned_32 := 16#A0000#;
   Bytes_Per_Line               : constant Unsigned_32 := 16#00000050#;

   --  BIOS constants
   BIOS_VGA_Mode                : constant Unsigned_8  := 16#12#;
   BIOS_Video_Interrupt         : constant Unsigned_16 := 16#10#;

   --  VGA controller values
   VGA_Set_Color_Register       : constant Unsigned_8  := 0;
   VGA_Set_Write_Plane_Register : constant Unsigned_8  := 1;
   VGA_Sequential_Register      : constant Unsigned_8  := 2;
   VGA_Put_Rule_Register        : constant Unsigned_8  := 3;
   VGA_Set_Read_Plane_Register  : constant Unsigned_8  := 4;
   VGA_Set_Bitmask_Register     : constant Unsigned_8  := 8;
   VGA_All_Planes               : constant Unsigned_8  := 16#FF#;

   --  VGA controller adresses
   Vga_Seqential_Index          : constant Unsigned_16 := 16#03C4#;
   Vga_Seqential                : constant Unsigned_16 := 16#03C5#;
   Vga_Graphics_Index           : constant Unsigned_16 := 16#03CE#;
   Vga_Graphics                 : constant Unsigned_16 := 16#03CF#;

   --  Default palette colors
   Default_Palette : constant Color_Palette := (16#00#, 16#01#, 16#02#, 16#03#,
                                                16#04#, 16#05#, 16#14#, 16#07#,
                                                16#38#, 16#39#, 16#3A#, 16#3B#,
                                                16#3C#, 16#3D#, 16#3E#, 16#3F#,
                                                16#00#);

   ---------------
   -- Font Type --
   ---------------
   Font_Max_Char   : constant Natural := 255;
   Font_Max_Height : constant Natural := Font_Height - 1;

   type Font_Char is array (0..Font_Max_Height) of Unsigned_8;
   type Font_Type is array (0..Font_Max_Char) of Font_Char;

   ---------------------
   -- Floodfill Stack --
   ---------------------
   Stack_Limit : constant Integer := 196608;
   --  Large enough to fill the whole screen
   --  Pre-allocated for efficienty

   type Floodfill_Stack is array (1 .. Stack_Limit) of Natural;
   type Floodfill_Stack_Access is access all Floodfill_Stack;

   X_Stack : Floodfill_Stack_Access := new Floodfill_Stack;
   Y_Stack : Floodfill_Stack_Access := new Floodfill_Stack;

   ------------------------
   --  Global Variables  --
   ------------------------
   System_Font      : Font_Type;          --  The system font
   Original_Mode    : Unsigned_8;         --  Keep original mode
   Original_Palette : Color_Palette;      --  Keep original palette
   VGA_Status       : Boolean := False;   --  Package status

   -------------------------
   -- Intrinsic Functions --
   -------------------------
   function Shift_Right(Val : Unsigned_8; Count : Natural) return Unsigned_8;
   pragma Import(Intrinsic, Shift_Right);

   function Shift_Right(Val : Unsigned_16; Count : Natural) return Unsigned_16;
   pragma Import(Intrinsic, Shift_Right);

   function Shift_Left(Val : Unsigned_8; Count : Natural) return Unsigned_8;
   pragma Import(Intrinsic, Shift_Left);

   function Shift_Left(Val : Unsigned_16; Count : Natural) return Unsigned_16;
   pragma Import(Intrinsic, Shift_Left);

   function Rotate_Right(Val : Unsigned_16; Count : Natural) return Unsigned_16;
   pragma Import(Intrinsic, Rotate_Right);

   ----------------------------------------------------
   -- NAME:    Set_VGA_Graphics_Register             --
   -- PURPOSE: Write a value to a VGA register       --
   -- INPUT:   Register - The register to write to   --
   --          Value    - The value to write into it --
   ----------------------------------------------------
   procedure Set_VGA_Graphics_Register(Register : in Unsigned_8;
                                       Value    : in Unsigned_8) is
   begin
      Outportb(VGA_Graphics_Index, Register);
      Outportb(VGA_Graphics, Value);
   end Set_VGA_Graphics_Register;
   pragma Inline(Set_VGA_Graphics_Register);

   ---------------------------------------
   -- NAME:    VGA_Set_Write_Plane      --
   -- PURPOSE: Set the new write planes --
   -- INPUT:   Plane - The new planes   --
   ---------------------------------------
   procedure VGA_Set_Write_Plane(Plane : in Unsigned_8) is
   begin
      Set_VGA_Graphics_Register(VGA_Set_Write_Plane_Register, Plane);
   end VGA_Set_Write_Plane;
   pragma Inline(VGA_Set_Write_Plane);

   --------------------------------------
   -- NAME:    VGA_Set_Read_Plane      --
   -- PURPOSE: Set the new read planes --
   -- INPUT:   Plane - The new planes  --
   --------------------------------------
   procedure VGA_Set_Read_Plane(Plane : in unsigned_8) is
   begin
      Set_VGA_Graphics_Register(VGA_Set_Read_Plane_Register, Plane);
   end VGA_Set_Read_Plane;
   pragma Inline(VGA_Set_Read_Plane);

   -------------------------------------------
   -- NAME:    VGA_Set_Sequential_Plane     --
   -- PURPOSE: Set the new sequential plane --
   -- INPUT:   Plane - The new plane        --
   -------------------------------------------
   procedure VGA_Set_Sequential_Plane(Plane : in Unsigned_8) is
   begin
      Outportb(Vga_Seqential_Index, VGA_Sequential_Register);
      Outportb(Vga_Seqential, Plane);
   end VGA_Set_Sequential_Plane;
   pragma Inline(VGA_Set_Sequential_Plane);

   ----------------------------------------------
   -- NAME:    VGA_Set_Put_Rule                --
   -- PURPOSE: Sets the screen overwrite mode  --
   -- INPUTS:  Rule - Mode to use              --
   ----------------------------------------------
   procedure VGA_Set_Put_Rule(Rule : in Put_Rule_Type) is
   begin
      Set_VGA_Graphics_Register(VGA_Put_Rule_Register,
        Put_Rule_Type'ENUM_REP(Rule));
   end VGA_Set_Put_Rule;
   pragma Inline(VGA_Set_Put_Rule);

   -----------------------------------------
   -- NAME:    VGA_Set_Bitmask            --
   -- PURPOSE: Sets the bitmask register  --
   -- INPUTS:  Mask - The new bitmask     --
   -----------------------------------------
   procedure VGA_Set_Bitmask(Mask : in Unsigned_8) is
   begin
      Set_VGA_Graphics_Register(VGA_Set_Bitmask_Register, Mask);
   end VGA_Set_Bitmask;
   pragma Inline(VGA_Set_Bitmask);

   ---------------------------------------------------
   -- NAME:    VGA_Set_Color                        --
   -- PURPOSE: Tell the VGA chip what color to use  --
   -- INPUTS:  Color - The color to use             --
   ---------------------------------------------------
   procedure VGA_Set_Color(Color : in Screen_Color) is
   begin
      Set_VGA_Graphics_Register(VGA_Set_Color_Register,
                                Screen_Color'Pos(Color));
   end VGA_Set_Color;
   pragma Inline(VGA_Set_Color);

   --------------------------------------------------------
   -- NAME:    Calculate_Pixel_Bitmask                   --
   -- PURPOSE: Calculates the bitmask for a single pixel --
   -- INPUTS:  X - Horizontal position                   --
   -- RETURNS: The bitmask                               --
   --------------------------------------------------------
   function Calculate_Pixel_Bitmask(X : in Horizontal_Location)
     return unsigned_8 is
      Mask : constant Unsigned_8 := 2#10000000#;
   begin
      return Shift_Right(Mask, X mod 8);
   end Calculate_Pixel_Bitmask;
   pragma Inline(Calculate_Pixel_Bitmask);

   ---------------------------------------------------------------
   -- NAME:    Calculate_Left_Bytemask                          --
   -- PURPOSE: Calculates a bitmask for the left side of a byte --
   -- INPUTS:  X - Horizontal position                          --
   -- RETURNS: The bytemask                                     --
   ---------------------------------------------------------------
   function Calculate_Left_Bytemask(X : in Horizontal_Location)
     return Unsigned_8 is
      Mask : constant Unsigned_8 := 2#11111111#;
   begin
      return Shift_Right(Mask, X mod 8);
   end Calculate_Left_Bytemask;
   pragma Inline(Calculate_Left_Bytemask);

   ----------------------------------------------------------------
   -- NAME:    Calculate_Right_Bytemask                          --
   -- PURPOSE: Calculates a bitmask for the right side of a byte --
   -- INPUTS:  X - Horizontal position                           --
   -- RETURNS: The bytemask                                      --
   ----------------------------------------------------------------
   function Calculate_Right_Bytemask(X : in Horizontal_Location)
     return Unsigned_8 is
      Mask : constant Unsigned_8 := 2#11111111#;
   begin
      return Shift_Left(Mask, 7 - (X mod 8));
   end Calculate_Right_Bytemask;
   pragma Inline(Calculate_Right_Bytemask);

   ----------------------------------------------------
   -- NAME:    Calculate_Byte_Address                --
   -- PURPOSE: Calculate the byte address of a pixel --
   -- INPUTS:  X - Horizontal position               --
   --          Y - Vertical position                 --
   -- RETURNS: fysical byte address                  --
   ----------------------------------------------------
   function Calculate_Byte_Address(X : in Horizontal_Location;
                                   Y : in Vertical_Location)
     return Unsigned_32 is
   begin
      return Video_Base_Address +
             Unsigned_32(Y) * Bytes_Per_Line +
             Unsigned_32(X / 8);
   end Calculate_Byte_Address;
   pragma Inline(Calculate_Byte_Address);

   -----------------------------------------------------
   -- NAME:    VGA_Get_Byte                           --
   -- PURPOSE: Reads a byte from conventional memory  --
   -- INPUTS:  Address - Physical address of the byte --
   -- RETURNS: The byte                               --
   -----------------------------------------------------
   function VGA_Get_Byte(Address : in Unsigned_32) return Unsigned_8 is
   begin
      return Farpeekb(Dos_DS, Address);
   end VGA_Get_Byte;
   pragma Inline(VGA_Get_Byte);

   -----------------------------------------------------
   -- NAME:    VGA_Get_NS_Byte                        --
   -- PURPOSE: Reads a byte from conventional memory  --
   --          using a preset selector                --
   -- INPUTS:  Address - Physical address of the byte --
   -- RETURNS: The byte                               --
   -----------------------------------------------------
   function VGA_Get_NS_Byte(Address : in Unsigned_32) return Unsigned_8 is
   begin
      return Farnspeekb(Address);
   end VGA_Get_NS_Byte;
   pragma Inline(VGA_Get_NS_Byte);

   -----------------------------------------------------
   -- NAME:    VGA_Put_NS_Byte                        --
   -- PURPOSE: Write a byte to conventional memory    --
   --          using a preset selector                --
   -- INPUTS:  Address - Physical address of the byte --
   --          Value   - The byte to write            --
   -----------------------------------------------------
   procedure VGA_Put_NS_Byte(Address : in Unsigned_32;
                             Value   : in Unsigned_8) is
      Dummy : Unsigned_8; -- To get byte into latch
   begin
      Dummy := VGA_Get_NS_Byte(Address);
      Farnspokeb(Address, Value);
   end VGA_Put_NS_Byte;
   pragma Inline(VGA_Put_NS_Byte);

   -----------------------------------------------------
   -- NAME:    VGA_Put_Byte                           --
   -- PURPOSE: Write a byte to conventional memory    --
   -- INPUTS:  Address - Physical address of the byte --
   --          Value   - The byte to write            --
   -----------------------------------------------------
   procedure VGA_Put_Byte(Address : in Unsigned_32;
                          Value   : in Unsigned_8) is
      Dummy    : Unsigned_8; -- To get byte into latch
   begin
      Dummy := VGA_Get_Byte(Address);
      Farpokeb(Dos_DS, Address, Value);
   end VGA_Put_Byte;
   pragma Inline(VGA_Put_Byte);

   -----------------------------------------
   -- NAME:    VGA_Set_Screen_Selector    --
   -- PURPOSE: Preset the memory selector --
   -----------------------------------------
   procedure VGA_Set_Screen_Selector is
   begin
      Farsetsel(Dos_DS);
   end VGA_Set_Screen_Selector;
   pragma Inline(VGA_Set_Screen_Selector);

   ----------------------------------------------------
   -- NAME:    VGA_Preset_Plot                       --
   -- PURPOSE: Plots a pixel using a preset selector --
   -- INPUTS:  X - Horizontal location               --
   --          Y - Vertical location                 --
   ----------------------------------------------------
   procedure VGA_Preset_Plot(X : in Integer;
                             Y : in Integer) is
   begin
      if X in 0..Horizontal_Maximum and Y in 0..Vertical_Maximum then
         VGA_Set_Bitmask(Calculate_Pixel_Bitmask(X));
         VGA_Put_NS_Byte(Calculate_Byte_Address(X, Y), 16#FF#);
      else
         raise Outside_Screen_Error;
      end if;
   end VGA_Preset_Plot;
   pragma Inline(VGA_Preset_Plot);

   ---------------------------------------
   -- NAME:    Reset_VGA_Registers      --
   -- PURPOSE: Resets the VGA registers --
   ---------------------------------------
   procedure Reset_VGA_Registers is
   begin
      VGA_Set_Color(Screen_Color'Val(15));
      VGA_Set_Bitmask(VGA_All_Planes);
      VGA_Set_Write_Plane(0);
      VGA_Set_Read_Plane(VGA_All_Planes);
      VGA_Set_Sequential_Plane(16#0F#);
   end Reset_VGA_Registers;

   ---------------------------------------------------
   -- NAME:    Check_Proper_Mode                    --
   -- PURPOSE: Check that the system is in VGA mode --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA     --
   ---------------------------------------------------
   procedure Check_Proper_Mode is
   begin
      if VGA_Status = False then
         raise Not_In_VGA_Error;
      end if;
   end Check_Proper_Mode;
   pragma Inline(Check_Proper_Mode);

   ----------------------------------------------------------------
   -- NAME:    Initialize                                        --
   -- PURPOSE: Initialize a Screen_Buffer                        --
   -- INPUTS:  Buffer - Empty Screen_Buffer                      --
   -- OUTPUTS: Buffer - Initialized Screen_Buffer                --
   -- EXCEPTS: Screen_Buffer_Error - if buffer size is wrong     --
   --          Storage_Error       - if no more memory available --
   -- NOTES:   Hidden in a controlled object                     --
   ----------------------------------------------------------------
   procedure Initialize(Buffer : in out Screen_Buffer) is
      Horizontal_Bytes : Integer := Buffer.Width / 8;
   begin
      if Buffer.Width = 0 or Buffer.Height = 0 then
         raise Screen_Buffer_Error;
      else
         if Buffer.Width mod 8 /= 0 then
            Horizontal_Bytes := Horizontal_Bytes + 1;
         end if;
         Buffer.Data := new Planes(1..(4 * Horizontal_Bytes * Buffer.Height));
      end if;
   end Initialize;

   -------------------------------------------------
   -- NAME:    Finalize                           --
   -- PURPOSE: Destroy a Screen_Buffer            --
   -- INPUTS:  Buffer - Initialized Screen_Buffer --
   -- OUTPUTS: Buffer - empty Screen_Buffer       --
   -- NOTES:   Hidden in a controlled object      --
   -------------------------------------------------
   procedure Finalize(Buffer : in out Screen_Buffer) is
      procedure Free is new Ada.Unchecked_Deallocation(Planes, Planes_Access);
   begin
      Free(Buffer.Data);
   end Finalize;

   ---------------------------------------------
   -- NAME:    Current_Video_Mode             --
   -- PURPOSE: Returns the current video mode --
   -- RETURNS: The video mode                 --
   ---------------------------------------------
   function Current_Video_Mode return Unsigned_8 is
      Regs : Dpmi_Regs;
   begin
      Regs.Ax := 16#0F00#;
      Dpmi_Int(BIOS_Video_Interrupt, Regs);
      return Unsigned_8(Regs.Ax and 16#00FF#);
   end Current_Video_Mode;

   -------------------------------------------------
   -- NAME:    Set_Video_Mode                     --
   -- PURPOSE: Set a new video mode               --
   -- NOTES:   1. Does not check if switch worked --
   -------------------------------------------------
   procedure Set_Video_Mode(Mode : in Unsigned_8) is
      Regs : Dpmi_Regs;
   begin
      Regs.Ax := Unsigned_16(Mode);
      Dpmi_Int(BIOS_Video_Interrupt, Regs);
   end Set_Video_Mode;

   ---------------------------------------------
   -- NAME:    Init_System_Font               --
   -- PURPOSE: Reads the system font from ROM --
   -- NOTES:   1. Sets the global font        --
   ---------------------------------------------
   procedure Init_System_Font is
      Regs    : Dpmi_Regs;
      Address : Unsigned_32;
   begin
      Regs.Ax := 16#1130#;
      Regs.Bx := 16#0200#;
      Dpmi_Int(BIOS_Video_Interrupt, Regs);
      Address := 16 * Unsigned_32(Regs.Es) + Unsigned_32(Regs.Bp);
      for Num in 0..Font_Max_Char loop
         For Byte in 0..Font_Max_Height loop
            System_Font(Num)(Byte) := VGA_Get_Byte(Address);
            Address := Address + 1;
         end loop;
      end loop;
   end Init_System_Font;

   ----------------------------------------------------
   -- NAME:    Get_Pixel_Fast                        --
   -- PURPOSE: Get the color of the specified pixel  --
   --          using predefines                      --
   -- INPUTS:  X_Pos - Horizontal pixel position     --
   --          Y_Pos - Vertical pixel position       --
   -- RETURNS: The color of the pixel                --
   ----------------------------------------------------
   function Get_Pixel_Fast(X_Pos : in Horizontal_Location;
                           Y_Pos : in Vertical_Location) return Screen_Color is
      Byte     : Unsigned_8;
      Color    : Unsigned_8           := 0;
      Bit_Mask : constant Unsigned_8  := Calculate_Pixel_Bitmask(X_Pos);
      Address  : constant Unsigned_32 := Calculate_Byte_Address(X_Pos, Y_Pos);
   begin
      VGA_Set_Bitmask(2#11111111#);
      for Plane in reverse 0..3 loop
         Color := Shift_Left(Color, 1);
         VGA_Set_Read_Plane(Unsigned_8(Plane));
         Byte := VGA_Get_NS_Byte(Address);
         if (Byte and Bit_Mask) /= 0 then
            Color := Color or 2#00000001#;
         end if;
      end loop;
      return Screen_Color'VAL(Color);
   end Get_Pixel_Fast;
   pragma Inline (Get_Pixel_Fast);

---------------------------
--  IMPLEMENTATION PART  --
---------------------------

   ------------------------------------------------
   -- NAME:    VGA_Mode                          --
   -- PURPOSE: Switch the display in VGA Mode    --
   -- EXCEPTS: No_VGA_Error - if the call failed --
   ------------------------------------------------
   procedure VGA_Mode is
   begin
      if VGA_Status = False then
         Original_Mode := Current_Video_Mode;
         Set_Video_Mode(BIOS_VGA_Mode);
         if Current_Video_Mode /= BIOS_VGA_Mode then
            raise No_VGA_Error;
         end if;
         VGA_Status := True;
         Original_Palette := Read_All_Palette;
         Write_All_Palette(Default_Palette);
         Init_System_Font;
      end if;
   end VGA_Mode;

   ----------------------------------------------------------------
   -- NAME:    TXT_Mode                                          --
   -- PURPOSE: Return from VGA Mode to original text mode        --
   -- EXCEPTS: Not_In_VGA_Error  - if not in VGA mode            --
   --          No_TXT_Mode_Error - if return to textmode failed  --
   -- NOTES:   1. Always returns to the mode from which the last --
   --             VGA_Mode was called, not to a fixed textmode   --
   --          2. Doesn't remember previous mode changes         --
   ----------------------------------------------------------------
   procedure TXT_Mode is
   begin
      Check_Proper_Mode;
      Write_All_Palette(Original_Palette);
      Reset_VGA_Registers;
      Set_Video_Mode(Original_Mode);
      if Current_Video_Mode /= Original_Mode then
         raise No_TXT_Mode_Error;
      end if;
      VGA_Status := False;
   end TXT_Mode;

   -------------------------------------------------------------------
   -- NAME:    Clear_Screen                                         --
   -- PURPOSE: Clears the screen by filling it with a single color  --
   -- INPUTS:  Color - Color to fill screen with, default is black  --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                --
   -- NOTES:   1. Faster then filling the screen with a loop as it  --
   --             uses a dedicated algorithm                        --
   -------------------------------------------------------------------
   procedure Clear_Screen(Color : in Screen_Color := Black) is
      Dummy     : Unsigned_8; -- To get byte into latch
      Address   : Unsigned_32 := Video_Base_Address;
      Num_Bytes : Positive :=
        (Horizontal_Maximum + 1) * (Vertical_Maximum + 1) / 8;
   begin
      Check_Proper_Mode;
      VGA_Set_Color(Color);
      VGA_Set_Bitmask(2#11111111#);
      VGA_Set_Write_Plane(VGA_All_Planes);
      VGA_Set_Screen_Selector;
      for I in 0..Num_Bytes loop
         Dummy := Farnspeekb(Address);
         Farnspokeb(Address, 16#FF#);
         Address := Address + 1;
      end loop;
      VGA_Set_Write_Plane(0);
   end Clear_Screen;

   ----------------------------------------------------
   -- NAME:    Put_Pixel                             --
   -- PURPOSE: Set a pixel to the specified color    --
   -- INPUTS:  X_Pos - Horizontal pixel position     --
   --          Y_Pos - Vertical pixel position       --
   --          Color - Color to set the pixel to     --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode --
   -- NOTES:   1. Position only garanteed if range   --
   --             checking is enabled                --
   ----------------------------------------------------
   procedure Put_Pixel(X_Pos : in Horizontal_Location;
                       Y_Pos : in Vertical_Location;
                       Color : in Screen_Color) is
   begin
      Check_Proper_Mode;
      VGA_Set_Color(Color);
      VGA_Set_Write_Plane(VGA_All_Planes);
      VGA_Preset_Plot(X_Pos, Y_Pos);
      VGA_Set_Write_Plane(0);
   end Put_Pixel;

   ----------------------------------------------------
   -- NAME:    Get_Pixel                             --
   -- PURPOSE: Get the color of the specified pixel  --
   -- INPUTS:  X_Pos - Horizontal pixel position     --
   --          Y_Pos - Vertical pixel position       --
   -- RETURNS: The color of the pixel                --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode --
   -- NOTES:   1. Result only garanteed if range     --
   --             checking is enabled                --
   ----------------------------------------------------
   function Get_Pixel(X_Pos : in Horizontal_Location;
                      Y_Pos : in Vertical_Location) return Screen_Color is
   begin
      Check_Proper_Mode;
      VGA_Set_Screen_Selector;
      return Get_Pixel_Fast(X_Pos, Y_Pos);
   end Get_Pixel;

   ------------------------------------------------------------------------
   -- NAME:    Vertical_Line                                             --
   -- PURPOSE: Draw a single vertical line                               --
   -- INPUTS:  X_Pos  - Constant horizontal position                     --
   --          First  - Vertical starting point                          --
   --          Last   - Vertical ending point                            --
   --          Color  - Line color                                       --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                     --
   -- NOTES:   1. If First is equal to Last, a single pixel is set       --
   --          2. If First is larger then Last the direction is reversed --
   --          3. Position only garanteed if range checking is enabled   --
   -- WARNING: Clashes with Ada.Characters.Latin_1.Vertical_Line !!!     --
   ------------------------------------------------------------------------
   procedure Vertical_Line(X_Pos : in Horizontal_Location;
                           First : in Vertical_Location;
                           Last  : in Vertical_Location;
                           Color : in Screen_Color) is
      Address  : Unsigned_32;
      Y1       : Vertical_Location   := First;
      Y2       : Vertical_Location   := Last;
      Bit_Mask : constant Unsigned_8 := Calculate_Pixel_bitmask(X_Pos);
   begin
      Check_Proper_Mode;
      if First = Last then
         Put_Pixel(X_Pos, First, Color);
      else
         if First > Last then
            Y1 := Last;
            Y2 := First;
         end if;
      end if;
      VGA_Set_Color(Color);
      VGA_Set_Write_Plane(VGA_All_Planes);
      VGA_Set_Bitmask(Bit_Mask);
      Address := Calculate_Byte_Address(X_Pos, Y1);
      VGA_Set_Screen_Selector;
      for Y in Y1..Y2 loop
         VGA_Put_NS_Byte(Address, 16#FF#);
         Address := Address + Bytes_Per_Line;
      end loop;
      VGA_Set_Write_Plane(0);
   end Vertical_Line;

   ------------------------------------------------------------------------
   -- NAME:    Horizontal_Line                                           --
   -- PURPOSE: Draw a single horizontal line                             --
   -- INPUTS:  First  - Horizontal starting point                        --
   --          Last   - Horizontal ending point                          --
   --          Y_Pos  - Constant vertical position                       --
   --          Color  - Line color                                       --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                     --
   -- NOTES:   1. If First is equal to Last, a single pixel is set       --
   --          2. If First is larger then Last the direction is reversed --
   --          3. Position only garanteed if range checking is enabled   --
   ------------------------------------------------------------------------
   procedure Horizontal_Line(First : in Horizontal_Location;
                             Last  : in Horizontal_Location;
                             Y_Pos : in Vertical_Location;
                             Color : in Screen_Color) is
      Left_Mask  : Unsigned_8;
      Right_Mask : Unsigned_8;
      Address    : Unsigned_32;
      X1         : Natural := First;
      X2         : Natural := Last;
   begin
      Check_Proper_Mode;
      if First = Last then
         Put_Pixel(First, Y_Pos, Color);
      else
         If First > Last then
            X1 := Last;
            X2 := First;
         end if;
         VGA_Set_Color(Color);
         VGA_Set_Screen_Selector;
         VGA_Set_Write_Plane(VGA_All_Planes);
         Left_Mask  := Calculate_Left_Bytemask(X1);
         Right_Mask := Calculate_Right_Bytemask(X2);
         Address    := Calculate_Byte_Address(X1, Y_Pos);
         if (X1 / 8) = (X2 / 8) then
            -- X1 and X2 are within one byte
            VGA_Set_Bitmask(Left_Mask and Right_Mask);
            VGA_Put_NS_Byte(Address, 16#FF#);
         elsif (X1 / 8 + 1) = (X2 / 8) then
            -- X1 and X2 are adjectent bytes
            VGA_Set_Bitmask(Left_Mask);
            VGA_Put_NS_Byte(Address, 16#FF#);
            VGA_Set_Bitmask(Right_Mask);
            VGA_Put_NS_Byte(Address+1, 16#FF#);
         else
            -- X1 and X2 have full bytes in between
            VGA_Set_Bitmask(Left_Mask);
            VGA_Put_NS_Byte(Address, 16#FF#);
            VGA_Set_Bitmask(16#FF#);
            X1 := X1 - X1 mod 8 + 8;
            for I in (X1 / 8)..(X2 / 8 - 1) loop
               Address := Address + 1;
               VGA_Put_NS_Byte(Address, 16#FF#);
            end loop;
            VGA_Set_Bitmask(Right_Mask);
            VGA_Put_NS_Byte(Address+1, 16#FF#);
         end if;
         VGA_Set_Write_Plane(0);
      end if;
   end Horizontal_Line;

   -----------------------------------------------------------------------
   -- NAME:    Draw_Line                                                --
   -- PURPOSE: Draw a single line                                       --
   -- INPUTS:  X1    - Horizontal start position                        --
   --          Y1    - Vertical start position                          --
   --          X2    - Horizontal end position                          --
   --          Y2    - Vertical end position                            --
   --          Color - Line color                                       --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                    --
   -- NOTES:   1. If the line is strictly vertical or horizontal,       --
   --             the thus specialized functions are much faster        --
   --          2. If X1 > X2 or Y1 > Y2 the direction is reversed       --
   --          3. if X1 = X2 and Y1 = Y2 a single pixel is set          --
   --          4. Position only garanteed if range checking is enabled  --
   -----------------------------------------------------------------------
   procedure Draw_Line(X1    : in Horizontal_Location;
                       Y1    : in Vertical_Location;
                       X2    : in Horizontal_Location;
                       Y2    : in Vertical_Location;
                       Color : in Screen_Color) is

      type Direction_Type is (Normal, Reversed);

      Step             : Integer;
      Step_X, Step_Y   : Integer;
      Delta_X, Delta_Y : Integer;
      Y_Pos            : Integer := Y1;
      X_Pos            : Integer := X1;
      Direction        : Direction_Type := Normal;

      procedure Swap(A : in out Integer;
                     B : in out Integer) is
         Temp : constant Integer := A;
      begin
         A := B;
         B := Temp;
      end Swap;
      pragma Inline(Swap);

   begin
      Check_Proper_Mode;
      VGA_Set_Color(Color);
      VGA_Set_Screen_Selector;
      Delta_X := abs(X2 - X_Pos);
      Delta_Y := abs(Y2 - Y_Pos);
      VGA_Set_Write_Plane(VGA_All_Planes);
      if X2 > X_Pos then
         Step_X := 1;
      else
         Step_X := -1;
      end if;
      if Y2 > Y_Pos then
         Step_Y := 1;
      else
         Step_Y := -1;
      end if;
      if Delta_Y > Delta_X then
         Swap(X_Pos, Y_Pos);
         Swap(Step_X, Step_Y);
         Swap(Delta_X, Delta_Y);
         Direction := Reversed;
      end if;
      Step := 2 * Delta_Y - Delta_X;
      for I in 1..Delta_X loop
         if Direction = Reversed then
            VGA_Preset_Plot(Y_Pos, X_Pos);
         else
            VGA_Preset_Plot(X_Pos, Y_Pos);
         end if;
         while Step >= 0 loop
            Y_Pos   := Y_Pos   + Step_Y;
            Step := Step - 2 * Delta_X;
         end loop;
         X_Pos   := X_Pos   + Step_X;
         Step := Step + 2 * Delta_Y;
      end loop;
      VGA_Preset_Plot(X2, Y2);
      VGA_Set_Write_Plane(0);
   end Draw_Line;

   ---------------------------------------------------------------
   -- NAME:    Draw_Circle                                      --
   -- PURPOSE: Draw a full single line circle                   --
   -- INPUTS:  X_Pos  - Horizontal position of center           --
   --          Y_Pos  - Vertical position of center             --
   --          Radius - Circle radius in pixels                 --
   --          Color  - Circle line color                       --
   -- EXCEPTS: Not_In_VGA_Error     - if not in VGA mode        --
   --          Outside_Screen_Error - writes outside the screen --
   ---------------------------------------------------------------
   procedure Draw_Circle(X_Pos  : in Horizontal_Location;
                         Y_Pos  : in Vertical_Location;
                         Radius : in Positive;
                         Color  : in Screen_Color) is
      X    : Integer := 0;
      Y    : Integer := Radius;
      Diff : Integer := 2 * (1 - Radius);
   begin
      Check_Proper_Mode;
      VGA_Set_Color(Color);
      VGA_Set_Screen_Selector;
      VGA_Set_Write_Plane(VGA_All_Planes);
      VGA_Preset_Plot(X_Pos + Radius, Y_Pos);
      VGA_Preset_Plot(X_Pos - Radius, Y_Pos);
      While Y > 0 loop
         VGA_Preset_Plot(X_Pos + X, Y_Pos + Y);
         VGA_Preset_Plot(X_Pos + X, Y_Pos - Y);
         VGA_Preset_Plot(X_Pos - X, Y_Pos + Y);
         VGA_Preset_Plot(X_Pos - X, Y_Pos - Y);
         if Diff + Y > 0 then
            Y    := Y - 1;
            Diff := Diff - 2 * Y - 1;
         end if;
         if X > Diff then
            X    := X + 1;
            Diff := Diff + 2 * X + 1;
         end if;
      end loop;
      VGA_Set_Write_Plane(0);
   end Draw_Circle;

   ----------------------------------------------------------------------
   -- NAME:    Fill_Box                                                --
   -- PURPOSE: Fills a rectangular box with a single color             --
   -- INPUTS:  X1    - Horizontal start position                       --
   --          Y1    - Vertical start position                         --
   --          X2    - Horizontal end position                         --
   --          Y2    - Vertical end position                           --
   --          Color - Fill color                                      --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                   --
   -- NOTES:   1. If X1 = X2 and Y1 = Y2 a single pixel is set         --
   --          2. If X1 = X2 or Y1 = Y2 a single line is drawn         --
   --          3. If X1 > X2 or Y1 > Y2 the direction is reversed      --
   --          4. Position only garanteed if range checking is enabled --
   ----------------------------------------------------------------------
   procedure Fill_Box(X1    : in Horizontal_Location;
                      Y1    : in Vertical_Location;
                      X2    : in Horizontal_Location;
                      Y2    : in Vertical_Location;
                      Color : in Screen_Color) is

      Left_Mask  : Unsigned_8;
      Right_Mask : Unsigned_8;
      Address    : Unsigned_32;
      Top_Pos    : Vertical_Location   := Y1;
      Left_Pos   : Horizontal_Location := X1;
      Right_Pos  : Horizontal_Location := X2;
      Bottom_Pos : Vertical_Location   := Y2;

      procedure Draw_Strip(Y1      : in Vertical_Location;
                           Y2      : in Vertical_Location;
                           Address : in Unsigned_32) is
         Memory : Unsigned_32 := Address;
      begin
         for Y in Y1..Y2 loop
            VGA_Put_NS_Byte(Memory, 16#FF#);
            Memory := Memory + Bytes_Per_Line;
         end loop;
      end Draw_Strip;
      Pragma Inline(Draw_Strip);

   begin
      Check_Proper_Mode;
      if X1 = X2 and Y1 = Y2 then
         Put_Pixel(X1, Y1, Color);
      elsif X1 = X2 then
         Vertical_Line(X1, Y1, Y2, Color);
      elsif Y1 = Y2 then
         Horizontal_Line(X1, X2, Y1, Color);
      else
         if X1 > X2 then
            Left_Pos := X2;
            Right_Pos := X1;
         end if;
         if Y1 > Y2 then
            Top_Pos := Y2;
            Bottom_Pos := Y1;
         end if;
         VGA_Set_Color(Color);
         VGA_Set_Screen_Selector;
         VGA_Set_Write_Plane(VGA_All_Planes);
         Left_Mask  := Calculate_Left_Bytemask(Left_Pos);
         Right_Mask := Calculate_Right_Bytemask(Right_Pos);
         Address    := Calculate_Byte_Address(Left_Pos, Top_Pos);
         if (Left_Pos / 8) = (Right_Pos / 8) then
            -- Left_Pos and Right_Pos are within one byte
            VGA_Set_Bitmask(Left_Mask and Right_Mask);
            Draw_Strip(Top_Pos, Bottom_Pos, Address);
         elsif (Left_Pos / 8 + 1) = (Right_Pos / 8) then
            -- Left_Pos and Right_Pos are adjectent bytes
            VGA_Set_Bitmask(Left_Mask);
            Draw_Strip(Top_Pos, Bottom_Pos, Address);
            VGA_Set_Bitmask(Right_Mask);
            Draw_Strip(Top_Pos, Bottom_Pos, Address+1);
         else
            -- Left_Pos and Right_Pos have full bytes in between
            VGA_Set_Bitmask(Left_Mask);
            Draw_Strip(Top_Pos, Bottom_Pos, Address);
            VGA_Set_Bitmask(16#FF#);
            Left_Pos := Left_Pos - Left_Pos mod 8 + 8;
            for I in (Left_Pos / 8)..(Right_Pos / 8 - 1) loop
               Address := Address + 1;
               Draw_Strip(Top_Pos, Bottom_Pos, Address);
            end loop;
            VGA_Set_Bitmask(Right_Mask);
            Draw_Strip(Top_Pos, Bottom_Pos, Address+1);
         end if;
         VGA_Set_Write_Plane(0);
      end if;
   end Fill_Box;

   -----------------------------------------------------------------
   -- NAME:    Flood_Fill                                         --
   -- PURPOSE: Fills a screen region with a single color          --
   -- INPUTS:  X     - Horizontal position within region          --
   --          Y     - Vertical position within region            --
   --          Color - Fill color                                 --
   -- EXCEPTS: Not_In_VGA_Error  - if not in VGA mode             --
   --          Flood_Fill_Failed - Region not completedly filled  --
   -- NOTES:   1. Works best on small areas                       --
   --          2. Large areas are slow because of stack unwinding --
   -----------------------------------------------------------------
   procedure Flood_Fill(X     : in Horizontal_Location;
                        Y     : in Vertical_Location;
                        Color : in Screen_Color) is
      type Neighbours is (Right, Up, Left, Down);
      Start_Color    : Screen_Color;
      Stack_Pointer  : Integer  := 2;
      Y_Start, Y_Pos : Vertical_Location;
      X_Start, X_Pos : Horizontal_Location;
      X_Neighbour    : constant array (Neighbours) of Integer := ( 1, 0, -1, 0);
      Y_Neighbour    : constant array (Neighbours) of Integer := ( 0, 1, 0, -1);
   begin
      Check_Proper_Mode;
      VGA_Set_Color(Color);
      VGA_Set_Screen_Selector;
      VGA_Set_Write_Plane(VGA_All_Planes);
      Start_Color := Get_Pixel_Fast(X, Y);
      VGA_Preset_Plot(X, Y);
      X_Stack(1) := X;
      Y_Stack(1) := Y;
      while Stack_Pointer > 1 loop
         Stack_Pointer := Stack_Pointer - 1;
         X_Start := X_Stack(Stack_Pointer);
         Y_Start := Y_Stack(Stack_Pointer);
         for Pixel in Neighbours'Range loop
            X_Pos := X_Start + X_Neighbour(Pixel);
            Y_Pos := Y_Start + Y_Neighbour(Pixel);
            if X_Pos in Horizontal_Location'Range and
               Y_Pos in Vertical_Location'Range then
               if Get_Pixel_Fast(X_Pos, Y_Pos) = Start_Color then
                  VGA_Preset_Plot(X_Pos, Y_Pos);
                  X_Stack(Stack_Pointer) := X_Pos;
                  Y_Stack(Stack_Pointer) := Y_Pos;
                  Stack_Pointer := Stack_Pointer + 1;
                  if Stack_Pointer = Stack_Limit then
                     raise Flood_Fill_Failed;
                  end if;
               end if;
            end if;
         end loop;
      end loop;
      VGA_Set_Write_Plane(0);
   end Flood_Fill;

   ---------------------------------------------------------------
   -- NAME:    Write_String                                     --
   -- PURPOSE: Writes a string in the system font               --
   -- INPUTS:  X_Pos - Horizontal start position                --
   --          Y_Pos - Vertical start position                  --
   --          Color - The text color                           --
   --          Text  - The string to write                      --
   -- EXCEPTS: Not_In_VGA_Error     - if not in VGA mode        --
   --          Outside_Screen_Error - writes outside the screen --
   -- NOTES:   1. X_Pos, Y_Pos is upper-left position           --
   --          2. Position only garanteed if range checking is  --
   --             enabled                                       --
   ---------------------------------------------------------------
   procedure Write_String(X_Pos : in Horizontal_Location;
                          Y_Pos : in Vertical_Location;
                          Color : in Screen_Color;
                          Text  : in String) is

      procedure Plot_Line(X       : in Horizontal_Location;
                          Line    : in Integer;
                          Address : in Unsigned_32;
                          Text    : in String) is

         Mask      : Unsigned_8;
         Count     : Natural          := 1;
         Register  : Unsigned_16      := 0;
         Offset    : constant Integer := X_Pos mod 8;
         Shift_Val : constant Integer := 8 - Offset;
         Screen    : Unsigned_32      := Address +
           Unsigned_32(Line) * Bytes_Per_Line;
      begin
         loop
            if X + Count * 8 + 8 <= Horizontal_Maximum then
               Register := Register and 16#FF00#;
               Register := Register or
                 unsigned_16(System_Font(Character'Pos(Text(Count)))(Line));
               Register := Rotate_Right(Register, Offset);
               Mask := Unsigned_8(Register and 16#00FF#);
               VGA_Set_Bitmask(Mask);
               VGA_Put_NS_Byte(Screen, 16#FF#);
               Register := Shift_Right(Register, Shift_Val);
            end if;
            Screen := Screen + 1;
            Count  := Count + 1;
            exit when Count > Text'Length;
         end loop;
         if X + Count * 8 + 8 <= Horizontal_Maximum then
            Register := Rotate_Right(Register, Offset);
            Mask := Unsigned_8(Register and 16#00FF#);
            VGA_Set_Bitmask(Mask);
            VGA_Put_Byte(Screen, 16#FF#);
         end if;
      end Plot_Line;
      pragma Inline(Plot_Line);

      Address : Unsigned_32;

   begin
      Check_Proper_Mode;
      VGA_Set_Color(Color);
      VGA_Set_Screen_Selector;
      VGA_Set_Write_Plane(VGA_All_Planes);
      Address := Calculate_Byte_Address(X_Pos, Y_Pos);
      for Y in 0..Font_Max_Height loop
         if Y_Pos + Y <= Vertical_Maximum then
            Plot_Line(X_Pos, Y, Address, Text);
         end if;
      end loop;
      VGA_Set_Write_Plane(0);
   end;

   ----------------------------------------------------------------------
   -- NAME:    Read_All_Palette                                        --
   -- PURPOSE: Returns the current palette                             --
   -- RETURNS: The palette                                             --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                   --
   --          Storage_Error    - if unable to allocate a conventional --
   --                             memory transfer buffer               --
   ----------------------------------------------------------------------
   function Read_All_Palette return Color_Palette is
      Regs   : Dpmi_Regs;
      Result : Color_Palette;
      Temp   : char_array(0..Screen_Color'POS(Screen_Color'LAST))
        := (others => nul);
   begin
      Check_Proper_Mode;
      if Current_Info.Size_Of_Transfer_Buffer < Color_Palette'Length then
         raise Storage_Error;
      end if;
      Regs.Ax := 16#1009#;
      Regs.Es := Unsigned_16(
        Shift_Right(Current_Info.Linear_Address_Of_Transfer_Buffer, 4)
        and 16#FFFF#);
      Regs.Dx := Unsigned_16(
        Current_Info.Linear_Address_Of_Transfer_Buffer and 16#0F#);
      Dpmi_Int(BIOS_Video_Interrupt, Regs);
      Dosmemget(
        Offset => Current_Info.Linear_Address_Of_Transfer_Buffer,
        Length => Temp'LENGTH,
        Buffer => Temp);
      for I in Color_Palette'RANGE loop
         Result(I) := Character'POS(To_Ada(Temp(Screen_Color'POS(I))));
      end loop;
      return Result;
   end Read_All_Palette;

   ----------------------------------------------------------------------
   -- NAME:    Write_All_Palette                                       --
   -- PURPOSE: Makes the palette the current palette                   --
   -- INPUTS:  Palette - The color palette to set                      --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                   --
   --          Storage_Error    - if unable to allocate a conventional --
   --                             memory transfer buffer               --
   ----------------------------------------------------------------------
   procedure Write_All_Palette(Palette : in Color_Palette) is
      Regs : Dpmi_Regs;
      Temp : char_array(0..Screen_Color'POS(Screen_Color'LAST));
   begin
      Check_Proper_Mode;
      if Current_Info.Size_Of_Transfer_Buffer < Color_Palette'LENGTH then
         raise Storage_Error;
      end if;
      for I in Screen_Color'RANGE loop
         Temp(Screen_Color'POS(I)) := To_C(Character'VAL(Integer(Palette(I))));
      end loop;
      Dosmemput(
        Offset => Current_Info.Linear_Address_Of_Transfer_Buffer,
        Length => Temp'LENGTH,
        Buffer => Temp);
      Regs.Ax := 16#1002#;
      Regs.Es := Unsigned_16(
        Shift_Right(Current_Info.Linear_Address_Of_Transfer_Buffer, 4)
        and 16#FFFF#);
      Regs.Dx := Unsigned_16(
        Current_Info.Linear_Address_Of_Transfer_Buffer and 16#0F#);
      Dpmi_Int(BIOS_Video_Interrupt, Regs);
   end Write_All_Palette;

   ---------------------------------------------------------------------
   -- NAME:    Set_Palette                                            --
   -- PURPOSE: Sets a palette color to a new value                    --
   -- INPUTS:  Palette - The color palette to change                  --
   --          Color   - The color value within the palette to change --
   --          Value   - The new value for the color                  --
   -- OUTPUTS: Palette - Loaded with current color palette            --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                  --
   -- NOTES:   1. See any PC reference how to code a RGB color value  --
   --          2. Changing the palette invalidates screen color names --
   ---------------------------------------------------------------------
   procedure Set_Palette(Palette : in out Color_Palette;
                         Color   : in     Screen_Color;
                         Value   : in     Color_Value) is
   begin
      Check_Proper_Mode;
      Palette(Color) := Value;
   end Set_Palette;

   ----------------------------------------------------------------------
   -- NAME:    Get_Palette                                             --
   -- PURPOSE: Gets an individual palette color from a palette         --
   -- INPUTS:  Palette - The color palette to read                     --
   --          Color   - The color value within the palette to read    --
   -- RETURNS: The color value                                         --
   -- EXCEPTS: Not_In_VGA_Error - if not in VGA mode                   --
   -- NOTES:   1. See any PC reference how to decode a RGB color value --
   ----------------------------------------------------------------------
   function Get_Palette(Palette : in Color_Palette;
                        Color   : in Screen_Color)
                        return Color_Value is
   begin
      Check_Proper_Mode;
      Return Color_Value(Palette(Color));
   end Get_Palette;

   -------------------------------------------------------------------------
   -- NAME:    Get_Buffer                                                 --
   -- PURPOSE: Copy a screen area to a Screen_Buffer                      --
   -- INPUTS:  Buffer - The screen buffer                                 --
   --          Left   - Left position of screen area                      --
   --          Top    - Top position of screen area                       --
   -- OUTPUTS: Buffer - Loaded with area                                  --
   -- EXCEPTS: Not_In_VGA_Error     - if not in VGA mode                  --
   --          Outside_Screen_Error - Area outside the screen             --
   -- NOTES:   1. Area size is defined when creating the Screen_Buffer    --
   --          2. Position only garanteed if range checking is enabled    --
   -------------------------------------------------------------------------
   procedure Get_Buffer(Buffer : in out Screen_Buffer;
                        Left   : in     Horizontal_Location;
                        Top    : in     Vertical_Location) is
      Bits_Left : Integer;
      Register  : Unsigned_16;
      Address   : Unsigned_32;
      Count     : Integer := 1;
      Offset    : constant Natural := Left mod 8;
      Rest_Bits : constant Natural := 8 - Offset;
   begin
      Check_Proper_Mode;
      if Left + Buffer.Width > Horizontal_Maximum or
        Top + Buffer.Height > Vertical_Maximum then
         raise Outside_Screen_Error;
      else
         VGA_Set_Bitmask(16#FF#);
         VGA_Set_Screen_Selector;
         for Plane in reverse 0..3 loop
            VGA_Set_Read_Plane(Unsigned_8(Plane));
            for Line in Top..Top + Buffer.Height - 1 loop
               Address := Calculate_Byte_Address(Left, Line);
               Register := Shift_Left(
                 unsigned_16(VGA_Get_NS_Byte(Address)), 8);
               Address := Address + 1;
               Register := Register or Unsigned_16(VGA_Get_NS_Byte(Address));
               Address := Address + 1;
               Register := Shift_Left(Register, Offset);
               Buffer.Data(Count) := Unsigned_8(Shift_Right(Register, 8));
               Count := Count + 1;
               Bits_Left := Buffer.Width - 8;
               While Bits_Left > 0 loop
                  Register := Shift_Left(Register, Rest_Bits);
                  Register := Register and 16#FF00#;
                  Register := Register or
                    Unsigned_16(VGA_Get_NS_Byte(Address));
                  Address := Address + 1;
                  Register := Shift_Left(Register, Offset);
                  Buffer.Data(Count) := Unsigned_8(Shift_Right(Register, 8));
                  Count := Count + 1;
                  Bits_Left := Bits_Left - 8;
               end loop;
            end loop;
         end loop;
      end if;
   end Get_Buffer;

   ----------------------------------------------------------------------
   -- NAME:    Put_Buffer                                              --
   -- PURPOSE: Copy a Screen_Buffer to the screen                      --
   -- INPUTS:  Buffer - The screen buffer                              --
   --          Left   - Left position of screen area                   --
   --          Top    - Top position of screen area                    --
   --          Rule   - Screen buffer put mode                         --
   -- EXCEPTS: Not_In_VGA_Error     - if not in VGA mode               --
   --          Outside_Screen_Error - Writin outside the screen        --
   -- NOTES:   1. Area size is defined when creating the Screen_Buffer --
   --          2. Position only garanteed if range checking is enabled --
   ----------------------------------------------------------------------
   procedure Put_Buffer(Buffer : in Screen_Buffer;
                        Left   : in Horizontal_Location;
                        Top    : in Vertical_Location;
                        Rule   : in Put_Rule_Type := Put_Force) is
      Bits_Left   : Integer;
      Left_Mask   : Unsigned_8;
      Right_Mask  : Unsigned_8;
      Register    : Unsigned_16;
      Address     : Unsigned_32;
      Count       : Integer := 1;
      Left_Offset : constant Integer := Left mod 8;
      Rest_Value  : constant Integer := 8 - Left_Offset;
   begin
      Check_Proper_Mode;
      if Left + Buffer.Width > Horizontal_Maximum or
        Top + Buffer.Height > Vertical_Maximum then
         raise Outside_Screen_Error;
      else
         VGA_Set_Put_Rule(Rule);
         VGA_Set_Screen_Selector;
         Left_Mask  := Shift_Right(16#FF#, Left_Offset);
         Right_Mask := Unsigned_8(Shift_Right(Unsigned_16 (16#FF00#),
           ((Left + Buffer.Width - 1) mod 8) + 1) and 16#00FF#);
         if Left / 8 = (Left + Buffer.Width - 1) / 8 then
            Left_Mask := Left_Mask and Right_Mask;
         end if;
         for Plane in reverse 0..3 loop
            VGA_Set_Sequential_Plane(Shift_Left(Unsigned_8(1), Plane));
            for Line in Top..Top + Buffer.Height - 1 loop
               Address := Calculate_Byte_Address(Left, Line);
               Register := Unsigned_16(Buffer.Data(Count));
               Count := Count + 1;
               Register := Rotate_Right(Register, Left_Offset);
               VGA_Set_Bitmask(Left_Mask);
               VGA_Put_NS_Byte(Address, Unsigned_8(Register and 16#00FF#));
               Address := Address + 1;
               Bits_Left := Buffer.Width - Rest_Value;
               VGA_Set_Bitmask(16#FF#);
               While Bits_Left >= 8 loop
                  Register := Shift_Right(Register, Rest_Value);
                  Register := Register and 16#FF00#;
                  Register := Register or Unsigned_16(Buffer.Data(Count));
                  Count := Count + 1;
                  Bits_Left := Bits_Left - 8;
                  Register := Rotate_Right(Register, Left_Offset);
                  VGA_Put_NS_Byte(Address, Unsigned_8(Register and 16#00FF#));
                  Address := Address + 1;
               end loop;
               VGA_Set_Bitmask(Right_Mask);
               if Bits_Left > 0 then
                  if Bits_Left > Left_Offset then
                     Register := Shift_Right(Register, Rest_Value);
                     Register := Register and 16#FF00#;
                     Register := Register or Unsigned_16(Buffer.Data(Count));
                     Count := Count + 1;
                     Register := Rotate_Right(Register, Left_Offset);
                     VGA_Put_NS_Byte(Address,
                       Unsigned_8(Register and 16#00FF#));
                  else
                     VGA_Put_NS_Byte(Address,
                       Unsigned_8(Shift_Right(Register, 8) and 16#00FF#));
                  end if;
               end if;
            end loop;
         end loop;
         VGA_Set_Put_Rule(Put_Force);
         VGA_Set_Sequential_Plane(16#0F#);
      end if;
   end Put_Buffer;

end VGA_Graphics;
