WITH Ada.Text_IO;
WITH Ada.Float_Text_IO;
WITH Screen;
PROCEDURE Home_Budget IS
------------------------------------------------------------------
--| Prints a summary of all expenses by budget category.   
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: July 1995                                     
------------------------------------------------------------------

  MaxExpense :  CONSTANT Float := 10_000.00;            -- max expense amount

  TYPE Categories IS (Entertainment, Food, Clothing, Rent,
                      Tuition, Insurance, Miscellaneous);
  TYPE Commands IS (E, F, C, R, T, I, M, Q);

  PACKAGE Category_IO IS NEW Ada.Text_IO.Enumeration_IO(Enum => Categories);
  PACKAGE Command_IO  IS NEW Ada.Text_IO.Enumeration_IO(Enum => Commands);

  SUBTYPE Expenses IS Float RANGE 0.00..MaxExpense;     -- expense type   
  TYPE BudgetArray IS ARRAY (Categories) OF Expenses;   -- array type   

  Budget : BudgetArray;                 -- array of ten totals   

  PROCEDURE Initialize (Budget : OUT BudgetArray) IS
  -- Pre:  None
  -- Post: Each array element Budget(Category) is 0.00
     
  BEGIN  -- Initialize   

    Budget := (OTHERS => 0.00);

  END Initialize;


  PROCEDURE DisplayTitles IS
  -- Pre:  None
  -- Post: Displays a list of expense categories with abbreviations

    WhichRow: Screen.Depth;

  BEGIN

    Screen.ClearScreen;

    Screen.MoveCursor(Row => 3, Column => 20);
    Ada.Text_IO.Put(Item => "Expense Categories");
    Ada.Text_IO.New_Line;
    Ada.Text_IO.New_Line;

    WhichRow := 5;
    FOR C IN Commands'First..Commands'Pred(Commands'Last) LOOP
      Screen.MoveCursor(Row => WhichRow, Column => 20);
      Command_IO.Put(Item => C, Width => 3);
      Category_IO.Put(Item => Categories'Val(Commands'Pos(C)));
      WhichRow := WhichRow + 1;
    END LOOP;

    Screen.MoveCursor(Row => WhichRow, Column => 20);
    Command_IO.Put(Item => Commands'Last, Width => 3);
    Ada.Text_IO.Put(Item => "when data entry is completed");

  END DisplayTitles;

  PROCEDURE GetCommand(Command: OUT Commands) IS
  -- Pre:  None
  -- Post: A valid Command is returned in Command

  BEGIN

    LOOP
      BEGIN    -- exception handler block
        Screen.MoveCursor(Row => 18, Column => 15);
        Ada.Text_IO.Put("Please enter first letter of category > ");
        Command_IO.Get(Item => Command);
        Screen.MoveCursor(Row => 19, Column => 15);
        Ada.Text_IO.Put("Category accepted, thank you");
        EXIT;
      EXCEPTION
        WHEN Ada.Text_IO.Data_Error =>
          Screen.Beep;
          Screen.MoveCursor(Row => 19, Column => 15);
          Ada.Text_IO.Put("Sorry, invalid category!    ");
          Ada.Text_IO.Skip_Line;
      END;     -- exception handler block
    END LOOP;
    -- assert: valid command input received

  END GetCommand;

  PROCEDURE GetExpense(Expense: OUT Expenses) IS
  -- Pre:  None
  -- Post: a valid Expense is returned in Expense

  BEGIN

    LOOP
      BEGIN    -- exception handler block
        Screen.MoveCursor(Row => 20, Column => 15);
        Ada.Text_IO.Put("Please enter expense as floating point number > ");
        Ada.Float_Text_IO.Get(Item => Expense);
        Screen.MoveCursor(Row => 21, Column => 15);
        Ada.Text_IO.Put("Expense accepted, thank you");
        EXIT;
      EXCEPTION
        WHEN Ada.Text_IO.Data_Error =>
          Screen.Beep;
          Screen.MoveCursor(Row => 21, Column => 15);
          Ada.Text_IO.Put("Sorry, invalid expense!    ");
          Ada.Text_IO.Skip_Line;
      END;     -- exception handler block
    END LOOP;
    -- assert: valid expense received

  END GetExpense;

  PROCEDURE Post (Budget : IN OUT BudgetArray) IS
  -- Pre:  Each array element Budget(c) is 0.0
  -- Post: Each array element Budget(c) is the sum of expense
  --        amounts for category c.
     
    Sentinel : CONSTANT Commands := Q; -- sentinel command    

    NextCommand :  Commands;          -- command
    NextCategory : Categories;        -- expenditure category   
    NextExpense :  Expenses;          -- expenditure amount   

  BEGIN  -- Post   

    LOOP
      -- invariant:
      -- no prior value of NextCommand  is Sentinel
         
      GetCommand(Command => NextCommand);
      EXIT WHEN NextCommand = Sentinel;

      NextCategory := Categories'Val(Commands'Pos(NextCommand));
      GetExpense(Expense => NextExpense);
      Budget(NextCategory) := Budget(NextCategory) + NextExpense;
    END LOOP;   

  END Post;

  PROCEDURE Report (Budget : IN BudgetArray) IS
  -- Pre:  Each array element Budget(c) is assigned a value.
  -- Post: Each array element Budget(c) is displayed.

    WhichRow: Screen.Depth;
     
  BEGIN  -- Report   

    Screen.ClearScreen;

    Screen.MoveCursor(Row => 3, Column => 20);
    Ada.Text_IO.Put(Item => "Category      Expense");
    Ada.Text_IO.New_Line;
    Ada.Text_IO.New_Line;

    WhichRow := 5;
    FOR Category IN Categories LOOP
      Screen.MoveCursor(Row => WhichRow, Column => 20);
      Category_IO.Put(Item => Category, Width=>13);   -- Print row   
      Ada.Float_Text_IO.Put(Item=>Budget(Category), 
                    Fore=>7, Aft=>2, Exp=>0);  
      WhichRow := WhichRow + 1;
    END LOOP;

    Screen.MoveCursor(Row => 23,Column => 1);

  END Report;

BEGIN  -- Home_Budget   

  -- prepare terminal screen for data entry
  DisplayTitles;

  -- Initialize array Budget to all zeros.   
  Initialize (Budget);

  -- Read and process each expenditure.   
  Post (Budget);

  -- Print the expenditures in each category.   
  Report (Budget);

END Home_Budget;
