WITH Ada.Float_Text_IO;
WITH Ada.Text_IO;
PACKAGE BODY Geometry.IO IS
------------------------------------------------------------------
--| Body of Input/Output Package for Geometric Figures
--| Author: Michael B. Feldman, The George Washington University
--| Last Modified: September 1995            
------------------------------------------------------------------

  MaxSize: CONSTANT NonNegFloat := 1_000_000.0;
  
  PACKAGE FigKind_IO IS NEW Ada.Text_IO.Enumeration_IO (Enum => FigKind);

  -- Local procedure ReadShape and RobustGet are used only within 
  -- the package, therefore not exported.

  PROCEDURE ReadShape (Item : OUT FigKind) IS 
  -- Pre:  none
  -- Post: Item contains a figure kind. ReadShape reads robustly.
  
    TempItem: FigKind;

  BEGIN -- ReadShape
            
    LOOP
      BEGIN
        Ada.Text_IO.Put(Item => "Enter a shape: rectangle, circle, square > ");
        FigKind_IO.Get(Item => TempItem);
        Item := TempItem;
        EXIT;
      EXCEPTION
        WHEN Ada.Text_IO.Data_Error =>
          Ada.Text_IO.Put ("Value not a valid shape. Please try again.");
          Ada.Text_IO.New_Line;
          Ada.Text_IO.Skip_Line;
      END;
    END LOOP;
    -- assert: Item is rect, circle, or square

  END ReadShape;

  PROCEDURE RobustGet (Item : OUT NonNegFloat;
                 MinVal : IN NonNegFloat; 
                 MaxVal : IN NonNegFloat) IS
  -- Pre:  MinVal and MaxVal are defined
  -- Post: MinVal <= Item <= MaxVal

    SUBTYPE TempType IS NonNegFloat RANGE MinVal..MaxVal;
    TempItem : TempType;     -- temporary copy of MinVal   

  BEGIN -- RobustGet
            
    LOOP
      BEGIN       -- exception handler block
        Ada.Text_IO.Put(Item => "Enter a floating-point value between ");
        Ada.Float_Text_IO.Put(Item => MinVal, Fore=> 1, Aft => 2, Exp => 0);  
        Ada.Text_IO.Put(Item => " and ");
        Ada.Float_Text_IO.Put(Item => MaxVal, Fore=> 1, Aft => 2, Exp => 0);  
        Ada.Text_IO.Put(Item => " > ");
        Ada.Float_Text_IO.Get(Item => TempItem);
        Item := TempItem;
        EXIT;     -- valid data
      EXCEPTION   -- invalid data
        WHEN Constraint_Error =>
          Ada.Text_IO.Put ("Value entered is out of range. Please try again.");
          Ada.Text_IO.New_Line;
          Ada.Text_IO.Skip_Line;
        WHEN Ada.Text_IO.Data_Error =>
          Ada.Text_IO.Put ("Value entered not floating point. Please try again.");
          Ada.Text_IO.New_Line;
          Ada.Text_IO.Skip_Line;
      END;        -- exception handler block
    END LOOP;
    -- assert: Item is in the range MinVal to MaxVal   

  END RobustGet;

  PROCEDURE Get (Item : OUT Figure) IS

    Shape  : FigKind;
    Height : NonNegFloat;
    Width  : NonNegFloat;
    Side   : NonNegFloat;
    Radius : NonNegFloat;

  BEGIN  -- Get   

    -- Read the shape character and define the discriminant
    ReadShape(Shape);

    -- Select the proper variant and read pertinent data   
    CASE Shape IS
      WHEN Rectangle =>
        Ada.Text_IO.Put(Item => "Enter width.");
        Ada.Text_IO.New_Line;
        RobustGet(Item => Width, MinVal => 0.0, MaxVal => MaxSize);   
        Ada.Text_IO.Put(Item => "Enter height.");
        Ada.Text_IO.New_Line;
        RobustGet(Item => Height, MinVal => 0.0, MaxVal => MaxSize);
        Item := MakeRectangle(Width, Height);
   
      WHEN Square    =>
        Ada.Text_IO.Put(Item => "Enter length of side.");
        Ada.Text_IO.New_Line;
        RobustGet(Item => Side, MinVal => 0.0, MaxVal => MaxSize);   
        Item := MakeSquare(Side);

      WHEN Circle    =>
        Ada.Text_IO.Put(Item => "Enter circle radius.");
        Ada.Text_IO.New_Line;
        RobustGet(Item => Radius, MinVal => 0.0, MaxVal => MaxSize);
        Item := MakeCircle(Radius);

    END CASE;

  END Get;
    
  PROCEDURE Put (Item: IN Figure) IS
    
  BEGIN -- DisplayFigure   
    
    -- Display shape and characteristics   
    Ada.Text_IO.Put(Item => "Figure shape: ");
    FigKind_IO.Put(Item => Shape(Item), Width => 1);
    Ada.Text_IO.New_Line;

    CASE Item.FigShape IS
      WHEN Rectangle =>
        Ada.Text_IO.Put(Item => "height = ");
        Ada.Float_Text_IO.Put(Item => Height(Item), Fore=>1, Aft=>2, Exp=>0);
        Ada.Text_IO.Put(Item => "; width = ");
        Ada.Float_Text_IO.Put(Item => Width(Item), Fore=>1, Aft=>2, Exp=>0);

      WHEN Square =>
        Ada.Text_IO.Put(Item => "side = ");
        Ada.Float_Text_IO.Put(Item => Height(Item), Fore=>1, Aft=>2, Exp=>0);
        
      WHEN Circle =>
        Ada.Text_IO.Put(Item => "radius = ");
        Ada.Float_Text_IO.Put(Item => Radius(Item), Fore=>1, Aft=>2, Exp=>0);
        
    END CASE;

    Ada.Text_IO.Put(Item => "; perimeter = ");
    Ada.Float_Text_IO.Put(Item => Perimeter(Item), Fore=>1, Aft=>2, Exp=>0);
    Ada.Text_IO.Put(Item => "; area = ");
    Ada.Float_Text_IO.Put(Item => Area(Item), Fore=>1, Aft=>2, Exp=>0);
    Ada.Text_IO.New_Line;

  END Put;

END Geometry.IO;
