/* arjopt.c - mini-menu for ARJ.EXE, 1991 Mike McCombs */

/*------------------------------------------------------------------------
- Full compilation requires:                                             -
-                           ARJOPT.C                                     -
-                           ARJOPT.H                                     -
-                           EDIT.C                                       -
-                           GLOBBER.C                                    -
-                           GLOBBER.H                                    -
------------------------------------------------------------------------*/

/*/////////////////////////// HISTORY /////////////////////////////////*/
/* 05/20/92 McCombs - added harderror_handler() to catch drive errors; */
/*                    requiring additional code in dup() and execute() */
/* 05/08/92 McCombs - rewrote execute() to restart ARJOPT with original*/
/*                    parameters for user to view changes (if any);    */
/*                    also added support to main() for saving params   */
/*                    User must now intentionally quit to exit.        */
/*                  - added pause() call to duplicate() so user could  */
/*                    see what happened since it no longer exits       */
/*                  - modified exit routine for new situation          */
/* 05/07/92 McCombs - added pause() to allow complete viewing with the */
/*                    list command                                     */
/* 05/05/92 McCombs - replaced WILDARGS.OBJ with GLOBBER.C for greatly */
/*          Kercheval enhanced wildcard parsing. Now does UNIX/sh      */
/*          Jung      style globbing. Rewrote main() and added parse().*/
/*                    Substituted Robert Jung's buf_upper() for the    */
/*                    original toupper() of John Kercheval             */
/*                  - changed usage_display() to reflect changes       */
/*                  - changed mode of usage_display() - now on -h or -?*/
/*                  - changed default (no command line) to all-files   */
/* 04/25/92 McCombs - added file code to set newfile date/time to      */
/*                    oldfile date/time in duplicate() for copy and    */
/*                    move commands                                    */
/* 04/14/92 McCombs - added check to prevent file deletion if not      */
/*                    copied in move command                           */
/* 04/11/92 McCombs - added space availability checking for copy and   */
/*                    move commands - will try other files if one will */
/*                    not fit                                          */
/*                  - added warning if unable to delete a file in      */
/*                    delete or move command                           */
/*                  - enabled "renaming" of file for move command if   */
/*                    target on same drive                             */
/* 02/22/92 McCombs - increased copy buffer size for faster copying    */
/*                  - added Build_SFX to Add menu                      */
/* 10/06/91 McCombs - set array size to 1K for extended screen modes   */
/* 10/05/91 McCombs - changed exit() to _exit() for error exits        */
/*                  - added test for no file for non-wildcard entries  */
/* 10/04/91 McCombs - added set_video() to test current video mode     */
/* 10/03/91 McCombs - finished port to Borland C++. Lost alphabetising */
/*                    from Microsoft, so changed first file-first      */
/*                    letter search to NEXT file-first letter search   */
/*                  - added letter_upper() for foreign support - taken */
/*                    from buf_upper by Rob Jung                       */
/* 10/02/91 McCombs - added textmode() to adjust for VPIX and similar  */
/*                    DOS windows                                      */
/* 10/01/91 McCombs - adjusted multi_exec() which was failing to pass  */
/*                    command properly                                 */
/*                  - added _chmod() test to get_files() to avoid some */
/*                    sub-directory display problems                   */
/* 09/27/91 McCombs - added "all-direction" rollover to select window  */
/*                  - added esc as == to tab in select()               */
/*                  - added esc a "quit current task" to box_command() */
/* 09/26/91 Jung    - added check_key() and default to select()        */
/*                  - added enter as == to space in select()           */
/* 09/25/91 McCombs - first beta version complete                      */
/*/////////////////////////////////////////////////////////////////////*/

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <dos.h>
#include <io.h>
#include <dir.h>
#include <fcntl.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <string.h>
#include <alloc.h>
#include <process.h>
#include <signal.h>
#include "arjopt.h"
#include "globber.h"

/* character array and globals used throughout */
static int cerror_flag = 0; /* status of hard error function */
static char *args[128];     /* temp storage for argv[] */
static char **ary;          /* directory listing */
static int count;           /* number of entries in ary[] */
int LAST_FILE, SCRN_HEIGHT; /* for variable video mode */
char user_info[81];         /* user requested input */
struct text_info mode_info; /* video mode structure */

int harderror_handler(int err_value, int ax, int bp, int si)
{  /* what to do if the drive door is open disk protected, etc. */
	int key, s = 0;

	cerror_flag = 1;
	printf(" - Unable to access drive %c:", toupper(user_info[0]));
	printf("\nPlease correct and press ENTER to continue or press ESC to exit.");
	while (!s)
	{
		key = getkey(); /* defined in edit.c, no use doing it again */
		switch (key)
		{
		case KEY_ENTER: /* user decides to stay and try again */
			s = 1;
			break;
		case KEY_ESCAPE: /* user decides to call it quits */
			s = 2;
			break;
		default:   /* anything else simply doesn't work */
			break;
		}
	}
	if(s == 2)
		hardresume(2); /* DOS */
	else
		hardretn(1); /* program */
	return(s);
}
/* Hard error handlers have to be able to handle device and media errors */
/* This one does nothing but return to the program at the line after the */
/* one that caused it. At those points, it is tested for, and if envoked,*/
/* the program is simply recycled through the failed portion after the   */
/* user is given an opportunity to correct the problem. This works well  */
/* for both drive and printer errors, though there are no print options  */
/* in ARJOPT - note the language in harderror_handler() is specific to   */
/* floppy drives. Since these are almost always wrong drive, open door or*/
/* faulty media problems, no further explanation is given - though, of   */
/* course, it is possible to do so. These are personal design decisions, */
/* feel free to modify to your own taste. I use simple "goto" statements */
/* as they are clear in this context. I don't otherwise recommend them.  */

void display_usage(void) /* usage display */
{
	clrscr();
	printf("\nARJOPT 1.41A menu/directory for ARJ.EXE - Copyright (c) 1992 Michael D McCombs\n");
	printf("\nUSAGE:       ARJOPT [ -h | -? ] ( {FILENAME} )\n");
	printf("\n             The file specification may be any UNIX-like wildcard ");
	printf("\n             combination for directory display. More than one   ");
	printf("\n             expression may be used; i.e.: ARJOPT *.EXE VOL??.*");
	printf("\n             or ARJOPT t[abc-fjl]*.? [a-e]?[!t] (See DOC for details.)\n");
	printf("\nControl keys:\n");
	printf("\n             TAB    - toggle windows (directory and command)");
	printf("\n             SPACE  - select/deselect a file (directory)");
	printf("\n             ENTER  - select a command (command) or file (directory)");
	printf("\n             ESCAPE - escape current task (both)");
	printf("\n             Arrows - move around the screen (both)");
	printf("\n             Letter - go to a command (command) or file (directory)");
	printf("\n\nStandard editing keys work while entering information requested.\n");
	exit(0);
}

void say_done (void) /* exit printout */
{
	printf("\n");
	printf("\nARJOPT task completed.  Goodbye...     \n");
}

void set_video(void)    /* get and set mode and colors */
{
	gettextinfo(&mode_info);
	SCRN_HEIGHT = mode_info.screenheight;
	if(SCRN_HEIGHT <= 25)      /* "normal" 25 x 80 text mode */
		textmode(3);
	else                       /* VGA or EGA extended line mode */
		textmode(64);
	textcolor(7);
	textbackground(1);
	clrscr();
}

void reset_video(void) /* reset to original */
{
	textmode(-1);
	clrscr();
}

void credits_line(void) /* put credit and directory line on screen */
{
	char directory[64];
	char dirout[80];
	int len, i;

	gotoxy(1, 1);
	cprintf("ARJOPT version 1.41A");

	/* get cwd and build directory line */
	getcwd(directory, 64);
	sprintf(dirout, " %s ", directory);
	len = strlen(dirout);
	for(i = len; i < 80;i++)
		dirout[i] = '';
	dirout[80] = '\0';

	/* put it to screen */
	gotoxy(1, 2);
	cprintf("%s", dirout);

	return;
}

void make_bar(void)
{
	/* display breakbars */
	cprintf("͹");
	gotoxy(41, 2);
	cprintf("");
	gotoxy(41, 3);
	cprintf("");
	return;
}

void parse(char * pathname) /* expands ambiguous (wildcard) entries */
{
	 struct file_info_struct ff; /* the file find structure */
	 int len, result, i = 0;

	 /* initialize ff */
	 strcpy( ff.file_pattern,pathname );
	 ff.file_attributes = 0x40 /* normal */ | FA_RDONLY | FA_ARCH |
								 FA_SYSTEM | FA_HIDDEN;
/* attributes are variable - this set shows everything except directories */
/* and volume labels - can be changed for future expansion                */

	 /* find the initial file matching pattern */
	 if ( find_firstfile( &ff ) ) {

		  /* loop through all matching files in the parameter */
		  do {
				len = strlen(ff.file.name);
				ary[count] = (char *)malloc(16);       /* plug into ary */
				memcpy(ary[count], ff.file.name, len + 1);
				len = strlen(ary[count]);
				for(i = len; i < 15; i++)
					ary[count][i] = ' ';
				ary[count][15] = '\0';
				for(i = 0; i < count; i++)
				{                       /* check if found previously */
					result = memcmp(ary[i], ary[count], 15);
					if(result == 0)      /* if so, overwrite it */
						goto restart;
				}
				count++;
				restart:
		  } while ( find_nextfile( &ff ) );
	 }
}

void get_files(void) /* display directory to screen */
{
	int i = 0, j, k;

	/* set up display */
	window(1, 3, 80, SCRN_HEIGHT - 3);

	/* display up to 5 columns of right number of files each */
	for(k = 1; k < 80; k = k + 16)  /* makes columns */
	{
		for(j = 1; i < count && j < SCRN_HEIGHT - 4; j++, i++)
		{                            /* makes entries to columns */
			gotoxy(k, j);
			cprintf("%s", ary[i]);
		}
	}
	LAST_FILE = i;    /* IMPORTANT - else overflows and */
							/* causes havoc in select() */
	return;
}

void pause(void)  /* display a pause, wait for user input */
{
	int s = 0;
	unsigned int key;

	printf("\n\nPress ENTER to continue:\n");
	while (!s)
	{
		  key = getkey(); /* defined in edit.c, no use doing it again */
		  switch (key)
		  {
					 case KEY_ENTER:
								s = 1;
								break;
					 default:
								break;
		  }
	}
	return;
}

char letter_upper(char t)   /* upper case after checking with country () */
{                           /* small version of buf_upper() by Rob Jung  */
	 static void far (*case_map)(void);
	 static int init_flag = 0;
	 struct COUNTRY cp;

	 if (!init_flag)
	 {
		init_flag = -1;
		if (country(0, &cp) != NULL)
		{
			init_flag = 1;
			case_map =
		MK_FP((unsigned int) ((unsigned long) cp.co_case >> 16), (unsigned int) cp.co_case);
		}
	 }

	 /* Modify the following code with extreme care */

	 if (init_flag > 0)
	 {
		 _AL = t;
		 if (_AL >= 'a' && _AL <= 'z')
			t = toupper(_AL);
		 else if (_AL >= 0x80)
		 {
			(*case_map)();
			t = _AL;
		 }
	 }
	 else
	 {
		 _AL = t;
		 if (_AL >= 'a' && _AL <= 'z')
			t = toupper(_AL);
	 }
	 return(t);
}

int check_key(char *command, unsigned key) /* check for command first letter */
{                                          /* added by Rob Jung              */
	 int pos;

	 key = letter_upper((char)key);
	 pos = 0;
	 for (pos = 0; ; pos = next_word(command, pos))
	 {
	 if (command[pos] == (char) key)
		 return pos;
	 if (command[pos] == 'Q' || command[pos] == 'R')
		 break;
	 }
	 return -1;
}

void instruct (int type) /* instruction boxes */
{
	window(43, SCRN_HEIGHT - 1, 80, SCRN_HEIGHT);
	switch(type)
	{
	case 1:  /* for command box entry */
		gotoxy(1, 1);
		cprintf("ENTER selects command, ESC Quits     ");
		gotoxy(1, 2);
		cprintf("TAB toggles to file window           ");
		break;
	case 2:  /* for file selection */
		gotoxy(1, 1);
		cprintf("SPACE selects/deselects file         ");
		gotoxy(1, 2);
		cprintf("TAB toggles to command window        ");
		break;
	case 3: /* for user_input() */
		gotoxy(1, 1);
		cprintf("ESCAPE aborts entry, ENTER ends entry");
		gotoxy(1, 2);
		cprintf("Arrows, Home, End, and Ins are active");
		break;
	case 4:  /* for delete_query() */
		gotoxy(1, 1);
		cprintf("\"No\" is default                      ");
		gotoxy(1, 2);
		cprintf("                                     ");
		break;
	}
	return;
}

void select(void) /* select and deselect files */
{
	int quit_flag = 0;
	int i = 0, j = 1, k = 1, l = 0; /* i is ary line number, j and k positions */
	unsigned key, key1;             /* and l is "previous file" */
	int a, b;                       /* for rollover calculations */
	struct text_info save_position;   /* position save for searches */

	/* put on instructions */
	instruct(2);

	/* set window and start position */
	window(1, 3, 80, SCRN_HEIGHT - 3);
	gotoxy(j, k);

	/* get figures for rollover */
	a = (LAST_FILE)/(SCRN_HEIGHT - 5);
	b = (LAST_FILE)%(SCRN_HEIGHT - 5);

	/* main select loop */
	while(!quit_flag)
	{
	textcolor(15);
	gotoxy(j, k);
	cprintf("%s", ary[i]);
	gotoxy(j, k);

	/* key switch */
	key = getkey();
	advance:
	switch(key)
	{
		case KEY_RIGHT:
			l = i;                   /* set file to unhighlight */
			if(i < LAST_FILE - (SCRN_HEIGHT - 5))  /* if there's a file to the */
			{                                      /* right, advance indeces */
				i = i + (SCRN_HEIGHT - 5);
				j = j + 16;
			}
			else
			{                        /* otherwise, roll to col 1 */
				i = k - 1;
				j = 1;
			}
			break;
		case KEY_LEFT:
			l = i;
			if(i >= (SCRN_HEIGHT - 5)) /* if there's a file to the */
			{                          /* left, decrease indeces */
				i = i - (SCRN_HEIGHT - 5);
				j = j - 16;
			}
			else
			{                        /* else, go to right column */
				if(k <= b)
				{
					i = ((SCRN_HEIGHT - 5) * a) + i;
					j = (16 * a) + 1;
				}
				else
				{
					i = ((SCRN_HEIGHT - 5) * (a - 1)) + i;
					j = (16 * (a - 1)) + 1;
				}
			}
			break;
		case KEY_UP:
			l = i;
			if(i == 0)        /* first file, go to last file */
			{
				i = LAST_FILE;
				k = b + 1;
				j = (16 * a) + 1;
			}
			if(k > 1)                  /* if not top of column */
			{
				i--;
				k--;
			}
			else                       /* from top of column */
			{
				i--;
				j = j - 16;
				k = (SCRN_HEIGHT - 5);
			}
			break;
		case KEY_DN:
			l = i;
			if(i == LAST_FILE - 1)    /* opposite of KEY_UP */
			{
				i = 0;
				k = 1;
				j = 1;
				break;
			}
			if(k == (SCRN_HEIGHT - 5))
			{
				i++;
				j = j + 16;
				k = 1;
			}
			else
			{
				i++;
				k++;
			}
			break;
		case ' ':
		case '\r':
			if (ary[i][14] == '\33')      /* if marked, unmark */
				ary[i][14] = ' ';
			else
				ary[i][14] = '\33';        /* if unmarked, mark */
			key = KEY_DN;
			goto advance;                 /* restart switch - advances cursor */
		case KEY_ESCAPE:
		case KEY_TAB:                    /* leave loop (goto command)*/
			l = i;
			quit_flag = 1;
			break;
		default:
			gettextinfo(&save_position);   /* remember where you start */
			key1 = letter_upper((char)key);/* capitalize letter */
			l = i;
			if(i == LAST_FILE - 1)         /* advance one space */
			{
				i = 0;
				k = 1;
				j = 1;
			}
			else if(k == (SCRN_HEIGHT - 5))
			{
				i++;
				j = j + 16;
				k = 1;
			}
			else
			{
				i++;
				k++;
			}
			while(key1 != ary[i][0]) /* if character,   */
			{                        /* search for file */
				if(i == l)            /* starting with it*/
					break;
				if(i == LAST_FILE - 1)
				{
					i = 0;
					k = 1;
					j = 1;
				}
				else if(k == (SCRN_HEIGHT - 5))
				{
					i++;
					j = j + 16;
					k = 1;
				}
				else
				{
					i++;
					k++;
				}
			}
			if(key1 != ary[i][0])  /* if not found */
			{
				j = save_position.curx;
				k = save_position.cury;
				gotoxy(j, k);
				i = l;
				printf("\7");  /* let user know he hit bad key */
			}
			break;
	}
	textcolor(7);
	cprintf("%s", ary[l]);     /* unhighlight last file */
	gotoxy(j, k);
	}
	/* set up for command_box() */
	window(1, SCRN_HEIGHT - 3, 80, SCRN_HEIGHT);
	return;
}

void user_input(int input_type) /* get user_info */
{
	int i;

	instruct(3);
	window(1, SCRN_HEIGHT - 2, 80, SCRN_HEIGHT);

	/* one of two types of info */
	gotoxy(2, 2);
	if(input_type == 1)
	cprintf("Enter archive name:                 ");
	else
	cprintf("Enter directory name:               ");
	gotoxy(2, 3);

	for(i = 0; i < 39; i++)
	user_info[i] = ' ';
	cprintf("%s", user_info);
	gotoxy(2, 3);
	(void)editline(user_info);
	return;
}

int duplicate(char *oldfile, char command) /* copy oldfile to newfile */
{
	FILE *In, *Out;
	int nr, len, drive;
	int in, out, result;
	int oldhandle, newhandle;
	unsigned long spaceA;
	char newfile[64];
	char buffer[10240];
	struct ffblk ffblk;
	struct dfree dtab;
	struct ftime datetime;

	/* build newfile name depending on entry ending: ':', '\', or other */
	len = strlen(user_info);
	if(user_info[len - 1] == '\\' || user_info[len - 1] == ':')
		sprintf(newfile,"%s%s", user_info, oldfile);
	else
		sprintf(newfile, "%s\\%s", user_info, oldfile);

	/* get current file size and target room */
	if(user_info[1] == ':') drive = toupper(user_info[0]) - 65;
	else drive = getdisk();

	dup_retry:
	getdfree(drive + 1, &dtab);

	if(cerror_flag)   /* check if attempted access caused hard error */
	{
		cerror_flag = 0;  /* if it did, try again */
		printf("\nRetrying...");
		goto dup_retry;
	}

	spaceA = (unsigned long)dtab.df_avail *
		((unsigned long)dtab.df_sclus * (unsigned long)dtab.df_bsec);

	findfirst(oldfile, &ffblk, 0);

	/* abort file copy if not enough room */
	if(spaceA < (unsigned long)ffblk.ff_fsize)
	{
		printf(" - Not enough room in target directory");
		return(-1); /* but go back and try the next one, do NOT delete */
	}

	/* try renaming (on same drive) first, if a move command */
	if(command == 'M')
	{
		result = rename(oldfile, newfile);
		if(result == 0) return(-1); /* deletion not needed if successful */
	}

	/* else,  open oldfile */
	in = open(oldfile, O_RDONLY | O_BINARY);
	if(in < 0)
	{
		reset_video();
		printf("\n\nCan't open %s to copy it\n", oldfile);
		_exit(1);
	}
	/* open newfile */
	out = open(newfile, O_WRONLY | O_CREAT | O_BINARY, S_IREAD | S_IWRITE);
	if(out < 0)
	{
		reset_video();
		printf("\n\nCan't write %s\n", newfile);
		_exit(1);
	}
	/* copy oldfile to buffer then to newfile */
	nr = _read(in, buffer, 10240);
	while(nr > 0)
	{
		_write(out, buffer, nr);
		nr = _read(in, buffer, 10240);
	}
	close(in);
	close (out);

	/* set newfile datetime group to oldfile datetime */
	In = fopen(oldfile, "r");
	oldhandle = fileno(In);
	getftime(oldhandle, &datetime);

	Out = fopen(newfile, "r+");
	newhandle = fileno(Out);
	result = setftime(newhandle, &datetime);
	if(result == -1)
		printf(" - Could not preserve old date...");

	fclose(In);
	fclose(Out);
	return(0);
}

void make_arjopt(void) /* put marked files into ARJOPT.$$$ */
{
	int len, i, j;
	FILE *temp;

	if((temp = fopen("ARJOPT.$$$", "a")) == NULL)
	{
		reset_video();
		printf("\n\nCan't open scratch file\n");
		_exit(1);
	}
	for(i = 0; i < LAST_FILE; i++)
	{
	if(ary[i][14] == '\33')
	{
		len = strlen(ary[i]);
		for(j = len - 1; ary[i][j] == ' ' || ary[i][j] == '\33'; j--)
			ary[i][j] = '\0';
		fprintf(temp, "%s\n", ary[i]);
	}
	}
	flushall();
	fclose(temp);
	return;
}

void multi_exec(char command) /* multiple executions of ARJ for l, e, x */
{
	int len, i, j;

	reset_video();
	printf("\n\n");
	for(i = 0; i < LAST_FILE; i++)
	{
	if(ary[i][14] == '\33')
	{
		 len = strlen(ary[i]);
		 for(j = len - 1; ary[i][j] == ' ' || ary[i][j] == '\33'; j--)
			  ary[i][j] = '\0';
		 printf("ARJ %c %s\n", command, ary[i]);
		 if(command == 'l')
		 {
			spawnlp(P_WAIT, "ARJ.EXE", "ARJ", "l", ary[i], NULL);
			pause();
		 }
		 else if (command == 'e')
			spawnlp(P_WAIT, "ARJ.EXE", "ARJ", "e", ary[i], NULL);
		 else
			spawnlp(P_WAIT, "ARJ.EXE", "ARJ", "x", ary[i], NULL);
	}
	}
	return;
}

void copy_exec(char command) /* multiple executions for d, m, c */
{
	int len, i, j, k;

	reset_video();
	printf("\n\n");
	for(i = 0; i < LAST_FILE; i++)
	{
	if(ary[i][14] == '\33')
	{
		 len = strlen(ary[i]);
		 for(j = len - 1; ary[i][j] == ' ' || ary[i][j] == '\33'; j--)
			ary[i][j] = '\0';
		 if(command == 'M')
		 {
			printf("\nMoving %s to (directory) %s", ary[i], user_info);
			k = duplicate(ary[i], 'M');
			if(k == 0)  /* skip deletion if not copied or renaming worked*/
			{
				k = remove(ary[i]);
				if(k == -1) printf(" - Unable to delete file!");
			}
		 }
		 else if(command == 'C')
		 {
			 printf("\nCopying %s to (directory) %s", ary[i], user_info);
			 duplicate(ary[i], 'C');
		 }
		 else /* command = "D" */
		 {
			 printf("\nDeleting %s", ary[i]);
			 k = remove(ary[i]);
			 if(k == -1) printf(" - Unable to delete file!");
		 }
	 }
	 }
	 pause(); /* let the user see what was done */
	 return;
}

void execute(char key) /* perform proper action depending upon command */
{
	int i, go_ahead = 0, exit_flag = 0;
	FILE *test;

	/* test for escape from user_input() */
	if(user_info[0] == '\30')
	{
		user_info[0] = '\0';
		return;
	}

	/* test if file(s) selected */
	for(i = 0; i < LAST_FILE; i++)
	{
		if(ary[i][14] == '\33')
			go_ahead = 1;
	}
	if(!go_ahead)
	{
	textcolor(15);
	gotoxy(2, 2);
	cprintf("No file(s) selected.                   ");
	textcolor(7);
	gotoxy(2, 3);
	cprintf("Press any key to continue              ");
	getch();
	return;
	}

	reset_video();

	/* test drive if user_input specifies a floppy drive */
	if((user_info[1] == ':' && toupper(user_info[0]) == 'A') ||
	  (user_info[1] == ':' && toupper(user_info[0]) == 'B'))
	{
		exec_retry:
		test = fopen(user_info, "w");
		if(cerror_flag) /* test if caused hard error */
		{
			cerror_flag = 0;
			printf("\nRetrying...");
			goto exec_retry;
		}
		else
		{
			fclose(test);
			unlink(user_info);
		}
	}

	/* execute according to command */
	switch(key)
	{
	case 'A':
		make_arjopt();
		printf("\n\nARJ a %s !ARJOPT.$$$\n", user_info);
		spawnlp(P_WAIT, "ARJ.EXE", "ARJ", "a", user_info, "!ARJOPT.$$$", NULL);
		unlink("ARJOPT.$$$");
		exit_flag = 1;
		break;
	case 'B':
		make_arjopt();
		printf("\n\nARJ a -je %s !ARJOPT.$$$\n", user_info);
		spawnlp(P_WAIT, "ARJ.EXE", "ARJ", "a",  "-je", user_info, "!ARJOPT.$$$", NULL);
		unlink("ARJOPT.$$$");
		exit_flag = 1;
		break;
	case 'E':
		multi_exec('e');
		exit_flag = 1;
		break;
	case 'L':
		multi_exec('l');
		exit_flag = 1;
		break;
	case 'S':
		make_arjopt();
		printf("\n\nARJ m %s !ARJOPT.$$$\n", user_info);
		spawnlp(P_WAIT, "ARJ.EXE", "ARJ", "m", user_info, "!ARJOPT.$$$", NULL);
		unlink("ARJOPT.$$$");
		exit_flag = 1;
		break;
	case 'C':
		copy_exec('C');
		exit_flag = 1;
		break;
	case 'M':
		copy_exec('M');
		exit_flag = 1;
		break;
	case 'D':
		copy_exec('D');
		exit_flag = 1;
		break;
	case 'X':
		multi_exec('x');
		exit_flag = 1;
		break;
	}
	if(exit_flag)                  /* this "recycles" ARJOPT with same     */
		execvp("ARJOPT.EXE", args); /* parameters to show changes - remove  */
		/*exit(0);*/                /* execvp() line and uncomment exit()   */
	else                           /* to quit after each action. return is */
		return;                     /* unused and there for future changes  */
}

void delete_query(void) /* "are you sure you want to do this?" */
{
	unsigned key;

	instruct(4);
	window(1, SCRN_HEIGHT - 2, 80, SCRN_HEIGHT);

	/* set display */
	gotoxy(2, 2);
	cprintf("                                   ");
	gotoxy(1, 3);
	cprintf("                                   ");
	gotoxy(2, 2);
	cprintf("Delete marked files? (Y/ )");
	textcolor(15);
	gotoxy(26, 2);
	cprintf("N");
	textcolor(7);
	gotoxy(26, 2);

	key = getkey();
	switch(key)
	{
	case 'Y':
	case 'y':
		execute('D');
		break;
	case KEY_ENTER:
	case KEY_ESCAPE:
	case 'N':
	case 'n':
		break;
	default:
		printf("\7");
		break;
	}
	return;
}

void explain(char key) /* display proper command explanation */
{
	gotoxy(2, 3);
	cprintf("                                       ");
	gotoxy(2, 3);
	textcolor(7);
	switch(key)
	{
	case 'A':
		cprintf("Add selected file(s) to archive");
		break;
	case 'B':
		cprintf("Build file(s) into self-extractor");
		break;
	case 'E':
		cprintf("Extract files from selected archive");
		break;
	case 'L':
		cprintf("List files in selected archive");
		break;
	case 'S':
		cprintf("Move selected file(s) to archive");
		break;
	case 'M':
		cprintf("Move selected file(s) elsewhere");
		break;
	case 'R':
		cprintf("Return to main menu");
		break;
	case 'C':
		cprintf("Copy selected file(s)");
		break;
	case 'D':
		cprintf("Delete selected file(s)");
		break;
	case 'X':
		cprintf("Extract with full path(s)");
		break;
	case 'Q':
		cprintf("Quit ARJOPT");
		break;
	}
	return;
}

void sub_command_box(char key) /* display proper sub-menu */
{
	char add_commands[] = {"Add  Build_SFX  Store  Return "};
	char ext_commands[] = {"Extract  Xtract  Return "};
	char copy_commands[] = {"Copy  Move  Return "};

	/* set display */
	gotoxy(2, 2);
	cprintf("                                   ");

	switch(key)
	{
	case 'A':
		command_box(add_commands, 1);
		break;
	case 'E':
		command_box(ext_commands, 1);
		break;
	case 'C':
		command_box(copy_commands, 1);
		break;
	}
	return;
}

void command_box(char *commands, int exec_flag) /* handle command menus */
{
	int pos = 0;
	int quit_flag = 0;
	int i, len;
	unsigned key;

	/* set up display */
	window(1, SCRN_HEIGHT - 2, 80, SCRN_HEIGHT);
	make_bar();
	len = strlen(commands);

	/* display commands */
	gotoxy(2, 2);
	cprintf("                              ");
	textcolor(7);
	gotoxy(2, 2);
	cprintf("%s", commands);

/* start in select file window  - remove if statement and change start() */
/* call to command_box() to "command_box(main_commands, 0)" to start in  */
/* command window                                                        */
	if(exec_flag == 2)
	{
	select();
	exec_flag = 0;
	}

	/* main command loop */
	while(!quit_flag)
	{
	/* place instructions */
	instruct(1);
	window(1, SCRN_HEIGHT - 2, 80, SCRN_HEIGHT);

	/* display commands */
	gotoxy(2, 2);
	cprintf("                              ");
	gotoxy(2, 2);
	cprintf("%s", commands);

	/* hilight current command */
	textcolor(15);
	i = pos;
	while (commands[i] != ' ')
	{
		gotoxy(2 + i, 2);
		cprintf("%c", commands[i]);
		i++;
	}
	gotoxy(2 + pos, 2);

	/* display proper explanation */
	explain(commands[pos]);
	gotoxy(2 + pos, 2);

	/* command switch */
	key = getkey();
	switch(key)
	{
		case KEY_ESCAPE:
			exec_flag = 0;
			quit_flag = 1;
			break;
		case KEY_RIGHT:
			if(commands[pos] == 'Q' || commands[pos] == 'R')
				pos = 0;
			else
				pos = next_word(commands, pos);
			break;
		case KEY_LEFT:
			if(pos == 0)
			{
				pos = len ;
				pos = prev_word(commands, pos);
			}
			else
				pos = prev_word(commands, pos);
			break;
		case KEY_TAB:
			select();
			break;
		case KEY_ENTER:
			if(commands[pos] == 'Q' || commands[pos] == 'R')
			{
				exec_flag = 0;
				quit_flag = 1;
				break;
			}
			else if(commands[pos] == 'L')
				execute(commands[pos]);
			else if(commands[pos] == 'D')
				delete_query();
			if(exec_flag == 1)   /* if set by sub_command_box() */
			{
				if(commands[pos] == 'A' || commands[pos] == 'S' || commands[pos] == 'B')
					user_input(1);
				else
					user_input(2);
				gotoxy(1, 3);
				execute(commands[pos]);
			}
			else
				sub_command_box(commands[pos]);
			break;
		default:
			i = check_key(commands, key);
			if (i < 0)
				 printf("\7");
			else
				 pos = i;
			break;
	}
	}
	return;
}

void start(void)
{
	char main_commands[] = {"Add  Extract  List  Copy  Del  Quit "};

	set_video();                    /* set video mode per user default */
	credits_line();                 /* print the creditline */
	get_files();                    /* get and print wildcards */
	harderr(harderror_handler);     /* set hard error handler */
	command_box(main_commands, 2);  /* print and enter command area */
}

int main(int argc, char *argv[])  /* get args and setup */
{
	int i, j;

	/* save command line for future use in reloading */
	for(i = 0; i < argc; i++)
		args[i] = argv[i];

	/* allocate string pointer space */
	if((ary = (char **)malloc(1024)) == NULL)
	{
		printf("\nCouldn't allocate string address list\n");
		_exit(1);
	}
	/* initialize count */
	count = 0;

	/* if no args, default to all-files */
	if(argc < 2)
	{
		argc = 2;
		argv[1] = "*";
	}

	/* be sure not asking for another directory or drive */
	for(i = 1; i < argc; i++)
	{
	if((strstr(argv[i], "\\") != NULL)||(strstr(argv[i], ":") != NULL))
	{
		printf("\nARJOPT reads current directory only, do not specify one...\n");
		_exit(1);
	}
	}

	/* parse arguments on command line */
	for ( argc--,argv++; argc; argc--,argv++ )
	{
		  /* parse the argument list */
		  switch ( argv[0][0] )
		  {
				case '-':
					 switch ( argv[0][1] ) {
						  case 'h':
						  case 'H':
						  case '?':
								display_usage();
								break;
					 }
					 break;

				/* this is a file parameter */
				default:
					 parse( *argv );
					 break;
		  }
	}

	/* if no files found, bow out... */
	if(count == 0)
	{
		printf("\nNo matching files found for specifications. Goodbye...\n");
		_exit(1);
	}

	/* do program */
	start();

	atexit(say_done);

	reset_video();
	return (0); /* exit success */
}

