/*

Micro BASIC Interpeter (mbasic.c)

This is the main module.

*** Copyleft - Andre Murta - August 1992/December 2002 ***

Adapted from Herbert Schildt Little C (Dr. Dobbs Journal, August 1989)

*/

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <stdlib.h>
#include <getargs.h>
#include <stdarg.h>
#include <conio.h>
#include <sys/stat.h>
#include "mbasic.h"
#include "structures.h"

/**********************************************************/
/******************** BEGIN definitions *******************/
/**********************************************************/
char *prog; /* holds expression to be analyzed */
char *p_buf; /* program buffer */
int done;
long curr_instruction=0; /* instruction index */
short OPTION_BASE = 1; /* holds OPTION BASE index */
struct data_read *memdata; /* Array information in memory */
struct node *array_items;  /* memory structure pointer for first list */
struct variables vars[NUM_VARS]; /* Variables structure */
struct user_function user_func_table[NUM_FUNCS]; /* user defined functions */
struct label label_table[NUM_LAB]; /* labels */
struct for_stack fstack[FOR_NEST]; /* FOR structure */
struct while_stack wstack[WHILE_NEST]; /* WHILE structure */
char *gstack[SUB_NEST]; /* GOSUB control vector */
struct dtfiles data_files[NUM_FILES]; /* data files control structure */
char token[STRING_LEN]; /* Hold string representation of token */
char token_type, tok;
short local_exp; /* Hold if expression call is inside user function */
int current_func; /* Hold index of function being executed, -1 main program  */
int ftos; /* index to top of FOR stack */
int wtos; /* index to top of WHILE stack */
int gtos; /* index to top of GOSUB stack */
int verbose = 0;        /* setup verbose mode */
unsigned int random_seed; /* this variable is the seed which will be used by RANDOMIZE and RND */
/* commands table */
struct commands table[] = {
    "print",        PRINT,      "input",        INPUT,
    "if",           IF,         "then",         THEN,
    "else",         ELSE,       "goto",         GOTO,
    "for",          FOR,        "next",         NEXT,
    "to",           TO,         "downto",       DOWNTO,
    "gosub",        GOSUB,      "return",       RETURN,
    "while",        WHILE,      "wend",         WEND,
    "step",         STEP,       "rem",          REM,
    "let",          LET,        "dim",          DIM,
    "cls",          CLS,        "clear",        CLEAR,
    "flash",        FLASH,      "randomize",    RANDOMIZE,
    "data",         DATA,       "read",         READ,
    "line",         LINE,       "locate",       LOCATE,
    "color",        COLOR,      "textmode",     TEXTMODE,
    "window",       _WINDOW,    "files",        FILES,
    "delline",      DELLINE,    "insline",      INSLINE,
    "deffn",        DEFFN,      "chain",        CHAIN,
    "highvideo",    HIGHVIDEO,  "normvideo",    NORMVIDEO,
    "lowvideo",     LOWVIDEO,   "open",         OPEN,
    "as",           AS,         "reading",      READING,
    "writing",      WRITING,    "appending",    APPENDING,
    "random",       RANDOM,     "sequential",   SEQUENTIAL,
    "close",        CLOSE,      "field",        FIELD,
    "get",          GET,        "put",          PUT,
    "seek",         SEEK,       "nonewline",    NONEWLINE,
    "fprint",       FPRINT,     "finput",       FINPUT,
    "end",          END,        "",             END,
};
int pquiet=0, pverbose=0, pversion=0, phelp=0; /* these variables keep the execution mode */
char *filename, *logname; /* holds program name */
/* These are the parameters option */
ARG Argtab[]={	{'q', BOOLEAN, &pquiet, "run in quiet mode"},
                {'v', BOOLEAN, &pverbose, "run in verbose mode"},
                {'s', BOOLEAN, &pversion,  "show engine version"},
                {'h', BOOLEAN, &phelp,  "show help"},
                {'l', STRING, (int *)&logname,  "log file"},
                {'f', STRING, (int *)&filename,  "script to execute"}
};
#define TABSIZE (sizeof(Argtab)/sizeof(ARG))
FILE *log_file; /* logfile file pointer */

/* function headers */
void wpush(struct while_stack i);
struct while_stack wpop();
int exec_video(short mode); /* exec_video() needs header declaration */
/* array support functions*/
extern struct node * InsertAtEnd(struct node newVal, struct node **array_items);
extern void deleteNodeValue(struct node delVal, short dim, struct node ** array_items);
extern struct node * search(struct node data, short dim, struct node *pstack);
extern struct node * getMemory(void);
/* DATA & READ support functions */
extern struct data_read * InsertData(struct data_read newVal, struct data_read **items);
extern void deleteItemAtPos(int pos, struct data_read **items);
extern struct data_read * returnItemAtPos(int pos, struct data_read *items);
extern struct data_read * getDataMemory(void);
/* internal string memory functions */
extern char *getmem(int size, unsigned short clean);
extern char *allocateString(char *s);
/********************************************************/
/******************** END definitions *******************/
/********************************************************/

/*
    Similiar to James Bielman verbose_printf() but prints the
    parameters to a log file if the user say so.
*/
int log_printf(char *format, ...)
{
	va_list ap;
	char infotext[STRING_LEN];

	va_start(ap, format);
	if(verbose) {
	    vsprintf(infotext, format, ap);
	    fputs(infotext, log_file);
	}
	va_end(ap);
}

/* Determine file size */
unsigned file_size(char *filename)
{
    struct stat buffer;

    /* call stat to get file info */
    stat(filename, &buffer);

	return buffer.st_size+1;
}

int main(int argc, char **argv)
{
    int answer;
    int quiet=0, return_value;
    unsigned size;
    /* holds temporary file name */
    char *tempfile;

    /* Make sure filename is clean */
    memset(filename, '\0', 255);

    /* Process command-line options */
    argc=getargs(argc,argv,Argtab,TABSIZE);
    if(pquiet) {
        quiet = 1;
        log_printf("Quiet mode enabled\n");
    }
    if(pverbose) {
        verbose = 1;
        printf("Verbose mode enabled\n");
        if((char *)logname == NULL) {
            if((log_file = fopen("log.txt", "w")) == NULL) {
                puts("Cannot create logfile.");
                exit(0);
            }
        } else {
            if((log_file = fopen(logname, "w")) == NULL) {
                puts("Cannot create logfile.");
                exit(0);
            }
        }
    }
    if(pversion) {
        printf("Micro BASIC Version %1.2f\n", MBASIC_VERSION);
        printf("Parser Version %1.2f\n", PARSER_VERSION);
        printf("Engine built for %d bits\n", ENVIRONMENT_VERSION);
        exit(0);
    }
    if(phelp) {
        printf("Micro BASIC Version %1.2f\n", MBASIC_VERSION);
        printf("Parser Version %1.2f\n", PARSER_VERSION);
        printf("Copyright (C) 1992 by Andre Murta\n\n");
        printf("Usage: %s {OPTIONS}\n\n", argv[0]);
        printf("-q, \t\tSuppress startup messages\n");
        printf("-v, \t\tPrint extra information\n");
        printf("-n, \t\tPrint version information\n");
        printf("-h, \t\tPrint this help and exit\n");
        printf("-l<logfile>, \tLogfile name (use with -v option)\n");
        printf("-f<program>, \tScript to be executed\n\n");
        exit(0);
    }

    if(!quiet) {
        printf("The Micro BASIC Language %1.2f\n", MBASIC_VERSION);
        printf("Copyright (C) 1992 by Andre Murta\n\n");
    }

    if((char *)filename == NULL) {
        printf("no input files\n");
        exit(-1);
    }

    log_printf("\n *** Loading file: %s\n", filename);

    /* create processed temporary file, this file will be actually read by the interpreter engine */
    tempfile = allocateString(tempnam(".","tmp"));
    return_value = create_temp_program(filename, tempfile);
    if(return_value == -1) {
        printf("Failed to open program file\n");
        exit(1);
    } else if(return_value == -2) {
        printf("Failed to create temporary file\n");
        exit(2);
    }

    /* Allocating memory for the program */
    size = file_size(tempfile);
    log_printf(" *** Allocating %#x bytes to hold the program...\n", size);
    if(!(p_buf = (char *)malloc(size))) {
        printf("%s: not enough memory to load program\n", argv[0]);
        exit(3);
    }

    /* Load the program */
    log_printf(" *** Loading program into memory...\n");
    /* if(!load_program(p_buf, filename)) { */
    if(!load_program(p_buf, tempfile)) {
        printf("%s: couldn't load file: %s\n", argv[0], filename);
        exit(4);
    }

    /* remove temporary file after it is in memory */
    if(unlink(tempfile) != 0) {
        printf("Failed to remove temporary file.\n");
        exit(5);
    }

 	log_printf(" *** Allocating longjmp buffer...\n");
 	prog = p_buf;

 	log_printf(" *** Resolving labels and functions...\n");
 	scan_program();

 	log_printf(" *** Initializing functions space...\n");
 	local_exp = FALSE;
 	current_func = -1;

 	log_printf(" *** Initializing FOR stack...\n");
 	ftos = gtos = wtos = done = 0;

    log_printf(" *** Initializing data stack...\n");
    init_vars();

    log_printf(" *** Initializing vectors environment...\n");
    array_items = NULL;
    log_printf(" *** Initializing memory environment...\n");
    memdata = NULL;

    do {
        token_type = get_token();

        exec_command();
    } while(!done);

    log_printf(" *** Freeing program buffer...\n");
    free(p_buf);

    if(pverbose) fclose(log_file);
}

/*************************************************************/
/* could be impoved to allow numbered line in REM statements */
/*************************************************************/
int create_temp_program(char *inname, char *outname)
{
	FILE *fp1,*fp2;
	int i;
	char line[STRING_LEN], tststr[3], ch;

	if(!(fp1 = fopen(inname, "r"))) return -1;
	if(!(fp2 = fopen(outname, "w"))) return -2;

	do {
	    /* reset char index */
	    i=0;

        /* get the string from the file */
        fgets(line, STRING_LEN, fp1);
	    if(!feof(fp1)) {
            /* avoid spaces */
            while(line[i] == ' ') i++;
    
            /* build test string */
            tststr[0] = toupper(line[i]);
            tststr[1] = toupper(line[i+1]);
            tststr[2] = toupper(line[i+2]);
            tststr[3] = '\0';
    
            if(tststr[0] == 'R' && tststr[1] == 'E' && tststr[2] == 'M')
                fputs("\n", fp2);
            else
                fputs(line, fp2);
        }
	} while(!feof(fp1));

	fclose(fp1);
	fclose(fp2);

	return 1;
}

/*****************************************************/
/* This routine executes each program statement.     */
/* its basically a loop in the program code, parsing */
/* each statement and performing the code related    */
/* to it.                                            */
/*****************************************************/
exec_command()
{
	if(token_type == VARIABLE)
	{
		putback();
		assignment();
	}
	else
        switch(tok)
        {
            case REM:
		        find_eol();
		        break;

            case DEFFN:
                find_eol(); /* deffn is treated in scan_program() and primitive() */
                break;

            case LET:
                assignment();
                break;

			case PRINT:
				exec_print();
				break;

			case GOTO:
				exec_goto();
				break;

			case IF:
				exec_if();
				break;

			case FOR:
				exec_for();
				break;

			case NEXT:
				exec_next();
				break;

            case WHILE:
                exec_while();
                break;

            case WEND:
                exec_wend();
                break;

			case INPUT:
				exec_input();
				break;

			case GOSUB:
				exec_gosub();
				break;

			case RETURN:
				exec_return();
				break;

            case DIM:
                exec_dim();
                break;

            case CLS:
                exec_cls(); /* HW dependent */
                break;

            case CLEAR:
                exec_clear(); /* HW dependent */
                break;

            case FLASH:
                exec_flash(); /* HW dependent */
                break;

            case RANDOMIZE:
                exec_randomize();
                break;

            case DATA:
                exec_data();
                break;

            case READ:
                exec_read();
                break;

            case LOCATE:
                exec_locate(); /* HW dependent */
                break;

            case COLOR:
                exec_color(); /* HW dependent */
                break;

            case TEXTMODE:
                exec_textmode(); /* HW dependent */
                break;

            case _WINDOW:
                exec_window(); /* HW dependent */
                break;

            case FILES:
                exec_files(); /* HW dependent */
                break;

            case DELLINE:
                exec_delline(); /* HW dependent */
                break;

            case INSLINE:
                exec_insline(); /* HW dependent */
                break;

            case CHAIN:
                exec_chain();  /* HW dependent */
                break;

            case HIGHVIDEO:
                exec_video(1);  /* HW dependent */
                break;

            case NORMVIDEO:
                exec_video(0);  /* HW dependent */
                break;

            case LOWVIDEO:
                exec_video(-1);  /* HW dependent */
                break;

            case OPEN:
                exec_open();
                break;

            case CLOSE:
                exec_close();
                break;

            case FIELD:
                exec_field();
                break;

            case GET:
                exec_get();
                break;

            case PUT:
                exec_put();
                break;

            case SEEK:
                exec_seek();
                break;

            case FPRINT:
                exec_fprint();
                break;

            case FINPUT:
                exec_finput();
                break;

			case END:
			case FINISHED:
				done = 1;
				break;

            /* More observative about syntax errors */
            default: if(tok != EOL && token_type != NUMBER) serror(0);
                     /* this is not a instruction, so, decrement counter */
                     curr_instruction--;
                     break;
		}
    /* increment instruction counter */
    curr_instruction++;
}

/* load program code from file to buffer */
int load_program(char *p, char *fname)
{
	FILE *fp;
	int i = 0;

	if(!(fp = fopen(fname, "r"))) return 0;

	i = 0;
	do
	{
		*p = getc(fp);
		p++; i++;

	} while(!feof(fp));
	fclose(fp);
	return 1;
}

/* Initialize the globals variables table space */
int init_vars()
{
    int i;

    for(i=0; i<NUM_VARS; ++i) {
        vars[i].type = -1;
        vars[i].isdim = FALSE;
        vars[i].dim = FALSE;
        vars[i].is_field = FALSE;
        vars[i].data_file_index = 0;
        vars[i].size = 0;
        memset(vars[i].name, '\0', VAR_LEN);
   }
}

/*
   Search for a var index, check data table, return the variable index or
   create a new entry in the table if necessary. 
*/
int find_var(char *s)
{
  register int i;

  /* if variable already exists, return it's pointer */
  for (i = 0; i < NUM_VARS; ++i)
    if(!strcmp(vars[i].name, s))
      return i;

  /* if variable does not exist, return a blank pointer */
  for (i = 0; i < NUM_VARS; ++i)
    if(vars[i].name[0] == '\0')
      return i;

  serror(15); /* Too many variables */
}

/*************************/
/* is the var declared ? */
/*************************/
int is_var_in_use(char *s)
{
    register int i;

    for (i=0; i<NUM_VARS; ++i)
        if(!strcmp(vars[i].name, s))
            return TRUE; /* found */

    return FALSE; /* not found */
}

/*******************************************************/
/* Converts all alpha chars of the string to uppercase */
/*******************************************************/
int to_upper(char *s)
{
   int i;

   for(i=0; i<strlen(s); i++)
      s[i] = toupper(s[i]);
}

/********************/
/* init label table */
/********************/
int label_init()
{
    register t;

    for(t = 0; t < NUM_LAB; ++t) label_table[t].name[0] = '\0';
}

/***************************/
/* init UD functions table */
/***************************/
int func_init()
{
    register t;

    for(t = 0; t < NUM_FUNCS; ++t) {
        user_func_table[t].name[0] = '\0';
        user_func_table[t].argc = 0;
    }

}

/***************************************************
    Scan program code and keep informations about
    labels and user defined functions entry point
****************************************************/
int scan_program()
{
    int addr, arg_index;
    char *temp;

    func_init();
    label_init();
    temp = prog;

    get_token();
    if(token_type ==  NUMBER) {
        strcpy(label_table[0].name, token);
        label_table[0].p = prog;
    }
    if(tok ==  DEFFN) {
        get_token();
        if(token_type != VARIABLE) serror(0);
        strcpy(user_func_table[0].name, token);
        user_func_table[0].loc = prog;

        get_token();
        if(tok == LPAREN) {
            arg_index = 0;
            do {
                get_token();
                if(token_type != VARIABLE) serror(0);
                strcpy(user_func_table[0].args[arg_index].name, token);
                arg_index++;

                get_token();
                if(tok != COMMA && tok != RPAREN) serror(26);
            } while(tok != RPAREN);
            user_func_table[0].argc = arg_index;
        } else if(tok != EQ) serror(0);
        /* next item after = */
        get_token();
        user_func_table[0].loc = prog;
    }

    find_eol();

    do {
        get_token();
        if(token_type == NUMBER) {
            addr = get_next_label(token);
                if(addr == -1 || addr == -2)
                    (addr == -1) ? serror(5) : serror(6);

            strcpy(label_table[addr].name, token);
            label_table[addr].p = prog;
        }
        if(tok == DEFFN) {
            get_token();
            if(token_type != VARIABLE) serror(0);
            addr = get_next_func(token);
                if(addr == -1 || addr == -2)
                    (addr == -1) ? serror(29) : serror(30);

            strcpy(user_func_table[addr].name, token);

            get_token();
            if(tok == LPAREN) {
                arg_index = 0;
                do {
                    get_token();
                    if(token_type != VARIABLE) serror(0);
                    strcpy(user_func_table[addr].args[arg_index].name, token);
                    arg_index++;
    
                    get_token();
                    if(tok != COMMA && tok != RPAREN) serror(26);
                } while(tok != RPAREN);
                user_func_table[addr].argc = arg_index;
            } else if(tok != EQ) serror(0);
            /* next item after = */
            get_token();
            user_func_table[addr].loc = prog;
        }
        if(tok != EOL) find_eol();
    } while(tok != FINISHED);
    prog = temp;
}

int find_eol()
{
	while(*prog != '\n' && *prog != '\0') ++prog;
}

int get_next_label(char *s)
{
    register int t;

    for(t = 0; t < NUM_LAB; ++t) {
        if(label_table[t].name[0] == 0) return t;
        if(!strcmp(label_table[t].name, s)) return -2;
    }

    return -1;
}

int get_next_func(char *s)
{
    register int t;

    for(t = 0; t < NUM_FUNCS; ++t) {
        if(user_func_table[t].name[0] == 0) return t;
        if(!strcmp(user_func_table[t].name, s)) return -2;
    }

    return -1;
}

char *find_label(char *s)
{
    register t;

    for(t = 0; t < NUM_LAB; ++t)
        if(!strcmp(label_table[t].name, s)) return label_table[t].p;

    return '\0';
}

/***********************************************
    Executes a variable assignment in MBASIC
************************************************/
int assignment()
{
    char _var[VAR_LEN], infotext[STRING_LEN];
    int var;
    struct variables value, x, y, z;
    struct node vec;
    short isdim = FALSE;

    /* Get the variable name */
    get_token();
    if(token_type != VARIABLE) {
        serror(4);
        return;
    }

    strcpy(_var, token);
    to_upper(_var);
    var = find_var(_var);

    get_token();
    if(tok == LPAREN) { /* It is a vector assignment */
        isdim = TRUE; /* This will be useful below */
        if(vars[var].dim < 1) serror(23); /* var has no dim */

        /* get the value */
        get_exp(&x);
        if(x.type != NUMERIC_VAR) serror(0);  /* check type and bounds */
        if((int)x.value.dv < OPTION_BASE || (int)x.value.dv > vars[var].range.x) serror(25);

        get_token();
        if(tok == COMMA) {
            if(vars[var].dim < 2) serror(21); /* invalid range  */

            /* get the value */
            get_exp(&y);

            if(y.type != NUMERIC_VAR) serror(0); /* check type and bounds */
            if((int)y.value.dv < OPTION_BASE || (int)y.value.dv > vars[var].range.y) serror(25);

            get_token();
            if(tok == COMMA) {
                if(vars[var].dim < 3) serror(21); /* invalid range */

                /* get the value */
                get_exp(&z);

                if(z.type != NUMERIC_VAR) serror(0); /* check type and bounds */
                if((int)z.value.dv < OPTION_BASE || (int)z.value.dv > vars[var].range.z) serror(25);

                get_token();
                if(tok != RPAREN) serror(22);
            } else if(tok != RPAREN) serror(0);
        } else if(tok != RPAREN) serror(0);
    }

    if(isdim) get_token(); /* to have the '=' if it is a vector */
    if(tok != EQ) serror(3);

    /* get the value */
    get_exp(&value);

    /* assign the var */
    var = find_var(_var);

    if(!isdim) { /* OK... it's not a vector */
        /* printf("type: %d\n", value.type); */
        vars[var].isdim = FALSE;
        vars[var].type = value.type;
        strcpy(vars[var].name, _var);
        /* printf("Name: %s\n", _var); */
        switch(value.type) {
    	    case NUMERIC_VAR: vars[var].value.dv = value.value.dv;

                              sprintf(infotext, "instr:[%d] ASSIGNMENT: %s=%g\n", curr_instruction, _var, vars[var].value.dv);
                              log_printf(infotext);

    	                      break;
    	    case STRING_VAR: strcpy(vars[var].value.sv, value.value.sv);

                             sprintf(infotext, "instr:[%d] ASSIGNMENT: %s=%s\n", curr_instruction, _var, vars[var].value.sv);
                             log_printf(infotext);

    	                     break;
        }

        return; /* next code is related to vector vars */
    }

    /* prepare vector data for updating */
    vec.var_index = var;
    vec.index.x = (int)(x.value.dv); vec.index.y = (int)(y.value.dv); vec.index.z = (int)(z.value.dv);
    vec.type = value.type;
    switch(value.type) {
	    case NUMERIC_VAR: vec.value.dv = value.value.dv;
                          switch(vars[var].dim) {
                            case 1: sprintf(infotext, "instr:[%d] ASSIGNMENT: %s(%d)=%g\n", curr_instruction, _var, (int)x.value.dv,
                                                                                             vec.value.dv);
                                    log_printf(infotext);
                                    break;
                            case 2: sprintf(infotext, "instr:[%d] ASSIGNMENT: %s(%d,%d)=%g\n", curr_instruction, _var, (int)x.value.dv,
                                                                                          (int)y.value.dv,
                                                                                          vec.value.dv);
                                    log_printf(infotext);
                                    break;
                            case 3: sprintf(infotext, "instr:[%d] ASSIGNMENT: %s(%d,%d,%d)=%g\n", curr_instruction,
                                                                                             _var, (int)x.value.dv,
                                                                                             (int)y.value.dv,
                                                                                             (int)z.value.dv,
                                                                                             vec.value.dv);
                                    log_printf(infotext);
                                    break;
                          }
	                      break;
	    case STRING_VAR: strcpy(vec.value.sv, value.value.sv);
	                     break;
    }


    /* If index is already there, remove it */
    if(search(vec, vars[var].dim, array_items)) deleteNodeValue(vec, vars[var].dim, &array_items);
    /* Insert at the new value at the end of the "list" */
    array_items = InsertAtEnd(vec, &array_items);
}

/****************************************
    Parse and perform a PRINT command
****************************************/
int exec_print()
{
	struct variables answer;
	int len=0, spaces;
	char last_delim, infotext[STRING_LEN];

    sprintf(infotext, "instr:[%d] COMMAND: PRINT: ", curr_instruction);

	do {
		get_exp(&answer);
        if(answer.type == NUMERIC_VAR) {
            len += cprintf("%g", answer.value.dv);
            sprintf(infotext,"%s%g, ", infotext, answer.value.dv);
        } else {
            cprintf("%s", answer.value.sv);
            sprintf(infotext,"%s%s, ", infotext, answer.value.sv);
        }

		get_token();
        last_delim = tok;

        if(tok == COMMA) {
            spaces = 8 - (len % 8);
            len += spaces;
            while(spaces) {
                cprintf(" ");
                spaces--;
            }
        }
        else if(tok == DCOMMA) /* nothin */ ;
        else break;
    } while(tok == DCOMMA || tok == COMMA);

    if(last_delim != NONEWLINE) printf("\n");
    else get_token();

    sprintf(infotext, "%s\n", infotext);
    log_printf(infotext);
}

/****************************************
    Parse and perform a FPRINT command
****************************************/
int exec_fprint()
{
	struct variables answer, filepointer;
	int len=0, spaces;
	char buffer[STRING_LEN], infotext[STRING_LEN];

    get_token();
    if(tok != SHARP) serror(34);

    sprintf(infotext, "instr:[%d] COMMAND: FPRINT: ", curr_instruction);

    get_exp(&filepointer);
    if(filepointer.type != NUMERIC_VAR) serror(0);
    if(filepointer.value.dv < 1 && filepointer.value.dv > NUM_FILES) serror(21);

    sprintf(infotext, "%sFILE INDEX #%d: ", infotext);

    if(data_files[(int)filepointer.value.dv].method != SEQUENTIAL) serror(41);

    get_token();
    if(tok != COMMA) serror(26);

    memset(buffer, '\0', STRING_LEN);

	do {
		get_exp(&answer);
        if(answer.type == NUMERIC_VAR)
            len += sprintf(buffer,"%s%g", buffer, answer.value.dv);
        else
            sprintf(buffer,"%s%s", buffer, answer.value.sv);

		get_token();

        if(tok == COMMA) {
            spaces = 8 - (len % 8);
            len += spaces;
            while(spaces) {
                sprintf(buffer,"%s ",buffer);
                spaces--;
            }
        }
        else if(tok == DCOMMA) /* nothin */ ;
        else break;
    } while(tok == DCOMMA || tok == COMMA);

    /* append newline to the buffer */
    sprintf(buffer,"%s\n", buffer);
    sprintf(infotext, "%s%s", infotext, buffer);

    /* actually write the text to the sequential file */
    fputs(buffer, data_files[(int)filepointer.value.dv].fp);
}

/*****************************************
    Parse and perform an INPUT command
*****************************************/
int exec_input()
{
    char _var[VAR_LEN];
    int var;
    double value;
    char strvalue[STRING_LEN], infotext[STRING_LEN];
    struct variables x, y, z;
    struct node vec;
    unsigned short isdim=FALSE;

    sprintf(infotext, "instr:[%d] COMMAND: INPUT: ", curr_instruction);

    get_token();

    if(token_type == QUOTE) {
        cprintf(token);
        get_token();
        if(tok != COMMA && tok != DCOMMA) serror(0);
        get_token();
    }
    else cprintf("? ");

    if(tok == DOLLAR) {
        get_token();
        if(token_type != VARIABLE) serror(0);

        strcpy(_var, token);
        to_upper(_var);
        var = find_var(_var);

        sprintf(infotext, "%s(STRING)%s", infotext, _var);

        get_token();
        if(tok == LPAREN) {
            isdim = TRUE; /* This will be useful below */
            if(vars[var].dim < 1) serror(23); /* var has no dim */

            /* get the value */
            get_exp(&x);
            if(x.type != NUMERIC_VAR) serror(0);  /* check type and bounds */
            if((int)x.value.dv < OPTION_BASE || (int)x.value.dv > vars[var].range.x) serror(25);

            get_token();
            if(tok == COMMA) {
                if(vars[var].dim < 2) serror(21); /* invalid range  */

                /* get the value */
                get_exp(&y);

                if(y.type != NUMERIC_VAR) serror(0); /* check type and bounds */
                if((int)y.value.dv < OPTION_BASE || (int)y.value.dv > vars[var].range.y) serror(25);

                get_token();
                if(tok == COMMA) {
                    if(vars[var].dim < 3) serror(21); /* invalid range */

                    /* get the value */
                    get_exp(&z);

                    if(z.type != NUMERIC_VAR) serror(0); /* check type and bounds */
                    if((int)z.value.dv < OPTION_BASE || (int)z.value.dv > vars[var].range.z) serror(25);

                    get_token();
                    if(tok != RPAREN) serror(22);
                } else if(tok != RPAREN) serror(0);
            } else if(tok != RPAREN) serror(0);

            if(vars[var].dim == 1) sprintf(infotext,"%s(%d)", infotext, (int)x.value.dv);
            else if(vars[var].dim == 2) sprintf(infotext, "%s(%d,%d)", infotext, (int)x.value.dv, (int)y.value.dv);
            else if(vars[var].dim == 3) sprintf(infotext, "%s(%d,%d,%d)", infotext, (int)x.value.dv, (int)y.value.dv, (int)z.value.dv);
        } else putback();

        gets(strvalue);

        if(!isdim) {
            vars[var].type = STRING_VAR;
            strcpy(vars[var].name, _var);
            strcpy(vars[var].value.sv, strvalue);
    
            sprintf(infotext, "%s=%s\n", infotext, strvalue);
            log_printf(infotext);
    
            return;
        }

        /* prepare vector data for updating */
        vec.var_index = var;
        vec.index.x = (int)(x.value.dv); vec.index.y = (int)(y.value.dv); vec.index.z = (int)(z.value.dv);
        vec.type = STRING_VAR;
    	strcpy(vec.value.sv, strvalue);

        /* If index is already there, remove it */
        if(search(vec, vars[var].dim, array_items)) deleteNodeValue(vec, vars[var].dim, &array_items);
        /* Insert the new value at the end of the "list" */
        array_items = InsertAtEnd(vec, &array_items);

        sprintf(infotext, "%s=%s\n", infotext, strvalue);
        log_printf(infotext);

        return;
    }

    if(token_type != VARIABLE) serror(0);
    strcpy(_var, token);
    to_upper(_var);
    var = find_var(_var);

    sprintf(infotext, "%s(NUMERIC)%s", infotext, _var);

    get_token();
    if(tok == LPAREN) {
        isdim = TRUE; /* This will be useful below */
        if(vars[var].dim < 1) serror(23); /* var has no dim */

        /* get the value */
        get_exp(&x);
        if(x.type != NUMERIC_VAR) serror(0);  /* check type and bounds */
        if((int)x.value.dv < OPTION_BASE || (int)x.value.dv > vars[var].range.x) serror(25);

        get_token();
        if(tok == COMMA) {
            if(vars[var].dim < 2) serror(21); /* invalid range  */

            /* get the value */
            get_exp(&y);

            if(y.type != NUMERIC_VAR) serror(0); /* check type and bounds */
            if((int)y.value.dv < OPTION_BASE || (int)y.value.dv > vars[var].range.y) serror(25);

            get_token();
            if(tok == COMMA) {
                if(vars[var].dim < 3) serror(21); /* invalid range */

                /* get the value */
                get_exp(&z);

                if(z.type != NUMERIC_VAR) serror(0); /* check type and bounds */
                if((int)z.value.dv < OPTION_BASE || (int)z.value.dv > vars[var].range.z) serror(25);

                get_token();
                if(tok != RPAREN) serror(22);
            } else if(tok != RPAREN) serror(0);
        } else if(tok != RPAREN) serror(0);

        if(vars[var].dim == 1) sprintf(infotext,"%s(%d)", infotext, (int)x.value.dv);
        else if(vars[var].dim == 2) sprintf(infotext, "%s(%d,%d)", infotext, (int)x.value.dv, (int)y.value.dv);
        else if(vars[var].dim == 3) sprintf(infotext, "%s(%d,%d,%d)", infotext, (int)x.value.dv, (int)y.value.dv, (int)z.value.dv);
    } else putback();

    scanf("%lf", &value);

    if(!isdim) {
        vars[var].type = NUMERIC_VAR;
        strcpy(vars[var].name, _var);
        vars[var].value.dv = value;
    
        sprintf(infotext, "%s=%g\n", infotext, value);
        log_printf(infotext);

        return;
    }

    /* prepare vector data for updating */
    vec.var_index = var;
    vec.index.x = (int)(x.value.dv); vec.index.y = (int)(y.value.dv); vec.index.z = (int)(z.value.dv);
    vec.type = NUMERIC_VAR;
	vec.value.dv = value;

    /* If index is already there, remove it */
    if(search(vec, vars[var].dim, array_items)) deleteNodeValue(vec, vars[var].dim, &array_items);
    /* Insert at the new value at the end of the "list" */
    array_items = InsertAtEnd(vec, &array_items);

    sprintf(infotext, "%s=%g\n", infotext, value);
    log_printf(infotext);

    return;
}

/******************************************
    Parse and perform a FINPUT command
******************************************/
int exec_finput()
{
    char _var[VAR_LEN];
    int var;
    char strvalue[STRING_LEN], infotext[STRING_LEN];
    struct variables filepointer, x, y, z;
    struct node vec;
    unsigned short isdim=FALSE;

    sprintf(infotext, "COMMAND: FINPUT: ");

    get_token();

    if(tok != SHARP) serror(34);

    get_exp(&filepointer);
    if(filepointer.type != NUMERIC_VAR) serror(0);
    if(filepointer.value.dv < 1 && filepointer.value.dv > NUM_FILES) serror(21);

    sprintf(infotext, "%sFILE INDEX #%d: ",infotext, (int)filepointer.value.dv);

    /* First, test if file space is really in use */
    if(!data_files[(int)filepointer.value.dv].locked) serror(38);

    /* So, test if file space is opened sequentially */
    if(data_files[(int)filepointer.value.dv].method != SEQUENTIAL) serror(41);

    get_token();
    if(tok != COMMA && tok != DCOMMA) serror(0);

    get_token();
    if(tok != DOLLAR) serror(42);

    get_token();
    if(token_type != VARIABLE) serror(0);

    strcpy(_var, token);
    to_upper(_var);
    var = find_var(_var);

    sprintf(infotext, "%s%s", infotext, _var);

    get_token();
    if(tok == LPAREN) {
        isdim = TRUE; /* This will be useful below */
        if(vars[var].dim < 1) serror(23); /* var has no dim */

        /* get the value */
        get_exp(&x);
        if(x.type != NUMERIC_VAR) serror(0);  /* check type and bounds */
        if((int)x.value.dv < OPTION_BASE || (int)x.value.dv > vars[var].range.x) serror(25);

        get_token();
        if(tok == COMMA) {
            if(vars[var].dim < 2) serror(21); /* invalid range  */

            /* get the value */
            get_exp(&y);

            if(y.type != NUMERIC_VAR) serror(0); /* check type and bounds */
            if((int)y.value.dv < OPTION_BASE || (int)y.value.dv > vars[var].range.y) serror(25);

            get_token();
            if(tok == COMMA) {
                if(vars[var].dim < 3) serror(21); /* invalid range */

                /* get the value */
                get_exp(&z);

                if(z.type != NUMERIC_VAR) serror(0); /* check type and bounds */
                if((int)z.value.dv < OPTION_BASE || (int)z.value.dv > vars[var].range.z) serror(25);

                get_token();
                if(tok != RPAREN) serror(22);
            } else if(tok != RPAREN) serror(0);
        } else if(tok != RPAREN) serror(0);

        if(vars[var].dim == 1) sprintf(infotext,"%s(%d)", infotext, (int)x.value.dv);
        else if(vars[var].dim == 2) sprintf(infotext, "%s(%d,%d)", infotext, (int)x.value.dv, (int)y.value.dv);
        else if(vars[var].dim == 3) sprintf(infotext, "%s(%d,%d,%d)", infotext, (int)x.value.dv, (int)y.value.dv, (int)z.value.dv);
    } else putback();

    /* get the string from the file */
    fgets(strvalue, STRING_LEN, data_files[(int)filepointer.value.dv].fp);

    if(!isdim) {
        vars[var].type = STRING_VAR;
        strcpy(vars[var].name, _var);
        strcpy(vars[var].value.sv, strvalue);

        sprintf(infotext, "%s=%s\n", infotext, strvalue);
        log_printf(infotext);

        return;
    }

    /* prepare vector data for updating */
    vec.var_index = var;
    vec.index.x = (int)(x.value.dv); vec.index.y = (int)(y.value.dv); vec.index.z = (int)(z.value.dv);
    vec.type = STRING_VAR;
	strcpy(vec.value.sv, strvalue);

    /* If index is already there, remove it */
    if(search(vec, vars[var].dim, array_items)) deleteNodeValue(vec, vars[var].dim, &array_items);
    /* Insert at the new value at the end of the "list" */
    array_items = InsertAtEnd(vec, &array_items);

    sprintf(infotext, "%s=%s\n", infotext, strvalue);
    log_printf(infotext);

    return;
}

/****************************************
    Parse and perform a GOTO command
****************************************/
int exec_goto()
{
    char *loc, infotext[STRING_LEN];

    sprintf(infotext,"instr:[%d] COMMAND: GOTO: ", curr_instruction);

    get_token();

    sprintf(infotext,"%s%s\n",infotext, token);

    loc = find_label(token);
    if(loc == '\0') serror(7);
    else prog = loc;

    log_printf(infotext);
}

/***************************************************
    Parse and perform an IF/THEN/[ELSE] command
***************************************************/
int exec_if()
{
    struct variables cond;
    char op;
    char infotext[STRING_LEN];

    sprintf(infotext, "instr:[%d] COMMAND: IF: ", curr_instruction);

    /* Test expression */
    get_exp(&cond);

    sprintf(infotext, "%sRESULT=%g:(", infotext, cond.value.dv);
    if(cond.value.dv) sprintf(infotext, "%sTRUE)\n", infotext, cond.value.dv);
    else sprintf(infotext, "%sFALSE)\n", infotext, cond.value.dv);

    /* Check for THEN statement */
    get_token();
    if(tok != THEN) {
        serror(8);
        return;
    }

    if(cond.value.dv) {
        /* Execute until ELSE or end-of-line */
        get_token();
        while(tok != ELSE && tok != EOL && tok != FINISHED) {
            exec_command();
            get_token();
        }

        if(tok == ELSE) find_eol();
    } else {
        /* Skip ahead until ELSE */
        while(tok != ELSE && tok != EOL)
            get_token();
    }

    log_printf(infotext);
}

/*
    Push a value into the FOR stack
*/
void fpush(struct for_stack i)
{
    if(ftos > FOR_NEST)
        serror(10);

    fstack[ftos] = i;
    ftos++;
}

/*
    Pop a value from the FOR stack
*/
struct for_stack fpop()
{
    ftos--;
    if(ftos < 0) serror(11);
    return(fstack[ftos]);
}

/****************************************
    Parse and perform a FOR command
****************************************/
int exec_for()
{
    struct for_stack i;
    char _var[VAR_LEN], infotext[STRING_LEN];
    struct variables value, for_target, for_step, x, y, z;
    struct node vec;

    sprintf(infotext, "instr:[%d] COMMAND: FOR: ", curr_instruction);

    get_token(); /* get control var */
    if(token_type != VARIABLE) {
        serror(4);
        return;
    }

    /* assign the var */
    strcpy(_var, token);
    to_upper(_var);
    i.var = find_var(_var);
    vars[i.var].type = NUMERIC_VAR; /* var type MUST be numeric */
    i.isdim = FALSE; /* assume it is not a vector attribution */

    sprintf(infotext, "%s%s", infotext, _var);

    get_token();
    if(tok == LPAREN) { /* It is a vector assignment */
        i.isdim = TRUE; /* UPS!!! It is a vector. This will be useful below */
        if(vars[i.var].dim < 1) serror(23); /* var has no dim */

        /* get the value */
        get_exp(&x);
        if(x.type != NUMERIC_VAR) serror(0);  /* check type and bounds */
        if((int)x.value.dv < OPTION_BASE || (int)x.value.dv > vars[i.var].range.x) serror(25);
        i.index.x = (int)x.value.dv;

        get_token();
        if(tok == COMMA) {
            if(vars[i.var].dim < 2) serror(21); /* invalid range  */

            /* get the value */
            get_exp(&y);

            if(y.type != NUMERIC_VAR) serror(0); /* check type and bounds */
            if((int)y.value.dv < OPTION_BASE || (int)y.value.dv > vars[i.var].range.y) serror(25);
            i.index.y = (int)y.value.dv;

            get_token();
            if(tok == COMMA) {
                if(vars[i.var].dim < 3) serror(21); /* invalid range */

                /* get the value */
                get_exp(&z);

                if(z.type != NUMERIC_VAR) serror(0); /* check type and bounds */
                if((int)z.value.dv < OPTION_BASE || (int)z.value.dv > vars[i.var].range.z) serror(25);
                i.index.z = (int)z.value.dv;

                get_token();
                if(tok != RPAREN) serror(22);
            } else if(tok != RPAREN) serror(0);
        } else if(tok != RPAREN) serror(0);

        if(vars[i.var].dim == 1) sprintf(infotext,"%s(%d)", infotext, (int)x.value.dv);
        else if(vars[i.var].dim == 2) sprintf(infotext, "%s(%d,%d)", infotext, (int)x.value.dv, (int)y.value.dv);
        else if(vars[i.var].dim == 3) sprintf(infotext, "%s(%d,%d,%d)", infotext, (int)x.value.dv, (int)y.value.dv, (int)z.value.dv);
    }

    if(i.isdim) get_token(); /* to have the '=' if it is a vector */
    if(tok != EQ) serror(3);

    /* get the value */
    get_exp(&value);

    if(value.type != NUMERIC_VAR) serror(18); /* type mismatch */
    strcpy(vars[i.var].name, _var);
    vars[i.var].isdim = i.isdim;
    vars[i.var].value.dv = value.value.dv;

    sprintf(infotext, "%s=%g", infotext, value.value.dv);

    if(i.isdim) {
        /* prepare vector data for updating */
        vec.var_index = i.var;
        vec.index.x = i.index.x; vec.index.y = i.index.y; vec.index.z = i.index.z;
        vec.type = vars[i.var].type;
        /* is it is here, we garantee we have a numeric value */
        vec.value.dv = value.value.dv;
        /* If index is already there, remove it */
        if(search(vec, vars[i.var].dim, array_items)) deleteNodeValue(vec, vars[i.var].dim, &array_items);
        /* Insert at the new value at the end of the "list" */
        array_items = InsertAtEnd(vec, &array_items);
    }

    get_token();

    if(tok != TO && tok != DOWNTO) serror(9);
    if(tok == TO) {
	    i.method = TO;

        sprintf(infotext, "%s: TO ", infotext);

    	get_exp(&for_target);
    	if(for_target.type != NUMERIC_VAR) serror(18); /* type mismatch */
    	i.target = for_target.value.dv;

        sprintf(infotext, "%s %g ", infotext, for_target.value.dv);

        get_token();
        if(tok != STEP) {
            putback();
            i.step = 1;
        } else {
            get_exp(&for_step);
            if(for_step.type != NUMERIC_VAR) serror(18); /* type mismatch */
            i.step = for_step.value.dv;
            sprintf(infotext, "%sSTEP %g", infotext, for_step.value.dv);
        }

        sprintf(infotext, "%s\n", infotext);

    	if(value.value.dv >= vars[i.var].value.dv)
    	{
            i.loc = prog;
            fpush(i);
        }
        else while(tok != NEXT) get_token();
    } else {
        i.method = DOWNTO;

        sprintf(infotext, "%s: DOWNTO ", infotext);

        get_exp(&for_target);
        if(for_target.type != NUMERIC_VAR) serror(18); /* type mismatch */
        i.target = for_target.value.dv;

        sprintf(infotext, "%s %g ", infotext, for_target.value.dv);

        get_token();
        if(tok != STEP) {
            putback();
            i.step = 1;
        } else {
            get_exp(&for_step);
            if(for_step.type != NUMERIC_VAR) serror(18); /* type mismatch */
            i.step = for_step.value.dv;
            sprintf(infotext, "%sSTEP %g", infotext, for_step.value.dv);
        }

        sprintf(infotext, "%s\n", infotext);

        if(value.value.dv <= vars[i.var].value.dv)
        {
            i.loc = prog;
            fpush(i);
        }
        else while(tok != NEXT) get_token();
    }

    log_printf(infotext);
}

/****************************************
    Parse and perform a NEXT command
****************************************/
int exec_next()
{
    struct for_stack i;
    struct node vec;
    char infotext[STRING_LEN];

    sprintf(infotext, "instr:[%d] COMMAND: NEXT: ", curr_instruction);

    i = fpop();

    if(i.method == TO) {
        sprintf(infotext, "%sMETHOD: TO: ", infotext);
        vars[i.var].value.dv += i.step;
        sprintf(infotext, "%s%s",infotext, vars[i.var].name);
        if(i.isdim) {
            /* prepare vector data for updating */
            vec.var_index = i.var;
            vec.index.x = i.index.x; vec.index.y = i.index.y; vec.index.z = i.index.z;
            vec.type = vars[i.var].type;
            /* it is here, we garantee we have a numeric value */
            vec.value.dv = vars[i.var].value.dv;
    
            if(vars[i.var].dim == 1) sprintf(infotext, "%s(%d)", infotext, (int)i.index.x);
            else if(vars[i.var].dim == 2) sprintf(infotext, "%s(%d,%d)", infotext, (int)i.index.x, (int)i.index.y);
            else if(vars[i.var].dim == 3) sprintf(infotext, "%s(%d,%d,%d)", infotext, (int)i.index.x, (int)i.index.y, (int)i.index.z);

            /* If index is already there, remove it */
            if(search(vec, vars[i.var].dim, array_items)) deleteNodeValue(vec, vars[i.var].dim, &array_items);
            /* Insert at the new value at the end of the "list" */
            array_items = InsertAtEnd(vec, &array_items);
        }
        sprintf(infotext, "%s=%g\n", infotext, vars[i.var].value.dv);
        if(vars[i.var].value.dv > i.target) return;
        fpush(i);
        prog = i.loc;
    } else {
        sprintf(infotext, "%sMETHOD: DOWNTO: ", infotext);
        vars[i.var].value.dv -= i.step;
        sprintf(infotext, "%s%s",infotext, vars[i.var].name);
        if(i.isdim) {
            /* prepare vector data for updating */
            vec.var_index = i.var;
            vec.index.x = i.index.x; vec.index.y = i.index.y; vec.index.z = i.index.z;
            vec.type = vars[i.var].type;
            /* is it is here, we garantee we have a numeric value */
            vec.value.dv = vars[i.var].value.dv;
    
            if(vars[i.var].dim == 1) sprintf(infotext, "%s(%d)", infotext, (int)i.index.x);
            else if(vars[i.var].dim == 2) sprintf(infotext, "%s(%d,%d)", infotext, (int)i.index.x, (int)i.index.y);
            else if(vars[i.var].dim == 3) sprintf(infotext, "%s(%d,%d,%d)", infotext, (int)i.index.x, (int)i.index.y, (int)i.index.z);

            /* If index is already there, remove it */
            if(search(vec, vars[i.var].dim, array_items)) deleteNodeValue(vec, vars[i.var].dim, &array_items);
            /* Insert at the new value at the end of the "list" */
            array_items = InsertAtEnd(vec, &array_items);
        }
        sprintf(infotext, "%s=%g\n", infotext, vars[i.var].value.dv);
        if(vars[i.var].value.dv < i.target) return;
        fpush(i);
        prog = i.loc;
    }

    log_printf(infotext);
}

/*****************************************
    Parse and perform a WHILE statement
*****************************************/
int exec_while()
{
    struct variables cond;
    char *temp;
    struct while_stack i;
    int wblk_count = 1;
    char infotext[STRING_LEN];

    sprintf(infotext, "instr:[%d] COMMAND: WHILE: ", curr_instruction);

    /* put WHILE back in the parser */
    putback();

    /* set address */
    i.loc = prog;

    /* discard WHILE*/
    get_token();

    /* check condition expression */
    get_exp(&cond);

    sprintf(infotext, "%s CONDITION: %g:(", infotext, cond.value.dv);
    if(cond.value.dv) sprintf(infotext, "%sTRUE)\n", infotext);
    else sprintf(infotext, "%sFALSE)\n", infotext);

    if(cond.value.dv)
    {
       i.cond = 1;
       wpush(i);
    }
    else
    {
       i.cond = 0;
       wpush(i);

       while(wblk_count > 0)
       {
          get_token();
          if(tok == WHILE) wblk_count++;
          else if(tok == WEND) wblk_count--;
       }

       putback();
    }

    log_printf(infotext);
}

/*
    Push a value into the WHILE stack
*/
void wpush(struct while_stack i)
{
    if(wtos > WHILE_NEST)
       serror(16);
    wstack[wtos] = i;
    wtos++;
}

/*
    Pop a value from the while stack
*/
struct while_stack wpop()
{
    wtos--;
    if(wtos < 0) serror(17);
    return(wstack[wtos]);
}

/*****************************************
    Parse and perform a WEND statement
*****************************************/
int exec_wend()
{
    struct while_stack i;
    char infotext[STRING_LEN];

    sprintf(infotext, "instr:[%d] COMMAND: WEND: ", curr_instruction);

    i = wpop();

    sprintf(infotext, " POP: %g", i.cond);

    log_printf(infotext);

    if(!i.cond) return;

    prog = i.loc;
}

/*
    Push a value into the subroutine stack
*/
int gpush(char *s)
{
    gtos++;

    if(gtos == SUB_NEST) {
        serror(12);
        return;
    }

    gstack[gtos] = s;
}

/*
    Pop a value from the subroutine stack
*/
char *gpop()
{
    if(gtos == 0) {
        serror(13);
        return 0;
    }
    return(gstack[gtos--]);
}

/*****************************************
    Parse and perform a GOSUB statement
*****************************************/
int exec_gosub()
{
    char *loc;
    char infotext[STRING_LEN];

    sprintf(infotext, "instr:[%d] COMMAND: GOSUB: ", curr_instruction);

    get_token();

    sprintf(infotext, "%s%s\n", infotext, token);
    log_printf(infotext);

    loc = find_label(token);
    if(loc == '\0')
        serror(7);
    else {
        gpush(prog);
        prog = loc;
    }
}

/*****************************************
    Parse and perform a RETURN statement
*****************************************/
int exec_return()
{
    char infotext[STRING_LEN];

    log_printf("instr:[%d] COMMAND: RETURN", curr_instruction);

    prog = gpop();
}

/*****************************************
    Parse and perform a DIM statement
*****************************************/
int exec_dim()
{
    char _var[VAR_LEN], infotext[STRING_LEN];
    int var;
    struct variables value;
    int x,y,z,dim;

    x=y=z=-1;

    sprintf(infotext, "instr:[%d] COMMAND: DIM: ", curr_instruction);

    /* Get the variable name */
    get_token();
    if(token_type != VARIABLE) {
        serror(4);
        return;
    }

    strcpy(_var, token);
    to_upper(_var);

    sprintf(infotext, "%s %s(", infotext, _var);

    /* get the ( */
    get_token();
    if(tok != LPAREN) {
        serror(0);
        return;
    }

    /* get the value */
    get_exp(&value);
    if(value.type!= NUMERIC_VAR) serror(0);
    x = (int)value.value.dv;

    dim = 1;

    get_token();
    if(tok == COMMA) {
        /* get the value */
        get_exp(&value);
        if(value.type!= NUMERIC_VAR) serror(0);
        y = (int)value.value.dv;

        dim = 2;

        get_token();
        if(tok == COMMA) {
            /* get the value */
            get_exp(&value);
            if(value.type!= NUMERIC_VAR) serror(0);
            z = (int)value.value.dv;

            dim = 3;

            get_token();
            if(tok != RPAREN) serror(22);
        } else if(tok != RPAREN) serror(0);
    } else if(tok != RPAREN) serror(0);

    if(dim == 1) sprintf(infotext, "%s%d)\n", infotext, x);
    else if(dim == 2) sprintf(infotext, "%s%d,%d)\n", infotext, x, y);
    else if(dim == 3) sprintf(infotext, "%s%d,%d,%d)\n", infotext, x, y, z);

    /* assign the var */
    var = find_var(_var);
    strcpy(vars[var].name, _var);
    vars[var].isdim = TRUE;
    vars[var].dim = dim;
    vars[var].range.x = x; vars[var].range.y = y; vars[var].range.z = z;

    log_printf(infotext);
}

/*****************************************
    Parse and perform a CLS statement
*****************************************/
int exec_cls()
{
    get_token();

    if(tok != EOL)
        if(tok == LINE) {
            clreol();
            log_printf("instr:[%d] COMMAND: CLS LINE\n", curr_instruction);
            return;
        } else serror(0);
    else putback();

    log_printf("instr:[%d] COMMAND: CLS\n", curr_instruction);

    clrscr();
}

/*****************************************
    Parse and perform a CLEAR statement
*****************************************/
int exec_clear()
{
    log_printf("instr:[%d] COMMAND: CLEAR\n", curr_instruction);
    ScreenClear();
}

/*****************************************
    Parse and perform a FLASH statement
*****************************************/
int exec_flash()
{
    register char i;

    log_printf("instr:[%d] COMMAND: FLASH\n", curr_instruction);

    for(i=1; i<11; ++i)
        ScreenVisualBell();
}

/*********************************************
    Parse and perform a RANDOMIZE statement
*********************************************/
int exec_randomize()
{
    struct variables value;
    char infotext[STRING_LEN];

    sprintf(infotext, "instr:[%d] COMMAND: RANDOMIZE: ", curr_instruction);

    get_exp(&value);
    if(value.type != NUMERIC_VAR) serror(18);
    if(value.value.dv < 0) serror(27);

    sprintf(infotext, "%s%g\n", value.value.dv);

    random_seed = (unsigned int)value.value.dv;
    srand((unsigned int)value.value.dv);

    log_printf(infotext);
}

/*****************************************
    Parse and perform a DATA statement
*****************************************/
int exec_data()
{
	struct variables value;
	struct data_read item;
	char infotext[STRING_LEN];
	unsigned long ammount=0;

    sprintf(infotext, "instr:[%d] COMMAND: DATA: ALLOCATED ", curr_instruction);

	do {
		get_exp(&value);
		item.type = value.type;
        switch(value.type) {
            case NUMERIC_VAR: item.value.dv = value.value.dv;
                              ammount += 8; /* double type occupies 8 bytes of memory */
                              break;
            case STRING_VAR: strcpy(item.value.sv, value.value.sv);
                             ammount += strlen(value.value.sv);
                             break;
        }
        memdata = InsertData(item, &memdata);

		get_token();
    } while(tok == COMMA);

    sprintf(infotext, "%s%ld BYTES OF MEMORY\n", infotext, ammount);

    log_printf(infotext);
    /* PrintItemList(memdata); */
}

/*****************************************
    Parse and perform a READ statement
*****************************************/
int exec_read()
{
    char _var[VAR_LEN], infotext[STRING_LEN];
    int var,count=0;
	struct data_read *item;
    struct variables x, y, z;
    struct node vec;
    short isdim = FALSE;
    unsigned long ammount;

    sprintf(infotext, "instr:[%d] COMMAND: READ: RELEASED ", curr_instruction);

	do {
        /* Get the variable name */
        get_token();
        if(token_type != VARIABLE) {
            serror(4);
            return;
        }

        strcpy(_var, token);
        to_upper(_var);
        var = find_var(_var);

        get_token();
        if(tok == LPAREN) { /* It is a vector assignment */
            isdim = TRUE; /* This will be useful below */
            if(vars[var].dim < 1) serror(23); /* var has no dim */

            /* get the value */
            get_exp(&x);
            if(x.type != NUMERIC_VAR) serror(0);  /* check type and bounds */
            if((int)x.value.dv < OPTION_BASE || (int)x.value.dv > vars[var].range.x) serror(25);

            get_token();
            if(tok == COMMA) {
                if(vars[var].dim < 2) serror(21); /* invalid range  */

                /* get the value */
                get_exp(&y);

                if(y.type != NUMERIC_VAR) serror(0); /* check type and bounds */
                if((int)y.value.dv < OPTION_BASE || (int)y.value.dv > vars[var].range.y) serror(25);

                get_token();
                if(tok == COMMA) {
                    if(vars[var].dim < 3) serror(21); /* invalid range */

                    /* get the value */
                    get_exp(&z);

                    if(z.type != NUMERIC_VAR) serror(0); /* check type and bounds */
                    if((int)z.value.dv < OPTION_BASE || (int)z.value.dv > vars[var].range.z) serror(25);

                    get_token();
                    if(tok != RPAREN) serror(22);
                } else if(tok != RPAREN) serror(0);
            } else if(tok != RPAREN) serror(0);
        } else putback(); /* token is not '[' */

        /* if we have no data we have an error */
        if(!getItemListSize(memdata)) serror(28);

        /* get the first stored data */
        item = returnItemAtPos(0, memdata);

        /* assign the var */
        var = find_var(_var);

        if(!isdim) { /* OK... it's not a vector */
            vars[var].type = item->type;
            strcpy(vars[var].name, _var);
            switch(item->type) {
        	    case NUMERIC_VAR: vars[var].value.dv = item->value.dv;
        	                      ammount += 8;
        	                      break;
        	    case STRING_VAR: strcpy(vars[var].value.sv, item->value.sv);
        	                     ammount += strlen(item->value.sv);
        	                     break;
            }
        } else {
            /* prepare vector data for updating */
            vec.var_index = var;
            vec.index.x = (int)(x.value.dv); vec.index.y = (int)(y.value.dv); vec.index.z = (int)(z.value.dv);
            vec.type = item->type;
            switch(item->type) {
        	    case NUMERIC_VAR: vec.value.dv = item->value.dv;
        	                      ammount += 8;
        	                      break;
        	    case STRING_VAR: strcpy(vec.value.sv, item->value.sv);
        	                     ammount += strlen(item->value.sv);
        	                     break;
            }
    
            /* If index is already there, remove it */
            if(search(vec, vars[var].dim, array_items)) deleteNodeValue(vec, vars[var].dim, &array_items);
            /* Insert at the new value at the end of the "list" */
            array_items = InsertAtEnd(vec, &array_items);
        }

        /* The value is copied into the var, we can kill it from the list */
        deleteItemAtPos(0, &memdata);     

		get_token();
    } while(tok == COMMA);

    sprintf(infotext, "%s%ld BYTES OF MEMORY\n", infotext, ammount);

    log_printf(infotext);
}

/*******************************************
    Parse and perform a LOCATE statement
*******************************************/
int exec_locate()
{
    struct variables y,x;

    get_exp(&y);
    if(y.type != NUMERIC_VAR) serror(0);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&x);
    if(x.type != NUMERIC_VAR) serror(0);

    gotoxy((int)x.value.dv, (int)y.value.dv);

    log_printf("instr:[%d] COMMAND: LOCATE: [%d,%d]\n", curr_instruction, (int)x.value.dv, (int)y.value.dv);
}

/*****************************************
    Parse and perform a COLOR statement
*****************************************/
int exec_color()
{
    struct variables tcolor,bcolor,blinking;

    get_exp(&tcolor);
    if(tcolor.type != NUMERIC_VAR) serror(0);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&bcolor);
    if(bcolor.type != NUMERIC_VAR) serror(0);

    get_token();
    if(tok == COMMA) {
        get_exp(&blinking);
        if(blinking.type != NUMERIC_VAR) serror(0);
        if((int)blinking.value.dv) {
            textcolor((int)tcolor.value.dv | BLINK);
            textbackground((int)bcolor.value.dv);
        } else {
            textcolor((int)tcolor.value.dv);
            textbackground((int)bcolor.value.dv);
        }

        return;
    } else {
        putback();
        blinking.value.dv = 0;
    }

    textcolor((int)tcolor.value.dv);
    textbackground((int)bcolor.value.dv);

    log_printf("instr:[%d] COMMAND: COLOR: [%d,%d,%d]\n", curr_instruction, (int)tcolor.value.dv,
                                                                            (int)bcolor.value.dv,
                                                                            (int)blinking.value.dv);
}

/********************************************
    Parse and perform a TEXTMODE statement
********************************************/
int exec_textmode()
{
    struct variables nlines;

    get_exp(&nlines);
    if(nlines.type != NUMERIC_VAR) serror(0);

    _set_screen_lines((int)nlines.value.dv);

    log_printf("instr:[%d] COMMAND: TEXTMODE: [%d]\n", curr_instruction, (int)nlines.value.dv);
}

/*******************************************
    Parse and perform a WINDOW statement
*******************************************/
int exec_window()
{
    struct variables left, top, right, bottom;

    get_exp(&left);
    if(left.type != NUMERIC_VAR) serror(0);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&top);
    if(top.type != NUMERIC_VAR) serror(0);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&right);
    if(right.type != NUMERIC_VAR) serror(0);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&bottom);
    if(bottom.type != NUMERIC_VAR) serror(0);

    window((int)left.value.dv, (int)top.value.dv, (int)right.value.dv, (int)bottom.value.dv);

    log_printf("instr:[%d] COMMAND: WINDOW [%d,%d,%d,%d]\n", curr_instruction, (int)left.value.dv,
                                                                               (int)top.value.dv,
                                                                               (int)right.value.dv,
                                                                               (int)bottom.value.dv);
}

/*****************************************
    Parse and perform a FILES statement
*****************************************/
int exec_files()
{
    struct variables mask;
    char buff[STRING_LEN];

    get_exp(&mask);
    if(mask.type != STRING_VAR) serror(0);

    sprintf(buff,"dir %s", mask.value.sv);

    system(buff);

    log_printf("instr:[%d] COMMAND: FILES: [%s]\n", curr_instruction, mask.value.sv);
}

/********************************************
    Parse and perform a DELLINE statement
*********************************************/
int exec_delline()
{
    delline();

    log_printf("instr:[%d] COMMAND: DELLINE\n", curr_instruction);
}

/********************************************
    Parse and perform an INSLINE statement
********************************************/
int exec_insline()
{
    insline();

    log_printf("instr:[%d] COMMAND: INSLINE\n", curr_instruction);
}

/*****************************************
    Parse and perform a CHAIN statement
*****************************************/
int exec_chain()
{
    struct variables app;
    char buff[STRING_LEN];

    get_exp(&app);
    if(app.type != STRING_VAR) serror(0);

    sprintf(buff,"mbasic -q -f%s", app.value.sv);

    system(buff);

    log_printf("instr:[%d] COMMAND: CHAIN: [%s]\n", curr_instruction, app.value.sv);
}

/****************************************************************
    Parse and perform a HIGHVIDEO/NORMVIDEO/LOWVIDEO statement
****************************************************************/
int exec_video(short mode)
{
    if(mode == 1) {
        highvideo();
        log_printf("instr:[%d] COMMAND: HIGHVIDEO\n", curr_instruction);
    } else if(mode == 0) {
        normvideo();
        log_printf("instr:[%d] COMMAND: NORMVIDEO\n", curr_instruction);
    } else if(mode == -1) {
        lowvideo();
        log_printf("instr:[%d] COMMAND: LOWVIDEO\n", curr_instruction);
    }
}

/*******************************************
    Parse and perform an OPEN statement
*******************************************/
int exec_open()
{
    struct variables filename, filepointer;
    char access_mode[4], infotext[STRING_LEN];
    short mode;

    sprintf(infotext, "instr:[%d] COMMAND: OPEN: ", curr_instruction);

    get_exp(&filename);
    if(filename.type != STRING_VAR) serror(0);

    sprintf(infotext, "%s%s: ", infotext, filename.value.sv);

    get_token();
    if(tok != FOR) serror(31);

    get_token();
    switch(tok) {
        case READING:
                strcpy(access_mode, "rb");
                sprintf(infotext, "%s READING: ", infotext);
                break;
        case WRITING:
                strcpy(access_mode, "wb");
                sprintf(infotext, "%s WRITING: ", infotext);
                break;
        case APPENDING:
                strcpy(access_mode, "ab");
                sprintf(infotext, "%s APPENDING: ", infotext);
                break;
        default: serror(32);
    }

    get_token();
    if(tok != COMMA) serror(26);

    get_token();
    switch(tok) {
        case RANDOM:
                mode = RANDOM;
                sprintf(infotext, "%s RANDOM: ", infotext);
                break;
        case SEQUENTIAL:
                mode = SEQUENTIAL;
                sprintf(infotext, "%s SEQUENTIAL: ", infotext);
                break;
        default: serror(39);
    }

    get_token();
    if(tok != AS) serror(33);

    get_token();
    if(tok != SHARP) serror(34);

    get_exp(&filepointer);
    if(filepointer.type != NUMERIC_VAR) serror(0);
    if(filepointer.value.dv < 1 && filepointer.value.dv > NUM_FILES) serror(21);

    sprintf(infotext, "%s FILE INDEX #%d\n", infotext, (int)filepointer.value.dv);

    /* First, test if file space is already in use */
    if(data_files[(int)filepointer.value.dv].locked) {
        fclose(data_files[(int)filepointer.value.dv].fp);
        serror(37);
    }

    data_files[(int)filepointer.value.dv].method = mode;

    /* now actually opens the file */
    strcpy(data_files[(int)filepointer.value.dv].filename, filename.value.sv);
    if((data_files[(int)filepointer.value.dv].fp = fopen(data_files[(int)filepointer.value.dv].filename, access_mode)) == NULL) {
        data_files[(int)filepointer.value.dv].io_error = TRUE;
        data_files[(int)filepointer.value.dv].locked = FALSE;
    } else {
        data_files[(int)filepointer.value.dv].io_error = FALSE;
        data_files[(int)filepointer.value.dv].locked = TRUE;
        data_files[(int)filepointer.value.dv].recsize = 0;
    }

    log_printf(infotext);
}

/******************************************
    Parse and perform a close statement
******************************************/
int exec_close()
{
    struct variables filepointer;
    register int var;
    char infotext[STRING_LEN];

    sprintf(infotext, "instr:[%d] COMMAND: CLOSE: ", curr_instruction);

    get_token();
    if(tok != SHARP) serror(34);

    get_exp(&filepointer);
    if(filepointer.type != NUMERIC_VAR) serror(0);

    sprintf(infotext, "%s%d\n", (int)filepointer.value.dv);

    fclose(data_files[(int)filepointer.value.dv].fp);
    data_files[(int)filepointer.value.dv].locked = FALSE;

    for(var=0; var<NUM_VARS; ++var) {
        /* Check for all fields associated with this pointer and reset them */
        if(vars[var].data_file_index == (int)filepointer.value.dv) {
            vars[var].field_index = 0;
            vars[var].size = 0;
            if(vars[var].isdim) {
                if(vars[var].dim == 1)
                    vars[var].index.x = 0;
                else if(vars[var].dim == 2) {
                    vars[var].index.x = 0;
                    vars[var].index.y = 0;
                } else {
                    vars[var].index.x = 0;
                    vars[var].index.y = 0;
                    vars[var].index.z = 0;
                }
            }
        }
    }

    log_printf(infotext);
}

/*****************************************
    Parse and perform a FIELD statement
*****************************************/
int exec_field()
{
    char _var[VAR_LEN];
    int var, recsize=0;
    unsigned short count=0;
    struct variables value, x, y, z, filepointer, numexpr;
    struct node vec;
    short isdim = FALSE;
    unsigned long ammount_bytes=0;
    unsigned int ammount_fields=0;

    get_token();
    if(tok != SHARP) serror(34);

    get_exp(&filepointer);
    if(filepointer.type != NUMERIC_VAR) serror(0);
    if(filepointer.value.dv < 1 && filepointer.value.dv > NUM_FILES) serror(21);

    /* First, test if file space is really in use */
    if(!data_files[(int)filepointer.value.dv].locked) serror(38);

    /* So, test if file space opened randomically */
    if(data_files[(int)filepointer.value.dv].method != RANDOM) serror(40);

    get_token();
    if(tok != COMMA) serror(26);

    do {
        get_exp(&numexpr);
        if(numexpr.value.dv < 1 && numexpr.value.dv > 255) serror(21);
        recsize += (int)numexpr.value.dv;

        /* increase ammount of allocated bytes */
        ammount_bytes += recsize;

        get_token();
        if(tok != AS) serror(33);

        /* Get the variable name */
        get_token();
        if(token_type != VARIABLE) serror(4);

        /* increase ammount of declared fields*/
        ammount_fields += 1;

        strcpy(_var, token);
        to_upper(_var);
        /* If we are here, the var is acceptable */
        var = find_var(_var); 

        get_token();
        if(tok == LPAREN) { /* It is a vector assignment */
            isdim = TRUE; /* This will be useful below */
            if(vars[var].dim < 1) serror(23); /* var has no dim */

            /* get the value */
            get_exp(&x);
            if(x.type != NUMERIC_VAR) serror(0);  /* check type and bounds */
            if((int)x.value.dv < OPTION_BASE || (int)x.value.dv > vars[var].range.x) serror(25);

            get_token();
            if(tok == COMMA) {
                if(vars[var].dim < 2) serror(21); /* invalid range  */

                /* get the value */
                get_exp(&y);

                if(y.type != NUMERIC_VAR) serror(0); /* check type and bounds */
                if((int)y.value.dv < OPTION_BASE || (int)y.value.dv > vars[var].range.y) serror(25);

                get_token();
                if(tok == COMMA) {
                    if(vars[var].dim < 3) serror(21); /* invalid range */

                    /* get the value */
                    get_exp(&z);

                    if(z.type != NUMERIC_VAR) serror(0); /* check type and bounds */
                    if((int)z.value.dv < OPTION_BASE || (int)z.value.dv > vars[var].range.z) serror(25);

                    get_token();
                    if(tok != RPAREN) serror(22);
                } else if(tok != RPAREN) serror(0);
            } else if(tok != RPAREN) serror(0);
        } else putback();

        get_token();
        if(tok != COMMA && tok != EOL) serror(0);

        /* update record size in file space */
        data_files[(int)filepointer.value.dv].recsize = recsize;

        /* define variable as a field */
        strcpy(vars[var].name, _var); /* set var name */
        vars[var].type = STRING_VAR;  /* set var type */
        vars[var].is_field = TRUE;    /* set var as field */
        vars[var].field_index = count++;  /* set field index */
        vars[var].data_file_index = (int)filepointer.value.dv; /* set the file pointer associated to the field */
        vars[var].size = (int)numexpr.value.dv; /* set field size */
        if(isdim) {
            if(vars[var].dim == 1)
                vars[var].index.x = (int)x.value.dv;
            else if(vars[var].dim == 2) {
                vars[var].index.x = (int)x.value.dv;
                vars[var].index.y = (int)y.value.dv;
            } else {
                vars[var].index.x = (int)x.value.dv;
                vars[var].index.y = (int)y.value.dv;
                vars[var].index.z = (int)z.value.dv;
            }
        }
    } while(tok == COMMA);

    log_printf("instr:[%d] COMMAND: FIELD: INDEX #%d: (%d) FIELDS: ALLOCATED: %(ld) BYTES", curr_instruction,
                                                                                            (int)filepointer.value.dv,
                                                                                            ammount_fields,
                                                                                            ammount_bytes);
}

/*****************************************
    Parse and perform a GET statement
*****************************************/
int exec_get()
{
    register int i,j;
    unsigned short count=0;
    struct variables filepointer;
    struct node vec;
    char str[STRING_LEN];

    get_token();
    if(tok != SHARP) serror(34);

    get_exp(&filepointer);
    if(filepointer.type != NUMERIC_VAR) serror(0);
    if(filepointer.value.dv < 1 && filepointer.value.dv > NUM_FILES) serror(21);

    /* First, test if file space is really in use */
    if(!data_files[(int)filepointer.value.dv].locked) serror(38);

    /* So, test if file space opened randomically */
    if(data_files[(int)filepointer.value.dv].method != RANDOM) serror(40);

    /* Check how many fields are associated to this filespace */
    for(i=0; i<NUM_VARS; ++i)
        if(vars[i].is_field && vars[i].data_file_index == (int)filepointer.value.dv) count++;

    for(i=0; i<count; ++i) {
        for(j=0; j<NUM_VARS; ++j) {
            if(vars[j].is_field && vars[j].data_file_index == (int)filepointer.value.dv && vars[j].field_index == i) {
                memset(str, '\0', STRING_LEN);
                fread(&str, vars[j].size, 1, data_files[(int)filepointer.value.dv].fp);
                if(!vars[j].isdim) {
                    strcpy(vars[j].value.sv, str);
                } else {
                    /* prepare vector data for updating */
                    vec.var_index = j;
                    vec.index.x = vars[j].index.x;
                    vec.index.y = vars[j].index.y;
                    vec.index.z = vars[j].index.z;
                    vec.type = STRING_VAR;
                	strcpy(vec.value.sv, str);

                    /* If index is already there, remove it */
                    if(search(vec, vars[j].dim, array_items)) deleteNodeValue(vec, vars[j].dim, &array_items);
                    /* Insert at the new value at the end of the "list" */
                    array_items = InsertAtEnd(vec, &array_items);
                }
            }
        }
    }

    log_printf("instr:[%d] COMMAND: GET: INDEX #%d\n", curr_instruction, (int)filepointer.value.dv);
}

/*****************************************
    Parse and perform a PUT statement
*****************************************/
int exec_put()
{
    register int i,j;
    unsigned short count=0;
    struct variables filepointer;
    struct node vec, *return_vec;

    get_token();
    if(tok != SHARP) serror(34);

    get_exp(&filepointer);
    if(filepointer.type != NUMERIC_VAR) serror(0);
    if(filepointer.value.dv < 1 && filepointer.value.dv > NUM_FILES) serror(21);

    /* First, test if file space is really in use */
    if(!data_files[(int)filepointer.value.dv].locked) serror(38);

    /* So, test if file space opened randomically */
    if(data_files[(int)filepointer.value.dv].method != RANDOM) serror(40);

    /* Check how many fields are associated to this filespace */
    for(i=0; i<NUM_VARS; ++i)
        if(vars[i].is_field && vars[i].data_file_index == (int)filepointer.value.dv) count++;

    for(i=0; i<count; ++i) {
        for(j=0; j<NUM_VARS; ++j) {
            if(vars[j].is_field && vars[j].data_file_index == (int)filepointer.value.dv && vars[j].field_index == i) {
                if(!vars[j].isdim)
                    fwrite(vars[j].value.sv, vars[j].size, 1, data_files[(int)filepointer.value.dv].fp);
                else {
                    vec.var_index = j;
                    vec.index.x = vars[j].index.x;
                    vec.index.y = vars[j].index.y;
                    vec.index.z = vars[j].index.z;

                    if(return_vec = search(vec, vars[j].dim, array_items))
                        fwrite(return_vec->value.sv, vars[j].size, 1, data_files[(int)filepointer.value.dv].fp);
                }
            }
        }
    }

    log_printf("instr:[%d] COMMAND: PUT: INDEX #%d\n", curr_instruction, (int)filepointer.value.dv);
}

/*****************************************
    Parse and perform a SEEK statement
*****************************************/
int exec_seek() 
{
    struct variables filepointer, pos;

    get_token();
    if(tok != SHARP) serror(34);

    get_exp(&filepointer);
    if(filepointer.type != NUMERIC_VAR) serror(0);
    if(filepointer.value.dv < 1 && filepointer.value.dv > NUM_FILES) serror(21);

    get_token();
    if(tok != COMMA) serror(26);

    get_exp(&pos);
    if(pos.type != NUMERIC_VAR) serror(0);

    /* First, test if file space is really in use */
    if(!data_files[(int)filepointer.value.dv].locked) serror(38);

    if(data_files[(int)filepointer.value.dv].recsize > 0 && data_files[(int)filepointer.value.dv].method == RANDOM)
        fseek(data_files[(int)filepointer.value.dv].fp, (int)pos.value.dv*data_files[(int)filepointer.value.dv].recsize, SEEK_SET);
    else
        fseek(data_files[(int)filepointer.value.dv].fp, (int)pos.value.dv, SEEK_SET);

    log_printf("instr:[%d] COMMAND: SEEK: INDEX #%d\n", curr_instruction, (int)filepointer.value.dv);
}
