/*------------------------------------------------------------------------*
 * tzset(): POSIX-compatible timezone handling.                           *
 * ---------------------------------------------------------------------- *
 * This version of tzset() provides more correct analysis of the          *
 * environment variable TZ and DST rules than the function from standard  *
 * libraries of Borland C and Microsoft C.                                *
 *                                                                        *
 * Also provides _isDST() and _isindst() for internal use in the          *
 * Borland C or Microsoft C standart libraries.                           *
 *                                                                        *
 * Note: 'M' format of DST rules supported only, but time > 24 hours      *
 *       allowed, thus "Sun after fourth Sat" or "Sun >= 23"              *
 *       as in GB will be: "GMT0BST,M3.5.0/1,M10.4.6/25"                  *
 * BUG: 2100 year is wrongly treated as leap!                             *
 *                                                                        *
 *                                    (c) 1995,1997 by Timofei Bondatenko *
 *------------------------------------------------------------------------*/
#include <time.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h> /* TZNAME_MAX */

#ifdef   _MSC_VER
#define  tzset     _tzset
#define  tzname    _tzname
#define  timezone  _timezone
#define  daylight  _daylight
#define MONTH_DAY(x) (_days[x]+1)
extern int _days[13];
#else
#define MONTH_DAY(x)  (_monthDay[x])
const /*unsigned*/ int _monthDay[ ] =
{   0,   31,  59,  90, 120, 151, 181, 212, 243, 273, 304, 334, 365 };

#ifndef timezone
long    timezone;
#endif
#ifndef daylight
 int    daylight; /*  Apply daylight savings */
#endif
#endif /*_MSC_VER*/

#ifndef altzone
long    altzone; /*   timezone for daylight */
#endif

#ifndef TZNAME_MAX
#ifdef  tzname
#define TZNAME_MAX  3
#else
#define TZNAME_MAX 16 /* "GMT+10:30 DST" */
#endif
#endif

#ifndef     tzname
static char _stdZone[TZNAME_MAX+1] = "GMT",
            _dstZone[TZNAME_MAX+1] = "   ";
char *const tzname[2] = { _stdZone, _dstZone };
#endif

/******************* Startup setting, Compiler-dependent! *****************/
#if   defined(_MSC_VER)
void __tzset(void) { tzset(); }
#elif defined(__BORLANDC__)
#pragma startup tzset 30
#endif
/**************************************************************************/

struct _TimeRule { int mwd[3]; /* month, week, wday; */
                   int hour;
                 };

static struct _TimeRule summer, winter;
    /* Russia: last Sun of Mar 02:00 / last Sun of Oct 03:00
          USA: first Sun of Apr / last Sun of Oct 02:00a */
static const struct _TimeRule summDef = { {  4, 1, 0 }, 2 },
                              wintDef = { { 10, 5, 0 }, 2 };

typedef unsigned char uchar;

static long _parse_time(char **str)   /* time string: [+|-]00[:00[:00]] */
{                                     /* Returns: in seconds            */
 long tim = 0, mul = 60 * 60;
 char *s = *str;

 if (*s == '-') mul = -mul, s++;
 else if (*s == '+') s++;

 while(isdigit((uchar)*s))
   {
    tim += mul * atoi(s); mul /= 60;
    while(isdigit((uchar)*(++s)));
    if (mul && *s == ':') s++;
    else break;
   }
 *str = s;
 return tim;
}

static int _parse_date(struct _TimeRule *xd, char **str)
{                            /* Date format: Mmm.nn.dd[/hour] */
 int i = 0;                  /* Return:  0 if Ok              */
 char *s = *str;

 if (*s++ != ',' || *s++ != 'M') return -1;
 for(;;)
   if (isdigit((uchar)*s))
     {
      xd->mwd[i] = atoi(s);
      while(isdigit((uchar)*(++s)));
      if (++i == 3) break;
      if (*s == '.') s++;
      else return -1;
     }
   else return -1;

 xd->hour = *s == '/'? (s++, (int)(_parse_time(&s) / (60 * 60))): 2;
 *str = s;
 return 0; /* Ok */
}

static int _parse_name(char *tzn, char **str)
{
 int ii, ch;

 for(ii = 0; (ch = (uchar)(*str)[ii]) && !isdigit(ch) &&
              ch != '-' && ch != '+' && ch != ','; ii++)
   if (ii < TZNAME_MAX) *tzn++ = ch;
 if (ii)
   {
    *str += ii; *tzn = '\0';
   }
 else --ii;
 return ii;
}

/*--------------------------------------------------------------------------*
  tzset() sets local timezone info base on the "TZ" environment string:
  TZ=<STD><hour1>[<DST>[<hour2>]',M'<day1>['/'<hour3>]',M'<day2>['/'<hour4>]]
  hour  :: [+|-]hh[:mm[:ss]] time, offset of local timezone, relativety GMT:
           0 < on the west, 0 > on the east;
  day   :: mm.nn.ww date of change to DST/STD, where mm- month (1...12);
           ww- day of week (0=Sun...6); nn- week in the month (1...5=last).
           for example: M3.5.0 = last Sun in Mar;
  STD   ::= name of STD timezone (GMT, UTC, MSK, EST etc...);
           must not include characters "[0-9]:,+-";
  hour1 ::= offset of standard (astronomical) time - STD;
  DST   ::= name of DST (daylight saving) timezone (MSD, EDT ...)
            if acceptable; must not include characters "[0-9]:,+-";
  hour2 ::= offset of DST - optional, assuming (hour1 - 01:00:00) by default;
  day1,2 ::= date of change to DST/STD;
  hour3,4 ::= time of change to DST/STD (by default = 02:00am)
              can be >= 24:00, for example in GB: "M10.4.6/25" mean
                                 September, Sun after fourth Sat 1:00am.
*---------------------------------------------------------------------------*/

void tzset(void)
{
 char *env;
#ifndef tzname
 (char*)tzname[0] = _stdZone;
 (char*)tzname[1] = _dstZone;
#endif

 if (!(env = getenv("TZ")) || *env == ':') return;
#if   0  /* Same TZ name ? */
 if (!strcmp(env, tzname[0])) return; /* TimeZone changed? */
#elif 0  /* Hashing for skip parse */
{
 static int hash;
 unsigned char *sps = (unsigned char*)env;
 int hsh, sf; hsh = 0;
 while(*sps)
   {
    if (hsh < 0) hsh <<= 1, hsh++;
    else hsh <<= 1;
    hsh ^= *sps++;
   }
 if (hash == hsh) return;
 hash = hsh;
}
#endif   /* Hashing */

 if (0 > _parse_name(tzname[0], &env)) return;
 timezone = _parse_time(&env);
 strcpy(tzname[1], "   ");
 daylight = 0; summer = summDef; winter = wintDef;
 if (0 > _parse_name(tzname[1], &env)) return;
 daylight++;
 altzone = isdigit((uchar)*env) ||
                   '-' == *env ||
                   '+' == *env ? _parse_time(&env): timezone - 60 * 60;
 if (_parse_date(&summer, &env)) return;
 if (_parse_date(&winter, &env)) return;
}

#if 0
#define PRECIOUS_2100 0   /* 0 is Ok for years < 2100 */

/* Determinates where year is leap or not. year 0 = 1900 */

int __is_leap_year(int year)
{
 return !(year & 3)
#if PRECIOUS_2100
     && ((year % 100) || !((1900 + year) % 400))
#endif
/*****/;
}

/* Computes number of leap years after 1970 to (1900+year),
   not including (1900+year).
   For example: 72->0, 73->1, 74->1, 75->2 ... */

int __leap_years(int year)
{
 return (year-1 - 68 >> 2) /* 1968 was leap */
#if PRECIOUS_2100
      - (year-1) / 100 + (year-1 + 300) / 400
#endif
/*****/;
}

/* Computes day-of-year from
   year 0 = 1900, month = 0...11, mday = 1...31,
   month = 12 allowed, mday out of range allowed */

int __year_day(int year, int month, int mday)
{
 return mday - 1 + MONTH_DAY(month) +
       (month > 1 && __is_leap_year(year));
}

#else
#define __year_day(year,month,mday) \
       (mday - 1 + MONTH_DAY(month) + (month > 1 && !(year & 3)))
#endif

/*--------------------------------------------------------------------------*
 __mday() - helper for _isDST(), which determinates 'week's 'wday',
            as in 'M'-format.
 YEAR 0 = 1900... .
 Returns: div_t::quot = the day-of-year + hour/24, (0...365),
          div_t::rem = hour%24.
*---------------------------------------------------------------------------*/

static div_t __mday(int year, struct _TimeRule *rul)
{
 int ii, fday, lday;
 div_t dv;

 ii = rul->mwd[0]; /* month */
 fday = MONTH_DAY(ii - 1);  /* first day of the month */
 lday = MONTH_DAY(ii) - 7;  /* first day of the last 7-days in month */
 if (!(year & 3) && ii > 1) /* Leap && > Jan */
   {
    lday++;
    if (ii > 2) fday++; /* > Mar */
   }
 ii = rul->mwd[2] - /* 365 % 7 == 1 */
     ((year - 70)/* *365*/ + fday + 4 + /* 01-01-70 was Thursday */
     ((year - 68 - 1) >> 2)) % 7; /* leap days, 1968 was leap */
 fday += ii < 0 ? 7 + ii: ii; /* fday = first mwd[2] wday in the month */
 for(ii = 0; ++ii < rul->mwd[1] && fday < lday; fday += 7);
 dv = div(rul->hour, 24);
 dv.quot += fday;
 return dv;
}

/*--------------------------------------------------------------------------*
 _isDST() - determines whether daylight savings is in effect (for Borland C).
 If month is 0, yday is the day of the year (0...365), otherwise yday is the
 day of the month (0...30), year = 0 --> 1970.
 Returns 1 if daylight savings is in effect for the given date.
*---------------------------------------------------------------------------*/
#if defined(__BORLANDC__) && !defined(__FLAT__)
#define _isDST  __isDST
int pascal near
#else
int
#endif
    _isDST(int hour, int yday, int month, int year)
{
 div_t to_dst, to_std;

 year += 70;         /* leap year < 2100 !! */
 if (!daylight /*|| year <= 80*/) return 0;

 if (month)             /* if month+day of month given */
   yday = __year_day(year, month - 1, yday + 1);
 to_dst = __mday(year, &summer);
 to_std = __mday(year, &winter);
 if (to_dst.quot > to_std.quot)  /* Ausralia & Argentina */
   { if (yday > to_dst.quot || yday < to_std.quot) return 1; }
 else { if (yday > to_dst.quot && yday < to_std.quot) return 1; }

 if (yday == to_dst.quot)
   {
    if (hour >= to_dst.rem) return 1;
   }
 else if (yday == to_std.quot)
   {
    if (hour < to_std.rem)
      return /*hour < dv.rem - (timezone - altzone)/(60*60)? 1: -*/1;
   }
 return 0;
}

/*--------------------------------------------------------------------------*
 _isindst() - same as _isDST() for Microsoft C.
              determines whether daylight savings is in effect.
 *--------------------------------------------------------------------------*/

int _isindst(struct tm *pt)
{
 return _isDST(pt->tm_hour, pt->tm_yday, 0, pt->tm_year - 70);
}
/* end of tzset.c ---------------------------------------------------------*/