*=================================================================
* EXTRA.PRG
* Copyright (c) 1996-1997  Daniele Giacomini
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License as
* published by the Free Software Foundation; either version 2 of
* the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public
* License along with this program; if not, write to the Free
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
* USA.
*
*=================================================================
* EXTRA.PRG
*
* This file contains some standard _extra_ functions.
*
*
* The mouse library used:
*
*      ClipMous.LIB
*      Simple library for mouse support under Clipper
*      (c) Martin Brousseau
*      FREEWARE
*      original package: CLIPMOUS.ZIP
*
*
*=================================================================
* Functions:
*
*
* characters()          Show a list of characters and return the
*                       corresponding value.
*
* dbRddSelect()         Select a RDD from a list of available
*                       ones.
*
* dbModifyStructure()   Create or modify file structure.
*
* fldArray()            Show a list of Fields of the active Alias
*                       and returns an array with the data of the
*                       selected field.
*
* fldNormal()           Show a list of Fields of the active Alias
*                       and returns the name of the selecte one.
*
* fldIndex()            Show a list of Fields of the active Alias
*                       and returns the name of the selecte one
*                       transformed in a way to be easyly used
*                       for indexes.
*
* fldRelation()         Show a list of Fields of the active Alias
*                       and returns the name of the selecte one
*                       transformed in a way to be easyly used
*                       for relations.
*
* fldAlign()            Show a list of Fields of the active Alias
*                       and returns the name of the selecte one
*                       transformed in a way to be easyly used
*                       for text substitution.
*
* frm()                 Form file editing.
*
* lbl()                 Label file editing.
*
* selectAlias()         Select an Alias name form a list.
*
* selectArray()         Return an array containing the Aliases
*                       names.
*
* selectOrder()         Select an Order number from a list.
*
*=================================================================
*
*=================================================================
* STATIC FUNCTION NAME TRANSLATION
*=================================================================

#define DBMODIFYSTRUCTURE_VALID_NAME            _Sub_00001
#define DBMODIFYSTRUCTURE_VALID_TYPE            _Sub_00002
#define DBMODIFYSTRUCTURE_VALID_DIM             _Sub_00003
#define DBMODIFYSTRUCTURE_VALID_DEC             _Sub_00004

*=================================================================
* FILE INCLUSION
*=================================================================

#include "standard.ch"

*=================================================================
* STATIC
*=================================================================



*=================================================================
* CHARACTERS()
*=================================================================
function characters()
*
* This function shows a list of characters.
*
* The decimal value corresponding to the selected character is
* returned.
*

    local cOldColor     := setcolor()
    local lOldSetMouse  := setMouse( .F. )
    local cOldScreen
    local nSetCursor    := setcursor ( SETCURSOR_NORMAL )
    local bOld_F1       := setkey( K_F1, NIL )
    local nOldRow       := row()
    local nOldCol       := col()
    local nChoice       := 0
    local nTop          := 0
    local nLeft         := maxcol()-14
    local nBottom       := maxrow()
    local nRight        := maxcol()
    local acItems       := {}
    local nI

    *-------------------------------------------------------------
    * Prepare the array for aChoice().
    *-------------------------------------------------------------

    for nI := 1 to 256
        aadd( acItems, "chr(" + str(nI, 3) + ") == " + chr(nI) )
    end

    *-------------------------------------------------------------
    * Create a kind of window.
    *-------------------------------------------------------------

    cOldScreen    :=;
        mouseScrSave( nTop, nLeft, nBottom, nRight )

    setcolor( COLOR_BODY )

    dispBoxShadow(;
        nTop, nLeft, nBottom, nRight,;
        1,;
        dispBoxColor( 1 ),;
        dispBoxColor( 2 );
    )

    say(;
        nTop+1, nLeft+1,;
        padc( "CHR()", nRight-nLeft-1 ),;
        ,;
        COLOR_HEAD;
    )

    *-------------------------------------------------------------
    * Start aChoice() to select the character.
    *-------------------------------------------------------------

    nChoice :=;
        achoice(;
            nTop+2, nLeft+1, nBottom-1, nRight-1,;
            acItems, , , .F.;
        )

    *-------------------------------------------------------------
    * Restore old values.
    *-------------------------------------------------------------

    mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )
    setcursor ( nSetCursor )
    setcolor( cOldColor )
    setkey( K_F1, bOld_F1 )
    setpos( nOldRow, nOldCol )
    setMouse( lOldSetMouse )

    *-------------------------------------------------------------
    * Return selected character number.
    *-------------------------------------------------------------

    return nChoice

#ifndef RUNTIME
*=================================================================
* DBMODIFYSTRUCTURE()
*=================================================================

#define DBMODIFYSTRUCTURE_HEAD;
    "Database file structure"

#define DBMODIFYSTRUCTURE_HEAD_SAVE;
    "Save Structure"

#define DBMODIFYSTRUCTURE_SHURE;
    "Inproper structure modification;" +;
    "may result in a loss of data."

#define DBMODIFYSTRUCTURE_SAVEAS;
    "Save file as:"

#define DBMODIFYSTRUCTURE_NEW_RDD;
    "RDD:"

#define DBMODIFYSTRUCTURE_NO_FILE;
    "The file do not exists."

#define DBMODIFYSTRUCTURE_TEMP_DELETE;
    "Do you want to delete the temporary file?"

#define DBMODIFYSTRUCTURE_ERROR_NAME;
    "The name may contain letters, numbers and '_' " +;
    "and must start with a letter." +;
    NL(1) +;
    "Space and other simbols are not allowed inside the name."

#define DBMODIFYSTRUCTURE_ERROR_FIELD_TYPE;
    "Field type may be 'C', 'N', 'L', 'D' and 'M'"

#define DBMODIFYSTRUCTURE_ERROR_FIELD_NUMERIC_TOO_LITTE;
    "A numeric field must have a length greater then 0"

#define DMBODIFYSTRUCTURE_ERROR_FIELD_DECIMAL_TOO_BIG;
    "Decimal length may be 0 or less than integer length -1"

#define DMBODIFYSTRUCTURE_ERROR_FIELD_DECIMAL_TOO_LITTLE;
    "Decimal length may not be negative"

#define DBMODIFYSTRUCTURE_HELP_SAVE;
    "dbModifyStructure()" +;
    NL(3) +;
    "Save a " + _EXTENTION_DBF + " file." +;
    NL(3) +;
    "Insert the name and the database driver " +;
    "to use to save the file."

#define DBMODIFYSTRUCTURE_HELP;
    "dbModifyStructure()" +;
    NL(3) +;
    "This function permits to create or modify a new database file." +;
    NL(3) +;
    "FIELD NAME is the column name that may be long " +;
    "10 character max, it " +;
    "must begin with a letter and may contains also " +;
    "numbers and '_'; the space ' ' is not allowed." +;
    NL(2) +;
    "TYPE is the column type that may be 'C' for character, 'N' for " +;
    "numeric, 'D' for data, 'L' for logic and 'M' for " +;
    "memo." +;
    NL(2) +;
    "LENGTH is the column width in character." +;
    NL(2) +;
    "DECIMAL is the decimal portion of LENGTH when the " +;
    "column type identifies a number. The maximum decimal " +;
    "dimention is LENGTH -2 as the decimal point takes one " +;
    "place." +;
    NL(3) +;
    "The maximum column (field) width may be different " +;
    "depending on the active database driver." + NL(1) +;
    "For DBFNTX, the standard, the maximum dimentions are " +;
    "as follows:" +;
    NL(2) +;
    "C - Character - max length  = 255" + NL(1) +;
    "                max decimal = 250" + NL(1) +;
    "    Width = LENGTH + (DECIMAL * 256)" + NL(1) +;
    "    Max Width = 64255" +;
    NL(2) +;
    "N - Numeric - max length  = 16" + NL(1) +;
    "              max decimal = LENGTH-2" + NL(1) +;
    "    Width = LENGTH" +;
    NL(2) +;
    "D - Data - length  = leave 0" + NL(1) +;
    "           decimal = leave 0" +;
    NL(2) +;
    "L - Logic - length  = leave 0" + NL(1) +;
    "            decimal = leave 0" +;
    NL(2) +;
    "M - Memo - length  = leave 0" + NL(1) +;
    "           decimal = leave 0" +;
    NL(3) +;
    "After completing the column (field) description, " +;
    "press [Esc] to exit: the name to use to save " +;
    "the new structure will be asked."

*=================================================================
function dbModifyStructure( cName )
*
* dbModifyStructure( <cName> ) --> <x> see below
*
*
* <cName>   database file name.
*
*
* -->   x = NIL     if the the function termiates without
*                   completing the task.
*
*       x = array   if successfull:
*
*           x[1] = char     the filename created or modified,
*           x[2] = char     the database driver used,
*           x[3] = array    the structure array.
*
*
* It modifies or creates the .DBF file <cName>.
*

    local aoGet                 := {}
    local aButtons              := {}
    local aTab                  := TAB_DEFAULT
    
    local bOldErrorHandler
    local cOldScreen
    local cOldColor             := setcolor( COLOR_BASE )
    local nOldCursor            := setcursor( SETCURSOR_NORMAL )
    local nOldSelect            := select()
    local lPreviousHelp
    local bOld_F1
    local bOld_F2               := setkey( K_F2, NIL )
    local nOldRow               := row()
    local nOldCol               := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth
    local lOldDeleted           := set( _SET_DELETED, .T. )

    local aStruct               := {}
    local acCol
    local acColSayPic
    local acColHead
    local abColValid
    local abColMsg
    local nOrder                := 0
    local lCreate               := .F.
    local lGoOn                 := .F.
    local cNewName
    local cNewRdd
    local nI
    local cTempPath             := strTempPath()

    local aReturn               := {}

    *-------------------------------------------------------------
    * Check if a previous help exists. If so, no local help
    * will be set up.
    *-------------------------------------------------------------

    if setkey( K_F1 ) == NIL

        lPreviousHelp := .F.

        bOld_F1 :=;
            setkey(;
                K_F1,;
                { || Text( DBMODIFYSTRUCTURE_HELP ) };
                )

    else

        lPreviousHelp := .T.

    end

    *-------------------------------------------------------------
    * Start a sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Check if it is a creation or a structure modification.
        * Check also file existance.
        *---------------------------------------------------------

        if valtype( cName ) <> "C";
            .or. empty( cName )

            *-----------------------------------------------------
            * No name was passed: the file must be created.
            *-----------------------------------------------------

            lCreate := .T.
            cName := padr( "*." + _EXTENTION_DBF, _MAX_STRING_LEN )

        else

            *-----------------------------------------------------
            * The file must be modified.
            * Correct possible problems with file extention.
            *-----------------------------------------------------

            lCreate := .F.
            cName := strAddExtention( cName, _EXTENTION_DBF )

            *-----------------------------------------------------
            * Check if the file that should be modified exists.
            * If it don't, tell it and break.
            *-----------------------------------------------------

            if !file( cName )
                alertBox(;
                    cName + NL(1) +;
                    DBMODIFYSTRUCTURE_NO_FILE;
                    )

                break                                   // BREAK

            end

        end

        *---------------------------------------------------------
        * If you are modifying the structure, watch your step!
        * If the user don't want to continue, break.
        *---------------------------------------------------------

        if !lCreate

            if  alertBox(;
                    cName + NL(1) +;
                        DBMODIFYSTRUCTURE_SHURE, ;
                    { _MENU_EXIT, _MENU_CONTINUE };
                ) == 2

                *-------------------------------------------------
                * OK, proceed.
                *-------------------------------------------------

            else

                break                                   // BREAK

            end

        end

        *---------------------------------------------------------
        * Prepare an array for structure editing.
        *---------------------------------------------------------

        if lCreate

            *-----------------------------------------------------
            * New file, empty array: only the first empty line
            *-----------------------------------------------------

            aStruct := { { space(10), space(1), 0, 0 } }

        else

            *-----------------------------------------------------
            * To read the structure of the file, we have to open
            * it and that may cause runtime errors.
            *-----------------------------------------------------

            bOldErrorHandler := errorblock( {|e| ErrorHandler(e)} )
            begin sequence

                *-------------------------------------------------
                * If the file has a structure defect,
                * an error occurs and the file DBF_MODIFY
                * must be closed.
                *-------------------------------------------------

                dbusearea( .T., NIL, cName,;
                    "DBF_MODIFY", .F., .F. )

            recover

                ("DBF_MODIFY")->(dbclosearea())

                break                                   // BREAK

            end //sequence
            errorblock( bOldErrorHandler )

            *-----------------------------------------------------
            * If the file is locked, break.
            *-----------------------------------------------------

            if neterr()

                alertBox( alltrim( cName ) + NL(1) +;
                    _ERROR_FLOCK_FAILURE )

                break                                   // BREAK

            end

            *-----------------------------------------------------
            * The file was successfully opened.
            * Now create the structure array and correct it for
            * the purpose of editing.
            *-----------------------------------------------------

            aStruct := dbstruct()

            for nI := 1 to len( aStruct )
                aStruct[nI][1] := padr( aStruct[nI][1], 10 )
            next

            *-----------------------------------------------------
            * Make a temporary copy.
            * Rename is not possible as a memo (.dbt) file
            * can be present.
            *-----------------------------------------------------

            select( "DBF_MODIFY" )
            dbCopy(;
                cTempPath+"\TMP_MDST",;
                NIL,;
                { || WaitFileEval() };
                )
            waitFileEval(.T.)

            *-----------------------------------------------------
            * Close now the original file.
            *-----------------------------------------------------

            ("DBF_MODIFY")->( dbCloseArea() )

        end

        *---------------------------------------------------------
        * Finally we have the structure array.
        * We can now modify it.
        *---------------------------------------------------------

        *---------------------------------------------------------
        * Create a kind of window right aligned.
        *---------------------------------------------------------

        nBottom     := maxrow()
        nTop        := 0
        nRight      := maxcol()
        nLeft       := nRight - 33
        nWidth      := nRight - nLeft +1
        
        cOldScreen  := mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc(;
                DBMODIFYSTRUCTURE_HEAD,;
                nWidth-2;
                ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Prepare browse arrays
        *---------------------------------------------------------

        acColSayPic  := {;
            "!!!!!!!!!!",;
            "!",;
            "99999",;
            "9999";
            }

        acColHead    := {;
            "Field Name",;
            "Type",;
            "Length",;
            "Decimal";
            }

        abColValid   := {;
            {|a, i, j| DBMODIFYSTRUCTURE_VALID_NAME( a, i, j ) },;
            {|a, i, j| DBMODIFYSTRUCTURE_VALID_TYPE( a, i, j ) },;
            {|a, i, j| DBMODIFYSTRUCTURE_VALID_DIM( a, i, j ) },;
            {|a, i, j| DBMODIFYSTRUCTURE_VALID_DEC( a, i, j ) };
            }

        abColMsg     := {;
            {||""},;
            {||"C, N, D, L, M"},;
            {|a, i, j| iif( a[i][2] == "N", "", "" ) +;
                iif( a[i][2] == "C", "Teorical Max: 64000", "" ) +;
                iif( a[i][2] == "D", "8", "" ) +;
                iif( a[i][2] == "M", "10", "" ) +;
                iif( a[i][2] == "L", "1", "") },;
            {|a, i, j| iif( a[i][2] == "N" .and. a[i][3] > 2,;
                "Max " + alltrim( str( a[i][3]-2 ) ), "" ) +;
                iif( a[i][2] == "C",;
                "0", "" ) } }

        * ATB( <nTop>, <nLeft>, <nBottom>, <nRight>,
        *   <aArray>, [<nSubscript>],
        *   [<acColSayPic>],
        *   [<acColTopSep>], [<acColBodySep>], [<acColBotSep>],
        *   [<acColHead>], [<acColFoot>],
        *   [<abColValid>],
        *   [<abColMsg>],
        *   [<cColor>], [<abColColors>],
        *   [<lModify>],
        *   [lButtons|aButtons]
        *   )  --> aArray

        *---------------------------------------------------------
        * Do the browse.
        *---------------------------------------------------------

        aStruct := atb(;
            nTop+3, nLeft+1, nBottom-1, nRight-1,;
            aStruct,,;
            acColSayPic,;
            ,,,;
            acColHead,,;
            abColValid,;
            abColMsg,;
            ,,;
            .T.,;
            .T.;
            )

        *---------------------------------------------------------
        * Close window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * Prepare for save.
        *---------------------------------------------------------

        cNewName :=;
            padr( strAddExtention( cName, _EXTENTION_DBF ),;
            _MAX_STRING_LEN )

        cNewRdd :=;
            padr( rddsetdefault(),;
            _MAX_STRING_LEN )

        *---------------------------------------------------------
        * Create a kind of window aligned at bottom screen.
        * The help now changes.
        *---------------------------------------------------------

        if !lPreviousHelp

            setkey(;
                K_F1,;
                { || Text( DBMODIFYSTRUCTURE_HELP_SAVE ) };
            )

        end

        nBottom             := maxrow()
        nTop                := nBottom - 7
        nLeft               := 0
        nRight              := maxcol()
        nWidth              := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 16, 8 }

        cOldScreen          :=; 
            mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( DBMODIFYSTRUCTURE_HEAD_SAVE, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                   // FOREVER

            say(;
                nTop+3, nLeft+1,;
                DBMODIFYSTRUCTURE_SAVEAS;
            )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cNewName := x, cNewName ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { || trueSetkey( K_F2, {|| gvFileDir( @cNewName ) } ) },;
                { || gvFileExtention( @cNewName, _EXTENTION_DBF ) };
            )

            say(;
                row()+1, nLeft+1,;
                DBMODIFYSTRUCTURE_NEW_RDD,;
            )
            tab( aTab )
            get(;
                aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cNewRdd := x, cNewRdd ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { || trueSetkey( K_F2, {|| gvSubst( @cNewRdd, dbRddSelect() ) } ) },;
                { || !empty( cNewRdd ) };
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------
                                        
            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lGoOn := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN
                                         
                *-------------------------------------------------
                * [PgDn] means confirm.
                * Before, check for valid data.
                *-------------------------------------------------

                do case
                case; 
                    strAddExtention(cNewName, _EXTENTION_DBF);
                        <> strAddExtention( cName, _EXTENTION_DBF );
                    .and.; 
                    file(; 
                        strAddExtention(; 
                            cNewName,;
                            _EXTENTION_DBF; 
                        ); 
                    )

                    *---------------------------------------------
                    * The file name was changed or it is a new
                    * file and it already exists.
                    * overwrite?
                    *---------------------------------------------

                    if alertBox( alltrim(cNewName) + NL(1) +;
                        _ERROR_PROMPT_ALREADY_EXIST_OVERWRITE,;
                        { _MENU_NO, _MENU_YES } ) == 2

                        *-----------------------------------------
                        * Ok, overwrite!
                        *-----------------------------------------

                        lGoOn := .T.

                        exit                            // EXIT

                    else

                        *-----------------------------------------
                        * No! Don't save. Loop again to change
                        * the name
                        *-----------------------------------------

                    end

                case strCutExtention( cNewName ) == ""

                    *---------------------------------------------
                    * No valid name was given. Loop again.
                    *---------------------------------------------

                otherwise

                    *---------------------------------------------
                    * No other known problems: exit loop.
                    *---------------------------------------------

                    lGoOn := .T.
                    exit                                // EXIT

                end

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * Close window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        *---------------------------------------------------------
        * The editing is terminated.
        * Save if it was so required, else break.
        *---------------------------------------------------------

        if lGoOn

            // Prepare <cNewName>
            cNewName := strAddExtention( cNewname, _EXTENTION_DBF )

            // Prepare <aReturn>
            aadd( aReturn, alltrim(cNewName) )
            aadd( aReturn, alltrim(cNewRdd) )
            aadd( aReturn, aStruct )

        else

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * If still here, the file is created.
        *---------------------------------------------------------

        bOldErrorHandler := errorblock( {|e| ErrorHandler(e)} )
        begin sequence

            *-----------------------------------------------------
            * Create the file, but before, delete a possible
            * memo file with the same name.
            *-----------------------------------------------------

            cNewName := strCutExtention( cNewName )
            ferase( cNewName+".DBT" )
            dbcreate( cNewName, aStruct, alltrim( cNewRdd ) )

            *-----------------------------------------------------
            * We have now an empty new file. If the purpose was
            * to modify a structure, we now have to fill the
            * new file with old record.
            *-----------------------------------------------------

            if !lCreate

                *-------------------------------------------------
                * Open the file in exclusive mode.
                * I hope that noone has already locked it!
                *-------------------------------------------------

                dbusearea(;
                    .T.,;
                    alltrim(cNewRdd),;
                    alltrim(cNewName),;
                    "_NEW_FILE_",;
                    .F., .F.;
                    )

                *-------------------------------------------------
                * Append old data.
                *-------------------------------------------------

                dbApp( cTempPath+"\TMP_MDST",, {||WaitFileEval()} )

                waitFileEval( .T. )

                *-------------------------------------------------
                * Close it.
                *-------------------------------------------------

                ("_NEW_FILE_")->(dbclosearea())

                *-------------------------------------------------
                * Should I delete the temporary file?
                *-------------------------------------------------

                if alertBox(;
                    cTempPath+"\TMP_MDST" + NL(1) +;
                    DBMODIFYSTRUCTURE_TEMP_DELETE,;
                    { _MENU_NO, _MENU_YES };
                    ) == 2

                    ferase( cTempPath+"\TMP_MDST." + _EXTENTION_DBF )
                    ferase( cTempPath+"\TMP_MDST.DBT" )

                end

            end

        recover

            *-----------------------------------------------------
            * If troubles occurred, try to remedy.
            *-----------------------------------------------------

            (cNewName)->( dbCloseArea() )

            dbusearea( .T., NIL, cTempPath+"\TMP_MDST", "TMP_MDST", .F. )
            select( "TMP_MDST" )

            dbCopy( cNewName, {|| waitFileEval() } )

            ("TMP_MDST")->( dbCloseArea() )

            waitFileEval( .T. )

        end //sequence
        errorblock(bOldErrorHandler)

    end //sequence

    *-------------------------------------------------------------
    * Restore values.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )
    set( _SET_DELETED, lOldDeleted )
    dbselectarea( nOldSelect )

    return aclone( aReturn )

*-----------------------------------------------------------------
static function DBMODIFYSTRUCTURE_VALID_NAME(;
    aArray, nRow, nCol;
    )
*
*
*

    local cName     := upper( rtrim( aArray[nRow][nCol] ) )
    local lValid    := .T.
    local nI

    begin sequence

        *---------------------------------------------------------
        * Check first character.
        *---------------------------------------------------------

        if (;
                  substr( cName, 1, 1 ) >= "A";
            .and. substr( cName, 1, 1 ) <= "Z";
            );
            .or.;
                  substr( cName, 1, 1 ) == "_"
            // Ok.
        else

            lValid := .F.

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Check the other characters.
        *---------------------------------------------------------

        for nI := 2 to len( cName )
            if (;
                      substr( cName, nI, 1 ) >= "0";
                .and. substr( cName, nI, 1 ) <= "Z";
                );
                .or.;
                      substr( cName, nI, 1 ) == "_"
                // Ok.
            else

                lValid := .F.

                break                                   // BREAK

            end
        next

        *---------------------------------------------------------
        * It seems valid.
        *---------------------------------------------------------

    end //sequence

    if !lValid
        waitFor( DBMODIFYSTRUCTURE_ERROR_NAME )
    end

    return lValid

*-----------------------------------------------------------------
static function DBMODIFYSTRUCTURE_VALID_TYPE(;
    aArray, nRow, nCol;
    )
*
*
*

    local cType := upper( aArray[nRow][nCol] )

    if !(;
              cType == "C";
         .or. cType == "N";
         .or. cType == "D";
         .or. cType == "L";
         .or. cType == "M";
         )
        waitFor(;
            DBMODIFYSTRUCTURE_ERROR_FIELD_TYPE;
            )
         return .F.
    end

    return .T.

*-----------------------------------------------------------------
static function DBMODIFYSTRUCTURE_VALID_DIM(;
    aArray, nRow, nCol;
    )
*
*
*

    if aArray[nRow][2] == "N";
        .and. aArray[nRow][3] <= 0
        //
        waitFor(;
            DBMODIFYSTRUCTURE_ERROR_FIELD_NUMERIC_TOO_LITTE;
            )
        return .F.
    end

    return .T.

*-----------------------------------------------------------------
static function DBMODIFYSTRUCTURE_VALID_DEC(;
    aArray, nRow, nCol;
    )
*
*
*

    do case
    case aArray[nRow][2] == "N";
        .and. aArray[nRow][4] > 0;
        .and. aArray[nRow][4] > aArray[nRow][3]-2
        //
        waitFor(DMBODIFYSTRUCTURE_ERROR_FIELD_DECIMAL_TOO_BIG)
        return .F.
    case aArray[nRow][4] < 0
        waitFor( DMBODIFYSTRUCTURE_ERROR_FIELD_DECIMAL_TOO_LITTLE )
        return .f.
    end

    return .T.

#endif //RUNTIME

*=================================================================
* DBRDDSELECT()
*=================================================================

#define DBRDDSELECT_HEAD;
    "Replaceable Database Drivers"

#define DBRDDSELECT_HELP;
    "dbRddSelect()" +;
    NL(3) +;
    "This function helps you to select a Database Driver from " +;
    " the available ones."

*=================================================================
function dbRddSelect()
*
* dbRddSelect() --> cRDD
*
* It returns a RDD name from the available ones.
*

    local aoGet             := {}
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NONE )
    local bOld_F1           := setkey( K_F1 )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local aRDD              := {}
    local aRDDX             := {}
    local nI                := 0

    *-------------------------------------------------------------
    * Check if a previous help exists. If so, no local help
    * will be set up.
    *-------------------------------------------------------------

    if bOld_F1 == NIL

        *---------------------------------------------------------
        * Set up a minimal help.
        *---------------------------------------------------------

        setkey( K_F1, { || Text(DBRDDSELECT_HELP) } )

    else
    
        *---------------------------------------------------------
        * There is already a help.
        *---------------------------------------------------------

    end

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        aRDD := rddlist( RDT_FULL )

        *---------------------------------------------------------
        * Delete the driver names that do not contains index.
        *---------------------------------------------------------

        nI := 1

        for nI = 1 to len( aRDD )

            if right( aRDD[nI], 1 ) == "X"  // driver with index

                aadd( aRDDX, aRDD[nI] )

            end

        next

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom     := maxrow()
        nRight      := maxcol()
        nTop        := 0
        nLeft       := nRight - 32
        nWidth      := nRight - nLeft +1

        cOldScreen := mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( DBRDDSELECT_HEAD, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        nI :=;
            achoice (;
                nTop+2, nLeft+1, nBottom-1, nRight-1,;
                aRDDX,,,.T.;
            )

        *---------------------------------------------------------
        * Close window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

    end //sequence

    *-------------------------------------------------------------
    * Restore old values.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    if nI <> 0

        return aRDDX[nI]

    end

    *-------------------------------------------------------------
    * if nI == 0, [Esc]
    *-------------------------------------------------------------

    return rddsetdefault()

*=================================================================
*  FLD ...
*=================================================================

#define FLD_AL_BUTTON_LEFT      "Left"
#define FLD_AL_BUTTON_CENTER    "Center"
#define FLD_AL_BUTTON_RIGHT     "Right"

#define FLD_ARRAY_HEAD;
    "Name     Type Length Decimal"

#define FLD_AL_CHR_HEAD;
    "Character alignment"

#define FLD_AL_MEMO_HEAD;
    "Memo max dimention"

#define FLD_AL_NUM_HEAD;
    "Number alignment"

#define FLD_AL_PROMPT_NORMAL;
    "Normal?"
#define FLD_AL_CHR_PROMPT_DIMENSION;
    "Dimention"
#define FLD_AL_CHR_PROMPT_ALIGNMENT;
    "Alignment"

#define FLD_AL_MEMO_PROMPT_DIMENSION;
    "Max dimention"

#define FLD_AL_NUM_PROMPT_LENGTH;
    "Length"
#define FLD_AL_NUM_PROMPT_DECIMAL;
    "Decimal"
#define FLD_AL_MEMO_PROMPT_LENGTH;
    "Insert memo length."

#define FLD_ERROR_MEMO_FILEDS_NOT_ALLOWED;
    "MEMO fields cannot be included."

#define FLD_TRUE;
    "True "
#define FLD_FALSE;
    "False"

#define FLD_HELP_FLD_CHOICE;
    "fldArray()" +;
    NL(3) +;
    "This function permits to select a Field " +;
    "(a column) from the active Alias. " +;
    NL(2) +;
    "The user have to move the cursor " +;
    "with the arrow keys, []/[], " +;
    "and select the name pressing [Enter]." +;
    NL(2) +;
    "Pressing [Esc], the function terminates."

#define FLD_AL_CHR_HELP;
    "fldAlign()" +;
    NL(3) +;
    "This function helps to define the dimension " +;
    "and the alignment of a field: " +;
    "in this case it is a character field." +;
    NL(2) +;
    "When normal is true (.T.), the field is left with " +;
    "no transformations." +;
    NL(2) +;
    "Otherwise, the dimension and alignment must " +;
    "be spesified exactly." +;
    NL(2) +;
    "Pressing [Esc] will be returned a normal " +;
    "alignment. Pressing [Pag] will be confirmd the choice selected."

#define FLD_AL_CHR_NUM;
    "flsAlign()" +;
    NL(3) +;
    "This function helps to define the dimension " +;
    "and the alignment of a field: " +;
    "in this case it is a numeric field." +;
    NL(2) +;
    "When normal is true (.T.), the number appears with " +;
    "a default transformation into string; " +;
    "otherwise, the dimension must " +;
    "be specified exactly. " +;
    NL(2) +;
    "Pressing [Esc] will be returned a normal " +;
    "alignment. Pressing [Pag] will be confirmed " +;
    "the choice selected." +;
    NL(3) +;
    "Examples:" +;
    NL(2) +;
    "Length, Decimals,  Number        Result    " + NL(1) +;
    "  10        0      12345,456     _____12345" + NL(1) +;
    "  10        2      12345,456     __12345.45" + NL(1) +;
    "  14        3      12345,456     _____12345.456" + NL(1) +;
    "   5        3      12345,456     *****" +;
    NL(2) +;
    "With the last case the result is an error."

*=================================================================
* FIELD - FIELD SELECTION
*=================================================================
function fldArray()
*
* fldArray() --> aFieldData
*            --> {}
*
* This function shows a list of Fields of the active Alias.
* If a field is selected, an array with its data is returned,
* else an empty array is retuned.
*

    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NONE )
    local bOld_F1           :=;
        setkey( K_F1, {|| Text( FLD_HELP_FLD_CHOICE )} )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local aField            := {}
    local aFieldEasy        := {}
    local nI                := 0

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * This function needs an active Alias.
        * If it doesn't exist, break.
        *---------------------------------------------------------

        if alias() == ""

             alertBox( _ERROR_NO_ALIAS )

             break                                     // BREAK

        end

        *---------------------------------------------------------
        * Copy the active Alias structure into <aField>.
        *---------------------------------------------------------

        aField := dbstruct()

        *---------------------------------------------------------
        * Prepare another array that will be used with the
        * function aChoice()
        *---------------------------------------------------------

        for nI := 1 to len( aField )
            aadd( aFieldEasy,;
                left( aField[nI][1] +;
                space(20), 11) +;
                aField[nI][2] + "    " +;
                str( aField[nI][3], 4, 0) + "   " +;
                str( aField[nI][4], 4, 0)  )
        next

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nTop       := 0
        nLeft      := maxcol()-32
        nBottom    := maxrow()
        nRight     := maxcol()
        nWidth     := nRight - nLeft +1

        cOldScreen := mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
            )

        say(;
            nTop+1, nLeft+1,;
            padc(;
                FLD_ARRAY_HEAD,;
                nWidth-2;
                ),;
            ,;
            COLOR_HEAD;
            )

        *---------------------------------------------------------
        * aChoice() does the real work.
        *---------------------------------------------------------

        nI :=;
            achoice(;
                nTop+3, nLeft+1, nBottom-1, nRight-1,;
                aFieldEasy,,,.T.;
            )

        *---------------------------------------------------------
        * Close window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )
    setpos( nOldRow, nOldCol )

    *-------------------------------------------------------------
    * Return an array containing the selected field data:
    * aField[nI] contains another array.
    *-------------------------------------------------------------

    if nI > 0
        return aclone( aField[nI] )
    end

    *-------------------------------------------------------------
    * An empty array is returned if no selection was made.
    *-------------------------------------------------------------

    return {}

*=================================================================
function fldNormal()
*
* fldNormal() --> cFieldName
*
* This function calls fldArray() and returns the selected
* field name or "" if no choice was made.
*

    local aMyField            := {}
    local cReturnString       := ""

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Select a field with fldArray().
        *---------------------------------------------------------

        aMyField := fldArray()

        *---------------------------------------------------------
        * Prepare the return string.
        *---------------------------------------------------------

        do case
        case len( aMyField ) == 0
            cReturnString := ""
        otherwise
            cReturnString := aMyField[1]
        end

    end //sequence

    *-------------------------------------------------------------
    * Return the selected field.
    *-------------------------------------------------------------

    return cReturnString

*=================================================================
function fldIndex()
*
* fldIndex() --> cString
*
* this function calls fldArray() and returns a character
* string usefull for index creation or "" if no choice
* was made.
*

    local aMyField            := {}
    local cReturnString       := ""

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Select a field with fldArray().
        *---------------------------------------------------------

        aMyField := fldArray()

        *---------------------------------------------------------
        * Create the right string.
        *---------------------------------------------------------

        do case
        case len( aMyField ) == 0
            cReturnString := ""
        case aMyField[2] == "C"
            cReturnString := "+upper(" + aMyField[1] + ")"
        case aMyField[2] == "N"
            cReturnString :=;
                "+str(" +;
                aMyField[1] +;
                "," +;
                alltrim(str(aMyField[3])) +;
                "," +;
                alltrim(str(aMyField[4])) +;
                ")"
        case aMyField[2] == "D"
            cReturnString := "+dtos(" + aMyField[1] + ")"
        case aMyField[2] == "L"
            cReturnString :=;
                "+transform(" + aMyField[1] + ",'L')"
        case aMyField[2] == "M"
            alertBox( FLD_ERROR_MEMO_FILEDS_NOT_ALLOWED )
            cReturnString := ""
        end

    end //sequence

    *-------------------------------------------------------------
    * Return the string.
    *-------------------------------------------------------------

    return cReturnString

*=================================================================
function fldRelation()
*
* fldRelation() --> cString
*
* This function calls fldArray() and returns a character
* string usefull for relation creation or "" if no choice
* was made.
*

    local aMyField            := {}
    local cAlias              := alias()
    local cReturnString       := ""

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Select a field with fldArray().
        *---------------------------------------------------------

        aMyField := fldArray()

        *---------------------------------------------------------
        * Create the right string.
        *---------------------------------------------------------

        do case
        case len( aMyField ) == 0
            cReturnString := ""
        case aMyField[2] == "C"
            cReturnString :=;
                "+upper(" +;
                cAlias +;
                "->" +;
                aMyField[1] +;
                ")"
        case aMyField[2] == "N"
            cReturnString :=;
                "+str(" +;
                cAlias + "->" + aMyField[1] +;
                "," +;
                alltrim(str(aMyField[3])) +;
                "," +;
                alltrim(str(aMyField[4])) +;
                ")"
        case aMyField[2] == "D"
            cReturnString :=;
                "+dtos(" +;
                cAlias +;
                "->" +;
                aMyField[1] +;
                ")"
        case aMyField[2] == "L"
            cReturnString :=;
                "+transform(" +;
                cAlias +;
                "->" +;
                aMyField[1] +;
                ",'L')"
        case aMyField[2] == "M"
            alertBox( FLD_ERROR_MEMO_FILEDS_NOT_ALLOWED )
            cReturnString := ""
        end

    end //sequence

    *-------------------------------------------------------------
    * Return the string.
    *-------------------------------------------------------------

    return cReturnString

*=================================================================
function fldAlign()
*
* fldAlign() --> cString
*
* This function calls fldArray() and asks for a picture.
* It returns a character string usefull for text substitution
* or "" if no choice was made.
*

    local aMyField          := {}
    local cReturnString     := ""
    local lNormal           := .F.
    local nDim              := 20
    local nMemoDim          := 80
    local cAlign            := "L"
    local nNumLen           := 10
    local nDecLen           := 0
    local lGoOn             := .F.

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Select a field with fldArray().
        *---------------------------------------------------------

        aMyField := fldArray()

        *---------------------------------------------------------
        * Create the right string.
        *---------------------------------------------------------

        do case
        case len( aMyField ) == 0

            cReturnString := ""

        case aMyField[2] == "C"

            lGoOn :=;
                fldAlChr( @lNormal, @nDim, @cAlign, aMyField[1] )

            do case
            case !lGoOn

                cReturnString := ""

            case lNormal

                cReturnString := aMyField[1]

            case cAlign == "L"

                cReturnString :=;
                    [padr(alltrim(] +;
                    aMyField[1] +;
                    [),] +;
                    alltrim(str(nDim)) +;
                    [)]

            case cAlign == "C"

                cReturnString :=;
                    [padc(alltrim(] +;
                    aMyField[1] +;
                    [),] +;
                    alltrim(str(nDim)) +;
                    [)]

            case cAlign == "R"

                cReturnString :=;
                    [padl(alltrim(] +;
                    aMyField[1] +;
                    [),] +;
                    alltrim(str(nDim)) +;
                    [)]

            otherwise

                cReturnString := aMyField[1]

            end

        case aMyField[2] == "N"

            nNumLen := aMyField[3]
            nDecLen := aMyField[4]
            lGoOn := fldAlNum( @lNormal, @nNumLen, @nDecLen )

            do case
            case !lGoOn

                cReturnString := ""

            case lNormal

                cReturnString :=;
                    [str(] +;
                    aMyField[1] +;
                    [)]

            case nNumLen > nDecLen

                cReturnString :=;
                    [str(] +;
                    aMyField[1] +;
                    [,] +;
                    alltrim(str(nNumLen)) +;
                    [,] +;
                    alltrim(str(nDecLen)) +;
                    [)]

            otherwise

                cReturnString :=;
                    [str(] +;
                    aMyField[1] +;
                    [)]

            end

        case aMyField[2] == "D"

            cReturnString :=;
                [dtoc(] +;
                aMyField[1] +;
                [)]

        case aMyField[2] == "L"

            cReturnString :=;
                [iif(] +;
                aMyField[1] +;
                [,"<] +;
                FLD_TRUE +;
                [>","<] +;
                FLD_FALSE +;
                [>" )]

        case aMyField[2] == "M"

            if  fldAlMemo( @nMemoDim )

                cReturnString :=;
                    [left(] +;
                    aMyField[1] +;
                    [,] +;
                    alltrim(str(nMemoDim)) +;
                    [)]

            else

                cReturnString := ""

            end

        end

    end //sequence

    *-------------------------------------------------------------
    * Return the string.
    *-------------------------------------------------------------

    return cReturnString

*-----------------------------------------------------------------
static function fldAlChr( lNormal, nDim, cAlign )
*
* fldAlChr( <lNormal>, <nDim>, <cAlign> ) --> lSuccess
*
* <@lNormal>        True if the string should not be aligned.
* <@Dim>            the dimention in characters.
* <@cAlign>         the alignment: "L", "C", "R".
*
* This function ask for data alignment and returns a string
* that when interpreted will give a corrected aligned data.
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOld_F1           :=;
        setkey( K_F1, {|| Text( FLD_AL_CHR_HELP ) } )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth
    local lSuccess          := .F.

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom             := maxrow()
        nTop                := nBottom - 8
        nLeft               := 0
        nRight              := 39
        nWidth              := nRight - nLeft +1

        coordinate( @nTop, @nLeft, @nBottom, @nRight, "C", "b" )

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 16, 8 }

        cOldScreen          := mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( FLD_AL_CHR_HEAD, nRight-nLeft-1 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                   // FOREVER

            say( nTop+3, nLeft+1, FLD_AL_PROMPT_NORMAL )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, lNormal := x, lNormal ) },;
                "L",;
                ,;
                { || trueSetkey( K_F2, NIL ) };
            )
            button( @aButtons, row(), col()+3,;
                _BUTTON_CHANGE, , {|| (lNormal := !lNormal), aoGet[1]:display() } )
            say( row()+1, nLeft+1, FLD_AL_CHR_PROMPT_DIMENSION )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, nDim := x, nDim ) },;
                "999",;
                ,;
                { || trueSetkey( K_F2, NIL ) .and. !lNormal };
            )
            say( row()+1, nLeft+1, FLD_AL_CHR_PROMPT_ALIGNMENT )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, cAlign := x, cAlign ) },;
                "!",;
                ,;
                { || trueSetkey( K_F2, NIL ) .and. !lNormal },;
                { || cAlign == "L" .or. cAlign == "C" .or. cAlign == "R" };
            )
            button( @aButtons, row(), col()+3,;
                FLD_AL_BUTTON_LEFT, , {|| (cAlign := "L"), aoGet[3]:display() } )
            button( @aButtons, row(), col()+1,;
                FLD_AL_BUTTON_CENTER, , {|| (cAlign := "C"), aoGet[3]:display() } )
            button( @aButtons, row(), col()+1,;
                FLD_AL_BUTTON_RIGHT, , {|| (cAlign := "R"), aoGet[3]:display() } )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, 1, aButtons )
            aoGet    := {}
            aButtons := {}

            *-----------------------------------------------------
            * Test exit key.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lSuccess := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                *-------------------------------------------------

                lSuccess := .T.

                exit                                    // EXIT

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * Close window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return lSuccess

*-----------------------------------------------------------------
static function fldAlMemo( nDim )
*
* fldAlChr( @<nDim> ) --> lSuccess
*
* @<Dim>            the dimention in characters.
*
* This function ask for data alignment and returns a string
* that when interpreted will give a corrected aligned data.
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOld_F1           :=;
        setkey( K_F1, {|| Text( FLD_AL_CHR_HELP ) } )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth
    local lSuccess          := .F.

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom             := maxrow()
        nTop                := nBottom - 6
        nLeft               := 0
        nRight              := 39
        nWidth              := nRight - nLeft +1

        coordinate( @nTop, @nLeft, @nBottom, @nRight, "C", "b" )

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 16, 8 }

        cOldScreen          := mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( FLD_AL_MEMO_HEAD, nRight-nLeft-1 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                   // FOREVER

            say( nTop+3, nLeft+1, FLD_AL_MEMO_PROMPT_DIMENSION )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, nDim := x, nDim ) },;
                "9999",;
                ,;
                { || trueSetkey( K_F2, NIL ) };
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, 1, aButtons )
            aoGet    := {}
            aButtons := {}

            *-----------------------------------------------------
            * Test exit key.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lSuccess := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                *-------------------------------------------------

                lSuccess := .T.

                exit                                    // EXIT

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * Close window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return lSuccess

*-----------------------------------------------------------------
static function fldAlNum( lNormal, nNumLen, nDecLen )
*
* fldAlNum( <lNormal>, <nDim>, <cAlign> ) --> lSuccess
*
* <@lNormal>        True if the string should not be aligned.
* <@nNumLen>        the total dimention in characters.
* <@nDecLen>        the decimal positions.
*
* This function ask for data alignment and returns a string
* that when interpreted will give a corrected aligned data.
*
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOld_F1           :=;
        setkey( K_F1, {|| Text( FLD_AL_CHR_NUM ) } )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth
    local lSuccess          := .F.

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom             := maxrow()
        nTop                := nBottom - 8
        nLeft               := 0
        nRight              := 39
        nWidth              := nRight - nLeft +1

        coordinate( @nTop, @nLeft, @nBottom, @nRight, "C", "b" )

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 16, 8 }

        cOldScreen          := mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( FLD_AL_NUM_HEAD, nRight-nLeft-1 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                   // FOREVER

            say( nTop+3, nLeft+1, FLD_AL_PROMPT_NORMAL )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, lNormal := x, lNormal ) },;
                "L",;
                ,;
                { || trueSetkey( K_F2, NIL ) };
            )
            button( @aButtons, row(), col()+3,;
                _BUTTON_CHANGE, , {|| (lNormal := !lNormal), aoGet[1]:display() } )
            say( row()+1, nLeft+1, FLD_AL_NUM_PROMPT_LENGTH )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, nNumLen := x, nNumLen ) },;
                "999",;
                ,;
                { || trueSetkey( K_F2, NIL ) .and. !lNormal };
            )
            say( row()+1, nLeft+1, FLD_AL_NUM_PROMPT_DECIMAL )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, nDecLen := x, nDecLen ) },;
                "999",;
                ,;
                { || trueSetkey( K_F2, NIL ) .and. !lNormal };
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, 1, aButtons )
            aoGet    := {}
            aButtons := {}

            *-----------------------------------------------------
            * Test exit key.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lSuccess := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN // ok

                *-------------------------------------------------
                * [PgDn] means confirm.
                *-------------------------------------------------

                lSuccess := .T.

                exit                                    // EXIT

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * Close window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return lSuccess

#ifndef RUNTIME
*=================================================================
* LBL()
*=================================================================
#define LBL_REMARK    1
#define LBL_HEIGHT    2
#define LBL_WIDTH     3
#define LBL_MARGIN    4
#define LBL_LINES     5
#define LBL_SPACES    6
#define LBL_ACROSS    7

#define LBL_SIZE      1034  // dimensione standard dei file di etichetta

#define LBL_WINDOW_LABEL;
    "LABEL"

#define LBL_PROMPT_LABEL_REMARK;
    "Remark:"
#define LBL_PROMPT_LABEL_HEIGHT;
    "Height:"
#define LBL_PROMPT_LABEL_WIDTH;
    "Width: "
#define LBL_PROMPT_LABEL_MARGIN;
    "Margin:"
#define LBL_PROMPT_LABEL_LINES;
    "Lines: "
#define LBL_PROMPT_LABEL_SPACES;
    "Spaces:"
#define LBL_PROMPT_LABEL_ACROSS;
    "Across:"

#define LBL_PROMPT_SAVE_FILE_LABEL;
    "Name to use to save the label:"

#define LBL_HELP_LABEL;
    "lbl()" +;
    NL(3) +;
    "Standard label file editing." +;
    NL(2) +;
    "It creates or modifies a label file, " +;
    "." + _EXTENTION_LABEL + ", under the dBaseIII+ standard." +;
    NL(2) +;
    "Labels may be printed in more than one column and can " +;
    "contain 16 lines maximum." +;
    NL(2) +;
    "The label data is the following." +;
    NL(2) +;
    "REMARK     A label remark that will not be printed." +;
    NL(2) +;
    "HEIGHT     The label vertical dimension." +;
    NL(2) +;
    "WIDTH      The label horizontal dimension." +;
    NL(2) +;
    "MARGIN     The left margin in characters." +;
    NL(2) +;
    "LINES      The vertical spacing between labels." +;
    NL(2) +;
    "SPACES     The horizzontal spacing between labels in characters." +;
    NL(2) +;
    "ACROSS     The number of label columns." +;
    NL(2) +;
    "1          The first line inside the labels." +;
    NL(2) +;
    "..." +;
    NL(2) +;
    "16         The 16th line inside the labels." +;
    NL(2) +;
    "The number of lines inside the lables depend on the " +;
    "HEIGHT and the maximum value is 16."

#define LBL_HELP_LABEL_SAVE;
    "lbl()" +;
    NL(3) +;
    "Label save" +;
    NL(2) +;
    "It permit to specify the name to use to save this label."

#define LBL_ERROR_CANNOT_OPEN;
    "Cannot open this file. See FERROR() error n. "

#define LBL_ERROR_CANNOT_CREATE;
    "Cannot create this file. See FERROR() error n. "

#define LBL_ERROR_FILE_TOO_LITTLE;
    "File too little or empty."

#define LBL_ERROR_FILE_NOT_SAVED;
    "File not saved!"

#define LBL_ERROR_FILE_NOT_VALID;
    "File not valid."

#define LBL_ERROR_SOMETHING_GONE_WRONG;
    "Something is gone wrong during the label save."

#define LBL_WAIT_LOADING;
    "Loading..."

#define LBL_WAIT_SAVING;
    "Saving..."

*=================================================================
function lbl( cFileName )
*
* lbl ( [<cFileName>] ) --> NIL
*
* <cFileName>  the label file name to modify.
*              If the name is not given, it is created.
*

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor         := setcolor()
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local lPreviousHelp
    local bOld_F1           := setkey( K_F1 )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local axLblDim          :=;
        { space(60), 5, 35, 0, 1, 0, 1 }  // default
    local acLblCont[16]
    local nI
    local cOldName
    local lSave             := .F.
    local lGoOn             := .F.

    *-------------------------------------------------------------
    * Check if a previous help exists. If so, no local help
    * will be set up.
    *-------------------------------------------------------------

    if setkey( K_F1 ) == NIL

        lPreviousHelp := .F.

    else

        lPreviousHelp := .T.

    end

    *-------------------------------------------------------------
    * Define local Help for this level.
    *-------------------------------------------------------------

    if !lPreviousHelp

        setkey( K_F1, { || Text( LBL_HELP_LABEL_SAVE ) } )

    end

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * If <cFileName> is not given, the label is created, else
        * it is modifyed.
        *---------------------------------------------------------

        if valtype( cFileName ) <> "C"

            cFileName :=;
                strAddExtention( _UNTITLED, _EXTENTION_LABEL )

            cOldName := NIL

            *-----------------------------------------------------
            * It must be created;
            * the array <acLblCont> is prepared.
            *-----------------------------------------------------

             for nI = 1 to 16

                 acLblCont[nI] := space(60)

             next

         else

            *-----------------------------------------------------
            * The name must be alltrimed.
            *-----------------------------------------------------

            cFileName := alltrim( cFileName )

            if file( cFileName )

                cOldName := alltrim(cFileName)

                *-------------------------------------------------
                * The label file is loaded.
                *-------------------------------------------------

                lblLoad( cFileName, @axLblDim, @acLblCont )

            else

                cOldName := NIL

                *-------------------------------------------------
                * The label file was not found, so it is created:
                * the array <acLblCont> is prepared.
                *-------------------------------------------------

                for nI = 1 to 16

                    acLblCont[nI] := space(60)

                next

            end

        end

        *---------------------------------------------------------
        * The label data is edited.
        *---------------------------------------------------------

        lSave := lblEdit( @axLblDim, @acLblCont, lPreviousHelp )

        *---------------------------------------------------------
        * Label data is saved.
        *---------------------------------------------------------

        if lSave

            cFileName := padr( cFileName, _MAX_STRING_LEN )

            *-----------------------------------------------------
            * Create a kind of window.
            *-----------------------------------------------------

            nBottom             := maxrow()
            nTop                := nBottom - 6
            nLeft               := 0
            nRight              := maxcol()
            nWidth              := nRight - nLeft +1

            aTab[TAB_LEFT]      := nLeft
            aTab[TAB_RIGHT]     := nRight
            aTab[TAB_TAB_ARRAY] := { 16, 4 }
        
            cOldScreen          := mouseScrSave( nTop, nLeft, nBottom, nRight )
        
            setcolor( COLOR_BODY )
        
            scroll( nTop, nLeft, nBottom, nRight )
        
            dispBoxShadow(;
                nTop, nLeft, nBottom, nRight,;
                1,;
                dispBoxColor( 1 ),;
                dispBoxColor( 2 );
            )
        
            say(;
                nTop+1, nLeft+1,;
                padc( LBL_WINDOW_LABEL, nWidth-2 ),;
                ,;
                COLOR_HEAD;
            )

            *-----------------------------------------------------
            * Do editing.
            *-----------------------------------------------------

            while .T.                                   // FOREVER

                say( nTop+3, nLeft+1, LBL_PROMPT_SAVE_FILE_LABEL )
                tab( aTab )
                get(;
                    @aoGet, row(), col(),;
                    { |x| iif( pcount() > 0, cFileName := x, cFileName ) },;
                    picChrMax( col(), nRight-1 ),;
                    ,;
                    { ||; 
                        setkey( K_F2, {|| gvFileDir( @cFileName ) } ),;
                        .T.; 
                    },;
                    { || gvFileExtention( @cFileName, _EXTENTION_LABEL ) },;
                )

                button( @aButtons, row()+2, nLeft+1,;
                    _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
                button( @aButtons, row(), col()+1,;
                    _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
                button( @aButtons, row(), col()+1,;
                    _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
                button( @aButtons, row(), col()+1,;
                    _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

                *-------------------------------------------------
                * Read.
                *-------------------------------------------------

                read( aoGet, 1, aButtons )
                aoGet    := {}
                aButtons := {}

                *-------------------------------------------------
                * Test exit key.
                *-------------------------------------------------

                do case
                case lastkey() = K_ESC

                    *---------------------------------------------
                    * [Esc] means leave.
                    *---------------------------------------------

                    lGoOn := .F.

                    exit                                // EXIT

                case lastkey() = K_PGDN

                    *---------------------------------------------
                    * [PgDn] means confirm.
                    * Before:
                    * check for valid data.
                    *---------------------------------------------

                    do case
                    case;
                        alltrim(cFileName) <> cOldName      .and.;
                        file(;
                            strAddExtention(;
                                cFileName,;
                                _EXTENTION_LABEL;
                            );
                        )                                   .and.;
                        alertBox(;
                            alltrim(cFileName) + NL(1) +;
                                _ERROR_PROMPT_ALREADY_EXIST_OVERWRITE,;
                            { _MENU_NO, _MENU_YES };
                        ) == 1

                        *-----------------------------------------
                        * The file already exists and the user
                        * don't want to overwrite: loop again.
                        *-----------------------------------------

                    case strCutExtention( cFileName ) == ""

                        *-----------------------------------------
                        * No name was given: loop again.
                        *-----------------------------------------

                    otherwise

                        *-----------------------------------------
                        * No other errors are known: go on.
                        *-----------------------------------------

                        lGoOn := .T.

                        exit                            // EXIT

                    end

                otherwise

                    *---------------------------------------------
                    * Exit with different keys is not allowed.
                    * Loop again.
                    *---------------------------------------------

                end

            end

            *-----------------------------------------------------
            * Proceed if so was required.
            *-----------------------------------------------------

            if lGoOn

                *-------------------------------------------------
                * Save.
                *-------------------------------------------------

                if  lblWrite(;
                        alltrim(cFileName),;
                        axLblDim,;
                        acLblCont;
                    )
                    
                    *---------------------------------------------
                    * Ok. Successfull save.
                    *---------------------------------------------

                else

                    alertBox( LBL_ERROR_FILE_NOT_SAVED )

                end

            else

                *-------------------------------------------------
                * Don't save.
                *-------------------------------------------------

            end

            *-----------------------------------------------------
            * Close window.
            *-----------------------------------------------------

            mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

        else

            *-----------------------------------------------------
            * Don't save.
            *-----------------------------------------------------

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return NIL

*-----------------------------------------------------------------
static function lblLoad( cFileName, axLblDim, acLblCont )

    local cBuffer   := space( LBL_SIZE )
    local nHandle   := 0
    local nByteRead := 0
    local nOffset   := 0
    local nI

    *-------------------------------------------------------------
    * Open the label file
    *-------------------------------------------------------------

    nHandle := fopen( cFileName )

    *-------------------------------------------------------------
    * Test for file open error.
    *-------------------------------------------------------------

    if !ferror() == 0

        alertBox(;
            cFileName + NL(1) +;
            LBL_ERROR_CANNOT_OPEN + str(ferror());
        )

        return .F.                                      // RETURN

    end

    *-------------------------------------------------------------
    * Read the entire label file.
    *-------------------------------------------------------------

    nByteRead := fread( nHandle, @cBuffer, LBL_SIZE )

    *-------------------------------------------------------------
    * Close the label file.
    *-------------------------------------------------------------

    fclose( nHandle )

    *-------------------------------------------------------------
    * Check if the length is so as expected.
    *-------------------------------------------------------------

    if nByteRead < LBL_SIZE

        alertBox(;
            cFileName + NL(1) +;
            LBL_ERROR_FILE_TOO_LITTLE;
        )

        return .F.                                      // RETURN

    end

    waitFor( LBL_WAIT_LOADING )

    *-------------------------------------------------------------
    * Extract label dimensions.
    *-------------------------------------------------------------

    axLblDim[LBL_REMARK] := substr( cBuffer, 2, 60 )
    axLblDim[LBL_HEIGHT] := bin2I( substr( cBuffer, 62, 2 ) )
    axLblDim[LBL_WIDTH]  := bin2I( substr( cBuffer, 64, 2 ) )
    axLblDim[LBL_MARGIN] := bin2I( substr( cBuffer, 66, 2 ) )
    axLblDim[LBL_LINES]  := bin2I( substr( cBuffer, 68, 2 ) )
    axLblDim[LBL_SPACES] := bin2I( substr( cBuffer, 70, 2 ) )
    axLblDim[LBL_ACROSS] := bin2I( substr( cBuffer, 72, 2 ) )

    *-------------------------------------------------------------
    * Extract label lines content.
    *-------------------------------------------------------------

    nOffset := 74

    for nI = 1 to 16

        acLblCont[nI] := substr( cBuffer, nOffset, 60 )

        nOffset += 60

    next

    *-------------------------------------------------------------
    * Close the wait window.
    *-------------------------------------------------------------

    waitFor()

    *-------------------------------------------------------------
    * Termiante with success.
    *-------------------------------------------------------------

    return .T.

*-----------------------------------------------------------------
static function lblEdit( axLblDim, acLblCont, lPreviousHelp )

    local aoGet             := {}
    local aButtons          := {}
    local aTab              := TAB_DEFAULT

    local bOldErrorHandler
    local cOldScreen
    local cOldColor         := setcolor( COLOR_BASE )
    local nOldCursor        := setcursor( SETCURSOR_NORMAL )
    local bOld_F1           := setkey( K_F1 )
    local bOld_F2           := setkey( K_F2, NIL )
    local nOldRow           := row()
    local nOldCol           := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lSave             := .F.

    *-------------------------------------------------------------
    * Set up local help
    *-------------------------------------------------------------

    if !lPreviousHelp

        setkey( K_F1, {|| Text( LBL_HELP_LABEL ) } )

    end

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom     := maxrow()
        nTop        := nBottom - 24     // max
        nLeft       := 0
        nRight      := maxcol()
        nWidth      := nRight - nLeft +1

        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 14, 12, 14, 12, 14, 12 }
        
        cOldScreen  := mouseScrSave( nTop, nLeft, nBottom, nRight )
        
        setcolor( COLOR_BODY )
        
        scroll( nTop, nLeft, nBottom, nRight )
        
        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )
        
        say(;
            nTop+1, nLeft+1,;
            padc( LBL_WINDOW_LABEL, nWidth-2 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------

        while .t.                                       // FOREVER

            say( nTop+3, nLeft+1, LBL_PROMPT_LABEL_REMARK )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axLblDim[ LBL_REMARK ] := x, axLblDim[ LBL_REMARK ] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    setkey( K_F2, NIL ),;
                    .T.;
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, LBL_PROMPT_LABEL_HEIGHT )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axLblDim[ LBL_HEIGHT ] := x, axLblDim[ LBL_HEIGHT ] ) },;
                "999",;
                ,;
                { ||;
                    setkey( K_F2, NIL ),;
                    .T.;
                },;
                { ||;
                    axLblDim[ LBL_HEIGHT ] > 0      .and.;
                    axLblDim[ LBL_HEIGHT ] <= 16;
                };
            )
            tab( aTab )
            say( row(), col(), LBL_PROMPT_LABEL_WIDTH )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axLblDim[ LBL_WIDTH ] := x, axLblDim[ LBL_WIDTH ] ) },;
                "999",;
                ,;
                { ||;
                    setkey( K_F2, NIL ),;
                    .T.;
                },;
                { || axLblDim[ LBL_WIDTH ] > 0 };
            )
            tab( aTab )
            say( row(), col(), LBL_PROMPT_LABEL_MARGIN )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axLblDim[ LBL_MARGIN ] := x, axLblDim[ LBL_MARGIN ] ) },;
                "999",;
                ,;
                { ||;
                    setkey( K_F2, NIL ),;
                    .T.;
                },;
                { || axLblDim[ LBL_MARGIN ] >= 0 };
            )

            say( row()+1, nLeft+1, LBL_PROMPT_LABEL_LINES )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axLblDim[ LBL_LINES ] := x, axLblDim[ LBL_LINES ] ) },;
                "999",;
                ,;
                { ||;
                    setkey( K_F2, NIL ),;
                    .T.;
                },;
                { || axLblDim[ LBL_LINES ] >= 0 };
            )
            tab( aTab )
            say( row(), col(), LBL_PROMPT_LABEL_SPACES )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axLblDim[ LBL_SPACES ] := x, axLblDim[ LBL_SPACES ] ) },;
                "999",;
                ,;
                { ||;
                    setkey( K_F2, NIL ),;
                    .T.;
                },;
                { || axLblDim[ LBL_SPACES ] >= 0 };
            )
            tab( aTab )
            say( row(), col(), LBL_PROMPT_LABEL_ACROSS )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axLblDim[ LBL_ACROSS ] := x, axLblDim[ LBL_ACROSS ] ) },;
                "999",;
                ,;
                { ||;
                    setkey( K_F2, NIL ),;
                    .T.;
                },;
                { || axLblDim[ LBL_ACROSS ] > 0 };
            )


            say( row()+1, nLeft+1, "Line  1" )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, acLblCont[1] := x, acLblCont[1] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    axLblDim[LBL_HEIGHT] >= 1       .and.;
                    trueSetkey(;
                        K_F2,;
                        {|| keyboard( fldAlign() ) };
                    );
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, "Line  2" )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, acLblCont[2] := x, acLblCont[2] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    axLblDim[LBL_HEIGHT] >= 2       .and.;
                    trueSetkey(;
                        K_F2,;
                        {|| keyboard( fldAlign() ) };
                    );
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, "Line  3" )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, acLblCont[3] := x, acLblCont[3] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    axLblDim[LBL_HEIGHT] >= 3       .and.;
                    trueSetkey(;
                        K_F2,;
                        {|| keyboard( fldAlign() ) };
                    );
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, "Line  4" )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, acLblCont[4] := x, acLblCont[4] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    axLblDim[LBL_HEIGHT] >= 4       .and.;
                    trueSetkey(;
                        K_F2,;
                        {|| keyboard( fldAlign() ) };
                    );
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, "Line  5" )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, acLblCont[5] := x, acLblCont[5] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    axLblDim[LBL_HEIGHT] >= 5       .and.;
                    trueSetkey(;
                        K_F2,;
                        {|| keyboard( fldAlign() ) };
                    );
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, "Line  6" )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, acLblCont[6] := x, acLblCont[6] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    axLblDim[LBL_HEIGHT] >= 6       .and.;
                    trueSetkey(;
                        K_F2,;
                        {|| keyboard( fldAlign() ) };
                    );
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, "Line  7" )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, acLblCont[7] := x, acLblCont[7] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    axLblDim[LBL_HEIGHT] >= 7       .and.;
                    trueSetkey(;
                        K_F2,;
                        {|| keyboard( fldAlign() ) };
                    );
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, "Line  8" )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, acLblCont[8] := x, acLblCont[8] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    axLblDim[LBL_HEIGHT] >= 8       .and.;
                    trueSetkey(;
                        K_F2,;
                        {|| keyboard( fldAlign() ) };
                    );
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, "Line  9" )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, acLblCont[9] := x, acLblCont[9] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    axLblDim[LBL_HEIGHT] >= 9       .and.;
                    trueSetkey(;
                        K_F2,;
                        {|| keyboard( fldAlign() ) };
                    );
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, "Line 10" )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, acLblCont[10] := x, acLblCont[10] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    axLblDim[LBL_HEIGHT] >= 10       .and.;
                    trueSetkey(;
                        K_F2,;
                        {|| keyboard( fldAlign() ) };
                    );
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, "Line 11" )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, acLblCont[11] := x, acLblCont[11] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    axLblDim[LBL_HEIGHT] >= 11       .and.;
                    trueSetkey(;
                        K_F2,;
                        {|| keyboard( fldAlign() ) };
                    );
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, "Line 12" )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, acLblCont[12] := x, acLblCont[12] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    axLblDim[LBL_HEIGHT] >= 12       .and.;
                    trueSetkey(;
                        K_F2,;
                        {|| keyboard( fldAlign() ) };
                    );
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, "Line 13" )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, acLblCont[13] := x, acLblCont[13] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    axLblDim[LBL_HEIGHT] >= 13       .and.;
                    trueSetkey(;
                        K_F2,;
                        {|| keyboard( fldAlign() ) };
                    );
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, "Line 14" )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, acLblCont[14] := x, acLblCont[14] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    axLblDim[LBL_HEIGHT] >= 14       .and.;
                    trueSetkey(;
                        K_F2,;
                        {|| keyboard( fldAlign() ) };
                    );
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, "Line 15" )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, acLblCont[15] := x, acLblCont[15] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    axLblDim[LBL_HEIGHT] >= 15       .and.;
                    trueSetkey(;
                        K_F2,;
                        {|| keyboard( fldAlign() ) };
                    );
                },;
                { || .T. };
            )

            say( row()+1, nLeft+1, "Line 16" )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, acLblCont[16] := x, acLblCont[16] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    axLblDim[LBL_HEIGHT] >= 16       .and.;
                    trueSetkey(;
                        K_F2,;
                        {|| keyboard( fldAlign() ) };
                    );
                },;
                { || .T. };
            )

            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------

            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lSave := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN

                *-------------------------------------------------
                * [PgDn] means confirm.
                *-------------------------------------------------

                lSave := .T.

                exit                                    // EXIT

            otherwise

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------

            end

        end

        *---------------------------------------------------------
        * Delete window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

    end //sequence

    *-------------------------------------------------------------
    * Restore old data and return.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return lSave

*----------------------------------------------------------------
static function lblWrite( cFileName, axLblDim, acLblCont )

    local cBuffer
    local nHandle   := 0
    local nByteWritten := 0
    local nI

    *-------------------------------------------------------------
    * Must be alltrimed.
    *-------------------------------------------------------------

    cFileName := alltrim( cFileName )

    *-------------------------------------------------------------
    * Create the label file.
    *-------------------------------------------------------------

    nHandle := fcreate( cFileName )

    *-------------------------------------------------------------
    * Check if an error occurred.
    *-------------------------------------------------------------

    if !ferror() == 0

        alertBox(;
            cFileName + NL(1) +;
            LBL_ERROR_CANNOT_CREATE + ;
            str(ferror());
        )

        return .F.                                      // RETURN

    end

    WaitFor( LBL_WAIT_SAVING )

    *-------------------------------------------------------------
    * Buffer prepare.
    *-------------------------------------------------------------

    cBuffer := chr(2)

    cBuffer += axLblDim[LBL_REMARK]
    cBuffer += i2bin(axLblDim[LBL_HEIGHT])
    cBuffer += i2bin(axLblDim[LBL_WIDTH])
    cBuffer += i2bin(axLblDim[LBL_MARGIN])
    cBuffer += i2bin(axLblDim[LBL_LINES])
    cBuffer += i2bin(axLblDim[LBL_SPACES])
    cBuffer += i2bin(axLblDim[LBL_ACROSS])

    for nI = 1 to 16

        cBuffer += acLblCont[nI]

    next

    cBuffer += chr(2)

    *-------------------------------------------------------------
    * Write buffer into the label file.
    *-------------------------------------------------------------

    nByteWritten := fwrite( nHandle, cBuffer )

    *-------------------------------------------------------------
    * Check if write was successfull.
    *-------------------------------------------------------------

    if nByteWritten != LBL_SIZE

        alertBox( LBL_ERROR_SOMETHING_GONE_WRONG )

    end

    *-------------------------------------------------------------
    * Close the label file.
    *-------------------------------------------------------------

    fclose( nHandle )

    WaitFor()

    *-------------------------------------------------------------
    * Terminate with success.
    *-------------------------------------------------------------

    return .T.

#endif //RUNTIME

#ifndef RUNTIME
*=================================================================
* FRM()
*=================================================================

* xcFormVal array
#define FRM_PAGE_HDR                 1
#define FRM_GRP_EXPR                 2
#define FRM_SUB_EXPR                 3
#define FRM_GRP_HDR                  4
#define FRM_SUB_HDR                  5
#define FRM_PAGE_WIDTH               6
#define FRM_LINES_PAGE               7
#define FRM_LEFT_MARG                8
#define FRM_RIGHT_MARG               9
#define FRM_COL_COUNT               10
#define FRM_DBL_SPACED              11
#define FRM_SUMMARY                 12
#define FRM_PE                      13
#define FRM_PEBP                    14
#define FRM_PEAP                    15
#define FRM_PLAINPAGE               16

* buffer dimensions
#define FRM_SIZE_FILE_BUFF        1990   // file length
#define FRM_SIZE_LENGTHS_BUFF      110
#define FRM_SIZE_OFFSETS_BUFF      110
#define FRM_SIZE_EXPR_BUFF        1440
#define FRM_SIZE_FIELDS_BUFF       300
#define FRM_SIZE_PARAMS_BUFF        24

* buffer offsets (start points)
* contined into cBuffer, that is the file (.frm) loaded
* into a variable and seen from the point fo view
* of CA-Clipper and not from the one of Dos.
#define FRM_LENGTHS_OFFSET           5
#define FRM_OFFSETS_OFFSET         115
#define FRM_EXPR_OFFSET            225
#define FRM_FIELDS_OFFSET         1665
#define FRM_PARAMS_OFFSET         1965

* Offsets contained in acFormBuffer[ FRM_FIELDS_BUFF ]
#define FRM_FIELD_OFFSET               0
#define FRM_FIELD_WIDTH_OFFSET         1
#define FRM_FIELD_TOTALS_OFFSET        6
#define FRM_FIELD_DECIMALS_OFFSET      7
#define FRM_FIELD_CONTENT_EXPR_OFFSET  9
#define FRM_FIELD_HEADER_EXPR_OFFSET  11

* Offsets contained in cParamsBuff
#define FRM_PAGE_HDR_OFFSET         1
#define FRM_GRP_EXPR_OFFSET         3
#define FRM_SUB_EXPR_OFFSET         5
#define FRM_GRP_HDR_OFFSET          7
#define FRM_SUB_HDR_OFFSET          9
#define FRM_PAGE_WIDTH_OFFSET      11
#define FRM_LNS_PER_PAGE_OFFSET    13
#define FRM_LEFT_MRGN_OFFSET       15
#define FRM_RIGHT_MRGN_OFFSET      17
#define FRM_COL_COUNT_OFFSET       19
#define FRM_DBL_SPACE_OFFSET       21
#define FRM_SUMMARY_RPT_OFFSET     22
#define FRM_PE_OFFSET              23
#define FRM_PLNPG_PEAP_PEBP_OFFSET 24

* FormBuffers
#define FRM_LENGTHS_BUFF            1
#define FRM_OFFSETS_BUFF            2
#define FRM_EXPR_BUFF               3
#define FRM_PARAMS_BUFF             4
#define FRM_FIELDS_BUFF             5

* anNum[5]
#define FRM_PAGE_HDR_NUM            1
#define FRM_GRP_EXPR_NUM            2
#define FRM_SUB_EXPR_NUM            3
#define FRM_GRP_HDR_NUM             4
#define FRM_SUB_HDR_NUM             5

#define FRM_HEAD_REPORT_ORDER;
    "Order"

#define FRM_HEAD_REPORT_HEADER;
    "Column Header"

#define FRM_HEAD_REPORT_CONTENT;
    "Content"

#define FRM_HEAD_REPORT_WIDTH;
    "Width"

#define FRM_HEAD_REPORT_DECIMALS;
    "dec."

#define FRM_HEAD_REPORT_TOTAL;
    "Totals"

#define FRM_DIALOG_BOX_TOP_REPORT;
    "REPORT FORM ..."

#define FRM_MESSAGELINE_F2;
    iif(;
        alias() == "",;
        NIL,;
        messageline(;
            "Press [F2] for a list of fields",, row()+1;
        );
    )

#define FRM_PROMPT_COMPRESSED_PRINT;
    "Compressed Print ? (Y/T/N/F) "

#define FRM_PROMPT_OPEN_FILE_REPORT;
    "Report file name:"

#define FRM_PROMPT_WHILE_EXPRESSION;
    "WHILE expression condition:"

#define FRM_PROMPT_FOR_EXPRESSION;
    "FOR expression condition:"

#define FRM_PROMPT_TO_FILE;
    "Periferal or destination file:"

#define FRM_PROMPT_NEW_FILE_NAME;
    "Insert new name."

#define FRM_PROMPT_SAVE_REPORT;
    "Name to use to save the report:"

#define FRM_PROMPT_REPORT_PAGE_WIDTH;
    "Page Width"

#define FRM_PROMPT_REPORT_LINES_PER_PAGE;
    "Lines per Page"

#define FRM_PROMPT_REPORT_LEFT_MARGIN;
    "Left Margin"

#define FRM_PROMPT_REPORT_RIGHT_MARGIN;
    "Right Margin"

#define FRM_PROMPT_REPORT_DOUBLE_SPACED;
    "Double Spaced?"

#define FRM_PROMPT_REPORT_PAGE_EJECT_BEFORE;
    "Eject Before Print?"

#define FRM_PROMPT_REPORT_PAGE_EJECT_AFTER;
    "Eject After Print?"

#define FRM_PROMPT_REPORT_PLAIN_PAGE;
    "Plain Page?"

#define FRM_PROMPT_REPORT_PAGE_HEADER;
    "Page Header"

#define FRM_PROMPT_REPORT_GROUP_HEADER;
    "Group Header"

#define FRM_PROMPT_REPORT_GROUP_EXPRESSION;
    "Group Expression"

#define FRM_PROMPT_REPORT_SUMMARY_REPORT_ONLY;
    "Summary Report Only?"

#define FRM_PROMPT_REPORT_PAGE_EJECT_AFTER_GROUP;
    "Page Eject After Group?"

#define FRM_PROMPT_REPORT_SUB_GROUP_HEADER;
    "Sub Group Header"

#define FRM_PROMPT_REPORT_SUB_GROUP_EXPRESSION;
    "Sub Group Expression"

#define FRM_HELP_REPORT_EDIT_HEAD;
    "frm()" +;
    NL(3) +;
    "Standard report form editing." +;
    NL(2) +;
    "It creates or modifies a report file, " +;
    "." + _EXTENTION_FORM + " under the dBaseIII+ standard." +;
    NL(2) +;
    "The informations are divided into two parts: " +;
    NL(1) +;
    "* the head and groups," +;
     NL(1) +;
    "* the columns." +;
    NL(2) +;
    "PAGE WIDTH                 the page width in characters;" +;
    NL(2) +;
    "LINES PER PAGE             the usable lines per per page;" +;
    NL(2) +;
    "LEFT MARGIN                the left margin in characters;" +;
    NL(2) +;
    "DOUBLE SPACED?             double spaced print, yes or no;" +;
    NL(2)+;
    "PAGE EJECT BEFORE PRINT?   form feed before print, " +;
    "yes or no;" +;
    NL(2) +;
    "PAGE EJECT AFTER PRINT?    form feed after print, " +;
    "yes or no;" +;
    NL(2) +;
    "PLAIN PAGE?                plain page, yes or no;" +;
    NL(2) +;
    "PAGE HEADER                the page header, max 4 lines " +;
    "(the separation between one line and the other is " +;
    "obtained writing a semicolon, ';');" +;
    NL(2)+;
    "GROUP HEADER               the group title;" +;
    NL(2) +;
    "GROUP EXPRESSION           the group expression (when it " +;
    "changes, the group changes);" +;
    NL(2) +;
    "SUMMARY REPORT ONLY?       only totals and no columns, " +;
    "yes or no;" +;
    NL(2)+;
    "PAGE EJECT AFTER GROUP?    form feed when the group " +;
    "changes, yes or no;" +;
    NL(2)+;
    "SUB GROUP HEADER           sub group title;" +;
    NL(2) +;
    "SUB GROUP EXPRESSION       the sub group expression."

#define FRM_HELP_REPORT_EDIT_COLUMNS;
    "frm()" +;
    NL(3) +;
    "Standard report form editing." +;
    NL(2) +;
    "It creates or modifies a report file, " +;
    "." + _EXTENTION_FORM + " under the dBaseIII+ standard." +;
    NL(2) +;
    "The informations are divided into two parts: " +;
    NL(1) +;
    "* the head and groups," +;
     NL(1) +;
    "* the columns." +;
    NL(2) +;
    "COLUMN HEADER  column head description " +;
    "(it can contain 4 lines separated with a semicolon);" +;
    NL(2)+;
    "CONTENT        the column expression;" +;
    NL(2) +;
    "WIDTH          the column witdth;" +;
    NL(2) +;
    "DEC.           the decimal lengh for numeric columns;" +;
    NL(2)+;
    "TOTALS         totals to be calculated, yes or no " +;
    "(usefull only for numeric columns)." +;
    NL(3) +;
    "The editing of the data is made using the following keys:" +;
    NL(2) +;
    "[Pag]             previous page;" +; 
    NL(1) +;
    "[Pag]             next page;" +; 
    NL(1) +;
    "[Ctrl]+[Pag]      top of table;" +; 
    NL(1) +;
    "[Ctrl]+[Pag]      bottom of table;" +; 
    NL(1) +;
    "[Ctrl]+[Home]      first column;" +; 
    NL(1) +;
    "[Ctrl]+[End]       last column;" +; 
    NL(1) +;
    "[Ctrl]+[Enter]     append;" +; 
    NL(1) +;
    "[CTRL]+[F1]        cut;" +;
    NL(1) +;
    "[CTRL]+[F2]        copy;" +; 
    NL(1) +;
    "[CTRL]+[F3]        paste;" +; 
    NL(1) +;
    "[Ctrl]+[Del]       delete;" +;
    NL(1) +;
    "[Ctrl]+[Y]         delete."

#define FRM_HELP_REPORT_SAVE;
    "frm()" +;
    NL(3) +;
    "Report Form save." +;
    NL(2) +;
    "It specify the name to use to save this report."

#define FRM_ERROR_CANNOT_OPEN;
    "Cannot open this file. See FERROR() error n. "

#define FRM_ERROR_CANNOT_CREATE;
    "Cannot create this file. See FERROR() error n. "

#define FRM_ERROR_FILE_TOO_LITTLE;
    "File too little or empty."

#define FRM_ERROR_FILE_NOT_SAVED;
    "File not saved!"

#define FRM_ERROR_FILE_NOT_VALID;
    "File not valid."

#define FRM_ERROR_SOMETHING_GONE_WRONG;
    "Something is gone wrong trying to save."

#define FRM_ERROR_PRINT_FILE_NOT_FOUND;
    "The file do not exists."

#define FRM_WAIT_LOADING;
    "Loading..."

#define FRM_WAIT_SAVING;
    "Saving..."

*=================================================================
function frm( cFileName )
*
* frm( <cFileName> ) --> NIL
*
* <cFileName>  the report file name to modify.
*              If not present, it will be created.
*
* Offsets:
*    CA-Clipper array and character variables pointers starts
*    from position 1. This means that the first element is
*    number one and not number 0.
*    Addresses writtend inside the Report Form file starts
*    from position 0 (zero).
*    All "DOS offsets" starts here from 0.
*
* Description from the original "RL" program.
*    Report file length is 7C6h (1990d) bytes.
*    Expression length array starts at 04h (4d) and can
*       contain upto 55 short (2 byte) numbers.
*    Expression offset index array starts at 72h (114d) and
*       can contain upto 55 short (2 byte) numbers.
*    Expression area starts at offset E0h (224d).
*    Expression area length is 5A0h (1440d).
*    Expressions in expression area are null terminated.
*    Field expression area starts at offset 680h (1664d).
*    Field expressions (column definition) are null terminated.
*    Field expression area can contain upto 25 12-byte blocks.
*

    local aoGet                 := {}
    local aButtons              := {}
    local aTab                  := TAB_DEFAULT
    
    local bOldErrorHandler
    local cOldScreen
    local cOldColor             := setcolor()
    local nOldCursor            := setcursor( SETCURSOR_NORMAL )
    local lPreviousHelp
    local bOld_F1               := setkey( K_F1 )
    local bOld_F2               := setkey( K_F2, NIL )
    local nOldRow               := row()
    local nOldCol               := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth
    local lOldDeleted           := set( _SET_DELETED, .T. )

    local axFormVal[16]
    local aStruct               := {}

    local cOldName
    local lSave
    local lGoOn

    *-------------------------------------------------------------
    * Check if a previous help exists. If so, no local help
    * will be set up.
    *-------------------------------------------------------------

    if setkey( K_F1 ) == NIL

        lPreviousHelp := .F.

    else

        lPreviousHelp := .T.

    end

    *-------------------------------------------------------------
    * Define local Help for this level.
    *-------------------------------------------------------------

    if !lPreviousHelp

        setkey( K_F1, { || Text( FRM_HELP_REPORT_SAVE ) } )

    end

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------
    
    begin sequence

        *---------------------------------------------------------
        * Header array initialization.
        *---------------------------------------------------------
        
        axFormVal[FRM_PAGE_WIDTH]       := 80
        axFormVal[FRM_LINES_PAGE]       := 58
        axFormVal[FRM_LEFT_MARG]        := 8
        axFormVal[FRM_RIGHT_MARG]       := 0
        axFormVal[FRM_COL_COUNT]        := 0
        axFormVal[FRM_DBL_SPACED]       := "N"
        axFormVal[FRM_SUMMARY]          := "N"
        axFormVal[FRM_PE]               := "N"
        axFormVal[FRM_PLAINPAGE]        := "N"
        axFormVal[FRM_PEAP]             := "Y"
        axFormVal[FRM_PEBP]             := "N"
        axFormVal[FRM_PAGE_HDR]         := space(240)
        axFormVal[FRM_GRP_EXPR]         := space(200)
        axFormVal[FRM_SUB_EXPR]         := space(200)
        axFormVal[FRM_GRP_HDR]          := space(50)
        axFormVal[FRM_SUB_HDR]          := space(50)

        *---------------------------------------------------------
        * If <cFileName> contains an existing form name, this
        * form is modified, else a new one is created.
        *---------------------------------------------------------

        if valtype( cFileName ) <> "C"

            *-----------------------------------------------------
            * A default file name is defined.
            *-----------------------------------------------------
            
            cFileName :=;
                strAddExtention( _UNTITLED, _EXTENTION_FORM )
            
            *-----------------------------------------------------
            * There is no previous name. <cOldName> is used
            * when saving to determinate if a file overwrite is
            * possibly dangerous.
            *-----------------------------------------------------
            
            cOldName := NIL

            *-----------------------------------------------------
            * An empty sample of the column definition is created.
            *-----------------------------------------------------
            
            aadd( aStruct, { space(260), space(254), 0, 0, space(1) } )
            
            *-----------------------------------------------------
            * The form will be created.
            *-----------------------------------------------------

        else

            *-----------------------------------------------------
            * The file name must be alltrimed.
            *-----------------------------------------------------
            
            cFileName := alltrim( cFileName )

            *-----------------------------------------------------
            * The file must exist.
            *-----------------------------------------------------
            
            if file( cFileName )

                *-------------------------------------------------
                * The actual name is saved.
                *-------------------------------------------------
                
                cOldName := alltrim(cFileName)

                *-------------------------------------------------
                * The form file is loaded.
                *-------------------------------------------------
                
                frmLoad( cFileName, @axFormVal, @aStruct )

            else

                *-------------------------------------------------
                * There is no previous name. <cOldName> is used
                * when saving to determinate if a file overwrite 
                * is possibly dangerous.
                *-------------------------------------------------
            
                cOldName := NIL

                *-------------------------------------------------
                * An empty sample of the column definition is 
                * created.
                *-------------------------------------------------
            
                aadd(; 
                    aStruct,; 
                    { space(260), space(254), 0, 0, space(1) }; 
                )
            
                *-------------------------------------------------
                * The form will be created.
                *-------------------------------------------------
            
            end
        
        end

        *---------------------------------------------------------
        * Edit the form.
        *---------------------------------------------------------
        
        lSave := frmEdit( cFileName, @axFormVal, @aStruct, lPreviousHelp )

        *---------------------------------------------------------
        * Save it if so was selected.
        *---------------------------------------------------------
        
        if lSave

            cFileName := padr( cFileName, _MAX_STRING_LEN )

            *-----------------------------------------------------
            * Create a kind of window.
            *-----------------------------------------------------

            nBottom             := maxrow()
            nTop                := nBottom - 6
            nLeft               := 0
            nRight              := maxcol()
            nWidth              := nRight - nLeft +1
            
            aTab[TAB_LEFT]      := nLeft
            aTab[TAB_RIGHT]     := nRight
            aTab[TAB_TAB_ARRAY] := { 16, 4 }

            cOldScreen          :=; 
                mouseScrSave( nTop, nLeft, nBottom, nRight )
            
            setcolor( COLOR_BODY )
        
            scroll( nTop, nLeft, nBottom, nRight )
            
            dispBoxShadow(;
                nTop, nLeft, nBottom, nRight,;
                1,;
                dispBoxColor( 1 ),;
                dispBoxColor( 2 );
            )

            say(;
                nTop+1, nLeft+1,;
                padc(;
                    FRM_DIALOG_BOX_TOP_REPORT,;
                    nWidth-2;
                    ),;
                ,;
                COLOR_HEAD;
            )

            *-----------------------------------------------------
            * Do editing.
            *-----------------------------------------------------
            
            while .t.                                   // FOREVER

                say( nTop+3, nLeft+1, FRM_PROMPT_SAVE_REPORT )
                tab( aTab )
                get(;
                    @aoGet, row(), col(),;
                    { |x| iif( pcount() > 0, cFileName := x, cFileName ) },;
                    picChrMax( col(), nRight-1 ),;
                    ,;
                    { ||; 
                        setkey( K_F2, {|| gvFileDir( @cFileName ) } ),; 
                        .T.;
                    },;
                    { ||; 
                        gvFileExtention( @cFileName, _EXTENTION_FORM ); 
                    };
                )
                
                button( @aButtons, row()+2, nLeft+1,;
                    _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
                button( @aButtons, row(), col()+1,;
                    _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
                button( @aButtons, row(), col()+1,;
                    _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
                button( @aButtons, row(), col()+1,;
                    _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )

                *-------------------------------------------------
                * Read.
                *-------------------------------------------------

                read( aoGet, , aButtons )
                aoGet   := {}
                aButtons := {}

                *-------------------------------------------------
                * Check what was read.
                *-------------------------------------------------

                do case
                case lastkey() = K_ESC  // exit

                    *---------------------------------------------
                    * [Esc] means leave.
                    *---------------------------------------------

                    lGoOn := .F.
                    
                    exit                                // EXIT
                
                case lastkey() = K_PGDN // ok

                    *---------------------------------------------
                    * [PgDn] means confirm.
                    * Before, check for valid data.
                    *---------------------------------------------

                    do case
                    case; 
                        alltrim(cFileName) <> cOldName  .and.;
                        file(; 
                            strAddExtention(; 
                                cFileName,;
                                _EXTENTION_FORM; 
                            ); 
                        )                               .and.;
                        alertBox(; 
                            alltrim(cFileName) + NL(1) +;
                                _ERROR_PROMPT_ALREADY_EXIST_OVERWRITE,;
                            { _MENU_NO, _MENU_YES }; 
                        ) == 1
                        
                        *-----------------------------------------
                        * The user don't want to overwrite, so
                        * loop again.
                        *-----------------------------------------

                    case strCutExtention( cFileName ) == ""

                        *-----------------------------------------
                        * No file name was given, so loop again.
                        *-----------------------------------------

                    otherwise

                        *-----------------------------------------
                        * No other error is considered: exit
                        * the editing loop.
                        *-----------------------------------------
                        
                        lGoOn := .T.
                        
                        exit                            // EXIT
                    
                    end

                otherwise

                    *---------------------------------------------
                    * Exit with different keys is not allowed.
                    * Loop again.
                    *---------------------------------------------
                
                end

            end

            *-----------------------------------------------------
            * Delete window.
            *-----------------------------------------------------

            mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

            *-----------------------------------------------------
            * If save was confirmed, save the form file.
            *-----------------------------------------------------
            
            if lGoOn

                *-------------------------------------------------
                * Ok save.
                *-------------------------------------------------
                
                if  !frmSave(; 
                        alltrim( cFileName ),; 
                        axFormVal,; 
                        @aStruct; 
                    )

                    alertBox( FRM_ERROR_FILE_NOT_SAVED )

                end

            end

        end

    end //sequence

    *-------------------------------------------------------------
    * Restore old data.
    *-------------------------------------------------------------
    
    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )
    set( _SET_DELETED, lOldDeleted )

    return NIL

*-----------------------------------------------------------------
static function frmLoad( cFileName, axFormVal, aStruct )

    local cBuffer             := space( FRM_SIZE_FILE_BUFF )
    local acFormBuffer[5]
    local nPlusByte           := 0
    local nPointer            := 0
    local nHandle             := fopen( cFileName )
    local nByteRead           := 0
    local nFieldOffset        := 0
    local nI                  := 0
    local lFileError          := .F.

    *-------------------------------------------------------------
    * Check if the form file open was succesfully executed.
    *-------------------------------------------------------------
    
    if !ferror() == 0

        alertBox(; 
            cFileName + NL(1) +;
            FRM_ERROR_CANNOT_OPEN +; 
            str( ferror() ); 
        )

        *---------------------------------------------------------
        * No load is executed.
        *---------------------------------------------------------
        
        return .F.                                      // RETURN
    
    end

    *-------------------------------------------------------------
    * Read the entire file.
    *-------------------------------------------------------------
    
    nByteRead := fread( nHandle, @cBuffer, FRM_SIZE_FILE_BUFF )

    *-------------------------------------------------------------
    * Check if was read all.
    *-------------------------------------------------------------
    
    if  nByteRead < FRM_SIZE_FILE_BUFF

        alertBox(; 
            cFileName + NL(1) +;
            FRM_ERROR_FILE_TOO_LITTLE; 
        )

        *---------------------------------------------------------
        * The data is not valid.
        *---------------------------------------------------------
        
        return .F.                                      // RETURN

    elseif; 
        !(; 
            bin2i(;
                substr( cBuffer, 1, 2 );
            ) == 2                                      .and.;
            bin2i(;
                substr( cBuffer, FRM_SIZE_FILE_BUFF - 1, 2 );
            ) == 2; 
        )

        *---------------------------------------------------------
        * If it don't starts and ends with the number 2
        * it is not a report form.
        *---------------------------------------------------------
        
        alertBox(; 
            cFileName + NL(1) +;
            FRM_ERROR_FILE_NOT_VALID; 
        )

        return .F.                                      // RETURN

    end

    *-------------------------------------------------------------
    * Show a wait box.
    *-------------------------------------------------------------
    
    waitFor( FRM_WAIT_LOADING )

    *-------------------------------------------------------------
    * Data extractions.
    *-------------------------------------------------------------
    
    acFormBuffer[ FRM_LENGTHS_BUFF ] := ;
        substr(cBuffer, FRM_LENGTHS_OFFSET, FRM_SIZE_LENGTHS_BUFF)

    acFormBuffer[ FRM_OFFSETS_BUFF ] := ;
        substr(cBuffer, FRM_OFFSETS_OFFSET, FRM_SIZE_OFFSETS_BUFF)

    acFormBuffer[ FRM_EXPR_BUFF ] := ;
        substr( cBuffer, FRM_EXPR_OFFSET, FRM_SIZE_EXPR_BUFF )

    acFormBuffer[ FRM_FIELDS_BUFF ] := ;
        substr( cBuffer, FRM_FIELDS_OFFSET, FRM_SIZE_FIELDS_BUFF )

    acFormBuffer[ FRM_PARAMS_BUFF ] := ;
        substr( cBuffer, FRM_PARAMS_OFFSET, FRM_SIZE_PARAMS_BUFF )

    axFormVal[FRM_PAGE_WIDTH] := ;
        bin2i(; 
            substr(; 
                acFormBuffer[ FRM_PARAMS_BUFF ],;
                FRM_PAGE_WIDTH_OFFSET,; 
                2; 
            ); 
        )

    axFormVal[FRM_LINES_PAGE] := ;
        bin2i(; 
            substr(; 
                acFormBuffer[ FRM_PARAMS_BUFF ],;
                FRM_LNS_PER_PAGE_OFFSET,; 
                2; 
            ); 
        )

    axFormVal[FRM_LEFT_MARG] := ;
        bin2i(; 
            substr(; 
                acFormBuffer[ FRM_PARAMS_BUFF ],;
                FRM_LEFT_MRGN_OFFSET,; 
                2; 
            ); 
        )

    axFormVal[FRM_RIGHT_MARG] := ;
        bin2i(; 
            substr(; 
                acFormBuffer[ FRM_PARAMS_BUFF ],;
                FRM_RIGHT_MRGN_OFFSET,; 
                2; 
            ); 
        )

    axFormVal[FRM_COL_COUNT] := ;
        bin2i(; 
            substr(; 
                acFormBuffer[ FRM_PARAMS_BUFF ],;
                FRM_COL_COUNT_OFFSET,; 
                2; 
            ); 
        )

    axFormVal[FRM_DBL_SPACED] := ;
        substr(; 
            acFormBuffer[ FRM_PARAMS_BUFF ],;
            FRM_DBL_SPACE_OFFSET,; 
            1; 
        )

    axFormVal[FRM_SUMMARY] := ;
        substr(; 
            acFormBuffer[ FRM_PARAMS_BUFF ],;
            FRM_SUMMARY_RPT_OFFSET,; 
            1; 
        )

    axFormVal[FRM_PE] := ;
        substr(; 
            acFormBuffer[ FRM_PARAMS_BUFF ],; 
            FRM_PE_OFFSET,; 
            1; 
        )

    nPlusByte := ;
        asc(; 
            substr(; 
                acFormBuffer[ FRM_PARAMS_BUFF ],;
                FRM_PLNPG_PEAP_PEBP_OFFSET,; 
                1;
            ); 
        )

    if  int( nPlusByte/4 ) == 1

        axFormVal[FRM_PLAINPAGE] := "Y"

        nPlusByte -= 4

    end

    if int( nPlusByte/2 ) == 1

        axFormVal[FRM_PEAP] := "Y"

        nPlusByte -= 2

    end

    if int( nPlusByte/1 ) == 1

        axFormVal[FRM_PEBP] := "N"

        nPlusByte -= 1

    end

    *-------------------------------------------------------------
    * String expressions extraction.
    *-------------------------------------------------------------

    *-------------------------------------------------------------
    * Page Header.
    *-------------------------------------------------------------
    
    nPointer := ;
        bin2i(; 
            substr(; 
                acFormBuffer[ FRM_PARAMS_BUFF ],;
                FRM_PAGE_HDR_OFFSET,; 
                2; 
            ); 
        )

    axFormVal[FRM_PAGE_HDR] := ;
        padr( frmGetExpr( @acFormBuffer, nPointer ), 240 )

    *-------------------------------------------------------------
    * Grouping Expression.
    *-------------------------------------------------------------
    
    nPointer := ;
        bin2i(; 
            substr(; 
                acFormBuffer[ FRM_PARAMS_BUFF ],;
                FRM_GRP_EXPR_OFFSET,; 
                2; 
            ); 
        )

    axFormVal[FRM_GRP_EXPR] := ;
        padr( frmGetExpr( @acFormBuffer, nPointer ), 200 )

    *-------------------------------------------------------------
    * Sub-grouping Expression.
    *-------------------------------------------------------------
    
    nPointer := ;
        bin2i(; 
            substr(; 
                acFormBuffer[ FRM_PARAMS_BUFF ],;
                FRM_SUB_EXPR_OFFSET,; 
                2; 
            ); 
        )

    axFormVal[FRM_SUB_EXPR] := ;
        padr( frmGetExpr( @acFormBuffer, nPointer ), 200 )

    *-------------------------------------------------------------
    * Group Header.
    *-------------------------------------------------------------
    
    nPointer := ;
        bin2i(; 
            substr(; 
                acFormBuffer[ FRM_PARAMS_BUFF ],;
                FRM_GRP_HDR_OFFSET,; 
                2; 
            ); 
        )

    axFormVal[FRM_GRP_HDR] := ;
        padr( frmGetExpr( @acFormBuffer, nPointer ), 50 )

    *-------------------------------------------------------------
    * Sub-group header.
    *-------------------------------------------------------------
    
    nPointer := ;
        bin2i(; 
            substr(; 
                acFormBuffer[ FRM_PARAMS_BUFF ],;
                FRM_SUB_HDR_OFFSET,; 
                2; 
            ); 
        )

    axFormVal[FRM_SUB_HDR] := ;
        padr( frmGetExpr( @acFormBuffer, nPointer ), 50 )

    *-------------------------------------------------------------
    * Fields extraction and transfer inside an array.
    *-------------------------------------------------------------

    nFieldOffset := 12

    for nI = 1 to axFormVal[FRM_COL_COUNT]

        nFieldOffset := ;
            frmGetField(; 
                @acFormBuffer,; 
                nFieldOffset,; 
                @lFileError,; 
                @aStruct; 
            )
    
    next

    *-------------------------------------------------------------
    * Close the wait box.
    *-------------------------------------------------------------
    
    waitFor()

    *-------------------------------------------------------------
    * Return True if all was successfully done.
    *-------------------------------------------------------------
    
    return !lFileError

*-----------------------------------------------------------------
static function frmGetExpr( acFormBuffer, nPointer )
*
* The expression is empty if:
*    - passed pointer is equal to 65535;
*    - character following character pointed to by
*      pointer is CHR(0) (NULL).
*

    local nExprOffset         := 0
    local nExprLength         := 0
    local nOffsetOffset       := 0
    local cString             := ""

    *-------------------------------------------------------------
    * If <nPointer> contains 65535 it is to be considered as a
    * NIL: pointer to nothing, so there is no expression.
    * In this case the function will return an empty string.
    *-------------------------------------------------------------
    
    if nPointer <> 65535

        *---------------------------------------------------------
        * Convert DOS FILE offset to CLIPPER string offset.
        *---------------------------------------------------------
        
        nPointer += 1

        *---------------------------------------------------------
        * Calculate offset into OFFSETS_BUFF.
        *---------------------------------------------------------
        
        if nPointer > 1

           nOffsetOffset := ( nPointer * 2 ) - 1
        
        end

        nExprOffset := ;
            bin2i(; 
                substr(; 
                    acFormBuffer[ FRM_OFFSETS_BUFF ],;
                    nOffsetOffset,; 
                    2;
                ); 
            )

        nExprLength := ;
            bin2i(; 
                substr(; 
                    acFormBuffer[ FRM_LENGTHS_BUFF ],;
                    nOffsetOffset,; 
                    2;
                ); 
            )

        *---------------------------------------------------------
        * EXPR_OFFSET points to a NULL, so add one (+1)
        * to get the string and subtract one (-1) from
        * EXPR_LENGTH for correct length.
        *---------------------------------------------------------
        
        nExprOffset += 1

        nExprLength -= 1

        *---------------------------------------------------------
        * Extract string.
        *---------------------------------------------------------
        
        cString :=; 
            substr(; 
                acFormBuff[ FRM_EXPR_BUFF ],;
                nExprOffset,; 
                nExprLength; 
            )

        *---------------------------------------------------------
        * dBASE does this so we must do it too.
        * Character following character pointed to by pointer
        * is NULL.
        *---------------------------------------------------------
        
        if  chr(0) == substr( cString, 1, 1)            .and.;
            len( substr( cString, 1, 1 ) ) == 1

            cString = ""

        end

    end

    *-------------------------------------------------------------
    * The extracted string is returned.
    *-------------------------------------------------------------
    
    return cString

*-----------------------------------------------------------------
static function frmGetField(; 
    acFormBuffer, nFieldOffset, lFileError, aStruct; 
    )
*
* The Header or Contents expressions are empty if:
*    - passed pointer is equal to 65535;
*    - character following character pointed to by
*      pointer is CHR(0) (NULL).
*

    local nPointer  := 0
    local nNumber   := 0

    local nWidth
    local cTotals
    local nDecimals
    local cContent
    local cHeader

    *-------------------------------------------------------------
    * Column width.
    *-------------------------------------------------------------
    
    nWidth :=;
        bin2i(; 
            substr(; 
                acFormBuffer[ FRM_FIELDS_BUFF ],;
                nFieldOffset + FRM_FIELD_WIDTH_OFFSET,; 
                2;
            ); 
        )

    *-------------------------------------------------------------
    * Totals.
    *-------------------------------------------------------------
    
    cTotals :=;
        substr(; 
            acFormBuffer[ FRM_FIELDS_BUFF ],;
            nFieldOffset + FRM_FIELD_TOTALS_OFFSET,; 
            1;
        )

    if  empty( cTotals ) 
    
        *---------------------------------------------------------
        * This is not necessary, but I think it will be more
        * clear when editing the column informations.
        *---------------------------------------------------------
        
        cTotals := "N"

    end
    
    *-------------------------------------------------------------
    * Decimals width.
    *-------------------------------------------------------------
    
    nDecimals :=;
       bin2i(; 
           substr(; 
               acFormBuffer[ FRM_FIELDS_BUFF ],;
               nFieldOffset + FRM_FIELD_DECIMALS_OFFSET,; 
               2; 
           ); 
       )

    *-------------------------------------------------------------
    * Content expression.
    *-------------------------------------------------------------
    
    nPointer :=;
        bin2i(; 
            substr(; 
                acFormBuffer[ FRM_FIELDS_BUFF ],;
                nFieldOffset + FRM_FIELD_CONTENT_EXPR_OFFSET,; 
                2; 
            ); 
        )

    cContent := ;
        padr( frmGetExpr( @acFormBuffer, nPointer ), 254 )
    
    *-------------------------------------------------------------
    * Header expression.
    *-------------------------------------------------------------
    
    nPointer :=;
        bin2i(; 
            substr(; 
                acFormBuffer[ FRM_FIELDS_BUFF ],;
                nFieldOffset + FRM_FIELD_HEADER_EXPR_OFFSET,; 
                2;
            );
        )

    cHeader := padr( frmGetExpr( @acFormBuffer, nPointer ), 260 )

    *-------------------------------------------------------------
    * Extracted data is transfered inside the <aStruct> array
    * used then for the editing.
    *-------------------------------------------------------------
    
    aadd(; 
        aStruct,; 
        { cHeader, cContent, nWidth, nDecimals, cTotals }; 
    )
    
    *-------------------------------------------------------------
    * Next offset is returned.
    *-------------------------------------------------------------
    
    return nFieldOffset + 12

*-----------------------------------------------------------------
static function frmEdit( cFileName, axFormVal, aStruct, lPreviousHelp )

    local aoGet                 := {}
    local aButtons              := {}
    local aTab                  := TAB_DEFAULT
    
    local bOldErrorHandler
    local cOldScreen
    local cOldColor             := setcolor( COLOR_BASE )
    local nOldCursor            := setcursor( SETCURSOR_NORMAL )
    local bOld_F1               := setkey( K_F1 )
    local bOld_F2               := setkey( K_F2, NIL )
    local nOldRow               := row()
    local nOldCol               := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local lSave                 := .F.

    local acCol
    local acColSayPic
    local acColHead
    local abColValid
    local abColMsg

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------
    
    begin sequence

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------
        
        nBottom             := maxrow()
        nTop                := nBottom - 23
        nLeft               := 0
        nRight              := maxcol()
        nWidth              := nRight - nLeft +1
            
        aTab[TAB_LEFT]      := nLeft
        aTab[TAB_RIGHT]     := nRight
        aTab[TAB_TAB_ARRAY] := { 26, 8, 26, 8, 4 }

        cOldScreen          :=; 
            mouseScrSave( nTop, nLeft, nBottom, nRight )
            
        setcolor( COLOR_BODY )
        
        scroll( nTop, nLeft, nBottom, nRight )
            
        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc(;
                FRM_DIALOG_BOX_TOP_REPORT,;
                nWidth-2;
                ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Set up local help.
        *---------------------------------------------------------

        if !lPreviousHelp

            setkey( K_F1, { || Text( FRM_HELP_REPORT_EDIT_HEAD ) } )

        end
        
        *---------------------------------------------------------
        * Do editing.
        *---------------------------------------------------------
            
        while .t.                                       // FOREVER

            say( nTop+3, nLeft+1, FRM_PROMPT_REPORT_PAGE_WIDTH )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axFormVal[ FRM_PAGE_WIDTH ] := x, axFormVal[ FRM_PAGE_WIDTH ] ) },;
                "999",;
                ,;
                { || messageline(), trueSetkey( K_F2, NIL ) },;
                { || .T. };
            )
            tab( aTab )
            say( row(), col(), FRM_PROMPT_REPORT_LINES_PER_PAGE )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axFormVal[ FRM_LINES_PAGE ] := x, axFormVal[ FRM_LINES_PAGE ] ) },;
                "999",;
                ,;
                { || messageline(), trueSetkey( K_F2, NIL ) },;
                { || .T. };
            )
            
            say( row()+1, nLeft+1, FRM_PROMPT_REPORT_LEFT_MARGIN )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axFormVal[ FRM_LEFT_MARG ] := x, axFormVal[ FRM_LEFT_MARG ] ) },;
                "999",;
                ,;
                { || messageline(), trueSetkey( K_F2, NIL ) },;
                { || .T. };
            )
            tab( aTab )
            say( row(), col(), FRM_PROMPT_REPORT_RIGHT_MARGIN )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axFormVal[ FRM_RIGHT_MARG ] := x, axFormVal[ FRM_RIGHT_MARG ] ) },;
                "999",;
                ,;
                { || messageline(), trueSetkey( K_F2, NIL ) },;
                { || .T. };
            )

            say( row()+2, nLeft+1, FRM_PROMPT_REPORT_DOUBLE_SPACED )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axFormVal[ FRM_DBL_SPACED ] := x, axFormVal[ FRM_DBL_SPACED ] ) },;
                "!",;
                ,;
                { || messageline(), trueSetkey( K_F2, NIL ) },;
                { ||; 
                    axFormVal[FRM_DBL_SPACED ] = "Y"    .or.; 
                    axFormVal[FRM_DBL_SPACED ] = "N";
                };
            )
            
            say( row()+2, nLeft+1, FRM_PROMPT_REPORT_PAGE_EJECT_BEFORE )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axFormVal[ FRM_PEBP ] := x, axFormVal[ FRM_PEBP ] ) },;
                "!",;
                ,;
                { || messageline(), trueSetkey( K_F2, NIL ) },;
                { ||; 
                    axFormVal[FRM_PEBP ] = "Y"    .or.; 
                    axFormVal[FRM_PEBP ] = "N";
                };
            )
            tab( aTab )
            say( row(), col(), FRM_PROMPT_REPORT_PAGE_EJECT_AFTER )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axFormVal[ FRM_PEAP ] := x, axFormVal[ FRM_PEAP ] ) },;
                "!",;
                ,;
                { || messageline(), trueSetkey( K_F2, NIL ) },;
                { ||; 
                    axFormVal[FRM_PEAP ] = "Y"    .or.; 
                    axFormVal[FRM_PEAP ] = "N";
                };
            )
            
            say( row()+2, nLeft+1, FRM_PROMPT_REPORT_PLAIN_PAGE )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axFormVal[ FRM_PLAINPAGE ] := x, axFormVal[ FRM_PLAINPAGE ] ) },;
                "!",;
                ,;
                { || messageline(), trueSetkey( K_F2, NIL ) },;
                { ||; 
                    axFormVal[ FRM_PLAINPAGE ] = "Y"    .or.; 
                    axFormVal[ FRM_PLAINPAGE ] = "N";
                };
            )
            
            say( row()+2, nLeft+1, FRM_PROMPT_REPORT_PAGE_HEADER )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axFormVal[ FRM_PAGE_HDR ] := x, axFormVal[ FRM_PAGE_HDR ] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { || messageline(), trueSetkey( K_F2, NIL ) },;
                { || .T. };
            )

            say( row()+2, nLeft+1, FRM_PROMPT_REPORT_GROUP_HEADER )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axFormVal[ FRM_GRP_HDR ] := x, axFormVal[ FRM_GRP_HDR ] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { || messageline(), trueSetkey( K_F2, NIL ) },;
                { || .T. };
            )
            
            say( row()+1, nLeft+1, FRM_PROMPT_REPORT_GROUP_EXPRESSION )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axFormVal[ FRM_GRP_EXPR ] := x, axFormVal[ FRM_GRP_EXPR ] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    FRM_MESSAGELINE_F2,;
                    setkey( K_F2, { || keyboard( fldNormal() ) } ),;
                    .T.;
                },;
                { || .T. };
            )

            say( row()+2, nLeft+1, FRM_PROMPT_REPORT_SUMMARY_REPORT_ONLY )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axFormVal[ FRM_SUMMARY ] := x, axFormVal[ FRM_SUMMARY ] ) },;
                "!",;
                ,;
                { || messageline(), trueSetkey( K_F2, NIL ) },;
                { ||; 
                    axFormVal[ FRM_SUMMARY ] = "Y"    .or.; 
                    axFormVal[ FRM_SUMMARY ] = "N";
                };
            )
            tab( aTab )
            say( row(), col(), FRM_PROMPT_REPORT_PAGE_EJECT_AFTER_GROUP )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axFormVal[ FRM_PE ] := x, axFormVal[ FRM_PE ] ) },;
                "!",;
                ,;
                { || messageline(), trueSetkey( K_F2, NIL ) },;
                { ||; 
                    axFormVal[ FRM_PE ] = "Y"    .or.; 
                    axFormVal[ FRM_PE ] = "N";
                };
            )
            
            say( row()+2, nLeft+1, FRM_PROMPT_REPORT_SUB_GROUP_HEADER )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axFormVal[ FRM_SUB_HDR ] := x, axFormVal[ FRM_SUB_HDR ] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { || messageline(), trueSetkey( K_F2, NIL ) },;
                { || .T. };
            )
            
            say( row()+1, nLeft+1, FRM_PROMPT_REPORT_SUB_GROUP_EXPRESSION )
            tab( aTab )
            get(;
                @aoGet, row(), col(),;
                { |x| iif( pcount() > 0, axFormVal[ FRM_SUB_EXPR ] := x, axFormVal[ FRM_SUB_EXPR ] ) },;
                picChrMax( col(), nRight-1 ),;
                ,;
                { ||;
                    FRM_MESSAGELINE_F2,;
                    setkey( K_F2, { || keyboard( fldNormal() ) } ),;
                    .T.;
                },;
                { || .T. };
            )
            
            button( @aButtons, row()+2, nLeft+1,;
                _BUTTON_ESC_CANCEL, , {||keyboard(chr(K_ESC))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F1_HELP, , {|| eval(setkey(K_F1))} )
            button( @aButtons, row(), col()+1,;
                _BUTTON_F2_LIST, , {|| iif( valtype(setkey(K_F2)) == "B", eval(setkey(K_F2)), NIL ) } )
            button( @aButtons, row(), col()+1,;
                _BUTTON_PGDN_CONFIRM, , {||keyboard(chr(K_PGDN))} )
            
            *-----------------------------------------------------
            * Read.
            *-----------------------------------------------------

            read( aoGet, , aButtons )
            aoGet   := {}
            aButtons := {}

            *-----------------------------------------------------
            * Check what was read.
            *-----------------------------------------------------
            
            do case
            case lastkey() = K_ESC

                *-------------------------------------------------
                * [Esc] means leave.
                *-------------------------------------------------

                lSave := .F.

                exit                                    // EXIT

            case lastkey() = K_PGDN 
            
                *-------------------------------------------------
                * [PgDn] means confirm.
                * Close last and open next window.
                *-------------------------------------------------
                
                mouseScrRestore(; 
                    nTop, nLeft, nBottom, nRight, cOldScreen; 
                )

                nBottom       := maxrow()
                nTop          := 1 //nBottom - 22
                nLeft         := 0
                nRight        := maxcol()
                nWidth        := nRight - nLeft +1

                cOldScreen    :=; 
                    mouseScrSave( nTop, nLeft, nBottom, nRight )

                setcolor( COLOR_BODY )

                scroll( nTop, nLeft, nBottom, nRight )

                dispBoxShadow(;
                    nTop, nLeft, nBottom, nRight,;
                    1,;
                    dispBoxColor( 1 ),;
                    dispBoxColor( 2 );
                )

                say(;
                    nTop+1, nLeft+1,;
                    padc( FRM_DIALOG_BOX_TOP_REPORT, nWidth-2 ),;
                    ,;
                    COLOR_HEAD;
                )

                *-------------------------------------------------
                * Change the help.
                *-------------------------------------------------

                if !lPreviousHelp

                    setkey(; 
                        K_F1,;
                        { || Text( FRM_HELP_REPORT_EDIT_COLUMNS ) }; 
                    )

                end

                *-------------------------------------------------
                * Prepare column informations
                *-------------------------------------------------

                acCol :=; 
                    {; 
                        "Header",;
                        "Content",;
                        "Width",;
                        "Decimals",;
                        "Totals"; 
                    }
                
                acColSayPic :=; 
                    {; 
                        "@s27",;
                        "@s27",;
                        "999",;
                        "99",;
                        "!"; 
                    }

                acColHead :=; 
                    {;  
                        FRM_HEAD_REPORT_HEADER,;
                        FRM_HEAD_REPORT_CONTENT,;
                        FRM_HEAD_REPORT_WIDTH,;
                        FRM_HEAD_REPORT_DECIMALS,;
                        FRM_HEAD_REPORT_TOTAL;
                    }

                abColValid :=; 
                    {; 
                        {||.T.},;
                        {||.T.},;
                        {||.T.},;
                        {||.T.},;
                        { |a, i, j|; 
                            a[i][j] == "Y"  .or.; 
                            a[i][j] == "N"  .or.; 
                            a[i][j] == " "; 
                        }; 
                    }                      
                    

                abColMsg :=; 
                    {; 
                        { ||; 
                            setkey( K_F2, NIL ),; 
                            ""; 
                        },;
                        { ||;
                            FRM_MESSAGELINE_F2,;
                            setkey(; 
                                K_F2,; 
                                { || keyboard( fldNormal() ) }; 
                            ),; 
                            ""; 
                        },;
                        { ||; 
                            setkey( K_F2, NIL ),; 
                            ""; 
                        },;
                        { ||; 
                            setkey( K_F2, NIL ),; 
                            ""; 
                        },;
                        { ||; 
                            setkey( K_F2, NIL ),; 
                            "Y/N"; 
                        };
                    }

                * ATB( <nTop>, <nLeft>, <nBottom>, <nRight>,
                *   <aArray>, [<nSubscript>],
                *   [<acColSayPic>],
                *   [<acColTopSep>], [<acColBodySep>], [<acColBotSep>],
                *   [<acColHead>], [<acColFoot>],
                *   [<abColValid>],
                *   [<abColMsg>],
                *   [<cColor>], [<abColColors>],
                *   [<lModify>],
                *   [lButtons|aButtons]
                *   )  --> aArray

                atb(; 
                    nTop+2, nLeft+1, nBottom-1, nRight-1,;
                    aStruct,,;
                    acColSayPic,;
                    ,,,;
                    acColHead,,;
                    abColValid,; 
                    abColMsg,;
                    ,,;
                    .T.,;
                    .T.;
                )

                *-------------------------------------------------
                * Close window.
                *-------------------------------------------------
                
                mouseScrRestore(; 
                    nTop, nLeft, nBottom, nRight, cOldScreen; 
                )

                lSave := .T.

                exit                                    // EXIT

            otherwise     

                *-------------------------------------------------
                * Exit with different keys is not allowed.
                * Loop again.
                *-------------------------------------------------
            
            end
        
        end

        *---------------------------------------------------------
        * Delete window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

    end //sequence

    *-------------------------------------------------------------
    * Restore old values.
    *-------------------------------------------------------------
    
    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    *-------------------------------------------------------------
    * Return True if the form is to be saved.
    *-------------------------------------------------------------
    
    return lSave

*----------------------------------------------------------------
static function frmSave( cFileName, axFormVal, aStruct )

    local nHandle             := 0
    local nByteWritten        := 0
    local cBuffer             := ""

    local nLastExpr           := 0  // initial value
    local nExprCount          := -1 // initial value
    local nLengthOffset       :=;
        FRM_LENGTHS_OFFSET - 1 // DOS offsets
    local nOffsetsOffset      := FRM_OFFSETS_OFFSET - 1
    local nFieldsOffset       := FRM_FIELDS_OFFSET - 1
    local nLengthsOffset      := FRM_LENGTHS_OFFSET - 1

    local anNum[5]

    local i                   := 0
    local j                   := 0

    local lOk                 := .F.

    nHandle := fcreate( cFileName )

    if !ferror() == 0
        alertBox( cFileName + NL(1) +;
            FRM_ERROR_CANNOT_CREATE + str(ferror()) )
        return .F.
    end

    *-------------------------------------------------------------
    * Write the Report Form skeleton.
    *-------------------------------------------------------------

    cBuffer :=;
        chr(2) + chr(0) +;
        replicate( chr(0), ( FRM_SIZE_FILE_BUFF - 4 ) ) +;
        chr(2) + chr(0)

    nByteWritten := fwrite( nHandle, cBuffer, FRM_SIZE_FILE_BUFF )

    if nByteWritten != FRM_SIZE_FILE_BUFF

        alertBox( FRM_ERROR_SOMETHING_GONE_WRONG )

        return .F.                                      // RETURN

    end

    *-------------------------------------------------------------
    * Write Page Heading info.
    *-------------------------------------------------------------

    if  (;
            anNum[ FRM_PAGE_HDR_NUM ] :=;
                frmWriteExpr(;
                    nHandle, axFormVal[ FRM_PAGE_HDR ], .T.,;
                    @nLastExpr, @nExprCount,;
                    @nLengthsOffset, @nOffsetsOffset;
                );
         ) = -1

        *---------------------------------------------------------
        * File write return an error.
        *---------------------------------------------------------

        return .F.                                      // RETURN

    end

    *-------------------------------------------------------------
    * Write Grouping expression info.
    *-------------------------------------------------------------

    if  (;
            anNum[ FRM_GRP_EXPR_NUM ] :=;
                frmWriteExpr(;
                    nHandle, axFormVal[ FRM_GRP_EXPR ], .T.,;
                    @nLastExpr, @nExprCount,;
                    @nLengthsOffset, @nOffsetsOffset;
                );
        ) = -1

        *---------------------------------------------------------
        * File write return an error.
        *---------------------------------------------------------

        return .F.                                      // RETURN

    end

    if  (;
            anNum[ FRM_SUB_EXPR_NUM ] :=;
                frmWriteExpr(;
                    nHandle, axFormVal[ FRM_SUB_EXPR ], .T.,;
                    @nLastExpr, @nExprCount,;
                    @nLengthsOffset, @nOffsetsOffset;
                );
        ) = -1

        *---------------------------------------------------------
        * File write return an error.
        *---------------------------------------------------------

        return .F.                                      // RETURN

    end

    if  (;
            anNum[ FRM_GRP_HDR_NUM ] :=;
                frmWriteExpr(;
                    nHandle, axFormVal[ FRM_GRP_HDR ], .T.,;
                    @nLastExpr, @nExprCount,;
                    @nLengthsOffset, @nOffsetsOffset;
                );
         ) = -1

        *---------------------------------------------------------
        * File write return an error.
        *---------------------------------------------------------

        return .F.                                      // RETURN

    end

    if  (;
            anNum[ FRM_SUB_HDR_NUM ] :=;
                frmWriteExpr(;
                    nHandle, axFormVal[ FRM_SUB_HDR ], .T.,;
                    @nLastExpr, @nExprCount,;
                    @nLengthsOffset, @nOffsetsOffset;
                );
         ) = -1

        *---------------------------------------------------------
        * File write return an error.
        *---------------------------------------------------------

        return .F.                                      // RETURN

    end

    *-------------------------------------------------------------
    * If arrived here it's ok.
    *-------------------------------------------------------------

    lOk := .T.

    *-------------------------------------------------------------
    * In this moment, the structure file for Report Form
    * is opened.
    *-------------------------------------------------------------

    j := len(aStruct)

    axFormVal[ FRM_COL_COUNT ] := j

    for i = 1 to j

        if  i == j                  .and.;
            empty( aStruct[i][2] )

            lOk := .T.

            axFormVal[ FRM_COL_COUNT ]-- // reduce

        else

            if  !frmWriteField( nHandle,;
                    @nFieldsOffset, @nLengthsOffsets,;
                    @nOffsetsOffset, @nLastExpr,;
                    @nExprCount, @aStruct, i;
                )

                lOk := .F.

                *-------------------------------------------------
                * If a column insert don't works, exit.
                *-------------------------------------------------

            end

        end

    next

    *-------------------------------------------------------------
    * Column info written ok?
    *-------------------------------------------------------------

    if lOk

        *----------------------------------------------------------
        * Write last 24 bytes of report and update
        * next_free_offset.
        *----------------------------------------------------------

        lOk :=;
            frmWriteParams(;
                nHandle, anNum,;
                axFormVal, nLastExpr;
            )

    end

    *-------------------------------------------------------------
    * Close
    *-------------------------------------------------------------

    if !fclose( nHandle )

        lOk := .F.

    end

    *-------------------------------------------------------------
    * Return the write result.
    *-------------------------------------------------------------

    return lOk

*----------------------------------------------------------------
static function frmWriteExpr(;
        nHandle, cString, lBlank, nLastExpr,;
        nExprCount, nLengthsOffset, nOffsetsOffset;
    )
*
* cString    string to be written inside the expression area.
*
* lBlank     test for dBASE like blank expression handling
*            and return a 65535 if expression to write is blank.
*
* -->
*    numeric, expression count (0 to 55 inclusive) or
*    65535 (if blank = .T. and EMPTY(string) = .T.) or
*    -1 (if WRITE/SEEK error).
*

    local lStatus             := .F.
    local cWriteItem          := ""
    local nWriteLen           := 0
    local nWriteCount         := 0
    local nReturnCount        := 0 // expression count: 65535 if empty, -1 if error.

    *-------------------------------------------------------------
    * For dBASE compatability.
    *-------------------------------------------------------------

    if lBlank .and. len(cString) == 0

        *---------------------------------------------------------
        * Empty expression
        *---------------------------------------------------------

        return 65535                                    // RETURN

    end

    cWriteItem := cString + chr(0)

    nWriteLen  := len(cWriteItem)

    *-------------------------------------------------------------
    * Move to the next free area (DOS offsets).
    *-------------------------------------------------------------

    fseek( nHandle, FRM_EXPR_OFFSET - 1 + nLastExpr )

    if ferror() <> 0

        *---------------------------------------------------------
        * Error
        *---------------------------------------------------------

        return -1                                       // RETURN

    end

    *-------------------------------------------------------------
    * Write the expression.
    *-------------------------------------------------------------

    nWriteCount := fwrite( nHandle, cWriteItem, nWriteLen )

    if nWriteCount == 0 .or. ferror() <> 0

        *---------------------------------------------------------
        * Error
        *---------------------------------------------------------

        return -1                                       // RETURN

    end

    fseek( nHandle, nOffsetsOffset)

    if ferror() <> 0

        *---------------------------------------------------------
        * Error
        *---------------------------------------------------------

        return -1                                       // RETURN

    end

    *-------------------------------------------------------------
    * Add an offset to the offsets array.
    *-------------------------------------------------------------

    nWriteCount := fwrite( nHandle, i2Bin( nLastExpr ), 2 )

    if nWriteCount == 0 .or. ferror() <> 0

        *---------------------------------------------------------
        * Error
        *---------------------------------------------------------

        return -1                                       // RETURN

    end

    fseek( nHandle, nLengthsOffset )

    if ferror() <> 0

        *---------------------------------------------------------
        * Error
        *---------------------------------------------------------

        return -1                                       // RETURN

    end

    *-------------------------------------------------------------
    * Add the expression length to the lengths array.
    *-------------------------------------------------------------

    nWriteCount := fwrite( nHandle, i2bin( nWriteLen ), 2)

    if nWriteCount == 0 .or. ferror() <> 0

        *---------------------------------------------------------
        * Error
        *---------------------------------------------------------

        return -1                                       // RETURN

    end

    *-------------------------------------------------------------
    * Move offsets to next position.
    *-------------------------------------------------------------

    nLastExpr += nWriteLen

    nLengthsOffset += 2

    nOffsetsOffset += 2

    nExprCount += 1         // global increment.

    return nExprCount

*----------------------------------------------------------------
static function frmWriteField(;
        nHandle, nFieldsOffset,;
        nLengthsOffset, nOffsetsOffset,;
        nLastExpr, nExprCount, aStruct, i;
    )
*
* -->   logical, success or fail of write operation.
*

    local cWriteItem          := ""
    local nWriteLen           := 0
    local nWriteCount         := 0
    local nCntsOffset         := 65535
    local nHdrOffset          := 65535

    *-------------------------------------------------------------
    * Write Contents.
    *-------------------------------------------------------------

    nCntsOffset :=;
        frmWriteExpr(;
            nHandle,;
            trim( aStruct[i][2] ), .F.,;
            @nLastExpr, @nExprCount,;
            @nLengthsOffset, @nOffsetsOffset;
        )

    *-------------------------------------------------------------
    * WRITE ok?
    *-------------------------------------------------------------

    if nCntsOffset != -1

        *---------------------------------------------------------
        * Write Header.
        *---------------------------------------------------------

        nHdrOffset :=;
            frmWriteExpr( nHandle,;
                trim( aStruct[i][1] ),;
                .T.,;
                @nLastExpr,;
                @nExprCount,;
                @nLengthsOffset,;
                @nOffsetsOffset;
            )

        *---------------------------------------------------------
        * WRITE ok?
        *---------------------------------------------------------

        if nHdrOffset != -1

            *-----------------------------------------------------
            * Seek to the next free FIELDS area.
            *-----------------------------------------------------

            nFieldsOffset := nFieldsOffset + 12

            fseek( nHandle, nFieldsOffset )

            *-----------------------------------------------------
            * SEEK ok?
            *-----------------------------------------------------

            if  ferror() == 0

                cWriteItem :=;
                    i2bin( aStruct[i][3] ) +;
                    replicate( chr(0), 3) +;
                    aStruct[i][5] +;
                    i2bin( aStruct[i][4] ) +; &&&Field->Decimals ) +;
                    i2bin( nCntsOffset ) +;
                    i2bin( nHdrOffset )

                nWriteLen := len( cWriteItem )

                *-------------------------------------------------
                *  Write the FIELDS info.
                *-------------------------------------------------

                nWriteCount :=;
                    fwrite( nHandle, cWriteItem, nWriteLen )

                if  nWriteCount = 0         .or.;
                    ferror() <> 0

                    return .F.                          // RETURN

                else

                    return .T.                          // RETURN

                end

            end

        end

    end

    return .T.

*-----------------------------------------------------------------
static function frmWriteParams(;
        nHandle, anNum, axFormVal,;
        nLastExpr;
    )
*
*
*  Writes the last 24 bytes of the report file plus
*   updates the first un-used offset. (last_offset)
*
* -->
*    logical, success or fail of write operation.
*

    local status              := .F.
    local cWriteItem          := ""
    local nWriteLen           := 0
    local nWriteCount         := 0
    local nPlusByte           := 0

    *-------------------------------------------------------------
    * Calculate plus byte.
    *-------------------------------------------------------------

    if axFormVal[ FRM_PLAINPAGE ] == "Y"

        nPlusByte += 4

    end

    if axFormVal[ FRM_PEAP ] == "Y"

        nPlusByte += 2

    end

    if axFormVal[ FRM_PEBP ] == "N"

        nPlusByte += 1

    end

    *-------------------------------------------------------------
    * Prepare miscellaneous data area string for write ops.
    *-------------------------------------------------------------

    cWriteItem =;
        i2bin( anNum[ FRM_PAGE_HDR_NUM ] ) +;
        i2bin( anNum[ FRM_GRP_EXPR_NUM ] ) +;
        i2bin( anNum[ FRM_SUB_EXPR_NUM ] ) +;
        i2bin( anNum[ FRM_GRP_HDR_NUM ] ) +;
        i2bin( anNum[ FRM_SUB_HDR_NUM ] ) +;
        i2bin( axFormVal[ FRM_PAGE_WIDTH ] ) +;
        i2bin( axFormVal[ FRM_LINES_PAGE ] ) +;
        i2bin( axFormVal[ FRM_LEFT_MARG ] ) +;
        i2bin( axFormVal[ FRM_RIGHT_MARG ] ) +;
        i2bin( axFormVal[ FRM_COL_COUNT ] ) +;
        axFormVal[ FRM_DBL_SPACED ] +;
        axFormVal[ FRM_SUMMARY ] +;
        axFormVal[ FRM_PE ] +;
        chr( nPlusByte )

    nWriteLen := len( cWriteItem )

    *-------------------------------------------------------------
    * Seek to first parameters area (DOS offsets).
    *-------------------------------------------------------------

    fseek( nHandle, FRM_PARAMS_OFFSET -1  )

    *-------------------------------------------------------------
    * SEEK ok?
    *-------------------------------------------------------------

    if ferror() == 0

        nWriteCount := fwrite( nHandle, cWriteItem, nWriteLen )

        if  nWriteCount = 0             .or.;
            ferror() <> 0

            return .F.                                  // RETURN

        end

    else

        return .F.                                      // RETURN

    end

    fseek( nHandle, 2 )  // next_free_offset

    if ferror() == 0

        *---------------------------------------------------------
        * Update the next free expression offset.
        *---------------------------------------------------------

        nWriteCount := fwrite( nHandle, i2bin( nLastExpr ), 2)

        *---------------------------------------------------------
        * write error.
        *---------------------------------------------------------

        if  nWriteCount = 0         .or.;
            ferror() <> 0

            return .F.                                  // RETURN

        end

    end

    *-------------------------------------------------------------
    * If here, ok.
    *-------------------------------------------------------------

    return .T.

#endif //RUNTIME

*=================================================================
* SELECTALIAS()
* SELECTARRAY()
* SELECTORDER()
*=================================================================

#define SELECT_MESSAGE_NEW_AREA;
    "New Area"

#define SELECT_MESSAGE_NATURAL_ORDER;
    "Natural"

#define SELECT_WINDOW_TOP_SELECT;
    "Alias list"

#define SELECT_WINDOW_TOP_ORDER;
    "Order list for the active Alias"

#define SELECT_HELP_SELECT;
    "ALIAS SELECTION" +;
    NL(2) +;
    "Select an Alias or a new area."

#define SELECT_HELP_ORDER;
    "ORDER SELECTION" +;
    NL(2) +;
    "Select an index order."

*=================================================================
function selectAlias()
*
* selectAlias()    --> cAlias
*                  --> ""
*

    local nSelect

    nSelect := selectArray()[1]

    if  nSelect <> NIL                      .and.;
        nSelect <> 0

        return alias(nSelect)

    end

    return ""

*=================================================================
function selectArray()
*
* selectChoice()  --> aSelect
*                 --> NIL
* selectChoice()[1] == Area number
* selectChoice()[2] == Alias
* selectChoice()[3] == Ordsetfocus
* selectChoice()[4] == OrdKey
*

    local bOldErrorHandler
    local cOldScreen
    local cOldColor           := setcolor( COLOR_BASE )
    local nOldCursor          := setcursor( SETCURSOR_NORMAL )
    local nOldSelect          := select()
    local bOld_F1             := setkey( K_F1 )
    local bOld_F2             := setkey( K_F2, NIL )
    local nOldRow             := row()
    local nOldCol             := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local aSelect             := {}
    local aSelectEasy         := {}
    local nI                  := 0
    local aSelected           :=;
        { NIL, NIL, NIL, NIL }

    *-------------------------------------------------------------
    * Check if a previous help exists. If so, no local help
    * will be set up.
    *-------------------------------------------------------------

    if bOld_F1 == NIL

        *---------------------------------------------------------
        * Set up a minimal help.
        *---------------------------------------------------------

        setkey( K_F1, { || Text(SELECT_HELP_SELECT) } )

    else
    
        *---------------------------------------------------------
        * There is already a help.
        *---------------------------------------------------------

    end

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Read the Aliases and add the "new area" as numnber 0
        * as the first on the list.
        *---------------------------------------------------------

        aadd( aSelect, { 0, SELECT_MESSAGE_NEW_AREA, "", "" } )

        for nI := 1 to _MAX_SELECT

            if alias(nI) == ""

                *-----------------------------------------------------
                * The area contains no Alias.
                * (the <> cannot be used to test empty strings)
                *-----------------------------------------------------

            else

                *-----------------------------------------------------
                * The area contains an Alias.
                *-----------------------------------------------------

                aadd(;
                    aSelect,;
                        {;
                            nI,;
                            alias(nI),;
                            (alias(nI))->(ordsetfocus()),;
                            (alias(nI))->(ordkey(ordsetfocus()));
                        };
                    )

            end

        next

        *---------------------------------------------------------
        * Translate the array into another.
        *---------------------------------------------------------

        for nI := 1 to len( aSelect )

            aadd(;
                aSelectEasy,;
                str( aSelect[nI,1], 3 ) +;
                    " " +;
                    padr( aSelect[nI,2], 10 ) +;
                    " " +;
                    aSelect[nI,3] +;
                    " " +;
                    aSelect[nI,4];
            )

        next

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nTop        := 0
        nLeft       := maxcol()-50
        nBottom     := maxrow()
        nRight      := maxcol()
        nWidth      := nRight - nLeft +1

        cOldScreen  :=;
            mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( SELECT_WINDOW_TOP_SELECT, nRight-nLeft-1 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Start aChoice() to select the area.
        *---------------------------------------------------------

        nI :=;
            achoice(;
                nTop+2, nLeft+1, nBottom-1, nRight-1,;
                aSelectEasy, , , .T.;
            )


        if nI <> 0

            aSelected := aSelect[nI]

        end

        *---------------------------------------------------------
        * Delete window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

    end //sequence

    *-------------------------------------------------------------
    * Restore old values.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    return aSelected

*-----------------------------------------------------------------
function selectOrder()
*
* selectOrder() --> nOrder
*                --> NIL
*
* It returns the order number selectable form the order list.
*

    local bOldErrorHandler
    local cOldScreen
    local cOldColor           := setcolor( COLOR_BASE )
    local nOldCursor          := setcursor( SETCURSOR_NONE )
    local bOld_F1             := setkey( K_F1 )
    local bOld_F2             := setkey( K_F2, NIL )
    local nOldRow             := row()
    local nOldCol             := col()
    local nTop
    local nLeft
    local nBottom
    local nRight
    local nWidth

    local aOrder              := {}
    local aOrderEasy          := {}
    local nI                  := 0

    *-------------------------------------------------------------
    * Check if a previous help exists. If so, no local help
    * will be set up.
    *-------------------------------------------------------------

    if bOld_F1 == NIL

        *---------------------------------------------------------
        * Set up a minimal help.
        *---------------------------------------------------------

        setkey( K_F1, { || Text(SELECT_HELP_ORDER) } )

    else
    
        *---------------------------------------------------------
        * There is already a help.
        *---------------------------------------------------------

    end

    *-------------------------------------------------------------
    * Start a new sequence.
    *-------------------------------------------------------------

    begin sequence

        *---------------------------------------------------------
        * Check it there is an Active Alias.
        *---------------------------------------------------------

        if alias() == ""

            alertBox( _ERROR_NO_ALIAS )

            break                                       // BREAK

        end

        *---------------------------------------------------------
        * Read the indexes including Zero as the natural order
        *---------------------------------------------------------

        aadd( aOrder, { 0, SELECT_MESSAGE_NATURAL_ORDER, "" } )

        for nI := 1 to _MAX_ORDER

            if ordkey(nI) == ""

                *-------------------------------------------------
                * There is no index inside this order number.
                * (with empty strings, <> can't be used.)
                *-------------------------------------------------

            else

                *-------------------------------------------------
                * Add the order key to the attay.
                *-------------------------------------------------

                aadd( aOrder, { nI, ordname(nI), ordkey(nI) } )

            end

        next

        *---------------------------------------------------------
        * Translate the array into another.
        *---------------------------------------------------------

        for nI := 1 to len( aOrder )

            aadd(;
                aOrderEasy,;
                str( aOrder[nI,1], 3 ) +;
                    " " +;
                    aOrder[nI,2] +;
                    " " +;
                    aOrder[nI,3];
                )

        next

        *---------------------------------------------------------
        * Create a kind of window.
        *---------------------------------------------------------

        nBottom     := maxrow()
        nRight      := maxcol()
        nTop        := 0
        nLeft       := nRight - 40
        nWidth      := nRight - nLeft +1

        cOldScreen  :=;
            mouseScrSave( nTop, nLeft, nBottom, nRight )

        setcolor( COLOR_BODY )

        scroll( nTop, nLeft, nBottom, nRight )

        dispBoxShadow(;
            nTop, nLeft, nBottom, nRight,;
            1,;
            dispBoxColor( 1 ),;
            dispBoxColor( 2 );
        )

        say(;
            nTop+1, nLeft+1,;
            padc( SELECT_WINDOW_TOP_ORDER, nRight-nLeft-1 ),;
            ,;
            COLOR_HEAD;
        )

        *---------------------------------------------------------
        * Start aChoice() to select the order.
        *---------------------------------------------------------

        nI :=;
            achoice(;
                nTop+2, nLeft+1, nBottom-1, nRight-1,;
                aOrderEasy, , , .T.;
            )

        *---------------------------------------------------------
        * Delete window.
        *---------------------------------------------------------

        mouseScrRestore( nTop, nLeft, nBottom, nRight, cOldScreen )

    end //sequence

    *-------------------------------------------------------------
    * Restore old values.
    *-------------------------------------------------------------

    setcursor( nOldCursor )
    setcolor( cOldColor )
    setpos( nOldRow, nOldCol )
    setkey( K_F1, bOld_F1 )
    setkey( K_F2, bOld_F2 )

    if nI <> 0

        return aOrder[nI,1]

    end

    *-------------------------------------------------------------
    * If [Esc] was pressed.
    *-------------------------------------------------------------

    return NIL

*=================================================================
* END
*=================================================================

