/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
    USA
*/

/*
   splitmer.c -- (c) Copyright Volker Kiefel 1998-2000

   command line program: split (archive files) into smaller files (to
   be stored on a diskette) and merge these files.

   the progam is primarily designed to split archive files for storage
   on some diskettes. The following conventions for naming of the large
   file and the split files are implied:

   largefl.zip -> 0000larg.zip 0001larg.zip 0002larg.zip
   0000larg.zip builds larg.zip or output filename and seeks for more
   files 0001larg.zip 0002larg.zip 0003larg.zip etc.
*/

#include <stdio.h>
#include <string.h>
#include "str_fun.h"
#include "fl_ut.h"

 long block_size;
 char ein_name[PFAD_LEN];
 FILE * fpein;
 char aus_name[PFAD_LEN];
 FILE * fpaus;
 char kommando[PFAD_LEN];
 char paramtext[PFAD_LEN];
 long block_groesse;

 void hilfe(char * av[]);
 int spalten(void);
 int zusammenfuegen(void);

int main(int argc, char * argv[])
{
   strcpy(ein_name,"");strcpy(aus_name,"");strcpy(kommando,"");
   strcpy(paramtext,"");
   block_groesse=1400000L;
   if (argc==1)
   {
     hilfe(argv); return 0;
   }
   if (argc==2 && (strcmp(argv[1],"-?")==0))
   {
     hilfe(argv); return 0;
   }
   if (parameter_auswerten_str(argc, argv, "-s",paramtext)==1 &&
     parameter_auswerten_str(argc, argv, "-m",paramtext)==1)
   {
     fprintf(stderr,"\nProblem: Commands `-s' and `-m' requested -- "
                     "not allowed\n");
     return 1;
   }
   if (parameter_auswerten_str(argc, argv, "-s",paramtext)==0 &&
     parameter_auswerten_str(argc, argv, "-m",paramtext)==0)
   {
     fprintf(stderr,"\nProblem: Neither command `-s' nor `-m' requested -- "
                     "nothing to do!\n");
     return 1;
   }

   /* Split files...*/
   if (parameter_auswerten_str(argc, argv, "-s",paramtext)==1)
   {
      if (parameter_auswerten_str(argc, argv, "-i",ein_name)==1)
      {
         if (!isdosfilename2(ein_name))
         {
           fprintf(stderr,"\nProblem: Command `-s' requested, input file "
                            "name `%s' invalid\n",ein_name);
           return 1;
         }
      }
      else
      {
         fprintf(stderr,"\nProblem: Command `-s' requested, input file "
                          "unknown\n");
         return 1;
      }
      if (parameter_auswerten_long(argc,argv,"-n",&block_groesse))
      {
         if (block_groesse < 60000L)
         {
           block_groesse = 60000L;
         }
      }
      else
      {
         fprintf(stderr,"\nProblem: Command `-s' requested, block size "
                          "unclear\n");
         return 1;
      }
      printf("\nSplit file `%s' -- block size: %li \n",
          ein_name,block_groesse);

      if (parameter_auswerten_str(argc, argv, "-o",aus_name)==1)
      {
          printf("Output file name ignored\n");
          fflush(stdout);
      }

      spalten();

   }
   /* Merge files...*/
   else if (parameter_auswerten_str(argc, argv, "-m",paramtext)==1)
   {
      if (parameter_auswerten_str(argc, argv, "-i",ein_name)==1)
      {
         if (!isdosfilename2(ein_name))
         {
           fprintf(stderr,"\nProblem: Command `-m' requested, input file "
                            "name `%s' invalid\n",ein_name);
           return 1;
         }
         if (locate_substring("0000",ein_name)!=0)
         {
           fprintf(stderr,"\nProblem: Command `-m' requested,\ninput file "
                            "name `%s' does not match "
                            "pattern `0000----.---'\n",ein_name);
           return 1;
         }
         if (locate_substring(".",ein_name)<5)
         {
           fprintf(stderr,"\nProblem: Command `-m' requested,\ninput file "
                            "name `%s' does not match "
                            "pattern `0000-[1..4]-.-[0..3]-'\n",ein_name);
           return 1;
         }
         if (!isfile(ein_name))
         {
           fprintf(stderr,"\nProblem: `%s' not found\n",ein_name);
           return 1;
         }

         if (parameter_auswerten_str(argc, argv, "-o",aus_name)==1)
         {
            if (!isdosfilename2(aus_name))
            {
              fprintf(stderr,"\nProblem: Command `-m' requested, output file "
                               "name `%s' invalid\n",aus_name);
              return 1;
            }
         }
         else
         {
            /* aus_name automatisch generieren... */
            extract_subs(aus_name,ein_name,4,PFAD_LEN);

         }
         printf("\nMerge files starting with `%s'\n",ein_name);
         zusammenfuegen();
      }
      else
      {
         fprintf(stderr,"\nProblem: Command `-m' requested, input file "
                          "unknown\n");
         return 1;
      }
   }
   else
   {
      fprintf(stderr,"\nProblem: No command `-m' or `-s' requested\n");
      return 1;
   }

   return 0;
}

void hilfe(char * av[])
{
  printf(
  "\nUsage: %s -<Command> -i<InputFile> [-o<OutputFile>]"
  "\n"
  "\nCommands:"
  "\n  -s   split InputFile"
  "\n  -n#  block size in bytes (default=1400000, min=60000)"
  "\n  -m   merge series of files, beginning with InputFile"
  "\n"
  "\nExamples of valid command lines:                                (explanation)"
  "\n  splitmer -s -ilargef.zip        (splits `largeref.zip', default block size)"
  "\n  splitmer -s -n1200000 -ilargef.zip           (... 1200000 bytes block size)"
  "\n  splitmer -m -i0000larg.zip    (merges series of files beginning with -i...)"
  "\n  splitmer -m -i0000larg.zip -olargef.zip      (... output file `largef.zip')\n"
  "\nSplitmer Version 1.3 -- Copyright (c) V. Kiefel 1998-2000"
  "\n",
  av[0]);
}

int spalten(void)
{
   long i,j, ein_datei_groesse;
   long di;
   char ex_name[PFAD_LEN];
   char teil_name[PFAD_LEN];
   int c;

   if (!isfile(ein_name))
   {
     fprintf(stderr,"\nProblem: `%s' not found\n",ein_name);
     return 0;
   }
   ein_datei_groesse=filesize(ein_name);
   di=0;
   strcpy(aus_name,"");
   strcpy(teil_name,"");
   strcpy(ex_name,"");
   extractitem(1,ein_name,".",ex_name);
   extract_subs(teil_name,ex_name,0,4);
   strcat(teil_name,".");
   extractitem(2,ein_name,".",ex_name);
   strcat(teil_name,ex_name);

   upper(teil_name);
   j=0; /* Zaehler in der gespaltenen Datei */
   if ((fpein=fopen(ein_name,"rb"))==NULL)
   {
     fprintf(stderr,"\nProblem: Cannot open `%s'\n",ein_name);
     return 0;
   }
   i=0;
   for (;;)
   {
      if (j==0)
      {
        sprintf(aus_name,"%04li%s",di,teil_name);
        if (isfile(aus_name))
        {
          fprintf(stderr,"\nProblem: File `%s' already exists\n",aus_name);
          return 0;
        }
        printf("%s\n",aus_name);
        if ((fpaus=fopen(aus_name,"wb"))==NULL)
        {
          fprintf(stderr,"\nProblem: Cannot create `%s'\n",aus_name);
          return 0;
        }

      }

      c = getc(fpein);
      putc(c,fpaus);
      j++;
      i++;
      if (j==block_groesse || i==ein_datei_groesse)
      {
         fclose(fpaus);
         if (i==ein_datei_groesse)
         {
            break;
         }
         else
         {
            j=0;
            di++;
         }
      }
   }
   fclose(fpein);
   return 1;
}

int zusammenfuegen(void)
{
   long i, j, ein_datei_groesse;
   char teil_name[PFAD_LEN];
   int c;

   if (isfile(aus_name))
   {
     fprintf(stderr,"\nProblem: File `%s' already exists\n",aus_name);
     return 0;
   }
   if ((fpaus=fopen(aus_name,"wb"))==NULL)
   {
     fprintf(stderr,"\nProblem: Cannot create `%s'\n",aus_name);
     return 0;
   }
   i=0; j=0;
   extract_subs(teil_name,ein_name,4,PFAD_LEN);
   for(;;)
   {
      sprintf(ein_name,"%04li%s",i,teil_name);
      if (!isfile(ein_name)) break;
      printf("%s\n",ein_name);
      ein_datei_groesse = filesize(ein_name);
      if ((fpein=fopen(ein_name,"rb"))==NULL)
      {
        fprintf(stderr,"\nProblem: Cannot open `%s'\n",ein_name);
        return 0;
      }
      for (j=0;j<ein_datei_groesse;j++)
      {
         c = getc(fpein);
         putc(c,fpaus);
      }
      fclose(fpein);
      i++;
   }
   fclose(fpaus);
   return 1;
}
