/* Copyright (C) Stephen Chung, 1991-1993.  All rights reserved. */

#include "jwp.h"
#ifdef CTL3D
    #include <ctl3d.h>
#endif CTL3D


#define KANJIBOXBORDER    AVGWIDTH
#define KANJILISTBORDER   BORDERSPACE
#define SCROLLBLOCKS      5
#define OKURIGANA         0x20

#define NRSORT  2
#define MAXLIST 50
#define STROKES 0
#define BUSHU   1

typedef struct {
	unsigned char strokes;
    unsigned char bushu;
    unsigned short int nelson;
    unsigned char grade;
    unsigned char on:4;
    unsigned char kun:4;
    unsigned char unknown:4;
    unsigned char meanings:4;
    unsigned char padding;              // To pad it to 32 bits
	unsigned long int offset;
} INFORECORD;

typedef struct {
    unsigned char sortkeys[NRSORT];
    KANJI jis;
} IDXRECORD;

static OFSTRUCT kinfoidxof;
static BOOL KinfoIdx = FALSE;
static int KinfoMax = -1;
static int SortKey;
static int IdxBase, NrIndex;
static int CurrentKeys[NRSORT];
static int MaxKeys[NRSORT];
static int fd;


static IDXRECORD KanjiList[MAXLIST];
static int start, stop;
static int NrInList, NrSelected;
static int EdgeGap;

static KANJI InfoChar;


#define NRJISROWS  6

static int PageNum;
static int SelRow, SelCol;
static int RowDiv, ColDiv;
static int RowPos, ColPos;

BOOL FAR PASCAL KanjiInfoProc (HWND, WORD, WORD, LONG);

static KANJI BushuSymbols[] = {
    0x306c, 0x2143, 0x5026, 0x254e, 0x3235,     /* 01 - 05 */
    0x502d, 0x4673, 0x5035, 0x3f4d, 0x5139,     /* 06 - 10 */
    0x467e, 0x482c, 0x5144, 0x514c, 0x5152,     /* 11 - 15 */
    0x515c, 0x5161, 0x4561, 0x4e4f, 0x5231,     /* 16 - 20 */
    0x5238, 0x5239, 0x523e, 0x3d3d, 0x4b4e,     /* 21 - 25 */
    0x5247, 0x524c, 0x5253, 0x4b74, 0x387d,     /* 26 - 30 */
    0x5378, 0x455a, 0x3b4e, 0x5469, 0x546a,     /* 31 - 35 */
    0x4d3c, 0x4267, 0x3d77, 0x3b52, 0x555f,     /* 36 - 40 */
    0x4023, 0x3e2e, 0x5577, 0x5579, 0x4366,     /* 41 - 45 [ 45 qustionable ] */
    0x3b33, 0x406e, 0x3929, 0x384a, 0x3652,     /* 46 - 50 */
    0x3433, 0x5676, 0x5678, 0x572e, 0x5730,     /* 51 - 55 */
    0x5735, 0x355d, 0x4730, 0x5744, 0x5746,     /* 56 - 60 [ 58 questionable ] */
    0x3f34, 0x5879, 0x384d, 0x3c6a, 0x3b59,     /* 61 - 65 */
    0x5a3d, 0x4a38, 0x454d, 0x3654, 0x4a7d,     /* 66 - 70 */
    0x5a5b, 0x467c, 0x5b29, 0x376e, 0x4c5a,     /* 71 - 75 */
    0x3767, 0x3b5f, 0x5d46, 0x5d55, 0x5d59,     /* 76 - 80 */
    0x4866, 0x4c53, 0x3b61, 0x5d63, 0x3f65,     /* 81 - 85 */
    0x3250, 0x445e, 0x4963, 0x602b, 0x602d,     /* 86 - 90 */
    0x4a52, 0x3267, 0x356d, 0x3824, 0x383c,     /* 91 - 95 */
    0x364c, 0x313b, 0x3424, 0x3445, 0x4038,     /* 96 - 100 */
    0x4d51, 0x4544, 0x4925, 0x614b, 0x6222,     /* 101 - 105 [ 104 questionable ] */
    0x4772, 0x4869, 0x3b2e, 0x4c5c, 0x4c37,     /* 106 - 110 */
    0x4c70, 0x4050, 0x3c28, 0x633b, 0x3253,     /* 111 - 115 [ 114 questionable ] */
    0x376a, 0x4e29, 0x435d, 0x4a46, 0x3b65,     /* 116 - 120 */
    0x344c, 0x6626, 0x4d53, 0x3129, 0x4f37,     /* 121 - 125 */
    0x3c29, 0x6650, 0x3c2a, 0x6666, 0x4679,     /* 126 - 130 */
    0x3f43, 0x3c2b, 0x3b6a, 0x3131, 0x4065,     /* 131 - 135 */
    0x4124, 0x3d2e, 0x3a31, 0x3f27, 0x6767,     /* 136 - 140 */
    0x6948, 0x436e, 0x376c, 0x3954, 0x3061,     /* 141 - 145 */
    0x403e, 0x382b, 0x3351, 0x3840, 0x432b,     /* 146 - 150 */
    0x4626, 0x6c35, 0x6c38, 0x332d, 0x4056,     /* 151 - 155 */
    0x4176, 0x422d, 0x3f48, 0x3c56, 0x3f49,     /* 156 - 160 */
    0x4324, 0x6d68, 0x4d38, 0x4653, 0x4850,     /* 161 - 165 [ 162 questionable ] */
    0x4e24, 0x3662, 0x4439, 0x4c67, 0x496c,     /* 166 - 170 */
    0x7030, 0x7032, 0x312b, 0x4044, 0x4873,     /* 171 - 175 */
    0x4c4c, 0x3357, 0x706a, 0x706c, 0x323b,     /* 176 - 180 */
    0x4a47, 0x4977, 0x4874, 0x3f29, 0x3c73,     /* 181 - 185 */
    0x3961, 0x474f, 0x397c, 0x3962, 0x7175,     /* 186 - 190 */
    0x7228, 0x722e, 0x722f, 0x3534, 0x357b,     /* 191 - 195 */
    0x443b, 0x7343, 0x3c2f, 0x734e, 0x4b63,     /* 196 - 200 */
    0x322b, 0x3550, 0x3975, 0x7363, 0x7366,     /* 201 - 205 */
    0x4524, 0x385d, 0x414d, 0x4921, 0x736e,     /* 206 - 210 */
    0x736f, 0x4e36, 0x737d, 0x737e, 0x2121,     /* 211 - 215 */
    0x2121, 0x2121, 0x2121, 0x2121, 0x2121,     /* 216 - 220 */
    0x2121, 0x2121, 0x2121, 0x2121, 0x2121,     /* 221 - 225 */
    0x2121, 0x2121, 0x2121, 0x2121, 0x2121,     /* 226 - 230 */
    0x2121, 0x2121, 0x2121, 0x2121, 0x2121,     /* 231 - 235 */
    0x2121, 0x2121, 0x2121, 0x2121, 0x2121,     /* 236 - 240 */
    0x2121, 0x2121, 0x2121, 0x2121, 0x2121,     /* 241 - 245 */
    0x2121, 0x2121, 0x2121, 0x2121, 0x2121,     /* 246 - 250 */
    0x2121, 0x2121, 0x2121, 0x2121, 0x2121      /* 251 - 255 */
};




void SetKanjiInfoChar (KANJI ch)
{
	InfoChar = ch;
}



static int KeysComp (int keys[], IDXRECORD record, int SortKey)
{
    int i;

    if (keys[SortKey] != record.sortkeys[SortKey])
        return (keys[SortKey] - record.sortkeys[SortKey]);

    for (i = 0; i < NRSORT; i++) {
        if (i == SortKey) continue;
        if (keys[i] != record.sortkeys[i])
            return (keys[i] - record.sortkeys[i]);
    }

	return (0);
}



static int BinarySearchIndex (int fd, int which, BOOL lower)
{
	int top, bottom, middle;
	int diff;
	IDXRECORD record;

	/* Now binary search */

	top = 0;
	bottom = NrIndex - 1;

	for (;;) {
		middle = (top + bottom) / 2;

        lseek(fd, IdxBase * which + middle * sizeof(IDXRECORD), 0L);
        read(fd, &record, sizeof(IDXRECORD));

        diff = KeysComp(CurrentKeys, record, which);

        if (diff == 0) {
            /* Move to the beginning */
            while (middle >= 0) {
                lseek(fd, IdxBase * which + middle * sizeof(IDXRECORD), 0);
				read(fd, &record, sizeof(IDXRECORD));
                if (KeysComp(CurrentKeys, record, which) > 0) break;
                middle--;
            }

            return (middle + 1);
        }

        if (top >= bottom - 1) return (lower ? bottom : top);

        if (diff > 0) top = middle;
		else bottom = middle;
	}
}



static void PerformSearch (HWND hwnd, int fd, BOOL direction, BOOL MainKey)
{
    int i, j;
    IDXRECORD record;
    HCURSOR hCursor;


    hCursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
    ShowCursor(TRUE);

    /* Now binary search the list */

    j = BinarySearchIndex(fd, SortKey, TRUE);
	lseek(fd, IdxBase * SortKey + j * sizeof(IDXRECORD), 0);
    read(fd, &record, sizeof(IDXRECORD));

    if (MainKey) {
        if (!direction) {
            if (CurrentKeys[SortKey] != record.sortkeys[SortKey]) j--;
        }
    } else {
        if (direction) {
            if (CurrentKeys[SortKey] != record.sortkeys[SortKey]) j--;
        } else {
            if (KeysComp(CurrentKeys, record, SortKey) < 0) {
                lseek(fd, IdxBase * SortKey + (j-1) * sizeof(IDXRECORD), 0);
                read(fd, &record, sizeof(IDXRECORD));
                if (CurrentKeys[SortKey] == record.sortkeys[SortKey]) j--;
            }
        }
    }


    start = j - NrSelected;
    stop = start + NrInList - 1;

    lseek(fd, SortKey * IdxBase, 0);
    if (start > 0) lseek(fd, start * sizeof(IDXRECORD), 1);

    for (j = 0; j < NrInList; j++) {
        if (start + j < 0 || start + j >= NrIndex) {
            KanjiList[j].jis = 0;
        } else {
            read(fd, &record, sizeof(IDXRECORD));
            KanjiList[j] = record;
        }

        if (j != NrSelected) continue;

        for (i = 0; i < NRSORT; i++) {
            if (record.sortkeys[i] != CurrentKeys[i]) {
                CurrentKeys[i] = record.sortkeys[i];
                SetScrollPos(GetDlgItem(hwnd, 4221 + i), SB_CTL, CurrentKeys[i], TRUE);
                SetDlgItemInt(hwnd, 4231 + i, CurrentKeys[i], TRUE);
            }
        }
    }

    InvalidateRect(GetDlgItem(hwnd, 4201), NULL, FALSE);
    SendDlgItemMessage(hwnd, 4201, WM_USER, 0, 0L);

    ShowCursor(FALSE);
    SetCursor(hCursor);
}



BOOL FAR PASCAL LookupProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    int i, j;
    long int offset;
    IDXRECORD record;

	switch (message) {
		case WM_INITDIALOG:
            if (!KinfoIdx) {
                /* Open index file */

                fd = OpenFile(global.kinfoidx, &kinfoidxof, OF_READ);
                if (fd <= 0) {
                    ErrorMessage(global.hwnd, "Cannot find Kanji info index file!");
                    EndDialog(hwnd, FALSE);
                    return (TRUE);
                }
                KinfoIdx = TRUE;

                lseek(fd, 0L, 2);       /* Seek to the end */
                offset = tell(fd);      /* File size */

                if (offset % sizeof(IDXRECORD) != 0) {
                    ErrorMessage(global.hwnd, "Funny, size of idx = %d", offset);
                }

                if ((offset / sizeof(IDXRECORD)) % NRSORT != 0) {
                    ErrorMessage(global.hwnd, "Funny, %d not divisible by %d", offset, NRSORT);
                }

                /* Offset of the last record */

                IdxBase = offset / NRSORT;
                NrIndex = IdxBase / sizeof(IDXRECORD);
                offset = IdxBase - sizeof(IDXRECORD);

                /* Get the maximums */

                for (i = 0; i < NRSORT; i++) {
                    lseek(fd, IdxBase * i + offset, 0);
                    read(fd, &record, sizeof(IDXRECORD));
                    MaxKeys[i] = record.sortkeys[i];
                }
            } else {
                fd = OpenFile(NULL, &kinfoidxof, OF_READ | OF_REOPEN);
            }

			SetDlgItemInt(hwnd, 4231, 1, TRUE);
            SetDlgItemInt(hwnd, 4232, 1, TRUE);

            SetWindowPos(GetDlgItem(hwnd, 4241), NULL, 0, 0,
                            SYSFONT->width, SYSFONT->height,
                            SWP_NOMOVE | SWP_NOZORDER);

            SetScrollRange(GetDlgItem(hwnd, 4221), SB_CTL, 1, MaxKeys[STROKES], TRUE);
            SetScrollPos(GetDlgItem(hwnd, 4221), SB_CTL, 1, TRUE);
            SetScrollRange(GetDlgItem(hwnd, 4222), SB_CTL, 1, MaxKeys[BUSHU], TRUE);
            SetScrollPos(GetDlgItem(hwnd, 4222), SB_CTL, 1, TRUE);

            SetScrollRange(GetDlgItem(hwnd, 4201), SB_HORZ, 0, NrIndex - 1, TRUE);
            SetScrollPos(GetDlgItem(hwnd, 4201), SB_HORZ, 1, TRUE);

            SortKey = STROKES;
            for (i = 0; i < NRSORT; i++) CurrentKeys[i] = 1;

            SendDlgItemMessage(hwnd, 4211, BM_SETCHECK, 1, 0L);
            SendDlgItemMessage(hwnd, 4212, BM_SETCHECK, 0, 0L);

            PerformSearch (hwnd, fd, TRUE, TRUE);

            SendMessage(hwnd, WM_USER, 0, 0L);  /* Draw the Bushu icon */

            if (global.active == NULL)
                EnableWindow(GetDlgItem(hwnd, IDOK), FALSE);    /* Insert button */

            CenterDialogBox(hwnd);

            return (TRUE);

        case WM_HSCROLL: {
            HWND child;
			BOOL direction;

			child = HIWORD(lParam);

            switch (GetWindowWord(child, GWW_ID)) {
                case 4221:      /* Strokes scrollbar */
                    i = STROKES;
                    break;

                case 4222:      /* Bushu scrollbar */
                    i = BUSHU;
                    break;

                default: return (TRUE);
            }

			switch (wParam) {
                case SB_TOP:
                    CurrentKeys[i] = 1;
                    direction = FALSE;
                    break;

                case SB_BOTTOM:
                    CurrentKeys[i] = MaxKeys[i];
                    direction = TRUE;
                    break;

                case SB_LINEUP:
                    if (CurrentKeys[i] <= 1) return (TRUE);
					CurrentKeys[i]--;
                    direction = FALSE;
                    break;

                case SB_PAGEUP:
                    if (CurrentKeys[i] <= 1) return (TRUE);
					CurrentKeys[i] -= (MaxKeys[i] / SCROLLBLOCKS);
                    if (CurrentKeys[i] < 1) CurrentKeys[i] = 1;
                    direction = FALSE;
                    break;

                case SB_LINEDOWN:
                    if (CurrentKeys[i] >= MaxKeys[i]) return (TRUE);
					CurrentKeys[i]++;
                    direction = TRUE;
                    break;

                case SB_PAGEDOWN:
                    if (CurrentKeys[i] >= MaxKeys[i]) return (TRUE);
                    CurrentKeys[i] += (MaxKeys[i] / SCROLLBLOCKS);
                    if (CurrentKeys[i] > MaxKeys[i]) CurrentKeys[i] = MaxKeys[i];
                    direction = TRUE;
                    break;

                case SB_THUMBPOSITION:
                case SB_THUMBTRACK:
					CurrentKeys[i] = LOWORD(lParam);
                    direction = TRUE;
                    break;

                default: return (TRUE);
            }

            SetScrollPos(child, SB_CTL, CurrentKeys[i], TRUE);
            SetDlgItemInt(hwnd, 4231 + i, CurrentKeys[i], TRUE);

            if (wParam == SB_THUMBTRACK) {
                if (4231 + i == 4232) {     /* Bushu */
                    SendMessage(hwnd, WM_USER, 0, 0L);  /* Draw the Bushu icon */
                }

                return (TRUE);
            }

            /* Set everything else to 1? */

            if (i == SortKey) {
                for (j = 0; j < NRSORT; j++) {
                    if (j == SortKey) continue;
                    CurrentKeys[j] = 1;
                    SetScrollPos(GetDlgItem(hwnd, 4221 + j), SB_CTL, 1, TRUE);
                    SetDlgItemInt(hwnd, 4231 + j, 1, TRUE);

                    if (4231 + j == 4232) {     /* Bushu */
                        SendMessage(hwnd, WM_USER, 0, 0L);  /* Draw the Bushu icon */
                    }
                }
            }

            PerformSearch (hwnd, fd, direction, i == SortKey);

            SendMessage(hwnd, WM_USER, 0, 0L);  /* Draw the Bushu icon */

            return (TRUE);
		}

        case WM_PAINT:
            SendMessage(hwnd, WM_USER, 0, 0L);  /* Draw the Bushu icon */
            break;

        case WM_USER: {
            int r, Bushu;
            HDC hdc;
			BYTE far *cbufp;
			extern BOOL Dialogs3D;

			Bushu = CurrentKeys[1];

			if (Bushu <= 0 || Bushu > 214) break;

			hdc = GetDC(GetDlgItem(hwnd, 4241));
			SetTextColor(hdc, GetSysColor(COLOR_WINDOWTEXT));
			SetBkColor(hdc, GetSysColor(COLOR_WINDOW));

#ifdef CTL3D
			if (Dialogs3D) {
				RECT rect;
				HBRUSH hbrush;

				hbrush = Ctl3dCtlColorEx(WM_CTLCOLOR, hdc, MAKELONG(hwnd, CTLCOLOR_DLG));
                GetClientRect(GetDlgItem(hwnd, 4241), &rect);
				FillRect(hdc, &rect, hbrush);
            } else
#endif CTL3D
            {
				HBRUSH hbrush;
				RECT rect;

                hbrush = CreateSolidBrush(GetSysColor(COLOR_WINDOW));
                GetClientRect(GetDlgItem(hwnd, 4241), &rect);
                FillRect(hdc, &rect, hbrush);
                DeleteObject(hbrush);
            }

			r = Jis2Index(BushuSymbols[Bushu - 1], SYSFONT->holes);
			if (r >= 0) {
				r = GetKanjiBitmap(SYSFONT, r, &cbufp);
                DisplayKanjiBitmap(hdc, 0, SYSFONT->height,
                                    SYSFONT->width, SYSFONT->height,
                                    r, SRCAND, cbufp);
            }

            ReleaseDC(GetDlgItem(hwnd, 4241), hdc);
            return (TRUE);
        }

        case WM_COMMAND:
            switch (wParam) {
                case 4211:      /* Strokes first */
                    SendDlgItemMessage(hwnd, 4211, BM_SETCHECK, 1, 0L);
                    SendDlgItemMessage(hwnd, 4212, BM_SETCHECK, 0, 0L);
                    SortKey = STROKES;
                    PerformSearch (hwnd, fd, TRUE, TRUE);
					return (TRUE);

				case 4212:      /* Bushu first */
					SendDlgItemMessage(hwnd, 4211, BM_SETCHECK, 0, 0L);
					SendDlgItemMessage(hwnd, 4212, BM_SETCHECK, 1, 0L);
					SortKey = BUSHU;
                    PerformSearch (hwnd, fd, TRUE, TRUE);
                    return (TRUE);

                case 4213:      /* Info */
					SetKanjiInfoChar(KanjiList[NrSelected].jis);

                    DialogBox (hInstance, "KanjiInfo", hwnd, KanjiInfoProc);
                    return (TRUE);

                case IDOK: {        /* Insert button */
                    KANJI buf[2];

                    if (curfile != NULL) {
                        buf[0] = KanjiList[NrSelected].jis;
                        buf[1] = 0;

                        TurnOffSelection(curfile);
                        UndoAddTyping(curfile, curfile->current, 1, FALSE);
                        InsertString(curfile, curfile->current, buf,
                                        OP_REFORMAT | OP_UPDATE | OP_MOVESEL | OP_MOVETOEND);
                    }
                }
                /* ... drops into the next case ... */

                case IDCANCEL:
                    EndDialog(hwnd, FALSE);
                    close(fd);
                    return (TRUE);

                default: break;
            }
            break;
    }

    return (FALSE);
}



LONG FAR PASCAL LookupListProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    int i, j, r;
    RECT rect;
    HDC hdc;
    HWND parent;
    HBRUSH hbrush;
    HPEN hpen;
    PAINTSTRUCT ps;
    IDXRECORD record;
    BYTE far *cbufp;

    switch (message) {

        case WM_CREATE:
            GetClientRect(hwnd, &rect);
            i = GetSystemMetrics(SM_CYHSCROLL) + SYSFONT->height + 1
                + 2 * KANJILISTBORDER;

            SetWindowPos(hwnd, NULL, 0, 0, rect.right, i, SWP_NOMOVE | SWP_NOZORDER);

            NrInList = (rect.right - 2 * KANJILISTBORDER) / SYSFONT->width;
            NrInList /= 2;

            if (NrInList % 2 != 0) NrSelected = NrInList / 2;
            else NrSelected = (NrInList / 2) - 1;

            if (NrInList <= 1) NrInList = NrSelected = 1;
            EdgeGap = (rect.right - (NrInList * 2 - 1) * SYSFONT->width) / 2;
            if (EdgeGap < 0) EdgeGap = 1;
            return (0);

        case WM_GETDLGCODE:
            return (DLGC_WANTARROWS);

        case WM_LBUTTONDOWN: {
            int x;

            x = LOWORD(lParam);
            GetClientRect(hwnd, &rect);

            /* Find out which character it clicks on */

            i = (x + (SYSFONT->width / 2) - EdgeGap) / (2 * SYSFONT->width);
            if (KanjiList[i].jis == 0 || i < 0 || i > NrInList) {
                MessageBeep(0);
                return (0);
            }

            /* Change the high-light */

            hdc = GetDC(hwnd);
            SetTextColor(hdc, GetSysColor(COLOR_WINDOWTEXT));
            SetBkColor(hdc, GetSysColor(COLOR_WINDOW));
            PatBlt(hdc, (NrSelected * 2 * SYSFONT->width) + EdgeGap - (SYSFONT->width / 2),
                    2, 2 * SYSFONT->width, rect.bottom - 4, DSTINVERT);
            PatBlt(hdc, (i * 2 * SYSFONT->width) + EdgeGap - (SYSFONT->width / 2),
                    2, 2 * SYSFONT->width, rect.bottom - 4, DSTINVERT);
            ReleaseDC(hwnd, hdc);

            NrSelected = i;


            /* Change the scroll bars */

            for (j = 0; j < NRSORT; j++) {
                CurrentKeys[j] = KanjiList[NrSelected].sortkeys[j];
                SetScrollPos(GetDlgItem(GetParent(hwnd), 4221 + j), SB_CTL, CurrentKeys[j], TRUE);
                SetDlgItemInt(GetParent(hwnd), 4231 + j, CurrentKeys[j], TRUE);

                if (4231 + j == 4232) {     /* Bushu */
                    SendMessage(GetParent(hwnd), WM_USER, 0, 0L);  /* Draw the Bushu icon */
                }
            }
            return (0);
        }

        case WM_LBUTTONDBLCLK:
            SendMessage(GetParent(hwnd), WM_COMMAND, IDOK, 0L);
            return (0);

        case WM_PAINT:
            GetClientRect(hwnd, &rect);
            hdc = BeginPaint(hwnd, &ps);

            SetTextColor(hdc, GetSysColor(COLOR_WINDOWTEXT));
            SetBkColor(hdc, GetSysColor(COLOR_WINDOW));
            hbrush = SelectObject(hdc, GetStockObject(NULL_BRUSH));
            hpen = SelectObject(hdc, CreatePen(PS_SOLID, 1, GetSysColor(COLOR_WINDOWFRAME)));

            Rectangle(hdc, 0, 0, rect.right, rect.bottom);

            DeleteObject(SelectObject(hdc, CreatePen(PS_SOLID, 1, GetSysColor(COLOR_BTNSHADOW))));

            Rectangle(hdc, 1, 1, rect.right - 1, rect.bottom - 1);

            SelectObject(hdc, CreateSolidBrush(GetSysColor(COLOR_WINDOW)));

            //EnableFontCache(FALSE);

			for (i = 0, j = EdgeGap; i < NrInList; i++, j += 2 * SYSFONT->width) {
                if (KanjiList[i].jis == 0) {
                    PatBlt(hdc, j, KANJILISTBORDER, SYSFONT->width,
                            SYSFONT->height, PATCOPY);
                    continue;
                }

                r = Jis2Index(KanjiList[i].jis, SYSFONT->holes);
                if (r < 0) r = Jis2Index(BADKANJI, SYSFONT->holes);
                r = GetKanjiBitmap(SYSFONT, r, &cbufp);

				if (i == NrSelected) {
					HBRUSH hbrush1;

					hbrush1 = SelectObject(hdc, CreateSolidBrush(~GetSysColor(COLOR_WINDOW)));
					PatBlt(hdc, j - (SYSFONT->width / 2), 2,
							2 * SYSFONT->width, rect.bottom - 4, PATCOPY);
					DeleteObject(SelectObject(hdc, hbrush1));

                    DisplayKanjiBitmap(hdc, j, KANJILISTBORDER + SYSFONT->height,
                                        SYSFONT->width, SYSFONT->height,
                                        r, NOTSRCCOPY, cbufp);
                } else {
                    DisplayKanjiBitmap(hdc, j, KANJILISTBORDER + SYSFONT->height,
                                        SYSFONT->width, SYSFONT->height,
                                        r, SRCCOPY, cbufp);
                }
            }

            DeleteObject(SelectObject(hdc, hpen));
            DeleteObject(SelectObject(hdc, hbrush));

            //EnableFontCache(TRUE);

            EndPaint(hwnd, &ps);
            return (0);

        case WM_USER:
			SetScrollPos(hwnd, SB_HORZ, start + NrSelected, TRUE);
            return (0);

        case WM_KEYDOWN:
            switch (wParam) {
                case VK_LEFT:   SendMessage(hwnd, WM_HSCROLL, SB_LINEUP, 0L); break;
                case VK_RIGHT:  SendMessage(hwnd, WM_HSCROLL, SB_LINEDOWN, 0L); break;
                case VK_PRIOR:  SendMessage(hwnd, WM_HSCROLL, SB_PAGEUP, 0L); break;
                case VK_NEXT:   SendMessage(hwnd, WM_HSCROLL, SB_PAGEDOWN, 0L); break;
                case VK_UP:     SendMessage(hwnd, WM_HSCROLL, SB_LINEUP, 0L); break;
                case VK_DOWN:   SendMessage(hwnd, WM_HSCROLL, SB_LINEDOWN, 0L); break;
            }
            return (0);

        case WM_HSCROLL:
            switch (wParam) {
                case SB_LINEUP:
                    if (start + NrSelected <= 0) return (TRUE);
                    start--;
                    stop = start + NrInList - 1;
                    for (i = NrInList - 1; i >= 0; i--) KanjiList[i] = KanjiList[i-1];
                    if (start < 0) {
                        KanjiList[0].jis = 0;
                    } else {
                        lseek(fd, SortKey * IdxBase + start * sizeof(IDXRECORD), 0);
                        read(fd, &record, sizeof(IDXRECORD));
                        KanjiList[0] = record;
                    }
                    InvalidateRect(hwnd, NULL, FALSE);
                    break;

                case SB_PAGEUP:
                    if (start + NrSelected <= 0) return (TRUE);
                    start -= NrInList;
                    stop = start + NrInList - 1;
                    if (start + NrSelected <= 0) start = -NrSelected + 1;
                    if (start >= 0) {
                        lseek(fd, SortKey * IdxBase + start * sizeof(IDXRECORD), 0);
                    } else {
                        lseek(fd, SortKey * IdxBase, 0);
                    }
                    for (i = 0; i < NrInList; i++) {
                        if (start + i < 0 || start + i >= NrIndex) {
                            KanjiList[i].jis = 0;
                        } else {
                            read(fd, &record, sizeof(IDXRECORD));
                            KanjiList[i] = record;
                        }
                    }
                    InvalidateRect(hwnd, NULL, FALSE);
                    break;

                case SB_LINEDOWN:
                    if (start + NrSelected >= NrIndex - 1) return (TRUE);
                    start++;
                    stop = start + NrInList - 1;
                    for (i = 0; i < NrInList - 1; i++) KanjiList[i] = KanjiList[i+1];
                    if (start + NrInList - 1 >= NrIndex) {
                        KanjiList[NrInList - 1].jis = 0;
                    } else {
                        lseek(fd, SortKey * IdxBase + stop * sizeof(IDXRECORD), 0);
                        read(fd, &record, sizeof(IDXRECORD));
                        KanjiList[NrInList - 1] = record;
                    }
                    InvalidateRect(hwnd, NULL, FALSE);
                    break;

                case SB_PAGEDOWN:
                    if (start + NrSelected >= NrIndex - 1) return (TRUE);
                    start += NrInList;
                    stop = start + NrInList - 1;
                    if (start + NrSelected >= NrIndex - 1) start = NrIndex - NrSelected;
                    for (i = 0; i < NrInList; i++) {
                        if (start + i < 0 || start + i >= NrIndex) {
                            KanjiList[i].jis = 0;
                        } else {
                            read(fd, &record, sizeof(IDXRECORD));
                            KanjiList[i] = record;
                        }
                    }
                    InvalidateRect(hwnd, NULL, FALSE);
                    break;

                default: return (TRUE);
            }

            parent = GetParent(hwnd);

            for (i = 0; i < NRSORT; i++) {
				if (KanjiList[NrSelected].sortkeys[i] != CurrentKeys[i]) {
					CurrentKeys[i] = KanjiList[NrSelected].sortkeys[i];
                    SetScrollPos(GetDlgItem(parent, 4221 + i), SB_CTL, CurrentKeys[i], TRUE);
                    SetDlgItemInt(parent, 4231 + i, CurrentKeys[i], TRUE);

                    if (4231 + i == 4232) {     /* Bushu */
                        SendMessage(parent, WM_USER, 0, 0L);  /* Draw the Bushu icon */
                    }
                }
            }
            return (TRUE);
	}

	return (DefWindowProc(hwnd, message, wParam, lParam));
}



static KANJI CheckHexJISCode (HWND hwnd, char *s)
{
    KANJI ch;
    int i;

    if (strlen(s) != 4) goto BadCode;

    for (i = 0; s[i]; i++) {
        if (strchr("0123456789ABCDEF", s[i]) == NULL)
            goto BadHexNumber;
    }

    ch = 0;

    for (i = 0; i < 4 && s[i]; i++) {
        ch <<= 4;
        if ('0' <= s[i] && s[i] <= '9') ch |= (s[i] - '0');
        else ch |= (s[i] - 'A' + 10);
    }
    ch &= 0x7f7f;

    if (Jis2Index(ch, FALSE) < 0) goto BadCode;

    return (ch);


BadCode:

    ErrorMessage(hwnd, "'%s' is not a valid JIS code.  Please try again.", s);
    return (0);

BadHexNumber:

    ErrorMessage(hwnd, "'%s' is not a valid HEX number.\n\n"
                          "Remember, you must type the\n"
                          "JIS code in HEX, e.g. 3A4B", s);

    return (0);
}



BOOL FAR PASCAL JISInputProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
	switch (message) {
        case WM_INITDIALOG:
            SendDlgItemMessage(hwnd, 4201, EM_LIMITTEXT, 4, 0L);
            if (global.active == NULL)
                EnableWindow(GetDlgItem(hwnd, IDOK), FALSE);    /* Insert button */

            CenterDialogBox(hwnd);
            return (TRUE);

        case WM_COMMAND:
			switch (wParam) {
                case 4211: {    /* Info button */
					KANJI ch;
					char textbuf[5];

					GetDlgItemText(hwnd, 4201, textbuf, 5);
                    ch = CheckHexJISCode(hwnd, textbuf);

					if (!ch) {
						SetDlgItemText(hwnd, 4201, "");
                        SetFocus(GetDlgItem(hwnd, 4201));
						return (TRUE);
					}

                    SetKanjiInfoChar(ch);

                    DialogBox (hInstance, "KanjiInfo", hwnd, KanjiInfoProc);
                    return (TRUE);
				}

				case IDOK: {        /* Insert that character? */
					char textbuf[5];
					KANJI buf[2];

					if (HIWORD(lParam) != BN_CLICKED) return (TRUE);

					/* Value OK? */

					GetDlgItemText(hwnd, 4201, textbuf, 5);

                    buf[0] = CheckHexJISCode(hwnd, textbuf);
					if (!buf[0]) {
						SetDlgItemText(hwnd, 4201, "");
						SetFocus(GetDlgItem(hwnd, 4201));
						return (TRUE);
					}

                    buf[1] = 0;

                    if (curfile != NULL) {
                        TurnOffSelection(curfile);
                        UndoAddTyping(curfile, curfile->current, 1, FALSE);
                        InsertString(curfile, curfile->current, buf,
                                        OP_REFORMAT | OP_UPDATE | OP_MOVESEL | OP_MOVETOEND);
                        SendMessage(hwnd, WM_COMMAND, IDCANCEL, 0L);
                    }

                    EndDialog(hwnd, FALSE);
                    return (TRUE);
				}

                case IDCANCEL:
                    EndDialog(hwnd, FALSE);
                    return (TRUE);
            }
			break;
    }
    return (FALSE);
}



BOOL FAR PASCAL KanjiInfoProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    static int Kinfo = 0;       /* 0 = not open, 1 = open, 2 = can't find */
    static int Bushu = 0;
    static OFSTRUCT kinfoof;
    static KANJI far *OnKunPtr;


	switch (message) {

        case WM_INITDIALOG: {
            INFORECORD record;
			KANJI ch;
			FILE *fp;
            char buffer[MAXLINELEN];
            int i, j;
            int seq, fd, onkunx;
            BOOL okurigana;

            CenterDialogBox(hwnd);

            Bushu = 0;

            ch = InfoChar & 0x7f7f;

            SendDlgItemMessage(hwnd, 4201, WM_USER, ch, 0L);

            if (ch > 0) {
                sprintf(buffer, "%X (%X) [ %c%c ]", (int) ch, ((int) ch) | 0x8080,
                                    HIBYTE(ch) & 0x007f, LOBYTE(ch) & 0x007f);
            } else {
                strcpy(buffer, "Unknown");
            }
            SetDlgItemText(hwnd, 4213, buffer);

            if (!ISKANJI(ch)) return (TRUE);

			/* Now figure out the sequence number of the kanji */

            ch -= 0x3021;
            seq = HIBYTE(ch) * 94 + LOBYTE(ch);

            switch (Kinfo) {
                case 0:
                    fd = OpenFile(global.kinfo, &kinfoof, OF_READ);
                    if (fd <= 0) {
                        ErrorMessage(hwnd, "Cannot open Kanji Info file!");
                        Kinfo = 2;
                        return (TRUE);
                    }
                    Kinfo = 1;
                    break;

                case 1:
                    fd = OpenFile(NULL, &kinfoof, OF_READ | OF_REOPEN);
                    break;

                case 2:
                default:
                    return (TRUE);
            }

            if (KinfoMax < 0) {
                lseek(fd, 0L, 0);
                read(fd, &record, sizeof(INFORECORD));

                KinfoMax = record.offset / sizeof(INFORECORD);
                if (KinfoMax < 0) ErrorMessage(hwnd, "Corrupted kanji info file!");
            }

            if (seq > KinfoMax) {
                close(fd);
                return (TRUE);
            }

            lseek(fd, (long int) seq * sizeof(INFORECORD), 0);
            read(fd, &record, sizeof(INFORECORD));

            SetDlgItemInt(hwnd, 4211, record.strokes, TRUE);
            SetDlgItemInt(hwnd, 4212, record.bushu, TRUE);
            if (record.grade <= 0) {
                SetDlgItemText(hwnd, 4214, "None");
            } else {
                SetDlgItemInt(hwnd, 4214, record.grade, TRUE);
            }
            if (record.nelson <= 0) {
                SetDlgItemText(hwnd, 4215, "None");
            } else {
                SetDlgItemInt(hwnd, 4215, record.nelson, TRUE);
            }

            Bushu = record.bushu;

            SetWindowPos(GetDlgItem(hwnd, 4231), NULL, 0, 0,
                            SYSFONT->width, SYSFONT->height,
                            SWP_NOMOVE | SWP_NOZORDER);

            /* Now read the On/Kun readings and meanings */

            OnKunPtr = MemAlloc(MAXLINELEN * sizeof(KANJI));
            if (OnKunPtr == NULL) return (TRUE);

            onkunx = 0;

            fp = fdopen(fd, "rb");
            if (fp == NULL) {
				FreeMem((void *) OnKunPtr);
				close(fd);
				goto NoInfo;
            }

            fseek(fp, record.offset, 0);

            SendDlgItemMessage(hwnd, 4221, WM_SETREDRAW, FALSE, 0L);
            SendDlgItemMessage(hwnd, 4222, WM_SETREDRAW, FALSE, 0L);
            SendDlgItemMessage(hwnd, 4223, WM_SETREDRAW, FALSE, 0L);

            for (i = 0; i < record.on; i++) {
                fgets(buffer, MAXLINELEN, fp);
                for (j = strlen(buffer); j >= 0 && buffer[j] <= ' '; j--);
                buffer[j+1] = '\0';

                for (j = 0; buffer[j]; j++) OnKunPtr[onkunx+j] = 0x2400 | (buffer[j] & 0x007f);
                OnKunPtr[onkunx+j] = 0;
                SendDlgItemMessage(hwnd, 4221, LB_ADDSTRING, 0, (LONG) (OnKunPtr + onkunx));
                onkunx += (j + 1);
            }

            for (i = 0; i < record.kun; i++) {
                fgets(buffer, MAXLINELEN, fp);
                for (j = strlen(buffer); j >= 0 && buffer[j] <= ' '; j--);
                buffer[j+1] = '\0';

                okurigana = FALSE;

                for (j = 0; buffer[j]; j++) {
                    if (buffer[j] == OKURIGANA) {
                        OnKunPtr[onkunx+j] = 0x214a;    /* ( */
                        okurigana = TRUE;
                    } else {
                        OnKunPtr[onkunx+j] = 0x2400 | (buffer[j] & 0x007f);
                    }
                }
                if (okurigana) {
                    OnKunPtr[onkunx+j] = 0x214b;    /* ) */
                    j++;
                }
                OnKunPtr[onkunx+j] = 0;
                SendDlgItemMessage(hwnd, 4222, LB_ADDSTRING, 0, (LONG) (OnKunPtr + onkunx));
                onkunx += (j + 1);
			}

            for (i = 0; i < record.meanings; i++) {
                fgets(buffer, MAXLINELEN, fp);
                for (j = strlen(buffer); j >= 0 && buffer[j] <= ' '; j--);
                buffer[j+1] = '\0';

                SendDlgItemMessage(hwnd, 4223, LB_ADDSTRING, 0, (LONG) buffer);
            }

			fclose(fp);

            SendDlgItemMessage(hwnd, 4221, WM_SETREDRAW, TRUE, 0L);
            SendDlgItemMessage(hwnd, 4222, WM_SETREDRAW, TRUE, 0L);
            SendDlgItemMessage(hwnd, 4223, WM_SETREDRAW, TRUE, 0L);
            InvalidateRect(GetDlgItem(hwnd, 4221), NULL, TRUE);
            InvalidateRect(GetDlgItem(hwnd, 4222), NULL, TRUE);
            InvalidateRect(GetDlgItem(hwnd, 4223), NULL, TRUE);

            return (TRUE);

NoInfo:     ErrorMessage(hwnd, "You must select (highlight) a KANJI");
            EndDialog(hwnd, FALSE);
            return (TRUE);
        }

        case WM_PAINT: {
            int r;
            HDC hdc;
            BYTE far *cbufp;

            if (Bushu <= 0 || Bushu > 214) break;

            hdc = GetDC(GetDlgItem(hwnd, 4231));
            SetTextColor(hdc, GetSysColor(COLOR_WINDOWTEXT));
			SetBkColor(hdc, GetSysColor(COLOR_WINDOW));

#ifdef CTL3D
			if (Dialogs3D) {
				RECT rect;
				HBRUSH hbrush;

				hbrush = Ctl3dCtlColorEx(WM_CTLCOLOR, hdc, MAKELONG(hwnd, CTLCOLOR_DLG));
                GetClientRect(GetDlgItem(hwnd, 4241), &rect);
				FillRect(hdc, &rect, hbrush);
            } else
#endif CTL3D
            {
				HBRUSH hbrush;
				RECT rect;

                hbrush = CreateSolidBrush(GetSysColor(COLOR_WINDOW));
				GetClientRect(hwnd, &rect);
                FillRect(hdc, &rect, hbrush);
                DeleteObject(hbrush);
            }

            r = Jis2Index(BushuSymbols[Bushu - 1], SYSFONT->holes);
            if (r >= 0) {
                r = GetKanjiBitmap(SYSFONT, r, &cbufp);
                DisplayKanjiBitmap(hdc, 0, SYSFONT->height,
                                    SYSFONT->width, SYSFONT->height,
                                    r, SRCAND, cbufp);
            }
            ReleaseDC(GetDlgItem(hwnd, 4231), hdc);
            break;
        }

        case WM_COMMAND:
            switch (wParam) {
                case IDOK:
                case IDCANCEL:
                    EndDialog(hwnd, FALSE);
                    return (TRUE);

                default: break;
            }
            break;

        case WM_DESTROY:
			FreeMem((void *) OnKunPtr);
            return (TRUE);
        
        case WM_COMPAREITEM:
        case WM_DELETEITEM:
        case WM_DRAWITEM:
        case WM_MEASUREITEM:
            return (JlistProc(hwnd, message, wParam, lParam, TRUE, (LONG) NULL));
    }

    return (FALSE);
}



LONG FAR PASCAL BigKanjiProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    static OFSTRUCT of;
    static FONTHEADER fh;
    static int LargeFont = 0;   /* 0 = Not yet open, 1 = opened, 2 = not available */

	switch (message) {

        case WM_CREATE:
            SetWindowWord(hwnd, 0, 0);
            return (0);

        case WM_PAINT: {
            int i, j, fd;
            int width, height;
            LONG offset;
            LONG area, LargestArea;
            KANJI index;
			BYTE far *cbufp, buffer[BUFSIZE], tempbuf[BUFSIZE];
            HDC hdc, hdcmem;
            HANDLE hbitmap;
            HBRUSH hbrush;
            HPEN hpen;
			RECT rect;
            PAINTSTRUCT ps;


            hdc = BeginPaint(hwnd, &ps);

            SetTextColor(hdc, GetSysColor(COLOR_WINDOWTEXT));
            SetBkColor(hdc, GetSysColor(COLOR_WINDOW));
            hbrush = SelectObject(hdc, CreateSolidBrush(GetSysColor(COLOR_WINDOW)));
            hpen = SelectObject(hdc, CreatePen(PS_SOLID, 1, GetSysColor(COLOR_WINDOWFRAME)));

            GetClientRect(hwnd, &rect);
            Rectangle(hdc, 0, 0, rect.right, rect.bottom);

            DeleteObject(SelectObject(hdc, CreatePen(PS_SOLID, 1, GetSysColor(COLOR_BTNSHADOW))));
            Rectangle(hdc, 1, 1, rect.right - 1, rect.bottom - 1);

            DeleteObject(SelectObject(hdc, hpen));
            DeleteObject(SelectObject(hdc, hbrush));

            index = GetWindowWord(hwnd, 0);

            if (!ISKANJI(index) || index <= 0) {
                EndPaint(hwnd, &ps);
                return (0);
            }

			if (LargeFont == 0) {
                /* Find out which font is the largest */

                for (i = 0; i < global.nr_fonts; i++) {
                    if (i == 0) {
                        j = 0;
                        LargestArea = (long int) InstalledFonts[0].width * (long int) InstalledFonts[0].height;
                        continue;
                    }
                    area = (long int) InstalledFonts[i].width * (long int) InstalledFonts[i].height;
                    if (area > LargestArea) {
                        j = i;
                        LargestArea = area;
                    }
                }

                /* Try to open the font */

                fd = OpenFile(InstalledFonts[j].filename, &of, OF_READ);
                if (fd <= 0) {
                    ErrorMessage(hwnd, "Cannot open font file %s.  Using system font.",
                                    InstalledFonts[j].filename);

                    LargeFont = 2;
                    goto UseSystem;
                }

                lseek(fd, 0L, 0);
                read(fd, &fh, sizeof(FONTHEADER));
                LargeFont = 1;
				goto UseLarge;
            } else if (LargeFont == 1) {
				fd = OpenFile(NULL, &of, OF_READ | OF_REOPEN);

UseLarge:       width = fh.width;
                height = fh.height;

                offset = Jis2Index(index, fh.holes);
                if (offset >= 0) {
                    offset *= fh.charsize;
                    offset += fh.offset;

                    lseek(fd, offset, 0);
                    read(fd, tempbuf, fh.charsize);
                    AllignKanjiBitmap(buffer, tempbuf, fh.charsize, height);
                }
                close(fd);
            } else {
UseSystem:      i = Jis2Index(index, SYSFONT->holes);
                if (i >= 0) {
                    EnableFontCache(FALSE);
                    i = GetKanjiBitmap (SYSFONT, i, &cbufp);
                    EnableFontCache(TRUE);
                    width = SYSFONT->width;
                    height = SYSFONT->height;
                    _fmemcpy(buffer, cbufp, i);
                }
            }

			hbitmap = CreateBitmap(width, height, 1, 1, buffer);
            hdcmem = CreateCompatibleDC(hdc);
            SelectObject(hdcmem, hbitmap);
            StretchBlt(hdc, KANJIBOXBORDER, KANJIBOXBORDER,
                        rect.right - 2 * KANJIBOXBORDER,
                        rect.bottom - 2 * KANJIBOXBORDER,
                        hdcmem, 0, 0, width, height, SRCCOPY);
            DeleteDC(hdcmem);
			DeleteObject(hbitmap);
            EndPaint(hwnd, &ps);
            return (0);
        }

        case WM_USER:
            wParam &= 0x7f7f;

            if (GetWindowWord(hwnd, 0) != wParam) {
                SetWindowWord(hwnd, 0, wParam);
                InvalidateRect(hwnd, NULL, TRUE);
                UpdateWindow(hwnd);
            }
            return (0);
    }

	return (DefWindowProc(hwnd, message, wParam, lParam));
}



BOOL FAR PASCAL JISTableDlgProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
	char tempbuf[5];

    switch (message) {
        case WM_INITDIALOG:
            PageNum = 0;
            SelRow = SelCol = -1;
			SetDlgItemInt(hwnd, 4221, 1, TRUE);
			SetDlgItemText(hwnd, 4222, "21");
            EnableWindow(GetDlgItem(hwnd, 4212), FALSE);
            EnableWindow(GetDlgItem(hwnd, 4213), FALSE);
            SetScrollRange(GetDlgItem(hwnd, 4201), SB_VERT, 0, 0x7e - 0x21 - 1, FALSE);
            SetScrollPos(GetDlgItem(hwnd, 4201), SB_VERT, 0, TRUE);

            if (global.active == NULL)
                EnableWindow(GetDlgItem(hwnd, IDOK), FALSE);    /* Insert button */

            CenterDialogBox(hwnd);

            /* RowPos and ColPos should have been set by JISTableProc   */
            /* already because child windows are sent WM_CREATE's first */

            if (RowPos < 2 || ColPos < 2) {
                ErrorMessage(hwnd, "Your monitor currently does not have enough resolution to\n"
                                      "display the JIS Table.  You must either use a higher resolution\n"
                                      "video-mode, or select a lower-resolution system font.");
				EndDialog(hwnd, FALSE);
			}

			return (TRUE);

		case WM_KEYDOWN:
			switch (wParam) {
				case VK_NEXT:
                    if (PageNum + 0x21 >= 0x7e) return (TRUE);
					if (PageNum <= 0)
						EnableWindow(GetDlgItem(hwnd, 4212), TRUE);
					PageNum++;
					if (PageNum + 0x21 >= 0x7e)
						EnableWindow(GetDlgItem(hwnd, 4211), FALSE);
					break;

				case VK_PRIOR:
                    if (PageNum <= 0) return (TRUE);
					if (PageNum + 0x21 >= 0x7e)
						EnableWindow(GetDlgItem(hwnd, 4211), TRUE);
					PageNum--;
					if (PageNum <= 0)
						EnableWindow(GetDlgItem(hwnd, 4212), FALSE);
					break;

				case VK_END:
					if (PageNum <= 0)
						EnableWindow(GetDlgItem(hwnd, 4212), TRUE);
					PageNum = 0x7e - 0x21 - 1;
					EnableWindow(GetDlgItem(hwnd, 4211), FALSE);
					break;

				case VK_HOME:
                    if (PageNum + 0x21 >= 0x7e)
                        EnableWindow(GetDlgItem(hwnd, 4211), TRUE);
                    PageNum = 0;
                    EnableWindow(GetDlgItem(hwnd, 4212), FALSE);
                    break;

                case VK_SELECT:
                    PageNum = LOWORD(lParam);
                    EnableWindow(GetDlgItem(hwnd, 4211), PageNum + 0x21 < 0x7e);
                    EnableWindow(GetDlgItem(hwnd, 4212), PageNum > 0);
                    break;

                default: return (0);
            }

            SelRow = SelCol = -1;
            EnableWindow(GetDlgItem(hwnd, 4213), FALSE);
            SetDlgItemInt(hwnd, 4221, PageNum + 1, TRUE);
            SetScrollPos(GetDlgItem(hwnd, 4201), SB_VERT, PageNum, TRUE);
			sprintf(tempbuf, "%X", PageNum + 0x21);
            SetDlgItemText(hwnd, 4222, tempbuf);
            InvalidateRect(GetDlgItem(hwnd, 4201), NULL, FALSE);
            return (FALSE);

        case WM_COMMAND:
			switch (wParam) {
                case 4211:          /* Next button */
					SendDlgItemMessage(hwnd, 4201, WM_KEYDOWN, VK_NEXT, 0L);
                    return (TRUE);

                case 4212:          /* Prev button */
                    SendDlgItemMessage(hwnd, 4201, WM_KEYDOWN, VK_PRIOR, 0L);
                    return (TRUE);

                case 4213: {        /* Info button */
                    KANJI ch;

                    if (SelRow < 0 || SelCol < 0) {
                        MessageBeep(0);
                        return (TRUE);
                    }

                    ch = ((SelRow + 2) << 4) | SelCol;
                    ch |= ((PageNum + 0x21) << 8);

                    SetKanjiInfoChar(ch);

                    DialogBox (hInstance, "KanjiInfo", hwnd, KanjiInfoProc);
                    return (TRUE);
                }

                case IDOK: {        /* Insert button */
                    KANJI buf[2];

                    if (curfile != NULL && SelRow >= 0 && SelCol >= 0) {
                        buf[0] = ((SelRow + 2) << 4) | SelCol;
                        buf[0] |= ((PageNum + 0x21) << 8);
                        buf[1] = 0;

                        TurnOffSelection(curfile);
                        UndoAddTyping(curfile, curfile->current, 1, FALSE);
                        InsertString(curfile, curfile->current, buf,
                                OP_REFORMAT | OP_UPDATE | OP_MOVESEL | OP_MOVETOEND);
                    }
					EndDialog(hwnd, FALSE);
                    return (TRUE);
                }

                case IDCANCEL:
					EndDialog(hwnd, FALSE);
                    return (TRUE);
            }
            return (FALSE);
    }

    return (FALSE);
}



static BOOL FindNextChar (int page, int *row, int *col, BOOL hori, int step)
{
    int x, y, r;
    KANJI index;

    y = *row;
    x = *col;

    for (; ; ) {
        if (hori) {
            x += step;

            if (x >= 16) {
                x = 0; y++;
                if (y >= NRJISROWS) return (FALSE);
            }
            if (x < 0) {
                x = 15; y--;
                if (y < 0) return (FALSE);
            }
        } else {
            y += step;

            if (y >= NRJISROWS) {
                y = 0; x++;
                if (x >= 16) return (FALSE);
            }
            if (y < 0) {
                y = NRJISROWS - 1; x--;
                if (x < 0) return (FALSE);
            }
        }

        index = ((y + 2) << 4) | x;
        index |= ((page + 0x21) << 8);

        r = Jis2Index(index, SYSFONT->holes);

        if (r >= 0) {
            *row = y;
            *col = x;
            return (TRUE);
        }
    }
}



LONG FAR PASCAL JISTableProc (HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
	int i, j, r;
    KANJI index;
	HDC hdc;
    PAINTSTRUCT ps;
    RECT rect;
    HBRUSH hbrush;
    HPEN hpen;
	BYTE far *cbufp;
    static int capturing;

	switch (message) {
		case WM_CREATE:
            SetScrollRange(GetDlgItem(hwnd, 4201), SB_VERT, 0, 0x7e - 0x21 - 1, TRUE);
            SetScrollPos(GetDlgItem(hwnd, 4201), SB_VERT, 0, TRUE);

			GetClientRect(hwnd, &rect);
			RowDiv = rect.bottom / NRJISROWS;
			ColDiv = rect.right / 16;

            i = RowDiv * NRJISROWS;
            j = ColDiv * 16 + GetSystemMetrics(SM_CXVSCROLL);

            SetWindowPos(hwnd, NULL, 0, 0, j + 1, i + 1, SWP_NOMOVE | SWP_NOZORDER);

            RowPos = (RowDiv - SYSFONT->height) / 2;
            ColPos = (ColDiv - SYSFONT->width) / 2;

            capturing = FALSE;

            return (0);

        case WM_GETDLGCODE:
            return (DLGC_WANTARROWS);

        case WM_VSCROLL:
            switch (wParam) {
                case SB_TOP:
                    SendMessage(GetParent(hwnd), WM_KEYDOWN, VK_HOME, 0L);
                    break;

                case SB_BOTTOM:
                    SendMessage(GetParent(hwnd), WM_KEYDOWN, VK_END, 0L);
                    break;

                case SB_PAGEUP:
                case SB_LINEUP:
                    SendMessage(GetParent(hwnd), WM_KEYDOWN, VK_PRIOR, 0L);
                    break;

                case SB_PAGEDOWN:
                case SB_LINEDOWN:
                    SendMessage(GetParent(hwnd), WM_KEYDOWN, VK_NEXT, 0L);
                    break;

                case SB_THUMBPOSITION:
                    SendMessage(GetParent(hwnd), WM_KEYDOWN, VK_SELECT, LOWORD(lParam));
                    break;

                case SB_THUMBTRACK: {
                    char tempbuf[10];

                    i = LOWORD(lParam);
                    SetDlgItemInt(GetParent(hwnd), 4221, i + 1, TRUE);
                    sprintf(tempbuf, "%X", i + 0x21);
                    SetDlgItemText(GetParent(hwnd), 4222, tempbuf);
                    break;
                }

            }
            return (0);

        case WM_KEYDOWN:
            switch (wParam) {
                case VK_NEXT:
				case VK_PRIOR:
                case VK_HOME:
                case VK_END:
                case VK_SELECT:
                    SendMessage(GetParent(hwnd), WM_KEYDOWN, wParam, lParam);
                    return (0);

                case VK_UP:
                case VK_DOWN:
                case VK_LEFT:
                case VK_RIGHT: {
                    BOOL hori;
                    int step;

                    if (wParam == VK_UP || wParam == VK_DOWN) hori = FALSE;
                    else hori = TRUE;

                    if (wParam == VK_UP || wParam == VK_LEFT) step = -1;
                    else step = +1;

                    i = SelCol;
                    j = SelRow;

                    if (i < 0 || j < 0) {
                        j = 0; i = -1;
                        hori = TRUE;
                        step = +1;
                    }

                    if (!FindNextChar(PageNum, &j, &i, hori, step)) {
                        MessageBeep(0);
                        return (0);
                    }

                    SendMessage(hwnd, WM_LBUTTONDOWN, 0, MAKELONG((i * ColDiv), (j * RowDiv)));
                    return (0);
                }

                case VK_ESCAPE:
                    SendMessage(GetParent(hwnd), WM_COMMAND, IDCANCEL, lParam);
                    return (0);
            }
            break;

        case WM_LBUTTONDBLCLK:
            if (SelRow >= 0 && SelCol >= 0) {
                SendMessage(GetParent(hwnd), WM_COMMAND, IDOK, 0L);
            } else {
                MessageBeep(0);
            }
            return (0);

        case WM_LBUTTONDOWN:
            i = LOWORD(lParam);
            j = HIWORD(lParam);

            GetClientRect(hwnd, &rect);

            i /= ColDiv;
            j /= RowDiv;

            if (j < 0 || j >= NRJISROWS || i < 0 || i >= 16) return (0);

            index = ((j + 2) << 4) | i;
            index |= ((PageNum + 0x21) << 8);

            r = Jis2Index(index, SYSFONT->holes);

            if (r < 0) {
                MessageBeep(0);
                return (0);
            }

            hdc = GetDC(hwnd);

            SetTextColor(hdc, GetSysColor(COLOR_WINDOWTEXT));
            SetBkColor(hdc, GetSysColor(COLOR_WINDOW));

            if (SelRow >= 0 && SelCol >= 0) {
                if (i != SelCol || j != SelRow) {
                    PatBlt(hdc, SelCol * ColDiv + 1, SelRow * RowDiv + 1,
                           ColDiv - 1, RowDiv - 1, DSTINVERT);
                    SelCol = i; SelRow = j;
                } else {
                    SelRow = SelCol = -1;
                }
            } else {
                SelCol = i; SelRow = j;
            }

            PatBlt(hdc, i * ColDiv + 1, j * RowDiv + 1, ColDiv - 1, RowDiv - 1, DSTINVERT);
            ReleaseDC(hwnd, hdc);

            if (SelRow >= 0 && SelCol >= 0) {
                index = ((SelRow + 2) << 4) | SelCol;
                index |= ((PageNum + 0x21) << 8);

                if (ISKANJI(index)) {
                    EnableWindow(GetDlgItem(GetParent(hwnd), 4213), TRUE);
                } else {
                    EnableWindow(GetDlgItem(GetParent(hwnd), 4213), FALSE);
                }
            } else {
                EnableWindow(GetDlgItem(GetParent(hwnd), 4213), FALSE);
            }

            return (0);

        case WM_PAINT:
            hdc = BeginPaint(hwnd, &ps);

            SetTextColor(hdc, GetSysColor(COLOR_WINDOWTEXT));
            SetBkColor(hdc, GetSysColor(COLOR_WINDOW));
            hbrush = SelectObject(hdc, CreateSolidBrush(GetSysColor(COLOR_WINDOW)));
            hpen = SelectObject(hdc, CreatePen(PS_SOLID, 1, GetSysColor(COLOR_WINDOWFRAME)));

            GetClientRect(hwnd, &rect);
            Rectangle(hdc, 0, 0, rect.right, rect.bottom);

            DeleteObject(SelectObject(hdc, CreatePen(PS_SOLID, 1, GetSysColor(COLOR_BTNSHADOW))));

            for (i = 1; i < NRJISROWS + 1; i++) {
                MoveTo(hdc, 1, i * RowDiv - 1);
                LineTo(hdc, rect.right - 1, i * RowDiv - 1);
			}

            for (i = 1; i <= 16; i++) {
                MoveTo(hdc, i * ColDiv - 1, 1);
                LineTo(hdc, i * ColDiv - 1, rect.bottom - 1);
            }

            DeleteObject(SelectObject(hdc, CreatePen(PS_SOLID, 1, GetSysColor(COLOR_WINDOWFRAME))));

            for (i = 1; i < NRJISROWS; i++) {
				MoveTo(hdc, 0, i * RowDiv);
				LineTo(hdc, rect.right, i * RowDiv);
			}

            for (i = 1; i < 16; i++) {
                MoveTo(hdc, i * ColDiv, 0);
                LineTo(hdc, i * ColDiv, rect.bottom);
            }

            EnableFontCache(FALSE);

            for (i = 0; i < NRJISROWS; i++) {
				for (j = 0; j < 16; j++) {
                    index = ((i + 2) << 4) | j;
                    index |= ((PageNum + 0x21) << 8);

                    r = Jis2Index(index, SYSFONT->holes);

                    if (r < 0) {
                        rect.left = j * ColDiv + 1;
                        rect.top = i * RowDiv + 1;
                        rect.right = j * ColDiv + ColDiv + 1;
                        rect.bottom = i * RowDiv + RowDiv + 1;
                        FillRect(hdc, &rect, GetStockObject(GRAY_BRUSH));
                        continue;
                    }

                    r = GetKanjiBitmap(SYSFONT, r, &cbufp);
					DisplayKanjiBitmap(hdc, j * ColDiv + ColPos,
                                        i * RowDiv + SYSFONT->height + RowPos,
                                        SYSFONT->width, SYSFONT->height,
                                        r, SRCCOPY, cbufp);

                    if (i == SelRow && j == SelCol) {
                        PatBlt(hdc, j * ColDiv + 1, i * RowDiv + 1,
                                ColDiv - 1, RowDiv - 1, DSTINVERT);
                    }
                }
            }

            EnableFontCache(TRUE);

            DeleteObject(SelectObject(hdc, hbrush));
            DeleteObject(SelectObject(hdc, hpen));

            EndPaint(hwnd, &ps);
            return (0);
    }

    return (DefWindowProc(hwnd, message, wParam, lParam));
}
