/***********************************************************************/
/* UTIL.C - Utility routines                                           */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1996 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                 Email:             M.Hessling@qut.edu.au
 * PO Box 203                    Phone:                    +617 3849 7731
 * Bellara                       http://www.gu.edu.au/gwis/the/markh.html
 * QLD 4507                      **** Maintainer PDCurses & REXX/SQL ****
 * Australia                     ************* Author of THE ************
 */

/*
$Id: util.c 2.1 1995/06/24 16:31:38 MH Rel MH $
*/

#include <the.h>
#include <proto.h>

/*--------------------------- common data -----------------------------*/
 static CHARTYPE *recv[MAX_RECV];
 static LENGTHTYPE recv_len[MAX_RECV];
 static short add_recv=(-1);
 static short retr_recv=(-1);
 static short num_recv=0;

#ifdef USE_EXTCURSES
 chtype color_pair[COLOR_PAIRS];
 static chtype fore_color[8];
 static chtype back_color[8];
#endif

/*man***************************************************************************
NAME
     memreveq - search buffer reversed for character

SYNOPSIS
     short memreveq(buffer,chr,max_length)
     CHARTYPE *buffer;
     CHARTYPE ch;
     short max_length;

DESCRIPTION
     The memreveq function searches the buffer from the right for the
     first character equal to the supplied character.

RETURN VALUE
     If successful, returns the position of first matching character
     or (-1) if unsuccessful.

SEE ALSO
     strzreveq, memrevne
*******************************************************************************/
#ifdef HAVE_PROTO
short memreveq(CHARTYPE *buffer,CHARTYPE ch,short max_len)
#else
short memreveq(buffer,ch,max_len)
CHARTYPE *buffer,ch;
short max_len;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short len=max_len;
/*--------------------------- processing ------------------------------*/
 for (--len; len>=0 && buffer[len]!=ch; len--);
 return(len);
}
/*man***************************************************************************
NAME
     memrevne - search buffer reversed for NOT character

SYNOPSIS
     #include "the.h"

     short memrevne(buffer,known_char,max_len)
     CHARTYPE *buffer;
     CHARTYPE known_char;
     short max_len;

DESCRIPTION
     The memrevne function searches the buffer from the right for first
     character NOT equal to the supplied character.

RETURN VALUE
     If successful, returns the position of first NON-matching character
     or (-1) if unsuccessful.

SEE ALSO
     strzrevne, strzne
*******************************************************************************/
#ifdef HAVE_PROTO
short memrevne(CHARTYPE *buffer,CHARTYPE known_char,short max_len)
#else
short memrevne(buffer,known_char,max_len)
CHARTYPE *buffer;
CHARTYPE known_char;
short max_len;
#endif
{
/*--------------------------- local data ------------------------------*/
 register short len=max_len;
/*--------------------------- processing ------------------------------*/
 for (--len; len>=0 && buffer[len]==known_char; len--);
 return(len);
}
/*man***************************************************************************
NAME
     meminschr - insert character into buffer

SYNOPSIS
     #include "the.h"

     CHARTYPE *meminschr(buffer,chr,location,max_length,curr_length)
     CHARTYPE *buffer;
     CHARTYPE chr;
     short location,max_length,curr_length;

DESCRIPTION
     The meminschr inserts the supplied 'chr' into the buffer 'buffer'
     before the 'location' specified. 'location' is an offset (0 based)
     from the start of 'buffer'.
     The 'buffer' will not be allowed to have more than 'max_length'
     characters, so if the insertion of the character causes the
     'max_length' to be exceeded, the last character of 'buffer' will
     be lost.

RETURN VALUE
     A pointer to the same 'buffer' as was supplied.

SEE ALSO
    meminsstr, memdelchr

*******************************************************************************/
#ifdef HAVE_PROTO
CHARTYPE *meminschr(CHARTYPE *buffer,CHARTYPE chr,short location,
                short max_length,short curr_length)
#else
CHARTYPE *meminschr(buffer,chr,location,max_length,curr_length)
CHARTYPE *buffer;
CHARTYPE chr;
short location,max_length,curr_length;
#endif
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
/*--------------------------- processing ------------------------------*/
 for (i=curr_length;i > location;i--)
     if (i < max_length)
       buffer[i] = buffer[i-1];
 if (location < max_length)
    buffer[location] = chr;
 return(buffer);
}
/*man***************************************************************************
NAME
     meminsmem - insert memory into buffer

SYNOPSIS
     #include "the.h"

     CHARTYPE *meminsmem(buffer,str,len,location,max_length,curr_length)
     CHARTYPE *buffer;
     CHARTYPE *str;
     short len,location,max_length,curr_length;

DESCRIPTION
     The meminsmem function inserts the supplied 'str' into the buffer 'buffer'
     before the 'location' specified. 'location' is an offset (0 based)
     from the start of 'buffer'.
     The 'buffer' will not be allowed to have more than 'max_length'
     characters, so if the insertion of the string causes the
     'max_length' to be exceeded, the last character(s) of 'buffer' will
     be lost.

RETURN VALUE
     A pointer to the same 'buffer' as was supplied.

SEE ALSO
    meminschr

*******************************************************************************/
#ifdef HAVE_PROTO
CHARTYPE *meminsmem(CHARTYPE *buffer,CHARTYPE *str,short len,short location,
                short max_length,short curr_length)
#else
CHARTYPE *meminsmem(buffer,str,len,location,max_length,curr_length)
CHARTYPE *buffer,*str;
short len,location,max_length,curr_length;
#endif
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
/*--------------------------- processing ------------------------------*/
 for (i=curr_length;i > location;i--)
     if (i+len-1 < max_length)
       buffer[i+len-1] = buffer[i-1];
 for (i=0;i<len;i++)
     if (location+i < max_length)
       buffer[location+i] = str[i];
 return(buffer);
}
/*man***************************************************************************
NAME
     memdelchr - delete character(s) from buffer

SYNOPSIS
     #include "the.h"

     CHARTYPE *memdelchr(buffer,location,curr_length,num_chars)
     CHARTYPE *buffer;
     short location,curr_length,num_chars;

DESCRIPTION
     The memdelchr deletes the supplied number of characters from the
     buffer starting at the 'location' specified. 'location' is an offset (0 based)
     from the start of 'buffer'.
     For each character deleted, what was the last character in buffer;
     based on 'curr_length' will be replaced with a space.

RETURN VALUE
     A pointer to the same 'buffer' as was supplied.

SEE ALSO
    meminschr

*******************************************************************************/
#ifdef HAVE_PROTO
CHARTYPE *memdelchr(CHARTYPE *buffer,short location,short curr_length,short num_chars)
#else
CHARTYPE *memdelchr(buffer,location,curr_length,num_chars)
CHARTYPE *buffer;
short location,curr_length,num_chars;
#endif
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
/*--------------------------- processing ------------------------------*/
 for (i=location;i <curr_length;i++)
     if (i+num_chars >= curr_length)
        buffer[i] = ' ';
      else
        buffer[i] = buffer[i+num_chars];
 return(buffer);
}
/*man***************************************************************************
NAME
     strzne - search string for NOT character

SYNOPSIS
     #include "the.h"

     short strzne(str,chr)
     CHARTYPE *str;
     CHARTYPE ch;

DESCRIPTION
     The strzne function searches the string from the left for the first
     character NOT equal to the supplied character.

RETURN VALUE
     If successful, returns the position of first NON-matching character
     or (-1) if unsuccessful.

SEE ALSO
     strzrevne, memrevne
*******************************************************************************/
#ifdef HAVE_PROTO
short strzne(CHARTYPE *str,CHARTYPE ch)
#else
short strzne(str,ch)
CHARTYPE *str;
CHARTYPE ch;
#endif
{
/*--------------------------- local data ------------------------------*/
 register short len=0;
 register short  i = 0;
/*--------------------------- processing ------------------------------*/
 len = strlen((DEFCHAR *)str);
 for (; i<len && str[i]==ch; i++);
 if (i>=len)
    i = (-1);
 return(i);
}
/*man***************************************************************************
NAME
     my_strdup - equivalent to strdup

SYNOPSIS
     CHARTYPE *my_strdup(str)
     CHARTYPE *str;

DESCRIPTION
     The my_strdup function duplicates the supplied string.

RETURN VALUE
     If successful, returns a pointer to the copy of the supplied string
     or NULL if unsuccessful.
*******************************************************************************/
#ifdef HAVE_PROTO
CHARTYPE *my_strdup(CHARTYPE *str)
#else
CHARTYPE *my_strdup(str)
CHARTYPE *str;
#endif
{
/*--------------------------- local data ------------------------------*/
 register short len=0;
 CHARTYPE *tmp=NULL;
/*--------------------------- processing ------------------------------*/
 len = strlen((DEFCHAR *)str);
 if ((tmp = (CHARTYPE *)(*the_malloc)((len+1)*sizeof(CHARTYPE))) == (CHARTYPE *)NULL)
    return((CHARTYPE *)NULL);
 strcpy((DEFCHAR *)tmp,(DEFCHAR *)str);
 return(tmp);
}
/*man***************************************************************************
NAME
     memne - search buffer for NOT character

SYNOPSIS
     #include "the.h"

     short memne(buffer,chr,length)
     CHARTYPE *buffer;
     CHARTYPE chr;
     short length;

DESCRIPTION
     The memne function searches the buffer from the left for the first
     character NOT equal to the supplied character.

RETURN VALUE
     If successful, returns the position of first NON-matching character
     or (-1) if unsuccessful.

SEE ALSO
     strzrevne, memrevne, strzne
*******************************************************************************/
#ifdef HAVE_PROTO
short memne(CHARTYPE *buffer,CHARTYPE chr,short length)
#else
short memne(buffer,chr,length)
CHARTYPE *buffer;
CHARTYPE chr;
short length;
#endif
{
/*--------------------------- local data ------------------------------*/
 register short  i = 0;
/*--------------------------- processing ------------------------------*/
 for (; i<length && buffer[i]==chr; i++);
 if (i>=length)
    i = (-1);
 return(i);
}
/*man***************************************************************************
NAME
     strzrevne - search string reversed for NOT character

SYNOPSIS
     #include "the.h"

     short strzrevne(str,chr)
     CHARTYPE *str;
     CHARTYPE ch;

DESCRIPTION
     The strzrevne function searches the string from the right for the
     first character NOT equal to the supplied character.

RETURN VALUE
     If successful, returns the position of first NON-matching character
     or (-1) if unsuccessful.

SEE ALSO
     strzne, memrevne
*******************************************************************************/
#ifdef HAVE_PROTO
short strzrevne(CHARTYPE *str,CHARTYPE ch)
#else
short strzrevne(str,ch)
CHARTYPE *str;
CHARTYPE ch;
#endif
{
/*--------------------------- local data ------------------------------*/
 register short len=0;
/*--------------------------- processing ------------------------------*/
 len = strlen((DEFCHAR *)str);
 for (--len; len>=0 && str[len]==ch; len--);
 return(len);
}
/*man***************************************************************************
NAME
     strzreveq - search string reversed for character

SYNOPSIS
     short strzreveq(str,chr)
     CHARTYPE *str;
     CHARTYPE ch;

DESCRIPTION
     The strzreveq function searches the string from the right for the
     first character equal to the supplied character.

RETURN VALUE
     If successful, returns the position of first matching character
     or (-1) if unsuccessful.

SEE ALSO
     strzrevne
*******************************************************************************/
#ifdef HAVE_PROTO
short strzreveq(CHARTYPE *str,CHARTYPE ch)
#else
short strzreveq(str,ch)
CHARTYPE *str,ch;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short len=0;
/*--------------------------- processing ------------------------------*/
 len = strlen((DEFCHAR *)str);
 for (--len; len>=0 && str[len]!=ch; len--);
 return(len);
}
/*man***************************************************************************
NAME
     strtrunc - truncate leading and trailing spaces from string

SYNOPSIS
     #include "the.h"

     CHARTYPE *strtrunc(string)
     CHARTYPE *string;

DESCRIPTION
     The strtrunc function truncates all leading and trailing spaces 
     from the supplied string.

RETURN VALUE
     A pointer to the original string, now truncated.

SEE ALSO

*******************************************************************************/
#ifdef HAVE_PROTO
CHARTYPE *strtrunc(CHARTYPE *string)
#else
CHARTYPE *strtrunc(string)
CHARTYPE *string;
#endif
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
 return(MyStrip(string,STRIP_BOTH,' '));
}
/*man***************************************************************************
NAME
     MyStrip - truncate leading and/or trailing spaces from string

SYNOPSIS
     #include "the.h"

     CHARTYPE *MyStrip(string,option,ch)
     CHARTYPE *string;
     char option;
     char ch;

DESCRIPTION
     The MyStrip function truncates all leading and/or trailing ch
     from the supplied string.
     The value of the "option" argument can be one of:
        STRIP_LEADING
        STRIP_TRAILING
        STRIP_BOTH
        STRIP_NONE
     These are defined elsewhere with the values:
        STRIP_LEADING    1
        STRIP_TRAILING   2
        STRIP_BOTH       STRIP_LEADING|STRIP_TRAILING
        STRIP_NONE       0

RETURN VALUE
     A pointer to the original string, now truncated.

SEE ALSO

*******************************************************************************/
#ifdef HAVE_PROTO
CHARTYPE *MyStrip(CHARTYPE *string,char option,char ch)
#else
CHARTYPE *MyStrip(string,option,ch)
CHARTYPE *string;
char option,ch;
#endif
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short pos=0;
/*--------------------------- processing ------------------------------*/
 if (strlen((DEFCHAR *)string) == 0)
    return(string);
 if (option & STRIP_TRAILING)
   {
    pos = strzrevne(string,ch);
    if (pos == (-1))
       *(string) = '\0';
    else
       *(string+pos+1) = '\0';
   }
 if (option & STRIP_LEADING)
   {
    pos = strzne(string,ch);
    if (pos == (-1))
       *(string) = '\0';
    else
      {
       for (i=0;*(string+i)!='\0';i++)
          *(string+i) = *(string+i+pos);
       *(string+i) = '\0';
      }
   }
 return(string);
}
/*man***************************************************************************
NAME
     memfind - finds a needle in a haystack respecting case and arbitrary
               characters if set.

SYNOPSIS
     short memfind(haystack,needle,hay_len,nee_len,case_ignore,arbsts,arb)
     CHARTYPE *haystack;                            string to be searched
     CHARTYPE *needle;        string to search for - may contain arbchars
     short hay_len;                                    length of haystack
     short nee_len;                                      length of needle
     bool case_ignore;                      TRUE if search to ignore case
     bool arbsts;          TRUE if need to check for arbitrary characters
     CHARTYPE single                       the single arbitrary character
     CHARTYPE multiple                   the multiple arbitrary character

DESCRIPTION
     The memfind function locates a needle in a haystack. Both the needle
     and haystack may contain null characters. If case_ignore is TRUE,
     then upper and lower case characters are treated equal. If arbsts
     is ON, any arbitrary character, specified by arb, in needle, will
     match ANY character in the haystack.

RETURN VALUE
     The first occurrence (0 based) of needle in haystack, or (-1) if
     the needle does not appear in the haystack.
*******************************************************************************/
#ifdef HAVE_PROTO
short memfind(CHARTYPE *haystack,CHARTYPE *needle,short hay_len,short nee_len,
            bool case_ignore,bool arbsts,CHARTYPE arb_single,CHARTYPE arb_multiple)
#else
short memfind(haystack,needle,hay_len,nee_len,case_ignore,arbsts,arb_single,arb_multiple)
CHARTYPE *haystack;
CHARTYPE *needle;
short hay_len;
short nee_len;
bool case_ignore;
bool arbsts;
CHARTYPE arb_single;
CHARTYPE arb_multiple;
#endif
/*--------------------------- local data ------------------------------*/
{
 register CHARTYPE c1=0,c2=0;
 register CHARTYPE *buf1=NULL,*buf2=NULL;
 register short i=0,j=0;
 short matches=0;
/*--------------------------- processing ------------------------------*/
 for (i=0;i<(hay_len-nee_len+1);i++)
    {
     buf1 = haystack+i;
     buf2 = needle;
     matches=0;
     for (j=0;j<nee_len;j++)
        {
         if (case_ignore)
           {
            if (isupper(*buf1))
               c1 = tolower(*buf1);
            else
               c1 = *buf1;
            if (isupper(*buf2))
               c2 = tolower(*buf2);
            else
               c2 = *buf2;
           }
         else
           {
            c1 = *buf1;
            c2 = *buf2;
           }
         if (arbsts)
           {
            if (c1 != c2 && c2 != arb_single)
               break;
            else
               matches++;
           }
         else
           {
            if (c1 != c2)
               break;
            else
               matches++;
           }
         ++buf1;
         ++buf2;
        }
     if (matches == nee_len)
        return(i);
    }
 return(-1);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short memcmpi(CHARTYPE *buf1,CHARTYPE *buf2,short len)
#else
short memcmpi(buf1,buf2,len)
CHARTYPE *buf1,*buf2;
short len;
#endif
/***********************************************************************/
/* Function  : Compares two memory buffers for equality;               */
/*             case insensitive. Same as memicmp() Microsoft C.        */
/* Parameters: buf1     - first buffer                                 */
/*             buf2     - second buffer                                */
/*             len      - number of characters to compare.             */
/* Return    : <0 if buf1 < buf2,                                      */
/*             =0 if buf1 = buf2,                                      */
/*             >0 if buf1 > buf2,                                      */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
/*--------------------------- processing ------------------------------*/
 for(i=0;i<len;i++)
   {
    CHARTYPE c1,c2;
    if (isupper(*buf1))
       c1 = tolower(*buf1);
    else
       c1 = *buf1;
    if (isupper(*buf2))
       c2 = tolower(*buf2);
    else
       c2 = *buf2;
    if (c1 != c2)
       return(c1-c2);
    ++buf1;
    ++buf2;
   }
 return(0);
}
/***********************************************************************/
#ifdef HAVE_PROTO
CHARTYPE *make_upper(CHARTYPE *str)
#else
CHARTYPE *make_upper(str)
CHARTYPE *str;
#endif
/***********************************************************************/
/* Function  : Makes the supplied string uppercase.                    */
/*             Equivalent to strupr() on some platforms.      .        */
/* Parameters: str      - string to uppercase                          */
/* Return    : str uppercased                                          */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 CHARTYPE *save_str=str;
/*--------------------------- processing ------------------------------*/
 while(*str)
   {
    if (islower(*str))
       *str = toupper(*str);
    ++str;
   }
 return(save_str);
}
/*man***************************************************************************
NAME
     equal - determine if strings are equal up to specified length

SYNOPSIS
     unsigned short equal(con,str,min_len)
     CHARTYPE *con,*str;
     short min_len;

DESCRIPTION
     The equal function determines if a two strings are equal, irrespective
     of case, up to the length of the second string. The length of the
     second string must be greater than or equal to the specified minimum
     length for the strings to be considered equal.

RETURN VALUE
     If 'equal' TRUE else FALSE.
*******************************************************************************/
#ifdef HAVE_PROTO
unsigned short equal(CHARTYPE *con,CHARTYPE *str,short min_len)
#else
unsigned short equal(con,str,min_len)
CHARTYPE *con,*str;
short min_len;
#endif
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    equal");
#endif
 if (min_len == 0)
   {
#ifdef TRACE
    trace_return();
#endif
    return(FALSE);
   }
 if (memfind(con,str,min(strlen((DEFCHAR *)str),strlen((DEFCHAR *)con)),
     min(strlen((DEFCHAR *)str),strlen((DEFCHAR *)con)),TRUE,FALSE,'\0','\0') == 0
 &&  strlen((DEFCHAR *)str) >= min_len
 &&  strlen((DEFCHAR *)con) >= strlen((DEFCHAR *)str))
   {
#ifdef TRACE
    trace_return();
#endif
    return(TRUE);
   }
#ifdef TRACE
 trace_return();
#endif
 return(FALSE);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short valid_integer(CHARTYPE *str)
#else
short valid_integer(str)
CHARTYPE *str;
#endif
/***********************************************************************/
/* Function  : Checks that string contains only 0-9,- or +.            */
/* Parameters: *str     - string to be checked                         */
/* Return    : TRUE or FALSE                                           */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short num_signs=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    valid_integer");
#endif
 for (i=0; i<strlen((DEFCHAR *)str); i++)
    {
     if (*(str+i) == '-' || *(str+i) == '+')
        num_signs++;
     else
        if (!isdigit(*(str+i)))
          {
#ifdef TRACE
           trace_return();
#endif
           return(FALSE);
          }
    }
 if (num_signs > 1)
   {
#ifdef TRACE
    trace_return();
#endif
    return(FALSE);
   }
#ifdef TRACE
 trace_return();
#endif
 return(TRUE);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short valid_positive_integer(CHARTYPE *str)
#else
short valid_positive_integer(str)
CHARTYPE *str;
#endif
/***********************************************************************/
/* Function  : Checks that string contains only 0-9, or +.             */
/* Parameters: *str     - string to be checked                         */
/* Return    : TRUE or FALSE                                           */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short num_signs=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    valid_positive_integer");
#endif
 for (i=0; i<strlen((DEFCHAR *)str); i++)
    {
     if (*(str+i) == '+')
        num_signs++;
     else
        if (!isdigit(*(str+i)))
          {
#ifdef TRACE
           trace_return();
#endif
           return(FALSE);
          }
    }
 if (num_signs > 1)
   {
#ifdef TRACE
    trace_return();
#endif
    return(FALSE);
   }
#ifdef TRACE
 trace_return();
#endif
 return(TRUE);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short strzeq(CHARTYPE *str,CHARTYPE ch)
#else
short strzeq(str,ch)
CHARTYPE *str;
CHARTYPE ch;
#endif
/***********************************************************************/
/* Function  : Locate in ASCIIZ string, character                      */
/* Parameters: *str     - string to be searched                        */
/*             ch       - character to be searched for                 */
/* Return    : position in string of character - (-1) if not found     */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short len=0;
 register short  i = 0;
/*--------------------------- processing ------------------------------*/
 len = strlen((DEFCHAR *)str);
 for (; i<len && str[i]!=ch; i++);
 if (i>=len)
    i = (-1);
 return(i);
}

/***********************************************************************/
#ifdef HAVE_PROTO
CHARTYPE *strtrans(CHARTYPE *str,CHARTYPE oldch,CHARTYPE newch)
#else
CHARTYPE *strtrans(str,oldch,newch)
CHARTYPE *str;
CHARTYPE oldch,newch;
#endif
/***********************************************************************/
/* Function  : Translate all occurrences of oldch to newch in str      */
/* Parameters: *str     - string to be amendedd                        */
/*             oldch    - character to be replaced                     */
/*             newch    - character to replace oldch                   */
/* Return    : same string but with characters translated              */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short  i=0;
/*--------------------------- processing ------------------------------*/
 for (i=0;i<strlen((DEFCHAR *)str); i++)
   {
    if (*(str+i) == oldch)
       *(str+i) = newch;
   }
 return(str);
}
/***********************************************************************/
#ifdef HAVE_PROTO
LINE *add_line(LINE *first,LINE *curr,CHARTYPE *line,
               LENGTHTYPE len,SELECTTYPE select)
#else
LINE *add_line(first,curr,line,len,select)
LINE *first;
LINE *curr;
CHARTYPE *line;
LENGTHTYPE len;
SELECTTYPE select;
#endif
/***********************************************************************/
/* Adds a member of the linked list for the specified file containing  */
/* the line contents and length.                                       */
/* PARAMETERS:                                                         */
/* first      - pointer to first line for the file                     */
/* curr       - pointer to current line for the file                   */
/* line       - contents of line to be added                           */
/* len        - length of line to be added                             */
/* select     - select level of new line                               */
/* RETURN:    - pointer to current item in linked list or NULL if error*/
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern LINE *curr_line;
 extern LINE *next_line;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    add_line");
#endif
 next_line = lll_add(first,curr,sizeof(LINE));
 if (next_line == NULL)
   {
#ifdef TRACE
    trace_return();
#endif
    return(NULL);
   }
 curr_line = next_line;

 curr_line->line = (CHARTYPE *)(*the_malloc)((len+1)*sizeof(CHARTYPE));
 if (curr_line->line == NULL)
   {
#ifdef TRACE
    trace_return();
#endif
    return(NULL);
   }
 memcpy(curr_line->line,line,len);
 *(curr_line->line+len) = '\0'; /* for functions that expect ASCIIZ string */
 curr_line->length = len;
 curr_line->select = select;
 curr_line->pre = NULL;
 curr_line->name = NULL;
#ifdef TRACE
 trace_return();
#endif
 return(curr_line);
}
/***********************************************************************/
#ifdef HAVE_PROTO
LINE *append_line(LINE *curr,CHARTYPE *line,LENGTHTYPE len)
#else
LINE *append_line(curr,line,len)
LINE *curr;
CHARTYPE *line;
LENGTHTYPE len;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    append_line");
#endif
 curr->line = (CHARTYPE *)(*the_realloc)(curr->line,(curr->length+len+1)*sizeof(CHARTYPE));
 if (curr->line == NULL)
   {
#ifdef TRACE
    trace_return();
#endif
    return(NULL);
   }
 memcpy(curr->line+curr->length,line,len);
 curr->length += len;
 *(curr->line+curr->length) = '\0'; /* for functions that expect ASCIIZ string */
#ifdef TRACE
 trace_return();
#endif
 return(curr);
}
/***********************************************************************/
#ifdef HAVE_PROTO
LINE *delete_line(LINE *first,LINE *curr,short direction)
#else
LINE *delete_line(first,curr,direction)
LINE *first,*curr;
short direction;
#endif
/***********************************************************************/
/* Deletes a member of the linked list for the specified file.         */
/* PARAMETERS:                                                         */
/* first      - pointer to first line for the file                     */
/* curr       - pointer to current line for the file                   */
/* direction  - direction in which to delete.                          */
/* RETURN:    - pointer to current item in linked list or NULL if error*/
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    delete_line");
#endif
 if (curr->name != (CHARTYPE *)NULL)
    (*the_free)(curr->name);
 (*the_free)(curr->line);
 curr = lll_del(&first,NULL,curr,direction);
#ifdef TRACE
 trace_return();
#endif
 return(curr);
}

/***********************************************************************/
#ifdef HAVE_PROTO
void put_string(WINDOW *win,short row,short col,CHARTYPE *string,short len)
#else
void put_string(win,row,col,string,len)
WINDOW *win;
short row,col;
CHARTYPE *string;
short len;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern chtype etmode_table[256];
/*--------------------------- local data ------------------------------*/
 register short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    put_string");
#endif
 wmove(win,row,col);
 for (i=0;i<len;i++)
   {
    waddch(win,etmode_table[*(string+i)]);
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
void put_char(WINDOW *win,chtype ch,CHARTYPE add_ins)
#else
void put_char(win,ch,add_ins)
WINDOW *win;
chtype ch;
CHARTYPE add_ins;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern chtype etmode_table[256];
 extern bool   etmode_flag[256];
/*--------------------------- local data ------------------------------*/
 chtype chr_attr=0,chr=0;
 chtype attr=0L;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    put_char");
#endif
 chr_attr = ch;

#if 0
 if (!ETMODEx)
   {
    chr = chr_attr & A_CHARTEXT;
    if (chr > 126 || chr < 32)
      {
       switch(compatible)
         {
          case COMPAT_THE:
          case COMPAT_KEDIT:
#if 0
               attr = (~chr_attr) & A_ATTRIBUTES & ~A_ALTCHARSET;
#endif
               if (chr > 126)
                  chr_attr = NONDISPx | attr;
               else
                  if (chr < 32)
                     chr_attr = ('@' + chr) | attr;
               break;
          case COMPAT_XEDIT:
#if 0
               attr = chr_attr & A_ATTRIBUTES;
#endif
               chr_attr = NONDISPx | attr;
               break;
         }
      }
   }
#endif
 chr = chr_attr & A_CHARTEXT;
 if (etmode_flag[chr])  /* etmode character has attributes, use them */
    chr_attr = etmode_table[chr];
 else
   {
    attr = chr_attr & A_ATTRIBUTES;
    chr_attr = etmode_table[chr] | attr;
   }

 if (add_ins == ADDCHAR)
    waddch(win,chr_attr);
 else
    winsch(win,chr_attr);
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
short set_up_windows(short scrn)
#else
short set_up_windows(scrn)
short scrn;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
 extern CHARTYPE display_screens;
 extern bool horizontal;
 extern WINDOW *divider;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short y=0,x=0;
 FILE_DETAILS fp;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    set_up_windows");
#endif
/*---------------------------------------------------------------------*/
/* If curses has not started exit gracefully...                        */
/*---------------------------------------------------------------------*/
 if (!curses_started)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* Allocate space for a file descriptor colour attributes...           */
/*---------------------------------------------------------------------*/
 if ((fp.attr = (COLOUR_ATTR *)(*the_malloc)(ATTR_MAX*sizeof(COLOUR_ATTR))) == NULL)
   {
    display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 if (screen[scrn].screen_view)
   {
    memcpy(fp.attr,screen[scrn].screen_view->file_for_view->attr,ATTR_MAX*sizeof(COLOUR_ATTR));
    memcpy(fp.attr+ATTR_DIVIDER,CURRENT_FILE->attr+ATTR_DIVIDER,sizeof(COLOUR_ATTR));
   }
 else
    set_up_default_colours(&fp,(COLOUR_ATTR *)NULL,ATTR_MAX);
/*---------------------------------------------------------------------*/
/* Save the position of the cursor in each window, and then delete the */
/* window. Recreate each window, that has a valid size and move the    */
/* cursor back to the position it had in each window.                  */
/*---------------------------------------------------------------------*/
 for (i=0;i<VIEW_WINDOWS;i++)
   {
    y = x = 0;
    if (screen[scrn].win[i] != (WINDOW *)NULL)
      {
       getyx(screen[scrn].win[i],y,x);
       delwin(screen[scrn].win[i]);
       screen[scrn].win[i] = (WINDOW *)NULL;
      }
    if (screen[scrn].rows[i] != 0
    &&  screen[scrn].cols[i] != 0)
      {
       screen[scrn].win[i] = newwin(screen[scrn].rows[i],screen[scrn].cols[i],
                                 screen[scrn].start_row[i],screen[scrn].start_col[i]);
       if (screen[scrn].win[i] == (WINDOW *)NULL)
         {
          display_error(30,(CHARTYPE *)"creating window",FALSE);
#ifdef TRACE
          trace_return();
#endif
          return(RC_OUT_OF_MEMORY);
         }
       wmove(screen[scrn].win[i],y,x);
      }
    }
 wattrset(screen[scrn].win[WINDOW_MAIN],set_colour(fp.attr+ATTR_FILEAREA));

 if (screen[scrn].win[WINDOW_ARROW] != (WINDOW *)NULL)
   {
    wattrset(screen[scrn].win[WINDOW_ARROW],set_colour(fp.attr+ATTR_ARROW));
    mvwaddstr(screen[scrn].win[WINDOW_ARROW],0,0,"====> ");
    wnoutrefresh(screen[scrn].win[WINDOW_ARROW]);
   }

 if (screen[scrn].win[WINDOW_IDLINE] != (WINDOW *)NULL)
   {
    wattrset(screen[scrn].win[WINDOW_IDLINE],set_colour(fp.attr+ATTR_IDLINE));
    wmove(screen[scrn].win[WINDOW_IDLINE],0,0);
    my_wclrtoeol(screen[scrn].win[WINDOW_IDLINE]);
   }

 if (screen[scrn].win[WINDOW_PREFIX] != (WINDOW *)NULL)
    wattrset(screen[scrn].win[WINDOW_PREFIX],set_colour(fp.attr+ATTR_PENDING));

 if (screen[scrn].win[WINDOW_COMMAND] != (WINDOW *)NULL)
   {
    wattrset(screen[scrn].win[WINDOW_COMMAND],set_colour(fp.attr+ATTR_CMDLINE));
    wmove(screen[scrn].win[WINDOW_COMMAND],0,0);
    my_wclrtoeol(screen[scrn].win[WINDOW_COMMAND]);
    wnoutrefresh(screen[scrn].win[WINDOW_COMMAND]);
    wmove(screen[scrn].win[WINDOW_COMMAND],0,0);
   }
/*---------------------------------------------------------------------*/
/* Set up divider window...                                            */
/*---------------------------------------------------------------------*/
 if (display_screens > 1
 &&  !horizontal)
   {
    if (divider != (WINDOW *)NULL)
       delwin(divider);
    divider = newwin(screen[1].screen_rows,2,screen[1].screen_start_row,
                     screen[1].screen_start_col-2);
    if (divider == (WINDOW *)NULL)
      {
       display_error(30,(CHARTYPE *)"creating window",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_OUT_OF_MEMORY);
      }

#if 0
# if defined(A_ALTCHARSET) && !defined(USE_NCURSES)
    wattrset(divider,A_ALTCHARSET|set_colour(fp.attr+ATTR_DIVIDER));
# else
    wattrset(divider,set_colour(fp.attr+ATTR_DIVIDER));
# endif
#else
    wattrset(divider,set_colour(fp.attr+ATTR_DIVIDER));
#endif

    draw_divider();
   }
/*---------------------------------------------------------------------*/
/* Free up  space for a file descriptor colour attributes...           */
/*---------------------------------------------------------------------*/
 (*the_free)(fp.attr);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short draw_divider(void)
#else
short draw_divider()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern WINDOW *divider;
/*--------------------------- local data ------------------------------*/
#ifndef HAVE_WVLINE
 register int i=0;
#endif
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    draw_divider");
#endif

#ifdef HAVE_WVLINE
 wmove(divider,0,0);
 wvline(divider,0,screen[1].screen_rows);
 wmove(divider,0,1);
 wvline(divider,0,screen[1].screen_rows);
#else
 for (i=0;i<screen[1].screen_rows;i++)
   {
    wmove(divider,i,0);
    waddch(divider,'|');
    waddch(divider,'|');
   }
#endif
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short create_statusline_window(void)
#else
short create_statusline_window()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
 extern ROWTYPE STATUSLINEx;
 extern WINDOW *foot;
 extern short terminal_lines;
/*--------------------------- local data ------------------------------*/
 COLOUR_ATTR attr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    create_statusline_window");
#endif
 if (!curses_started)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
 if (CURRENT_VIEW == NULL
 ||  CURRENT_FILE == NULL)
    set_up_default_colours((FILE_DETAILS *)NULL,&attr,ATTR_STATAREA);
 else
    memcpy(&attr,CURRENT_FILE->attr+ATTR_STATAREA,sizeof(COLOUR_ATTR));
 if (foot != (WINDOW *)NULL)
   {
    delwin(foot);
    foot = (WINDOW *)NULL;
   }
 switch(STATUSLINEx)
   {
    case 'B':
         foot = newwin(1,COLS,terminal_lines-1,0);
         wattrset(foot,set_colour(&attr));
         clear_footing();
         break;
    case 'T':
         foot = newwin(1,COLS,0,0);
         wattrset(foot,set_colour(&attr));
         clear_footing();
         break;
    default:
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef HAVE_PROTO
void pre_process_line(VIEW_DETAILS *the_view,LINETYPE line_number,LINE *known_curr)
#else
void pre_process_line(the_view,line_number,known_curr)
VIEW_DETAILS *the_view;
LINETYPE line_number;
LINE *known_curr;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern unsigned short pre_rec_len;
 extern CHARTYPE *pre_rec;
 extern LENGTHTYPE rec_len;
 extern CHARTYPE *rec;
/*--------------------------- local data ------------------------------*/
 LINE *curr=known_curr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    pre_process_line",(LINE *)NULL);
#endif
/*---------------------------------------------------------------------*/
/* If we haven't been passed a valid LINE*, go and get one for the     */
/* supplied line_number.                                               */
/*---------------------------------------------------------------------*/
 if (curr == (LINE *)NULL)
    curr = lll_find(the_view->file_for_view->first_line,the_view->file_for_view->last_line,
                 line_number,the_view->file_for_view->number_lines);
 memset(rec,' ',max_line_length);
 memcpy(rec,curr->line,curr->length);
 rec_len = curr->length;
/*---------------------------------------------------------------------*/
/* Now set up the prefix command from the linked list...               */
/*---------------------------------------------------------------------*/
 if (curr->pre == NULL)
   {
    memset(pre_rec,' ',PREFIX_WIDTH);
    pre_rec_len = 0;
   }
 else
   {
    memset(pre_rec,' ',PREFIX_WIDTH);
    strcpy((DEFCHAR *)pre_rec,(DEFCHAR *)curr->pre->ppc_command);
    pre_rec_len = strlen((DEFCHAR *)pre_rec);
    pre_rec[pre_rec_len] = ' ';
    pre_rec[PREFIX_WIDTH] = '\0';
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
short post_process_line(VIEW_DETAILS *the_view,LINETYPE line_number,LINE *known_curr)
#else
short post_process_line(the_view,line_number,known_curr)
VIEW_DETAILS *the_view;
LINETYPE line_number;
LINE *known_curr;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
 extern bool prefix_changed;
 extern CHARTYPE *rec;
 extern LENGTHTYPE rec_len;
/*--------------------------- local data ------------------------------*/
 LINE *curr=known_curr;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    post_process_line",(LINE *)NULL);
#endif
/*---------------------------------------------------------------------*/
/* If there are no lines in the file associated with the view, exit... */
/*---------------------------------------------------------------------*/
 if (the_view->file_for_view->first_line == NULL)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* If we haven't been passed a valid LINE*, go and get one for the     */
/* supplied line_number.                                               */
/*---------------------------------------------------------------------*/
 if (curr == (LINE *)NULL)
    curr = lll_find(the_view->file_for_view->first_line,the_view->file_for_view->last_line,
                 line_number,the_view->file_for_view->number_lines);
/*---------------------------------------------------------------------*/
/* First copy the pending prefix command to the linked list.           */
/* Only do it if the prefix command has a value or there is already a  */
/* pending prefix command for that line.                               */
/*---------------------------------------------------------------------*/
 if (prefix_changed)
    add_prefix_command(curr,line_number,FALSE);
/*---------------------------------------------------------------------*/
/* If the line hasn't changed, return.                                 */
/*---------------------------------------------------------------------*/
 if (rec_len == curr->length && (memcmp(rec,curr->line,curr->length) == 0))
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* Increment the alteration counters...                                */
/*---------------------------------------------------------------------*/
 if ((rc = increment_alt(the_view->file_for_view)) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* Add the old line contents to the line recovery list.                */
/*---------------------------------------------------------------------*/
 add_to_recovery_list(curr->line,curr->length);
/*---------------------------------------------------------------------*/
/* Realloc the dynamic memory for the line if the line is now longer.  */
/*---------------------------------------------------------------------*/
 if (rec_len > curr->length)
   {
    curr->line = (CHARTYPE *)(*the_realloc)((void *)curr->line,(rec_len+1)*sizeof(CHARTYPE));
    if (curr->line == NULL)
      {
       display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(RC_OUT_OF_MEMORY);
      }
   }
/*---------------------------------------------------------------------*/
/* Copy the contents of rec into the line.                             */
/*---------------------------------------------------------------------*/
 memcpy(curr->line,rec,rec_len);
 curr->length = rec_len;
 *(curr->line+rec_len) = '\0';
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef HAVE_PROTO
bool blank_field(CHARTYPE *field)
#else
bool blank_field(field)
CHARTYPE *field;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    blank_field");
#endif
 if (strzne(field,' ') == (-1))
   {
#ifdef TRACE
    trace_return();
#endif
    return(TRUE);                /* field is NULL or just contains spaces */
   }
#ifdef TRACE
 trace_return();
#endif
 return(FALSE);
}
/***********************************************************************/
#ifdef HAVE_PROTO
void adjust_marked_lines(bool insert_line,LINETYPE base_line,LINETYPE num_lines)
#else
void adjust_marked_lines(insert_line,base_line,num_lines)
bool insert_line;
LINETYPE base_line;
LINETYPE num_lines;
#endif
/***********************************************************************/
{
/*---------------------------------------------------------------------*/
/* When lines are deleted, the base line is the first line in the file */
/* irrespective of the direction that the delete is done.              */
/*---------------------------------------------------------------------*/
/*-------------------------- external data ----------------------------*/
 extern VIEW_DETAILS *vd_mark;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    adjust_marked_lines");
#endif
/*---------------------------------------------------------------------*/
/* If there are no marked lines in the current view, return.           */
/*---------------------------------------------------------------------*/
 if (MARK_VIEW != CURRENT_VIEW)
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
 switch(insert_line)
   {
    case TRUE:/* INSERT */
         if (base_line < CURRENT_VIEW->mark_start_line)
           {
            CURRENT_VIEW->mark_start_line += num_lines;
            CURRENT_VIEW->mark_end_line += num_lines;
            break;
           }
         if (base_line >= CURRENT_VIEW->mark_start_line
         &&  base_line < CURRENT_VIEW->mark_end_line)
           {
            CURRENT_VIEW->mark_end_line += num_lines;
            break;
           }
         break;
    case FALSE:  /* DELETE */
         if (base_line <= CURRENT_VIEW->mark_start_line
         &&  base_line+num_lines-1L >= CURRENT_VIEW->mark_end_line)
           {
            CURRENT_VIEW->marked_line = FALSE;
            MARK_VIEW = (VIEW_DETAILS *)NULL;
            break;
           }
         if (base_line+num_lines-1L < CURRENT_VIEW->mark_start_line)
           {
            CURRENT_VIEW->mark_start_line -= num_lines;
            CURRENT_VIEW->mark_end_line -= num_lines;
            break;
           }
         if (base_line > CURRENT_VIEW->mark_end_line)
           {
            break;
           }
         if (base_line+num_lines-1L > CURRENT_VIEW->mark_end_line)
           {
            CURRENT_VIEW->mark_end_line = base_line - 1L;
            break;
           }
         if (base_line < CURRENT_VIEW->mark_start_line)
           {
            CURRENT_VIEW->mark_start_line = base_line;
            CURRENT_VIEW->mark_end_line = base_line +
                                         (CURRENT_VIEW->mark_end_line -
                                          (base_line + num_lines));
            break;
           }
         CURRENT_VIEW->mark_end_line -= num_lines;
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
void adjust_pending_prefix(VIEW_DETAILS *view,bool insert_line,LINETYPE base_line,LINETYPE num_lines)
#else
void adjust_pending_prefix(view,insert_line,base_line,num_lines)
VIEW_DETAILS *view;
bool insert_line;
LINETYPE base_line;
LINETYPE num_lines;
#endif
/***********************************************************************/
{
/*---------------------------------------------------------------------*/
/* When lines are deleted, the base line is the first line in the file */
/* irrespective of the direction that the delete is done.              */
/*---------------------------------------------------------------------*/
/*--------------------------- local data ------------------------------*/
 PPC *curr_ppc=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    adjust_pending_prefix");
#endif
/*---------------------------------------------------------------------*/
/* If there are no pending prefix commands in the view, return.        */
/*---------------------------------------------------------------------*/
 if (view->file_for_view->first_ppc == NULL)
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
 curr_ppc = view->file_for_view->first_ppc;
 while (curr_ppc != NULL)
   {
    switch(insert_line)
      {
       case TRUE:/* INSERT */
            if (base_line < curr_ppc->ppc_line_number)
              {
               curr_ppc->ppc_line_number += num_lines;
               break;
              }
            break;
       case FALSE:  /* DELETE */
            if (base_line+num_lines-1L < curr_ppc->ppc_line_number)
              {
               curr_ppc->ppc_line_number -= num_lines;
               break;
              }
            if (base_line > curr_ppc->ppc_line_number)
               break;
#if OLD_CLEAR
            (void)delete_pending_prefix_command(curr_ppc,view->file_for_view,(LINE *)NULL);
#else
            clear_pending_prefix_command(curr_ppc,(LINE *)NULL);
#endif
            break;
      }
    curr_ppc = curr_ppc->next;
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
CHARTYPE case_translate(CHARTYPE key)
#else
CHARTYPE case_translate(key)
CHARTYPE key;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    case_translate");
#endif
 if (CURRENT_VIEW->case_enter == CASE_UPPER
 && islower(key))
   {
#ifdef TRACE
    trace_return();
#endif
    return(toupper(key));
   }
 if (CURRENT_VIEW->case_enter == CASE_LOWER
 && isupper(key))
   {
#ifdef TRACE
    trace_return();
#endif
    return(tolower(key));
   }
#ifdef TRACE
 trace_return();
#endif
 return(key);
}
/***********************************************************************/
#ifdef HAVE_PROTO
void add_to_recovery_list(CHARTYPE *line,LENGTHTYPE len)
#else
void add_to_recovery_list(line,len)
CHARTYPE *line;
LENGTHTYPE len;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool batch_only;
/*--------------------------- local data ------------------------------*/
 register short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    add_to_recovery_list");
#endif
/*---------------------------------------------------------------------*/
/* Ignore if running in batch.                                         */
/*---------------------------------------------------------------------*/
 if (batch_only)
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* Ignore empty lines.                                                 */
/*---------------------------------------------------------------------*/
 if (len == 0)
   {
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* First time through, set line array to NULL,  to indicated unused.   */
/* This setup MUST occur before the freeing up code.                   */
/*---------------------------------------------------------------------*/
 if (add_recv == (-1))
   {
    for (i=0;i<MAX_RECV;i++)
       recv[i] = NULL;
    add_recv = 0;               /* set to point to next available slot */
   }
/*---------------------------------------------------------------------*/
/* Now we are here, lets add to the array.                             */
/*---------------------------------------------------------------------*/
 if (recv[add_recv] == NULL)  /* haven't malloced yet */
   {
    if ((recv[add_recv] = (CHARTYPE *)(*the_malloc)((len+1)*sizeof(CHARTYPE))) == NULL)
      {
       display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return;
      }
   }
 else
   {
    if ((recv[add_recv] = (CHARTYPE *)(*the_realloc)(recv[add_recv],(len+1)*sizeof(CHARTYPE))) == NULL)
      {
       display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
       trace_return();
#endif
       return;
      }
   }
 memcpy(recv[add_recv],line,len);
 recv_len[add_recv] = len;
 retr_recv = add_recv;
 add_recv = (++add_recv >= MAX_RECV) ? 0 : add_recv;
 num_recv = (++num_recv > MAX_RECV) ? MAX_RECV : num_recv;

#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
void get_from_recovery_list(short num)
#else
void get_from_recovery_list(num)
short num;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE *temp_cmd;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short num_retr = min(num,num_recv);
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    get_from_recovery_list");
#endif
/*---------------------------------------------------------------------*/
/* Return error if nothing to recover.                                 */
/*---------------------------------------------------------------------*/
 if (retr_recv == (-1))
   {
    display_error(0,(CHARTYPE *)"0 line(s) recovered",TRUE);
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* Retrieve each allocated recovery line and put back into the body.   */
/*---------------------------------------------------------------------*/
 post_process_line(CURRENT_VIEW,CURRENT_VIEW->focus_line,(LINE *)NULL);
 for (i=0;i<num_retr;i++)
   {
    if (recv[retr_recv] != NULL)
      {
       insert_new_line(recv[retr_recv],recv_len[retr_recv],1L,get_true_line(),TRUE,FALSE,CURRENT_VIEW->display_low);
       retr_recv = (--retr_recv < 0) ? num_recv-1 : retr_recv;
      }
   }
 sprintf((DEFCHAR *)temp_cmd,"%d line(s) recovered",num_retr);
 display_error(0,temp_cmd,TRUE);
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
void free_recovery_list(void)
#else
void free_recovery_list()
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    free_recovery_list");
#endif
 for (i=0;i<MAX_RECV;i++)
   {
    if (recv[i] != NULL)
      {
       (*the_free)(recv[i]);
       recv[i] = NULL;
      }
   }
 add_recv  = (-1);
 retr_recv = (-1);
 num_recv  = 0;
#ifdef TRACE
 trace_return();
#endif
 return;
}

#if THIS_APPEARS_TO_NOT_BE_USED
/***********************************************************************/
#ifdef HAVE_PROTO
WINDOW *adjust_window(WINDOW *win,short tr,short tc,short lines,short cols)
#else
WINDOW *adjust_window(win,tr,tc,lines,cols)
WINDOW *win;
short tr;
short tc;
short lines;
short cols;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 WINDOW *neww=NULL;
 short begy=0,begx=0,maxy=0,maxx=0,y=0,x=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    adjust_window");
#endif
/*---------------------------------------------------------------------*/
/* Get existing details about the current window.                      */
/*---------------------------------------------------------------------*/
 getbegyx(win,begy,begx);
 getmaxyx(win,maxy,maxx);
 if (maxy == lines && maxx == cols)  /* same size */
   {
    if (begy == tr && begx == tc)   /* same position */
      {
#ifdef TRACE
       trace_return();
#endif
       return(win); /* nothing to do, return same window */
      }
    else /* need to move window */
      {
       rc = mvwin(win,tr,tc);
#ifdef TRACE
       trace_return();
#endif
       return(win);
      }
   }
/*---------------------------------------------------------------------*/
/* To get here the window needs to be resized.                         */
/*---------------------------------------------------------------------*/
 getyx(win,y,x);
 delwin(win);
 neww = newwin(lines,cols,tr,tc);
 if (neww != (WINDOW *)NULL)
    wmove(neww,y,x);
#ifdef TRACE
 trace_return();
#endif
 return(neww);
}
#endif

/***********************************************************************/
#ifdef HAVE_PROTO
void draw_cursor(bool visible)
#else
void draw_cursor(visible)
bool visible;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool INSERTMODEx;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    draw_cursor");
#endif
#ifdef HAVE_CURS_SET
 if (visible)
   {
    if (INSERTMODEx)
       curs_set(2);   /* block cursor */
    else
       curs_set(1);   /* underline cursor */
   }
 else
    curs_set(0);      /* cursor off */
#endif

#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
short my_wclrtoeol(WINDOW *win)
#else
short my_wclrtoeol(win)
WINDOW *win;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short x=0,y=0,maxx=0,maxy=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    my_wclrtoeol");
#endif
 if (win != (WINDOW *)NULL)
   {
    getyx(win,y,x);
    getmaxyx(win,maxy,maxx);
    for (i=x;i<maxx;i++)
#if 0
       mvwaddch(win,y,i,' ');
#else
       waddch(win,' ');
#endif
    wmove(win,y,x);
   }
#ifdef TRACE
 trace_return();
#endif
 return(0);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short my_wdelch(WINDOW *win)
#else
short my_wdelch(win)
WINDOW *win;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short x=0,y=0,maxx=0,maxy=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    my_wdelch");
#endif

 getyx(win,y,x);
 getmaxyx(win,maxy,maxx);
 wdelch(win);
 mvwaddch(win,y,maxx-1,' ');
 wmove(win,y,x);

#ifdef TRACE
 trace_return();
#endif
 return(0);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short get_word(CHARTYPE *string,LENGTHTYPE length,LENGTHTYPE curr_pos,
               LENGTHTYPE *first_col,LENGTHTYPE *last_col)
#else
short get_word(string,length,curr_pos,first_col,last_col)
CHARTYPE *string;
LENGTHTYPE length,curr_pos;
LENGTHTYPE *first_col,*last_col;
#endif
/***********************************************************************/
{
#define FIRST_BLANK  0
#define SECOND_BLANK 1
#define FIRST_WORD   2
#define SECOND_WORD  3
/*--------------------------- local data ------------------------------*/
 short state=0,cursor_off=0;
 short num_cols=0;
 register short i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    get_word");
#endif
/*---------------------------------------------------------------------*/
/* If we are after the last column of the line, then just ignore the   */
/* command and leave the cursor where it is.                           */
/*---------------------------------------------------------------------*/
 if (curr_pos >= length)
   {
#ifdef TRACE
    trace_return();
#endif
    return(0);
   }
/*---------------------------------------------------------------------*/
/* Determine the end   of the next word, or go to the end of the line  */
/* if already at or past beginning of last word.                       */
/*---------------------------------------------------------------------*/
 cursor_off = num_cols = 0;
 if (*(string+curr_pos) == ' ')
    state = FIRST_BLANK;
 else
   {
    state = FIRST_WORD;
    for (i=curr_pos;;i--)
      {
       if (*(string+i) == ' '
       ||  i == -1)
         {
          cursor_off++;
          break;
         }
       cursor_off--;
      }
   }
 for (i=curr_pos+cursor_off;i<length;i++)
   {
    switch(state)
       {
        case FIRST_BLANK:
             if (*(string+i) != ' ')
                state = FIRST_WORD;
             break;
        case FIRST_WORD:
             if (*(string+i) == ' ')
                state = SECOND_BLANK;
             break;
        case SECOND_BLANK:
             if (*(string+i) != ' ')
                state = SECOND_WORD;
             break;
       }
    if (state == SECOND_WORD)
       break;
    num_cols++;
   }
 *first_col = curr_pos + cursor_off;
 *last_col = *first_col + num_cols - 1;
#ifdef TRACE
 trace_return();
#endif
 return(1);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short my_wmove(WINDOW *win,short scridx,short winidx,short y,short x)
#else
short my_wmove(win,scridx,winidx,y,x)
WINDOW *win;
short scridx,winidx,y,x;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool curses_started;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    my_wmove");
#endif
/*---------------------------------------------------------------------*/
/* If the scridx or winidx are -1, do not try to save the x/y position.*/
/*---------------------------------------------------------------------*/
 if (scridx != (-1)
 &&  winidx != (-1))
   {
    screen[scridx].screen_view->x[winidx] = x;
    screen[scridx].screen_view->y[winidx] = y;
   }
 if (curses_started)
    wmove(win,y,x);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short get_row_for_tof_eof(short row,CHARTYPE scridx)
#else
short get_row_for_tof_eof(row,scridx)
short row;
CHARTYPE scridx;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    get_row_for_tof_eof");
#endif
 if (screen[scridx].sl[row].line_type == LINE_OUT_OF_BOUNDS_ABOVE)
   {
    for(;screen[scridx].sl[row].line_type != LINE_TOF;row++)
/*    for(;screen[scridx].sl[row].line_type != LINE_TOF_EOF;row++) MH12 */
       ;
   }
 if (screen[scridx].sl[row].line_type == LINE_OUT_OF_BOUNDS_BELOW)
   {
/*    for(;screen[scridx].sl[row].line_type != LINE_TOF_EOF;row--) MH12 */
    for(;screen[scridx].sl[row].line_type != LINE_EOF;row--)
       ;
   }
#ifdef TRACE
 trace_return();
#endif
 return(row);
}
#ifndef HAVE_DOUPDATE
/***********************************************************************/
#ifdef HAVE_PROTO
int doupdate(void)
#else
int doupdate()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 unsigned short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("util.c:    doupdate");
#endif
 getyx(CURRENT_WINDOW,y,x);
 refresh();
 wmove(CURRENT_WINDOW,y,x);
 wrefresh(CURRENT_WINDOW);
#ifdef TRACE
 trace_return();
#endif
 return(0);
}
#endif

#ifdef USE_EXTCURSES
/***********************************************************************/
#ifdef HAVE_PROTO
int has_colors(void)
#else
int has_colors()
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
 return(TRUE);
}
/***********************************************************************/
#ifdef HAVE_PROTO
int start_color(void)
#else
int start_color()
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register int i=0;
/*--------------------------- processing ------------------------------*/
 for (i=0;i<COLOR_PAIRS;i++)
   color_pair[i] = NORMAL;
 fore_color[COLOR_BLACK  ] = F_BLACK  ;
 fore_color[COLOR_BLUE   ] = F_BLUE   ;
 fore_color[COLOR_GREEN  ] = F_GREEN  ;
 fore_color[COLOR_CYAN   ] = F_CYAN   ;
 fore_color[COLOR_RED    ] = F_RED    ;
 fore_color[COLOR_MAGENTA] = F_MAGENTA;
 fore_color[COLOR_YELLOW ] = F_BROWN  ;
 fore_color[COLOR_WHITE  ] = F_WHITE  ;
 back_color[COLOR_BLACK  ] = B_BLACK  ;
 back_color[COLOR_BLUE   ] = B_BLUE   ;
 back_color[COLOR_GREEN  ] = B_GREEN  ;
 back_color[COLOR_CYAN   ] = B_CYAN   ;
 back_color[COLOR_RED    ] = B_RED    ;
 back_color[COLOR_MAGENTA] = B_MAGENTA;
 back_color[COLOR_YELLOW ] = B_BROWN  ;
 back_color[COLOR_WHITE  ] = B_WHITE  ;
 return(0);
}
/***********************************************************************/
#ifdef HAVE_PROTO
int init_pair(int pairnum,chtype fore,chtype back)
#else
int init_pair(pairnum,fore,back)
int pairnum;
chtype fore,back;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register int i=0;
/*--------------------------- processing ------------------------------*/
 color_pair[pairnum] = fore_color[fore] | back_color[back];
 return(0);
}
#endif
