/*
 * fragment.c  -- program to break BIG programs up into <360Kb files
 *                   so that they can read into/out of PC's that only
 *                   have 360Kb drives.
 */

#include <stdio.h>
#include <string.h>

long filelength(int) ;			/* Wasn't declared in <stdio.h> ! */

#define	TRUE	1
#define FALSE	0

#define	DISK_SIZE_DESIRED	327680L
#define CHUNK_SIZE		16384
#define	MAX_FNAME_LEN		256

char buffer[CHUNK_SIZE] ;
int prompt = FALSE ;
char out_path[MAX_FNAME_LEN+1] ;
char out_template[MAX_FNAME_LEN+1] ;

void prompt_for_new_file(fragment_nbr)
int fragment_nbr ;
{
    fprintf(stderr, "Strike any key when ready for Fragment %0d.", fragment_nbr) ;
    fflush(stderr) ;
    fflush(stdin) ;
    getch() ;
}

FILE *next_output_file(out_template, suffix_ndx, fragment_nbr)
char *out_template ;
int suffix_ndx ;
int fragment_nbr ;
{
    FILE *out_stream ;
    char suffix[3+1] ;

    sprintf(suffix, "%0d", 100+fragment_nbr) ;
    strncpy(&out_template[suffix_ndx], &suffix[1], 2) ;

    if ((out_stream = fopen(out_template, "rb")) != NULL) {
	fprintf(stderr, "The output file '%s' already exists.\n", out_template) ;
	fprintf(stderr, "   Hit Ctrl-Break to abort, any other key to overwrite.\n") ;
	getch() ;
	fclose(out_stream) ;
    }
    if ((out_stream = fopen(out_template, "wb")) == NULL) {
	fprintf(stderr, "can't open %s for output.\n", out_template) ;
	exit(1) ;
    }
}

void fragment(big_stream, out_template, suffix_ndx, fragment_cnt)
FILE *big_stream ;
char *out_template ;
int   suffix_ndx ;
int   fragment_cnt ;
{
    long  bytes_written ;
    FILE *out_stream ;
    int   byte_cnt ;
    int   fragment_nbr ;
    int   chunk ;

    bytes_written = 0L ;
    for (fragment_nbr=0; fragment_nbr < fragment_cnt; fragment_nbr++) {

	if (prompt)
	    prompt_for_new_file(fragment_nbr) ;

	out_stream = next_output_file(out_template, suffix_ndx, fragment_nbr) ;

	for (chunk=0; chunk < DISK_SIZE_DESIRED/CHUNK_SIZE; chunk++) {
	    byte_cnt = fread(buffer, 1, CHUNK_SIZE, big_stream) ;
	    if (byte_cnt == 0)
		break ;
	    fprintf(stderr, "\rFragment %2d, Chunk %2d: %0d bytes Read     ",
			fragment_nbr, chunk, byte_cnt) ;
	    fflush(stderr) ;
	    if (fwrite(buffer, 1, byte_cnt, out_stream) != byte_cnt) {
		fprintf(stderr, "\n") ;
		fprintf(stderr, "Error Writing Fragment %2d, Chunk %2d.\n") ;
		fprintf(stderr, "Total of %0ld were successfully coalesced.\n",
				bytes_written) ;
		exit(1) ;
	    }
	    bytes_written += (long)byte_cnt ;
	    fprintf(stderr, "\rFragment %2d, Chunk %2d: %0d bytes Written    ",
			fragment_nbr, chunk, byte_cnt) ;
	    fflush(stderr) ;
	}

	fclose(out_stream) ;
	fprintf(stderr, "\rFragment %2d: Completed                                     \n", fragment_nbr) ;
    }

    fclose(big_stream) ;
    fprintf(stderr, "End of Input File successfully reached.  Fragmentation complete.\n") ;
}

void get_template(file_template, out_template, suffix_ndx)
char *file_template ;
char *out_template ;
int  *suffix_ndx ;
{
    char fname_template[8+1] ;
    char file_ext[1+3+1] ;
    char *file_name ;
    char *file_dot ;

    *out_path = '\0' ;
    file_name = file_template ;
    if (strchr(file_template, '\\') || strchr(file_template, ':')) {
	if (strchr(file_template, '\\'))
	    file_name = strrchr(file_template, '\\')+1 ;
        else
	    file_name = strrchr(file_template, ':')+1 ;
	if (file_name-file_template > MAX_FNAME_LEN) {
	    fprintf(stderr, "Directory path of %s is too long.\n", file_template) ;
	    exit(1) ;
	} else {
	    strncpy(out_path, file_template, file_name-file_template) ;
	    out_path[file_name-file_template+1] = '\0' ;
	}
    }

    *file_ext = '\0' ;
    file_dot = strchr(file_name, '.') ;
    if (file_dot)
	if (strlen(file_dot) > 4) {
	    fprintf(stderr, "The extension of '%s' is too long.\n", file_dot) ;
	    exit(1) ;
	} else {
	    strcpy(file_ext, file_dot) ;
	    *file_dot = '\0' ;
	}

    strcpy(fname_template, "________") ;
    if (strlen(file_name) > 6) {
	fprintf(stderr, "The filename of '%s' is too long.\n", file_name) ;
	exit(1) ;
    }
    strncpy(fname_template, file_name, strlen(file_name)) ;

    if (strlen(out_path)+strlen(fname_template)+strlen(file_ext) > MAX_FNAME_LEN) {
	fprintf("The resulting output pathname of '%s%s%s' is too long.\n",
		   out_path, fname_template,file_ext) ;
	exit(1) ;
    }

    strcpy(out_template, out_path) ;
    strcat(out_template, fname_template) ;
    *suffix_ndx = strlen(out_template) - 2 ;
    strcat(out_template, file_ext) ;
}

void main(argc, argv)
int argc ;
char *argv[] ;
{
    FILE *big_stream ;
    long  length ;
    int   fragment_cnt ;
    int   suffix_ndx ;

    argc-- ; /* throw away "program name" argument */
    argv++ ;

    /* parse off any "prompt option" argument that is present */

    prompt = FALSE ;
    if (argc && (stricmp(argv[0], "-p") == 0)) {
	argc-- ;
	argv++ ;
	prompt = TRUE ;
    }

    if ((argc != 2) || (*argv[0] == '?')) {
	fprintf(stderr, "\n") ;
	fprintf(stderr, "   fragment [-p] <big fname> <output file template>\n") ;
	fprintf(stderr, "\n") ;
	fprintf(stderr, "   Where:\n") ;
	fprintf(stderr, "\n") ;
	fprintf(stderr, "       -p optionally causes the program to pause between output files.\n") ;
	fprintf(stderr, "           (allowing you to, say, change output disks)\n") ;
	fprintf(stderr, "       <big fname> is the full pathname to the file to be fragmented.\n") ;
	fprintf(stderr, "       <output file template> is the path to the output directory\n") ;
	fprintf(stderr, "          along with a six letter filename template which will\n") ;
	fprintf(stderr, "          be suffixed with the fragment nbr and the extension.\n") ;
	fprintf(stderr, "\n") ;
	fprintf(stderr, "   Example:\n") ;
	fprintf(stderr, "\n") ;
	fprintf(stderr, "       fragment -p c:\\pcip\\src\\doc.tar a:doc.frg\n") ;
	fprintf(stderr, "\n") ;
	fprintf(stderr, "       Causes a fragmented copy of c:\\pcip\\src\\doc.tar\n") ;
	fprintf(stderr, "          to be written to a:doc___00.frg, a:doc___01.frg, etc.\n") ;
	fprintf(stderr, "       Since -p is present, the program will pause between each file.\n") ;
	exit(1) ;
    }

    if ((big_stream = fopen(argv[0], "rb")) == NULL) {
	fprintf(stderr, "can't open %s for input.\n", argv[0]) ;
	exit(1) ;
    }

    length = filelength(fileno(big_stream)) ;
    if (length == -1L) {
	fprintf(stderr, "filelength on %s failed.\n", argv[0]) ;
	fclose(big_stream) ;
	exit(1) ;
    }

    fragment_cnt = (length + DISK_SIZE_DESIRED - 1L) / DISK_SIZE_DESIRED ;
    if (fragment_cnt <= 1) {
	fprintf(stderr, "%s (%ld bytes) should already fit onto one disk.\n", argv[0], length) ;
	fprintf(stderr, "   disk size is considered to be >= %ld\n", DISK_SIZE_DESIRED) ;
	fclose(big_stream) ;
	exit(1) ;
    }

    get_template(argv[1], out_template, &suffix_ndx) ;

    fprintf(stderr, "Fragmentation will result in the creation of %d Fragments\n", fragment_cnt) ;
    if (fragment_cnt > 99)
	fprintf(stderr, "Sorry, can't handle more than 100 fragment files.\n") ;
    else
	fragment(big_stream, out_template, suffix_ndx, fragment_cnt) ;
}
