/*
 *      JET PAK - HP DeskJet and LaserJet series printer utilities
 *
 *      JETDMP program - output symbolic dump of soft font file
 *
 *      Version 1.1 (Public Domain)
 */

/* system include files */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/* application include files */
#include "patchlev.h"
#include "jetfont.h"
#include "jetmean.h"
#include "jetutil.h"
#include "jetbmp.h"

/* macro to print out first three descriptor table fields derived/forced */
#define _PFD(field)  field,(loc=nloc,nloc+=sizeof(field),loc),sizeof(field)
#define _PFF(field,val)     field,(loc=nloc,nloc+=val,loc),val

/*
 * MODULE GLOBAL DATA
 */

/* baseline distance for bottom pass */
static UNSIGNEDINT baseline;

/* portrait or landscape font */
static UNSIGNEDBYTE orientation;

/* scratch area for decompressed DJ bitmap */
static char scratch[BITMAP_SIZE_MAX];

/* font command being processed */
static FONT_COMMAND fc;

/* character code being processed */
static UNSIGNEDINT charcode;

/* suffix for output file */
static char output_suffix[SUFFIX_MAX] = ".dmp";

/*
 * LOCAL FUNCTIONS
 */

static void usage_wrong()
{
    /*
     * Print usage message and exit.
     */
    fprintf(stderr, USAGE_HEADER);
    fprintf(stderr, "Usage: JETDMP [-h] [-t filetype] fontfile [fontfile...]\n\n");
    fprintf(stderr, "Dump symbolic listings for soft fonts\n\n");
    fprintf(stderr, "  -h            print this usage information\n");
    fprintf(stderr, "  -t filetype   change output file type (default %s)\n", output_suffix);
    fprintf(stderr, "  fontfile      soft font file name\n");
    exit(1);
}

static int buffer_dump(fp, buf, len)
FILE *fp;
char *buf;
int len;
{
    /*
     * Dump a buffer, escaping '\' and '"' as necessary
     */

    if (putc('"', fp) < 0)
        return(ERROR);

    while (len-- > 0)
    {
        if (*buf == '\\' || *buf == '"')
            if (putc('\\', fp) < 0)
                return(ERROR);

        if (putc(*buf, fp) < 0)
            return(ERROR);

        buf++;
    }

    if (putc('"', fp) < 0)
        return(ERROR);

    return(OK);
}

static int filename_dump(fp, sp)
FILE *fp;
char *sp;
{
    /*
     * Dump a filename record
     */

    if (fprintf(fp, "%s ", KEYWORD_FILE) < 0) return(ERROR);
    if (buffer_dump(fp, sp, strlen(sp)) == ERROR) return(ERROR);
    if (fprintf(fp, "\n") < 0) return(ERROR);

    return(OK);
}

static int header_dump(fp)
FILE *fp;
{
    /*
     * Dump the font descriptor header record
     */
    int loc, nloc = 0;
    char name_combined[sizeof(fc.data.font.name)+sizeof(fc.data.font.name_extension)];

    if (fprintf(fp, "\n%s\n", KEYWORD_HEADER) < 0) return(ERROR);
    if (fprintf(fp, "\n#          Key to HEADER information:\n") < 0) return(ERROR);
    if (fprintf(fp, "#\n") < 0) return(ERROR);
    if (fprintf(fp, "# Value    Loc   Size  Type  Name               Units/meaning\n") < 0) return(ERROR);
    if (fprintf(fp, "#\n") < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Descriptor Size    Bytes\n",    _PFD(fc.data.font.size)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UB    Header Format      %s\n",       _PFD(fc.data.font.header_format), header_format_meaning(fc.data.font.header_format)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UB    Font Type          %s\n",       _PFD(fc.data.font.type), font_type_meaning(fc.data.font.type)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d        Reserved           \n",         _PFD(fc.data.font.reserved1)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Baseline Distance  Dots\n",     _PFD(fc.data.font.baseline_distance)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Cell Width         Dots\n",     _PFD(fc.data.font.cell_width)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Cell Height        Dots\n",     _PFD(fc.data.font.cell_height)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UB    Orientation        %s\n",       _PFD(fc.data.font.orientation), orientation_meaning(fc.data.font.orientation)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  B     Spacing            %s\n",       _PFD(fc.data.font.spacing), spacing_meaning(fc.data.font.spacing)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Symbol Set         %s\n",       _PFD(fc.data.font.symbol_set), symbol_set_meaning(fc.data.font.symbol_set)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Pitch              1/4 Dots\n", _PFD(fc.data.font.pitch)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Height             1/4 Dots\n", _PFD(fc.data.font.height)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    x Height           1/4 Dots\n", _PFD(fc.data.font.x_height)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6d # %-4d  %-4d  SB    Width Type         %s\n",       _PFD(fc.data.font.width_type), width_type_meaning(fc.data.font.width_type)) < 0) return(ERROR);
    if (fc.data.font.header_format == DJ500FONTFORMAT)
    {
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Style              %s\n",       _PFD(fc.data.font.style), style_meaning(fc.data.font.style)) < 0) return(ERROR);
    }
    else
    {
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UB    Style              %s\n",       _PFF(fc.data.font.style,1), style_meaning(fc.data.font.style)) < 0) return(ERROR);
    }
    if (fprintf(fp, "  %-6d # %-4d  %-4d  SB    Stroke Weight      %s\n",       _PFD(fc.data.font.stroke_weight), stroke_weight_meaning(fc.data.font.stroke_weight)) < 0) return(ERROR);
    if (fc.data.font.header_format == DJ500FONTFORMAT)
    {
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Typeface           %s\n",       _PFD(fc.data.font.typeface), typeface_meaning(fc.data.font.typeface)) < 0) return(ERROR);
    }
    else
    {
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UB    Typeface           %s\n",       _PFF(fc.data.font.typeface,1), typeface_meaning(fc.data.font.typeface)) < 0) return(ERROR);
    }
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UB    Slant              \n",         _PFD(fc.data.font.slant)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UB    Serif Style        %s\n",       _PFD(fc.data.font.serif_style), serif_style_meaning(fc.data.font.serif_style)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UB    Quality            %s\n",       _PFD(fc.data.font.quality), quality_meaning(fc.data.font.quality)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6d # %-4d  %-4d  SB    Placement          %s\n",       _PFD(fc.data.font.placement), placement_meaning(fc.data.font.placement)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6d # %-4d  %-4d  SB    Underline distance Dots\n",     _PFD(fc.data.font.underline_distance)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UB    Underline height   Dots\n",     _PFD(fc.data.font.underline_height)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Text Height        1/4 Dots\n", _PFD(fc.data.font.text_height)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Text Width         1/4 Dots\n", _PFD(fc.data.font.text_width)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    First Code         \n",         _PFD(fc.data.font.first_code)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Last Code          \n",         _PFD(fc.data.font.last_code)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  EB    Extended Pitch     \n",         _PFD(fc.data.font.pitch_extended)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  EB    Extended Height    \n",         _PFD(fc.data.font.height_extended)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d        Reserved           \n",         _PFD(fc.data.font.reserved2)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Font Number Top    \n",         _PFD(fc.data.font.font_number_top)) < 0) return(ERROR);
    if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Font Number Bottom \n",         _PFD(fc.data.font.font_number_bot)) < 0) return(ERROR);

    /* adjust for font name size (output out of sync, at the end) */
    nloc += sizeof(fc.data.font.name);

    if (fc.data.font.size > LJFDSIZE)
    {
        if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    H pixel resolution Dots/Inch\n", _PFD(fc.data.font.h_pixel_resolution)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    V pixel resolution Dots/Inch\n", _PFD(fc.data.font.v_pixel_resolution)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6d # %-4d  %-4d  SB    Top Dbl Und Dist   Dots\n",      _PFD(fc.data.font.tdu_distance)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  UB    Top Dbl Und Height Dots\n",      _PFD(fc.data.font.tdu_height)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6d # %-4d  %-4d  SB    Bot Dbl Und Dist   Dots\n",      _PFD(fc.data.font.bdu_distance)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  UB    Bot Dbl Und Height Dots\n",      _PFD(fc.data.font.bdu_height)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Printer Desc Size  Bytes\n",     _PFD(fc.data.font.specific_size)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Font Data Size     \n",          _PFD(fc.data.font.data_size)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  B     Unidirectional?    %s\n",        _PFD(fc.data.font.unidirection), boolean_meaning(fc.data.font.unidirection)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  B     Compressed Info?   %s\n",        _PFD(fc.data.font.compressed), boolean_meaning(fc.data.font.compressed)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  UB    Hold Time Factor   Relative\n",  _PFD(fc.data.font.hold_time_factor)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  B     No Half Pitch?     %s\n",        _PFD(fc.data.font.no_half_pitch), boolean_meaning(fc.data.font.no_half_pitch)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  B     No Double Pitch?   %s\n",        _PFD(fc.data.font.no_double_pitch), boolean_meaning(fc.data.font.no_double_pitch)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  B     No Half Height?    %s\n",        _PFD(fc.data.font.no_half_height), boolean_meaning(fc.data.font.no_half_height)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  B     No Bold?           %s\n",        _PFD(fc.data.font.no_bold), boolean_meaning(fc.data.font.no_bold)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  B     No Draft?          %s\n",        _PFD(fc.data.font.no_draft), boolean_meaning(fc.data.font.no_draft)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  B     Dark Bold Method?  %s\n",        _PFD(fc.data.font.bold_method), boolean_meaning(fc.data.font.bold_method)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d        Reserved           \n",          _PFD(fc.data.font.reserved3)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Baseline Offset 2  Dots\n",      _PFD(fc.data.font.baseline_offset_2)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Baseline Offset 3  Dots\n",      _PFD(fc.data.font.baseline_offset_3)) < 0) return(ERROR);
        if (fprintf(fp, "  %-6u # %-4d  %-4d  UI    Baseline Offset 4  Dots\n",      _PFD(fc.data.font.baseline_offset_4)) < 0) return(ERROR);
    }

    if (fc.data.font.header_format == DJ500FONTFORMAT)
    {
        /* print out combined name and name extension for DJ500 only */
        strncpy(name_combined, fc.data.font.name, sizeof(fc.data.font.name));
        strncpy(name_combined+sizeof(fc.data.font.name), fc.data.font.name_extension, sizeof(fc.data.font.name_extension));

        if (fprintf(fp, "\n%s ", KEYWORD_NAME) < 0) return(ERROR);
        if (buffer_dump(fp, name_combined, sizeof(name_combined)) == ERROR) return(ERROR);
        if (fprintf(fp, "\n") < 0) return(ERROR);
    }
    else
    {
        if (fprintf(fp, "\n%s ", KEYWORD_NAME) < 0) return(ERROR);
        if (buffer_dump(fp, fc.data.font.name, sizeof(fc.data.font.name)) == ERROR) return(ERROR);
        if (fprintf(fp, "\n") < 0) return(ERROR);
    }

    if (fprintf(fp, "\n%s ", KEYWORD_COMMENT) < 0) return(ERROR);
    if (buffer_dump(fp, fc.data.font.comment, strlen(fc.data.font.comment)) == ERROR) return(ERROR);
    if (fprintf(fp, "\n") < 0) return(ERROR);

    /* print key to the character descriptor information */
    switch (fc.data.font.header_format)
    {
    default:
        fprintf(stderr, WARNING_BAD_FONT_FORMAT,
                    os_dir, os_file, fc.data.font.header_format);

        /* !! INTENTIONAL DROP-THROUGH !! */

    case LJFONTFORMAT:
        if (fprintf(fp, "\n#        Key to CHAR information:\n") < 0) return(ERROR);
        if (fprintf(fp, "#\n") < 0) return(ERROR);
        if (fprintf(fp, "# Tag    Loc   Size  Type  Name               Units/meaning\n") < 0) return(ERROR);
        if (fprintf(fp, "#\n") < 0) return(ERROR);
        if (fprintf(fp, "# Form   0     1     UB    Format             4=LJ\n") < 0) return(ERROR);
        if (fprintf(fp, "# Cont   1     1     B     Is contination?    0=No,1=Yes\n") < 0) return(ERROR);
        if (fprintf(fp, "# Dsiz   2     1     UB    Descriptor size    Bytes\n") < 0) return(ERROR);
        if (fprintf(fp, "# Clas   3     1     UB    Class              1=LJ\n") < 0) return(ERROR);
        if (fprintf(fp, "# Orie   4     1     UB    Orientation        0=Portr,1=Lands\n") < 0) return(ERROR);
        if (fprintf(fp, "# Rese   5     1           Reserved           \n") < 0) return(ERROR);
        if (fprintf(fp, "# Loffst 6     2     SI    Left offset        Dots\n") < 0) return(ERROR);
        if (fprintf(fp, "# Toffst 8     2     SI    Top offset         Dots\n") < 0) return(ERROR);
        if (fprintf(fp, "# Cwidth 10    2     UI    Character width    Dots\n") < 0) return(ERROR);
        if (fprintf(fp, "# Chght  12    2     UI    Character height   Dots\n") < 0) return(ERROR);
        if (fprintf(fp, "# DeltaX 14    2     SI    Delta X            1/4 Dots\n") < 0) return(ERROR);
        break;
    case DJFONTFORMAT:
    case DJPFONTFORMAT:
    case DJ500FONTFORMAT:
        if (fprintf(fp, "\n#        Key to CHAR information:\n") < 0) return(ERROR);
        if (fprintf(fp, "#\n") < 0) return(ERROR);
        if (fprintf(fp, "# Tag    Loc   Size  Type  Name               Units/meaning\n") < 0) return(ERROR);
        if (fprintf(fp, "#\n") < 0) return(ERROR);
        if (fprintf(fp, "# Form   0     1     UB    Format             5=DJ,9=DJ+\n") < 0) return(ERROR);
        if (fprintf(fp, "# Cont   1     1     UB    Is contination?    0=No,1=Yes\n") < 0) return(ERROR);
        if (fprintf(fp, "# Dsiz   2     1     UB    Descriptor size    Bytes\n") < 0) return(ERROR);
        if (fprintf(fp, "# Type   3     1     UB    Character type     0=Single Pass,2-5=Multi Pass\n") < 0) return(ERROR);
        if (fprintf(fp, "# Wdth   4     1     UB    Character width    Dots\n") < 0) return(ERROR);
        if (fprintf(fp, "# Cwdt   5     1     UB    Compressed width   Dots\n") < 0) return(ERROR);
        if (fprintf(fp, "# Lpad   6     1     SB/UB Left pad           Dots\n") < 0) return(ERROR);
        if (fprintf(fp, "# Rpad   7     1     SB/UB Right pad          Dots\n") < 0) return(ERROR);
        break;
    }

    /* save baseline and orientation for bitmap marking */
    baseline = fc.data.font.baseline_distance;
    orientation = fc.data.font.orientation;

    return(OK);
}

static int character_dump(fp)
FILE *fp;
{
    /*
     * Dump a character descriptor record
     */

    if (fprintf(fp, "\n%s\n\n", KEYWORD_CHAR) < 0)
        return(ERROR);

    if (isascii(charcode) && isprint(charcode))
    {
        if (fprintf(fp, "# %c    ", charcode) < 0)
            return(ERROR);
    }
    else
    {
        if (fprintf(fp, "# \\%03o ", charcode) < 0)
            return(ERROR);
    }

    switch (fc.data.character.format)
    {
    default:
        fprintf(stderr, WARNING_BAD_CHAR_FORMAT,
                    os_dir, os_file, fc.data.character.format);

        /* !! INTENTIONAL DROP-THROUGH !! */

    case LJCHARFORMAT:
        if (fprintf(fp, "Form Cont Dsiz Clas Orie Rese Loffst Toffst Cwidth Chght  DeltaX\n") < 0)
            return(ERROR);

        if (fprintf(fp, "  %-4u %-4u %-4u %-4u %-4u %-4u %-4u %-6d %-6d %-6u %-6u %-6d\n",
                charcode,
                fc.data.character.format,
                fc.data.character.continuation,
                fc.data.character.data.ljchar.descriptor_size,
                fc.data.character.data.ljchar.class,
                fc.data.character.data.ljchar.orientation,
                fc.data.character.data.ljchar.reserved,
                fc.data.character.data.ljchar.left_offset,
                fc.data.character.data.ljchar.top_offset,
                fc.data.character.data.ljchar.character_width,
                fc.data.character.data.ljchar.character_height,
                fc.data.character.data.ljchar.delta_x) < 0)
            return(ERROR);
        break;

    case DJCHARFORMAT:
    case DJPCHARFORMAT:
    case DJ500CHARFORMAT:
        if (fprintf(fp, "Form Cont Dsiz Type Wdth Cwdt Lpad Rpad\n") < 0)
            return(ERROR);

        if (fprintf(fp, "  %-4u %-4u %-4u %-4u %-4u %-4u %-4u %-4d %-4d\n",
                charcode,
                fc.data.character.format,
                fc.data.character.continuation,
                fc.data.character.data.djchar.descriptor_size,
                fc.data.character.data.djchar.char_type,
                fc.data.character.data.djchar.character_width,
                fc.data.character.data.djchar.comp_width,
                fc.data.character.data.djchar.left_offset,
                fc.data.character.data.djchar.right_offset) < 0)
            return(ERROR);
        break;
    }

    return(OK);
}

static int bitmap_dump(fp, bp, cw, ch, baseline)
FILE *fp;
register UNSIGNEDBYTE *bp;
UNSIGNEDINT cw, ch;
SIGNEDINT baseline;
{
    /*
     * Dump a bitmap record
     */
    register UNSIGNEDINT pmask;
    UNSIGNEDINT x;
    UNSIGNEDBYTE *saved_bp;
    UNSIGNEDINT ow;

    if (fprintf(fp, "\n%s\n\n", KEYWORD_BITMAP) < 0)
        return(ERROR);

    if (orientation == LANDSCAPE)
    {
        if (baseline < 0)
            while (++baseline != 0)
                if (putc(' ', fp) < 0)
                    return(ERROR);

        if (fprintf(fp, "#<-BASELINE\n") < 0)
            return(ERROR);
    }

    while (ch-- > 0)
    {
        /* make one pass through the line to find rightmost set dot */
        saved_bp = bp;
        ow = 0;
        for (x = 0, pmask = 0x80; x < cw; x++, pmask >>= 1)
        {
            if (pmask == 0)
            {
                pmask = 0x80;
                bp++;
            }

            if (*bp & pmask)
                ow = x;
        }

        /* print bitmap omitting clear dots to right of set dots: the
           exception is that one clear dot is printed for completely
           clear lines so the token isn't lost!! */
        bp = saved_bp;
        for (x = 0, pmask = 0x80; x < cw; x++, pmask >>= 1)
        {
            if (pmask == 0)
            {
                pmask = 0x80;
                bp++;
            }

            if (*bp & pmask)
            {
                if (putc('@', fp) < 0)
                    return(ERROR);
            }
            else if (x <= ow)
            {
                if (putc('.', fp) < 0)
                    return(ERROR);
            }
        }
        bp++;

        if (orientation == PORTRAIT)
        {
            /* mark the baseline position */
            if (--baseline == 0)
                if (fprintf(fp, " # __BASELINE__") < 0)
                    return(ERROR);
        }

        if (putc('\n', fp) < 0)
            return(ERROR);
    }

    return(OK);
}

static void jetdmp()
{
    /*
     * Dump a complete soft font file
     */
    char inpath[OS_PATH_LEN], outpath[OS_PATH_LEN], *sp;
    FILE *infp, *outfp;
    SIGNEDINT offset;
    int r, fdc_found = FALSE;

    /* build the input and output file paths */
    strcpy(inpath, os_dir);
    strcat(inpath, os_file);
    strcpy(outpath, os_file);
    if ((sp = strrchr(outpath, '.')) == NULL)
        strcat(outpath, output_suffix);
    else
        strcpy(sp, output_suffix);

    /* rudimentary check for input overwriting output */
    if (strcmp(inpath, outpath) == 0)
    {
        fprintf(stderr, ERROR_OVERWRITE, os_dir, os_file);
        return;
    }

    if (!(infp = fopen(inpath, "rb")))
    {
        fprintf(stderr, ERROR_OPEN_READ_FAILED, os_dir, os_file);
        return;
    }
    if (!(outfp = fopen(outpath, "w")))
    {
        fprintf(stderr, ERROR_OPEN_WRITE_FAILED, os_dir, os_file, outpath);
        fclose(infp);
        return;
    }

    if (filename_dump(outfp, os_file) == ERROR)
    {
        fprintf(stderr, ERROR_FILE_WRITE_FAILED, os_dir, os_file);
        fclose(infp);
        fclose(outfp);
        return;
    }

    while ((r = font_command_read(infp, &fc)) == OK)
    {
        switch(fc.command_type)
        {
        case FDC:
            if (header_dump(outfp) == ERROR)
            {
                fprintf(stderr, ERROR_FILE_WRITE_FAILED, os_dir, os_file);
                fclose(infp);
                fclose(outfp);
                return;
            }

            fdc_found = TRUE;

            break;
        case CCC:
            charcode = fc.number;
            break;
        case CDC:
            if (character_dump(outfp) == ERROR)
            {
                fprintf(stderr, ERROR_FILE_WRITE_FAILED, os_dir, os_file);
                fclose(infp);
                fclose(outfp);
                return;
            }

            switch(fc.data.character.format)
            {
            default:

                /* !! INTENTIONAL DROP-THROUGH */

            case LJCHARFORMAT:
                /* get offset to baseline mark */
                if (orientation == PORTRAIT)
                    offset = fc.data.character.data.ljchar.top_offset;
                else
                    offset = fc.data.character.data.ljchar.left_offset;

                if (bitmap_dump(outfp,
                    fc.data.character.data.ljchar.bitmap,
                    fc.data.character.data.ljchar.character_width,
                    fc.data.character.data.ljchar.character_height,
                    offset) == ERROR)
                {
                    fprintf(stderr, ERROR_FILE_WRITE_FAILED, os_dir, os_file);
                    fclose(infp);
                    fclose(outfp);
                    return;
                }

                break;
            case DJCHARFORMAT:
            case DJPCHARFORMAT:
            case DJ500CHARFORMAT:
                /* get offset to baseline mark */
                if (orientation == PORTRAIT)
                {
                    if (pass_for_char_type(fc.data.character.data.djchar.char_type) == 0)
                        offset = (SIGNEDINT)baseline;
                    else
                        offset = (SIGNEDINT)PASS_HEIGHT+1;
                }
                else
                {
                    offset = ((SIGNEDINT)fc.data.character.data.djchar.comp_width) - 2*(SIGNEDINT)baseline;
                }

                if (bitmap_decompress(fc.data.character.data.djchar.bitmap,
                    fc.data.character.data.djchar.character_width,
                    PASS_HEIGHT, scratch, sizeof(scratch)) == ERROR)
                {
                    fprintf(stderr, ERROR_BITMAP_TOO_BIG, os_dir, os_file);
                    fclose(infp);
                    fclose(outfp);
                    return;
                }

                if (bitmap_dump(outfp, scratch,
                       fc.data.character.data.djchar.character_width,
                       PASS_HEIGHT, offset) == ERROR)
                {
                    fprintf(stderr, ERROR_FILE_WRITE_FAILED, os_dir, os_file);
                    fclose(infp);
                    fclose(outfp);
                    return;
                }

                break;
            }
            break;
        }
    }


    /* error scenarios: (!fdc_found) means no font descriptor found,
       probably processing a text file; (r != EOF) means a font
       command escape sequence wasn't read in correctly, probably
       processing a binary file or a truncated soft font file */
    if (!fdc_found || r != EOF)
        fprintf(stderr, ERROR_FILE_READ_FAILED, os_dir, os_file);
    else
        fprintf(stderr, OK_JETDMP, os_dir, os_file, outpath);

    fclose(infp);
    fclose(outfp);
}

/*
 *  EXTERNAL FUNCTION
 */

int main(argc, argv)
int argc;
char *argv[];
{
    char c;

    /* stop getopt() printing errors */
    opterr = FALSE;
    while ((c = getopt(argc, argv, "t:h")) != EOF)
    {
        switch (c)
        {
        case 't':
            strncpy(output_suffix+1, optarg, SUFFIX_MAX-2);
            output_suffix[SUFFIX_MAX-1] = '\0';
            break;
        case 'h':
        case '?':
            /* help required, or invalid option specified */
            usage_wrong();
        }
    }

    /* must specify at least one file */
    if (optind >= argc)
        usage_wrong();

    /* process file arguments */
    if (os_findfiles((argc - optind), &argv[optind]) == ERROR)
        fprintf(stderr, ERROR_OUT_OF_HEAP);

    while (os_getfile() != EOF)
        jetdmp();

    return(0);
}
