/**
 * \file lib_filters.h
 * \author Sorin SUCIU <sorin.suciu@gecadtech.com>
 * \date 19-05-2005
 *
 * Copyright &copy; 2005 GeCAD Technologies SRL. All Rights Reserved.
 * \brief interface to the external filters library
 */
#ifndef _LIB_FILTERS_H_
#define _LIB_FILTERS_H_

/// Class AxigenFilter.
/**
 * This is a pure virtual class used for making a connector to axigen's filtering
 * system. 
 */
class AxigenFilter
{
public:
	///Struct defining a char* elements array.
	/**
 	* Struct stringArray used for holding the RCPTs list, and the scannedReject() messages
 	*/
	struct stringArray {
        ///an array of strings
        char **elements;
        ///the length of the array
        int length;
	};
	///Struct defining the virus or spam results for the current mail
	/**
	 * An instance of the virus_spam structure will be passed to the write functions to be
	 * used for the Sieve virustest and spamtest extension
	 * 
	 * Thus, if the filter implementing this library is an anti-virus, it MUST set the virusLevel member
	 * and if it is an anti-spam filter, the spamLevel member. In the case the filter is both,
	 * it must set both the virusLevel and spamLevel members. If the filter is neither an anti-virus,
	 * nor an anti-spam filter, it will pass an instance with the members set to 0.
	 * 
	 * The values which are to be set are described as follows: (as defined in RFC 3685)
	 * 
	 * SPAMTEST:
	 *   0          message was not tested for spam
     *   1          message was tested and is clear of spam
     *   2 - 9      message was tested and has a varying likelihood of
     *              containing spam in increasing order
     *   10         message was tested and definitely contains spam
	 *
	 * VIRUSTEST:
	 *   0          message was not tested for viruses
     *   1          message was tested and contains no known viruses
     *   2          message was tested and contained a known virus which
     *              was replaced with harmless content
     *   3          message was tested and contained a known virus
     *              which was "cured" such that it is now harmless
     *   4          message was tested and possibly contains a
     *              known virus
     *   5          message was tested and definitely contains a
     *              known virus
	 */  
	struct virus_spam {
        ///the virusLevel in the case the filter is an antivirus
        int virusLevel;
        ///the spamLevel in the case the filter is an antispam
        int spamLevel;
        virus_spam():virusLevel(0),spamLevel(0) {}
	}; 
	///Enum of return types.
	/**
	 * Enum that defines the return types of most of the functions, and also defines the
	 * error codes that will be passed as an argument in the virtual functions error() and warning()
	 */
	enum ReturnTypes {
		//error return types
		RT_INVALID_STATE 		= -11,	/*!< functions were not called in the correct order (WILL NOT CALL error()) 
											- returned by read, clean and write functions*/
		RT_COMPLETE_ERROR 		= -10,	/*!< the function complete() returned -1 
											- returned by read*/
		RT_PROCESSING_ERROR 	= -9,	/*!< the function process() returned -1 
											- returend by write functions*/
		RT_INVALID_ARGUMENTS 	= -8,	/*!< invalid arguments were passed to a function 
											- returned by read and write functions*/
		RT_UNKNOWN_ERROR 		= -7,	/*!< unknown error 
											- returned by read and write functions*/
		RT_ERROR_QUEUE 			= -6,	/*!< error while creating/accessing mail queue 
											- returned by read and write functions*/
		RT_ERROR_INTERNAL 		= -5,	/*!< structures used in this library were not in correct state
											- returned by read and write functions*/
		RT_ERROR_PROTOCOL 		= -4,	/*!< an error occurred in the communication protocol 
											- returned by read and write functions*/
		RT_ERROR_SOCKET_WRITE 	= -2,	/*!< socket write error
											- returned by read and write functions*/
		RT_ERROR_SOCKET_READ 	= -1,	/*!< socket read error
											- returned by read and write functions*/
		RT_ERROR_CONN_CLOSED	= -12,	/*!< connection closed by the client
											- returned by read and write functions*/
		RT_READ_TIMEOUT			= 0,    /*!< timeout on read operation 
											- returned by read and write functions*/
		RT_WRITE_TIMEOUT		= 10,    /*!< timeout on write operation 
											- returned by read and write functions*/
		//success return types
		RT_CLEAN_COMPLETE 		= 15,	/*!< reseting the connector's state completed 
											- returned by clean*/
		RT_WRITE_AND_CLEAN_COMPLETE = 16, /*!< writing information and reseting the connector's state completed 
											- returned by write functions*/
		RT_PROCESSING_COMPLETE 	= 14,	/*!< reading information and processing completed 
											- returned by read*/
		//reserved return types
		RT_READ_COMPLETE 		= 11,	/*!< reading information from axigen completed (for a certain mail). will not appear in current version */
		RT_WRITE_COMPLETE 		= 13,	/*!< writing information to axigen completed (for a certain mail). will not appear in current version */
	};
    /// Constructor
    /**
     * constructs an AxigenFilter connector
     * \param filterName name of the filter (must have max. 128 chars) and MUST not contain spaces
     * \param descriptor the socket fileDescriptor used for communicating with axigen 
     * (must be valid)
     * \param queuePath location where the mail queue should be created if it cannot connect
     * with axigen's queue (must not be NULL)
     * \param bindTimeout the timeout used when trying to bind a mail in the queue (used if it shares
     * the queue with axigen), default -1 (infinite)
     * \param socketTimeout the socket timeout in milisecconds
     */
	AxigenFilter(const char *filterName, int descriptor, const char *queuePath, 
				 int bindTimeout = -1, int socketTimeout = 500);
	/// Destructor
	/** 
	 * destructs an AxigenFilter connector
	 */
	~AxigenFilter();
	///Initialises the library.
	/**
	 * This function initialises the library and MUST be called once per process, before any
	 * other function (this includes the constructor) is called
	 */
	static void init();
	///Cleanup of the library.
	/**
	 * This function performes the cleanup of the library, it MUST be called after all other functions
	 * and before the process ends, it also must be called once per process
	 */
	static void cleanup();

    ///Initialises the object.
    /**
     * This function initializes the object and MUST be called right after the constructor
     * \return -1 if the parameters passed to the constructor were invalid. In this case
     * the object must be destroyed.
     */
    int initialize();	
    
    ///Reads information from axigen, for a certain mail.
	/**
	 * This function also connects to axigen for retrieving queue information. It should be
	 * called when data is available for reading on the socket. This function writes on socket
	 * \return the error or success code as defined above. It also calls error() with the 
	 * respective error code when any error. If the error is RT_INVALID_ARGUMENTS, it refers to 
	 * an argument error in the constructor. For some of the codes defined above, for example, 
	 * RT_READ_TIMEOUT and RT_WRITE TIMEOUT, the error() function will not be called 
	 */
	ReturnTypes read();
	/// Resets the connector state and prepares for the next mail
	/**
	 * This function is called by any write function except when complete() returns -1,
	 * then it must be called manually.
	 * \param remove when true, it removes the mail form the local queue (if the queue is not shared)
	 * MUST always be set to true, except the case when writeNeedsCleaning() is the write function
	 * and complete() returns -1
	 * \return error or success code defined above
	 */
	ReturnTypes clean(bool remove = true);
	/// Writes a response to axigen for a certain mail
	/**
	 * The message was modified, cleaned and can be delivered to the user's mailbox
	 * \warning Implementations MUST call only one write function for a given mail
	 * \param message a message to the axigen
     * \param test a struct that will define the virus and/or spam level the filter reports
	 * \return an error or success code as defined above
	 */
	ReturnTypes writeCleannedOk(const char *message,virus_spam test);
	/// Writes a response to axigen for a certain mail
	/**
	 * The message was modified, cleaned and because the respective operation was
	 * unsuccessful or because of other reasons,  the message is not safe to be delivered
	 * to the user's mailbox (this does not necessary mean that it will not be delivered)
	 * \warning Implementations MUST call only one write function for a given mail
	 * \param message a message to the axigen
     * \param test a struct that will define the virus and/or spam level the filter reports
	 * \return an error or success code as defined above
	 */
	ReturnTypes writeCleannedReject(const char *message, virus_spam test);
	/// Writes a response to axigen for a certain mail
	/**
	 * The message was scanned, NOT modified and can be delivered to the user's mailbox
	 * \warning Implementations MUST call only one write function for a given mail
	 * \param message a message to the axigen
     * \param test a struct that will define the virus and/or spam level the filter reports
	 * \return an error or success code as defined above
	 */
	ReturnTypes writeScannedOk(const char *message, virus_spam test);
	/// Writes a response to axigen for a certain mail
	/**
	 * The message was scanned, NOT modified and because something was found wrong
	 * or because of other reasons,  the message is not safe to be delivered to the user's
	 * mailbox (this does not necessary mean that it will not be delivered)
	 * \warning Implementations MUST call only one write function for a given mail
	 * \param messages a list of messages to be send to axigen, for example the list of viruses found
     * \param test a struct that will define the virus and/or spam level the filter reports
	 * \return an error or success code as defined above
	 */
	ReturnTypes writeScannedReject(stringArray *messages, virus_spam test);
	/// Writes a response to axigen for a certain mail
	/**
	 * The message was scanned, NOT modified and the implementation, needs to clean it
	 * or take some other action, in any case, needs to MODIFY the mail message
	 * \warning Implementations MUST call only one write function for a given mail
	 * \return an error or success code as defined above
	 */
	ReturnTypes writeNeedsCleaning();
	/// Error callback function
	/** 
	 * Virtual function that will be called when an error occurs
	 * \param errorCode the error code as defined above
	 */
	virtual void error(ReturnTypes errorCode) = 0;
	/// Warning callback function
	/** 
	 * virtual function that will be called when a warning occurs
	 * \param warningCode the error code as defined above
	 */
	virtual void warning(ReturnTypes warningCode) = 0;
	/// Mail processing callback function
	/** 
	 * Virtual function that will be called from the function read() if no error has occurred
	 * it should be used for the processing of the mail
	 * \param filename the path to the mail stream, can be opened for write.
	 * \param rpath the return path from the mail envelope
	 * \param rcpts[] the list of rcpts from the mail envelope
	 * \param allowclean tells if the implementation is allowed to modify the mail. If allowclean
	 * is false, the impl. MUST not modyfy the file but call writeNeedsCleaning as the write function
	 * if it needs to. If allowclean is true, the impl. MUST not call writeNeedsCleaning as the
	 * write function but any other write function.
	 * \return should return 0 on success and -1 on error, it it return -1, the function error()
	 * will be called with the parameter: RT_PROCESSING_ERROR
	 * \retval 0 on success
	 * \retval 1 on error
	 */
	virtual int process(const char *filename, const char *rpath, stringArray *rcpts,
						bool allowclean) = 0;
	/// Cleanup callback function
	/**
	 * Virtual function that will be called from any of the write function if the writing on
	 * the socket was successful. The function should return 0 on success and -1 on error
	 * If the function returns -1, the function error() will be called with the argument set to:
	 * RT_COMPLETE_ERROR. In it returns 0, clean() will be called that will reset the state for the
	 * next mail. If it returns -1, the write function does not call clean() which MUST be called
	 * by the implementation at a later time but before the next call to read()
	 * \return should return 0 on success and -1 on error or if it is wanted the clean()
	 * function not to be called
	 * \retval 0 on success
	 * \retval 1 on error
	 */
	virtual int complete() = 0;

private:
	/**
	 * The implementation of axigen's connector.
	 * \warning Should not be used in implementations
	 */
	void *connector;
	/**
	 * A member used for holding a stream implementation.
	 * \warning Should not be used in implementations
	 */
	void *socket;
	/**
	 * The state of this object
	 * \warning Should not be used in implementations.
	 */
	int state;
	/**
	 * A function that will be called from any write function.
	 * \warning Should not be used in implementations
	 */
	ReturnTypes write(int type,const char *message, virus_spam test);
};

#endif //_LIB_FILTERS_H_
