; Thrust 98, by Richard Mitton ;
;                                                                            ;
; Copyright (c) 1997 Richard Mitton.                                         ;
; This program (and it's source code) are freely                             ;
; distributable, as long as the source code is kept intact.                  ;
;                                                                            ;
; Keys:                                                                      ;
;   Thrust             = LEFT ALT                                            ;
;   Turn clockwise     = LEFT SHIFT                                          ;
;   Turn anticlockwise = RIGHT SHIFT                                         ;
;   Pick up pod        = LEFT CTRL                                           ;
;   Quit               = ESC                                                 ;
;                                                                            ;
; This code should be assembled using TASM 4.1 or above:                     ;
;                                                                            ;
;    tasm /m2 thrust                                                         ;
;    tlink /t /x thrust                                                      ;
;                                                                            ;
; Project ideas:                                                             ;
;   * Baddies (simple guns, lasers, etc)                                     ;
;   * Different gravity on different levels                                  ;
;   * "Hurry up!" message after X amount of time, then a baddy slowly        ;
;     homes in on you.                                                       ;
;   * End-of-game baddy                                                      ;
;                                                                            ;
;;

.model tiny
.code
.386
org 100h

; The structure of a game object
obj struc
   x dd ?            ; 16.16 fixed point
   y dd ?            ; 16.16 fixed point
   xs dd ?           ; 16.16 fixed point
   ys dd ?           ; 16.16 fixed point
   angle db ?        ; 0-256 represents 0-360 degrees
obj ends

; Game constants ;
ROPE_LEN    equ 40   ; length of tractor-beam/rope thing
TURN_SPEED  equ 2    ; the speed you turn at
ACCEL       equ 5    ; acceleration = 1 shr ???
DRAG        equ 511  ; air resistance = 1 - (???/512)
POD_MASS    equ 1    ; mass of pod = 1 SHR ???
GRAVITY     equ 1    ; gravity = ???/128
MAX_DIRTY   equ 10   ; maximum no. of dirty (xor-ed) lines
INIT_LIVES  equ 5    ; how many lives you get at the start
INIT_FUEL   equ 2500 ; how much fuel you get at the start
NUM_LEVELS  equ 6    ; how many levels are there in the game

; These 3 are hard-coded in places
VIRT_WIDTH  equ 128  ; virtual width of screen in bytes (1 byte = 4 pixels)
                     ; must be a multiple of 16
MAP_WIDTH   equ 8    ; width of maps (in 64x32 tiles)
MAP_HEIGHT  equ 16   ; height of maps (in 64x32 tiles)

STAR_ROWS   equ 20   ; number of rows of stars to be added at the top
STAT_HEIGHT equ 8    ; height of status bar. Tiles must fit EXACTLY between
                     ; the top of the screen and the status bar.

; In-game flags ;
FLAG_POD     equ 1   ; If pod is attached
FLAG_SHIELD  equ 2   ; If shield is active
FLAG_XOR     equ 4   ; If using xor drawing mode
FLAG_GETPOD  equ 8   ; If player is picking up pod
FLAG_COLLIDE equ 16  ; If a collision has happened
FLAG_DIRTY   equ 32  ; If using dirty lines mode

; The game itself ;
start:

; On entry, DOS will set up the following registers ;
;   AX=0000  BX=0000  CX=0100  DX=0000  SP=FFFE  BP=0000  SI=0000  DI=0000   ;
;   DS=xxxx  ES=xxxx  SS=xxxx  CS=xxxx  IP=0100   NV UP EI PL NZ NA PO NC    ;
;;

; Clear high scores
   mov di, offset hs_names
   mov cx, 10*10 + 10*2
   rep stosb

   mov dx, 10
   mov di, offset hs_names
init_hs_loop:
   mov cx, 7
   mov si, offset myname
   rep movsb
   add di, 3
   dec dx
   jnz init_hs_loop

   mov di, offset hs_score
   mov cx, 10
   mov ax, 2000
set_hs_loop:
   stosw
   sub ax, 100
   loop set_hs_loop

; Set up system
   mov ax, 0A000h
   mov es, ax        ; es = video ram
; Set 320x200, 256 colours, Mode-X
   mov ax, 13h
   int 10h           ; mode 13h
   mov dx, 3C4h
   mov ax, 604h
   out dx, ax     ; turn off chain-4

   mov dx, 3D4h
   mov ax, 0E317h
   out dx, ax     ; turn off word mode
   mov ax, 14h
   out dx, ax     ; turn off doubleword mode

   mov ax, (VIRT_WIDTH shl 7) + 13h
   out dx, ax

; set split-screen for status bar ;

   ; Turn off pixel panning for split screen area
   mov dx, 3DAh
   in al, dx
   mov dx, 3C0h
   mov al, 30h
   out dx, al
   inc dx
   in al, dx
   or al, 20h
   dec dx
   out dx, al

   ; Tell it where to start the split at
   mov dx, 3D4h
   mov al, 18h
   out dx, al
   inc dx
   mov al, ((200-STAT_HEIGHT)*2 - 1) and 255
   out dx, al
   dec dx
   mov al, 7
   out dx, al
   inc dx
   in al, dx
   and al, not 10h
   or al, ((((200-STAT_HEIGHT)*2 - 1) and 100h) shr 8) shl 4
   out dx, al
   dec dx
   mov al, 9
   out dx, al
   inc dx
   in al, dx
   and al, not 40h
   or al, ((((200-STAT_HEIGHT)*2 - 1) and 200h) shr 9) shl 6
   out dx, al

; Title screen ;
title_screen:

; Clear all of uninitialised data section
   mov di, offset uninit_data
   mov cx, offset end_of_uninit_data - offset uninit_data
   xor al, al
clear_data_loop:
   mov [di], al
   inc di
   loop clear_data_loop

   mov [flags], FLAG_DIRTY

; Generate cosine table
; (credits to "William Wilson / Officium" aka "Wally / Fatal Rage" for
;  getting this into 24 bytes!)
   mov bx, 0798fh                   ; initial position
   xor cx, cx                       ; initial velocity
   mov bp, 512                      ; number required
cos_loop:
   mov ax, -39                      
   imul bx 
   add cx, dx
   add bx, cx
   mov cos[bp], bh 
   dec bp
   jns cos_loop

   call stop_sound
   call decompress_tiles

   call draw_title_screen

   xor ah, ah
   int 16h           ; wait for a key
   cmp al, 27
   je exit_to_dos

   call clear_keybuf

; Set up game
   mov [lives], INIT_LIVES
   mov [fuel], INIT_FUEL
   mov [level_num], 0
; mark
   mov [level], offset level_1 ;+ (16 + 4*2 + 2)*(4-1)
   mov [points], 0
   mov [restart_x], 160
   mov [restart_y], 32*(STAR_ROWS-2)
restart_game:
   mov ax, [restart_x]
   mov bx, [restart_y]
   mov [word ptr ship.x+2], ax
   mov [word ptr ship.y+2], bx
   cmp ax, 160
   ja restart_game_dont_clip_scroll_x
   mov ax, 160
restart_game_dont_clip_scroll_x:
   sub ax, 160
   mov [scroll_x], ax
   shr ax, 2
   mov bx, VIRT_WIDTH * STAT_HEIGHT
   add bx, ax
   call scroll_screen

   xor eax, eax
   mov [ship.xs], eax
   mov [ship.ys], eax
   mov [pod.x], eax
   mov [pod.y], eax
   mov [pod.xs], eax
   mov [pod.ys], eax
   mov [scroll_y], ax
   mov [ship.angle], 192

   mov si, [level]
   call decompress_map

   test [flags], FLAG_POD
   jz no_attach_pod

   mov eax, [ship.x]
   mov ebx, [ship.y]
   add ebx, ROPE_LEN shl 16
   mov [pod.x], eax
   mov [pod.y], ebx
no_attach_pod:

   mov ax, [restart_y]
   sub ax, 96
   mov [map_y], ax
   shr ax, 5

   call draw_level
   call update_status_bar

   mov [sound_freq], 800
   mov [sound_step], 20
   mov [sound_time], 10
   mov dx, 56*4
   mov bx, word ptr [ship.x+2]
   mov cx, word ptr [ship.y+2]
   call do_explosion
   call draw_level

;- Main loop ;
; Here's the main game loop                                                  ;
;;
main_loop:

   call play_sound         ; update sound fx

; Get keyboard input ;
; I use mostly shift and alt keys because the BIOS doesn't use key repeat    ;
; on them, so I won't have to install my own keyboard handler                ;
;;

   mov ah, 12h
   int 16h
   test al, 2
   jz kb_not_left
   sub [ship.angle], TURN_SPEED
kb_not_left:
   test al, 1
   jz kb_not_right
   add [ship.angle], TURN_SPEED
kb_not_right:
   test ah, 2
   jz kb_not_thrust

; Check for enough fuel
   mov bx, [fuel]
   or bx, bx
   jz kb_not_thrust
   dec bx
   mov [fuel], bx

; Thrust the ship forward in the current direction
   mov bl, [ship.angle]
   xor bh, bh
   mov cl, cos[bx]
   mov dl, sin[bx]
   movsx ecx, cl
   movsx edx, dl
   sal ecx, 9-ACCEL      ; convert from 0.7 to 16.16, then shift by ACCEL
   sal edx, 9-ACCEL
   add [ship.xs], ecx
   add [ship.ys], edx

kb_not_thrust:
   push ax
   test ah, 1
   jz turn_off_shield

   ; shield on
   or [flags], FLAG_SHIELD
   test [flags], FLAG_POD
   jnz kb_not_shield

   mov ax, [fuel]
   or ax, ax
   jz no_shield_fuel
   dec ax
no_shield_fuel:
   mov [fuel], ax

   mov ax, word ptr [ship.y+2]
   sub ax, word ptr [pod.y+2]
      ; Sanity check to see if pod is even anywhere
      ; near the ship, cos' get_tension crashes if it isn't.
   or ax, ax
   jns dont_negative_distance
   neg ax
dont_negative_distance:
   cmp ax, 40
   ja turn_off_getpod

   call get_tension        ; see if we are near the pod
   ror eax, 16
   or ax, ax               ; if pod and ship are being pushed apart,
                           ; then AX will be negative, so we are close to
                           ; the pod.
   jns maybe_pickup_pod

   or [FLAGS], FLAG_GETPOD
   mov [sound_freq], 50
   mov [sound_step], 0
   mov [sound_time], 10
   jmp kb_not_shield

turn_off_shield:
   and [flags], not FLAG_SHIELD  ; shield off
turn_off_getpod:
   and [flags], not FLAG_GETPOD
   jmp kb_not_shield

maybe_pickup_pod:
   ; If player was already in GETPOD mode, then switch them to normal
   ; pod mode when they move a certain distance apart
   test [flags], FLAG_GETPOD
   jz kb_not_shield

   or [flags], FLAG_POD
   and [flags], not FLAG_GETPOD
   mov eax, [ship.xs]
   mov [pod.xs], eax    ; set pod xs and ys to be the same as the ships,
   mov eax, [ship.ys]   ; so that it won't wobble around
   mov [pod.ys], eax

kb_not_shield:
   pop ax

   mov ah, 1
   int 16h
   jz no_other_keys_pressed
   xor ah, ah
   int 16h
   cmp al, 27
   je title_screen

no_other_keys_pressed:

; Do some physics ;
   test [flags], FLAG_POD
   jz no_pod_physics

; Here's the physics bit. Work out the tension in the tractor beam (as if
; it was a bit of steel), then pull the ship and pod together by that
; amount.
;
; For all you physics-fans out there, this uses a simplification of the
; Young's modulus formula E=(F*L)/(A*X). By assuming E,L,and A are constants,
; we just get F = k*X, so force is k*extension, where k is a constant.
;
; What this basically amounts to is: The farther the 2 objects move apart,
; the greater the acceleration that pulls them back.
;
; Project idea: Try and work out how this could be applied to the front
;               and back wheels of a car in order to get a extremely
;               cool car-handling system :-)

   call get_tension
   ; eax = tension
   push eax
   call get_podangle
   pop ecx

   mov esi, ecx
   sal esi, POD_MASS
   ; ebx = cos angle, edi = sin angle
   ; ecx = ship accel
   ; esi = pod accel

   mov ebp, 10000h

; First move the ship
   mov eax, ebx
   imul ecx
   idiv ebp
   add [ship.xs], eax
   mov eax, edi
   imul ecx
   idiv ebp
   add [ship.ys], eax

; Then move the pod in the opposite direction
   mov eax, ebx
   imul esi
   idiv ebp
   sub [pod.xs], eax
   mov eax, edi
   imul esi
   idiv ebp
   sub [pod.ys], eax

   add [pod.ys], GRAVITY*65536/128

   mov ebx, [pod.xs]
   mov ecx, [pod.ys]
   add [pod.x], ebx
   add [pod.y], ecx

no_pod_physics:

; Now we've done the tractor beam, do the ship movement ;
   add [ship.ys], GRAVITY*65536/128

; Add the ship's speed to it's position
   mov ebx, [ship.xs]
   mov ecx, [ship.ys]
   add ebx, [ship.x]
   add ecx, [ship.y]

; Keep ship locked on screen ;
   cmp ebx, 16 shl 16       ; 16-pixel safety margin
   jge dont_lock_ship_left
   mov ebx, 16 shl 16
   mov [ship.xs], 0
   mov [pod.xs], 0
dont_lock_ship_left:

   cmp ebx, (VIRT_WIDTH*4 - 16) shl 16
   jl dont_lock_ship_right
   mov ebx, (VIRT_WIDTH*4 - 16) shl 16
   mov [ship.xs], 0
   mov [pod.xs], 0
dont_lock_ship_right:

   cmp ecx, 16 shl 16       ; 16-pixel safety margin
   jge dont_lock_ship_top
   mov ecx, 16 shl 16
   mov [ship.ys], 0
   mov [pod.ys], 0
dont_lock_ship_top:

   cmp ecx, ((MAP_HEIGHT+STAR_ROWS)*32 - 16) shl 16
   jl dont_lock_ship_bottom
   mov ecx, ((MAP_HEIGHT+STAR_ROWS)*32 - 16) shl 16
   mov [ship.ys], 0
   mov [pod.ys], 0
dont_lock_ship_bottom:

   mov [ship.x], ebx
   mov [ship.y], ecx

; Put on some air resistance
   mov si, offset ship
   call do_drag
   mov si, offset pod
   call do_drag

; start screen redraw ;
   call update_status_bar
   call vsync
   call remove_dirty_lines

; scroll the screen ;
   mov cx, word ptr [ship.x+2]
   sub cx, 160

   ; Don't let screen scroll off left edge of game map
   or cx, cx
   jg no_screen_left_lock
   xor cx, cx
no_screen_left_lock:

   ; Don't let screen scroll off right edge of game map
   cmp cx, VIRT_WIDTH*4 - 320
   jl no_screen_right_lock
   mov cx, VIRT_WIDTH*4 - 320
no_screen_right_lock:

   mov [scroll_x], cx

   mov ax, VIRT_WIDTH
   imul [scroll_y]
   mov bx, ax
   add bx, VIRT_WIDTH * STAT_HEIGHT

   call set_hpp
   shr cx, 2
   add bx, cx
   call scroll_screen

   mov bx, word ptr [ship.y+2]
   sub bx, 96

   ; Don't let screen scroll off top edge of game map
   or bx, bx
   jg no_screen_top_lock
   xor bx, bx
no_screen_top_lock:

   ; Don't let screen scroll off bottom edge of game map
   cmp bx, (MAP_HEIGHT+STAR_ROWS)*32 - (200-STAT_HEIGHT)
   jl no_screen_bottom_lock
   mov bx, (MAP_HEIGHT+STAR_ROWS)*32 - (200-STAT_HEIGHT)
no_screen_bottom_lock:

   mov ax, bx
   push bx
   shr ax, 5
   cmp ax, [tile_y]
   mov [tile_y], ax
   jne maybe_draw_new_tiles

   mov dx, [map_ody]
   shr dx, 15
   mov ax, bx
   sub ax, [map_y]
   shr ax, 15
   cmp ax, dx        ; if SignOf(ax) = SignOf(dx) then don't draw new tiles
   je end_of_map_scrolling

maybe_draw_new_tiles:

   cmp bx, [map_y]
   jg scroll_down    ; Draw tiles and move down
   jl scroll_up      ; Draw tiles and move up
   jmp end_of_map_scrolling  ; Move but don't draw any tiles

scroll_down:
   mov di, [scroll_y]
   and di, not 31
   add di, STAT_HEIGHT
   mov ax, VIRT_WIDTH
   imul di
   mov di, ax
   push cx di
   shr bx, 5
   sub di, VIRT_WIDTH*32
   add bx, ((200-STAT_HEIGHT) shr 5)
   call draw_map_row
   pop di
   cmp di, VIRT_WIDTH*((200-STAT_HEIGHT)+32*2)
   ja scroll_down_bottom_clip
   add di, VIRT_WIDTH*((200-STAT_HEIGHT)+32)
   call draw_map_row
scroll_down_bottom_clip:
   pop cx
   jmp end_of_map_scrolling

scroll_up:
   mov di, [scroll_y]
   and di, not 31
   add di, STAT_HEIGHT
   mov ax, VIRT_WIDTH
   imul di
   mov di, ax
   push cx di
   shr bx, 5
   sub di, VIRT_WIDTH*32
   call draw_map_row
   pop di
   cmp di, VIRT_WIDTH*((200-STAT_HEIGHT)+32*2)
   ja scroll_up_top_clip
   add di, VIRT_WIDTH*((200-STAT_HEIGHT)+32)
   call draw_map_row
scroll_up_top_clip:
   pop cx
   jmp end_of_map_scrolling

end_of_map_scrolling:
   pop bx
   push bx
   sub bx, [map_y]
   mov [map_ody], bx
   add bx, [scroll_y]
   cmp bx, 32
   jge no_lower_wrap
   add bx, (200-STAT_HEIGHT) + 32*2
no_lower_wrap:
   cmp bx, (200-STAT_HEIGHT) + 32*2 + 32
   jl no_upper_wrap
   sub bx, (200-STAT_HEIGHT) + 32*2
no_upper_wrap:
   mov [scroll_y], bx

   pop bx
   mov [map_y], bx

   call draw_stuff

; check for crossing restart points ;
   mov si, [level]
   add si, MAP_WIDTH*MAP_HEIGHT/8 + 2     ; skip map, etc
   mov cx, 4
check_restart_points:
   xor ah, ah
   lodsb
   mov bl, al
   lodsb
   mov dx, word ptr [ship.y+2]
   shr dx, 5
   ; bx=restart_x, ax=restart_y, dx=ship_y
   cmp dx, ax
   jne check_next_restart_point

   shl bx, 6
   shl ax, 5
   add bx, 32
   mov [restart_x], bx
   mov [restart_y], ax

check_next_restart_point:
   loop check_restart_points

; check for finishing the level ;

   mov ax, word ptr [ship.y+2]
   cmp ax, 32
   jge havent_finished_level

; Make ship (and pod) 'explode' as they go into hyperspace
   call remove_dirty_lines
   mov [sound_freq], 400
   mov [sound_step], -2
   mov [sound_time], 30
   mov bx, word ptr [ship.x+2]
   mov cx, word ptr [ship.y+2]
   mov dx, 56*4
   call do_explosion

   test [flags], FLAG_POD
   jz no_explode_pod_finish_level

   ; Explode pod as well
   mov bx, word ptr [pod.x+2]
   mov cx, word ptr [pod.y+2]
   mov dx, 56*4
   call do_explosion
no_explode_pod_finish_level:

; wait for 2 seconds
   mov cx, 140
finish_level_wait_loop:
   call vsync
   push cx
   call play_sound
   pop cx
   loop finish_level_wait_loop

   call clear_vidmem
   call update_status_bar

   ; reset screen to top
   mov bx, VIRT_WIDTH * STAT_HEIGHT
   call scroll_screen
   xor cx, cx
   call set_hpp

   ; display scores
   test [flags], FLAG_POD
   jz no_pod_bonus         ; mission failed, so no points

   mov bx, [fuel]
   add [points], bx        ; add fuel left to points

   ; display fuel left
   mov di, 40*VIRT_WIDTH + ((80-11*2)/2)
   mov si, offset fuelstr
   mov cx, 4
   call print_str
   add di, 4
   mov bx, [fuel]
   call print_dec

   ; display lives left
   mov di, 50*VIRT_WIDTH + ((80-11*2)/2)
   mov si, offset lives_str
   mov cx, 5
   call print_str
   add di, 4
   mov ax, [lives]
   inc al
   call print_char
   mov ax, [lives]
   shl ax, 7         ; 128 points per life
   add [points], ax

   mov di, 70*VIRT_WIDTH + ((80-15*2)/2)
   mov si, offset pod_str
   mov cx, 15
   call print_str

   cmp [level_num], NUM_LEVELS-1
   jne mission_success

   mov di, 90*VIRT_WIDTH + ((80-7*2)/2)
   mov si, offset won_str
   mov cx, 7
   call print_str

   jmp mission_success
no_pod_bonus:
   mov di, 70*VIRT_WIDTH + ((80-14*2)/2)
   mov si, offset failed_str
   mov cx, 14
   call print_str
mission_success:
   call update_status_bar  

   call clear_keybuf
   mov cx, 140
wait_at_end_of_mission_loop:
   call vsync
   loop wait_at_end_of_mission_loop

   inc [level_num]
   cmp [level_num], NUM_LEVELS
   jge enter_highscore_table

; set up next level
   mov [fuel], INIT_FUEL
   mov [restart_x], 160
   mov [restart_y], 32*(STAR_ROWS-2)
   mov [flags], FLAG_DIRTY
   add [level], MAP_WIDTH*MAP_HEIGHT/8 + 4*2 + 2
   
   jmp restart_game
havent_finished_level:

; check for collisions ;
   test [flags], FLAG_COLLIDE
   jz main_loop

; They collided with something

   call remove_dirty_lines

   mov [sound_freq], 400
   mov [sound_step], -20
   mov [sound_time], 10
   mov bx, word ptr [ship.x+2]
   mov cx, word ptr [ship.y+2]
   mov dx, 44*4
   call do_explosion
   test [flags], FLAG_POD
   jz no_explode_pod

   ; Explode pod as well
   mov bx, word ptr [pod.x+2]
   mov cx, word ptr [pod.y+2]
   mov dx, 44*4
   call do_explosion

no_explode_pod:
   mov cx, 50
dead_wait_loop:
   call vsync         ; wait 50 frames, at 70fps = 0.6ish seconds
   loop dead_wait_loop

   and [flags], FLAG_POD  ; keep pod
   or [flags], FLAG_DIRTY
   dec [lives]
   jnz restart_game

; They are dead
; Enter name into high score table ;
enter_highscore_table:

   ; Find a place in the high score table
   mov bx, [points]
   xor si, si
   mov cx, 10
find_hs_loop:
   cmp bx, [hs_score + si]
   jge found_hs
   add si, 2
   loop find_hs_loop

   ; They didn't make it. They're not very good, are they?
   jmp title_screen

found_hs:
   ; BX=score, SI=offset of 1st entry to be shifted
   mov cx, 9*2
   sub cx, si      ; cx = no. of entries to be shifted*2

; shift all the highscores below this one down
   mov ax, 5
   imul cx
   mov cx, ax
   mov dx, si
   mov si, 10*10-1   ; SI = offset of last byte to be shifted,
                     ; CX = bytes to be moved

shift_hs_name_loop:
   mov al, [hs_names+si-10]
   mov [hs_names+si], al
   dec si
   loop shift_hs_name_loop

   mov cx, 9*2
   sub cx, dx
   shr cx, 1
   mov si, 9*2
shift_hs_score_loop:
   mov ax, [hs_score+si-2]
   mov [hs_score+si], ax
   sub si, 2
   loop shift_hs_score_loop

   mov si, dx
   mov [hs_score+si], bx
   mov ax, 5
   imul si
   push ax
   ; AX is now position in highscore names table of this highscore
   ; Prompt the user for their name

   ; clear out old high score
   mov cx, 10
   mov si, ax
   xor al, al
clear_hs_loop:
   mov [hs_names+si], al
   inc si
   loop clear_hs_loop

   call draw_title_screen
   pop si

   mov di, si
   shl di, 7      ; di*VIRT_WIDTH
   add di, ((80-20*2)/2) + 70*VIRT_WIDTH

   push si di
   ; Put line under the name
   mov cx, si
   add cx, 79
   mov di, cx

   and [flags], not FLAG_XOR
   mov bx, ((320-20*8)/2)
   mov si, ((320-20*8)/2) + 8*10
   mov bp, 15
   call line
   pop di si

   add si, offset hs_names
   mov cx, 10
get_hs_name_loop:
   xor ah, ah
   int 16h
   cmp al, 13        ; has user pressed enter?
   je finish_get_hs_name
   cmp al, 27        ; has user pressed escape?
   je finish_get_hs_name
   cmp al, 32        ; has user pressed space?
   jne not_space

   dec cx
   mov [si], ch
   inc si
   add di, 2
   jmp get_hs_name_loop
not_space:

   cmp al, 8         ; has user pressed backspace?
   jne not_backspace

   cmp cl, 10
   je not_backspace     ; can't backspace if this is the 1st letter

   inc cx
   dec si
   sub di, 2
   mov [si], ch         ; clear last letter
   push ax
   xor al, al
   call print_char
   pop ax

not_backspace:
   ; convert to upper case
   cmp al, 'a'
   jb not_lowercase
   cmp al, 'z'
   ja not_lowercase
   sub al, 'a'-'A'
not_lowercase:

; check for invalid letters
   cmp al, '0'
   jb get_hs_name_loop
   cmp al, 'Z'
   ja get_hs_name_loop
   cmp al, '9'
   jbe letter_in_range
   cmp al, 'A'
   jb get_hs_name_loop

letter_in_range:
   sub al, '0'-1
   cmp al, 'A'-('0'-1)
   jb dont_adjust_letter
   sub al, 'A'-'9'-1
dont_adjust_letter:

   call print_char
   add di, 2
   mov [si], al
   inc si
   loop get_hs_name_loop

finish_get_hs_name:
   call clear_keybuf
   jmp title_screen

exit_to_dos:
; Exit to DOS ;

   call stop_sound

   mov ax, 3
   int 10h
   ret

; draw_title_screen ;
; Draws the title screen and high score table                                ;
; Corrupts everything (and also game map, etc)                               ;
;;
draw_title_screen:
   mov bx, VIRT_WIDTH * STAT_HEIGHT
   call scroll_screen
   call clear_vidmem

; draw some stars in the background
   mov si, offset level_1
   call decompress_map     ; use this to generate stars at top of map
   mov [map_y], 0
   call draw_level         ; draw it

; bung on some text
   mov di, ((80-9*2)/2) + 30*VIRT_WIDTH
   mov cx, 9
   mov si, offset titlestr
   call print_str

   mov bx, ((80-9*2)/2)*4
   mov si, 320 - ((80-9*2)/2)*4 - 3
   mov cx, 40
   mov di, cx
   mov bp, 15
   call line         ; underline the title

; print "Highscores"
   mov di, ((80-10*2)/2) + 50*VIRT_WIDTH
   mov cx, 10
   mov si, offset highstr
   call print_str

; copyright message
   mov di, ((80-30*2)/2)
   mov cx, 30
   mov si, offset copystr
   call print_str

; Draw high-score table
   mov di, ((80-20*2)/2) + 70*VIRT_WIDTH
   xor dx, dx
   mov si, offset hs_names
highscore_loop:
   mov cx, 10
   call print_str

   ; print the score
   add di, 5*2
   mov bx, dx
   shl bx, 1
   mov bx, [hs_score+bx]
   call print_dec
   add di, 5*2 + VIRT_WIDTH*10 - 20*2
   inc dx
   cmp dl, 10
   jne highscore_loop
   ret         


; draw all the stuff (ship, pod, tractor beam, etc) ;
draw_stuff:
   or [flags], FLAG_XOR
   and [flags], not FLAG_COLLIDE

   mov [shape_colour], 15
   test [flags], FLAG_SHIELD
   jz no_shield_colour
   mov ax, [fuel]
   and al, 15
   mov [shape_colour], ax
no_shield_colour:

   mov si, offset ship_data
   mov bp, offset ship
   call draw_shape

   mov [shape_colour], 15
   mov si, offset pod_data
   mov bp, offset pod
   call draw_shape

   test [flags], (FLAG_POD or FLAG_GETPOD)
   jz no_draw_pod_line

   mov bx, word ptr [ship.x+2]
   mov cx, word ptr [ship.y+2]
   mov si, word ptr [pod.x+2]
   mov di, word ptr [pod.y+2]
   call add_dirty_line
   call map_line
no_draw_pod_line:

   ret

; add_dirty_line ;
; Adds a dirty line to the list.                                             ;
; Line coords are (BX,CX) - (SI,DI), colour is BP                            ;
; This dont corrupt nuffin' mate, coz its well cool.                         ;
;;
add_dirty_line:
   push ax di
   test [flags], FLAG_DIRTY
   jz dirty_list_full

   mov ax, [num_dirty]
   cmp ax, MAX_DIRTY
   jge dirty_list_full

   xchg ax, di
   shl di, 1
   mov [dirty_x1+di], bx
   mov [dirty_y1+di], cx
   mov [dirty_x2+di], si
   mov [dirty_y2+di], ax
   mov [dirty_c+di], bp

   inc [num_dirty]
 dirty_list_full:
   pop di ax
   ret

; remove_dirty_lines ;
; Draws over all the lines previously drawn with XOR on, so they disappear   ;
; Corrupts everything                                                        ;
;;
remove_dirty_lines:
   or [flags], FLAG_XOR          ; set xor mode on
   mov di, [num_dirty]
   or di, di         ; If none left, then stop
   jz remove_dirty_lines_finish

   dec di
   mov [num_dirty], di

   shl di, 1
   mov bp, [dirty_c+di]
   mov bx, [dirty_x1+di]
   mov cx, [dirty_y1+di]
   mov si, [dirty_x2+di]
   mov di, [dirty_y2+di]
   call map_line

   jmp remove_dirty_lines      ; and repeat

remove_dirty_lines_finish:
   ret

; do_explosion ;
; Animates an on-screen explosion at map coords (BX,CX)                      ;
; The screen will need redrawing afterwards.                                 ;
; DX is middle colour of explosion * 4                                       ;
;;
do_explosion:
   and [flags], not FLAG_XOR
   xor ax, ax
   sub cx, [map_y]
   add cx, [scroll_y]
   add cx, STAT_HEIGHT
   explosion_loop:
      mov bp, dx
      shr bp, 2
      pusha
      add bx, ax
      call plot
      popa
      pusha
      sub bx, ax
      call plot
      popa
      pusha
      add cx, ax
      call plot
      popa
      pusha
      sub cx, ax
      call plot
      call vsync
      call play_sound
      popa
      dec dx
      inc ax
      cmp ax, 4*4
      jne explosion_loop
   ret

; get_podangle ;
; Returns the angle of the pod from the ship (cosine in EBX, sine in EDI)    ;
; Corrupts EAX, ESI, CX, EDX                                                 ;
;;
get_podangle:
   ; work out sqrt(dx^2 + dy^2)

   mov ebp, 10000h
   mov eax, [pod.x]
   sub eax, [ship.x]
   imul eax
   idiv ebp
   mov ebx, eax
   mov eax, [pod.y]
   sub eax, [ship.y]
   imul eax
   idiv ebp
   add ebx, eax
   call sqrt
; We now have distance from ship-to-pod in esi. Use it to normalise
   mov eax, [pod.x]
   sub eax, [ship.x]
   imul ebp
   idiv esi
   mov ebx, eax

   mov eax, [pod.y]
   sub eax, [ship.y]
   imul ebp
   idiv esi
   mov edi, eax
   ret

; get_tension ;
; Calculates the tension in the beam between the ship and the pod            ;
; Returns in EAX
; Corrupts EBX, EBP, ESI, CX, EDX
;;
get_tension:
   mov ebp, 10000h
   mov eax, [ship.x]
   sub eax, [pod.x]
   imul eax
   idiv ebp
   mov ebx, eax
   mov eax, [ship.y]
   sub eax, [pod.y]
   imul eax
   idiv ebp
   add ebx, eax
   call sqrt

   sub esi, ROPE_LEN shl 16
   mov eax, esi
   sar eax, POD_MASS
   sar eax, 3     ; apply some drag
   ret

; draw_tile ;
; Draws tile pointed to by SI onto screen at DI                              ;
; NOTE: This assumes virtual width is a multiple of 16                       ;
;       and DI is on a 4-pixel boundary                                      ;
; Corrupts almost everything
;;
draw_tile:
   mov bh, 32

 draw_tile_y_loop:
   mov bl, 1
   draw_tile_plane_loop:
      mov dx, 3C4h
      mov al, 02h
      mov ah, bl
      out dx, ax     ; set write plane

      push di
      mov cx, 64/8
      rep movsw
      pop di

      shl bl, 1
      cmp bl, 16
      jl draw_tile_plane_loop

   add di, VIRT_WIDTH
   dec bh
   jnz draw_tile_y_loop
   
   ret

; draw_map_row ;
; Draws row BX of the map at screen offset DI                                ;
; Corrupts AX, CX, SI, DI                                                    ;
;;
draw_map_row:
   mov si, bx
   shl si, 3      ; si = si*MAP_WIDTH
   add si, offset map
   mov cx, MAP_WIDTH
  draw_row_loop:
      lodsb
      push bx cx si di
      cmp al, 1
      jne dont_recalc_stars

      ; Generate tile_stars on the fly by picking a section out of the
      ; big stars picture, and copying it across. This will generate
      ; stars that don't look tile-based, yet are not random.

      push ax
      and cx, 1      ; wrap to big tile width
      shl cx, 6+5    ; x*64*32

      mov si, bx     ; y
      and si, 3      ; wrap to fit in big tile
      shl si, 6+5+1  ; y*64*32*2, because 2 tiles per row in big tile
      add si, cx
      add si, offset tile_big_stars
      mov bx, offset tile_stars

      mov cx, 64*32/2
   make_star_tile_loop:
      lodsw
      mov [bx], ax
      add bx, 2
      loop make_star_tile_loop

      pop ax

   dont_recalc_stars:
      xor ah, ah
      shl ax, 11
      mov si, ax
      add si, offset uncompressed_tiles
      call draw_tile
      pop di si cx bx

      add di, 64/4
      loop draw_row_loop

   ret

; update_status_bar ;
; Redraws the status bar (at offset 0 of the video memory)                   ;
;;
update_status_bar:
; set write plane
   mov dx, 3C4h
   mov ax, 0F02h
   out dx, ax

   xor di, di
   mov cx, VIRT_WIDTH*STAT_HEIGHT
   xor al, al
   rep stosb
   mov di, 2
   mov cx, 4
   mov si, offset fuelstr
   call print_str
   add di, 2
   mov bx, [fuel]
   call print_dec

   add di, 20
   mov cx, 5
   mov si, offset lives_str
   call print_str
   add di, 2
   mov ax, [lives]
   inc al
   call print_char

   add di, 8
   mov cx, 6
   mov si, offset points_str
   call print_str
   add di, 2
   mov bx, [points]
   call print_dec
         
; set write plane
   mov dx, 3C4h
   mov ax, 0F02h
   out dx, ax
   xor di, di
   mov ax, 0101h
   mov cx, 40
   rep stosw
   ret

; draw_level ;
; Draws the current level onto the screen                                    ;
; Corrupts almost everything                                                 ;
;;
draw_level:
   mov bx, [map_y]
   shr bx, 5
   mov cx, ((200-STAT_HEIGHT) shr 5) + 1
   mov di, VIRT_WIDTH * STAT_HEIGHT

  draw_level_loop:
      push cx di
      call draw_map_row
      pop di cx
      add di, VIRT_WIDTH*((200-STAT_HEIGHT)+32*2)  ; Replicate on other screen

      push cx di
      call draw_map_row
      pop di cx
      sub di, VIRT_WIDTH*((200-STAT_HEIGHT)+32)
      inc bx
      loop draw_level_loop

   ret

; decompress_tiles ;
; Decompresses the tiles into a separate memory area                         ;
; Corrupts AL, BL, CX, SI, DI
;;
decompress_tiles:
   mov si, offset start_of_tiles
   mov di, offset uncompressed_tiles + 64*32*2
   ; leave room for black tile and stars tile

   mov cx, offset end_of_tiles - offset start_of_tiles
  decompress_tile_loop:
     lodsb
     mov bl, al
     shr bl, 4
     and al, 15         ; al = 1st pixel, bl = 2nd pixel
     add al, 16         ; shift to grey range
     add bl, 16
     mov [di], al
     inc di
     mov [di], bl
     inc di
     loop decompress_tile_loop

; Now generate some random stars
   mov si, offset tile_ground    ; Random data
   xor di, di
   mov bp, 30    ; number of stars
   mov ebx, 07080F0Fh      ; 4 colours - light grey, dark grey, white, white
 gen_stars_loop:
   lodsb
   mov cl, al
   and cl, 3
   shl cl, 3
   ror ebx, cl            ; BL is now randomly one of the 4 colours

   mov ah, al
   lodsb
   add di, ax           ; Move to random place
   and di, 128*128-1    ; Wrap to picture
   mov [tile_big_stars+di], bl

   imul di     ; Randomise a bit more

   dec bp
   jnz gen_stars_loop

   ret

; make_slanted_tile ;
; Makes a new sloping tile in DI from the ground tile.                       ;
; Takes initial left pos in AX, initial right pos in BX,                     ;
; change in left pos per row in SI, change in right pos per row in BP        ;
;;
make_slanted_tile:
   xor dx, dx
 slant_loop_1:
   mov cx, ax
   slant_loop_2:
      ; move from (CX, DX) to DI
      push ax bx
      mov bx, dx
      shl bx, 6      ; Convert x,y into the planar layout the tiles are in
      mov ax, cx
      and ax, 3
      shl ax, 4
      add bx, ax
      mov ax, cx
      shr ax, 2
      add bx, ax        ; Offset of pixel is now in BX
      mov al, [tile_ground + bx]   ; Move from source to dest
      mov [di + bx], al    
      pop bx ax

      inc cx            ; Next pixel
      cmp cx, bx
      jle slant_loop_2

   ; Next row
   add ax, si
   add bx, bp
   inc dx
   cmp dx, 32
   jne slant_loop_1
   ret

; decompress_map ;
; Decompress the map in SI into the global map area                          ;
;;
decompress_map:
   mov di, offset map
   mov al, 1                        ; Select "stars" tile
   mov cx, MAP_WIDTH*STAR_ROWS      ; First, add rows of stars at the top
 star_loop:
   mov [di], al
   inc di
   loop star_loop

   mov dx, MAP_WIDTH*MAP_HEIGHT/8   ; 8 tiles per byte
 map_loop:
   lodsb
   mov cx, 8
   decompress_map_tile_loop:
      mov bl, al
      and bl, 1
      shr al, 1
      shl bl, 1
      mov [di], bl
      inc di
      loop decompress_map_tile_loop

   dec dx
   jnz map_loop

; Get pod coords
   lodsb
   xor ah, ah
   shl ax, 6      ; ax*64
   add ax, 32
   mov bx, ax
   lodsb
   xor ah, ah
   shl ax, 5      ; ax*32
   add ax, 20
   mov word ptr [pod.x+2], bx
   mov word ptr [pod.y+2], ax

   ret

; calc_poly_offset ;
; Helper for draw_shape - returns coords of a point in BX,CX                 ;
;;
calc_poly_offset:
   lodsw
   movzx cx, ah

   xor ah, ah
   add al, [bp.angle]
   movzx di, al
   mov al, cos[di]
   cbw
   imul cx
   sar ax, 7
   mov bx, ax
   add bx, word ptr [bp.x+2]

   mov al, sin[di]
   cbw
   imul cx
   sar ax, 7
   mov cx, ax
   add cx, word ptr [bp.y+2]
   ret

; draw_shape ;
; Draws a polygon pointed to by SI, with object in BP                        ;
; Corrupts almost everything                                                 ;
;;
draw_shape:
   mov ax, word ptr [bp.y+2]
   mov bx, [map_y]
   cmp ax, bx
   jnb no_top_clip_drawshape
   ret
no_top_clip_drawshape:

   add bx, (200-STAT_HEIGHT)
   cmp ax, bx
   jb no_bottom_clip_drawshape
   ret

no_bottom_clip_drawshape:

   call calc_poly_offset
   push bx cx
   call calc_poly_offset
   push bx cx
   call calc_poly_offset
   push bx cx
   call calc_poly_offset
   push bx cx

   mov bp, sp
   mov cx, [bp+12]    ; Put first point back on the end again
   mov bx, [bp+14]    ; so that the polygon will be closed

   mov al, 4
 draw_shape_loop:
   mov bp, sp     ; Because we can't directly reference via SP
   mov di, [bp]
   mov si, [bp+2]
   push ax
   mov bp, [shape_colour]
   call add_dirty_line
   call map_line
   pop ax cx bx         ; Fetch next co-ords
   dec al
   jnz draw_shape_loop

   ret

; do_drag ;
; Applies air resistance to object in SI                                     ;
; Corrupts EAX, EBX, ECX, EDX                                                ;
;;
do_drag:
   mov eax, DRAG
   imul [si.xs]
   sar eax, 9
   mov [si.xs], eax
   mov eax, DRAG
   imul [si.ys]
   sar eax, 9
   mov [si.ys], eax
   ret

;;
; Miscellaneous routines                                                     ;
;;

; play_sound ;
; Plays a sound out of the PC speaker.                                       ;
; sound_freq is frequency                                                    ;
; Corrupts AX, BX, ECX, EDX                                                  ;
;;
play_sound:
   mov bx, [sound_freq]       ; frequency (hz)
   mov ax, [sound_time]
   or ax, ax
   jz finish_play_sound
   
   mov al, 0b6h
   out 43h, al

   movzx ecx, bx
   mov eax, 1193046
   cdq
   idiv ecx

   out 42h, al
   ror ax, 8
   out 42h, al
   in al, 61h
   or al, 3
   out 61h, al       ; output to speaker

   ; increase sound for next frame
   mov bx, [sound_step]
   add [sound_freq], bx
   dec [sound_time]
   jnz finish_play_sound

stop_sound:
   in al, 61h
   and al, 11111100b
   out 61h, al
finish_play_sound:
   ret

; clear_keybuf ;
; Clears the keyboard buffer
; Corrupts AX
;;
clear_keybuf:
   mov ah, 1
   int 16h
   jz keybuf_empty
   xor ah, ah
   int 16h
   jmp clear_keybuf

keybuf_empty:
   ret

; sqrt ;
; Utterly cool square-root routine using Newton's iterative method           ;
; Takes number in EBX, returns in ESI                                        ;
; Corrupts CX, EAX, EDX                                                      ;
; You have no idea how much I cursed at this while writing it ;-)            ;
;;
sqrt:
   test ebx, ebx
   jz sqrt_zero      ; if zero, quit now

   mov esi, ebx
   shr esi, 4
   mov cx, 20          ; number of iterations
 sqrtloop:
   mov eax, ebx
   xor edx, edx      ; clear edx for divide
   idiv esi
   add esi, eax
   shr esi, 1
   loop sqrtloop
   shl esi, 8        ; restore back to fixed point
 sqrt_zero:
   ret

; print_char ;
; Prints character in AL onto screen at DI.                                  ;
; Colour is always 15 (white)                                                ;
; DI must be on a 4-pixel boundary                                           ;
; AL can be 0 (space), 11->37 (A->Z) or 1->10 (0->9)                                     ;
; Corrupts nothing                                                           ;
;;
print_char:
   pusha
   mov si, offset font_dat
   xor ah, ah
   shl ax, 3      ; 1 letter is 8 bytes
   add si, ax

   mov dh, 8
print_char_y_loop:
   lodsb
   mov dl, al
   xor cx, cx
   print_char_x_loop:
      ; set write plane
      push di dx
      mov dx, 3C4h
      mov ax, 0102h
      shl ah, cl
      out dx, ax
      pop dx
      
      mov al, dl
      and al, 1
      shl al, 3
      or al, al
      jz no_add_font_colour_left
      add al, 7
    no_add_font_colour_left:
      stosb
      shr dl, 1

      mov al, dl
      and al, 1
      shl al, 3
      or al, al
      jz no_add_font_colour_right
      add al, 7
    no_add_font_colour_right:
      stosb
      shr dl, 1
      pop di
      inc cx
      cmp cl, 4
      jne print_char_x_loop

   add di, VIRT_WIDTH
   dec dh
   jnz print_char_y_loop
   popa
   ret

; print_str ;
; Prints string in SI on screen at DI. CX is no. of letters                  ;
; Corrupts CX, SI. DI is advanced for each letter                            ;
;;
print_str:
   lodsb
   call print_char
   add di, 2
   loop print_str
   ret

; print_dec ;
; Prints the contents of BX on screen (in decimal)                           ;
; Prints at offset DI                                                        ;
; Corrupts nothing                                                           ;
;;
print_dec:
   pusha
   mov cx, 5
   add di, 5*2
 print_dec_loop:
   mov ax, bx
   mov bx, 10
   cwd
   idiv bx
   xchg ax, bx
   mov al, dl
   and al, 15
   inc al
   call print_char
   sub di, 2
   loop print_dec_loop

   popa
   ret

; scroll_screen ;
; Sets the video memory pointer to BX                                        ;
; Corrupts AX, DX                                                            ;
;;
scroll_screen:
   mov dx, 3D4h
   mov ax, 0Ch
   out dx, ax
   inc dx
   mov al, bh
   out dx, ax
   dec dx
   mov ax, 0Dh
   out dx, ax
   inc dx
   mov al, bl
   out dx, ax
   ret

; set_hpp ;
; Sets the horizontal pan register to scroll horizontally to CX              ;
; Corrupts AX, DX                                                            ;
;;
set_hpp:
   ; adjust hpp register
   mov dx, 3C0h
   mov al, 33h
   out dx, al
   mov al, cl
   and al, 3
   shl al, 1
   out dx, al
   ret

; clear_vidmem ;
; Clears the entire video memory to black                                    ;
; Corrupts AX, CX, DX, DI                                                    ;
;;
clear_vidmem:
   mov dx, 3C4h
   mov ax, 0F02h
   out dx, ax     ; enable all planes
   xor ax, ax
   mov cx, 32768
   xor di, di
   rep stosw      ; clear video memory
   ret

; vsync ;
; Waits for the VGA to go into vertical retrace                              ;
; Corrupts AL, DX                                                            ;
;;
vsync:
   mov dx, 3dah
 vsync_out:
   in al, dx
   test al, 8
   jz vsync_out
 vsync_in:
   in al, dx
   test al, 8
   jnz vsync_in
   ret

; map_line ;
; Draws a line in color BP from (BX,CX) to (SI,DI)                           ;
; Assumes co-ordinates are MAP co-ordinates                                  ;
; Remember to set xor mode first!                                            ;
; Corrupts pretty much everything                                            ;
;;
map_line:
   sub cx, [map_y]
   add cx, [scroll_y]
   add cx, STAT_HEIGHT
   sub di, [map_y]
   add di, [scroll_y]
   add di, STAT_HEIGHT
   ; fall-through

; line ;
; Draws a line in color BP from (BX,CX) to (SI,DI)                           ;
; Co-ords are not relative to scrolling area                                 ;
; Remember to set xor mode first!                                            ;
; Corrupts pretty much everything                                            ;
;;
line:
   cmp bx, si
   jne line_is_not_one_dot
   cmp cx, di
   jne line_is_not_one_dot
   call plot
   ret

 line_is_not_one_dot:
   push si di
   sub si, bx
   jns line_x_positive
   neg si
 line_x_positive:
   sub di, cx
   jns line_y_positive
   neg di
 line_y_positive:

   cmp si, di
   jge line_horizontal

 ; Line is more vertical than horizontal
   pop di si
   sub si, bx
   sub di, cx
   jns line_no_vert_swap

 ; Line needs reversing
   add bx, si
   neg si
   add cx, di
   neg di
 line_no_vert_swap:

   mov ax, si
   shl eax, 16
   shl ebx, 16
   movzx edi, di
   cdq
   idiv edi
   add ebx, 32768
 line_vert_loop:
   push ax ebx cx di
   shr ebx, 16
   call plot
   pop di cx ebx ax
   inc cx
   add ebx, eax
   dec di
   jns line_vert_loop
   ret

 line_horizontal:
 ; Line is more horizontal than vertical
   pop di si
   sub di, cx
   sub si, bx
   jns line_no_horiz_swap

 ; Line needs reversing
   add bx, si
   neg si
   add cx, di
   neg di
 line_no_horiz_swap:

   mov ax, di
   shl eax, 16
   shl ecx, 16
   movzx esi, si
   cdq
   idiv esi
   add ecx, 32768
 line_horiz_loop:
   push ax bx ecx si
   shr ecx, 16
   call plot
   pop si ecx bx ax
   inc bx
   add ecx, eax
   dec si
   jns line_horiz_loop

   ret   

;- plot -;
; Plots a point at (bx,cx) in color BP                                       ;
; Does not take scrolling into account                                       ;
; Remember to set xor mode first!                                            ;
; Corrupts AX, BX, CX, DX, DI                                                ;
;;
plot:
; offset = (y*VIRT_WIDTH+x/4)
   and bx, (VIRT_WIDTH*4)-1       ; wrap to screen
   mov ax, VIRT_WIDTH
   imul cx
   mov cl, bl
   and cl, 3
   shr bx, 2
   add ax, bx
   mov di, ax

; set write plane
   mov dx, 3C4h
   mov ax, 0102h
   shl ah, cl
   out dx, ax

; set read plane (for XOR)
   mov dl, 0CEh      ; dx=3CEh
   mov al, 04h
   mov ah, cl
   out dx, ax

   mov ax, bp
   test [flags], FLAG_XOR
   jz no_plot_xor

   mov ah, es:[di]
   cmp ah, 16
   jb plot_nocollision
   cmp ah, 31
   ja plot_nocollision
   or [flags], FLAG_COLLIDE
plot_nocollision:

   xor al, ah

no_plot_xor:
   stosb
   ret

;- Data section -;
; This is fixed data to be included in the .COM file                         ;
;;

include textdata.inc

; Data for the shape of the ship
; Format: angle, distance
ship_data db   0, 10
          db  86,  5
          db  64,  0
          db 170,  5

; Data for the shape of the pod
; Format: angle, distance
pod_data  db  32,  5
          db  96,  5
          db 160,  5
          db 224,  5

; The maps for each level
; If you don't want 4 restart points, repeat some more than once, and they
; get counted as one.

include map1.inc      ; Ground data - 1-bit per tile, so 8 tiles to a byte.
db 2, 20    ; (x, y) coords of pod (in tiles)
; 4 restart points
db  3, 18    ; map position (x, y)
db  3, 18    ; map position (x, y)
db  3, 18    ; map position (x, y)
db  3, 18    ; map position (x, y)
    
include map2.inc
db 5, 29    ; (x, y) coords of pod (in tiles)
; 4 restart points
db  3, 20    ; map position (x, y)
db  3, 22    ; map position (x, y)
db  2, 25    ; map position (x, y)
db  2, 25    ; map position (x, y)
    
include map3.inc
db 6, 34    ; (x, y) coords of pod (in tiles)
; 4 restart points
db  6, 20    ; map position (x, y)
db  1, 26    ; map position (x, y)
db  1, 26    ; map position (x, y)
db  1, 26    ; map position (x, y)
    
include map4.inc
db 1, 34    ; (x, y) coords of pod (in tiles)
; 4 restart points
db  6, 20    ; map position (x, y)
db  1, 22    ; map position (x, y)
db  5, 12    ; map position (x, y)
db  5, 12    ; map position (x, y)
    
include map5.inc
db 5, 29    ; (x, y) coords of pod (in tiles)
; 4 restart points
db  5, 20    ; map position (x, y)
db  1, 25    ; map position (x, y)
db  2, 32    ; map position (x, y)
db  5, 31    ; map position (x, y)
    
include map6.inc
db 3, 32    ; (x, y) coords of pod (in tiles)
; 4 restart points
db  5, 20    ; map position (x, y)
db  2, 25    ; map position (x, y)
db  4, 31    ; map position (x, y)
db  4, 31    ; map position (x, y)
    

; Tiles that make up the map (the format is 32 rows, each row has 4 planes,
;                             each plane has 16 bytes)
start_of_tiles:
include ground1.inc
end_of_tiles:

include font.inc     ; The font

; Uninitialised section ;
; Because we don't initialise it, is won't actually                          ;
; be put into the .COM file. Isn't TASM wonderful?                           ;
;;
                                                                             
uninit_data:

ship     obj   ?        ; Your ship
pod      obj   ?        ; The pod you have to retrieve
flags    db    ?        ; Set with FLAG_XXX constants
fuel     dw    ?        ; Fuel left
lives    dw    ?        ; Number of lives left
map_y    dw    ?        ; Top of map currently displayed (in pixels)
tile_y   dw    ?        ; Top of map currently displayed (in tiles)
map_ody  dw    ?        ; Previous change in vertical motion (pixels)
scroll_x dw    ?        ; Current horizontal scroll position
scroll_y dw    ?        ; Current vertical scroll position

shape_colour   dw  ?    ; Colour to be used for draw_shape

sound_freq     dw  ?    ; Frequency for play_sound
sound_step     dw  ?    ; Change in frequency per frame
sound_time     dw  ?    ; No. of frames left to play sound for

num_dirty      dw  ?    ; Number of dirty lines currently in use
dirty_x1 dw MAX_DIRTY dup (?) ; 
dirty_y1 dw MAX_DIRTY dup (?) ; X and Y coords of each end of the lines
dirty_x2 dw MAX_DIRTY dup (?) ;
dirty_y2 dw MAX_DIRTY dup (?) ;
dirty_c  dw MAX_DIRTY dup (?) ; Colour of the line that was drawn

level     dw   ?        ; Pointer to current level data
level_num dw   ?        ; Current level number (0 -> max)
points    dw   ?        ; Your current score
restart_x dw   ?        ; Where the player will restart from when they die
restart_y dw   ?        ; (in pixels)

; uncompressed map data
map   db    (MAP_WIDTH*MAP_HEIGHT + MAP_WIDTH*STAR_ROWS) dup (?)

; uncompressed tiles
uncompressed_tiles:
tile_black     db 64*32 dup (?)     ; Blank tile
tile_stars     db 64*32 dup (?)     ; Stars background
tile_ground    db 64*32 dup (?)     ; Normal ground tile     
tile_big_stars db 128*128 dup (?)   ; Lots of stars

end_of_uninit_data:
; These are put here so they don't get cleared every time

hs_names db 10*10 dup (?)  ; The high score names
hs_score dw 10 dup (?)     ; The high scores

cos   db    64*3+1 dup (?)     ; Cosine starts here    (use +1 to correct bug
sin   db    64+256 dup (?)     ; Sine starts here        of sine starting at
                               ;                          the wrong place)

end_of_program:

end start
