/* 
 * FPLOT - fplib.c
 *
 * Copyright (C) 1997 - 2000  Michael C. Ring  
 *
 * Permission to use, copy, and distribute this software and its
 * documentation for any purpose with or without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and 
 * that both that copyright notice and this permission notice appear 
 * in supporting documentation.
 *
 * Permission to modify the software is granted, but not the right to
 * distribute the modified code.  Modifications are to be distributed 
 * as patches to released version.
 *  
 * This software is provided "as is" without express or implied warranty.
 */

/*
 *      this file has 'generic' functions, i.e. they do 
 *      not need access to the global variables
 *
 *      $Log: fplib.c,v $
 *      Revision 1.17  2000/01/14 22:38:32  mike
 *      change array dim to 22 for adivisions also
 *
 *      Revision 1.16  2000/01/14 22:31:04  mike
 *      add 3.6 increment at divisions == 9 to scale_values
 *
 *      Revision 1.15  2000/01/10 04:40:18  mike
 *      added some comments
 *
 *      Revision 1.14  2000/01/09 04:58:03  mike
 *      need to add unistd.h for DOS GCC compile
 *
 *      Revision 1.13  2000/01/09 03:25:09  mike
 *      allow memory_allocate and memory_free to malloc a full
 *      4 byte value if sizeof(int) is 4 bytes
 *
 *      Revision 1.12  2000/01/08 06:10:06  mike
 *      added proto.h and full prototypes
 *
 *      Revision 1.11  1999/01/13 02:01:49  mring
 *      cast ch1 to UINT before << 8
 *
 *      Revision 1.10  1998/11/26 05:59:46  mring
 *      put in new memory_allocate & memory_free
 *
 *      Revision 1.9  1998/10/16 21:17:57  mring
 *      added malloc-free-counters and function
 *
 *      Revision 1.8  1998/04/03 19:52:35  mring
 *      put in new font size look-up table
 *
 *      Revision 1.7  1998/03/16 00:00:23  mring
 *      minor tweak
 *
 *      Revision 1.6  1998/03/15 23:28:22  mring
 *      fixed obscure bug in format_double
 *
 *      Revision 1.5  1998/03/05 03:42:47  mring
 *      update for 0.5a
 *
 *      Revision 1.4  1997/12/05 02:20:18  mring
 *      update for version 0.4a
 *
 *      Revision 1.3  1997/11/27 16:41:45  mring
 *      minor tweak to rescale_divisions
 *
 *      Revision 1.2  1997/11/24 01:30:52  mring
 *      added copyright notice
 *
 *      Revision 1.1  1997/11/23 23:08:15  mring
 *      Initial revision
 */

#include "fplot.h"
#include "proto.h"

#ifdef DOS_GCC_DEFINED
#include <unistd.h>			     /* need proto for 'getpid' */
#endif

/* RCS TRACER */
/*
 * $Id: fplib.c,v 1.17 2000/01/14 22:38:32 mike Exp $
 */
                                     /* font size 14 point scaled 100:1 */
                                     /* from 0x20 to 0x7E (ASCII)       */
static  int     helvetica_size[95] = {
                390, 390, 500, 780, 780, 1248, 936, 312, 466, 466, 546, 820,
                390, 466, 390, 390, 780, 780, 780, 780, 780, 780, 780, 780,
                780, 780, 390, 390, 820, 820, 820, 780, 1422, 936, 936,
                1012, 1012, 936, 856, 1090, 1012, 390, 700, 936, 780, 1168,
                1012, 1090, 936, 1090, 1012, 936, 856, 1012, 936, 1326, 936,
                936, 856, 390, 390, 390, 658, 780, 312, 780, 780, 700, 780,
                780, 390, 780, 780, 312, 312, 700, 312, 1166, 780, 780,
                780, 780, 466, 700, 390, 780, 700, 1012, 700, 700, 700,
                466, 366, 466, 820
                } ;

static  int     gs_malloc_free_counter = 0;

/***********************************************************************/
/*
 *      the char \,(,) need to be 'escaped' to be printed by postscript
 *      this function will also add the leadin '(' and trailing ')'
 *
 *      the function returns the total number of points in the string
 *      assuming the Helvetica 14 pt font.
 */
long    convert_string_to_ps(s1,s2)
char    *s1, *s2;
{
char    ch, *p1, *p2;
long    total_points;

p1 = s1;
p2 = s2;
*p2++ = '(';      /* store lead in '(' */

total_points = 0L;
 
while (1)
  {
   if ((ch = *p1++) == '\0')  break;

   if (ch >= 0x20 && ch <= 0x7E)
     total_points += helvetica_size[(int)ch - 0x20];
   else
     total_points += 700;   /* just estimate if char is not ASCII */

   if (ch == '\\' || ch == '(' || ch == ')')
      *p2++ = '\\';

   *p2++ = ch;
  }

*p2++ = ')';            /* trailing ')' */
*p2   = '\0';           /* terminator   */

return(total_points);
}
/***********************************************************************/ 
/*
 *      Display the number in the simplest terms and still retain all
 *	the precision of the number. Very small numbers and very 
 *	large numbers are displayed in scientic notation.
 *     
 *      The function takes in a double and fills a char * with
 *	the desired text.
 *     
 *	This function is fairly simple in operation. It first displays
 *      the desired number with 8 significant digits after the decimal 
 *	point,
 *
 *	31.4 becomes 31.40000000, 9.3E-8 becomes 9.30000000E-008.
 *	
 *	For fixed point, simply reverse scan the string until 
 *	the char is not a '0' or the decimal point.
 *
 *	For scientific notation, first simplify the exponent,
 *	xxxE-008 becomes xxxE-8,  xxxE+007 becomes xxxE7.
 *
 *      Then reverse scan from the 'E' and replace all consecutive
 *	'0' (up to and including the decimal point) with spaces. 
 *	Then copy the temp string into the destination and just 
 *	ignore spaces in the copy.
 *
 *	Ex :	The number -3.7E+8 will become   -3.70000000E+008
 *					   ...   -3.70000000E8
 *					   ...   -3.7       E8
 *					   ...   -3.7E8
 *					  done
 *
 *	Ex :	The number 6.0E-9 will become     6.00000000E-009
 *					   ...    6.00000000E-9
 *					   ...    6         E-9
 *					   ...    6E-9
 *					  done
 */
void	format_double(d,s)
double	d;
char    *s;
{
int	i, flag;
char    *p, *ps, buf[32];

flag = 0;

if (fabs(d) < 1.0E-12)
  {
   strcpy(s,"0");
   return;
  }

if ((fabs(d) < 9.99E-4) || (fabs(d) > 9.99E4))
  flag = 1;

if (flag == 0)
  {
   sprintf(buf,"%.8f",d);
   p  = buf;
   i  = strlen(buf);
   p += i - 1;

   while (1)
     {
      if (*p == '0')
 	 p--;
      else
	{
	 if (*p == '.')
	   break;
         
	 p++;
	 break;
	}
     }

   *p = '\0';
   strcpy(s,buf);
  }
else
  {
   sprintf(buf,"%.8E",d);
   p = buf;

   while (*p++ != 'E');
   i  = atoi(p);
   sprintf(p,"%d",i);

   p -= 2;
   while (1)
     {
      if (*p == '0' || *p == '.')  
	*p = ' ';
      else
	break;

      p--;
     }

   p  = buf;
   ps = s;

   while (1)
     {
      if ((*ps = *p) == '\0')
	break;

      if (*p++ != ' ')
	 ps++;
     }
  }
}
/***********************************************************************/ 
/*
 *      For an arbitrary min/max of a range, this function will
 *      determine the optimal numbers to display on the graph and it
 *      will also determine the number of divisions (how many
 *      gridlines on the graph).
 */
void	scale_values(flag,v_min,v_max,divisions)
double  *v_min, *v_max;
int     flag, *divisions;
{
int     adivisions[22], index, i, adiv;
double  cc[22], c1, c10, d1, d2, vperdiv, xmin, xmax, delta;

xmin = *v_min;
xmax = *v_max;

c1     = 1.0;
c10    = 10.0;

cc[0]  = 1.0;    adivisions[0]  = 10;
cc[1]  = 1.2;    adivisions[1]  = 6;
cc[2]  = 1.4;    adivisions[2]  = 7;
cc[3]  = 1.6;    adivisions[3]  = 8;
cc[4]  = 1.8;    adivisions[4]  = 9;
cc[5]  = 2.0;    adivisions[5]  = 10;
cc[6]  = 2.4;    adivisions[6]  = 6;
cc[7]  = 2.8;    adivisions[7]  = 7;
cc[8]  = 3.0;    adivisions[8]  = 6;
cc[9]  = 3.2;    adivisions[9]  = 8;
cc[10] = 3.5;    adivisions[10] = 7;
cc[11] = 3.6;    adivisions[11] = 9;
cc[12] = 4.0;    adivisions[12] = 8;
cc[13] = 4.5;    adivisions[13] = 9;
cc[14] = 5.0;    adivisions[14] = 10;
cc[15] = 6.0;    adivisions[15] = 6;
cc[16] = 7.0;    adivisions[16] = 7;
cc[17] = 8.0;    adivisions[17] = 8;
cc[18] = 9.0;    adivisions[18] = 9;
cc[19] = 10.0;   adivisions[19] = 10;
cc[20] = 12.0;   adivisions[20] = 6;
cc[21] = 14.0;   adivisions[21] = 7;

delta = xmax - xmin;
if (delta < 1.0E-12)
  {
   xmax += 0.1;
   xmin -= 0.1;
   delta = xmax - xmin;
  }

d1 = log10(delta);
if (d1 >= 0.0)
  i = (int)d1;
else
  i = 0 - (int)(1.0 - d1);

d1 = pow(10.0,(double)i);

c1  *= d1;
c10 *= d1;

while (1)
  {
   if (delta >= c1 && delta <= c10)
     break;

   if (delta < c1)
     {
      c1  /= 10.0;
      c10 /= 10.0;
     }
   else
     {
      c1  *= 10.0;
      c10 *= 10.0;
     }
  }

for (i=0; i < 22; i++)
  cc[i] *= c1;

index = 1;
while (1)
  {
   if (delta <= cc[index])
     break;

   index++;
  }

d2 = fabs(delta - cc[index - 1]);
if (d2 < 1.1E-12)
  index--;

if (flag)			/* just compute the #divisions & return */
  {
   *divisions = adivisions[index]; 
   return;
  }

while (1)
  {
   adiv    = adivisions[index]; 
   vperdiv = cc[index] / adiv;
   
   d2 = xmax / vperdiv;
   if (d2 >= 0.0)
     {
      d1 = floor(d2);
      if (fabs(d2 - d1) > 1.1E-12) 
        d1 += 1.0;

      xmax = vperdiv * d1;
      xmin = xmax - vperdiv * adiv;
     }
   else
     {
      d1   = floor(-d2);
      xmax = -d1 * vperdiv;
      xmin = xmax - vperdiv * adiv;
     }

   if (xmin <= *v_min)
     break;
   
   if (fabs(xmin - *v_min) < 1.1E-12)
     break;
   
   xmin = *v_min;
   xmax = *v_max;
   index++;
  }

*divisions = adiv;
*v_min     = xmin;
*v_max     = xmax;
}
/***********************************************************************/
void	rescale_divisions(desired_div,current_div,clow,chigh)
int	desired_div, current_div;
double	*clow, *chigh;
{
double  delta, c1, c2, c95, c1org, c2org;
int	i, jdiv;

c1org = *clow;
c2org = *chigh;

delta = (c2org - c1org) / current_div;
i     = 0;
c95   = 0.95;

c1org += delta / 1000.0;

while (1)
  {
   c1 = c1org;
   c2 = c2org;

   if ((i & 1) == 0)
     c2 += c95 * delta;
   else
     c1 -= c95 * delta;

   c1org = c1;
   c2org = c2;
   
   scale_values(FALSE,&c1,&c2,&jdiv);

   if (++i == 10)	/* something is really broke if this happens */
     break;		/* maybe should have error message & exit ?? */
			/* this isn't catastrophic, so we'll go on   */

   if (jdiv == desired_div)
     break;
  }

*clow  = c1;
*chigh = c2;
}
/***********************************************************************/
/*
 *     Flip low/high if low > high, i.e. guarantee the greater
 *     of the values is in *high.
 */
void	adjust_min_max(low,high)
double  *low, *high;
{
double  tmp;

if (*low > *high)
  {
   tmp   = *low;
   *low  = *high;
   *high = tmp;
  }
}
/***********************************************************************/
/*
  Allocate extra memory to store signature information to detect
  if the memory blocks or pointers are getting corrupted.

  Allocate 48 extra bytes, 32 for the signature info and 16 to round 
  up the desired allocation to the next multiple of 16.

  NOTE : For 16 bit machines, a maximum of 65,480 bytes can be allocated 
         with this function. For >= 32 bit machines, 2^32 - 64 bytes
	 can be allocated with this function.

  This function has been redone since it would break if sizeof(long) > 4
  bytes. It is now smarter about these things and will work as long as
  sizeof(long) is <= 8 bytes.

  OFFSET        DESCRIPTION
  -------------------------------------------
     0          signature-1    0x7c143ae9
     8          size of block malloc'ed (N)
    16          pointer returned to caller
   N-16         end of malloc'ed block
   N-8          signature-2    0x3d726bc5
*/
void          *memory_allocate(numbytes)
unsigned int  numbytes;
{
int	      flag;
unsigned int  n;
unsigned char *cp, ch0, ch1, ch2, ch3;
void	      *vp;
long          *lp;
				             /* get a multiple of 16 bytes */
n = (numbytes + (48 * sizeof(char))) & ~0xF;

flag = sizeof(int);	        /* this should be a compile time check ... */

if ((vp = (void *)malloc(n)) == NULL)
  {
   fprintf(stderr,"ERROR : Out of memory\n");
   exit_program(90);
  }

ch0 = (unsigned char)(n & 0xFF);
ch1 = (unsigned char)((n >> 8) & 0xFF);

if (flag <= 2)			/* 16 bit machines */
  {
   ch2 = 0;
   ch3 = 0;
  }
else				/* >= 32 bit machines */
  {
   ch2 = (unsigned char)((n >> 16) & 0xFF);
   ch3 = (unsigned char)((n >> 24) & 0xFF);
  }

lp  = (long *)vp;
*lp = 0x7C143AE9;            /* write first signature */

cp  = (char *)vp;
cp += 8 * sizeof(char);    /* write size of the allocation */

*cp++ = ch3;
*cp++ = ch2;
*cp++ = ch1;
*cp++ = ch0;
*cp   = 0x1B;

cp  = (char *)vp;
cp += n - 8 * sizeof(char);
lp  = (long *)cp;

*lp = 0x3D726BC5;          /* write second signature */

gs_malloc_free_counter++;

cp  = (char *)vp;
cp += 16 * sizeof(char);
return ((void *)cp);
}
/***********************************************************************/ 
void	memory_free(vpf)
void    *vpf;
{
int	      flag;
unsigned int  n;
unsigned char *cp, ch0, ch1, ch2, ch3;
long          sig1, *lp;
void          *vp;

flag = sizeof(int);

cp   = (char *)vpf;
cp  -= 16 * sizeof(char);
vp   = (void *)cp;
lp   = (long *)vp;
sig1 = *lp;

cp += 8 * sizeof(char);
ch3 = *cp++;
ch2 = *cp++;
ch1 = *cp++;
ch0 = *cp++;

if (flag <= 2)			/* 16 bit machines */
  {
   n = (unsigned int)ch1 & 0xFF;
   n = (n << 8) | ((unsigned int)ch0 & 0xFF);
  }
else				/* >= 32 bit machines */
  {
   n = (unsigned int)ch3 & 0xFF;
   n = (n << 8) | ((unsigned int)ch2 & 0xFF);
   n = (n << 8) | ((unsigned int)ch1 & 0xFF);
   n = (n << 8) | ((unsigned int)ch0 & 0xFF);
  }

ch0 = *cp;
cp  = (char *)vp;
cp += n - 8 * sizeof(char);

lp  = (long *)cp;
                           /* verify signatures 1 & 2  */
if ((sig1 != 0x7C143AE9) || (*lp != 0x3D726BC5) || (ch0 != 0x1B))
  {
   fprintf(stderr,
           "ERROR : Attempted free of invalid or corrupted memory block\n");
   exit_program(92);
  }

*lp   = 0xAA55BB77;       /* erase our signatures */
lp    = (long *)vp;
*lp++ = 0xBB66FF99;  
*lp   = 0x7733CC88;

gs_malloc_free_counter--;
free(vp);
}
/***********************************************************************/ 
int	get_malloc_free_counter()
{
return (gs_malloc_free_counter);
}
/***********************************************************************/ 
void	get_tmp_file(ss)     
char    *ss;
{
int     k;
char    str8[16], str[64];
FILE    *ptr;

k = 25;
get_tmp_2(str8);

while (1)
  {
   if (--k == 0)  error_tmp_file();

#ifdef MSDOS
   fill_tmpvar_2(str);                  /*  for MS-DOS */
   strcat(str,str8);                    /*  use TMP env variable, or C:\ */
#else
   strcpy(str,"/tmp/fpl");              /*  for UNIX   */
   strcat(str,str8);
#endif

   if ((ptr = fopen(str, "r")) == NULL)
     break;
   else
     {
      fclose(ptr);
      str8[0] = 'H';
      str8[1] = (char)('A' + k);
     }
  }

if ((ptr = fopen(str, "w")) == NULL)  error_tmp_file();

fclose(ptr);
strcpy(ss,str);
}
/***********************************************************************/
void	get_tmp_2(s)
char    *s;
{
static  int filect = 122;
long        pid;

pid = (long)getpid() & 0xFFFFFL;

#ifdef MSDOS
sprintf(s,"PID%05lX.%03X",pid,filect);
#else
sprintf(s,"%05lx.%03x",pid,filect);
#endif

filect++;
filect &= 0xFFF;
}
/***********************************************************************/

#ifdef MSDOS                                 /* use TMP env var in DOS */

void	fill_tmpvar_2(s)                     /* if not there, use C:\  */
char    *s;
{
int     len;
char    *p;

strcpy(s,"C:\\");
if ((p = getenv("TMP")) != NULL)
  {
   strcpy(s,p);
   len = strlen(s) - 1;
   if (s[len] != '\\')
     {
      s[len + 1] = '\\';
      s[len + 2] = '\0';
     }
  }
}

#endif

/***********************************************************************/
void	error_tmp_file()
{
fprintf(stderr,"ERROR : Could not create temporary file\n");
exit_program(60);
}
/***********************************************************************/
char	*replace_crlf(s)
char    *s;
{
char    *p;

p = s;

while (1)
  {
   if (*p == '\0' || *p == 0x0D || *p == '\n')  break;
   p++;
  }

*p = '\0';
return(s);
}
/***********************************************************************/
void	delete_file(fn)
char 	*fn;
{
remove(fn);
}
/***********************************************************************/
/*
 *      this function will filter ^G (beep), ^L (formfeed), and 0x7F
 */
char    *filter_input_string(s)
char    *s;
{
char    ch, *p, *ps;

p  = s;
ps = s;

while (1)
  {
   ch = *p++;
   if ((*s = ch) == '\0')
     break;

   if (ch != 0x07 && ch != 0x0C && ch != 0x7F)
     s++;
  }
return(ps);
}
/***********************************************************************/
/*
 *      this function will convert a string to lowercase
 */
char    *lowercase(s)
char    *s;
{
char    *p;

p = s;

while (1)
  {
   if (*p >= 'A' && *p <= 'Z')
     *p += 'a' - 'A';

   if (*p++ == '\0')  break;
  }
return(s);
}
/***********************************************************************/
/*    returns char position of first occurence of s2 in s1
	  or -1 if no match found
*/
int     strposition(s1,s2)
char    *s1, *s2;
{
register char  ch1, ch2;
char           *p0, *p1, *p2;
int            ct;

ct = -1;
p0 = s1;

if (*s2 == '\0')  return(-1);

while (1)
  {
   ct++;
   p1  = p0;
   p2  = s2;
   ch2 = *p2;
   
   while (1)                       /* scan until first char matches */
     {
      if ((ch1 = *p1) == '\0')  return(-1);
      if (ch1 == ch2)           break;
      p1++;
      ct++;
     }

   p2++;                           /* check remainder of 2 strings */
   p1++;
   p0 = p1;

   while (1)
     {
      if ((ch2 = *p2) == '\0')  return(ct);
      if (*p1 != ch2)           break;
      p1++;
      p2++;
     }
  }
}
/***********************************************************************/
