/* Copyright Andrew J. Chalk, 1987, 1988.  All Rights Reserved.               */
/* Purpose: A utility to create and maintain a log computer usage             */
/* Usage: i)   Install RXINTMGR, STACKMGR, RXBATMGR,                          */
/*             ANSI.SYS and GLOBALV.                                          */
/*        ii)  Ensure EXECIO.EXE is in your path when LOGOFF is called.       */
/*        iii) Place the command "Logoff on" in your autoexec.bat             */
/*        iv)  When you finish work, type "logoff"                            */
/* REXXLOG will create a file called USAGE.LOG as specified below             */
/* This file contains a record of your computer usage.                        */
/* First version 5/15/87. Last update: 05-11-88.                              */

trace n
options newcom

/********************* VARIABLES GLOBAL TO THE PROGRAM ************************/
/* USERS: This is where you record the name of your logfile */
logfile = 'c:\rexx\myrexx\usage.log'            /* location of log */
ver = 'Version 1.2'                             /* program version number */

/******************************************************************************/

arg status .                           /* are we logging on or off?   */
  'cls'
  call intro                           /* Show the intro screen       */
  select
    when status = on then call logon   /* You want to logon..         */
    when status = '' then call logoff  /* You want to logoff..        */
    otherwise call error 1
  end
  call total_time                      /* Let's find out how long     */
  if dosdir('c:\lasting.bak') <> '' then call dosdel 'c:\lasting.bak'
exit(0)


/* THIS PROCEDURE PUTS AN INTRODUCTORY HEADER ON THE SCREEN */
intro : procedure expose ver

  h1 = 'REXXLOG -- THE REXX COMPUTER USAGE LOG'
  cpyrt = right('(C) Andrew J. Chalk, 1987,1988. All Rights Reserved',80)
  h2 = overlay(ver,cpyrt)
  say Centre('[33;40;1m' h1 '[0m',96)
  say h2
return


check_file : procedure expose logfile ver       /*  create a new log?  */

  if dosdir(logfile) <> '' then call check_date
  if result = 1 then return
  h1 = Center('REXXLOG -- THE REXX COMPUTER USAGE LOG',80)
  cpyrt = right('Copyright Andrew J. Chalk, 1987. All Rights Reserved.',80)
  h2 = overlay(ver,cpyrt)                      /* what an egotist! */
  t1 = left('Date',80)
  d1 = 'On Time'
  t1 = overlay(d1,t1,23)
  d2 = 'Off Time'
  t1 = overlay(d2,t1,32)
  d3 = 'Purpose of work'
  t1 = overlay(d3,t1,41)
  d4 = 'Time On'
  h3 = overlay(d4,t1,73)
  call lineout logfile, h1
  call lineout logfile, h2
  call lineout logfile, ' '
  call lineout logfile, h3
  call lineout logfile
  say Centre('[33;40;1m File' logfile 'not found. New Logfile Created. [0m',95)

return


check_date:procedure expose logfile       /* Has the month changed?   */
                                          /* If so, create an archive */
    file_month = substr(dosdir(logfile,d),1,2) /* the month of the logfile */
    actual_month = substr(date('U'),1,2)       /* the month at runtime */
    if actual_month = file_month then return 1
    fpath = substr(logfile,1,lastpos('\',logfile))
    fspec = dosdir(logfile,n)
    if pos('.',fspec) <> '' then fname = substr(fspec,1,pos('.',fspec)-1)
    else fname = fspec

    /* HOW TO USE SETS IN REXX */
    monthset = jan feb mar apr may jun jul aug sep oct nov dec
    month = word(monthset,file_month)
    'copy' logfile fpath||fname'.'month '> nul'
    'erase' logfile
return 0


logon : procedure expose logfile       /* logon procedure */

    call check_file    /* if logfile does not exist then create a new one */
    if dosdir('c:\lasting.glv') <> '' then return
    reason = purpose()
    day = left(date('w'),9)
    fdate = right(date(),11)
    info = day fdate'. On at' time()
    'globalv select worklog setpl instring1' info
    info = day fdate'. On at' time() 'to work on' reason
    'globalv select worklog setpl instring2' reason
    say Centre('[33;40;1m The Following Logon Information was Recorded by REXXLOG [0m',95)
    say Centre('[7m' info ' [0m',88)

return


purpose: procedure

    say '[1mEnter purpose of logging on (for tax records):[0m'
    say
    parse pull reason
    reason = strip(substr(reason,1,80),t)

return reason


logoff : procedure expose logfile

    call check_file
    if dosdir('c:\lasting.glv') = '' then return  /* already logged off! */
    'globalv select worklog get instring1'
    intime  = word(instring1,words(instring1))
    outtime = time()
    time_on = elapsed_time(intime,outtime)
    'globalv select worklog get instring2'
    temp1 = left(word(instring1,1),9)    /* the day */
    temp2 = left(word(instring1,2),2)    /* the date */
    temp3 = left(word(instring1,3),3)    /* the month */
    temp4 = left(word(instring1,4),4)    /* the year */
    temp5 = left(intime,8)
    temp6 = left(outtime,8)
    temp7 = left(instring2,31)
    temp8 = left(time_on,8)
    b3 = ''
    outinfo1 = temp1 temp2 temp3 temp4 temp5 temp6 temp7 temp8
    outinfo2 = instring1'. Off at' outtime'. Time on:' time_on
    outinfo3 = 'Work:' instring2
    call lineout logfile, outinfo1
    if result <> 0 then call error result
    call lineout logfile
    say Centre('[33;40;1m The Following Logoff Information was Recorded by REXXLOG [0m',95)
    say Centre('[7m' outinfo2 '[0m',88)
    say Centre('[1;44;35m' outinfo3 '[0m',94)
    if dosdir('c:\lasting.glv') <> '' then 'erase c:\lasting.glv'

return


elapsed_time:procedure                      /* How long were we on? */
arg start, finish
ehours = 0;emins=0;

    shour = substr(start,1,pos(':',start)-1)
    smin = substr(start,pos(':',start)+1,2)
    ssec = substr(start,7,2)
    fhour = substr(finish,1,pos(':',finish)-1)
    fmin = substr(finish,pos(':',finish)+1,2)
    fsec = substr(finish,7,2)

    start_secs = (shour*3600) + (smin*60) + ssec
    finish_secs = (fhour*3600) + (fmin*60) + fsec

    esecs = finish_secs - start_secs
    if esecs < 0 then esecs = esecs + 86400     /* midnight passed */
    if esecs > 3600 then do
        ehours = esecs % 3600
        esecs = esecs // 3600
    end
    if esecs > 60 then do
        emins = esecs % 60
        esecs = esecs // 60
    end
    if ehours < 10 then ehours = '0'ehours;
    if emins < 10 then emins = '0'emins
    if esecs < 10 then esecs = '0'esecs
    etime = ehours':'emins':'esecs

return etime


total_time:procedure expose logfile /* How long were we on this MONTH!*/

    thours = 0;tmins=0; tsecs=0;

    stack_state = stackstatus()          /* is the stack installed and large enough? */
    parse var stack_state flag id free
    if flag = n | flag = d then call error 3 /* stack not installed */
    filesize = word(dosdir(logfile),2)
    if filesize >= free then call error 4
    theirs = queued()


    address command 'execio * diskr' logfile '0 (fifo finis notype'
    if rc < 0 | rc = 28 then call error rc
    do 4                                 /* pull header */
        pull
    end

    do while queued() > theirs
        pull line
        time = word(line,words(line))      /* the time on is the last line */
        shour = substr(time,1,pos(':',time)-1)
        smin = substr(time,pos(':',time)+1,2)
        ssec = substr(time,7,2)
        secs = (shour*3600) + (smin*60) + ssec
        tsecs = secs + tsecs
    end
    if tsecs > 3600 then do
        thours = tsecs % 3600
        tsecs = tsecs // 3600
    end
    if tsecs > 60 then do
        tmins = tsecs % 60
        tsecs = tsecs // 60
    end
    if thours < 10 then thours = '0'thours
    if tmins < 10 then tmins = '0'tmins
    if tsecs < 10 then tsecs = '0'tsecs
    ttime = thours':'tmins':'tsecs
    say Centre('[36;47;1m Total Time On This Month is' ttime' [0m',94)

return ttime


error: procedure                            /* Errors happen! */
arg enum

select
    when enum = -8 then,
      say '[30;41m Error: [31;40m Insufficient REXX storage run command.[0m'
    when enum = -3 then,
      say '[30;41m Error: [31;40m Requested program could not be found. [0m'
    when enum = 1 then,
      say '[30;41m Error: [31;40m Incorrect syntax. [0m'
    when enum = 2 then,
      say '[30;41m Error: [31;40m Could not write line to'upper(logfile)||'. [0m'
    when enum = 3 then do
      say '[30;41m Error: [31;40m Stack manager disabled or not installed. [0m'
      exit(-1)
    end
    when enum = 4 then,
      say '[30;41m Error: [31;40m Insufficient REXX stack space. [0m'
    otherwise say '[30;41m Error: [31;40m Unidentified error. [0m'
end
call lineout logfile
exit(1)
