/*
 *  Routines for setting up the parameters/buffers/display
 *
 *  Copyright (C) 1995  Philip VanBaren
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include "specgram.h"
#include "display.h"
#include "extern.h"

int fftlen=1024;        /* Number of points for FFT */
long SampleRate=44100L; /* A/D sampling rate */
int logamp=0;           /* Flag set to 1 for log-based amplitude scale */
int windfunc=0;         /* Flag set to selected window function */
int gain=0;             /* Amount of db/octave gain */
int deriv=0;            /* Flag for doing differencing for 6db/octave gain */
float amp_scale=4.0;    /* Scale factor for the graph amplitude */
float log_scale=20.0;   /* Scale factor for the graph amplitude */
int log_base=8;         /* Base level for logarithm display */
int palette=0;          /* Flag indicating current palette number */
int diff_mode=0;        /* Flag for doing difference mode on palette */
int stereo=0;           /* Flag for Mono/Stereo mode */
int stereo_diff=0;      /* Flag for showing difference mode in stereo */
int buffer_offset=0;    /* Current offset into recording buffer */
int buffer_step;        /* Samples to step forward between FFTs */
int accel_factor=1;     /* FFTs performed per sampled buffer */
volatile long recorded=0; /* Number of buffers recorded (for rate calculation) */
long displayed=0;       /* Number of FFTs performed (for rate calculation) */
char *windfile=NULL;    /* Pointer to filename for saving window data */
double alpha;           /* Gaussian window parameter */

char *window_name[] = { "Hamming","Hanning","Blackman","Gaussian","Welch","Parzen","Rectangular" };
char *palette_name[] = { "HSV","Threshold","Cool","Hot","White","Bone","Copper" };

int Soundcard=SC_DEF;   /* Soundcard number (as defined in freq.h) */
int sb_irq=SB_IRQ;      /* IRQ used by the Soundblaster card */
int sb_dma=SB_DMA;      /* DMA channel used by the Soundblaster card */
int sb_addr=SB_ADDR;    /* I/O address of the Soundblaster card */
int sample_size;        /* Bits per sample (8 or 16) */
int mixers;             /* Mixers available (1) or not (0) */

char *audio_device=NULL;  /* Name of DSP device */
char *mixer_device=NULL;  /* Name of the mixer device */
char *output_device=NULL; /* Name of the output device */
/* Soundcard function pointers */
void (*reset_soundcard)(void)=NULL;
void (*halt_soundcard)(void)=NULL;
void (*cleanup_soundcard)(void)=NULL;
void (*recordblock)(void *)=NULL;
void (*set_mixer)(int,int)=NULL;


/* Parse a sound card number or label */
int parsecardname(char *label)
{
   int i=0,j;
   char name[10];

   /* Skip over any leading space */
   while(label[i]==' ') i++;

   /* Check for numerical values */
   if((label[i]>='0') && (label[i]<='9'))
      return(label[i]-'0');

   /* Convert the string to upper case */
   for(j=0;label[i] && (j<9);i++,j++)
      name[j]=toupper(label[i]);
   name[j]=0;

   /* Check for valid string values */
   #ifdef SC_SB8
      if(strncmp(name,SC_SB8_NAME,strlen(SC_SB8_NAME))==0)
         return SC_SB8;
   #endif
   #ifdef SC_PAS16
      if(strncmp(name,SC_PAS16_NAME,strlen(SC_PAS16_NAME))==0)
         return SC_PAS16;
   #endif
   #ifdef SC_VESA
      if(strncmp(name,SC_VESA_NAME,strlen(SC_VESA_NAME))==0)
         return SC_VESA;
   #endif
   #ifdef SC_SB16
      if(strncmp(name,SC_SB16_NAME,strlen(SC_SB16_NAME))==0)
         return SC_SB16;
   #endif
   #ifdef SC_MULAW
      if(strncmp(name,SC_MULAW_NAME,strlen(SC_MULAW_NAME))==0)
         return SC_MULAW;
   #endif
   #ifdef SC_LINUX
      if(strncmp(name,SC_LINUX_NAME,strlen(SC_LINUX_NAME))==0)
         return SC_LINUX;
   #endif
   DOUT("Error: no soundcard match");
   return(-1); /* If none match, return failure */
}

/* Parse the ini file, if it exists */
void parse_ini_file(void)
{
   FILE *fp;
   char buffer[80];
   int i;

   if((fp=fopen(ini_file,"r"))!=NULL)
   {
      while(!feof(fp))
      {
         fgets(buffer,sizeof(buffer),fp);
         for(i=0;(buffer[i]!=0) && (buffer[i]!=':');i++)
            buffer[i]=toupper(buffer[i]);

	 if(strncmp(buffer,"SOUNDCARD:",10)==0)
            Soundcard=parsecardname(buffer+10);
         sscanf(buffer,"SAMPLE RATE:%ld",&SampleRate);
         sscanf(buffer,"FFT LENGTH:%d",&fftlen);
         sscanf(buffer,"ACCEL FACTOR:%d",&accel_factor);
         sscanf(buffer,"STEP RATE:%d",&buffer_step);
         sscanf(buffer,"PIXEL HEIGHT:%d",&repcount);
         sscanf(buffer,"WINDOW FUNCTION:%d",&windfunc);
         sscanf(buffer,"LOG MODE:%d",&logamp);
         sscanf(buffer,"LOG SCALE BASE:%d",&log_base);
         sscanf(buffer,"LOG SCALE FACTOR:%f",&log_scale);
         sscanf(buffer,"LINEAR SCALE FACTOR:%f",&amp_scale);
         sscanf(buffer,"DB/OCTAVE GAIN:%d",&gain);
         sscanf(buffer,"EMBOSSED MODE:%d",&diff_mode);
         sscanf(buffer,"PALETTE:%d",&palette);
         sscanf(buffer,"STEREO:%d",&stereo);
         sscanf(buffer,"STEREO L-R:%d",&stereo_diff);

	 if(strncmp(buffer,"AUDIO DEVICE:",13)==0)
	 {
	    int i=13;
	    while(buffer[i]==' ') i++;  /* Remove leading spaces */
	    audio_device=(char *)malloc(strlen(buffer+i)+1);
	    strcpy(audio_device,buffer+i);
	    for(i=0;i<strlen(audio_device);i++) /* Remove trailing junk */
	    {
	       if(   (audio_device[i]==0x0d)
		  || (audio_device[i]==0x0a)
		  || (audio_device[i]==0x20))
		  audio_device[i]=0;
	    }
	 }
	 if(strncmp(buffer,"OUTPUT DEVICE:",14)==0)
	 {
	    int i=14;
	    while(buffer[i]==' ') i++;  /* Remove leading spaces */
	    output_device=(char *)malloc(strlen(buffer+i)+1);
	    strcpy(output_device,buffer+i);
	    for(i=0;i<strlen(output_device);i++) /* Remove trailing junk */
	    {
	       if(   (output_device[i]==0x0d)
		  || (output_device[i]==0x0a)
		  || (output_device[i]==0x20))
		  output_device[i]=0;
	    }
	 }
	 if(strncmp(buffer,"MIXER DEVICE:",13)==0)
	 {
	    int i=13;
	    while(buffer[i]==' ') i++;  /* Remove leading spaces */
	    mixer_device=(char *)malloc(strlen(buffer+i)+1);
	    strcpy(mixer_device,buffer+i);
	    for(i=0;i<strlen(mixer_device);i++) /* Remove trailing junk */
	    {
	       if(   (mixer_device[i]==0x0d)
		  || (mixer_device[i]==0x0a)
		  || (mixer_device[i]==0x20))
		  mixer_device[i]=0;
	    }
	 }
      }
      fclose(fp);
   }
}


/* Parse the command line */
void parse_command(int argc,char *argv[],char *environ[])
{
   int i=0;
   while(i<argc)
   {
      if(argv[i][0]=='-')
      {
         switch(argv[i][1])
         {
          case 'C':
          case 'c':
            Soundcard=parsecardname(argv[i]+2);
            break;
	  case 'D':   /* Set the DSP device string */
	  case 'd':
	    audio_device=malloc(strlen(argv[i])-1);
	    strcpy(audio_device,argv[i]+2);
	    break;
	  case 'O':   /* Set the output device string */
	  case 'o':
	    output_device=malloc(strlen(argv[i])-1);
	    strcpy(output_device,argv[i]+2);
	    break;
	  case 'X':   /* Set the Mixer device string */
	  case 'x':
	    mixer_device=malloc(strlen(argv[i])-1);
	    strcpy(mixer_device,argv[i]+2);
	    break;
          case 'R':
          case 'r':
            SampleRate=atol(&argv[i][2]);
            break;
	  case 'S':
	  case 's':
	    stereo=1;
	    break;
          case 'F':
          case 'f':
            fftlen=atoi(&argv[i][2]);
            break;
          case 'W':
          case 'w':
            if(argv[i][2]>='A')
            {
               windfile=&argv[i][2];
            }
            else
            {
               windfunc=argv[i][2]-'0';
               if(windfunc==3)
               {
                  alpha=0;
                  if(argv[i][3]==',')
                     alpha=atof(&argv[i][4]);
                  if(alpha<=0) alpha=1;
               }
            }
            break;
          case '?':
          case 'H':
          case 'h':
            #ifdef DOS
               puts("-Cnumber selects the soundcard (0=SB8, 1=PAS, 2=VESA, 3=SB16).");
            #endif
            #ifdef LINUX
               puts("-Cnumber selects the soundcard (0=AU, 1=DSP).");
            #endif
            #ifdef SUN
               puts("-Cnumber selects the soundcard (0=AU).");
            #endif
	    puts("-Ddevice selects the input device");
	    puts("-Odevice selects the output device");
	    puts("-Xdevice selects the Mixer device");
            puts("-Rnumber sets the sampling rate.");
	    puts("-S selects stereo mode.");
            puts("-Fnumber sets the length of the FFT.");
            puts("-W0 selects a Hamming window.  (offset sine) <--default");
            puts("-W1 selects a Hanning window.  (sine)");
            puts("-W2 selects a Blackman window. (two sines)");
            puts("-W3[,alpha] selects a Gaussian window.");
            puts("-W4 selects a Welch window.    (quadratic)");
            puts("-W5 selects a Parzen window.   (triangular)");
            puts("-W6 selects a Rectangular window.");
            puts("-Wfilename saves the windowing function to the specified file.");
            puts("-H or -? displays this message.");
            exit(0);
          default:
            printf("Ignoring unrecognized switch: %s\n",argv[i]);
            puts("Hit <ENTER> to continue...");
            getchar();
         }
      }
      else
         printf("Ignoring unrecognized parameter: %s  (use -? for help)\n",argv[i]);
      i++;
   }

   /* Set up the flags based on chosen gain */
   deriv=gain/6;
   if(deriv>2) deriv=2;
   gain=deriv*6;
   /* Double-check some values */
   if(stereo) stereo=1;   /* Make sure this is 0 or 1 */
   if(stereo_diff) stereo_diff=1;
   if(diff_mode) diff_mode=1;
   if(logamp) logamp=1;

   /* Set up the soundcard function pointers */
   #ifdef SC_SB8
      if(Soundcard==SC_SB8)
         init_sb8(environ);
   #endif
   #ifdef SC_PAS16
      if(Soundcard==SC_PAS16)
         init_pas16();
   #endif
   #ifdef SC_VESA
      if(Soundcard==SC_VESA)
         init_vesa();
   #endif
   #ifdef SC_SB16
      if(Soundcard==SC_SB16)
         init_sb16(environ);
   #endif
   #ifdef SC_LINUX
      if(Soundcard==SC_LINUX)
         init_linux_sc();
   #endif
   #ifdef SC_MULAW
      if(Soundcard==SC_MULAW)
         init_mulaw_sc();
   #endif
   if(reset_soundcard==NULL)
   {
      puts("Error: Invalid soundcard selection");
      puts("Valid choices are:");
      #ifdef SC_SB8
         printf("  %s (%d)\n",SC_SB8_NAME,SC_SB8);
      #endif
      #ifdef SC_PAS16
         printf("  %s (%d)\n",SC_PAS16_NAME,SC_PAS16);
      #endif
      #ifdef SC_VESA
         printf("  %s (%d)\n",SC_VESA_NAME,SC_VESA);
      #endif
      #ifdef SC_SB16
         printf("  %s (%d)\n",SC_SB16_NAME,SC_SB16);
      #endif
      #ifdef SC_MULAW
         printf("  %s (%d)\n",SC_MULAW_NAME,SC_MULAW);
      #endif
      #ifdef SC_LINUX
         printf("  %s (%d)\n",SC_LINUX_NAME,SC_LINUX);
      #endif
      exit(1);
   }
   if(!stereo) stereo_diff=0;
   if(fftlen<64) fftlen=64;
   if(fftlen>(1024/(stereo+1))) fftlen=1024/(stereo+1);
   /* Convert fftlen to a power of 2 */
   for(i=0;fftlen>1;fftlen>>=1,i++);
   if(i) fftlen<<=i;
   if(accel_factor<1) accel_factor=1;
   if(accel_factor>40) accel_factor=40;
   buffer_step=(int)ceil((double)fftlen/(double)accel_factor);
   if(SampleRate<5000L) SampleRate=5000L;
   if(SampleRate>100000L) SampleRate=100000L;
}


/* Allocate memory for and initialize the buffers */
void setup_buffers(int length)
{
   int i;

   if(sample_size==8)
   {
      if((buffer=malloc(length*2*sizeof(unsigned char)))==NULL)
      {
         puts("Unable to allocate enough memory for the buffers!");
         exit(1);
      }
      /* Pre-set the buffer to all zeros */
      memset(buffer,0x80,(stereo+1)*2*length*sizeof(unsigned char));
      offsetbuffer=(unsigned char *)buffer+length*(stereo+1);
   }
   else
   {
      if((buffer=malloc((stereo+1)*2*length*sizeof(short)))==NULL)
      {
         puts("Unable to allocate enough memory for the buffers!");
         exit(1);
      }
      /* Pre-set the buffer to all zeros */
      memset(buffer,0,(stereo+1)*2*length*sizeof(short));
      offsetbuffer=(short *)buffer+length*(stereo+1);
   }

   fftdata=(short *)malloc(length*sizeof(short));
   wind=(short *)malloc(length*sizeof(short));
   olddata=(short *)malloc((length/2)*sizeof(short));
   if(stereo)
   {
      fftdata2=(short *)malloc(length*sizeof(short));
      olddata2=(short *)malloc((length/2)*sizeof(short));
   }
   if(!fftdata || !wind || !olddata || (stereo && !fftdata2) || (stereo && !olddata2))
   {
      puts("Unable to allocate enough memory for the buffers!");
      exit(1);
   }
   for(i=0;i<length;i++)
   {
      fftdata[i]=0;
      if(stereo)
         fftdata2[i]=0;
   }
   for(i=0;i<length/2;i++)
   {
      olddata[i]=0;
      if(stereo)
         olddata2[i]=0;
   }
}

void compute_window_function(void)
{
   int i;

   /* Calculate FFT Windowing function */
   for(i=0;i<fftlen;i++)
   {
      double val;

      switch(windfunc)
      {
       case 1:    /* Hanning */
	 val = 0.5 - 0.5*cos(2*M_PI*i/fftlen);
	 break;

       case 2:    /* Blackman */
         val = 0.42-0.5*cos(2*M_PI*i/fftlen)+0.08*cos(4*M_PI*i/fftlen);
         break;

       case 3:    /* Gaussian */
         val = exp(-alpha / ((double)fftlen*(double)fftlen)
    	       * ((double)2*i-fftlen)*((double)2*i-fftlen));
         break;

       case 4:    /* Welch */
         val = 1 -  ((double)(2*i-fftlen)/(double)(fftlen+1))
                 *((double)(2*i-fftlen)/(double)(fftlen+1));
         break;

       case 5:    /* Parzen */
         val = 1 - fabs((double)(2*i-fftlen)/(double)(fftlen+1));
         break;

       case 6:    /* Rectangular */
         val = 1;
         break;

       default:   /* Hamming */
         val = 0.54-0.46*cos(2*M_PI*i/fftlen);
         break;
      }
      wind[i]=floor(val*32767+0.5);
   }

   /* If output requested, save the window function. (for the curious) */
   if(windfile)
   {
      FILE *fp;
      if((fp=fopen(windfile,"w"))==NULL)
      {
         printf("Error opening window output file: %s\n",windfile);
	 puts("Press <ENTER> to continue...");
         getchar();
      }
      else
      {
         for(i=0;i<fftlen;i++)
            fprintf(fp,"%d\n",wind[i]);
         fclose(fp);
      }
      windfile=NULL;
   }
}

void draw_axis(void)
{
   char ach[5];
   int i;
   int y;

   /* Set up some variables for the frequency bar */
   df=(int)(SampleRate/100L*(stereo+1));
   if(!repcount)
      repcount=1024/fftlen/(stereo+1);
   if(repcount*fftlen*(stereo+1) > 1024)
      repcount=1024/fftlen/(stereo+1);
   tf=(stereo+1)*fftlen*10*repcount;

   #ifdef DEBUG_OUTPUT
      printf("FFTLen:%d, stereo:%d, repcount:%d, df:%d, tf:%d\n",fftlen,stereo,repcount,df,tf);
   #endif
   draw_bar(0,MAXY-525,24,MAXY,0);

   for(i=1;i<=SampleRate/2000;i++)
   {
      y=(int)((long)repcount*1000L*fftlen*i/SampleRate);
      sprintf(ach,"%2d",i);
      draw_text_left(0,MAXY-6-y,ach);
      draw_putpixel(19,MAXY+1-y,TICK_COLOR);
      draw_putpixel(20,MAXY+1-y,TICK_COLOR);
      draw_putpixel(21,MAXY+1-y,TICK_COLOR);
      draw_putpixel(22,MAXY+1-y,TICK_COLOR);
      draw_putpixel(23,MAXY+1-y,TICK_COLOR);
   }

   if(stereo)
   {
      if(stereo_diff)
      {
         draw_text_left(0,MAXY-_font_height,"L+R");
         draw_text_left(0,MAXY-5-_font_height-repcount*fftlen/2,"L-R");
      }
      else
      {
         draw_text_left(0,MAXY-_font_height,"L");
         draw_text_left(0,MAXY-5-_font_height-repcount*fftlen/2,"R");
      }
      for(i=1;i<=SampleRate/2000;i++)
      {
         y=(int)((long)repcount*1000L*fftlen*i/SampleRate);
         sprintf(ach,"%2d",i);
         draw_text_left(0,MAXY-10-repcount*fftlen/2-y,ach);
         draw_putpixel(19,MAXY-4-repcount*fftlen/2-y,TICK_COLOR);
         draw_putpixel(20,MAXY-4-repcount*fftlen/2-y,TICK_COLOR);
         draw_putpixel(21,MAXY-4-repcount*fftlen/2-y,TICK_COLOR);
         draw_putpixel(22,MAXY-4-repcount*fftlen/2-y,TICK_COLOR);
         draw_putpixel(23,MAXY-4-repcount*fftlen/2-y,TICK_COLOR);
      }
   }
}

void show_help(void)
{
   draw_bar(0,0,MAXX,79,0);

   draw_text_left(0,0, " 'F' Change the FFT length");
   draw_text_left(0,16," 'R' Change the sampling rate");
   draw_text_left(0,32," 'A' Change the accel factor");
   draw_text_left(0,48," 'W' Switch windowing function");
   draw_text_left(0,64," 'P' Switch palettes");

   draw_text_left(266,0, "    'G' Adjust dB/octave gain");
   draw_text_left(266,16,"<Arrow> Adjust amplitude scales");
   draw_text_left(266,32,"<SPACE> Freeze display");
   draw_text_left(266,48,"    'L' Toggle log/linear amp");
   draw_text_left(266,64,"    'E' Toggle embossed mode");

   draw_text_left(533,0, "    'O' Output disp as GIF file");
   draw_text_left(533,16,"    'S' Save state to INI file");
   if(stereo)
      draw_text_left(533,32,"    'D' Toggle stereo modes");
   else
      draw_text_left(533,32,"'H','?' Toggle this help screen");
   draw_text_left(533,48,"'Q','X' Exit from the program");
   if(mixers)
      draw_text_left(533,64,"<> [] {} Set mic,ext,cd level");
}

void update_header(void)
{
   char ach[100];

   draw_bar(0,0,MAXX,79,0);
   sprintf(ach,"Sampling rate: %ld Hz",SampleRate);
   draw_text_left(SRX,SRY,ach);
   sprintf(ach,"FFT Length: %d points",fftlen);
   draw_text_left(FLX,FLY,ach);
   sprintf(ach,"Frequency resolution: %g Hz",(double)SampleRate/(double)fftlen);
   draw_text_left(FRX,FRY,ach);
   sprintf(ach,"%s window function",window_name[windfunc]);
   draw_text_left(WFX,WFY,ach);
   sprintf(ach,"Gain of %d dB per octave",gain);
   draw_text_left(DGX,DGY,ach);
   sprintf(ach,"%s palette",palette_name[palette]);
   draw_text_left(PLX,PLY,ach);
   sprintf(ach,"Accel:%2d",accel_factor);
   draw_text_left(AFX,AFY,ach);
   if(displayed>=100)
   {
      double rate,curr;
      ftime(&tb);
      curr=(double)(tb.time-basetime)+(double)tb.millitm/1000.0;
      rate=curr-starttime;
      if(rate>0)
         rate=(double)displayed*fftlen/(rate*SampleRate);
      else
         rate=0;
      sprintf(ach,"(%4.1f)",rate);
      draw_text_left(AFX+80,AFY,ach);
      #ifdef DEBUG_OUTPUT
         printf("%g-%g, %g\n",curr,starttime,rate);
      #endif
   }
   if(diff_mode)
      draw_text_left(MDX,MDY,"Embossed mode");
   else
      draw_text_left(MDX,MDY,"Amplitude mode");
   if(mixers)
   {
      sprintf(ach,"Mic:%3d  Ext:%3d  CD:%3d",mic_level,ext_level,int_level);
      draw_text_left(LVX,LVY,ach);
   }
   if(logamp)
      draw_text_left(AMX,AMY,"Log amplitude scale");
   else
      draw_text_left(AMX,AMY,"Linear amplitude scale");
}

void set_palette(int n)
{
   unsigned char p[256*3];

   /* Set up the palette */
   fillpalette(p,n);
   draw_setallpalette(p);
}

void fillpalette(unsigned char *p,int n)
{
   long c,color;

   p[0]=p[1]=p[2]=0;
   p[3]=p[4]=p[5]=255;
   p[6]=p[7]=p[8]=64;

   p+=9;
   for(c=MIN_COLOR;c<_draw_colors;c++)
   {
      color=c*256;
      color/=_draw_colors;
      if(n==HSV) /* HSV */
      {
         if(color<64)
         {
            *(p++)=0;
            *(p++)=(uchar)(color*4.0);
            *(p++)=255;
         }
         else if(color<128)
         {
            *(p++)=0;
            *(p++)=255;
            *(p++)=(uchar)(510.0-color*4.0);
         }
         else if(color<192)
         {
          *(p++)=(uchar)(color*4.0-510.0);
          *(p++)=255;
          *(p++)=0;
         }
         else
         {
          *(p++)=255;
          *(p++)=(uchar)(1020.0-color*4.0);
          *(p++)=0;
         }
      }
      else if(n==THRESH) /* Thresholded HSV */
      {
         if(color<16)
         {
          *(p++)=0;
          *(p++)=0;
          *(p++)=0;
         }
         else if(color<64)
         {
          *(p++)=0;
          *(p++)=(uchar)(color*4.0);
          *(p++)=255;
         }
         else if(color<128)
         {
          *(p++)=0;
          *(p++)=255;
          *(p++)=(uchar)(510.0-color*4.0);
         }
         else if(color<192)
         {
          *(p++)=(uchar)(color*4.0-510.0);
          *(p++)=255;
          *(p++)=0;
         }
         else
         {
          *(p++)=255;
          *(p++)=(uchar)(1020.0-color*4.0);
          *(p++)=0;
         }
      }
      else if(n==COOL) /* cool */
      {
         *(p++)=(uchar)color;
         *(p++)=(uchar)(255-color);
         *(p++)=255;
      }
      else if(n==HOT) /* hot */
      {
         if(color<96)
         {
          *(p++)=(uchar)(color*2.66667+0.5);
          *(p++)=0;
          *(p++)=0;
         }
         else if(color<192)
         {
          *(p++)=255;
          *(p++)=(uchar)(color*2.66667-254);
          *(p++)=0;
         }
         else
         {
          *(p++)=255;
          *(p++)=255;
          *(p++)=(uchar)(color*4.0-766.0);
         }
      }
      else if(n==BONE) /* bone */
      {
         if(color<96)
         {
          *(p++)=(uchar)(color*0.88889);
          *(p++)=(uchar)(color*0.88889);
          *(p++)=(uchar)(color*1.20000);
         }
         else if(color<192)
         {
          *(p++)=(uchar)(color*0.88889);
          *(p++)=(uchar)(color*1.20000-29);
          *(p++)=(uchar)(color*0.88889+29);
         }
         else
         {
          *(p++)=(uchar)(color*1.20000-60);
          *(p++)=(uchar)(color*0.88889+29);
          *(p++)=(uchar)(color*0.88889+29);
         }
      }
      else if(n==COPPER) /* copper */
      {
         if(color<208)
         {
          *(p++)=(uchar)(color*1.23);
          *(p++)=(uchar)(color*0.78);
          *(p++)=(uchar)(color*0.5);
         }
         else
         {
          *(p++)=255;
          *(p++)=(uchar)(color*0.78);
          *(p++)=(uchar)(color*0.5);
         }
      }
      else /* Black and white */
      {
         *(p++)=(uchar)color;
         *(p++)=(uchar)color;
         *(p++)=(uchar)color;
      }
   }
}

