{ NUMLIB.PAS : Numerical routine library

  Title   : NUMLIB
  Version : 4.0
  date    : Nov 11,1996
  Language: Borland Turbo Pascal 4.0 through 7.0 (all targets)
  Author  : J R Ferguson
  Usage   : Unit
}

UNIT NumLib;

INTERFACE


const

{ Pi      = 3.1415926536; }
  DblPi   = 6.2831853072;   { 2 * Pi }
  HalfPi  = 1.5707963268;   { Pi / 2 }
  Radian  = 57.295779513;   { number of degrees in een radian (180 / Pi) }
  Ln2     = 0.693147180560; { natural logarithm of 2 }
  Ln10    = 2.302585092994; { natural logarithm of 10 }


function PostInc(var n: integer): integer;
{ Increment integer n; return its original value. }

function PreInc(var n: integer): integer;
{ Increment integer n; return its new value. }

function PostDec(var n: integer): integer;
{ Decrement integer n; return its original value. }

function PreDec(var n: integer): integer;
{ Decrement integer n; return its new value. }

function even(n: integer): boolean;
{ Test if integer n is even. }

{ function odd(n: integer): boolean;     (standard Pascal function)}
{ Test if integer n is odd. }

function Imax(i,j: integer): integer;
{ Return the greatest of two integers. }

function Imin(i,j: integer): integer;
{ Return the smallest of two integers. }

function Rmax(x,y: real): real;
{ Return the greatest of two reals. }

function Rmin(x,y: real): real;
{ Return the smallest of two reals. }

function Isign(n: integer): integer;
{ Return +1 if n>0, 0 if n=0, -1 if n<0. }

function Rsign(x: real): real;
{ Return +1.0 if x>0, 0.0 if x=0, -1.0 if x<0. }




function log2(x: real): real;
{ Return the logarithm of x with base 2. }

function log10(x: real): real;
{ Return the logarithm of x with base 10. }

function logg(g,x: real): real;
{ Return the logarithm of x with base g. }

{function ln(x: real): real;         (standard Pascal function)}
{ Return the natural logarithm of x. }




function RpowerR(x,e: real): real; { x ** e }
{ Return x to the power of e.
  x and e are both of type real. The result is also of type real. }

function RpowerI(x: real; n: integer): real; { x ** n }
{ Return x to the power of n.
  x is of type real, n is of type integer. The result is of type real. }

function IpowerI(i,n: integer): integer; { i ** n }
{ Return i to the power of n.
  i and n are both of type integer. The result is also of type integer.
  n must be non-negative. }




function rad(x: real): real;
{ Returns the value of angle x in radians, when x is in degrees.
  Defined as rad(x) = x * Pi/180. }

function deg(x: real): real;
{ Returns the value of angle x in degrees, when x is in radians.
  Defined as deg(x) = x * 180/Pi. }

function tan(x: real): real;
{ Tangent of x radians.
  Definition : tan(x) = sin(x)/cos(x).
  Condition  : x <> Pi/2 + k*Pi   where k is an integer. }

function cot(x: real): real;
{ Cotangent of x radians.
  Definition : cot(x) = cos(x)/sin(x).
  Condition  : x <> k*Pi          where k is an integer. }

function sec(x: real): real;
{ Secant of x radians.
  Definition : sec(x) = 1/cos(x).
  Condition  : x <> Pi/2 + k*Pi   where k is an integer. }

function cosec(x: real): real;
{ Cosecant of x radians.
  Definition : cosec(x) = 1/sin(x).
  Condition  : x <> k*Pi          where k is an integer. }

function ArcSin(x: real): real;
{ Arc sine of x in radians.
  Definition : sin(ArcSin(x)) = 1
  Condition  : -1.0 <= x <=  1.0
  Range      : [-Pi/2 , +Pi/2] }

function ArcCos(x: real): real;
{ Arc cosine of x in radians.
  Definition : cos(ArcCos(x)) = 1
  Condition  : -1.0 <= x <=  1.0
  Range      : [0.0 , Pi] }

function ArcCot(x: real): real;
{ Arc cotangent of x in radians.
  Definition : cot(ArcCot(x)) = 1
  Condition  : none
  Range      : <0 , Pi>  }

{function ArcTan(x: real): real;            (standard Pascal function)}
{ Arc tangent of x in radians.
  Definition : tan(ArcTan(x)) = 1
  Condition  : none
  Range      : <-Pi/2 , Pi/2> }

function sinH(x: real): real;
{ Hyperbolic sine of x radians.
  Definition : sinH(x) = ( exp(x) - exp(-x) )/2 }

function cosH(x: real): real;
{ Hyperbolic cosine of x radians.
  Definition : cosH(x) = ( exp(x) + exp(-x) )/2 }

function tanH(x: real): real;
{ Hyperbolic tangent of x radians.
  Definition : tanH(x) = sinH(x) / cotH(x) }

function cotH(x: real): real;
{ Hyperbolic cotangent of x radians.
  Definition : cotH(x) = cosH(x) / sinH(x)
  Condition  : x <> 0.0 }



IMPLEMENTATION


function PostInc(var n: integer): integer;
begin PostInc:= n; Inc(n) end;

function PreInc(var n: integer): integer;
begin Inc(n); PreInc:= n end;

function PostDec(var n: integer): integer;
begin PostDec:= n; Dec(n) end;

function PreDec(var n: integer): integer;
begin Dec(n); PreDec:= n end;

function even(n: integer): boolean;
begin even:= not odd(n) end;

function Imax(i,j: integer): integer;
begin if i>j then Imax:= i else Imax:= j end;

function Imin(i,j: integer): integer;
begin if i<j then Imin:= i else Imin:= j end;

function Rmax(x,y: real): real;
begin if x>y then Rmax:= x else Rmax:= y end;

function Rmin(x,y: real): real;
begin if x<y then Rmin:= x else Rmin:= y end;

function Isign(n: integer): integer;
begin if n=0 then Isign:= 0 else if n>0 then Isign:= 1 else Isign:= -1 end;

function Rsign(x: real): real;
begin
  if x=0.0 then Rsign:= 0.0 else if x>0.0 then Rsign:= 1.0 else Rsign:= -1.0
end;

function log2(x: real): real;
begin log2:= ln(x)/Ln2 end;

function log10(x: real): real;
begin log10:= ln(x)/Ln10 end;

function logg(g,x: real): real;
begin logg:= ln(x)/ln(g) end;

function RpowerR(x,e: real): real;
begin if x=0.0 then RpowerR:= 0.0 else RpowerR:= exp(e*ln(x)) end;

function RpowerI(x: real; n: integer): real;
{ By Dennis E. Hamilton v1.08 Nov 11,1985
  Dr Dobbs Journal of Software Tools #112, Feb 1986 }
var i: integer; r: real;
begin
  if n=0
  then RpowerI:= x/x
  else begin
    i:= abs(n);
    while not odd(i) do begin x:= sqr(x); i:= i shr 1 end;
    r:= x;
    while i<>1 do begin
      repeat x:= sqr(x); i:= i shr 1 until odd(i);
      r:= r*x
    end;
    if n<0 then RpowerI:= 1.0/r else RpowerI:= r
  end
end;

function IpowerI(i,n: integer): integer;
var y: integer;
begin
  y:= 1;
  while n>0 do begin
    while not odd(n) do begin n:= n shr 1; i:= sqr(i) end;
    n:= pred(n); y:= y*i
  end;
  IpowerI:= y
end;

function rad(x: real): real;
begin rad:=x/Radian end;

function deg(x: real): real;
begin deg:= x*Radian end;

function tan(x: real): real;
begin tan:= sin(x)/cos(x) end;

function cot(x: real): real;
begin cot:= cos(x)/sin(x) end;

function sec(x: real): real;
begin sec:= 1.0/cos(x) end;

function cosec(x: real): real;
begin cosec:= 1.0/sin(x) end;

function ArcSin(x: real): real;
begin
  if x=1.0 then ArcSin:= HalfPi
  else if x=-1.0 then ArcSin:= -HalfPi
  else ArcSin:= ArcTan(x/sqrt(1.0-sqr(x)))
end;

function ArcCos(x: real): real;
begin
  if x=1.0 then ArcCos:= 0.0
  else if x=-1.0 then ArcCos:= Pi
  else ArcCos:= HalfPi-ArcTan(x/sqrt(1.0-sqr(x)))
end;

function ArcCot(x: real): real;
begin ArcCot:= HalfPi-ArcTan(x) end;

function sinH(x: real): real;
begin sinh:= (exp(x)-exp(-x))/2.0 end;

function cosH(x: real): real;
begin cosh:= (exp(x)+exp(-x))/2.0 end;

function tanH(x: real): real;
begin tanh:= 1.0-2.0/(exp(2.0*x)+1.0) end;

function cotH(x: real): real;
begin coth:= 1.0+2.0/(exp(2.0*x)-1.0) end;

END.
