(*****************************************************************************

  TextLine
    Version 1.0

  Purpose:
    This unit holds all the routines that handles basic text lines.  Text
    lines are structures that are similar to strings, but much larger.

  Features:
    This unit is not intended to operate as a stand alone unit.
    Routines to provide basic functions on the behalf of text lines.
    Code to check for errors as they occur.
    Code to perform automatic tab expansion during line reading.

  Limitations:
    This unit should not be overlaid.
    Several routines can't work with dynamic text lines, only static lines.

  CopyRight 1994, All rights reserved.
    By Paul Renaud

  Compiler:
    Turbo Pascal versions 5.0 to 6.0

  System:
    MS-DOS, MDOS

*****************************************************************************)

Unit TextLine;

  Interface

    Const
     { Define the maximum size of a single text line.  Range 100 to 65500 }
      Line_Limit = 8000;
      Automatic_Tab_Expansion = True;
     { Defines the tab character control code. }
      Tab_Character = #9;
     { Defines the standard tab displacement. }
      Tab_Amount = 8;

    Type
     { Range of characters in the line. }
      Line_Range_Type = 1 .. Line_Limit;
     { Defines the line data array. }
      Line_Array_Type = Packed array [ Line_Range_Type ] of Char;
     { Defines the size of the line data structure. }
      Line_Size_Type = 0 .. Line_Limit;
     { Defines a pointer to the line type data structure. }
      Line_Pointer_Type = ^Line_Type;
     { Defines the line type data structure. }
      Line_Type = Record
                    Size: Line_Size_Type;
                    Data: Line_Array_Type;
                  End;
     { Defines file pointer for specific procedures. }
      Point_Type = Record
                     Row: LongInt;
                     Column: Word;
                   End;

    Var
     { Defines the condition of the error }
      Line_Error: Byte;

(***********************************************************

  Function: Calculate size

    This function returns the maximum size needed to store
    the given line in a dynamic fashion.  ( taking only
    exactly the amount of memory necessary. )

***********************************************************)

    Function Calculate_Size( Var Line: Line_Type ): Word;

(***********************************************************

  Procedure: Retrieve line.

    This procedure takes a dynamically stored text line in
    Buffer and copies it to a static variable in Line.

***********************************************************)

    Procedure Retrieve_Line( Buffer: Pointer; Var Line: Line_Type );

(***********************************************************

  Procedure: Combine lines.

    This procedure adds the static Source text line to the
    static Destination line.  All extra data is ignored.

***********************************************************)

    Procedure Combine_Lines( Var Destination, Source: Line_Type );

(***********************************************************

  Procedure: Append line.

    This procedure adds the given character to the end of
    the text line if possible.

***********************************************************)

    Procedure Append_Line( Var Line: Line_Type; Data: Char );

(***********************************************************

  Procedure: Insert line.

    This procedure inserts the given character into the
    text line at the given location.

***********************************************************)

    Procedure Insert_Line( Var Line: Line_Type; Where: Word; Data: Char );

(***********************************************************

  Procedure: Delete line.

    This procedure deletes the characters in the text line
    at the given location.

***********************************************************)

    Procedure Delete_Line( Var Line: Line_Type; Where: Line_Range_Type; Amount: Word );

(***********************************************************

  Function: Copy line.

    This function makes a string copy of the given line
    starting at Start and ending when the length is equal
    to Length.   The string is extended with spaces when
    necessary.

***********************************************************)

    Function Copy_Line( Var Line: Line_Type; Start: Word; Length: Byte ): String;

(***********************************************************

  Procedure: Reduce line.

    This procedure removes the extra blanks at the end of
    the text line.  This is very useful for storage.

***********************************************************)

    Procedure Reduce_Line( Var Line: Line_Type );

(***********************************************************

  Function: Allocate line.

    This function takes the given text line and allocates
    a pointer for it and copies the data into the memory
    at the pointer.  If no memory is available, this
    function may return a nil pointer.

***********************************************************)

    Function Allocate_Line( Var Buffer: Line_Type ): Pointer;

(***********************************************************

  Function: Read line.

    This function attempts to read in a text line from the
    given text file.  If it fails, it returns false.

***********************************************************)

    Function Read_Line( Var InFile: Text; Var Line: Line_Type ): Boolean;

(***********************************************************

  Function: Write line.

    This function attempts to write out to the text file,
    the given text line.  If it fails, it returns false.

***********************************************************)

    Function Write_Line( Var OutFile: Text; Var Line: Line_Type ): Boolean;

(***********************************************************

  Function: Write line fast.

    This function attempts to write out to the text file,
    the given text line.  If it fails, it returns false.

***********************************************************)

    Function Write_Line_Fast( Var OutFile: File; Var Line: Line_Type ): Boolean;

(***********************************************************

  Procedure: Combine together.

    This procedure combines the two text lines together
    starting at Start and ending at Finish.

***********************************************************)

    Procedure Combine_Together( Var Destination, Source: Line_Type; Start, Finish: Word );

(***********************************************************

  Function: Less than.

    This function returns true if line 1 is less than
    line 2.

***********************************************************)

    Function Less_Then( Var Line1, Line2: Line_Type ): Boolean;

(***********************************************************

  Procedure: Change line.

    This procedure is designed to alter the given line at
    the given space, The old value at the given line of the
    given old length is removed and the new string is
    inserted at that place.

***********************************************************)

    Procedure Change_Line( Var Line: Line_Type; Position, Old_Length: Word; New_String: String );

{----------------------------------------------------------------------------}

  Implementation

    Const
     { Constant used to write a line return in the file. }
      Write_Line_Value: Packed array[ 1 .. 2 ] of Char = ( #13, #10 );

    {$DEFINE Quick} { Select alternate code for faster processing speed. }

{----------------------------------------------------------------------------}

(*************************************************

  Function: Calculate size.
    As previously defined.

*************************************************)

    Function Calculate_Size( Var Line: Line_Type ): Word;
      Begin
        Calculate_Size := ( Line.Size + SizeOf( Line_Size_Type ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Retrieve line.
    As previously defined.

*************************************************)

    Procedure Retrieve_Line( Buffer: Pointer; Var Line: Line_Type );
      Var
        Buffer_Pointer: Line_Pointer_Type absolute Buffer;
      Begin
        Move( Buffer_Pointer^, Line, Calculate_Size( Buffer_Pointer^ ) );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Allocate line.
    As previously defined.

*************************************************)

    Function Allocate_Line( Var Buffer: Line_Type ): Pointer;
      Var
        Allocate_Size: Word;
        Temporary_Holder: Pointer;
      Begin
        Allocate_Size := Calculate_Size( Buffer );
        GetMem( Temporary_Holder, Allocate_Size );
        If ( Temporary_Holder <> Nil )
          then
            Move( Buffer, Temporary_Holder^, Allocate_Size )
          else
            Line_Error := 203;
        Allocate_Line := Temporary_Holder;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Reduce line.
    As previously defined.

*************************************************)

    Procedure Reduce_Line( Var Line: Line_Type );
      Begin
        While ( ( Line.Size > 0 ) and ( Line.Data[ Line.Size ] = ' ' ) ) do
          Dec( Line.Size );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Append line.
    As previously defined.

*************************************************)

    Procedure Append_Line( Var Line: Line_Type; Data: Char );
      Begin
        If ( Line.Size < Line_Limit )
          then
            Begin
              Inc( Line.Size );
              Line.Data[ Line.Size ] := Data;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Expand line.
    This procedure expands the given text line
    to full size.

*************************************************)

    Procedure Expand_Line( Var Line: Line_Type );
      Var
        Count: Word;
      Begin
        For Count := 1 to Line.Size do
          Line.Data[ Count ] := UpCase( Line.Data[ Count ] );
       {$IFDEF Quick}
        Count := ( Line_Limit - Line.Size );
        If ( Count > 0 )
          then
            FillChar( Line.Data[ Succ( Line.Size ) ], Count, ' ' );
       {$ELSE}
        For Count := Succ( Line.Size ) to Line_Limit do
          Line.Data[ Count ] := ' ';
       {$ENDIF}
        Line.Size := Line_Limit;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Less than.
    As previously defined.

*************************************************)

    Function Less_Then( Var Line1, Line2: Line_Type ): Boolean;
      Var
        Pointer: Word;
      Begin
        If ( Line1.Size < Line_Limit )
          then
            Expand_Line( Line1 );
        If ( Line2.Size < Line_Limit )
          then
            Expand_Line( Line2 );
        Pointer := 1;
        While ( Pointer < Line_Limit ) and ( Line1.Data[ Pointer ] = Line2.Data[ Pointer ] ) do
          Inc( Pointer );
        Less_Then := ( Line1.Data[ Pointer ] < Line2.Data[ Pointer ] );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Read line.
    As previously defined.

*************************************************)

    Function Read_Line( Var InFile: Text; Var Line: Line_Type ): Boolean;
      Var
        Okay: Boolean;
        Character: Char;
        Count: Byte;
      Begin
        Okay := True;
        Line.Size := 0;
       {$I-}
        If Automatic_Tab_Expansion
          then
            While ( Okay and ( not EOLN( InFile ) ) and ( Line.Size < Line_Limit ) ) do
              Begin
                Read( InFile, Character );
                Line_Error := IoResult;
                Okay := ( Line_Error = 0 );
                If ( Character <> Tab_Character )
                  then
                    Begin
                      Inc( Line.Size );
                      Line.Data[ Line.Size ] := Character;
                    End
                  else
                    For Count := Succ( Line.Size ) to ( Succ( Line.Size div Tab_Amount ) * Tab_Amount ) do
                      If ( Line.Size < Line_Limit )
                        then
                          Begin
                            Inc( Line.Size );
                            Line.Data[ Line.Size ] := ' ';
                          End;
              End
          else
            While ( Okay and ( not EOLN( InFile ) ) and ( Line.Size < Line_Limit ) ) do
              Begin
                Inc( Line.Size );
                Read( InFile, Line.Data[ Line.Size ] );
                Line_Error := IoResult;
                Okay := ( Line_Error = 0 );
              End;
        If Okay
          then
            Begin
              ReadLn( InFile );
              Line_Error := IoResult;
              Okay := ( Line_Error = 0 );
            End;
       {$I+}
        Read_Line := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Write line.
    As previously defined.

*************************************************)

    Function Write_Line( Var OutFile: Text; Var Line: Line_Type ): Boolean;
      Var
        Count: Word;
        Okay: Boolean;
      Begin
        Okay := True;
       {$I-}
        For Count := 1 to Line.Size do
          If Okay
            then
              Begin
                Write( OutFile, Line.Data[ Count ] );
                Line_Error := IoResult;
                Okay := ( Line_Error = 0 );
              End;
        If Okay
          then
            Begin
              WriteLn( OutFile );
              Line_Error := IoResult;
              Okay := ( Line_Error = 0 );
            End;
       {$I+}
        Write_Line := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Write line fast.
    As previously defined.

*************************************************)

    Function Write_Line_Fast( Var OutFile: File; Var Line: Line_Type ): Boolean;
      Var
        Okay: Boolean;
      Begin
        Okay := True;
       {$I-}
        If ( Line.Size > 0 )
          then
            Begin
              BlockWrite( OutFile, Line.Data, Line.Size );
              Line_Error := IoResult;
              Okay := ( Line_Error = 0 );
            End;
        If Okay
          then
            Begin
              BlockWrite( OutFile, Write_Line_Value, SizeOf( Write_Line_Value ) );
              Line_Error := IoResult;
              Okay := ( Line_Error = 0 );
            End;
       {$I+}
        Write_Line_Fast := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Insert line.
    As previously defined.

*************************************************)

    Procedure Insert_Line( Var Line: Line_Type; Where: Word; Data: Char );
      Var
        Pointer: Word;
      Begin
        If ( Line.Size < Line_Limit )
          then
            If ( Where < Line.Size )
              then
                Begin
                 {$IFDEF Quick}
                  Move( Line.Data[ Where ], Line.Data[ Succ( Where ) ], Succ( Line.Size - Where ) );
                 {$ELSE}
                  For Pointer := Line.Size downto Where do
                    Line.Data[ Succ( Pointer ) ] := Line.Data[ Pointer ];
                 {$ENDIF}
                  Inc( Line.Size );
                  Line.Data[ Where ] := Data;
                End
              else
                Begin
                  Inc( Line.Size );
                  Line.Data[ Line.Size ] := Data;
                End
          else
            Begin
              If ( Where < Line.Size )
                then
                  Begin
                   {$IFDEF Quick}
                    Move( Line.Data[ Where ], Line.Data[ Succ( Where ) ], ( Line.Size - Where ) );
                   {$ELSE}
                    For Pointer := Pred( Line.Size ) downto Where do
                      Line.Data[ Succ( Pointer ) ] := Line.Data[ Pointer ];
                   {$ENDIF}
                    Line.Data[ Where ] := Data;
                  End
                else
                  Line.Data[ Line.Size ] := Data;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Delete line.
    As previously defined.

*************************************************)

    Procedure Delete_Line( Var Line: Line_Type; Where: Line_Range_Type; Amount: Word );
     {$IFNDEF Quick}
      Var
        Count,
        Point: Word;
      Begin
        For Count := 1 to Amount do
          If ( Where < Line.Size )
            then
              Begin
                For Point := Where to Pred( Line.Size ) do
                  Line.Data[ Point ] := Line.Data[ Succ( Point ) ];
                Dec( Line.Size );
              End;
      End;
     {$ELSE}
      Var
        Point: Word;
      Begin
        If ( Where < Line.Size ) and ( Amount > 0 )
          then
            Begin
              Point := ( Where + Amount );
              While ( Point > Line.Size ) do
                Dec( Point );
              Move( Line.Data[ Point ], Line.Data[ Where ], Succ( Line.Size - Point ) );
              Line.Size := ( Line.Size - Amount );
            End;
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Function: Copy line.
    As previously defined.

*************************************************)

    Function Copy_Line( Var Line: Line_Type; Start: Word; Length: Byte ): String;
     {$IFDEF Quick}
      Var
        Amount: Byte;
        Temporary_String: String;
      Begin
        FillChar( Temporary_String[ 1 ], Length, ' ' );
        Temporary_String[ 0 ] := Chr( Length );
        If ( Line.Size >= Start )
          then
            Begin
              Amount := Length;
              If ( Pred( Start + Amount ) > Line.Size )
                then
                  Amount := Succ( Line.Size - Start );
              If ( Amount > 0 )
                then
                  Move( Line.Data[ Start ], Temporary_String[ 1 ], Amount );
            End;
        Copy_Line := Temporary_String;
      End;
     {$ELSE}
      Var
        Pointer: Word;
        Temporary_String: String;
      Begin
        Temporary_String := '';
        If ( Start > 0 )
          then
            Begin
              Pointer := Start;
              While ( Pointer <= Line.Size ) and ( Length > 0 ) do
                Begin
                  Temporary_String := Temporary_String + Line.Data[ Pointer ];
                  Inc( Pointer );
                  Dec( Length );
                End;
            End;
        For Pointer := 1 to Length do
          Temporary_String := Temporary_String + ' ';
        Copy_Line := Temporary_String;
      End;
     {$ENDIF}

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Combine lines.
    As previously defined.

*************************************************)

    Procedure Combine_Lines( Var Destination, Source: Line_Type );
      Var
        Count: Word;
      Begin
       {$IFDEF QUICK}
        Count := Source.Size;
        While ( ( Destination.Size + Count ) > Line_Limit ) do
          Dec( Count );
        If ( Count > 0 )
          then
            Move( Source.Data[ 1 ], Destination.Data[ Succ( Destination.Size ) ], Count );
        Destination.Size := ( Destination.Size + Count );
       {$ELSE}
        For Count := 1 to Source.Size do
          Append_Line( Destination, Source.Data[ Count ] );
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Combine together.
    As previously defined.

*************************************************)

    Procedure Combine_Together( Var Destination, Source: Line_Type; Start, Finish: Word );
      Var
        Count: Word;
      Begin
        If ( Finish > Source.Size )
          then
            Finish := Source.Size;
       {$IFNDEF Quick}
        For Count := Start to Finish do
          Append_Line( Destination, Source.Data[ Count ] );
       {$ELSE}
        Count := Succ( Finish - Start );
        While ( ( Destination.Size + Count ) > Line_Limit ) do
          Dec( Count );
        If ( Count > 0 )
          then
            Move( Source.Data[ Start ], Destination.Data[ Succ( Destination.Size ) ], Count );
        Destination.Size := Destination.Size + Count;
       {$ENDIF}
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Change line.
    As previously defined.

*************************************************)

    Procedure Change_Line( Var Line: Line_Type; Position, Old_Length: Word; New_String: String );
      Var
        Count: Word;
      Begin
        Append_Line( Line, ' ' );
        Delete_Line( Line, Position, Old_Length );
        Append_Line( Line, ' ' );
        For Count := Length( New_String ) downto 1 do
          TextLine.Insert_Line( Line, Position, New_String[ Count ] );
      End;

{----------------------------------------------------------------------------}

  End.

