/* ----------------------------------------------------------------------------	*/
/*	Originators:																*/
/*  	Tom Arnold,	Quark, Inc.,	300 S. Jackson, 	Denver, 	CO 80209	*/
/*		Ron Perry, 	Atex, Inc.,		165 Lexington Road,	Billerica, 	MA 01821	*/
/* ----------------------------------------------------------------------------	*/
/*	The source code in this file is provided by Quark and Atex for use in the	*/
/*	development of a Publishing Interchange Language. The code provided herein	*/
/*	is non-proprietary, non-copyrighted, and is for use in the public domain.	*/
/*	The source code contained herein is provided "as is".						*/
/* ----------------------------------------------------------------------------	*/
/*	QUARK AND ATEX DISCLAIM ALL WARRANTIES, EXPRESSED OR IMPLIED, WITH REGARD	*/ 
/*	TO THE ENCLOSED SOURCE CODE. QUARK AND ATEX DISCLAIM ANY IMPLIED WARRANTY,	*/
/*	INCLUDING, BUT NOT LIMITED TO ANY IMPLIED WARRANTY OF FITNESS FOR A			*/
/*	PARTICULAR PURPOSE OR MERCHANTABILITY.										*/
/* ----------------------------------------------------------------------------	*/


/*$ab ----------------------------- abstract ----------------------------------
  genmain.c   	This is a sample "main" routine that demonstrates the use of the
  				PIL API to export a PIL file representing a page. The example here
  				shows some data structures from a hypothetical product, called 
  				PAGEKLUDGE.
  				
  				This file should be compiled and linked with: pilapi.obj, 
  				pilparse.obj, pilglob.obj, pilgen.obj, stdlib. Note that
  				stdlib is required for the appropriate library routines
  				that perform I/O and memory allocation invoked from genmain.c.
  				
  				Since the API does no file i/o, error reporting, or run-time 
  				memory allocation (except for automatic variables on the stack),
  				the application layer, in this case, genmain.c must do all that.
  				The K&R stdio and stdlib functions are used in these examples.
  				However, if your environment is different, you can make the
  				necessary changes as required.
  				
	Techniques Illustrated in genmain.c
	-----------------------------------
														Page Number In
		Topic										 PIL Programmer's Guide Function
		-----										------------------------
		Basic Uses & Techniques								6
			Export from a page design application to PIL	6	main()					
			Buffer allocation								7	ap_init()
			File Opens, closes, etc.						7	ap_init()
			Getchar and Putchar functions					7	ap_putc()
			Error handling, reporting						8	error_report()
		Advanced Functions									8
			Buffer management								8	main()
			Generating content entities						10	main()
			Application specific attributes					10	ap_init_obj()

	Functions contained here:
	-------------------------
		main
		ap_getc
		ap_putc
		ap_init
		ap_cleanup
		process_component
		error_report
		flush_writebuf
		ap_init_obj
		ap_make_nte

  				
	PIL API functions called:
	------------------------
	
		Major functions for parsing and generating:
		------------------------------------------
     
		pil_pg_init				Initializes parser and generator
		pil_create_component	Create a component of the requested type in the 
								pil buffer at the next available location. This
								allocates space in the buffer, clears it, and sets
								up the structure with language-specific defaults. 
		pil_put_component		Write a pil_component to the open pil file


		Utility functions for controlling parser/generator actions:
		----------------------------------------------------------
	
		pil_set_buffer			Set up the buffer that will be used for reads.


		Utility functions for status:
		----------------------------
	
		pil_last_error			Returns the most recent PIL_ERROR
		pil_get_line_num		Returns the line number where the parser encountered
								an error.


		Utility functions for creating PIL data structures:
		--------------------------------------------------
	
		pil_create_asi				Create an application specific item
		pil_create_asa				Create an application specific attribute
		pil_create_nametbleentry 	A name table entry
		pil_create_objid			an object id (for text flow or group)


		Utility functions for PIL file generation:
		-----------------------------------------
	
		pil_put_comment			Write a comment into the pil file

---------------------------------------------------------------------------- */


/*$au ------------------------------ audits -----------------------------------
    Author: Tom Arnold
            Quark, Inc.
            300 South Jackson
            Denver, Colorado 80209
            (303) 934-2211

    Ver     Date       Who  Etc
    v01.01  24-apr-91  tla	Revised for PIL 5.0
    v01.00  12-feb-91  tla	Initial version
--------------------------------------------------------------------------- */


/*$ep --------------------------- header files ---------------------------- */

#include	<stdio.h>	/* For fopen, fclose, etc.					*/
#include	<stdlib.h>	/* malloc, realloc							*/
#include	"pildefs.h"	/* Pick up PIL data structures and defines	*/
#ifdef THINK_C
#include	<console.h>	/* for ccommand (THINK C ONLY) 				*/
#endif
/* ----------------------- defines for this file only --------------------- */

#define		DEFAULT_OUTPUT_FILE	"out.pil"
#define		LCDNAMELENGTH		12		/* 8 char root, 3 char ext, null */
#define 	WBUFSIZE 			512		/* write buffer size */

typedef struct OBJ {					/* object structure of PAGEKLUDGE */
	char name[LCDNAMELENGTH];			/* name of the element */
	char editor[LCDNAMELENGTH];			/* name of person assigned to edit it */
	char type[LCDNAMELENGTH];			/* type of element */
	char file[LCDNAMELENGTH];			/* content file name */
	char format[LCDNAMELENGTH];			/* text format used for it */
	int	x,y;							/* origin on page */
	int	width,height;					/* size */
} object;

typedef struct PG{						/* page structure of PAGEKLUDGE */
	char name[LCDNAMELENGTH];			/* name of the page */
	int width;							/* size of the page in points */
	int height;
	int objcount;						/* number of objects */
	object *objs;						/* pointer to array of objects */
}page;


/*$ld --------------------------- static  data ----------------------------- */

static char *output = NULL;						/* output file name 		 */
static FILE *outfile = NULL;					/* file number for writing   */
static char writebuf[WBUFSIZE];					/* buffer for writing        */
static int wbufpos = 0;							/* initial position in buf   */
static char *pilbuf = NULL;						/* pointer to buffer for api */
static size_t pil_buffer_size = 2048;			/* default size for pilbuf 	 */
static size_t cnt_buffering_size = 512;			/* def sz of content blocks  */


static object pageobjs[] =	{ /* measurements are in whole points */
							{"PG1Banner","","banner","banner.txt","ban1",
							 54,36,7*72,72},
							{"hdfire","joe","head","","hdhelv36",
							 54,2*72+36,246,36},
							{"bt1fire","harry","bt","fire","bttimes10",
							 54,3*72,117,7*72},
							{"bt2fire","harry","bt","fire","bttimes10",
							 54+117+12,3*72,117,7*72},
							{"ph1fire","mary","photo","fire.tif","",
							 54+246+12,2*72+36,246,360},
							{"adsears","louise","","sears.eps","",
							 54+246+22,2*72+36+360,246,216}
							};
static page thepage = { "Page 1", 8*72+36, 11*72, 
						sizeof(pageobjs)/sizeof(object), pageobjs
					  };

static char *headline = "Billy Martin Fired Yet Again"; /* content for hdfire */


/*$pr ---------------------------- prototypes ----------------------------- */

#ifdef PIL_ANSI_PROTOTYPES
int			ap_init(char *,size_t);
PIL_VOID	ap_cleanup(PIL_VOID);
PIL_VOID	ap_putc(PIL_UINT8);
PIL_VOID	error_report(PIL_ERROR);
int			flush_writebuf(PIL_VOID);
pil_name_table_entry *ap_make_nte(object *);
PIL_ERROR	ap_init_obj(pil_object *,object *);
#else
int			ap_init();
PIL_VOID	ap_cleanup();
PIL_VOID	ap_putc();
PIL_VOID	error_report();
int			flush_writebuf();
PIL_ERROR ap_init_obj();
pil_name_table_entry *ap_make_nte();
#endif

/*$co ---------------------------- code start ---------------------------- */


/*-------------------------------------------------------------------------*\
  ap_init

	Entry:	out			Name of output file to open
			bufsize		Size of buffer to allocate, in bytes

	Exit:	0 for success, -1 for failure
\*-----------------------------------------------------------------------*/

int ap_init(out,bufsize)
char *out;
size_t	bufsize;
{
	/* initialize read and write buffers */
	wbufpos = 0;

	/* allocate buffer */
	if ((pilbuf = (char *)malloc(bufsize)) == NULL)
		return(-1);

	/* open output file, if needed */
	if (*out) 
	{
		if ((outfile = fopen (out, "w")) == NULL) return (-1);
	}
	return (0);
}


/*-------------------------------------------------------------------------*\
  ap_cleanup
  
  	Called at the end of generating, or on error. This function flushes and
  	output that may be buffered, closes the output file and frees the pil 
  	buffer.

	Entry:	No input

	Exit:	No return value.
\*-----------------------------------------------------------------------*/

PIL_VOID ap_cleanup()
{
	if (outfile != NULL) {
		flush_writebuf();
		wbufpos = 0;
		fclose (outfile);
		outfile = NULL;
	}
	
	if (pilbuf) {
		free (pilbuf);
		pilbuf = NULL;
	}
}


/*-----------------------------------------------------------------------*\
  flush_writebuf

	Flush the write buffer.
 
	Entry:	PIL_UINT8 to write.

	Exit:	-1 if error or eof, else character read, cast as an int.
\*-----------------------------------------------------------------------*/

int flush_writebuf()
{
    if (outfile == NULL) return (0);
    return ((int) fwrite (writebuf, (size_t) 1, (size_t) wbufpos, outfile));
}


/* ----------------------------------------------------------------------------	*\
  ap_putc

    The applications "putchar" function. A pointer to this function is 
	passed to the api via pil_pg_init(). This routine knows what file to 
	write to, and checks to see that it has been opened. If the file 
	number is -1, it indicates that either the file is not open, or 
	an error has occurred on a previous write. Otherwise, ap_putc 
	buffers up characters and writes them when the buffer is full.

    NOTE: to ensure that all characters are written, flush_writebuf()
    must be called after the last call to ap_putc.

    Entry:  PIL_UINT8 to write.

    Exit:   None
\* ----------------------------------------------------------------------------	*/

#ifdef PIL_ANSI_PROTOTYPES
PIL_VOID ap_putc(PIL_UINT8 c)
#else
PIL_VOID ap_putc(c)
PIL_UINT8 c;
#endif
{
    if (outfile == NULL) return ;
    if (wbufpos == WBUFSIZE) 
	{
        if (flush_writebuf() != wbufpos) outfile = NULL;
        wbufpos = 0;
    }
    writebuf[wbufpos++] = c;
}


/*-----------------------------------------------------------------------*\
  error_report

	The applications error reporting function.
	
 
	Entry:	PIL_ERROR code.

	Exit:	none
\*-----------------------------------------------------------------------*/

PIL_VOID error_report(err)
PIL_ERROR err;
{
	switch (err) {
		case PIL_OUT_OF_MEMORY:
			printf("PIL_ERROR: Not enough room left in pil buffer\n");
			break;
		case PIL_END_OF_FILE:
			printf("PIL_ERROR: End of file was encountered on a read\n");
			break;
		case PIL_NO_BUFPTR:
			printf("PIL_ERROR: The pil buffer was not provided\n");
			break;
		case PIL_NO_ASALISTPTR:
			printf("PIL_ERROR: Asa list was expected, but not provided\n");
			break;
		case PIL_BAD_STRUCT:
			printf("PIL_ERROR: Bad structure found when generating\n");
			break;
		case PIL_BAD_SYNTAX:
			printf("PIL_ERROR: Parser encountered illegal input, line %lu\n",
					pil_get_line_num());
			break;
		case PIL_BAD_FILE_TYPE:
			printf("PIL_ERROR: Input to the parser is NOT a PIL file\n");
			break;
		case PIL_PREV_ERR:
			printf("PIL_ERROR: Previous error encountered by parser\n");
			break;
		case PIL_BAD_CBUF_SIZE:
			printf("PIL_ERROR: Invalid (zero) content buffering size\n");
			break;
		case PIL_BAD_CMPNT_TYPE:
			printf("PIL_ERROR: Invalid component type\n");
			break;
		case PIL_BAD_VALUE_TYPE:
			printf("PIL_ERROR: Invalid pil_value type\n");
			break;
		case PIL_NO_RW_FUNCS:
			printf("PIL_ERROR: Both getc and putc func ptrs are null\n");
			break;
		case PIL_NO_R_FUNC:
			printf("PIL_ERROR: getc function pointer is null\n");
			break;
		case PIL_NO_W_FUNC:
			printf("PIL_ERROR: putc function pointer is null\n");
			break;
		case PIL_UNKN_HDR_BORD:
			printf("PIL_ERROR: Unknown byte order in cnt data hdr\n");
			break;
		case PIL_BAD_UNITS:
			printf("PIL_ERROR: Invalid Units (canvas or object)\n");
			break;
		case PIL_ZERO_DIMENSION:
			printf("PIL_ERROR: Object or Canvas with zero dimension\n");
			break;
		case PIL_BAD_NTE:
			printf("PIL_ERROR: A name table entry is missing a field\n");
			break;
		case PIL_OPEN_CLIPPER:
			printf("PIL_ERROR: An open shape can't be used as a clipper\n");
			break;
		case PIL_ZERO_SHAPE:
			printf("PIL_ERROR: A shape can't have zero dimension\n");
			break;
		case PIL_UNKN_SHAPE:
			printf("PIL_ERROR: Unknown shape\n");
			break;
		case PIL_UNKN_PATHPT:
			printf("PIL_ERROR: Unknown path point type\n");
			break;
		case PIL_ZERO_LENGTH:
			printf("PIL_ERROR: A line can't have zero length\n");
			break;
		case PIL_NEED_MOVE_TO:
			printf("PIL_ERROR: Path must start with a move-to\n");
			break;
		case PIL_OPEN_CONTAINER:
			printf("PIL_ERROR: A container must be a closed shape\n");
			break;
		case PIL_UNKN_CM:
			printf("PIL_ERROR: An unknown color model\n");
			break;
		case PIL_UNKN_RENDEROP:
			printf("PIL_ERROR: An unknown render operation\n");
			break;
		case PIL_UNKN_FILLRULE:
			printf("PIL_ERROR: An unknown fill rule\n");
			break;
		case PIL_UNKN_LINEPOS:
			printf("PIL_ERROR: Invalid Line position\n");
			break;
		case PIL_BAD_OBJID:
			printf("PIL_ERROR: object id list with null id pointer\n");
			break;
		case PIL_BAD_MITERLIM:
			printf("PIL_ERROR: Miter limit must be >= 1.0\n");
			break;
		case PIL_BAD_COLOR:
			printf("PIL_ERROR: Color value out of range\n");
			break;
		case PIL_BAD_HEADER:
			printf("PIL_ERROR: Incorrect values in a cnt header\n");
			break;
		case PIL_BAD_ARGS:
			printf("PIL_ERROR: Invalid arguments passed to a pil func\n");
			break;
		case PIL_WRONG_CMPNT:
			printf("PIL_ERROR: Incorrect component type provided\n");
			break;
		case PIL_UNKN_HDR_ENC:
			printf("PIL_ERROR: Unknown data encoding in a content header.\n");
			printf("Encoding must be ascii or binary\n");
			break;
		case PIL_OK:
		default:
			break;
	}
}


/*	Generate Only Example -- Export from PAGEKLUDGE to PIL
	To illustrate export of PIL, here is a hypothetical export program
	for an application called PAGEKLUDGE. PIL does not support every 
	feature of PAGEKLUDGE directly; for example, PAGEKLUDGE stores an
	editor's name with an object. This is represented as an application
	specific item in PIL. Conversely, PAGEKLUDGE does not support many
	of the PIL constructs. It deals with ads, which are always created
	in EPS; images, which are always TIFF; and text, which it only
	transfers as ascii files, never as formatted text. The content for 
	headlines (object type "head") is passed inline via pil-content
	entities.
	
	The generate example creates a pil file to represent this page:
  -------------------------------------------- One pil-layout
  | ---------------------------------------- |   One name-table
  | |                Banner                | |   One canvas, 8.5 x 11 inches
  | |             504pt x 144pt            | |     Object for banner
  | |--------------------------------------| |     Object for headline
  | |  Headline 246x36 |                   | |     Object for 1st body text col
  | |------------------|   PHOTO  246x360  | |     Object for 2nd body text col
  | | BT1    || BT2    |                   | |     Object for Photo
  | |        ||        |                   | |     Object for Display ad
  | |        ||        |                   | |     
  | |        ||<-12pt  |                   | |   text-flow for headline
  | |        ||  gutter|                   | |   text-flow for body text
  | |        ||        |                   | | One pil-content, for headline
  | |        ||        |                   | |
  | |        ||        |                   | |
  | |        ||        |                   | | The page itself has:
  | | 117pt  ||        |                   | |   half inch margins top & bot
  | | column ||        |                   | |   3/4 inch margins left and right
  | |        ||        |-------------------| |   4 column dummy, 1 pica gutters
  | |        ||        |                   | |
  | |        ||        |   AD   246x216    | |   All measurements in points
  | |        ||        |                   | |	 where 72 points == exactly 1 inch
  | |        ||        |                   | |
  | |        ||        |    54 pt. margin->| |
  | |        ||        |                   | |
  | |        ||        |   36 pt margin    | |
  | |        ||        |        v          | |
  | ---------------------------------------- | 
  --------------------------------------------
 */


main(argc,argv)
int	argc;
char *argv[];
{
	FAST	int i;
	PIL_ERROR	err;
	pil_component	*pc;
	pil_name_table_entry PTRPTR ntepp;
	object	*obj;
	
#ifdef THINK_C
	argc = ccommand(&argv);
#endif

	/* Establish the output file name */
	if (argc > 1) output = argv[1];
	else output = DEFAULT_OUTPUT_FILE;

	/* Initialize the application */
	if (ap_init(output,pil_buffer_size)) 
	{
		printf ("Could not initialize the application\n");
		exit (-1);
	}

	/* Initialize the PIL API */
	err = pil_pg_init(pilbuf,(PIL_UINT32) pil_buffer_size,
					  NULL,ap_putc,
					  (PIL_UINT32)cnt_buffering_size,
					  (PIL_INT32)-1,
					  NULL
					 ); 
	
	/* Is the PIL API ok? */					   
	if (err != PIL_OK) 
	{
		error_report(err);
		ap_cleanup();
		exit (-1);
	}

	/* Create the layout start */
	if (pil_create_component(&pc,PIL_LAYOUT_START_C) != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}

	if(pil_append_string(thepage.name, 
						  &((pil_layout_start *)pc)->name) != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	if (pil_put_component(pc) != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	
	/* create the name table */
	pil_set_buffer(pilbuf,(PIL_UINT32) pil_buffer_size);
	if (pil_create_component(&pc,PIL_NAME_TABLE_C) != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	/* Now loop through our page and create name table entries as needed */
	ntepp = &((pil_name_table *)pc)->entries;
	for (i = 0, obj = thepage.objs; i < thepage.objcount; i++, obj++) {
		if ((*ntepp = ap_make_nte(obj)) != NULL) {
			ntepp = &(*ntepp)->next;
		}
	}
	/* Write the whole name table */
	if (pil_put_component(pc) != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	
	/* create the canvas */
	if (pil_create_component(&pc,PIL_CANVAS_C) != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	/*	Units and dimensions are required */
	/*	Units: number of nanometers per point (where 72 points = 1 inch) */
	((pil_canvas *)pc)->units = 25400000.0/72.0;
	((pil_canvas *)pc)->width = 612;		/* 8.5 inches */
	((pil_canvas *)pc)->height = 792;		/* 11 inches */

	/*	Other fields of the canvas are optional */
	if( pil_append_string(thepage.name, 
						  &((pil_canvas *)pc)->username) != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	if( pil_append_string("page", 
						  &((pil_canvas *)pc)->cantype) != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	if (pil_put_component(pc) != PIL_OK) {
		ap_cleanup();
		error_report(pil_last_error());
		return (-1);
	}
	
	/* Now loop through our page, creating and writing objects. The	*/
	/*	buffer gets "reset" each time through so we can get use a 	*/
	/*	small buffer without running out of room. */
	for (i = 0, obj = thepage.objs; i < thepage.objcount; i++, obj++) {
		pil_set_buffer(pilbuf,(PIL_UINT32) pil_buffer_size);
		if (pil_create_component(&pc,PIL_OBJECT_C) != PIL_OK) {
			ap_cleanup();
			error_report(pil_last_error());
			return (-1);
		}
		if (ap_init_obj((pil_object *)pc,obj) != PIL_OK) {
			ap_cleanup();
			error_report(pil_last_error());
			return (-1);
		}
		if (pil_put_component(pc) != PIL_OK) {
			ap_cleanup();
			error_report(pil_last_error());
			return (-1);
		}
	}
	/*	Put a little comment in the PIL file to show the way */
	pil_put_comment("End of objects, text flows follow.");
	
	/*	Create a text flow for the headline. PAGEKLUDGE will create		*/
	/*	a pil-content object for the headline, with the same name as	*/
	/*	the object itself. (This example does not show how PAGEKLUDGE	*/
	/*	"knows" that the headline and body text need text flows, or 	*/
	/*	how to link them together. The sophisticated algorithm that 	*/
	/*	produces that informaation is highly proprietary to the owners	*/
	/*	of PAGEKLUDGE) 													*/
	/*	Note, that the label of the pil_text_flow can be the same as 	*/
	/*	the name of an object */
	pil_set_buffer(pilbuf,(PIL_UINT32) pil_buffer_size);
	if (pil_create_component(&pc,PIL_TEXT_FLOW_C) != PIL_OK) {
		ap_cleanup();
		error_report(pil_last_error());
		return (-1);
	}
	if(pil_append_string("hdfire",&((pil_text_flow *)pc)->label) != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	if (pil_create_objid(&((pil_text_flow *)pc)->objects,"hdfire") != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	if (pil_create_objid(&((pil_text_flow *)pc)->content,"hdfire") != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	if (pil_put_component(pc) != PIL_OK) {
		ap_cleanup();
		error_report(pil_last_error());
		return (-1);
	}

	/*	Create the text flow for the body text. This text flow links 	*/
	/*	together the two body text objects, and references the single  	*/
	/*	content file named "fire". It also links the text flow to another */
	/*	layout named "layout 2" and a text flow on it called "another flow" */
	pil_set_buffer(pilbuf,(PIL_UINT32) pil_buffer_size);
	if (pil_create_component(&pc,PIL_TEXT_FLOW_C) != PIL_OK) {
		ap_cleanup();
		error_report(pil_last_error());
		return (-1);
	}
	if( pil_append_string("fire",&((pil_text_flow *)pc)->label) != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	if( pil_append_string("layout 2",&((pil_text_flow *)pc)->to_layout) != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	if( pil_append_string("another flow",&((pil_text_flow *)pc)->to_label) != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	if (pil_create_objid(&((pil_text_flow *)pc)->objects,"bt1fire") != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	if (pil_create_objid(&((pil_text_flow *)pc)->objects->next,"bt2fire") != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	if (pil_create_objid(&((pil_text_flow *)pc)->content,"fire") != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	if (pil_put_component(pc) != PIL_OK) {
		ap_cleanup();
		error_report(pil_last_error());
		return (-1);
	}

	/* Create the layout end */
	if (pil_create_component(&pc,PIL_LAYOUT_END_C) != PIL_OK) {
		ap_cleanup();
		error_report(pil_last_error());
		return (-1);
	}
	if (pil_put_component(pc) != PIL_OK) {
		ap_cleanup();
		error_report(pil_last_error());
		return (-1);
	}
	
	/*	Now create the pil-content entity to contain the ascii */
	/*	text for the headline */
	
	pil_put_comment("Now for the content of our headline for the \"fire\" story");
	pil_set_buffer(pilbuf,(PIL_UINT32) pil_buffer_size);
	if (pil_create_component(&pc,PIL_CONTENT_START_C) != PIL_OK) {
		ap_cleanup();
		error_report(pil_last_error());
		return (-1);
	}
	if( pil_append_string("hdfire",&((pil_content_start *)pc)->name) != PIL_OK) {
		error_report(pil_last_error());
		ap_cleanup();
		return (-1);
	}
	if (pil_put_component(pc) != PIL_OK) {
		ap_cleanup();
		error_report(pil_last_error());
		return (-1);
	}

	/*	Create the header, fill it in, and write it out */
	if (pil_create_component(&pc,PIL_CONTENT_HDR_C) != PIL_OK) {
		ap_cleanup();
		error_report(pil_last_error());
		return (-1);
	}
	((pil_content_hdr *)pc)->byteorder = PIL_BIG_ENDIAN;
	((pil_content_hdr *)pc)->encoding = PIL_ASCII_CONTENT;
	((pil_content_hdr *)pc)->escapechar = PIL_DFT_CNT_ESC;
	((pil_content_hdr *)pc)->begblkchar = PIL_DFT_CNT_BEG;
	((pil_content_hdr *)pc)->endblkchar = PIL_DFT_CNT_END;
	if (pil_put_component(pc) != PIL_OK) {
		ap_cleanup();
		error_report(pil_last_error());
		return (-1);
	}
	
	/* 	Create the data	 */

	if (pil_create_component(&pc,PIL_CONTENT_DATA_C) != PIL_OK) {
		ap_cleanup();
		error_report(pil_last_error());
		return (-1);
	}
	
	/*	Notice that the data does NOT have to be put in the pil buffer, */
	/*	just pointed to by the data pointer. */

	((pil_content_data *)pc)->data = headline;
	((pil_content_data *)pc)->length = pil_strlen(headline);
	
	if (pil_put_component(pc) != PIL_OK) {
		ap_cleanup();
		error_report(pil_last_error());
		return (-1);
	}
	


	if (pil_create_component(&pc,PIL_CONTENT_END_C) != PIL_OK) {
		ap_cleanup();
		error_report(pil_last_error());
		return (-1);
	}
	if (pil_put_component(pc) != PIL_OK) {
		ap_cleanup();
		error_report(pil_last_error());
		return (-1);
	}
	ap_cleanup();	/* flush the output buffer, close down the show */
}

/*-------------------------------------------------------------------------*\
  ap_make_nte

	This routine makes a name table entry, if necessary, to represent the 
	content for an object. It illustrates a couple of things.
	
	First, since the nametable is written before the corresponding objects,
	this routine creates names for the entries and for the values in the 
	name table based on knowledge of how the names for the objects themselves,
	and the text-flows that follow them, will be generated.
	
	Second, it does not create a name table entry for every object. For
	body text, a text-flow will be written to link together all the pieces,
	and link a single content entity or file to the text flow. Therefore,
	only one name table entry is needed for the content of the body text.
	It is created when the first body text object is found.
	
	Third, it shows how to use "inline" to identify that the content
	named in an entry will follow in the same file.
	
	Entry:	pobj	pointer to a pil_object
			obj		pointer to a PAGEKLUDGE object

	Exit:	PIL_ERROR (PIL_OK for success, else whatever error was
			encountered by the API functions called).
\*-----------------------------------------------------------------------*/
pil_name_table_entry *ap_make_nte(obj)
object *obj;
{
pil_name_table_entry *ntep;
char		*ntename;
PIL_NT_TYPE	content_type;
PIL_NT_DMN	domain_type;
PIL_NT_VAL	value_type;

	/*	PAGEKLUDGE handles all photos in tiff, ads as eps,			*/
	/*	and everything else as ascii text. For headlines, it 		*/
	/*	creates content entities in pil which are passed INLINE,	*/
	/*	i.e., in the same pil file as the associated layout. All	*/
	/*	other content is assumed to be in a file in the same    	*/
	/*	directory as the pil file, that is, an "lcd filename" 		*/

	if (!pil_strcmp(obj->type,"photo")) {
		content_type = PIL_CNT_TIFF;
		domain_type = PIL_DMN_LCD_FILE;
		value_type = PIL_VAL_OTHER;
		ntename = obj->name;
	}
	else if (!pil_strcmp(obj->type,"ad")) {
		content_type = PIL_CNT_EPS;
		domain_type = PIL_DMN_LCD_FILE;
		value_type = PIL_VAL_OTHER;
		ntename = obj->name;
	}
	else if (!pil_strcmp(obj->type,"head")) {
		content_type = PIL_CNT_ASCII_TEXT;
		domain_type = PIL_DMN_INLINE;
		value_type = PIL_VAL_INLINE;
		ntename = obj->name;
	}
	/****************************************************************/
	/*	In PAGEKLUDGE the body text objects are named bt1story,		*/
	/*	bt2story, etc. Only the first should get a name table entry	*/
	/*	since there will only be one content file associated with	*/
	/*	the entire text flow. The name of the content referenced by	*/
	/*	the text flow will be the same as the file name known by 	*/
	/*	PAGEKLUDGE                                                  */
	/****************************************************************/
	else if (!pil_strcmp(obj->type,"bt")) {
		if (obj->name[2] == '1' && (obj->name[3] < '0' || obj->name[3] > '9')) {
			content_type = PIL_CNT_ASCII_TEXT;
			domain_type = PIL_DMN_LCD_FILE;
			value_type = PIL_VAL_OTHER;
			ntename = obj->file;
		}
		else {					/* skip it if it's not the first bt */
			return (NULL);
		}
	}
	else {
		content_type = PIL_CNT_ASCII_TEXT;
		domain_type = PIL_DMN_LCD_FILE;
		value_type = PIL_VAL_OTHER;
		ntename = obj->name;
	}
	
	/* also, the entry name and "value" (filename) are always the same */
	/* and the domain is always lcd file name */
	if (pil_create_nametbleentry(&ntep, ntename, content_type,NULL,
								 domain_type,NULL,
								 value_type,obj->file) 
							     != PIL_OK) {
		ntep = NULL;
	}
	return (ntep);
}

/*-------------------------------------------------------------------------*\
  ap_init_obj

	This routine initializes a pil_object based on the contents of a 
	PAGEKLUDGE object.
	
	It shows how to set the mandatory fields (origin, dimensions);
	how to give a pil_object an optional id; how to use a pil_graphic;
	how to attach an application specific attribute.
	
	Entry:	pobj	pointer to a pil_object
			obj		pointer to a PAGEKLUDGE object

	Exit:	PIL_ERROR (PIL_OK for success, else whatever error was
			encountered by the API functions called).
\*-----------------------------------------------------------------------*/
PIL_ERROR ap_init_obj(pobj,obj)
FAST pil_object *pobj;
FAST object *obj;
{
PIL_ERROR err;
pil_asa	*asa;
pil_asi *asi;

	pobj->dest_rect.ul.x = obj->x;
	pobj->dest_rect.ul.y = obj->y;
	pobj->dest_rect.width = obj->width;
	pobj->dest_rect.height = obj->height;
	
	/*	In PAGEKLUDGE all images have a rectangular "trap rule", */
	/*	which is a one-point wide black box, surrounding them.   */
	/*	This is represented by an object graphic in PIL.		 */
	if (!pil_strcmp(obj->type,"photo")) {
		pobj->clipshape.type = PIL_RECTANGLE_SHAPE;
		pobj->clipshape.shape.rect.ul.x = obj->x;
		pobj->clipshape.shape.rect.ul.y = obj->y;
		pobj->clipshape.shape.rect.width = obj->width;
		pobj->clipshape.shape.rect.height = obj->height;
		pobj->graphic.shape.type = PIL_CLIPPER_SHAPE;
		pobj->graphic.renderop = PIL_STROKE_RENDER_OP;
		pobj->graphic.color_model = PIL_DEV_GRAY_CM;
		pobj->graphic.stroke_color.devgray.value = 1.0;
		pobj->graphic.linewidth = 1;
		if ((err = pil_append_string(obj->name,&pobj->rcname)) != PIL_OK)
			return(err);
	}
	else if (!pil_strcmp(obj->type,"ad")) {
		if ((err = pil_append_string(obj->name,&pobj->rcname)) != PIL_OK)
			return(err);
	}
	
	/* 	Any type of object can have an "editor" responsible for it in  */
	/*	PAGEKLUDGE. Create an application specific item to convey it.  */
	if (obj->editor[0] != '\0') {
		if ((err = pil_create_asi(&pobj->asi,"sample-editor-name")) != PIL_OK) {
			return(err);
		}
			
		/* create the application specific item */
		if (err = pil_create_asa(&(pobj->asi->asa),
								 "editor",					/* attr name */
								 (PIL_UINT32)1,				/* # of vals */
								 PIL_STRING_CODE,			/* type of data */
								 obj->editor)				/* ptr to data */
								 != PIL_OK) {
			return(err);
		}							 
	}
	/* The name used by PAGEKLUDGE becomes the object id in PIL. */
	if ((err = pil_append_string(obj->name,&pobj->id)) != PIL_OK)
		return(err);
	/*	And the object types recognized by PAGEKLUDGE are either text 
		or pictures, or unknown if PAGEKLUDGE has a null string for its
		object type */
	if (!obj->type[0]) pobj->objtype = PIL_OBJ_UNKNOWN;
	else if (!pil_strcmp(obj->type,"photo")) pobj->objtype = PIL_OBJ_PICTURE;
	else pobj->objtype = PIL_OBJ_TEXT;
	return (PIL_OK);
		
}


/************************End of genmain.c***************************/


