/* ----------------------------------------------------------------------------	*/
/*	Originators:																*/
/*  	Tom Arnold,	Quark, Inc.,	300 S. Jackson, 	Denver, 	CO 80209	*/
/*		Ron Perry, 	Atex, Inc.,		165 Lexington Road,	Billerica, 	MA 01821	*/
/* ----------------------------------------------------------------------------	*/
/*	The source code in this file is provided by Quark and Atex for use in the	*/
/*	development of a Publishing Interchange Language. The code provided herein	*/
/*	is non-proprietary, non-copyrighted, and is for use in the public domain.	*/
/*	The source code contained herein is provided "as is".						*/
/* ----------------------------------------------------------------------------	*/
/*	QUARK AND ATEX DISCLAIM ALL WARRANTIES, EXPRESSED OR IMPLIED, WITH REGARD	*/ 
/*	TO THE ENCLOSED SOURCE CODE. QUARK AND ATEX DISCLAIM ANY IMPLIED WARRANTY,	*/
/*	INCLUDING, BUT NOT LIMITED TO ANY IMPLIED WARRANTY OF FITNESS FOR A			*/
/*	PARTICULAR PURPOSE OR MERCHANTABILITY.										*/
/* ----------------------------------------------------------------------------	*/


/*$ab ----------------------------- abstract ----------------------------------
  pilapi.c   	Contains interface routines for applications to access the pil
  				parser and generator.
  				
  				NO MEMORY IS ALLOCATED OR FREED BY THE FUNCTIONS IN THIS MODULE!
  				
  				NO I/O (FILE OPENS, CLOSES, READS, OR WRITES, ERROR PRINTING)
  				IS DONE DIRECTLY BY THE FUNCTIONS IN THIS MODULE.

  				ERRORS ARE COMMUNICATED TO THE APPLICATION BY RETURNING AN
  				ERROR CODE AND SETTING A GLOBAL ERROR WORD WITH THAT ERROR CODE. 
  				
  				A function is provided to return the last PIL_ERROR set.
  				
  				The functions in this module manipulate a buffer that is 
  				provided by the application. The parser/generator functions
  				use memory from the buffer sequentially, and keep track of 
  				what is used, by calling pil_alloc. However, there is no 
  				corresponding pil_free to give back parts of the buffer. It
  				is the application's responsibility to provide a buffer that
  				is big enough for the structures being built. When the buffer
  				is not large enough, PIL_OUT_OF_MEMORY will be returned by
  				the function that requested memory. It is then up to the 
  				application to provide a larger buffer, call pil_set_buffer,
  				and retry. In the case of parsing, PIL_OUT_OF_MEMORY requires 
  				restarting the parser at the beginning of the file.
  				
  				The application also provides pointers to functions to read
  				and write characters.  The application must open the 
  				necessary files for reading or writing the PIL, and set up the 
  				getchar or putchar function pointers, and provide a buffer for
  				the parser, before calling any function here that needs to read 
  				or write! (This is be done with pil_pg_init.)
  				
  				The application can call the functions here to:
  					read a pil file, putting pil structures into a buffer
  					create pil structures in a buffer
  					write pil structures from the buffer to a file
  					write comments to a file
  					
  			 	The functions that read or create pil structs will put structures 
  				at the next available location in the provided buffer, 
  				and return an error code (PIL_ERROR). A pointer to the structure
  				written, will be returned in a parameter passed to the function.

  				
	Major functions for parsing and generating:
	------------------------------------------
     
		pil_pg_init				Initializes parser and generator
		pil_get_component		Read the next thing from the file and put it in
								the pil buffer.
		pil_put_component		Write a pil_component to the open pil file
		pil_create_component	Create a component of the requested type in the 
								pil buffer at the next available location. This
								allocates space in the buffer, clears it, and sets
								up the structure with language-specific defaults. 


	Utility functions for controlling parser/generator actions:
	----------------------------------------------------------
	
		pil_set_asa_list		Set up a list of application-specific-attributes
								(asa) that the application wants the parser to 
								return, ignoring all others.
		pil_set_buffer			Set up the buffer that will be used for reads.
		pil_set_cnt_bsize		Set the content data buffering size. This value is
								initialized at pil_pg_init() time and can be modified
								via this function.
		pil_set_getc			Set get character function pointer
		pil_set_putc			Set put character function pointer


	Utility functions for status:
	----------------------------
	
        pil_get_mem_avail		Returns the number of bytes left in the pil buffer
		pil_last_error			Returns the most recent PIL_ERROR
		pil_get_version			Returns the language version and api version strings.
		pil_get_line_num		Returns the line number where the parser encountered
								an error.


	Utility functions for creating PIL data structures:
	--------------------------------------------------
	
        pil_alloc       			Allocates memory from the pil buffer
		pil_create_asi				Create an application specific item
		pil_create_asa				Create an application specific attribute
		pil_create_pvalue			A pil_value for a pil_asa
		pil_create_nametbleentry 	A name table entry
		pil_create_objid			an object id (for text flow or group)
		pil_create_pathpt			A path point (for a pil_path)
		pil_create_polypt			A polygon point
		pil_create_layout_start		Create layout start component
		pil_create_name_table		Create name table component
		pil_create_canvas			Create canvas component
		pil_create_object			Create object component
		pil_create_text_flow		Create text flow component
		pil_create_group			Create group component
		pil_create_layout_end		Create layout end component
		pil_create_content_start	Create content start component
		pil_create_content_hdr		Create content header component
		pil_create_content_data		Create content data component
		pil_create_content_end		Create content end component


	Utility functions for PIL file generation:
	-----------------------------------------
	
		pil_put_comment			Write a comment into the pil file
		pil_put_layout_start	Write a layout start component
		pil_put_name_table		Write a name table component
		pil_put_canvas			Write a canvas component
		pil_put_object			Write a object component
		pil_put_text_flow		Write a text flow component
		pil_put_group			Write a group component
		pil_put_layout_end		Write a layout end component
		pil_put_content_start	Write a content start component
		pil_put_content_hdr		Write a content header component
		pil_put_content_data	Write a content data component
		pil_put_content_end		Write a content end component
---------------------------------------------------------------------------- */


/*$au ------------------------------ audits -----------------------------------
    Author: Tom Arnold
            Quark, Inc.
            300 South Jackson
            Denver, Colorado 80209
            (303) 934-2211

    Ver     Date       Who  Etc
    v01.02  24-apr-91  tla	Upgraded for PIL 5.0
    v01.01  03-apr-91  tla	Fixed 2 bugs in pil_set_buffer, one in pil_alloc,
    	that could allow pointers to improper address boundaries, and allow
    	the buf to be overrun by several bytes. Changed pil_api_ver to b01.01
    v01.00  12-feb-91  tla	Initial version
---------------------------------------------------------------------------- */


/*$ep --------------------------- header files ---------------------------- */

#ifdef		THINK_DA	/* Support for THINK C Desk Accessory Architecture 	*/
#include 	<SANE.h>	/* Pickup num2str 									*/
#else					/* else 											*/
#include	<stdlib.h>	/* Use stdlib for atof								*/
#endif					/* end of THINK_DA 									*/
#include	"pildefs.h"	/* Pick up PIL data structures and defines			*/




/*$ed ------------------------ public global data -------------------------- */

#ifdef PIL_ANSI_PROTOTYPES
extern	PIL_VOID (FPTRTYPE *pil_putchar)(PIL_UINT8);/* Appl PutChar function */
extern	int (FPTRTYPE *pil_getchar)(PIL_VOID); 		/* Appl GetChar function */
#else
extern	PIL_VOID (FPTRTYPE *pil_putchar)();			/* Appl PutChar function */
extern	int (FPTRTYPE *pil_getchar)(); 				/* Appl GetChar function */
#endif
extern	char PTRPTR	pil_asa_list;		/* appl. specific attribute names    */
extern	PIL_INT32 pil_asa_count;		/* <0 = All, 0 = None ,>0 = use list */
extern	PIL_UINT32 pil_cnt_buf_size;	/* Buffering size for content data   */
extern	char PTRTYPE *pil_kws[];		/* Keyword pointer array             */


/*$ld --------------------------- static  data ----------------------------- */

/*	The PIL language version that this API interfaces to. The generator will */
/*	produce the version given below. The parser is able to parse any version */
/*	with the same major revision number (the part before the ".") and a minor */
/*	version number that is equal to or less than the minor version number of */
/*	pil_version. (E.g., if pil_version is "04.04" then the parser can also	 */
/*	understand version 4.01, 4.02, and 4.03) */

static char pil_version[6]={ '0', '5', '.', '0', '0', '\0' };

/*	And a version number for the API itself. This number also has a major and */
/*	a minor part. The major rev number changes when data structures or */
/*	function prototypes for the api change, or when functions are added or */
/*	removed from the api. The minor rev number changes with any code change to */
/*	this file, e.g., bug fixes, etc. */

static char	pil_api_ver[6]={ '0','2','.','0','0','\0' };

/* Last error code and PIL buffer accounting variables */

static PIL_ERROR	pil_error_val;		/* error word                       */
static char PTRTYPE *pil_bufstart;		/* beginning of buffer              */
static char PTRTYPE *pil_bufferptr;		/* next free space in the buffer    */
static PIL_UINT32	pil_bufremaining;	/* space left in buffer (bytes)     */

/*$co ---------------------------- code start ----------------------------- */
		

/* ------------------------------------------------------------------------	*/
/*		M A I N  S T R E A M  F U N C T I O N S								*/
/* ------------------------------------------------------------------------	*/
/*			pil_pg_init				(in this file)							*/
/*			pil_create_component	(in this file)							*/
/*			pil_get_component		(in pilparse.c)							*/
/*			pil_put_component		(in pilgen.c)							*/
/* ------------------------------------------------------------------------	*/


/*-----------------------------------------------------------------------*\
  pil_pg_init

	Tell the parser and generator to initialize any data structures needed.
	In addition, establish:
		o parser/generator buffer pointer and size
		o getchar function pointer
		o putchar function pointer
		o content data buffering size
		o which application specific items are to be reported to the caller
		
	ALL ARGUMENTS ARE REQUIRED. If only parsing is intended, pass NULL for
	putchar_p. If only generating is intended, pass NULL for getchar_p.

	The getchar function provided must:
		o Take no arguments (which means it must know what file to read)	
		o Return an int, with the ASCII character in the low-order 8 bits
		o Return a -1 for end of file

	The parser assumes that the application will have opened the
	required file for the getchar function, and has set up whatever 
	data (e.g., file pointer, read buffer, etc) the function requires 
	to read the PIL file, BEFORE calling any routines that call the 
	getchar function. These include: pil_get_component.

	The putchar function provided must:	
		o Take one argument, a character		
		o Returns nothing
	
	The generator assumes that the application will have opened the
	required file for the putchar function, BEFORE calling any routines 
	that call the putchar function. These include: pil_put_*.


	Entry:	bufptr		pointer to buffer for use of the parser/generator
			size		size of buffer in bytes
			getcharp	pointer to getchar function
			putcharp	pointer to putchar function
			cntbufsize	content data buffering size
			asi_count	<0 == All, 0 == None , >0 means use asi_names
			asi_names	application specific item names

	Exit:	error code (from pildefs.h)
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_pg_init (bufptr, size, getchar_p, putchar_p, cntbufsize,
asi_count, asi_name)

char PTRTYPE 	*bufptr;
PIL_UINT32		size;
#ifdef PIL_ANSI_PROTOTYPES
int 			(FPTRTYPE *getchar_p)(PIL_VOID);
PIL_VOID  		(FPTRTYPE *putchar_p)(PIL_UINT8);
#else
int 			(FPTRTYPE *getchar_p)();
PIL_VOID  		(FPTRTYPE *putchar_p)();
#endif
PIL_UINT32 		cntbufsize;
PIL_INT32		asi_count;
char PTRPTR		asi_name;
{
	if (pil_set_buffer (bufptr, size) != PIL_OK) return (pil_last_error());
	if (getchar_p == NULL && putchar_p == NULL) return (pil_set_error(PIL_NO_RW_FUNCS));
	pil_getchar = getchar_p;
	pil_putchar = putchar_p;


	if (pil_set_cnt_bsize (cntbufsize) != PIL_OK) return (pil_last_error());
	if (pil_set_asa_list (asi_count, asi_name) != PIL_OK) return (pil_last_error());
	pil_init_keywords ();
	pil_init_parser ();
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_create_component

	Creates a pil "component" of the requested type in the pil buffer.
	A pil component is defined in pilstruct.h.
	The component is put in the buffer at the next available location,
	and is initialized to the defaults as defined in the language spec.	
 
	Entry:	Pointer to a pil_component *, and type of requested component.

	Exit:	*cptrptr points to new component. Error code is returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_component (cptrptr,type)
FAST pil_component PTRPTR cptrptr;
PIL_COMP_TYPE type;
{
	if (type == PIL_LAYOUT_START_C)
		return (pil_create_layout_start((pil_layout_start PTRPTR)cptrptr));
	if (type == PIL_NAME_TABLE_C)
		return (pil_create_name_table((pil_name_table PTRPTR)cptrptr));
	if (type == PIL_CANVAS_C)
		return (pil_create_canvas((pil_canvas PTRPTR)cptrptr));
	if (type == PIL_OBJECT_C)
		return (pil_create_object((pil_object PTRPTR)cptrptr));
	if (type == PIL_TEXT_FLOW_C)
		return (pil_create_text_flow((pil_text_flow PTRPTR)cptrptr));
	if (type == PIL_GROUP_C)
		return (pil_create_group((pil_group PTRPTR)cptrptr));
	if (type == PIL_LAYOUT_END_C)
		return (pil_create_layout_end((pil_layout_end PTRPTR)cptrptr));
	if (type == PIL_CONTENT_START_C)
		return (pil_create_content_start((pil_content_start PTRPTR)cptrptr));
	if (type == PIL_CONTENT_HDR_C)
		return (pil_create_content_hdr((pil_content_hdr PTRPTR)cptrptr));
	if (type == PIL_CONTENT_DATA_C)
		return (pil_create_content_data((pil_content_data PTRPTR)cptrptr));
	if (type == PIL_CONTENT_END_C)
		return (pil_create_content_end((pil_content_end PTRPTR)cptrptr));
	return (pil_set_error(PIL_BAD_CMPNT_TYPE));
}


/* ------------------------------------------------------------------------	*/
/*		S E T  F U N C T I O N S											*/
/* ------------------------------------------------------------------------	*/
/*			pil_set_buffer													*/
/*			pil_set_cnt_bsize												*/
/*			pil_set_asa_list												*/
/*			pil_set_getc													*/
/*			pil_set_putc													*/
/* ------------------------------------------------------------------------	*/


/*-----------------------------------------------------------------------*\
  pil_set_buffer

	Give the parser/generator a place to put/look for pil
	structures. The buffer passed is used by the parser for all
	structures it creates, so the parser itself does not need to
	do any memory allocation or deallocation. 
	
	pil_set_buffer is called by pil_pg_init. It may also be called
	directly to reset the buffer pointer, for example, to reuse the
	same buffer on subsequent calls to pil_get_component or
	pil_create_component

	NOTE:	Some cpus require that certain sized structures must
			be aligned to a particular address boundaries. The most
			common form of this restriction is that integers or long 
			integers must not start on odd numbered addresses. To honor
			this requirement, pil_set_buffer make sure that the buffer
			address is a multiple of the constant PIL_MEM_ALIGN_SIZE, 
			which may be defined in pildefs.h
	
	Entry: Pointer to buffer, size of buffer in bytes.

	Exit:  PIL_NO_BUFPTR, if a NULL pointer or a size of zero is passed.
		   Else, PIL_OK. Initializes private data needed to maintain 
		   buffer.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_set_buffer (bufptr, size)
char PTRTYPE *bufptr;
PIL_UINT32	size;
{
	PIL_UINT32 align;
	if (bufptr == NULL || size == 0) return (pil_set_error(PIL_NO_BUFPTR));

#ifdef PIL_MEM_ALIGN_SIZE
	/* force alignment to the required address boundary */
	align = ((PIL_ADDR_T) bufptr % PIL_MEM_ALIGN_SIZE);
	if (align != 0)
	{
		bufptr += (PIL_MEM_ALIGN_SIZE - align);
		size -= (PIL_MEM_ALIGN_SIZE - align);
	}
#endif
	
	pil_bufstart = pil_bufferptr = bufptr;
	pil_bufremaining = size;	
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_set_cnt_bsize

	Sets the PIL content buffering size and returns PIL_OK if the given
	argument is valid. The buffering size must not be zero and must not
	be greater than the amount of memory remaining in the pil buffer.

	Entry: PIL_UINT32

	Exit: PIL_OK or PIL_BAD_CBUF_SIZE
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_set_cnt_bsize (cnt_bsize)
PIL_UINT32 cnt_bsize;
{
	if (cnt_bsize == 0) return (pil_set_error(PIL_BAD_CBUF_SIZE));
	if (cnt_bsize > pil_bufremaining) return (pil_set_error(PIL_BAD_CBUF_SIZE));
	pil_cnt_buf_size = cnt_bsize;
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_set_asa_list

	Set up a list of application-specific-attributes that the application 
	wants the parser to return, ignoring all others.
	NOTE: The list of attribute names is NOT copied. A pointer to it is 
	maintained for the parser. Therefore the application must not destroy,
	move, or change the list
 
	Entry:	Count of number of attributes to accept.
			Count = 0	don't return any application specific attributes
			Count < 0	return ALL application specific attributes
			Count > 0	return application specific attributes named in list
			Also, pointer to array of pointers to string representing the 
			names of the attributes to accept, if Count > 0.

	Exit:	PIL_NO_ASALISTPTR, if count > 0 and list ptr is null,
			else PIL_OK
\*-----------------------------------------------------------------------*/

PIL_ERROR	pil_set_asa_list (count, names)
PIL_INT32	count;
char PTRPTR names;			/* if count>0, names[] is char *names[count] */
{
	pil_asa_count = count;
	if (count > 0) 
	{
		if (names == NULL) return (pil_set_error(PIL_NO_ASALISTPTR));
		pil_asa_list = names;
	}
	else 
	{
		pil_asa_list = NULL;
	}
	return (PIL_OK);
}
	

/*-----------------------------------------------------------------------*\
  pil_set_getc

	Set the PIL get character function pointer

	Entry:	function pointer to your getc routine
	
	Exit:	PIL_NO_R_FUNC or PIL_OK
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_set_getc (getchar_p)
#ifdef		PIL_ANSI_PROTOTYPES
int (FPTRTYPE *getchar_p)(PIL_VOID);
#else
int (FPTRTYPE *getchar_p)();
#endif
{
	if (getchar_p == NULL) return (pil_set_error(PIL_NO_R_FUNC));
	pil_getchar = getchar_p;
	return (PIL_OK);
}
	

/*-----------------------------------------------------------------------*\
  pil_set_putc

	Set the PIL put character function pointer

	Entry:	function pointer to your putc routine. putc routine must take
			a PIL_UINT8 as input, and return nothing.
	
	Exit:	PIL_NO_W_FUNC or PIL_OK
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_set_putc (putchar_p)
#ifdef		PIL_ANSI_PROTOTYPES
PIL_VOID (FPTRTYPE *putchar_p)(PIL_UINT8);
#else
PIL_VOID (FPTRTYPE *putchar_p)();
#endif
{
	if (putchar_p == NULL) return (pil_set_error(PIL_NO_W_FUNC));
	pil_putchar = putchar_p;
	return (PIL_OK);
}


/* ------------------------------------------------------------------------	*/
/*		S T A T U S  F U N C T I O N S										*/
/* ------------------------------------------------------------------------	*/
/*			pil_get_version													*/
/*			pil_get_mem_avail												*/
/*			pil_last_error													*/
/*			pil_get_line_num (in pilparse.c)								*/
/* ------------------------------------------------------------------------	*/


/*-----------------------------------------------------------------------*\
  pil_get_version

	Get the version strings for the PIL language and the PIL API itself.

	Entry:	char **language - pointer to pointer for language version str.
		 	char **api - pointer to pointer for api version string

	Exit:	returns PIL_OK, 
			returns pointer to language and api version strings.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_get_version (language, api)
char	PTRPTR	language;
char	PTRPTR	api;
{
	*language = pil_version;
	*api = pil_api_ver;
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_get_mem_avail

	Returns the number of bytes available in the pil buffer.

	Entry: none

	Exit:  number of bytes unused at the end of the pil buffer
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_get_mem_avail (count)
PIL_UINT32 PTRTYPE *count;
{
	if (pil_bufferptr == NULL) 
	{
		*count = 0;
		return (pil_set_error(PIL_NO_BUFPTR));
	}
	else 
	{
		*count = pil_bufremaining;
		return (PIL_OK);
	}
}


/*-----------------------------------------------------------------------*\
  pil_last_error

	Returns the last PIL_ERROR recorded.

	Entry: none

	Exit:  value pil error word
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_last_error ()
{
	return (pil_error_val);
}


/* ------------------------------------------------------------------------	*/
/*		C R E A T I O N  F U N C T I O N S									*/
/* ------------------------------------------------------------------------	*/
/*	        pil_alloc														*/
/*			pil_create_asa													*/
/*			pil_create_asi													*/
/*			pil_create_nametbleentry 										*/
/*			pil_create_objid												*/
/*			pil_create_pathpt												*/
/*			pil_create_polypt												*/
/*			pil_create_pvalue												*/
/*			pil_create_layout_start											*/
/*			pil_create_name_table											*/
/*			pil_create_canvas												*/
/*			pil_create_object												*/
/*			pil_create_text_flow											*/
/*			pil_create_group												*/
/*			pil_create_layout_end											*/
/*			pil_create_content_start										*/
/*			pil_create_content_hdr											*/
/*			pil_create_content_data											*/
/*			pil_create_content_end											*/
/* ------------------------------------------------------------------------	*/


/*-----------------------------------------------------------------------*\
  pil_alloc

	Low-level routine that "allocates" memory chunks from the pil_buffer.
	Checks to see if the requested amount of memory is available in the
	buffer, and returns a pointer to the next available memory if it is.
	NOTE:	pil_set_buffer MUST be called before pil_alloc, or any 
			routine that calls pil_alloc. These include:
				pil_append_string
				pil_get_component
				pil_create_component
			If this is not done, the buffer pointer will be null and
			pil_alloc will return PIL_NO_BUFPTR.
	NOTE:	pil_alloc just uses memory sequentially from the global
			buffer set up by pil_set_buffer. It keeps track of what is 
			left.  
			Obviously, the application has a pointer to the buffer
			when it calls calls pil_set_buffer.
			HOWEVER, if the application wants to WRITE anything in that
			buffer, it should call pil_alloc to find the next place to 
			write. This will keep the application and parser from
			stepping on each other's data.
	NOTE:	Some cpus require that certain sized structures must
			be aligned to a particular address boundaries. The most
			common form of this restriction is that integers or long 
			integers must not start on odd numbered addresses. To honor
			this requirement, pil_alloc will only return pointers whose
			address is a multiple of the constant PIL_MEM_ALIGN_SIZE, 
			which may be defined in pildefs.h
	NOTE: 	there is no pil_free.
 
	Entry:	requested memory size
			pointer to pointer, which will be initialize to point to the 
				block allocated.

	Exit:	If o.k., PIL_OK and *memptrptr set to address of the
				requested block
			Else some other error code, and *memptrptr == NULL,
\*-----------------------------------------------------------------------*/

PIL_ERROR	pil_alloc(length, memptrptr)
PIL_UINT32	length;
PIL_VOID	PTRPTR memptrptr;
{	
#ifdef PIL_MEM_ALIGN_SIZE
	/* force alignment to a pointer-sized boundary */
	length += length % PIL_MEM_ALIGN_SIZE;
	length += (PIL_MEM_ALIGN_SIZE - (length % PIL_MEM_ALIGN_SIZE));
#endif


	if (pil_bufremaining < length) 
	{
		*memptrptr = NULL;
		return (pil_set_error(PIL_OUT_OF_MEMORY));
	}
	
	
	/* Make sure that the buffer pointer was initialized */
	if ((*memptrptr = (PIL_VOID PTRTYPE *) pil_bufferptr) == NULL) 
	{
		return (pil_set_error(PIL_NO_BUFPTR));
	}
	pil_bufremaining -= length;
	pil_bufferptr += length;
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_create_canvas

	Subroutine for pil_create_component.
	Creates a pil_canvas in the pil buffer.
	The pil_canvas is put in the buffer at the next available location.
 
	Entry:	Pointer to a pil_canvas *

	Exit:	*cptrptr points to new component. Error code is returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_canvas (cptrptr)
pil_canvas PTRPTR cptrptr;
{
	if (pil_alloc ((PIL_UINT32) sizeof(pil_canvas), 
		(PIL_VOID PTRPTR) cptrptr) != PIL_OK) 
		return (pil_last_error());


	pil_clear ((char PTRTYPE *) *cptrptr, (PIL_UINT32) sizeof(pil_canvas));
	(*cptrptr)->type = PIL_CANVAS_C;
	(*cptrptr)->clipshape.type = PIL_DFT_CNV_CLIPPER;
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_create_name_table

	Subroutine for pil_create_component.
	Creates a pil_name_table in the pil buffer.
	The pil_name_table is put in the buffer at the next available location,
	and is initialized to the defaults as defined in the language spec.	
 
	Entry:	Pointer to a pil_name_table *

	Exit:	*nptrptr points to new pil_name_table. Error code is returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_name_table (nptrptr)
pil_name_table PTRPTR nptrptr;
{
	if (pil_alloc ((PIL_UINT32) sizeof(pil_name_table), 
		(PIL_VOID PTRPTR) nptrptr) != PIL_OK) 
		return (pil_last_error());


	pil_clear ((char PTRTYPE *) *nptrptr, (PIL_UINT32) sizeof(pil_name_table));
	(*nptrptr)->type = PIL_NAME_TABLE_C;
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_create_nametblentry

	Creates a pil_name_table_entry in the pil buffer.
	The pil_name_table is put in the buffer at the next available location,
	and is initialized to the values passed.	
 
	Entry:	Pointer to a pil_name_table *
			char * 	name of entry
			PIL_INT16	content type of entry
			PIL_INT16	domain type
			char *	content string
			char *	value of entry

	Exit:	Error code is returned.
			nptrptr points to new pil_name_table_entry
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_nametbleentry (aptrptr, name, tcode, tstring,
dcode, dstring, vcode, vstring)

pil_name_table_entry PTRPTR aptrptr;	/* Destination for creation 		*/
char PTRTYPE 	*name;					/* entry name 						*/
PIL_NT_TYPE		tcode;					/* content type code 				*/
char PTRTYPE 	*tstring;				/* content type string 				*/
PIL_NT_DMN		dcode;					/* domain type code 				*/
char PTRTYPE 	*dstring;				/* domain name string 				*/
PIL_NT_VAL		vcode;					/* value type code 					*/
char PTRTYPE 	*vstring;				/* value name string 				*/
{
	FAST pil_name_table_entry PTRTYPE *nte;
	if (pil_alloc((PIL_UINT32)sizeof(pil_name_table_entry),
		(PIL_VOID PTRPTR)aptrptr) != PIL_OK) 
		return (pil_last_error());


	nte = *aptrptr;
	nte->next = NULL;
	if (pil_append_string(name,&nte->name) != PIL_OK) 
	return (pil_last_error());


	nte->content_type_code = tcode;
	if (pil_append_string(tstring,&nte->content_type_string) != PIL_OK)
	return (pil_last_error());


	nte->domain_type_code = dcode;
	if (pil_append_string(dstring,&nte->domain_type_string) != PIL_OK)
	return (pil_last_error());


	nte->value_type_code = vcode;
	if (pil_append_string(vstring,&nte->value_type_string) != PIL_OK)
	return (pil_last_error());
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_create_asi

	Creates a pil_asi in the pil buffer at the next 
	available location. It is given a name. The pil_asa
	can be created by calling pil_create_asa.
 
	Entry:	Pointer to a pil_asi *
			pointer to name of the pil_asi

	Exit:	*aptrptr points to new pil_asi. 
			Error code is returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_asi (aptrptr, name)
pil_asi PTRPTR aptrptr;
char PTRTYPE *name;
{
	FAST pil_asi PTRTYPE *asi;
	if (pil_alloc((PIL_UINT32)sizeof(pil_asi),
		(PIL_VOID PTRPTR)aptrptr) != PIL_OK) 
		return (pil_last_error());


	asi = *aptrptr;
	asi->next = NULL;
	asi->asa = NULL;
	if (pil_append_string(name, &asi->asi_name) != PIL_OK) 
		return (pil_last_error());
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_create_asa

	Creates a pil_asa in the pil buffer.
	The pil_asa is put in the buffer at the next 
	available location. It is given a name a list of values as passed
	to this function.
 
	Entry:	Pointer to a pil_asa *
			PIL_UINT32			number of PIL_VALUES for this attribute
			PIL_ASA_VAL_TYPE	code specifying data type for each pil_value
			char *				name of the pil_asa
			char *				pointer to an array of values of the 
								specified type

	Exit:	*aptrptr points to new pil_asa. 
			Error code is returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_asa (aptrptr, name, count, type, data)
pil_asa PTRPTR 		aptrptr;
char PTRTYPE 		*name;
PIL_UINT32			count;
PIL_ASA_VAL_TYPE	type;
char PTRTYPE 		*data;
{
	pil_value	PTRPTR pv;
	PIL_UINT32	itemsize;

	
	if (pil_alloc ((PIL_UINT32) sizeof(pil_asa), 
		(PIL_VOID PTRPTR) aptrptr) != PIL_OK) 
		return (pil_last_error());
		
		
	/* set the name, data type, and count of values for the attribute */
	(*aptrptr)->next = NULL;
	(*aptrptr)->count = count;
	(*aptrptr)->type = type;
	if (pil_append_string(name, &(*aptrptr)->attr_name) != PIL_OK)
		return (pil_last_error());
		
		
	if (count == 0)					/* no values, clear pointer */
	{
		(*aptrptr)->value = NULL;
	}
	else 							/* create the list of values */
	{
		/*  data is declared as a char * so that pointer addition can be done 	*/
		/*	in bytes. So now we get the size of the actual data items, and we 	*/
		/*	add that as we advance through the array. 							*/
		
		switch (type) 
		{
			case PIL_DOUBLE_CODE:
			case PIL_DOUBLE_LIST_CODE:
				itemsize = (PIL_UINT32) sizeof(PIL_DOUBLE);
				break;
				
			case PIL_INT16_CODE:
			case PIL_INT16_LIST_CODE:
			case PIL_UINT16_CODE:
			case PIL_UINT16_LIST_CODE:
				itemsize = (PIL_UINT32) sizeof(PIL_INT16);
				break;
				
			case PIL_INT32_CODE:
			case PIL_INT32_LIST_CODE:
			case PIL_UINT32_CODE:
			case PIL_UINT32_LIST_CODE:
				itemsize = (PIL_UINT32) sizeof(PIL_INT32);
				break;
				
			case PIL_STRING_CODE:
			case PIL_STRING_LIST_CODE:
				itemsize = (PIL_UINT32) sizeof(char PTRTYPE *);
				break;
				
			case PIL_INT8_CODE:
			case PIL_INT8_LIST_CODE:
			case PIL_UINT8_CODE:
			case PIL_UINT8_LIST_CODE:
				itemsize = (PIL_UINT32) sizeof(PIL_INT8);
				break;
				
			default:
				return (pil_set_error(PIL_BAD_VALUE_TYPE));
		}
		for (pv = &(*aptrptr)->value; count != 0; count--) 
		{
	    	if (pil_create_pvalue(pv,type,data) != PIL_OK) return (pil_last_error());
	    	data += itemsize;
	    	pv = &(*pv)->next;
		}
	}
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_create_pvalue

	Creates a pil_value in the pil buffer at the next available location.
 
	Entry:	Pointer to a pil_value *
			PIL_ASA_VAL_TYPE	code specifying data type for each pil_value
			char *				pointer to value for the pil_value

	Exit:	*aptrptr points to new pil_value. 
			Error code is returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_pvalue (aptrptr, type, data)
pil_value PTRPTR 	aptrptr;
PIL_ASA_VAL_TYPE	type;
FAST char PTRTYPE 	*data;
{
	FAST pil_value PTRTYPE *pv;


	if (pil_alloc ((PIL_UINT32) sizeof(pil_value), 
		(PIL_VOID PTRPTR) aptrptr) != PIL_OK) 
		return (pil_last_error());
		
		
	pv = *aptrptr;
	pil_clear ((char PTRTYPE *) pv, (PIL_UINT32) sizeof (pil_value));
	
	
	switch (type) 
	{
		case PIL_INT8_CODE:
		case PIL_INT8_LIST_CODE:
			pv->data.int8 = *(PIL_INT8 PTRTYPE *) data;
			break;
			
		case PIL_INT16_CODE:
		case PIL_INT16_LIST_CODE:
			pv->data.int16 = *(PIL_INT16 PTRTYPE *) data;
			break;
			
		case PIL_INT32_CODE:
		case PIL_INT32_LIST_CODE:
			pv->data.int32 = *(PIL_INT32 PTRTYPE *) data;
			break;
			
		case PIL_UINT8_CODE:
		case PIL_UINT8_LIST_CODE:
			pv->data.uint8 = *(PIL_UINT8 PTRTYPE *) data;
			break;
			
		case PIL_UINT16_CODE:
		case PIL_UINT16_LIST_CODE:
			pv->data.uint16 = *(PIL_UINT16 PTRTYPE *) data;
			break;
			
		case PIL_UINT32_CODE:
		case PIL_UINT32_LIST_CODE:
			pv->data.uint32 = *(PIL_UINT32 PTRTYPE *) data;
			break;
			
		case PIL_DOUBLE_CODE:
		case PIL_DOUBLE_LIST_CODE:
			pv->data.dbl = *(PIL_DOUBLE PTRTYPE *) data;
			break;
			
		case PIL_STRING_CODE:
		case PIL_STRING_LIST_CODE:
			if (pil_append_string(data, &pv->data.string)
				!= PIL_OK) return (pil_last_error());
			break;
			
		default:
			return (pil_set_error(PIL_BAD_VALUE_TYPE));
	}
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_create_objid

	Creates a pil_object_id_list in the pil buffer at the next 
	available location. 
 
	Entry:	Pointer to a pil_object_id_list *
			char *	pointer to name of the pil_object_id_list

	Exit:	*aptrptr points to new pil_object_id_list. 
			Error code is returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_objid (aptrptr, name)
pil_object_id_list PTRPTR aptrptr;
char PTRTYPE *name;
{
	if (pil_alloc ((PIL_UINT32) sizeof(pil_object_id_list),
		(PIL_VOID PTRPTR) aptrptr) != PIL_OK) 
		return (pil_last_error());


	(*aptrptr)->next = NULL;
	if (pil_append_string(name, &(*aptrptr)->id) != PIL_OK) 
		return (pil_last_error());
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_create_polypt

	Creates a pil_poly_pt in the pil buffer at the next 
	available location. 
 
	Entry:	Pointer to a pil_poly_pt *
			PIL_INT32	x value
			PIL_INT32 	y value

	Exit:	*aptrptr points to new pil_poly_pt. 
			Error code is returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_polypt (aptrptr, x, y)
pil_poly_pt PTRPTR aptrptr;
PIL_INT32	x,y;
{
	if (pil_alloc((PIL_UINT32)sizeof(pil_poly_pt),
		(PIL_VOID PTRPTR)aptrptr) != PIL_OK) 
		return (pil_last_error());


	(*aptrptr)->next = NULL;
	(*aptrptr)->p.x = x;
	(*aptrptr)->p.y = y;
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_create_pathpt

	Creates a pil_path_pt in the pil buffer at the next available location. 
 
	Entry:	Pointer to a pil_path_pt *
			char	type of point
			PIL_INT32	x value for first point
			PIL_INT32 	y value	for first point
			PIL_INT32	x value for second point (if any)
			PIL_INT32	y value for second point (if any)
			PIL_INT32	x value for third point (if any)
			PIL_INT32	y value for third point (if any)
			PIL_INT32	curve radius

	Exit:	*aptrptr points to new pil_poly_pt. 
			Error code is returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_pathpt (aptrptr, type, x1, y1, x2, y2, x3, y3, radius)
pil_path_pt PTRPTR 	aptrptr;
PIL_PATH_PART		type;
PIL_INT32			x1, y1;
PIL_INT32			x2, y2;
PIL_INT32			x3, y3;
PIL_INT32			radius;
{
	FAST pil_path_pt PTRTYPE *p;
	if (pil_alloc((PIL_UINT32)sizeof(pil_path_pt),
		(PIL_VOID PTRPTR)aptrptr) != PIL_OK) 
		return (pil_last_error());
		

	p = *aptrptr;
	p->type = type;
	p->next = NULL;
	
	
	switch (type) 
	{
		case PIL_MOVE_TO:
			p->p.mt.pt.x = x1;
			p->p.mt.pt.y = y1;
			break;
			
		case PIL_LINE_TO:
			p->p.lt.pt.x = x1;
			p->p.lt.pt.y = y1;
			break;
			
		case PIL_CURVE_TO:
			p->p.ct.ctrl_pt_1.x = x1;
			p->p.ct.ctrl_pt_1.y = y1;
			p->p.ct.ctrl_pt_2.x = x2;
			p->p.ct.ctrl_pt_2.y = y2;
			p->p.ct.end_pt.x = x3;
			p->p.ct.end_pt.y = y3;
			break;
			
		case PIL_ARC_TO:
			p->p.at.ctrl_pt_1.x = x1;
			p->p.at.ctrl_pt_1.y = y1;
			p->p.at.ctrl_pt_2.x = x2;
			p->p.at.ctrl_pt_2.y = y2;
			p->p.at.radius = radius;
			break;
	}
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_create_object

	Subroutine for pil_create_component.
	Creates a pil_object in the pil buffer.
	The pil_object is put in the buffer at the next available location,
	and is initialized to the defaults as defined in the language spec.	
 
	Entry:	Pointer to a pil_object *

	Exit:	*optrptr points to new pil_object. Error code is returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_object(optrptr)
pil_object PTRPTR optrptr;
{
	FAST pil_object PTRTYPE	*optr;
	if (pil_alloc ((PIL_UINT32) sizeof(pil_object), 
		(PIL_VOID PTRPTR) optrptr) != PIL_OK) 
		return (pil_last_error());


	optr = *optrptr;
	pil_clear ((char PTRTYPE *)optr, (PIL_UINT32) sizeof(pil_object));
	optr->type = PIL_OBJECT_C;
	
	
	optr->units 				= PIL_DFT_OBJ_UNITS;
	optr->rotation_angle 		= PIL_DFT_OBJ_ROTANG;
	optr->rotation_point.x 		= PIL_DFT_OBJ_ROTPT_X;
	optr->rotation_point.y 		= PIL_DFT_OBJ_ROTPT_Y;
	optr->clipshape.type 		= PIL_DFT_OBJ_CLIPPER;
	optr->container.type 		= PIL_DFT_OBJ_CONTAINER;
	optr->src_rect.ul.x 		= PIL_DFT_OBJ_SRCRECT_X;
	optr->src_rect.ul.y 		= PIL_DFT_OBJ_SRCRECT_Y;
	optr->src_rect.width 		= PIL_DFT_OBJ_SRCRECT_W;
	optr->src_rect.height 		= PIL_DFT_OBJ_SRCRECT_H;
	optr->rctype 				= PIL_DFT_OBJ_RCTYPE;
	optr->objtype				= PIL_DFT_OBJ_TYPE;
	optr->graphic.shape.type 	= PIL_DFT_OBJ_GRAPHIC;
	optr->graphic.fill_rule 	= PIL_DFT_FILL_RULE;
	optr->graphic.linecap 		= PIL_DFT_LINE_CAP;
	optr->graphic.linejoin 		= PIL_DFT_LINE_JOIN;
	optr->graphic.lineposition	= PIL_DFT_LINE_POSITION;
	optr->graphic.linewidth 	= PIL_DFT_LINE_WIDTH;
	optr->graphic.miter_limit 	= PIL_DFT_MITER_LIMIT;
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_create_text_flow

	Subroutine for pil_create_component.
	Creates a pil_text_flow in the pil buffer.
	The pil_text_flow is put in the buffer at the next available loc,
	and is initialized to the defaults as defined in the language spec.	
 
	Entry:	Pointer to a pil_text_flow *

	Exit:	*nptrptr points to new pil_text_flow. Error code is returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_text_flow(tfptrptr)
pil_text_flow	PTRPTR tfptrptr;
{
	if (pil_alloc ((PIL_UINT32) sizeof(pil_text_flow), 
		(PIL_VOID PTRPTR) tfptrptr) != PIL_OK) 
		return (pil_last_error());


	pil_clear ((char PTRTYPE *) *tfptrptr, (PIL_UINT32) sizeof(pil_text_flow));
	(*tfptrptr)->type = PIL_TEXT_FLOW_C;
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_create_group

	Subroutine for pil_create_component.
	Creates a pil_group in the pil buffer.
	The pil_group is put in the buffer at the next available loc,
	and is initialized to the defaults as defined in the language spec.	
 
	Entry:	Pointer to a pil_group *

	Exit:	*nptrptr points to new pil_group. Error code is returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_group (gptrptr)
pil_group PTRPTR gptrptr;
{
	if (pil_alloc ((PIL_UINT32) sizeof(pil_group), 
		(PIL_VOID PTRPTR) gptrptr) != PIL_OK) 
		return (pil_last_error());


	pil_clear ((char PTRTYPE *) *gptrptr, (PIL_UINT32) sizeof(pil_group));
	(*gptrptr)->type = PIL_GROUP_C;
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_create_layout_start

	Subroutine for pil_create_component.
	Creates a pil_layout_start in the pil buffer.
	The pil_layout_start is put in the buffer at the next available loc,
	and is initialized to the defaults as defined in the language spec.	
 
	Entry:	Pointer to a pil_layout_start *

	Exit:	*lsptrptr points to new pil_layout_start, Err code is returned
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_layout_start (lsptrptr)
pil_layout_start PTRPTR lsptrptr;
{
	if (pil_alloc ((PIL_UINT32) sizeof (pil_layout_start),
		(PIL_VOID PTRPTR) lsptrptr) != PIL_OK)
		return (pil_last_error());


	pil_clear ((char PTRTYPE *) *lsptrptr, (PIL_UINT32) sizeof(pil_layout_start));
	(*lsptrptr)->type = PIL_LAYOUT_START_C;
	return (pil_append_string (pil_version, &(*lsptrptr)->version));
}


/*-----------------------------------------------------------------------*\
  pil_create_layout_end

	Subroutine for pil_create_component.
	Creates a pil_layout_end in the pil buffer.
	The pil_layout_end is put in the buffer at the next available loc,
	and is initialized to the defaults as defined in the language spec.	
 
	Entry:	Pointer to a pil_layout_end *

	Exit:	*leptrptr points to new pil_layout_end. Error code is returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_layout_end (leptrptr)
pil_layout_end PTRPTR leptrptr;
{
	if (pil_alloc ((PIL_UINT32) sizeof (pil_layout_end),
		(PIL_VOID PTRPTR) leptrptr) != PIL_OK) 
		return (pil_last_error());
	(*leptrptr)->type = PIL_LAYOUT_END_C;
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_create_content_start

	Subroutine for pil_create_component.
	Creates a pil_content_start in the pil buffer.
	The pil_content_start is put in the buffer at the next available loc,
	and is initialized to the defaults as defined in the language spec.	
 
	Entry:	Pointer to a pil_content_start *

	Exit:	*csptrptr points to new pil_content_start, Err code returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_content_start (csptrptr)
pil_content_start PTRPTR csptrptr;
{
	if (pil_alloc ((PIL_UINT32) sizeof (pil_content_start),
		(PIL_VOID PTRPTR) csptrptr) != PIL_OK)
		return (pil_last_error());


	pil_clear ((char PTRTYPE *) *csptrptr, (PIL_UINT32) sizeof(pil_content_start));
	(*csptrptr)->type = PIL_CONTENT_START_C;
	return (pil_append_string (pil_version, &(*csptrptr)->version));
}


/*-----------------------------------------------------------------------*\
  pil_create_content_hdr

	Subroutine for pil_create_component.
	Creates a pil_create_content_hdr in the pil buffer.
	The pil_create_content_hdr is put in the buffer at the next available 
	location and is cleared. The application must fill in all the values,
	as all are required and there are no defaults.
 
	Entry:	chptrpptr - Pointer to a pil_content_hdr *

	Exit:	*chptrptr points to new pil_content_hdr, Err code returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_content_hdr (chptrptr)
pil_content_hdr	PTRPTR chptrptr;
{
	if (pil_alloc ((PIL_UINT32) sizeof(pil_content_hdr),
		(PIL_VOID PTRPTR) chptrptr) != PIL_OK)
		return (pil_last_error());


	pil_clear ((char PTRTYPE *) *chptrptr, (PIL_UINT32) sizeof(pil_content_hdr));
	(*chptrptr)->type = PIL_CONTENT_HDR_C;
	return (PIL_OK);
}


/*-----------------------------------------------------------------------*\
  pil_create_content_data

	Subroutine for pil_create_component.
	Creates a pil_content_data compontent in the pil buffer.
	The pil_content_data is put in the buffer at the next available loc,
	and is initialized to the defaults as defined in the language spec.	
	(The default is no data (zero length, NULL pointer).
	The application can initialize the length and the pointer to the data.
 
	Entry:	Pointer to a pil_content_data *

	Exit:	*cdptrptr points to new pil_content_data component
			Err code returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_content_data (cdptrptr)
pil_content_data PTRPTR cdptrptr;
{
	if (pil_alloc ((PIL_UINT32) sizeof(pil_content_data),
		(PIL_VOID PTRPTR) cdptrptr) != PIL_OK)
		return (pil_last_error());


	pil_clear ((char PTRTYPE *) *cdptrptr, (PIL_UINT32) sizeof(pil_content_data));
	(*cdptrptr)->type = PIL_CONTENT_DATA_C;
	return (PIL_OK);
}

/*-----------------------------------------------------------------------*\
  pil_create_content_end

	Subroutine for pil_create_component.
	Creates a pil_content_end in the pil buffer.
	The pil_content_end is put in the buffer at the next available loc.	
 
	Entry:	Pointer to a pil_content_end * 

	Exit:	*ceptrptr points to new pil_content_end. Error code is returned.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_create_content_end (ceptrptr)
pil_content_end	PTRPTR ceptrptr;
{
	if (pil_alloc ((PIL_UINT32) sizeof(pil_content_end),
		(PIL_VOID PTRPTR) ceptrptr) != PIL_OK)
		return (pil_last_error());
	(*ceptrptr)->type = PIL_CONTENT_END_C;
	return (PIL_OK);
}


/* ------------------------------------------------------------------------	*/
/*		U T I L I T Y  F U N C T I O N S									*/
/* ------------------------------------------------------------------------	*/
/*			pil_clear														*/
/*			pil_set_error													*/
/*			pil_append_string												*/
/* ------------------------------------------------------------------------	*/


/*-----------------------------------------------------------------------*\
  pil_clear

	Zero out a block of memory. This is provided because we don't want to
	assume the existence of ansi library routines such as memset(). However
	it may be wise to replace this call with whatever fast routine is 
	available on your platform to zero memory.	
 
	Entry:	Pointer to block of data, size in bytes

	Exit:	Writes zeros to the block.
\*-----------------------------------------------------------------------*/

PIL_VOID pil_clear (ptr, size)
FAST char PTRTYPE *ptr;
FAST PIL_UINT32 size;
{
	while (size--) *ptr++ = '\0';
}


/*-----------------------------------------------------------------------*\
  pil_set_error

	Sets the PIL_ERROR and returns the value set.

	Entry: PIL_ERROR

	Exit:  value  of pil error word
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_set_error (new_err)
PIL_ERROR new_err;
{
	return (pil_error_val = new_err);
}


/*-----------------------------------------------------------------------*\
  pil_append_string

	Low-level routine that copies a string into the pil buffer.
	Calls pil_alloc. See warning on pil_alloc.
 
	Entry:	Pointer to null-terminated (C-language style) string
			Pointer to a pointer in which to return new string address

	Exit:	Returns PIL_OK on success, PIL_OUT_OF_MEMORY on failure.
			Pointer to new string (in the buffer) is set if successful
			NULL if failure, and pil_error_val will be set.
\*-----------------------------------------------------------------------*/

PIL_ERROR pil_append_string (str, aptrptr)
char PTRTYPE *str;
char PTRPTR aptrptr;
{
	if (str == NULL) 
	{
		*aptrptr = NULL;
		return (PIL_OK);
	}
	else if (pil_alloc ((PIL_UINT32) pil_strlen (str) + 1, 
	(PIL_VOID PTRPTR) aptrptr) == PIL_OK) 
	{
		pil_strcpy (*aptrptr, str);
		return (PIL_OK);
	}
	else return (pil_last_error());
}


/* ------------------------------------------------------------------------	*/
/*		C  L I B R A R Y  F U N C T I O N S									*/
/* ------------------------------------------------------------------------	*/
/*			pil_atof														*/
/*			pil_strcmp														*/
/*			pil_strcpy														*/
/*			pil_strlen														*/
/* ------------------------------------------------------------------------	*/


PIL_DOUBLE pil_atof (ascii_num)
char PTRTYPE *ascii_num;
{
#ifdef THINK_DA
	PIL_UINT8	token [PIL_MAX_TOKEN_SIZE];
	PIL_UINT8	len;
	/* create a Pascal string for str2num */
	for (len = 1; (token[len] = *ascii_num) != '\0'; len++, ascii_num++) {}
	token[0] = len-1;
	return(str2num(token));
#else
#ifdef WINDOWS3
	/* stdlib routines require near pointers, so copy string to a local buf */
	PIL_UINT8	token [PIL_MAX_TOKEN_SIZE];
	pil_strcpy(token,ascii_num);
	return (atof (token));
#else
	return (atof (ascii_num));
#endif
#endif
}


PIL_INT16 pil_strcmp (s1, s2)
FAST char PTRTYPE *s1;
FAST char PTRTYPE *s2;
{
	while (*s1 == *s2++) if (*s1++ == '\0') return (0);
	return ((PIL_INT16) (*s1 - *--s2));
}


char PTRTYPE *pil_strcpy (dst, src)
FAST char PTRTYPE *dst;
FAST char PTRTYPE *src;
{
	FAST char PTRTYPE *outp;
	outp = dst;
	while (*dst++ = *src++) ;
	return (outp);
}


PIL_UINT32 pil_strlen (s)
FAST char PTRTYPE *s;
{
	FAST PIL_UINT32 n;
	n = 0;
	while (*s++) n++;
	return (n);
}


/* ------------------------------------------------------------------------	*/
/*		K E Y W O R D  T A B L E  F U N C T I O N S							*/
/* ------------------------------------------------------------------------	*/
/*			pil_init_keywords												*/
/* ------------------------------------------------------------------------	*/


/*-----------------------------------------------------------------------*\
  pil_init_keywords
  
	Subroutine for pil_pg_init. Initialize table of keyword strings used
	by the generator and parser.
 
	Entry:	none 

	Exit:	pil_kws array is initialized

\*-----------------------------------------------------------------------*/

PIL_VOID pil_init_keywords()
{
	pil_kws[PIL_KW_OBJECT]				=	"object";
	pil_kws[PIL_KW_TEXT_FLOW]			=	"text-flow";
	pil_kws[PIL_KW_GROUP]				=	"group";
	pil_kws[PIL_KW_UNKNOWN]				=	"unknown";
	pil_kws[PIL_KW_ASCII_TEXT]			=	"ascii-text";
	pil_kws[PIL_KW_ATEX_ARIS]			=	"atex-aris";
	pil_kws[PIL_KW_ATEX_ITF]			=	"atex-itf";
	pil_kws[PIL_KW_EPS]					=	"eps";
	pil_kws[PIL_KW_ICL]					=	"icl";
	pil_kws[PIL_KW_IT8_1]				=	"it8.1";
	pil_kws[PIL_KW_IT8_2]				=	"it8.2";
	pil_kws[PIL_KW_MS_RTF]				=	"microsoft-rtf";
	pil_kws[PIL_KW_MS_WORD_3]			=	"microsoft-word-3.0";
	pil_kws[PIL_KW_MS_WORD_4]			=	"microsoft-word-4.0";
	pil_kws[PIL_KW_MIF]					=	"mif";
	pil_kws[PIL_KW_PICT]				=	"pict";
	pil_kws[PIL_KW_PIL_LAYOUT_NTET]		=	"pil-layout";
	pil_kws[PIL_KW_POSTSCRIPT]			=	"postscript";
	pil_kws[PIL_KW_QUARK_XPRESSTAGS]	=	"quark-xpresstags";
	pil_kws[PIL_KW_SCITEX_CT]			=	"scitex-ct";
	pil_kws[PIL_KW_SCITEX_HANDSHAKE]	=	"scitex-handshake";
	pil_kws[PIL_KW_SCITEX_LW]			=	"scitex-lw";
	pil_kws[PIL_KW_SGML]				=	"sgml";
	pil_kws[PIL_KW_TIFF]				=	"tiff";
	pil_kws[PIL_KW_WP_100]				=	"word-perfect-1.0";
	pil_kws[PIL_KW_WP_102]				=	"word-perfect-1.02";
	pil_kws[PIL_KW_XYQUEST_XYWRITE]		=	"xyquest-xywrite";
	pil_kws[PIL_KW_MAC_FILENAME]		=	"mac-filename";
	pil_kws[PIL_KW_UNIX_FILENAME]		=	"unix-filename";
	pil_kws[PIL_KW_MSDOS_FILENAME]		=	"ms-dos-filename";
	pil_kws[PIL_KW_OS2_FILENAME]		=	"os2-filename";
	pil_kws[PIL_KW_VMS_FILENAME]		=	"vms-filename";
	pil_kws[PIL_KW_VM_FILENAME]			=	"vm-filename";
	pil_kws[PIL_KW_LCD_FILENAME]		=	"lcd-filename";
	pil_kws[PIL_KW_INLINE]				=	"inline";
	pil_kws[PIL_KW_NAME_TABLE]			=	"name-table";
	pil_kws[PIL_KW_CANVAS]				=	"canvas";
	pil_kws[PIL_KW_DIMENSIONS]			=	"dimensions";	
	pil_kws[PIL_KW_UNITS]				=	"units";
	pil_kws[PIL_KW_USER_NAME]			=	"user-name";
	pil_kws[PIL_KW_TYPE]				=	"type";
	pil_kws[PIL_KW_CLIPPER]				=	"clipper";
	pil_kws[PIL_KW_APPLICATION]			=	"application";
	pil_kws[PIL_KW_APP_NAME]			=	"app-name";
	pil_kws[PIL_KW_APP_INT8]			=	"int8";		
	pil_kws[PIL_KW_APP_INT16]			=	"int16";
	pil_kws[PIL_KW_APP_INT32]			=	"int32";
	pil_kws[PIL_KW_APP_UINT8]			=	"uint8";
	pil_kws[PIL_KW_APP_UINT16]			=	"uint16";
	pil_kws[PIL_KW_APP_UINT32]			=	"uint32";
	pil_kws[PIL_KW_APP_DOUBLE]			=	"double";
	pil_kws[PIL_KW_APP_STRING]			=	"string";
	pil_kws[PIL_KW_FILL_RULE]			=	"fill-rule";
	pil_kws[PIL_KW_RECTANGLE]			=	"rectangle";
	pil_kws[PIL_KW_CIRCLE]				=	"circle";
	pil_kws[PIL_KW_ELLIPSE]				=	"ellipse";
	pil_kws[PIL_KW_POLYGON]				=	"polygon";
	pil_kws[PIL_KW_CLOSED_PATH]			=	"closed-path";
	pil_kws[PIL_KW_EVEN_ODD]			=	"even-odd";
	pil_kws[PIL_KW_NZ_WINDING]			=	"non-zero-winding";
	pil_kws[PIL_KW_MOVE_TO]				=	"move-to";
	pil_kws[PIL_KW_LINE_TO]				=	"line-to";
	pil_kws[PIL_KW_CURVE_TO]			=	"curve-to";
	pil_kws[PIL_KW_ARC_TO]				=	"arc-to";
	pil_kws[PIL_KW_FLOW_LABEL]			=	"flow-label";
	pil_kws[PIL_KW_FLOW_OBJECTS]		=	"objects";
	pil_kws[PIL_KW_FLOW_FROM]			=	"from";
	pil_kws[PIL_KW_FLOW_TO]				=	"to";
	pil_kws[PIL_KW_FLOW_CONTENT]		=	"content";
	pil_kws[PIL_KW_ID]					=	"id";
	pil_kws[PIL_KW_GROUP_OBJECTS]		=	"objects";
	pil_kws[PIL_KW_ORIGIN]				=	"origin";
	pil_kws[PIL_KW_ROT_ANGLE]			=	"rot-angle";
	pil_kws[PIL_KW_ROT_POINT]			=	"rot-point";
	pil_kws[PIL_KW_CONTAINER]			=	"container";
	pil_kws[PIL_KW_GRAPHIC]				=	"graphic";
	pil_kws[PIL_KW_RC_TYPE]				=	"rc-type";
	pil_kws[PIL_KW_RC_NAME]				=	"rc-name";
	pil_kws[PIL_KW_BBOX]				=	"bbox";
	pil_kws[PIL_KW_RC_TYPE_TEXT]		=	"text";	
	pil_kws[PIL_KW_RC_TYPE_GEOMETRY]	=	"geometry";
	pil_kws[PIL_KW_RC_TYPE_LINEART]		=	"lineart";
	pil_kws[PIL_KW_RC_TYPE_IMAGE]		=	"image";
	pil_kws[PIL_KW_COLOR_MODEL]			=	"color-model";
	pil_kws[PIL_KW_DEV_GRAY]			=	"device-gray";
	pil_kws[PIL_KW_CAL_GRAY]			=	"calibrated-gray";
	pil_kws[PIL_KW_DEV_RGB]				=	"device-rgb";
	pil_kws[PIL_KW_CAL_RGB]				=	"calibrated-rgb";
	pil_kws[PIL_KW_DEV_CMYK]			=	"device-cmyk";
	pil_kws[PIL_KW_SPOT]				=	"spot";
	pil_kws[PIL_KW_PANTONE]				=	"pantone";
	pil_kws[PIL_KW_RENDER_OP]			=	"render-op";
	pil_kws[PIL_KW_STROKE]				=	"stroke";
	pil_kws[PIL_KW_FILL]				=	"fill";
	pil_kws[PIL_KW_FILL_STROKE]			=	"fill-and-stroke";
	pil_kws[PIL_KW_RENDER_ATTR]			=	"render-attributes";
	pil_kws[PIL_KW_STROKE_COLOR]		=	"stroke-color";
	pil_kws[PIL_KW_POSITION]			=	"position";
	pil_kws[PIL_KW_WIDTH]				=	"width";
	pil_kws[PIL_KW_CAP]					=	"cap";
	pil_kws[PIL_KW_JOIN]				=	"join";
	pil_kws[PIL_KW_MITER_LIMIT]			=	"miter-limit";
	pil_kws[PIL_KW_FILL_COLOR]			=	"fill-color";
	pil_kws[PIL_KW_INSIDE]				=	"inside";
	pil_kws[PIL_KW_OUTSIDE]				=	"outside";
	pil_kws[PIL_KW_CENTERED]			=	"centered";
	pil_kws[PIL_KW_BUTT_CAP]			=	"butt-cap";
	pil_kws[PIL_KW_ROUND_CAP]			=	"round-cap";
	pil_kws[PIL_KW_PROJECTING_CAP]		=	"projecting-cap";
	pil_kws[PIL_KW_MITER_JOIN]			=	"miter-join";
	pil_kws[PIL_KW_BEVEL_JOIN]			=	"bevel-join";
	pil_kws[PIL_KW_ROUND_JOIN]			=	"round-join";
	pil_kws[PIL_KW_LINE]				=	"line";
	pil_kws[PIL_KW_ARC]					=	"arc";
	pil_kws[PIL_KW_BEZIER]				=	"bezier";
	pil_kws[PIL_KW_POLY_LINE]			=	"poly-line";
	pil_kws[PIL_KW_OPEN_PATH]			=	"open-path";
	pil_kws[PIL_KW_DATA]				=	"data";
	pil_kws[PIL_KW_HEADER]				=	"header";
	pil_kws[PIL_KW_LITTLE_ENDIAN]		=	"little-endian";
	pil_kws[PIL_KW_BIG_ENDIAN]			=	"big-endian";
	pil_kws[PIL_KW_ASCII]				=	"ascii";
	pil_kws[PIL_KW_BINARY]				=	"binary";
	pil_kws[PIL_KW_LAYOUT]				=	"pil-layout-";
	pil_kws[PIL_KW_CONTENT]				=	"pil-content-";
	pil_kws[PIL_KW_LAYOUT_VS1]			=	"pil-layout-5.0";
	pil_kws[PIL_KW_LAYOUT_VS2]			=	"pil-layout-5.00";
	pil_kws[PIL_KW_LAYOUT_VS3]			=	"pil-layout-05.0";
	pil_kws[PIL_KW_LAYOUT_VS4]			=	"pil-layout-05.00";
	pil_kws[PIL_KW_CONTENT_VS1]			=	"pil-content-5.0";
	pil_kws[PIL_KW_CONTENT_VS2]			=	"pil-content-5.00";
	pil_kws[PIL_KW_CONTENT_VS3]			=	"pil-content-05.0";
	pil_kws[PIL_KW_CONTENT_VS4]			=	"pil-content-05.00";
	pil_kws[PIL_KW_CLIPPER_SHAPE]		=	"clipper-shape";
	pil_kws[PIL_KW_CONTAINER_SHAPE]		=	"container-shape";
	pil_kws[PIL_KW_OBJ_TEXT]			=	"text";
	pil_kws[PIL_KW_OBJ_PICTURE]			=	"picture";
	pil_kws[PIL_KW_OBJ_PRIMITIVE]		=	"primitive";
}
/******************* End of pilapi.c *********************/
