-- ----- dazuko.lua ---------------------------------------------
-- lua binding for the dazuko device
-- uses libdazuko, provides an OO interface
-- see the README for license details

_DAZUKO = {}
havelib = false

-- ===== library handling =======================================
-- {

function _DAZUKO:preplib()
	local dir, fn, ini
	local func, rc

	-- been there, done that?
	if (havelib) then
		return(true)
	end

	-- how to get the complete path name of the lib?
	-- but OTOH we don't seem to need it, "./" works
	dir = '.'
	fn = 'libluadazuko.so'
	ini = 'luaopen_libdazuko'

	-- load the library, run the init function
	func = loadlib(dir .. '/' .. fn, ini)
	if (func == nil) then
		return(false)
	end
	rc = func()

	-- check for success
	-- yes, this IS paranoid :)
	if ((type(rc) == "table") and
	    -- check that all the routines are there
	    (type(rc['register']) == "function") and
	    (type(rc['setmask']) == "function") and
	    (type(rc['addincl']) == "function") and
	    (type(rc['addexcl']) == "function") and
	    (type(rc['removeall']) == "function") and
	    (type(rc['getaccess']) == "function") and
	    (type(rc['unregister']) == "function") and
	    (type(rc['swallowsig']) == "function") and
	    -- check that all the constants are there
	    (type(rc['DAZUKO_ON_OPEN']) == "number") and
	    (type(rc['DAZUKO_ON_CLOSE']) == "number") and
	    (type(rc['DAZUKO_ON_EXEC']) == "number") and
	    (type(rc['DAZUKO_ON_CLOSE_MODIFIED']) == "number") and
	    (type(rc['DAZUKO_ON_UNLINK']) == "number") and
	    (type(rc['DAZUKO_ON_RMDIR']) == "number") and
	    -- this is for "unified syntax" and smaller diffs
	    0) then
		-- raise a flag that we've been here before
		havelib = true
		return(true)
	end
	return(false)

end

-- }
-- ===== the DAZUKO methods =====================================
-- {

-- register with the dazuko device
function _DAZUKO:register(group, mode)
	local id

	group = tostring(group)
	if (mode == nil) then mode = 'r' end
	mode = tostring(mode)
	id = libdazuko.register(group, mode)
	if (not id) then
		return(false)
	end

	self['id'] = id
	self['group'] = group
	self['mode'] = mode
	self['mask'] = 0
	self['incl'] = {}
	self['excl'] = {}

	return(true)
end

-- set the access mask
function _DAZUKO:setmask(mask)
	local rc, id

	mask = tonumber(mask) or 0

	id = self['id']
	rc = libdazuko.setmask(id, mask)
	if (not rc) then
		return(false)
	end

	self['mask'] = mask

	return(true)
end

-- add path(s) to the include or exclude list
function _DAZUKO:addpath(which, paths)
	local dazfunc
	local rc, id
	local idx, path

	-- determine which list to add to (and how to)
	if (which == 'incl') then
		dazfunc = libdazuko.addincl
	elseif (which == 'excl') then
		dazfunc = libdazuko.addexcl
	else
		return(false)
	end

	-- check which type of input we got
	if (type(paths) == "string") then
		path = tostring(paths)
		paths = { path }
	elseif (type(paths) == "table") then
		-- EMPTY
	else
		return(false);
	end

	-- we now have an array, loop through it
	id = self['id']
	for idx = 1, table.getn(paths) do
		path = tostring(paths[idx])
		rc = dazfunc(id, path)
		if (not rc) then
			return(false)
		end
		table.insert(self[which], path)
	end

	return(true)
end

-- calling "self:addpath()" won't work because addpath is "private"
-- so we (have to?) use the unusual(?) syntax we see here

-- add an include path
function _DAZUKO:addincl(paths)
	return(_DAZUKO.addpath(self, 'incl', paths))
end

-- add an exclude path
function _DAZUKO:addexcl(paths)
	return(_DAZUKO.addpath(self, 'excl', paths))
end

-- clear all includes and excludes
function _DAZUKO:removeall()
	local rc, id

	id = self['id']
	rc = libdazuko.removeall(id)
	if (not rc) then
		return(false)
	end

	self['incl'] = {}
	self['excl'] = {}

	return(true)
end

-- get, handle and return an access (in one call,
-- this is special compared to the pure C library)
function _DAZUKO:getaccess(cb)
	local id
	local hdl, data
	local deny

	-- insist in getting a function
	if (type(cb) ~= "function") then
		return(false)
	end

	-- get an access from the kernel
	id = self['id']
	rc = libdazuko.getaccess(id, cb)
	if (not rc) then
		return(false)
	end

	return(true)
end

-- unregister with the dazuko device
function _DAZUKO:unregister()
	local rc, id

	id = self['id']
	rc = libdazuko.unregister(id)
	if (not rc) then
		return(false)
	end

	self['id'] = nil
	self['group'] = nil
	self['mode'] = nil
	self['mask'] = nil
	self['incl'] = nil
	self['excl'] = nil

	return(true)
end

-- dump all parameters (development aid)
function _DAZUKO:dump()
	io.write("DAZUKO.dump()\n")
	io.write("\tid[", self['id'], "]\n")
	io.write("\tgroup[", self['group'], "], mode[", self['mode'], "], mask[", self['mask'], "]\n")
	if (self['incl'][1]) then
		io.write("\tincl[", table.concat(self['incl'], "] ["), "]\n")
	else
		io.write("\tincl[", "-- none --", "]\n")
	end
	if (self['excl'][1]) then
		io.write("\texcl[", table.concat(self['excl'], "] ["), "]\n")
	else
		io.write("\texcl[", "-- none --", "]\n")
	end
	io.write("end of dump\n")
	-- table.foreach(self, print)
end

-- }
-- ===== constructor ============================================
-- {

-- constructor
function DAZUKO()
	local ok
	local d

	ok = _DAZUKO.preplib()
	if (not ok) then
		return(nil)
	end

	d = {}
	d.register = _DAZUKO.register
	d.setmask = _DAZUKO.setmask
	d.addincl = _DAZUKO.addincl
	d.addexcl = _DAZUKO.addexcl
	d.removeall = _DAZUKO.removeall
	d.getaccess = _DAZUKO.getaccess
	d.unregister = _DAZUKO.unregister
	d.dump = _DAZUKO.dump

	return d
end

-- }
-- ----- E O F --------------------------------------------------
