-- ---- example.lua ---------------------------------------------
-- sample Lua script, demonstrates how to interface with Dazuko
-- (OO and procedure style, see bottom)
--
-- synopsis: [lua] example.lua <include> ...
--
-- see the README for license details

require 'dazuko'

-- handle a request
function dump(acc)
	-- dump the details
	io.write("\n")
	io.write("got an access:\n")
	table.foreach(acc, function(k, v)
		io.write("\t", tostring(k)
			,"\t", tostring(v)
			,"\n")
	end)
	-- return 0 -> grant access
	return(0)
end

-- preferences
local group, mode, mask, incl, excl

-- the main routine using OO style
function main_obj()
	local d, rc

	-- preparation
	d = DAZUKO()
	rc = d:register(group, mode)
	rc = d:setmask(mask)
	rc = d:addincl(incl)
	rc = d:addexcl(excl)

	-- loop getting accesses
	-- XXX how to register for SIGINT?
	while d:getaccess(dump) do
		-- EMPTY
	end

	-- cleanup
	rc = d:removeall()
	rc = d:unregister()
	d = nil
end

-- the main routine accessing libdazuko directly (procedural style)
function main_lib()
	local id, rc

	-- preparation
	id = libdazuko.register(group, mode)
	if (id == nil) then
		return
	end
	rc = libdazuko.setmask(id, mask)
	rc = libdazuko.addincl(id, unpack(incl))
	rc = libdazuko.addexcl(id, unpack(excl))

	-- loop getting accesses
	-- XXX how to register for SIGINT?
	while libdazuko.getaccess(id, dump) do
		-- EMPTY
	end

	-- cleanup
	rc = libdazuko.removeall(id)
	rc = libdazuko.unregister(id)
end

-- check parameters
function prep()
	local d
	local i

	-- load the library
	-- actually: have it loaded as a "side effect"
	d = DAZUKO()
	if (d == nil) then
		io.write("cannot load libdazuko / create instance\n")
		os.exit(1)
	end
	d = nil

	-- catch signals (they still interrupt getaccess()
	-- but they don't throw exceptions any longer)
	libdazuko.swallowsig()

	-- set up default values
	group = 'example:lua'
	mode = 'rw'
	mask = libdazuko.DAZUKO_ON_OPEN + libdazuko.DAZUKO_ON_CLOSE
	incl = {}
	excl = { '/proc/', '/dev/' }

	-- scan cmdline parameters
	for i = 1, table.getn(arg) do
		table.insert(incl, arg[i])
	end

	-- sanity check
	if (incl[1] == nil) then
		io.write("need at least one include path\n")
		os.exit(1)
	end

	-- dump parameters
	io.write("will register with group \"", group, "\" and mode \"", mode, "\"\n")
	io.write("access mask will be ", mask, "\n")
	io.write("include directories are:")
	table.foreach(incl, function(k, v) io.write("\n\t", v) end)
	io.write("\n")
	io.write("exclude directories are:")
	table.foreach(excl, function(k, v) io.write("\n\t", v) end)
	io.write("\n")
end

-- program entry point
prep()
if (true) then
	main_obj()
else
	main_lib()
end

-- ---- E O F ---------------------------------------------------
