/* ----- dazukomodule.c -------------------------------------- */
/* a Python binding for Dazuko */

/*
 * Copyright (c) 2004 Gerhard Sittig
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of Dazuko nor the names of its contributors may be used
 * to endorse or promote products derived from this software without specific
 * prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <Python.h>

#include <dazukoio.h>

static char RcsRevision[] = "$Revision: 1.1 $";

static PyObject *DazukoError = NULL;

static PyObject *
dazuko_register(PyObject *self, PyObject *args) {
	const char *group, *mode;
	int rc;

	mode = "r";
	if (! PyArg_ParseTuple(args, "s|s:register", &group, &mode))
		return NULL;

	rc = dazukoRegister(group, mode);
	if (rc != 0) {
		PyErr_SetString(DazukoError, "Register() failed");
		return(NULL);
	}
	Py_INCREF(Py_None);
	return(Py_None);
}

static PyObject *
dazuko_setAccessMask(PyObject *self, PyObject *args) {
	unsigned long mask;
	int rc;

	if (! PyArg_ParseTuple(args, "i:setAccessMask", &mask))
		return(NULL);

	rc = dazukoSetAccessMask(mask);
	if (rc != 0) {
		PyErr_SetString(DazukoError, "SetAccessMask() failed");
		return(NULL);
	}
	Py_INCREF(Py_None);
	return(Py_None);
}

static PyObject *
dazuko_addIncludePath(PyObject *self, PyObject *args) {
	const char *path;
	int rc;

	if (! PyArg_ParseTuple(args, "s:addIncludePath", &path))
		return(NULL);

	rc = dazukoAddIncludePath(path);
	if (rc != 0) {
		PyErr_SetString(DazukoError, "AddIncludePath() failed");
		return(NULL);
	}
	Py_INCREF(Py_None);
	return(Py_None);
}

static PyObject *
dazuko_addExcludePath(PyObject *self, PyObject *args) {
	const char *path;
	int rc;

	if (! PyArg_ParseTuple(args, "s:addExcludePath", &path))
		return(NULL);

	rc = dazukoAddExcludePath(path);
	if (rc != 0) {
		PyErr_SetString(DazukoError, "AddExcludePath() failed");
		return(NULL);
	}
	Py_INCREF(Py_None);
	return(Py_None);
}

static PyObject *
dazuko_removeAllPaths(PyObject *self, PyObject *args) {
	int rc;

	if (! PyArg_ParseTuple(args, ""))
		return(NULL);

	rc = dazukoRemoveAllPaths();
	if (rc != 0) {
		PyErr_SetString(DazukoError, "RemoveAllPaths() failed");
		return(NULL);
	}
	Py_INCREF(Py_None);
	return(Py_None);
}

static PyObject *
dazuko_getAccess(PyObject *self, PyObject *args) {
	struct dazuko_access *acc;
	int rc;
	PyObject *accref, *deny;
	PyObject *event, *flags, *mode, *uid, *pid;
	PyObject *filename, *file_size, *file_uid;
	PyObject *file_gid, *file_mode, *file_device;
	PyObject *result;

#define PREP_PARAM(name, format) do { \
	if (acc->set_ ##name) { \
		name = Py_BuildValue(format, acc-> name); \
	} else { \
		Py_INCREF(Py_None); \
		name = Py_None; \
	} \
} while (0)

	/* no input parameters */
	if (! PyArg_ParseTuple(args, ""))
		return(NULL);

	/* get an access (might block longer, allow threads) */
	Py_BEGIN_ALLOW_THREADS
	acc = NULL;
	rc = dazukoGetAccess(&acc);
	Py_END_ALLOW_THREADS

	/* shortcuts for the error cases */
	if (rc != 0) {
		PyErr_SetString(DazukoError, "GetAccess() failed (nonzero return code)");
		return(NULL);
	}
	if (acc == NULL) {
		PyErr_SetString(DazukoError, "GetAccess() failed (NULL access reference)");
		return(NULL);
	}
	if (! acc->set_event) {
		dazukoReturnAccess(&acc);
		PyErr_SetString(DazukoError, "GetAccess() failed (no access event code)");
		return(NULL);
	}

	/*
	 * setup the access parameters to get returned: create
	 * single objects, put them together into a dictionary
	 * and release the references since the dictionary will
	 * reference them from now on
	 */
	accref = Py_BuildValue("l", (long)acc);
	deny = Py_BuildValue("i", acc->deny);
	PREP_PARAM(event, "i");
	PREP_PARAM(flags, "i");
	PREP_PARAM(mode, "i");
	PREP_PARAM(uid, "i");
	PREP_PARAM(pid, "i");
	PREP_PARAM(filename, "s");
	PREP_PARAM(file_size, "l");
	PREP_PARAM(file_uid, "i");
	PREP_PARAM(file_gid, "i");
	PREP_PARAM(file_mode, "i");
	PREP_PARAM(file_device, "i");
	result = Py_BuildValue("{sOsOsOsOsOsOsOsOsOsOsOsOsO}"
		, "ref", accref, "deny", deny, "event", event
		, "flags", flags, "mode", mode, "uid", uid, "pid", pid
		, "filename", filename, "file_size", file_size
		, "file_uid", file_uid, "file_gid", file_gid
		, "file_mode", file_mode, "file_device", file_device
		);
	Py_XDECREF(accref);
	Py_XDECREF(deny);
	Py_XDECREF(event);
	Py_XDECREF(flags);
	Py_XDECREF(mode);
	Py_XDECREF(uid);
	Py_XDECREF(pid);
	Py_XDECREF(filename);
	Py_XDECREF(file_size);
	Py_XDECREF(file_uid);
	Py_XDECREF(file_gid);
	Py_XDECREF(file_mode);
	Py_XDECREF(file_device);

	return(result);
}

static PyObject *
dazuko_returnAccess(PyObject *self, PyObject *args) {
	long accref;
	int deny;
	struct dazuko_access *acc;
	int rc;

	deny = 0;
	if (! PyArg_ParseTuple(args, "l|i:returnAccess", &accref, &deny))
		return(NULL);
	acc = (struct dazuko_access *)accref;
	if (acc == NULL) {
		PyErr_SetString(DazukoError, "ReturnAccess() failed");
		return(NULL);
	}

	acc->deny = (deny != 0) ? 1 : 0;
	rc = dazukoReturnAccess(&acc);
	if (rc != 0) {
		PyErr_SetString(DazukoError, "ReturnAccess() failed");
		return(NULL);
	}
	Py_INCREF(Py_None);
	return(Py_None);
}

static PyObject *
dazuko_unregister(PyObject *self, PyObject *args) {
	int rc;

	if (! PyArg_ParseTuple(args, ""))
		return(NULL);

	rc = dazukoUnregister();
	if (rc != 0) {
		PyErr_SetString(DazukoError, "Unregister() failed");
		return(NULL);
	}
	Py_INCREF(Py_None);
	return(Py_None);
}

static char DazukoDoc[] = ""
	"Dazuko is a means to hook into a machine's file system and can be used\n"
	"to monitor (log for diagnostic purposes) or intercept (filter based on\n"
	"scan results) access to data (files and directories).  Dazuko is mostly\n"
	"used by antivirus scanners to achieve on access scan capabilites.\n"
	"\n"
	"See http://www.dazuko.org/ for additional information.\n"
	"\n"
	"The \"dazuko\" module provides a C like interface to Dazuko.  Roughly\n"
	"usage is like this:\n"
	"\n"
	"dazuko.register(\"group\", \"rw\")\n"
	"dazuko.setAccessMask(dazuko.ON_OPEN | dazuko.ON_CLOSE)\n"
	"dazuko.addIncludePath(\"/home\")\n"
	"dazuko.addExcludePath(\"/home/excl\")\n"
	"while running:\n"
	"	acc = dazuko.getAccess()\n"
	"	print acc['event'], acc['filename']\n"
	"	deny = check(acc)\n"
	"	dazuko.returnAccess(acc['ref'], deny)\n"
	"dazuko.unregister();\n"
	"\n"
	"NOTE: Make sure to spend as little time as necessary when holding an\n"
	"access -- the kernel is waiting for you!"
	;

static PyMethodDef DazukoMethods[] = {
    { "register",  dazuko_register, METH_VARARGS,
	"register(groupname)\n"
	"register(groupname, regmode)\n"
	"\n"
	"Registers the app with dazuko.  Needs to be done first.\n"
	"regmode can be \"rw\" or \"r\" (optional, \"r\" is the default)."
    },
    { "setAccessMask",  dazuko_setAccessMask, METH_VARARGS,
	"setAccessMask(3)\n"
	"setAccessMask(dazuko.ON_ACCESS | dazuko.ON_CLOSE)\n"
	"\n"
	"Sets the access mask. Constants are available."
    },
    { "addIncludePath",  dazuko_addIncludePath, METH_VARARGS,
	"addIncludePath(pathspec)\n"
	"\n"
	"Adds an include path.  Path specs must be absolute."
    },
    { "addExcludePath",  dazuko_addExcludePath, METH_VARARGS,
	"addExcludePath(pathspec)\n"
	"\n"
	"Adds an exclude path.  Path specs must be absolute."
    },
    { "removeAllPaths",  dazuko_removeAllPaths, METH_VARARGS,
	"removeAllPaths()\n"
	"\n"
	"Removes all previously set include and exclude paths."
    },
    { "getAccess",  dazuko_getAccess, METH_VARARGS,
	"acc = getAccess()\n"
	"\n"
	"Waits for an access, returns the parameters of the access.\n"
	"Accesses gotten MUST be returned to the kernel.  Make sure\n"
	"to spend as little time as necessary while holding the access,\n"
	"the kernel is waiting for you!"
    },
    { "returnAccess",  dazuko_returnAccess, METH_VARARGS,
	"returnAccess(acc['ref'])\n"
	"returnAccess(acc['ref'], do_deny)\n"
	"\n"
	"Return an access gotten from getAccess().  Have the kernel\n"
	"deny the operation to the original caller should do_deny\n"
	"have a nonzero value (optional, allow by default)."
    },
    { "unregister",  dazuko_unregister, METH_VARARGS,
	"unregister()\n"
	"\n"
	"Unregister with dazuko."
    },
    { NULL, NULL, 0, NULL, },
};

#ifndef PyMODINIT_FUNC
#define PyMODINIT_FUNC void
#endif
PyMODINIT_FUNC
initdazuko(void) {
	PyObject *m;
	char *rev, *sep;

	/* set up the module */
	m = Py_InitModule("dazuko", DazukoMethods);

	/* add an inline doc */
	PyModule_AddStringConstant(m, "__doc__", DazukoDoc);

	/* create the exception instance */
	DazukoError = PyErr_NewException("dazuko.error", NULL, NULL);
	Py_INCREF(DazukoError);
	PyModule_AddObject(m, "error", DazukoError);

	/* make our revision accessible */
	rev = RcsRevision;
	rev += 11;
	sep = strchr(rev, ' ');
	if (sep != NULL)
		*sep = '\0';
	PyModule_AddStringConstant(m, "VERSION", rev);

#define ADD_CONST(name) \
	PyModule_AddIntConstant(m, #name , DAZUKO_ ##name)

	/* add the access mask constants */
	ADD_CONST(ON_OPEN);
	ADD_CONST(ON_CLOSE);
	ADD_CONST(ON_EXEC);
	ADD_CONST(ON_CLOSE_MODIFIED);
	ADD_CONST(ON_UNLINK);
	ADD_CONST(ON_RMDIR);
}

/* ----- E O F ----------------------------------------------- */
