/*
** 1998-05-21 -	This module implements the built-in command "Copy". Very handy stuff!
** 1998-05-23 -	Added some error handling/reporting. Very far from bullet-proof, though. :(
** 1998-05-31 -	Added capability to copy device files (recreates them at destination using
**		mknod(), of course). Nice.
** 1998-06-04 -	Copied directories now try to have the same protection flags, rather than
**		trying to turn on as many bits as possible (as the old code did!).
** 1998-06-07 -	Added support for copying soft links. Originally wrote the code in cmd_move,
**		then moved it here and made the cmd_move code call it here.
** 1998-09-12 -	Changed all destination arguments to be full (with path and filename). This
**		makes the implementation of cmd_copys SO much easier.
** 1998-09-19 -	Now uses the new overwrite protection/confirmation module. Kinda cool.
** 1998-12-23 -	Now supports copying the source's access and modification dates, too.
** 1999-01-03 -	After complaints, I altered the copying command slightly; it no longer attempts
**		to avoid doing a new stat() call on the file being copied. This assures that
**		the size used for the copy is the most recent one known to the file system (if
**		you pretend there's no other programs running, that is).
** 1999-03-05 -	Moved over to new selection method, and it's semi-abstracted view of dir rows.
** 1999-04-06 -	Modified to use the new command configuration system.
** 2000-07-02 -	Initialized translation by marking up strings.
*/

#include "gentoo.h"

#include <fcntl.h>
#include <utime.h>

#include "cmd_delete.h"
#include "cmdseq_config.h"
#include "dialog.h"
#include "dirpane.h"
#include "errors.h"
#include "fileutil.h"
#include "overwrite.h"
#include "progress.h"
#include "strutil.h"

#include "cmd_copy.h"

#define	CMD_ID	"copy"

/* ----------------------------------------------------------------------------------------- */

typedef struct {			/* Options used by the "Copy" command (and relatives). */
	gboolean	modified;
	gboolean	copy_dates;		/* Copy access and modification dates? */
	gboolean	ignore_attrib_err;	/* Allow attribute-copying to fail silently. */
	gboolean	leave_fullsize;		/* If destination has same size as source, leave it even if error occured. */
	gsize		buf_size;		/* Buffer size. */
} OptCopy;

static OptCopy	copy_options;
static CmdCfg	*copy_cmc = NULL;

/* ----------------------------------------------------------------------------------------- */

/* 2001-04-29 -	Copy access and modification dates from <from> to (freshly created) <to>.
**		If <from_stat> is NULL, a local stat() call is made.
*/
static void copy_dates(const gchar *from, const struct stat *from_stat, const gchar *to)
{
	struct stat	sbuf;
	struct utimbuf	tim;

	if(!copy_options.copy_dates)
		return;
	if(from_stat == NULL)
	{
		if(lstat(from, &sbuf) != 0)
			return;
		from_stat = &sbuf;
	}
	tim.actime  = from_stat->st_atime;
	tim.modtime = from_stat->st_mtime;
	utime(to, &tim);	/* Implicit error handling through errno assumed. */
}

/* 2002-03-09 -	Copy mode from <from> to <to>. */
static void copy_mode(const gchar *from, const struct stat *from_stat, const gchar *to)
{
	struct stat	sbuf;

	if(from_stat == NULL)
	{
		if(lstat(from, &sbuf) != 0)
			return;
		from_stat = &sbuf;
	}
	chmod(to, from_stat->st_mode);
}

static void copy_owner(const char *from, const struct stat *from_stat, const char *to)
{
	struct stat	sbuf;

	if(from_stat == NULL)
	{
		if(lstat(from, &sbuf) != 0)
			return;
		from_stat = &sbuf;
	}
	chown(to, from_stat->st_uid, from_stat->st_gid);
}

/* 2003-10-21 -	Copy attributes: dates, mode/protection flags and owner. Optionally hides error. */
static gboolean copy_attributes(const char *from, const struct stat *from_stat, const char *to)
{
	gint		old_errno = errno;
	gboolean	ret;

	if(from_stat == NULL)
	{
		static struct stat	sbuf;

		if(lstat(from, &sbuf) == 0)
			from_stat = &sbuf;
		else
			return FALSE;
	}

	errno = 0;
	copy_dates(from, from_stat, to);
	copy_mode(from,  from_stat, to);
	copy_owner(from, from_stat, to);
	ret = errno == 0;
	if(copy_options.ignore_attrib_err)
		errno = old_errno;
	return ret;
}

/* ----------------------------------------------------------------------------------------- */

/* 2003-11-24 -	Placeholder. Implement! */
gboolean copy_socket(MainInfo *min, const gchar *from, const gchar *full_to, const struct stat *sstat)
{
	return FALSE;
}

/* 2003-11-25 -	Copy a symbolic link. Does not alter the contents of the link, which might mean
**		that links break.
*/
gboolean copy_link(MainInfo *min, const gchar *from, const gchar *full_to, const struct stat *sstat)
{
	gchar	buf[PATH_MAX];
	gint	len;

	if((len = readlink(from, buf, sizeof buf - 1)) > 0)
	{
		buf[len] = '\0';
		symlink(buf, full_to);
		/* Don't try copying any attributes for symlink, it won't fly! */
	}
	if(errno)
		err_set(min, errno, CMD_ID, from);
	return (errno == 0) ? TRUE : FALSE;
}

/* 2003-11-24 -	Copy a FIFO. */
gboolean copy_fifo(MainInfo *min, const gchar *from, const gchar *full_to, const struct stat *sstat)
{
	pgs_progress_item_begin(min, full_to, 0);
	if(mkfifo(full_to, sstat->st_mode) == 0)
		copy_attributes(from, sstat, full_to);
	pgs_progress_item_end(min);
	return errno == 0;
}

/* 1998-05-31 -	Copy a device special file. Pretty simple, since there are no contents
**		to worry about. Note that non-root users rarely have enough power for this.
** 1998-09-12 -	Modified; the <full_to> parameter now gives the complete destination name.
*/
gboolean copy_device(MainInfo *min, const gchar *from, const gchar *full_to, const struct stat *sstat)
{
	struct stat	stat;

	if(sstat == NULL)
	{
		if(lstat(from, &stat) == 0)
			sstat = &stat;
		else
		{
			err_set(min, errno, CMD_ID, from);
			return FALSE;
		}
	}
	pgs_progress_item_begin(min, full_to, stat.st_size);
	if(mknod(full_to, sstat->st_mode, sstat->st_rdev) == 0)
		copy_attributes(from, sstat, full_to);
	pgs_progress_item_update(min, sstat->st_size);
	pgs_progress_item_end(min);
	if(errno)
		err_set(min, errno, CMD_ID, from);
	return (errno == 0) ? TRUE : FALSE;
}

/* 2003-11-24 -	Copy regular file. */
gboolean copy_regular(MainInfo *min, const gchar *from, const gchar *full_to, const struct stat *sstat)
{
	gint	fd_in, fd_out;
	off_t	put = 0;

	pgs_progress_item_begin(min, full_to, sstat->st_size);
	if((fd_in = open(from, O_RDONLY)) >= 0)
	{
		if((fd_out = open(full_to, O_CREAT | O_WRONLY | O_TRUNC, sstat->st_mode | S_IWUSR)) >= 0)
		{
			put = fut_copy(fd_in, fd_out, copy_options.buf_size);
			close(fd_out);
		}
		close(fd_in);
		if((errno == 0) && (put == sstat->st_size))
			copy_attributes(from, sstat, full_to);
	}
	pgs_progress_item_end(min);
	if(errno)
	{
		if((put == sstat->st_size) && (put > 0) && (copy_options.leave_fullsize))	/* Leave fullsized copy? */
		{
			errno = 0;
			return TRUE;
		}
		err_set(min, errno, CMD_ID, full_to);
		if(fut_exists(full_to))				/* Preserves errno. */
			unlink(full_to);			/* Attempt to remove failed destination file. */
	}
	return (sstat->st_size == 0 || put == sstat->st_size) && (errno == 0);
}

/* 2003-11-24 -	Copy a "file". Since we speak Unix here, a file can actually be quite a lot,
**		the core defining criteria is mainly that a file is not a directory. It can be
**		a regular file, a device special file, a symlink, a fifo, or maybe even a socket.
*/
gboolean copy_file(MainInfo *min, const gchar *from, const gchar *full_to, const struct stat *sstat)
{
	struct stat	stat;
	OvwRes		owres;

	/* We really need to stat() the source in order to copy it. */
	if(sstat == NULL)
	{
		if(lstat(from, &stat) == 0)
			sstat = &stat;
		else
		{
			err_set(min, errno, CMD_ID, from);
			return FALSE;
		}
	}
	/* Check for overwrite, and clear the way if necessary. */
	owres = ovw_overwrite_file(min, full_to, from);
	if(owres == OVW_SKIP)
		return TRUE;
	else if(owres == OVW_CANCEL)
		return FALSE;
	else if(owres == OVW_PROCEED_FILE)
		del_delete_file(min, full_to);
	else if(owres == OVW_PROCEED_DIR)
		del_delete_dir(min, full_to, FALSE);
	/* Look at the mode stat field, and call the correct copying function. */
	err_clear(min);
	if(S_ISREG(sstat->st_mode))
		return copy_regular(min, from, full_to, sstat);
	else if(S_ISBLK(sstat->st_mode) || S_ISCHR(sstat->st_mode))
		return copy_device(min, from, full_to, sstat);
	else if(S_ISLNK(sstat->st_mode))
		return copy_link(min, from, full_to, sstat);
	else if(S_ISFIFO(sstat->st_mode))
		return copy_fifo(min, from, full_to, sstat);
	else if(S_ISSOCK(sstat->st_mode))
		return copy_socket(min, from, full_to, sstat);
	return FALSE;
}

/* ----------------------------------------------------------------------------------------- */

/* 2003-10-10 -	Check if the directory <to> is a child directory of <from>. If it
**		is, copying <from> into <to> is a genuine Bad Idea.
*/
static gboolean is_parent_of(const char *from, const char *to)
{
	gchar	buf1[PATH_MAX], buf2[PATH_MAX];

	if(fut_path_canonicalize(from, buf1, sizeof buf1) &&
	   fut_path_canonicalize(to,   buf2, sizeof buf2))
	{
		gsize	len = strlen(buf1);
		
		if(strncmp(buf1, buf2, len) == 0)
			return buf2[len] == G_DIR_SEPARATOR;	/* If last char is /, buf1 is really a prefix. */
	}
	return FALSE;
}

/* 1998-05-21 -	Copy a directory recursively, by first creating the appropriate
**		destination directory, and then copying all the source's contents
**		there. When directories are found in the source, this function is
**		called again to copy them.
** 1998-06-07 -	Added more error checking (now aborts on first failure, returning 0).
**		Does not remove partially failed directories.
** 1998-09-12 -	Modified; the <dest_name> is now the COMPLETE destination; it is
**		created directly, without appending anything. Also changed the order of
**		the fut_cd() and mkdir() calls, for better handling of unreable dirs.
** 2002-07-13 -	Will now completely delete an existing directory before copying.
** 2003-10-06 -	No longer deletes existing directory, only non-dir on name collision.
** 2003-10-12 -	Added simplistic check against source-contains-dest copying.
*/
gboolean copy_dir(MainInfo *min, const gchar *from, const gchar *dest_name)
{
	gboolean	aborted = FALSE;
	gchar		old_dir[PATH_MAX];
	DIR		*dir;
	struct dirent	*de;
	struct stat	sstat;
	OvwRes		owres;

	if(is_parent_of(from, dest_name))
	{
		gchar	ebuf[2 * PATH_MAX + 128];

		g_snprintf(ebuf, sizeof ebuf, _("Can't copy directory \"%s\"\nto \"%s\":\nthe source contains the destination."), from, dest_name);
		dlg_dialog_async_new_error(ebuf);
		return FALSE;
	}

	if(lstat(from, &sstat) != 0)
	{
		err_set(min, errno, CMD_ID, from);
		return FALSE;
	}
	owres = ovw_overwrite_file(min, dest_name, from);
	switch(owres)
	{
	case OVW_PROCEED_FILE:
		if(!del_delete_file(min, dest_name))
		{
			err_set(min, errno, CMD_ID, dest_name);
			return FALSE;
		}
		break;
	case OVW_PROCEED:
	case OVW_PROCEED_DIR:
		break;
	case OVW_SKIP:
	case OVW_CANCEL:
		return FALSE;
	}

	err_clear(min);
	pgs_progress_item_begin(min, dest_name, sstat.st_size);
	if(fut_cd(from, old_dir, sizeof old_dir))
	{
		if(owres == OVW_PROCEED_DIR || mkdir(dest_name, sstat.st_mode | S_IWUSR) == 0)
		{
			if((dir = opendir(".")) != NULL)
			{
				while(!aborted && !errno && (de = readdir(dir)) != NULL)
				{
					struct stat	stat;

					if(!min->cfg.dir_filter(de->d_name))
						continue;
					if(lstat(de->d_name, &stat) == 0)
					{
						gchar	dest2[PATH_MAX];

						g_snprintf(dest2, sizeof dest2, "%s%c%s", dest_name, G_DIR_SEPARATOR, de->d_name);
						if(S_ISDIR(stat.st_mode))
						{
							if(!copy_dir(min, de->d_name, dest2))
								aborted = TRUE;
						}
						else
						{
							if(!copy_file(min, de->d_name, dest2, &stat))
								aborted = TRUE;
						}
					}
				}
				closedir(dir);
			}
		}
		fut_cd(old_dir, NULL, 0);
		copy_attributes(from, &sstat, dest_name);
	}
	pgs_progress_item_end(min);
	if(errno)
		err_set(min, errno, CMD_ID, from);
	return (errno == 0 && !aborted) ? TRUE : FALSE;
}

/* ----------------------------------------------------------------------------------------- */

/* 2003-11-24 -	Copy selected files and/or directories to destination pane's directory. */
gint cmd_copy(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	gchar	old_path[PATH_MAX], dest[PATH_MAX];
	guint	num = 0;
	GSList	*slist, *iter;

	err_clear(min);

	if((src == NULL) || (dst == NULL))
		return 1;
	if(!fut_cd(src->dir.path, old_path, sizeof old_path))
		return 0;
	if((slist = dp_get_selection(src)) == NULL)
		return 1;

	ovw_overwrite_begin(min, _("\"%s\" Already Exists - Proceed With Copy?"), 0UL);
	pgs_progress_begin(min, _("Copying..."), PFLG_COUNT_RECURSIVE | PFLG_ITEM_VISIBLE | PFLG_BYTE_VISIBLE);
	for(iter = slist; !errno && (iter != NULL); iter = g_slist_next(iter))
	{
		g_snprintf(dest, sizeof dest, "%s/%s", dst->dir.path, DP_SEL_NAME(iter));
		if(S_ISDIR(DP_SEL_LSTAT(iter).st_mode))
		{
			if(!copy_dir(min, DP_SEL_NAME(iter), dest))
				break;
		}
		else
		{
			if(!copy_file(min, DP_SEL_NAME(iter), dest, &DP_SEL_LSTAT(iter)))
				break;
		}
		if(errno == 0)
			dp_unselect(src, DP_SEL_INDEX(src, iter));
		num++;
	}
	if(num)
		dp_rescan_post_cmd(dst);
	pgs_progress_end(min);
	ovw_overwrite_end(min);
	fut_cd(old_path, NULL, 0);

	if(errno && iter != NULL)
		err_set(min, errno, "copy", dp_full_name(src, DP_SEL_INDEX(src, iter)));
	dp_free_selection(slist);
	err_show(min);

	return errno == 0;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-04-06 -	Configuration initialization. */
void cfg_copy(MainInfo *min)
{
	if(copy_cmc == NULL)
	{
		/* Set the default values for module's options. */
		copy_options.modified	= FALSE;
		copy_options.copy_dates	= TRUE;
		copy_options.leave_fullsize = TRUE;
		copy_options.buf_size	= (1 << 18);

		copy_cmc = cmc_config_new("Copy", &copy_options);
		cmc_field_add_boolean(copy_cmc, "modified", NULL, offsetof(OptCopy, modified));
		cmc_field_add_boolean(copy_cmc, "copy_dates", _("Preserve Dates During Copy?"), offsetof(OptCopy, copy_dates));
		cmc_field_add_boolean(copy_cmc, "ignore_attrib_err", _("Ignore Failure to Copy Attributes (Date, Owner, Mode)?"), offsetof(OptCopy, ignore_attrib_err));
		cmc_field_add_boolean(copy_cmc, "leave_fullsize", _("Leave Failed Destination if Full Size?"), offsetof(OptCopy, leave_fullsize));
		cmc_field_add_size(copy_cmc, "buf_size", _("Buffer Size"), offsetof(OptCopy, buf_size), 1024, (1<<20), 1024);
		cmc_config_register(copy_cmc);
	}
}
