/////////////////////////////////////////////////////////////////////////////
//
// gk_wldap_link.h
//
// Copyright (C) 2003 Franz J Ehrengruber <franz@iptelenet.com>
//  
// PURPOSE OF THIS FILE: 
//   Provides the LDAP search functions based on the RFC 1823 MS LDAP-API
//   Windows LDAP integration (wldap.h/wldap32.lib)
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  
//
// History:
//      2003/10/01      initial version (Franz J Ehrengruber)
//                      based on ldaplink.h (Martin Froehlich)
//
/////////////////////////////////////////////////////////////////////////////

#if !defined(GK_WLDAP_LINK_H)	// make idempotent
#define GK_WLDAP_LINK_H "#(@) $Id: gk_wldap_link.h,v 1.1.2.4 2004/05/12 17:46:40 zvision Exp $"

#if defined(HAS_WLDAP) 

#if (_MSC_VER >= 1200)
#if defined _DEBUG
#pragma warning( disable : 4786 ) // warning about the identifier was truncated to 'number' characters in the debug information
#endif                            // The identifier string exceeded the maximum allowable length and was truncated.
#endif

#include <time.h>
#include <map>			// STL map
#include <vector>       // STL vector
#include <ptlib.h>		// the PWlib !! do NOT move
#include <winldap.h>	// Windows ldap include

#include "gk_wldap_interface.h"

/** Class that holds the names of the attributes, used for LDAP support. 
	The GK_WLDAP modules currently support the following LDAP attributes:

    Gk Config Name:  Ldap Display Name:  LDAP Attribute Name:  OIDs:
    -------------------------------------------------------------------------------------
	CommonName       Common-Name         cn                    2.5.4.3 (RFC)
	AccountStatus    H323-accountStatus  h323accountStatus     1.3.6.1.4.1.16617.2.5.1000
	H323ID           H323-id             H323id                1.3.6.1.4.1.16617.2.5.1001
	TelephonNo       H323-telephoneNo    H323telephonNo        1.3.6.1.4.1.16617.2.5.1002
	H235PassWord     H323-plainTextPw    H323plainTextPw       1.3.6.1.4.1.16617.2.5.1003
**/

typedef std::map<PString, PString> LDAPAttributeNamesClass;
typedef LDAPAttributeNamesClass::value_type LDAPANValuePair;

// LDAP config tags and names - tags named after config file tags, 
// used as indices to LDAPAttrTags
namespace lctn  {enum LDAPAttributeNamesEnum {DN=0, 
											 CommonName, 
											 AccountStatus, 
											 H323ID, 
											 TelephonNo, 
											 H235PassWord, 
											 MAX_ATTR_NO};

                 // list of names (keys) as used in config file
                 extern const char * LDAPAttrTags[MAX_ATTR_NO];
				}// namespace

typedef std::map<PString, PStringList> LDAPAttributeValueClass;
typedef LDAPAttributeValueClass::value_type LDAPAVValuePair;

/** Class that contains search queries
 */
class LDAPQuery {
public:
	enum LDAPQueryOp {
		LDAPand, LDAPor, LDAPnot, LDAPNONE };
	unsigned LDAPOperator;
	LDAPAttributeValueClass LDAPAttributeValues;
	LDAPQuery(): LDAPOperator(LDAPor) {}; // Default is "or"
};

typedef std::map<PString, LDAPAttributeValueClass> LDAPEntryClass;
typedef LDAPEntryClass::value_type LDAPECValuePair;

/** Class that contains search answers
 */
class LDAPAnswer {
public:
	LDAPAnswer();
	virtual ~LDAPAnswer();
	LDAPEntryClass LDAPec;	     // the attributes and their values
	LDAPMessage * result;
};

class LDAPCtrl {
public:
	LDAPCtrl(LDAPAttributeNamesClass *, // the Attribute names
		     LDAP_TIMEVAL timeout,
		     PString &,		// Name of the LDAP Server
		     PString &,		// Distinguished Name (DN) from where to search
		     PString &,		// UserDN of acting user
		     PString &,		// Pasword for simple auth. of BindUserDN
		     unsigned int,	// timeout in seconds (default 10)
		     int			// Port of the LDAP Server (default IANA port)
		    ); 
	virtual void Close(void);
	virtual ~LDAPCtrl();

	virtual int LdapUserLookup(const PString      & common_name, // [ in ] 
							   const PStringArray & attr_names,  // [ in ]
							   LDAPAnswer         * answer);     // [ out ]
	
	virtual int LdapDirectorySearch(const PString & filter, const char * attrs[], LDAPAnswer * answer);

	virtual int LdapCollectAttributes(LDAPAnswer * answer);

protected:
	// Some of this data might look superflous, but experience teaches to
	// keep the connection details. At least they come handy during a
	// debugging session
	LDAPAttributeNamesClass * AttributeNames; // names of the LDAP attributes
	LDAP_TIMEVAL timeout;
	PString ServerName;		// Name of the LDAP Server
	PString SearchBaseDN;	// Distinguished Name (DN) from where to search
	PString BindUserDN;		// UserDN of acting user
	PString BindUserPW;		// Pasword for simple auth. of BindUserDN
	unsigned int timelimit;	// timeout for operations in seconds
	int ServerPort;		    // Port of the LDAP Server

private:
	GK_LDAP * ldap;		    // The ldap connection
	
	bool known_to_be_bound;	// _known_ status of binding
	int Initialize(void);	// initially, called from constructors
	int Connect(void);      // connectint, called before binding
	int Bind(bool);		    // binding, may be enforced by passing true
	int Unbind(bool);		// unbinding, may be enforced by passing true

	mutable PMutex m_bindLock;
};

#endif // HAS_WLDAP

#endif // defined(WLDAPLINK_H)

//
// End of gk_wldap_link.h
//
