/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/locks.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/jfs/jfs_types.h>
#include <linux/jfs/jfs_filsys.h>
#include <linux/jfs/jfs_dtree.h>
#include <linux/jfs/jfs_unicode.h>
#include <linux/jfs/jfs_debug.h>

extern int jfs_create(struct inode *, struct dentry *, int);
extern int jfs_mkdir(struct inode *, struct dentry *, int);
extern int jfs_unlink(struct inode *, struct dentry *);
extern int jfs_rmdir(struct inode *, struct dentry *);
extern int jfs_link(struct dentry *,struct inode *, struct dentry *);
extern int jfs_symlink(struct inode *, struct dentry *, const char * name);
extern int jfs_rename(struct inode *, struct dentry *, struct inode *,
		       struct dentry *);
extern int jfs_mknod(struct inode *, struct dentry *, int, int);
#ifndef kern22
extern int jfs_fsync_file(struct file *, struct dentry *, int);
#endif

static ssize_t jfs_dir_read (
struct file	*filp,
char		*buf,
size_t		count,
loff_t		*ppos)
{
	return -EISDIR;
}

#ifdef kern22
static struct file_operations jfs_dir_operations = {
	fsync:		file_fsync,
#else
struct file_operations jfs_dir_operations = {
	fsync:		jfs_fsync_file,
#endif
	read:		jfs_dir_read,
	readdir:	jfs_readdir,
};

static struct dentry *jfs_lookup(
struct inode	*dip,
struct dentry	*dentry)
{
	btstack_t	btstack;
	ino_t		inum;
	struct inode	*ip;
	component_t	key;
	const char	*name = dentry->d_name.name;
	int32		len = dentry->d_name.len;
	int32		rc;

	jFYI(1,("jfs_lookup: name = %s\n", name));


	if ((name[0] == '.') && (len == 1))
		inum = dip->i_ino;
	else if (strcmp(name, "..") == 0)
		inum = ((dtroot_t *)dip->i_jfs_inode_ext)->header.idotdot;
	else
	{
		if ((rc = get_UCSname(&key, dentry, dip->i_sb->s_jfs_nls_tab)))
			return ERR_PTR(-rc);
		IREAD_LOCK(dip);
		rc = dtSearch(dip, &key, &inum, &btstack, JFS_LOOKUP);
		IREAD_UNLOCK(dip);
		free_UCSname(&key);
		if (rc == ENOENT)
		{
			d_add(dentry, NULL);
			return ERR_PTR(0);
		}
		else if (rc)
		{
			jERROR(1,("jfs_lookup: dtSearch returned %d\n", rc));
			return ERR_PTR(-rc);
		}
	}

	ip = iget(dip->i_sb, inum);
	if (ip == NULL)
	{
		jERROR(1,("jfs_lookup: iget failed on inum %d\n", (uint)inum));
		return ERR_PTR(-EACCES);
	}

	d_add(dentry, ip);

	return ERR_PTR(0);
}

struct inode_operations jfs_dir_inode_operations = {
#ifdef kern22
        default_file_ops:	&jfs_dir_operations,
#endif
	create:			jfs_create,
	lookup:			jfs_lookup,
	link:			jfs_link,
	unlink:			jfs_unlink,
	symlink:		jfs_symlink,
	mkdir:			jfs_mkdir,
	rmdir:			jfs_rmdir,
	mknod:			jfs_mknod,
	rename:			jfs_rename,
};
