/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/locks.h>
#include <linux/errno.h>
#include <linux/jfs/jfs_types.h>
#include <linux/jfs/jfs_filsys.h>
#include <linux/jfs/jfs_imap.h>
#include <linux/jfs/jfs_xtree.h>
#include <linux/jfs/jfs_extent.h>
#include <linux/jfs/jfs_unicode.h>
#include <linux/jfs/jfs_txnmgr.h>
#include <linux/jfs/jfs_debug.h>

extern struct inode_operations jfs_dir_inode_operations;
extern struct inode_operations jfs_file_inode_operations;
extern struct inode_operations jfs_symlink_inode_operations;
extern struct file_operations jfs_dir_operations;
extern struct file_operations jfs_file_operations;
#ifndef kern22
struct address_space_operations jfs_aops;
#endif
extern int jfs_ftruncate(struct inode *, loff_t);
extern int freeZeroLink(struct inode *);

void jfs_put_inode(struct inode *inode)
{
	jFYI(1,("In jfs_put_inode, inode = 0x%p\n", inode));
}

void jfs_read_inode(struct inode * inode)
{
	int32 rc;

	jFYI(1,("In jfs_read_inode, inode = 0x%p\n", inode));

	if ((rc = diRead(inode)))
	{
		jERROR(1,("jfs_read_inode: diRead failure! rc = %d\n", rc));
		make_bad_inode(inode);
	}
	else
	{
		if (S_ISREG(inode->i_mode)) {
			inode->i_op = &jfs_file_inode_operations;
#ifndef kern22
			inode->i_fop = &jfs_file_operations;
			inode->i_mapping->a_ops = &jfs_aops;
#endif
		}
		else if (S_ISDIR(inode->i_mode)) {
			inode->i_op = &jfs_dir_inode_operations;
#ifndef kern22
			inode->i_fop = &jfs_dir_operations;
#endif
		}
		else if (S_ISLNK(inode->i_mode)) {
			inode->i_op = &jfs_symlink_inode_operations;
#ifndef kern22
			inode->i_mapping->a_ops = &jfs_aops;
#endif
		}
#ifdef kern22
		else if (S_ISCHR(inode->i_mode))
			inode->i_op = &chrdev_inode_operations;
		else if (S_ISBLK(inode->i_mode))
			inode->i_op = &blkdev_inode_operations;
		else if (S_ISFIFO(inode->i_mode))
			init_fifo(inode);
		else
			jERROR(1,("jfs_read_inode: unknown file type 0%o\n",
				  inode->i_mode));
#else
		else
			init_special_inode(inode, inode->i_mode,
					   kdev_t_to_nr(inode->i_rdev));
#endif
	}
}

/*
 * Workhorse of both fsync & write_inode
 */
int jfs_commit_inode(struct inode *inode, int wait)
{
	int	rc = 0;
	int	tid;

	jFYI(1,("In jfs_commit_inode, inode = 0x%p\n", inode));

	txBegin(inode->i_sb, &tid, 1);
	rc = txCommit(tid, 1, &inode, wait?COMMIT_SYNC:0);
	txEnd(tid);
	return rc;
}

#ifdef kern22
void jfs_write_inode(struct inode *inode)
#else
void jfs_write_inode(struct inode *inode, int wait)
#endif
{
	/*
	 * If COMMIT_COMMITTING set, we are in the process of committing
	 * this inode already.
	 */
	if (inode->i_jfs_cflag & COMMIT_COMMITTING)
		return;

	inode->i_jfs_cflag |= COMMIT_COMMITTING;

	IWRITE_LOCK(inode);

#ifdef kern22
	if (jfs_commit_inode(inode, 0))
#else
	if (jfs_commit_inode(inode, wait))
#endif
	{
		jERROR(1,("jfs_write_inode: jfs_commit_inode failed!\n"));
	}
	inode->i_jfs_cflag &= ~COMMIT_COMMITTING;
	IWRITE_UNLOCK(inode);
}

#ifndef kern22
int jfs_fsync_inode(struct inode *inode, int datasync)
{
	int rc = 0;

	rc = fsync_inode_buffers(inode);
	if (datasync || !(inode->i_state & I_DIRTY))
		return rc;

	IWRITE_LOCK(inode);
	rc = jfs_commit_inode(inode, 1);
	IWRITE_UNLOCK(inode);
	return rc;
}
#endif

void jfs_delete_inode(struct inode *inode)
{
	jFYI(1,("In jfs_delete_inode, inode = 0x%p\n", inode));

	IWRITE_LOCK(inode);
	if (inode->i_jfs_cflag & COMMIT_FREEWMAP)
		(void) freeZeroLink(inode);

	diFree(inode);
	IWRITE_UNLOCK(inode);

#ifndef kern22
	clear_inode(inode);
#endif
}

#ifndef kern22

static int jfs_get_block(
struct inode		*ip,
long			lblock,
struct buffer_head	*bh_result,
int			create)
{
	int	rc = 0;
	xad_t	xad;
	int64	xaddr;
	uint8	xflag;
	int32	xlen;

	/*
	 * Take appropriate lock on inode
	 */
	if (ip->i_jfs_fileset != AGGREGATE_I) {
		if (create)
			IWRITE_LOCK(ip);
		else
			IREAD_LOCK(ip);
	}

	if (((lblock << ip->i_sb->s_blocksize_bits) < ip->i_size) &&
	    (xtLookup(ip, lblock, 1, &xflag, &xaddr, &xlen, 0) == 0) && xlen) {
		if (xflag & XAD_NOTRECORDED) {
			if (! create)
				/*
				 * Allocated but not recorded, read treats
				 * this as a hole
				 */
				goto unlock;
#ifdef _JFS_4K
			XADoffset(&xad, lblock);
			XADlength(&xad, xlen);
			XADaddress(&xad, xaddr);
#else /* _JFS_4K */
			/*
			 * As long as block size = 4K, this isn't a problem.
			 * We should mark the whole page not ABNR, but how
			 * will we know to mark the other blocks BH_New?
			 */
			BUG();
#endif /* _JFS_4K */
			rc = extRecord(ip, &xad);
			if (rc)
				goto unlock;
			bh_result->b_state |= (1UL << BH_New);
		}

		bh_result->b_dev = ip->i_dev;
		bh_result->b_blocknr = xaddr;
		bh_result->b_state |= (1UL << BH_Mapped);
		goto unlock;
	}
	if (! create)
		goto unlock;

	/*
	 * Allocate a new block
	 */
#ifdef _JFS_4K
	if ((rc = extHint(ip, lblock << ip->i_sb->s_blocksize_bits, &xad)))
		goto unlock;
	rc = extAlloc(ip, 1, lblock, &xad, FALSE);
	if (rc)
		goto unlock;

	bh_result->b_dev = ip->i_dev;
	bh_result->b_blocknr = addressXAD(&xad);
	bh_result->b_state |= ((1UL << BH_Mapped) | (1UL << BH_New));

#else /* _JFS_4K */
	/*
	 * We need to do whatever it takes to keep all but the last buffers
	 * in 4K pages - see jfs_write.c
	 */
	BUG();
#endif /* _JFS_4K */

unlock:
	/*
	 * Release lock on inode
	 */
	if (ip->i_jfs_fileset != AGGREGATE_I) {
		if (create)
			IWRITE_UNLOCK(ip);
		else
			IREAD_UNLOCK(ip);
	}
	return -rc;
}

static int jfs_writepage(
struct page	*page)
{
	return block_write_full_page(page, jfs_get_block);
}

static int jfs_readpage(
struct file	*file,
struct page	*page)
{
	return block_read_full_page(page, jfs_get_block);
}

static int jfs_prepare_write(
struct file	*file,
struct page	*page,
unsigned	from,
unsigned	to)
{
	return block_prepare_write(page, from, to, jfs_get_block);
}

static int jfs_bmap(
struct address_space	*mapping,
long			block)
{
	return generic_block_bmap(mapping, block, jfs_get_block);
}

struct address_space_operations jfs_aops = {
	readpage:	jfs_readpage,
	writepage:	jfs_writepage,
	sync_page:	block_sync_page,
	prepare_write:	jfs_prepare_write,
	commit_write:	generic_commit_write,
	bmap:		jfs_bmap,
};

#endif /* kern22 */
