/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/malloc.h>
#include <linux/version.h>
#include <linux/jfs/jfs_types.h>
#include <linux/jfs/jfs_filsys.h>
#include <linux/jfs/jfs_imap.h>
#include <linux/jfs/jfs_dinode.h>
#include <linux/jfs/jfs_debug.h>

/*
 * NAME:	ialloc()
 *
 * FUNCTION:	Allocate a new inode
 *
 */
struct inode *ialloc(
struct inode	*parent,
umode_t		mode)
{
	struct inode		*new_inode;
	int32			rc;
	struct super_block	*sb = parent->i_sb;

	new_inode = get_empty_inode();
	if (! new_inode)
	{
		jERROR(1,("ialloc: get_empty_inode returned NULL!\n"));
		return new_inode;
	}

	rc = diAlloc(parent, S_ISDIR(mode), new_inode);
	if (rc)
	{
		jERROR(1,("ialloc: diAlloc returned %d!\n", rc));	
		new_inode->i_nlink = 0;
		iput(new_inode);
		return NULL;
	}

	if ((! S_ISBLK(mode)) && (! S_ISCHR(mode)) && (! S_ISFIFO(mode))) {
		new_inode->i_jfs_inode_ext = kmalloc(288, GFP_KERNEL);
		if (new_inode->i_jfs_inode_ext == NULL)
		{
			jERROR(1,("ialloc: kmalloc failed!\n"));
			(void) diFree(new_inode);
			new_inode->i_nlink = 0;
			iput(new_inode);
			return NULL;
		}
		memset(new_inode->i_jfs_inode_ext, 0, 288);
	}

	new_inode->i_sb = parent->i_sb;
	new_inode->i_dev = parent->i_dev;

	new_inode->i_uid = current->fsuid;
	if (parent->i_mode & S_ISGID)
	{
		new_inode->i_gid = parent->i_gid;
		if (S_ISDIR(mode))
			mode |= S_ISGID;
	}
	else
		new_inode->i_gid = current->fsgid;

	new_inode->i_mode = mode;
	if (S_ISDIR(mode))
		new_inode->i_jfs_mode2 = IDIRECTORY|mode;
	else
		new_inode->i_jfs_mode2 = INLINEEA|ISPARSE|mode;
	new_inode->i_blksize = sb->s_blocksize;
	new_inode->i_blocks = 0;
	new_inode->i_mtime = new_inode->i_atime = new_inode->i_ctime =
			     CURRENT_TIME;
	/*
	 * i_jfs_otime is an OS/2 thing.  In 2.2 kernels, we can't afford
	 * the space in a FIFO inode, because the same space is used used
	 * by the pipe code.  OS/2 won't recognize a pipe anyway.
	 */
	if (! S_ISFIFO(mode))
			     new_inode->i_jfs_otime = new_inode->i_ctime;
	new_inode->i_version = ++event;
	new_inode->i_generation = sb->s_jfs_gengen++;

	new_inode->i_jfs_cflag = COMMIT_NEW;

	RDWRLOCK_INIT(&new_inode->i_jfs_rdwrlock);

	insert_inode_hash(new_inode);

	jFYI(1,("ialloc returns new_inode = 0x%p\n", new_inode));

	return new_inode;
}

/*
 * NAME:	iwritelocklist()
 *
 * FUNCTION:	Lock multiple inodes in sorted order to avoid deadlock
 *
 */
void iwritelocklist(int n, ...)
{
	va_list ilist;
	struct inode *sort[4];
	struct inode *ip;
	int k, m;

	va_start(ilist, n);
	for (k = 0; k < n; k++)
		sort[k] = va_arg(ilist, struct inode *);
	va_end(ilist);

	/* Bubble sort in descending order */
	do {
		m = 0;
		for (k = 0; k < n; k++)
			if ((k+1) < n && sort[k+1]->i_ino > sort[k]->i_ino) {
				ip = sort[k];
				sort[k] = sort[k+1];
				sort[k+1] = ip;
				m++;
			}
	} while (m);

	/* Lock them */
	for (k = 0; k < n; k++) {
		IWRITE_LOCK(sort[k]);
	}
}
