/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * Change History :
 */

/*
 *	jfs_umount.c
 *
 * note: file system in transition to aggregate/fileset:
 * (ref. jfs_mount.c)
 *
 * file system unmount is interpreted as mount of the single/only 
 * fileset in the aggregate and, if unmount of the last fileset, 
 * as unmount of the aggerate;
 */

#include <linux/fs.h>
#include <linux/jfs/jfs_types.h>
#include <linux/jfs/jfs_filsys.h>
#include <linux/jfs/jfs_superblock.h>
#include <linux/jfs/jfs_dmap.h>
#include <linux/jfs/jfs_imap.h>
#include <linux/jfs/jfs_metapage.h>
#include <linux/jfs/jfs_logmgr.h>
#include <linux/jfs/jfs_debug.h>

#ifdef _STILL_TO_PORT
#include "jfs_types.h"
#include "jfs_filsys.h"
#include "jfs_lock.h"
#include "jfs_inode.h"
#include "jfs_bufmgr.h"
#include "jfs_superblock.h"
#include "jfs_imap.h"
#include "jfs_dmap.h"
#include "jfs_dnlc.h"
#include "jfs_proto.h"
#include "jfs_dasdlim.h"					// D233382
#include "jfs_debug.h"
#endif /* _STILL_TO_PORT */

/* external references not covered in header files */
int32 readSuper(struct super_block *, metapage_t **);
//void dasd_umount(struct vfs *);

/* forward references */
static int32 updateSuper(struct super_block *);

/*
 * NAME:	jfs_umount(vfsp, flags, crp)
 *
 * FUNCTION:	vfs_umount()
 *
 * PARAMETERS:	vfsp	- virtual file system pointer
 *		flags	- unmount for shutdown
 *		crp	- credential
 *
 * RETURN :	EBUSY	- device has open files
 */
int32 jfs_umount(
	struct super_block	*sb)
{
	int32	rc = 0;
	log_t *log;
#ifdef _STILL_TO_PORT
        dev_t	fsdev;
        ino_t	fileset;
	int32	forced;
        struct inode *rootip;
	struct vnode *rootvp;
	struct inode	*ipimap, *ipaimap, *ipaimap2, *ipbmap, *ipmnt, *iplog;
#endif /* _STILL_TO_PORT */
	struct inode *ipbmap = sb->s_jfs_ipbmap;
	struct inode *ipimap = sb->s_jfs_ipimap;
	struct inode *ipaimap = sb->s_jfs_ipaimap;
	struct inode *ipaimap2 = sb->s_jfs_ipaimap2;

jFYI(1,("\n	UnMount JFS: sb:0x%p\n", sb));

#ifdef _STILL_TO_PORT
	/*
	 * purge volume information from DASD limit structures
	 */
	dasd_umount(vfsp);					// F226941

	/*
	 * serialize mount/unmount
	 */
	JFS_LOCK();

	/*
	 *	unmount fileset:
	 */
umntFileset:
	/* retrieve the mounted fs device */
        fsdev = ipmnt->i_dev;
	fileset = ipimap->i_number;

	/*
	 * Purge directory name lookup cache for the fileset. 
	 */
	ncUnmount(ipimap);

	/*
	 * close inodes of the fileset in inode cache
	 *
	 * There should be no active files in the fileset
	 * being unmounted (unless this is a shutdown
	 * unmount, in which case we force the vfs to be
	 * unmounted anyway).
	 */
jEVENT(0,("jfs_umount: close fileset inodes\n"));
	forced = (vfsp->vfs_flag & VFS_SHUTDOWN);
	if ((rc = iUnmount(vfsp, forced)) && !forced)
		goto out;

	/*
	 *	point of no return for unmount: 
	 *
	 * ignore failure of further intermediate steps and
	 * return success to notify LFS to cleanup.
	 */
	rc = 0;

	/*
	 * close root vode/inode (skipped in iunmount() of the fileset)
	 */
	rootvp = vfsp->vfs_mntd;
	vfsp->vfs_mntd = NULL;

	rootip = VP2IP(rootvp);
//	assert(rootip->i_count == 1);
	ICACHE_LOCK();
	iuncache(rootip, 1);
	ICACHE_UNLOCK();

#endif /* _STILL_TO_PORT */
	/*
	 * close fileset inode allocation map (aka fileset inode)
	 */
jEVENT(0,("jfs_umount: close ipimap:0x%p\n", ipimap));
	diUnmount(ipimap, 0);

#ifdef _STILL_TO_PORT
	bmInodeFlush(ipimap);
#endif /* _STILL_TO_PORT */

	diFreeSpecial(ipimap);
	sb->s_jfs_ipimap = NULL;

#ifdef _STILL_TO_PORT
	/* more fileset in aggregate ? */
	if (--ipmnt->i_count)
		goto out;
	
	/*
	 *	unmount aggregate (file system)
	 */
umntAggregate:
	MUTEXLOCK_FREE(&ipmnt->i_renamelock);
#endif /* _STILL_TO_PORT */

	/*
	 * close secondary aggregate inode allocation map
	 */
	ipaimap2 = sb->s_jfs_ipaimap2;
	if (ipaimap2)
	{
jEVENT(0,("jfs_umount: close ipaimap2:0x%p\n", ipaimap2));
		diUnmount(ipaimap2, 0);
#ifdef _STILL_TO_PORT
		bmInodeFlush(ipaimap2);
#endif /* _STILL_TO_PORT */
		diFreeSpecial(ipaimap2);
		sb->s_jfs_ipaimap2 = NULL;
	}

	/*
	 * close aggregate inode allocation map
	 */
	ipaimap = sb->s_jfs_ipaimap;
jEVENT(0,("jfs_umount: close ipaimap:0x%p\n", ipaimap));
	diUnmount(ipaimap, 0);
#ifdef _STILL_TO_PORT
	bmInodeFlush(ipaimap);
#endif /* _STILL_TO_PORT */
	diFreeSpecial(ipaimap);
	sb->s_jfs_ipaimap = NULL;

	/*
	 * close aggregate block allocation map
	 */
jEVENT(0,("jfs_umount: close ipbmap:%p\n", ipbmap));
	dbUnmount(ipbmap, 0);

#ifdef _STILL_TO_PORT
	bmInodeFlush(ipbmap);
#endif /*  _STILL_TO_PORT */

	diFreeSpecial(ipbmap);
	sb->s_jfs_ipimap = NULL;

#ifdef _STILL_TO_PORT
	/*
	 * free cache device for aggregate i/o
	 */
	cmUnmount(ipmnt->i_cachedev);
#endif /* _STILL_TO_PORT */

	/*
	 * ensure all file system file pages are propagated to their
	 * home blocks on disk (and their in-memory buffer pages are 
	 * invalidated) BEFORE updating file system superblock state
	 * (to signify file system is unmounted cleanly, and thus in 
	 * consistent state) and log superblock active file system 
	 * list (to signify skip logredo()).
	 */

	/*
	 *	update superblock and close log 
	 *
	 * if mounted read-write and log based recovery was enabled
	 */
	if (sb->s_jfs_log)
	{
		/*
		 * update superblock: synchronously write and 
		 * invalidate its in-memory buffer page
		 */
		rc = updateSuper(sb);

		/*
	 	 * close log: 
		 *
		 * remove file system from log active file system list.
	 	 */
		log = sb->s_jfs_log;
		rc = lmLogClose(sb, log);
	}

#ifdef _STILL_TO_PORT
	/*
	 *	close file system device
	 */
	ipmnt->i_devfp = NULL;

	/*
	 * close "mount" inode.
	 */
	ICACHE_LOCK();
	iunhash(ipmnt);
	ICACHE_UNLOCK();

out:
	JFS_UNLOCK();

	/* debug sanity check */
	/* iSanity(); */
	/* bmSanity(); */

#endif /* _STILL_TO_PORT */
jFYI(0,("	UnMount JFS Complete: %d\n", rc));
	return rc;
}


/*
 *	updateSuper(ipmnt)
 *
 * update superblock if it is mounted read-write
 */
static int32
updateSuper(
	struct super_block *sb)
{
	int32			rc = 0;
	metapage_t		*mp;
	struct jfs_superblock	*j_sb;

	if ((rc = readSuper(sb, &mp)))
		return rc;
	j_sb = (struct jfs_superblock *)(mp->data);

 	/*
	 * file system state transition: 
	 * if mounted-clean, mark it as unmounted-clean;
 	 * otherwise state remains as dirty.
	 */
#ifndef _JFS_NOJOURNAL
	if (j_sb->s_state == FM_MOUNT)
#endif
		j_sb->s_state = FM_CLEAN;

#ifdef _JFS_FASTDASD
	/*
	 * Synchronize DASD_PRIME flag in superblock with the one stored in
	 * the mount inode
	 */
	j_sb->s_flag = (j_sb->s_flag & ~JFS_DASD_PRIME) |
		     (sb->s_jfs_mntflag & JFS_DASD_PRIME);
#endif /* _JFS_FASTDASD */

	/* write out superblock synchronously and invalidate page */
	write_metapage(mp);

	return rc;
}
