
/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 
 *  JFS symlink handling code
 */


#include <linux/fs.h>
#include <asm/uaccess.h>
#include <linux/malloc.h>
#include <linux/jfs/jfs_types.h>
#include <linux/jfs/jfs_filsys.h>
#include <linux/jfs/jfs_lock.h>
#include <linux/jfs/jfs_inode.h>
#include <linux/jfs/jfs_dinode.h>
#include <linux/jfs/jfs_dtree.h>
#include <linux/jfs/jfs_imap.h>
#include <linux/jfs/jfs_xtree.h>
#include <linux/jfs/jfs_unicode.h>
#include <linux/jfs/jfs_debug.h>

static int jfs_readlink(struct dentry *, char *buffer, int buflen);
#ifdef kern22
static struct dentry *jfs_follow_link(struct dentry *dentry,
				      struct dentry *base,
				      unsigned int follow);
#else
static int jfs_follow_link(struct dentry *dentry,
				      struct nameidata *nd);
#endif

/*
 * symlinks can't do much...
 */
struct inode_operations jfs_symlink_inode_operations = {
	readlink:	jfs_readlink,
	follow_link:	jfs_follow_link,
};

#ifdef kern22
static struct dentry *jfs_follow_link(struct dentry *dentry,
				      struct dentry *base,
				      unsigned int follow)
#else
static int jfs_follow_link(struct dentry *dentry, struct nameidata *nd)
#endif
{
#ifndef kern22
	int rc;
#endif
	struct inode *ip = dentry->d_inode;
	char *link = NULL;
	metapage_t *mp = NULL;

	jFYI(1, ("jfs_follow_link: ip:0x%p\n", ip));

	/*
	 * read the target path name
	 */
	if (ip->i_size <= IDATASIZE) {
		/*
		 * fast symbolic link
		 * read target path name inline from on-disk inode
		 */

		link =
		    &(((char *) (ip->i_jfs_inode_ext))
		      [sizeof(dasd_t) + sizeof(dxd_t)]);
	} else {
		jFYI(1, ("jfs_follow_link not fast symbolic link\n"));
		/* 
		 * read target path name from a single extent
		 *
		 * target path name <= PATH_MAX < buffer page size
		 *
		 * even though the data of symlink object (target 
		 * path name) is treated as non-journaled user data,
		 * it is read/written thru buffer cache for performance.
		 */
		mp = read_metapage(ip, 0, PSIZE, 0);
		if (mp == NULL) {
			goto out2;
		}
		link = (char *) (mp->data);
	}

#ifdef kern22
	UPDATE_ATIME(ip);

	base = lookup_dentry(link, base, follow);
#else
	rc = vfs_follow_link(nd, link);
#endif
	if (mp) {
		release_metapage(mp);
	}

	jFYI(1, ("jfs_follow_link\n"));

#ifdef kern22
	return base;
#else
	return rc;
#endif

      out2:
#ifdef kern22
	return ERR_PTR(-EIO);
#else
	return -EIO;
#endif
}

static int jfs_readlink(struct dentry *dentry, char *buffer, int buflen)
{
	int32 rc = 0;
	struct inode *ip = dentry->d_inode;
	char *link = NULL;
	metapage_t *mp = NULL;

	jFYI(1,
	     ("jfs_readlink: ip:0x%p buflen:%d i_size:%ld \n", ip,
	      buflen, (ulong)ip->i_size));

	/* validate the buffer size vs target path name size
	 * (AES requires ERANGE if the link name won't fit)
	 */
#ifdef _I_DONT_THINK_WE_NEED_TO_CHECK
	if (ip->i_size > buflen) {
		rc = ERANGE;
		goto out;
	}
#endif				/* _I_DONT_THINK_WE_NEED_TO_CHECK  */
	/*
	 * read the target path name
	 */

	if (ip->i_size <= IDATASIZE) {
		/*
		 * fast symbolic link
		 *
		 * read target path name inline from on-disk inode
		 */

		link =
		    &(((char *) (ip->i_jfs_inode_ext))
		      [sizeof(dasd_t) + sizeof(dxd_t)]);
		jFYI(1, ("jfs_readlink link name:%s \n", link));
	} else {

		jFYI(1, ("jfs_readlink not fast symbolic \n"));
		/* 
		 * read target path name from a single extent
		 *
		 * target path name <= PATH_MAX < buffer page size
		 *
		 * even though the data of symlink object (target 
		 * path name) is treated as non-journaled user data,
		 * it is read/written thru buffer cache for performance.
		 */
		mp = read_metapage(ip, 0, PSIZE, 0);
		if (mp == NULL) {
            rc = -EIO;
            goto out;
        }
			
		link = mp->data;
	}
#ifdef kern22
	rc = ip->i_size;
	jFYI(1, ("jfs_readlink: size%d\n", rc));
    if (rc > buflen) {
        rc = buflen;
    }
	if (copy_to_user(buffer, link, rc))
		rc = -EFAULT;

	UPDATE_ATIME(ip);
#else
	rc = vfs_readlink(dentry, buffer, buflen, link);
#endif
	if (mp) {
		release_metapage(mp);
	}

      out:
	jFYI(1, ("jfs_readlink: rc:%d\n", rc));
	return rc;
}
