/*
*Copyright (C) 2003 VIA Technologies, Inc. All Rights Reserved.
*This program is free software; you can redistribute it and/or modify it under 
*the terms of the GNU *General Public License as published by the Free Software 
*Foundation; either version 2 of the License, or (at your option) any later 
*version.
*
*This program is distributed in the hope that it will be useful, but WITHOUT ANY
*WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
*A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
*You should have received a copy of the GNU General Public License along with
*this program; if not, write to the Free Software Foundation, Inc.,
*59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/fb.h>
#include <linux/init.h>
#include <linux/pci.h>

#include <video/fbcon.h>
#include <video/fbcon-cfb8.h>
#include <video/fbcon-cfb16.h>
#include <video/fbcon-cfb24.h>
#include <video/fbcon-cfb32.h>


#include "viafbdev.h"
#include "share.h"
#include "chip.h"
#include "debug.h"


#define VERSION		"0.8"

extern unsigned int get_memsize(void);
extern int setmode(int vmode_index,int hor_res,int ver_res,int video_bpp);
extern void write_reg_mask(u8 index, int io_port, u8 data, u8 mask);
extern void init_chip_info(void);
extern inline u8 read_reg(int io_port, u8 index);
extern void init_dac(int set_iga); 
extern int get_pixclock(int hres, int vres, int vmode_refresh);
extern int get_refresh(int hres, int vres, int pixclock);
extern void update_device_setting(int hres, int vres, int bpp, int vmode_refresh);

static int get_mode_index(int hres, int vres);
struct chip_information  	    chip_info;
struct crt_setting_information  crt_setting_info;
struct tv_setting_information	tv_setting_info;
struct tmds_setting_information	tmds_setting_info; 	
struct lvds_setting_information	lvds_setting_info; 

struct viafb_par {
	struct fb_var_screeninfo var;
	int bpp;
	int hres;
	int vres;
	int linelength;
	int vclk;		//in MHz

	int vtotal;
	int vdispend;
	int vsyncstart;
	int vsyncend;
	int vblankstart;
	int vblankend;

	int htotal;
	int hdispend;
	int hsyncstart;
	int hsyncend;
	int hblankstart;
	int hblankend;
};

struct viafb_info {
	struct fb_info_gen gen;
	unsigned int fbmem_virt;	//framebuffer virtual memory address
	unsigned int fbmem;		    //framebuffer physical memory address
	unsigned int memsize;		//size of fbmem
	unsigned int io;		    //io space address
	unsigned int io_virt;		//iospace virtual memory address
	struct viafb_par currentmode;
	unsigned char eng_oper;		//engine operation...
};

static struct fb_ops viafb_ops;

static struct viafb_info fb_info;
static struct display disp;

static struct { unsigned char red,green,blue,transp; } palette[256];

static struct fb_var_screeninfo default_var;


static char * viafb_name = "Via";

static int defaultaccel=0;

static int pseudo_pal[16];

/* video mode */
static char *mode __initdata = "640x480";
static int bpp __initdata = 32;
static int resMode = VIA_RES_640X480;
static int noaccel __initdata = 1;

int memsize=0;
int CRT_ON=1,DVI_ON=0,TV_ON=0,LCD_ON=0;

/* Display Mode Information */
int  refresh __initdata = 60;
int  tv_level __initdata = 0;
int  tv_system __initdata = NTSC;
int  tv_out_signal __initdata = TV_OUTPUT_COMPOSITE_SVIDEO;
int  tv_dedotcrawl __initdata = 0;


#ifndef VIA_MMIO
	#define VIA_MMIO 0
#endif

#if VIA_MMIO
//	#define t_outb(addr, data)  *(volatile unsigned char *)(fb_info.io_virt + (addr)) = (data)
//	#define t_inb(addr)         *(volatile unsigned char *)(fb_info.io_virt + (addr))

	#define t_outb(val,reg)	writeb(val,fb_info.io_virt + reg)
	#define t_inb(reg)	readb(fb_info.io_virt + reg)
#else
	#define t_outb(val,reg) outb(val,reg)
	#define t_inb(reg) inb(reg)
#endif


#define VIASETREG(addr, data)   outl(data,addr)
#define VIAGETREG(addr)         inl(addr)

/*
 * Accel functions called by the upper layers
 */
/*static void WaitIdle(void)
{
}*/
static void via_bmove (struct display *p, int sy, int sx,
				int dy, int dx, int height, int width)
{
}

static void via_clear_helper (int c, struct display *p,
				int sy, int sx, int height, int width)
{
}


#ifdef FBCON_HAS_CFB8
static void via_8bpp_clear (struct vc_data *conp, struct display *p,
				int sy, int sx, int height, int width)
{
	int c;
	c = attr_bgcol_ec(p,conp) & 0xFF;
	c |= c<<8;
	c |= c<<16;
	via_clear_helper(c,p,sy,sx,height,width);
}

static struct display_switch via_8bpp = {
	setup:		fbcon_cfb8_setup,
	bmove:		via_bmove,
	clear:		via_8bpp_clear,
	putc:		fbcon_cfb8_putc,
	putcs:		fbcon_cfb8_putcs,
	revc:		fbcon_cfb8_revc,
	clear_margins:	fbcon_cfb8_clear_margins,
	fontwidthmask:	FONTWIDTH (4) | FONTWIDTH (8) | FONTWIDTH (12) | FONTWIDTH (16)
};
#endif
#ifdef FBCON_HAS_CFB16
static void via_16bpp_clear (struct vc_data *conp, struct display *p,
				int sy, int sx, int height, int width)
{
	int c;
	c = ((u16*)p->dispsw_data)[attr_bgcol_ec(p,conp)];
	c = c | c<<16;
	via_clear_helper(c,p,sy,sx,height,width);
}
static struct display_switch via_16bpp = {
	setup:		fbcon_cfb16_setup,
	bmove:		via_bmove,
	clear:		via_16bpp_clear,
	putc:		fbcon_cfb16_putc,
	putcs:		fbcon_cfb16_putcs,
	revc:		fbcon_cfb16_revc,
	clear_margins:	fbcon_cfb16_clear_margins,
	fontwidthmask:	FONTWIDTH (4) | FONTWIDTH (8) | FONTWIDTH (12) | FONTWIDTH (16)
};
#endif
#ifdef FBCON_HAS_CFB32
static void via_32bpp_clear (struct vc_data *conp, struct display *p,
				int sy, int sx, int height, int width)
{
	int c;
	c = ((u32*)p->dispsw_data)[attr_bgcol_ec(p,conp)];
	via_clear_helper(c,p,sy,sx,height,width);
}
static struct display_switch via_32bpp = {
	setup:		fbcon_cfb32_setup,
	bmove:		via_bmove,
	clear:		via_32bpp_clear,
	putc:		fbcon_cfb32_putc,
	putcs:		fbcon_cfb32_putcs,
	revc:		fbcon_cfb32_revc,
	clear_margins:	fbcon_cfb32_clear_margins,
	fontwidthmask:	FONTWIDTH (4) | FONTWIDTH (8) | FONTWIDTH (12) | FONTWIDTH (16)
};
#endif



/* Fill in fix */
static int via_encode_fix(struct fb_fix_screeninfo *fix,
				  const void *par,
				  struct fb_info_gen *info)
{
	struct viafb_info * i = (struct viafb_info *)info;
	struct viafb_par * p = (struct viafb_par *)par;

        DEBUG_MSG(KERN_INFO "via_encode_fix!\n" );

	memset(fix, 0, sizeof(struct fb_fix_screeninfo));
	strcpy(fix->id,viafb_name);

	fix->smem_start = i->fbmem;
	fix->smem_len = i->memsize;

	fix->type = FB_TYPE_PACKED_PIXELS;
	fix->type_aux = 0;

	fix->visual = p->bpp==8 ? FB_VISUAL_PSEUDOCOLOR:FB_VISUAL_TRUECOLOR;

	fix->xpanstep = fix->ywrapstep = 0;
	fix->ypanstep = 1;
	fix->line_length = p->linelength;
	fix->mmio_start = 0;
	fix->mmio_len = 0;

	fix->accel = FB_ACCEL_NONE;
	
	return 0;
}

/* Fill in par from var */
static int via_decode_var(const struct fb_var_screeninfo *var,
				  void *par,
				  struct fb_info_gen *info)
{
	struct viafb_par * p = (struct viafb_par *)par;
	struct viafb_info * i = (struct viafb_info *)info;
	int vres,vfront,vback,vsync;

        DEBUG_MSG(KERN_INFO "via_decode_var!\n" );
  	p->var = *var;
	p->bpp = var->bits_per_pixel;

	if (p->bpp == 24 )
		p->bpp = 32;

	p->linelength = var->xres_virtual * p->bpp/8;

	switch (p->bpp) {
		case 8:
			p->var.red.offset = 0;
			p->var.green.offset = 0;
			p->var.blue.offset = 0;
			p->var.red.length = 6;
			p->var.green.length = 6;
			p->var.blue.length = 6;
			break;
		case 16:
			p->var.red.offset = 11;
			p->var.green.offset = 5;
			p->var.blue.offset = 0;
			p->var.red.length = 5;
			p->var.green.length = 6;
			p->var.blue.length = 5;
			break;
		case 32:
			p->var.red.offset = 16;
			p->var.green.offset = 8;
			p->var.blue.offset = 0;
			p->var.red.length = 8;
			p->var.green.length = 8;
			p->var.blue.length = 8;
			break;
		default:
			return -EINVAL;
	}

	/* convert from picoseconds to MHz */
        refresh= get_refresh(var->xres, var->yres, var->pixclock);
	p->vclk = 1000000/var->pixclock;
	

	if (p->bpp == 32)
		p->vclk *=2;

	p->hres = var->xres;
	vres = p->vres = var->yres;

	/* See if requested resolution fits in available memory */
	if (p->hres * p->vres * p->bpp/8 > i->memsize) {
		return -EINVAL;
	}

	vfront = var->upper_margin;
	vback =	var->lower_margin;
	vsync =	var->vsync_len;

	/* Compute horizontal and vertical VGA CRTC timing values */
	if (var->vmode & FB_VMODE_INTERLACED) {
		vres /= 2;
		vfront /=2;
		vback /=2;
		vsync /=2;
	}

	if (var->vmode & FB_VMODE_DOUBLE) {
		vres *= 2;
		vfront *=2;
		vback *=2;
		vsync *=2;
	}

	p->htotal = (p->hres + var->left_margin + var->right_margin + var->hsync_len)/8 - 10;
	p->hdispend = p->hres/8 - 1;
	p->hsyncstart = (p->hres + var->right_margin)/8;
	p->hsyncend = var->hsync_len/8;
	p->hblankstart = p->hdispend + 1;
	p->hblankend = p->htotal + 5;

	p->vtotal = vres + vfront + vback + vsync - 2;
	p->vdispend = vres - 1;
	p->vsyncstart = vres + vback;
	p->vsyncend = vsync;
	p->vblankstart = vres;
	p->vblankend = p->vtotal + 2;

	return 0;
}


/* Fill in var from info */
static int via_encode_var(struct fb_var_screeninfo *var,
				  const void *par,
				  struct fb_info_gen *info)
{
	struct viafb_par * p = (struct viafb_par *)par;
        
        DEBUG_MSG(KERN_INFO "via_encode_var!\n" ); 
	*var = p->var;
	var->bits_per_pixel = p->bpp;

	return 0;
}

/* Fill in par from hardware */
static void via_get_par(void *par, struct fb_info_gen *info)
{
	struct viafb_par * p = (struct viafb_par *)par;
	struct viafb_info * i = (struct viafb_info *)info;

        DEBUG_MSG(KERN_INFO "via_get_par!\n" );
	*p = i->currentmode;
}

/* Pan the display */
static int via_pan_display(const struct fb_var_screeninfo *var,
				   struct fb_info_gen *info)
{
	unsigned int offset;
	struct viafb_info * i = (struct viafb_info *)info;

        DEBUG_MSG(KERN_INFO "via_pan_display!\n" );
	offset = (var->xoffset + (var->yoffset * var->xres))
			* var->bits_per_pixel/32;
	i->currentmode.var.xoffset = var->xoffset;
	i->currentmode.var.yoffset = var->yoffset;
	
	return 0;
}

/* Set the hardware from par */
static void via_set_par(const void *par, struct fb_info_gen *info)
{
        int vmode_index;
	struct viafb_par * p = (struct viafb_par *)par;
	struct viafb_info * i = (struct viafb_info *)info;

        DEBUG_MSG(KERN_INFO "via_set_par!\n" ); 
	i->currentmode = *p;
        
        update_device_setting(i->currentmode.hres, i->currentmode.vres, i->currentmode.bpp, refresh);   
        vmode_index=get_mode_index(i->currentmode.hres, i->currentmode.vres);
	setmode(vmode_index,i->currentmode.hres,i->currentmode.vres,i->currentmode.bpp);

	via_pan_display(&p->var,info);
}

/* Get value of one color register */
static int via_getcolreg(unsigned regno, unsigned *red,
				 unsigned *green, unsigned *blue,
				 unsigned *transp, struct fb_info *info)
{
	struct viafb_info * i = (struct viafb_info *)info;
	int m = i->currentmode.bpp==8?256:16;

        DEBUG_MSG(KERN_INFO "via_getcolreg!\n" ); 
	if (regno >= m)
		return 1;

	*red = palette[regno].red;
	*green = palette[regno].green;
	*blue = palette[regno].blue;
	*transp = palette[regno].transp;

	return 0;
}

/* Set one color register */
static int via_setcolreg(unsigned regno, unsigned red, unsigned green,
				 unsigned blue, unsigned transp,
				 struct fb_info *info)
{
	struct viafb_info * i = (struct viafb_info *)info;
	int bpp = i->currentmode.bpp;
	int m = bpp==8?256:16;

        DEBUG_MSG(KERN_INFO "via_setcolreg!\n" ); 
	if (regno >= m)
		return 1;

	palette[regno].red = red;
	palette[regno].green = green;
	palette[regno].blue = blue;
	palette[regno].transp = transp;

	if (bpp==8) {
		t_outb(0xFF,0x3C6);
		t_outb(regno,0x3C8);

		t_outb(red>>10,0x3C9);
		t_outb(green>>10,0x3C9);
		t_outb(blue>>10,0x3C9);

	} else
	if (bpp == 16) 			/* RGB 565 */
			((u16*)info->pseudo_palette)[regno] = (red & 0xF800) |
			((green & 0xFC00) >> 5) | ((blue & 0xF800) >> 11);
	else
	if (bpp == 32)		/* ARGB 8888 */
		((u32*)info->pseudo_palette)[regno] =
			((transp & 0xFF00) <<16) 	|
			((red & 0xFF00) << 8) 		|
			((green & 0xFF00))		|
			((blue & 0xFF00)>>8);

	return 0;
}

/* blank screen */
static int via_blank(int blank_mode, struct fb_info_gen *info)
{
	DEBUG_MSG(KERN_INFO "via_blank!\n" );   
	/* clear DPMS setting */
		
	switch (blank_mode) {
	case VESA_NO_BLANKING:
		/* Screen: On, HSync: On, VSync: On */
		write_reg_mask(CR36, VIACR, 0x00, BIT4+BIT5);   // control CRT monitor power management
        break;
	case VESA_HSYNC_SUSPEND:
		/* Screen: Off, HSync: Off, VSync: On */
		write_reg_mask(CR36, VIACR, 0x10, BIT4+BIT5);   // control CRT monitor power management
		break;
	case VESA_VSYNC_SUSPEND:
		/* Screen: Off, HSync: On, VSync: Off */
		write_reg_mask(CR36, VIACR, 0x20, BIT4+BIT5);   // control CRT monitor power management
		break;
	case VESA_POWERDOWN:
		/* Screen: Off, HSync: Off, VSync: Off */
		write_reg_mask(CR36, VIACR, 0x30, BIT4+BIT5);   // control CRT monitor power management
		break;
    	}
    	

	return 0;
}

/* Set display switch used by console */
static void via_set_disp(const void *par, struct display *disp,
				 struct fb_info_gen *info)
{
	struct viafb_info * i = (struct viafb_info *)info;
	struct fb_info * ii = (struct fb_info *)info;
	struct viafb_par * p = (struct viafb_par *)par;
	//int isaccel = p->var.accel_flags & FB_ACCELF_TEXT;
	int isaccel = 0;

        DEBUG_MSG(KERN_INFO "via_set_disp!\n" );
	disp->screen_base = (char *)i->fbmem_virt;
	
#ifdef FBCON_HAS_CFB8
	if (p->bpp == 8 ) {
		if (isaccel)
			disp->dispsw = &via_8bpp;
		else
			disp->dispsw = &fbcon_cfb8;
	} else
#endif
#ifdef FBCON_HAS_CFB16
	if (p->bpp == 16) {
		if (isaccel){
			disp->dispsw = &via_16bpp;
	}else
			disp->dispsw = &fbcon_cfb16;
		
		disp->dispsw_data =ii->pseudo_palette;	/* console palette */
	} else
#endif
#ifdef FBCON_HAS_CFB32
	if (p->bpp == 32) {
		if (isaccel)
			disp->dispsw = &via_32bpp;
		else
			disp->dispsw = &fbcon_cfb32;
		disp->dispsw_data =ii->pseudo_palette;	/* console palette */
	} else
#endif
	disp->dispsw = &fbcon_dummy;


}

static struct fbgen_hwswitch via_hwswitch = {
	NULL, /* detect not needed */
	via_encode_fix,
	via_decode_var,
	via_encode_var,
	via_get_par,
	via_set_par,
	via_getcolreg,
	via_setcolreg,
	via_pan_display,
	via_blank,
	via_set_disp
};

static int get_mode_index(int hres, int vres)
{
    DEBUG_MSG(KERN_INFO "get_mode_index!\n" );     

    if (hres==640 && vres==480) {
	    resMode = VIA_RES_640X480;
	    mode = "640x480";
	} else if (hres==800 && vres==600) {
	    resMode = VIA_RES_800X600;
	    mode = "800x600";
	} else if (hres==1024 && vres==768) {
	    resMode = VIA_RES_1024X768;
	    mode = "1024x768";
	} else if (hres==1152 && vres==864) {
	    resMode = VIA_RES_1152X864;
	    mode = "1152x864";
	} else if (hres==1280 && vres==1024) {
	    resMode = VIA_RES_1280X1024;
	    mode = "1280x1024";
	} else if (hres==1600 && vres==1200) {
	    resMode = VIA_RES_1600X1200;
	    mode = "1600x1200";
	} else if (hres==1440 && vres==1050) {
	    resMode = VIA_RES_1440X1050;
	    mode = "1440x1050";
	} else if (hres==1280 && vres==768) {
	    resMode = VIA_RES_1280X768;
	    mode = "1280x768";
	} else if (hres==1280 && vres==960) {
	    resMode = VIA_RES_1280X960;
	    mode = "1280x960";
	} else if (hres==1920 && vres==1440) {
	    resMode = VIA_RES_1920X1440;
	    mode = "1920x1440";
	} else if (hres==848 && vres==480) {
	    resMode = VIA_RES_848X480;
	    mode = "848x480";
	} else if (hres==1400 && vres==1050) {
	    resMode = VIA_RES_1400X1050;
	    mode = "1400x1050";
	} else if (hres==720 && vres==480) {
	    resMode = VIA_RES_720X480;
	    mode = "720x480";
	} else if (hres==720 && vres==576) {
	    resMode = VIA_RES_720X576;
	    mode = "720x576";
	} else if (hres==1024 && vres==512) {
	    resMode = VIA_RES_1024X512;
	    mode = "1024x512";
	} else if (hres==856 && vres==480) {
	    resMode = VIA_RES_856X480;
	    mode = "856x480";
	} else if (hres==1024 && vres==576) {
	    resMode = VIA_RES_1024X576;
	    mode = "1024x576";
	} else {
	    resMode = VIA_RES_640X480;
	    mode = "640x480";
	}
	return(resMode);        
}

//static int via_iosize;


//static int __devinit via_pci_probe(struct pci_dev * dev, const struct pci_device_id * id)
static int __devinit via_pci_probe(void)
{
	
	//unsigned char revision;
	unsigned int default_xres,default_yres;
	char *tmpc,*tmpm;
	int vmode_index;


	DEBUG_MSG(KERN_INFO "VIAFB PCI Probe!!\n");
	init_chip_info();

	noaccel = 1;
	defaultaccel = !noaccel;
        if (chip_info.gfx_chip_name == UNICHROME_K800)
	     fb_info.fbmem = 0x0c000000;	
	else
	     fb_info.fbmem = read_reg(VIASR, SR30) << 24;

	fb_info.memsize = get_memsize();
	fb_info.fbmem_virt = (unsigned int)ioremap_nocache(fb_info.fbmem, fb_info.memsize);

	if (!fb_info.fbmem_virt) {
//		release_mem_region(fb_info.fbmem, fb_info.memsize);
		printk("ioremap failed\n");
		return -1;
	}

	fb_info.gen.parsize = sizeof (struct viafb_par);
	fb_info.gen.fbhw = &via_hwswitch;
	strcpy(fb_info.gen.info.modename, viafb_name);
	fb_info.gen.info.changevar = NULL;
	fb_info.gen.info.node = NODEV;
	fb_info.gen.info.fbops = &viafb_ops;
	fb_info.gen.info.disp = &disp;
	fb_info.gen.info.switch_con = &fbgen_switch;
	fb_info.gen.info.updatevar = &fbgen_update_var;
	fb_info.gen.info.blank = &fbgen_blank;
	fb_info.gen.info.flags = FBINFO_FLAG_DEFAULT;
	fb_info.gen.info.fontname[0] = '\0';
	fb_info.gen.info.pseudo_palette = pseudo_pal;

	tmpm = mode;
	tmpc = strsep(&tmpm,"x");
	default_xres = simple_strtoul(tmpc,NULL,0);
	default_yres = simple_strtoul(tmpm,NULL,0);
	
	vmode_index = get_mode_index(default_xres, default_yres);
	
	switch (bpp) {
	    case 0 ... 8:
	        bpp = 8;
	        break;
	    case 9 ... 16:
	        bpp = 16;
	        break;
	    case 17 ... 32:
	        bpp = 32;
	        break;
	    default :
	        bpp = 8;
	}
	
	default_var.xres=default_xres;
	default_var.yres=default_yres;
	default_var.xres_virtual=default_xres;
	default_var.yres_virtual=default_yres;
	default_var.bits_per_pixel=bpp;
	if (default_var.bits_per_pixel==15)
		default_var.bits_per_pixel=16;
	default_var.pixclock=get_pixclock(default_xres, default_yres, refresh);
	default_var.left_margin=(default_xres>>3)&0xf8;
	default_var.right_margin=32;
	default_var.upper_margin=16;
	default_var.lower_margin=4;
	default_var.hsync_len=default_var.left_margin;
	default_var.vsync_len=4;

	//if (defaultaccel && acc)
	//	default_var.accel_flags |= FB_ACCELF_TEXT;
	//else
	//	default_var.accel_flags &= ~FB_ACCELF_TEXT;
	default_var.accel_flags = 0;
	
	//default_var.accel_flags |= FB_ACCELF_TEXT;
	via_decode_var(&default_var, &fb_info.currentmode, &fb_info.gen);
	fbgen_get_var(&disp.var, -1, &fb_info.gen.info);
	default_var.activate = FB_ACTIVATE_NOW;
	fbgen_set_disp(-1, &fb_info.gen);

    if (register_framebuffer(&fb_info.gen.info) < 0) {
		printk("Could not register Via framebuffer\n");
		return -EINVAL;
	}

	DEBUG_MSG(KERN_INFO "fb%d: %s frame buffer device %dx%d-%dbpp\n",
	   GET_FB_IDX(fb_info.gen.info.node), fb_info.gen.info.modename,default_var.xres,
	   default_var.yres,default_var.bits_per_pixel);
	init_dac(IGA2);

	return 0;
}

//static void __devexit via_pci_remove(struct pci_dev * dev)
static void __devexit via_pci_remove(void)
{
    DEBUG_MSG(KERN_INFO "via_pci_remove!\n" );
	unregister_framebuffer(&fb_info.gen.info);
//	iounmap((void *)fb_info.io_virt);
	iounmap((void *)fb_info.fbmem_virt);
//	release_mem_region(fb_info.fbmem, fb_info.memsize);
//	release_region(fb_info.io, via_iosize);
}

/* List of boards that we are trying to support */
/*static struct pci_device_id via_devices[] __devinitdata = {
	{PCI_VENDOR_ID_VIA,	0x3122, PCI_ANY_ID,PCI_ANY_ID,0,0,0},
	{0,}
};*/	
	
//MODULE_DEVICE_TABLE(pci,via_devices); 

//static struct pci_driver viafb_pci_driver = {
//	name:"viafb",
//	id_table:via_devices,
//	probe:via_pci_probe,
//	remove:__devexit_p(via_pci_remove),
//};

int __init viafb_init(void)
{
    DEBUG_MSG(KERN_INFO "viafb_init!\n" ); 
	DEBUG_MSG(KERN_INFO "VIA Graphics Intergration Chipset framebuffer %s initializing\n", VERSION);
	via_pci_probe();
	//return pci_module_init(&viafb_pci_driver);

	return 0;
}

void __exit viafb_exit(void)
{
    DEBUG_MSG(KERN_INFO "viafb_exit!\n" ); 
	via_pci_remove();
	//pci_unregister_driver(&viafb_pci_driver);
}


/*
 * Parse user specified options (`video=via:')
 * example:
 * 	video=via:800x600,bpp=16,noaccel
 */
int __init viafb_setup(char *options)
{
	DEBUG_MSG(KERN_INFO "viafb_setup!\n" ); 
	return 0;
}


static struct fb_ops viafb_ops = {
	fb_get_fix:fbgen_get_fix,
	fb_get_var:fbgen_get_var,
	fb_set_var:fbgen_set_var,
	fb_get_cmap:fbgen_get_cmap,
	fb_set_cmap:fbgen_set_cmap,
	fb_pan_display:fbgen_pan_display,
};

#ifdef MODULE

int __init init_module(void)
{
  DEBUG_MSG(KERN_INFO "init_module!\n" );
  viafb_init();

  return 0;
}

#endif
module_exit(viafb_exit);

MODULE_PARM(noaccel,"i");
MODULE_PARM(memsize,"i");
MODULE_PARM(mode,"s");
MODULE_PARM_DESC(mode, "Set resolution (default=640x480)");
MODULE_PARM(bpp,"i");
MODULE_PARM_DESC(disp_mode, "Set color depth (default=32bpp)");
MODULE_PARM(refresh,"i");
MODULE_PARM_DESC(refresh, "Set CRT refresh rate (default = 60)");
MODULE_PARM(CRT_ON, "i");
MODULE_PARM_DESC(CRT, "Turn on/off flag of CRT device (default = ON)");
MODULE_PARM(DVI_ON, "i");
MODULE_PARM_DESC(DVI, "Turn on/off flag of DVI device (default = OFF)");
MODULE_PARM(LCD_ON, "i");
MODULE_PARM_DESC(LCD, "Turn on/off flag of LCD device (default = OFF)");
MODULE_PARM(TV_ON, "i");
MODULE_PARM_DESC(TV, "Turn on/off flag of TV device (default = OFF)");
MODULE_PARM(tv_system, "i");
MODULE_PARM_DESC(tv_system, "TV Signal Standard (NTSC, PAL,...,etc)");
MODULE_PARM(tv_level, "i");
MODULE_PARM_DESC(tv_level, "TV Horizontal/Vertical Contraction Control");
MODULE_PARM(tv_out_signal, "i");
MODULE_PARM_DESC(tv_out_signal, "Set TV output signal.(Default=Composite+SVideo)");
MODULE_PARM(tv_dedotcrawl, "i");
MODULE_PARM_DESC(tv_dedotcrawl, "Control TV DeDotCrawl function.(Default=disable)");
MODULE_LICENSE("GPL");

