/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "SplittableBuffer.h"
#include "DataChunk.h"
#include "StringUtils.h"
#include <cstdlib>
#include <cstring>
#include <cctype>
#include <cassert>

using namespace std;

inline void
SplittableBuffer::setupBorders()
{
	m_leftBorder.prev = &m_leftBorder;
	m_leftBorder.next = &m_rightBorder;
	m_rightBorder.prev = &m_leftBorder;
	m_rightBorder.next = &m_rightBorder;
}

SplittableBuffer::SplittableBuffer()
{
	setupBorders();
}

SplittableBuffer::SplittableBuffer(SplittableBuffer const& other)
{
	setupBorders();
	append(other);
}

SplittableBuffer::SplittableBuffer(ByteIterator const& begin, ByteIterator const& end)
{
	setupBorders();
	append(begin, end);
}

SplittableBuffer::~SplittableBuffer()
{
	clear();
}

SplittableBuffer&
SplittableBuffer::operator=(SplittableBuffer const& rhs)
{
	clear();
	append(rhs);
	return *this;
}

size_t
SplittableBuffer::size() const
{
	size_t size = 0;
	for (Range* range = m_leftBorder.next; range != &m_rightBorder; range = range->next) {
		for (BString* span = range->spanBegin; span != range->spanEnd; ++span) {
			size += span->size();
		}
	}
	return size;
}

void
SplittableBuffer::clear()
{
	Range* range = m_leftBorder.next;
	while (range != &m_rightBorder) {
		Range* next = range->next;
		delete range;
		range = next;
	}
	m_leftBorder.next = &m_rightBorder;
	m_rightBorder.prev = &m_leftBorder;
}

void
SplittableBuffer::append(SplittableBuffer const& buf)
{
	SpanIterator iter(buf.spanBegin());
	for (; !iter.isAtRightBorder(); ++iter) {
		append(iter.m_pSpan->chunk(), iter.m_pSpan->begin(), iter.m_pSpan->end());
	}
}

void
SplittableBuffer::prepend(SplittableBuffer const& buf)
{
	SpanIterator iter(buf.spanRBegin());
	for (; !iter.isAtLeftBorder(); --iter) {
		prepend(iter.m_pSpan->chunk(), iter.m_pSpan->begin(), iter.m_pSpan->end());
	}
}

void
SplittableBuffer::append(std::auto_ptr<DataChunk> chunk, size_t trim_front)
{
	IntrusivePtr<DataChunk const> chk(chunk.release());
	append(chk, chk->getDataAddr() + trim_front, chk->getDataAddr() + chk->getDataSize());
}

void
SplittableBuffer::prepend(std::auto_ptr<DataChunk> chunk, size_t trim_front)
{
	IntrusivePtr<DataChunk const> chk(chunk.release());
	prepend(chk, chk->getDataAddr() + trim_front, chk->getDataAddr() + chk->getDataSize());
}

void
SplittableBuffer::append(IntrusivePtr<DataChunk const> const& chunk, char const* begin, char const* end)
{
	if (begin == end) {
		return;
	}
	BString* last_span = lastSpan();
	if (last_span && last_span->chunk() == chunk && last_span->end() == begin) {
		last_span->trimBack(begin - end);
		return;
	}
	Range* range = m_rightBorder.prev;
	if (range->spanStorageEnd() == range->spanEnd) {
		range = Range::create(Range::APPEND);
		range->prev = m_rightBorder.prev;
		range->next = &m_rightBorder;
		m_rightBorder.prev->next = range;
		m_rightBorder.prev = range;
	}
	new(range->spanEnd++) BString(chunk, begin, end);
}

void
SplittableBuffer::prepend(IntrusivePtr<DataChunk const> const& chunk, char const* begin, char const* end)
{
	if (begin == end) {
		return;
	}
	BString* first_span = firstSpan();
	if (first_span && first_span->chunk() == chunk && first_span->begin() == end) {
		first_span->trimFront(begin - end);
		return;
	}
	Range* range = m_leftBorder.next;
	if (range->spanStorageBegin() == range->spanBegin) {
		range = Range::create(Range::PREPEND);
		range->prev = &m_leftBorder;
		range->next = m_leftBorder.next;
		m_leftBorder.next->prev = range;
		m_leftBorder.next = range;
	}
	
	new(--range->spanBegin) BString(chunk, begin, end);
}

void
SplittableBuffer::append(ByteIterator const& begin, ByteIterator const& end)
{
	SplittableBuffer::ByteIterator it = begin;
	for (;; it.nextSpan()) {
		bool last = it.isSameSpan(end);
		char const* b = &*it;
		char const* e = last ? &*end : it.spanEnd();
		append(it.m_pSpan->chunk(), b, e);
		if (last) {
			break;
		}
	}
}

void
SplittableBuffer::prepend(ByteIterator const& begin, ByteIterator const& end)
{
	SplittableBuffer::ByteIterator it = end;
	for (;; it.prevSpan()) {
		bool last = it.isSameSpan(begin);
		char const* b = last ? &*begin : it.spanBegin();
		char const* e = &*it;
		append(it.m_pSpan->chunk(), b, e);
		if (last) {
			break;
		}
	}
}

void
SplittableBuffer::appendDestructive(SplittableBuffer& buf)
{
#if 1
	// if possible, move Spans from the first range of buf to the last range of *this
	Range* src_range = buf.m_leftBorder.next;
	if (src_range->next == src_range) {
		return;
	}
	Range* dst_range = m_rightBorder.prev;
	size_t src_used = (char*)src_range->spanEnd - (char*)src_range->spanBegin;
	size_t dst_avail = (char*)dst_range->spanStorageEnd() - (char*)dst_range->spanEnd;
	if (src_used <= dst_avail) {
		memcpy(dst_range->spanEnd, src_range->spanBegin, src_used);
		// We copy spans in a bitwise manner. That's ok, as span
		// is just an IntrusivePtr and a couple of ordinary pointers.
		// IntrusivePtr is also just a pointer, so we are not copying
		// any atomic counters or anything like that.
		
		(char*&)dst_range->spanEnd += src_used;
		src_range->spanBegin = src_range->spanEnd;
		src_range->prev->next = src_range->next;
		src_range->next->prev = src_range->prev;
		delete src_range;
	}
#endif
	moveRanges(buf.m_leftBorder.next, &buf.m_rightBorder, &m_rightBorder);
}

void
SplittableBuffer::prependDestructive(SplittableBuffer& buf)
{
#if 1
	// if possible, move Spans from the last range of buf to the first range of *this
	Range* src_range = buf.m_rightBorder.prev;
	if (src_range->prev == src_range) {
		return;
	}
	Range* dst_range = m_leftBorder.next;
	size_t src_used = (char*)src_range->spanEnd - (char*)src_range->spanBegin;
	size_t dst_avail = (char*)dst_range->spanBegin - (char*)dst_range->spanStorageBegin();
	if (src_used <= dst_avail) {
		char* new_dst_begin = (char*)dst_range->spanBegin - src_used;
		memcpy(new_dst_begin, src_range->spanBegin, src_used);
		// We copy spans in a bitwise manner. That's ok, as span
		// is just an IntrusivePtr and a couple of ordinary pointers.
		// IntrusivePtr is also just a pointer, so we are not copying
		// any atomic counters or anything like that.
		
		dst_range->spanBegin = (BString*)new_dst_begin;
		src_range->spanEnd = src_range->spanBegin;
		src_range->prev->next = src_range->next;
		src_range->next->prev = src_range->prev;
		delete src_range;
	}
#endif
	moveRanges(buf.m_leftBorder.next, &buf.m_rightBorder, m_leftBorder.next);
}

void
SplittableBuffer::trimFront(size_t size)
{
	while (size != 0) {
		Range* const range = m_leftBorder.next;
		if (range == &m_rightBorder) {
			assert(size == 0);
			return;
		}
		while (true) {
			BString* const span = range->spanBegin;
			size_t span_size = span->size();
			if (size < span_size) {
				span->trimFront(size);
				return;
			} else {
				size -= span_size;
				span->~BString();
				if (++range->spanBegin == range->spanEnd) {
					range->prev->next = range->next;
					range->next->prev = range->prev;
					delete range;
					break;
				}
			}
		}
	}
}

void
SplittableBuffer::trimBack(size_t size) {
	while (size != 0) {
		Range* const range = m_rightBorder.prev;
		if (range == &m_leftBorder) {
			assert(size == 0);
			return;
		}
		while (true) {
			BString* const span = range->spanEnd - 1;
			size_t span_size = span->size();
			if (size < span_size) {
				span->trimBack(size);
				return;
			} else {
				size -= span_size;
				span->~BString();
				if (--range->spanEnd == range->spanBegin) {
					range->prev->next = range->next;
					range->next->prev = range->prev;
					delete range;
					break;
				}
			}
		}
	}
}

void
SplittableBuffer::trimFront(ByteIterator const& to)
{
	SplittableBuffer temp;
	Range* range = const_cast<Range*>(to.m_pRange);
	moveRanges(m_leftBorder.next, range, &temp.m_rightBorder);
	BString* span = range->spanBegin;
	for (; span != to.m_pSpan; ++span) {
		span->~BString();
	}
	range->spanBegin = span;
	span->data().begin = to.m_pData;
}

void
SplittableBuffer::trimBack(ByteIterator const& to)
{
	SplittableBuffer temp;
	--const_cast<ByteIterator&>(to);
	Range* range = const_cast<Range*>(to.m_pRange);	
	moveRanges(range->next, &m_rightBorder, &temp.m_rightBorder);
	BString* span = range->spanEnd - 1;
	for (; span != to.m_pSpan; --span) {
		span->~BString();
	}
	range->spanEnd = span + 1;
	span->data().end = to.m_pData + 1;
}

void
SplittableBuffer::splitFrontAppendBack(size_t size, SplittableBuffer& target)
{
	splitFrontAppendBack(begin() + size, target);
}

void
SplittableBuffer::splitFrontAppendBack(ByteIterator const& pos, SplittableBuffer& target)
{	
	moveRanges(m_leftBorder.next, const_cast<Range*>(pos.m_pRange), &target.m_rightBorder);
	
	BString const* span = pos.m_pRange->spanBegin;
	for (; span != pos.m_pSpan; ++span) {
		target.append(span->chunk(), span->begin(), span->end());
	}
	target.append(span->chunk(), span->begin(), pos.m_pData);
	
	BString* tspan = m_leftBorder.next->spanBegin;
	for (; tspan != pos.m_pSpan; ++tspan) {
		tspan->~BString();
	}
	m_leftBorder.next->spanBegin = tspan;
	tspan->data().begin = pos.m_pData;
}

void
SplittableBuffer::splitBackAppendFront(size_t size, SplittableBuffer& target)
{
	splitBackAppendFront(end() - size, target);
}

void
SplittableBuffer::splitBackAppendFront(ByteIterator const& pos, SplittableBuffer& target)
{
	moveRanges(pos.m_pRange->next, &m_rightBorder, target.m_leftBorder.next);
	
	BString const* span = pos.m_pRange->spanEnd - 1;
	for (; span != pos.m_pSpan; --span) {
		target.prepend(span->chunk(), span->begin(), span->end());
	}
	target.prepend(span->chunk(), pos.m_pData, span->end());
	
	BString* tspan = m_rightBorder.prev->spanEnd - 1;
	for (; tspan != pos.m_pSpan; --tspan) {
		tspan->~BString();
	}
	m_rightBorder.prev->spanEnd = tspan + 1;
	tspan->data().end = pos.m_pData + 1;
}

SplittableBuffer::ByteIterator
SplittableBuffer::find(ByteIterator const& begin,
	ByteIterator const& end, char ch)
{
	SplittableBuffer::ByteIterator it = begin;
	for (;; it.nextSpan()) {
		bool last = it.isSameSpan(end);
		char const* b = &*it;
		char const* e = last ? &*end : it.spanEnd();
		char const* p = (char const*)memchr(b, ch, e - b);
		if (p) {
			it += p - b;
			return it;
		}
		if (last) {
			break;
		}
	}
	return end;
}

SplittableBuffer::ByteIterator
SplittableBuffer::find(
	ByteIterator const& begin, ByteIterator const& end,
	char const* target_begin, char const* target_end)
{
	if (target_begin == target_end) {
		return begin;
	}
	ByteIterator it(begin);
	for (;; ++it) {
		it = find(it, end, *target_begin);
		if (it == end) {
			break;
		}
		if (StringUtils::startsWith(it, end, target_begin, target_end)) {
			return it;
		}
	}
	return end;
}

SplittableBuffer::ByteIterator
SplittableBuffer::ciFind(ByteIterator const& begin,
	ByteIterator const& end, char ch)
{
	char lch = tolower(ch);
	char uch = toupper(ch);
	ByteIterator pos = find(begin, end, lch);
	if (lch == uch) {
		return pos;
	}
	return find(begin, pos, uch);
}

SplittableBuffer::ByteIterator
SplittableBuffer::ciFind(
	ByteIterator const& begin, ByteIterator const& end,
	char const* target_begin, char const* target_end)
{
	if (target_begin == target_end) {
		return begin;
	}
	ByteIterator it(begin);
	for (;; ++it) {
		it = ciFind(it, end, *target_begin);
		if (it == end) {
			break;
		}
		if (StringUtils::ciStartsWith(it, end, target_begin, target_end)) {
			return it;
		}
	}
	return end;
}

std::string
SplittableBuffer::toString(ByteIterator const& begin, ByteIterator const& end)
{
	std::string str;
	str.reserve(end - begin);
	SplittableBuffer::ByteIterator it = begin;
	for (;; it.nextSpan()) {
		bool last = it.isSameSpan(end);
		char const* b = &*it;
		char const* e = last ? &*end : it.spanEnd();
		str.append((char const*)b, e - b);
		if (last) {
			break;
		}
	}
	return str;
}

BString
SplittableBuffer::toBString(ByteIterator const& begin, ByteIterator const& end)
{
	if (begin == end) {
		return BString();
	}
	
	ByteIterator end_m1(end);
	--end_m1;
	if (begin.m_pSpan == end_m1.m_pSpan) {
		return BString(begin.m_pSpan->chunk(), begin.m_pData, end_m1.m_pData + 1);
	}
	
	auto_ptr<DataChunk> chunk(DataChunk::create(end - begin));
	char* p = chunk->getDataAddr();
	SplittableBuffer::ByteIterator it = begin;
	for (;; it.nextSpan()) {
		bool last = it.isSameSpan(end);
		char const* b = &*it;
		char const* e = last ? &*end : it.spanEnd();
		size_t len = e - b;
		memcpy(p, b, len);
		if (last) {
			break;
		}
		p += len;
	}
	return BString(chunk);
}

void
SplittableBuffer::moveRanges(Range* begin, Range* end, Range* before_this)
{
	if (begin != end) {
		Range* pre_begin = begin->prev;
		Range* pre_end = end->prev;
		Range* after_this = before_this->prev;
		pre_begin->next = end;
		end->prev = pre_begin;
		begin->prev = after_this;
		pre_end->next = before_this;
		after_this->next = begin;
		before_this->prev = pre_end;
	}
}


/* =================== SplittableBuffer::ByteIterator ================ */

void
SplittableBuffer::ByteIterator::nextSpan()
{
	if (++m_pSpan == m_pRange->spanEnd) {
		m_pRange = m_pRange->next;
		m_pSpan = m_pRange->spanBegin;
	}
	m_pData = m_pSpan->begin();
}

void
SplittableBuffer::ByteIterator::prevSpan()
{
	if (m_pSpan-- == m_pRange->spanBegin) {
		m_pRange = m_pRange->prev;
		m_pSpan = m_pRange->spanEnd - 1;
	}
	m_pData = m_pSpan->end() - 1;
}

SplittableBuffer::ByteIterator&
SplittableBuffer::ByteIterator::operator++()
{
	if (++m_pData == m_pSpan->end()) {
		if (++m_pSpan == m_pRange->spanEnd) {
			m_pRange = m_pRange->next;
			m_pSpan = m_pRange->spanBegin;
		}
		m_pData = m_pSpan->begin();
	}
	return *this;
}

SplittableBuffer::ByteIterator&
SplittableBuffer::ByteIterator::operator--()
{
	if (m_pData-- == m_pSpan->begin()) {
		if (m_pSpan-- == m_pRange->spanBegin) {
			m_pRange = m_pRange->prev;
			m_pSpan = m_pRange->spanEnd - 1;
		}
		m_pData = m_pSpan->end() - 1;
	}
	return *this;
}

SplittableBuffer::ByteIterator&
SplittableBuffer::ByteIterator::operator+=(difference_type amount)
{
	while (amount != 0) {
		size_t distance = m_pSpan->end() - m_pData;
		if (amount < distance) {
			m_pData += amount;
			break;
		} else {
			amount -= distance;
			if (++m_pSpan == m_pRange->spanEnd) {
				m_pRange = m_pRange->next;
				m_pSpan = m_pRange->spanBegin;
			}
			m_pData = m_pSpan->begin();
		}
	}
	return *this;
}

SplittableBuffer::ByteIterator&
SplittableBuffer::ByteIterator::operator-=(difference_type amount)
{
	while (amount != 0) {
		size_t distance = m_pData + 1 - m_pSpan->begin();
		if (amount < distance) {
			m_pData -= amount;
			break;
		} else {
			amount -= distance;
			if (m_pSpan-- == m_pRange->spanBegin) {
				m_pRange = m_pRange->prev;
				m_pSpan = m_pRange->spanEnd - 1;
			}
			m_pData = m_pSpan->end() - 1;
		}
	}
	return *this;
}

SplittableBuffer::ByteIterator::difference_type
SplittableBuffer::ByteIterator::operator-(ByteIterator const& rhs) const
{
	difference_type distance = 0;
	Range const* range = rhs.m_pRange;
	BString const* span = rhs.m_pSpan;
	while (true) {
		distance += span->size();
		if (span == m_pSpan) {
			// testing equality of ranges isn't necessary,
			// as spans are stored inside Range objects
			break;
		}
		if (++span == range->spanEnd) {
			range = range->next;
			span = range->spanBegin;
		}
	}
	distance -= rhs.m_pData - rhs.m_pSpan->begin();
	distance -= m_pSpan->end() - m_pData;
	return distance;	
}

bool
SplittableBuffer::ByteIterator::less(ByteIterator const& than, bool or_equal) const
{
	for (Range const* range = m_pRange; range != than.m_pRange; range = range->next) {
		if (range->next == range) {
			return false;
		}
	}
	if (m_pRange != than.m_pRange) {
		return true;
	}
	if (m_pSpan == than.m_pSpan) {
		return or_equal ? m_pData <= than.m_pData : m_pData < than.m_pData;
	}
	return m_pSpan < than.m_pSpan;
}


/* =================== SplittableBuffer::SpanIterator ================ */

SplittableBuffer::SpanIterator&
SplittableBuffer::SpanIterator::operator++()
{
	if (++m_pSpan == m_pRange->spanEnd) {
		m_pRange = m_pRange->next;
		m_pSpan = m_pRange->spanBegin;
	}
	return *this;
}

SplittableBuffer::SpanIterator&
SplittableBuffer::SpanIterator::operator--()
{
	if (m_pSpan-- == m_pRange->spanBegin) {
		m_pRange = m_pRange->prev;
		m_pSpan = m_pRange->spanEnd - 1;
	}
	return *this;
}


/* ===================== SplittableBuffer::Range ================== */

SplittableBuffer::Range::Range(GrowDirection direction, size_t capacity)
: spanStorageEnd_(spanStorageBegin() + capacity)
{
	if (direction == APPEND) {
		spanBegin = spanEnd = spanStorageBegin();
	} else {
		spanBegin = spanEnd = spanStorageEnd();
	}
}

SplittableBuffer::Range::~Range()
{
	for (BString* span = spanBegin; span != spanEnd; ++span) {
		span->~BString();
	}
}

void*
SplittableBuffer::Range::operator new(size_t size, size_t extra_space)
{
	return malloc(size + extra_space);
}

void
SplittableBuffer::Range::operator delete(void* addr)
{
	return free(addr);
}

