/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "types.h"
#include "StringUtils.h"
#include "BString.h"
#include "DataChunk.h"
#include <memory>

using namespace std;

bool
StringUtils::equalCharPtr(const void* begin1, const void* end1, const void* begin2, const void* end2)
{
	size_t len1 = reinterpret_cast<const char*>(end1) - reinterpret_cast<const char*>(begin1);
	size_t len2 = reinterpret_cast<const char*>(end2) - reinterpret_cast<const char*>(begin2);
	if (len1 != len2) {
		return false;
	}
	return std::memcmp(begin1, begin2, len1) == 0;
}

bool
StringUtils::ciEqual(const std::string& str1, const std::string& str2)
{
	return ciEqual(str1.data(), str1.data()+str1.length(), str2.data(), str2.data()+str2.length());
}

bool
StringUtils::ciEqual(const char* str1, const char* str2)
{
	unsigned char* s1 = (unsigned char*)str1;
	unsigned char* s2 = (unsigned char*)str2;
	for (;; ++s1, ++s2) {
		if (tolower(*s1) != tolower(*s2)) {
			return false;    
		} else if (!*s1) {
			return true;
		}
	}
}

bool
StringUtils::ciLess(const std::string& str1, const std::string& str2)
{
	return ciLess(str1.data(), str1.data()+str1.length(), str2.data(), str2.data()+str2.length());
}

bool
StringUtils::ciLess(const char* str1, const char* str2)
{
	unsigned char* s1 = (unsigned char*)str1;
	unsigned char* s2 = (unsigned char*)str2;
	for (;; ++s1, ++s2) {
		int diff = tolower(*s1) - tolower(*s2);
		if (diff != 0) {
			return diff < 0; 
		} else if (!*s1) {
			return false;
		}
	}
}

bool
StringUtils::ciLessEqual(const std::string& str1, const std::string& str2)
{
	return ciLessEqual(str1.data(), str1.data()+str1.length(), str2.data(), str2.data()+str2.length());
}

bool
StringUtils::ciLessEqual(const char* str1, const char* str2)
{
	unsigned char* s1 = (unsigned char*)str1;
	unsigned char* s2 = (unsigned char*)str2;
	for (;; ++s1, ++s2) {
		int diff = tolower(*s1) - tolower(*s2);
		if (diff != 0) {
			return diff < 0; 
		} else if (!*s1) {
			return true;
		}
	}
}

BString
StringUtils::toLowerBString(const BString& str)
{
	auto_ptr<DataChunk> chunk(DataChunk::create(str.size()));
	char* dst = chunk->getDataAddr();
	char const* src = str.begin();
	char const* const src_end = str.end();
	for (; src != src_end; ++src, ++dst) {
		*dst = tolower(uint8_t(*src));
	}
	return BString(chunk);
}

BString
StringUtils::toUpperBString(const BString& str)
{
	auto_ptr<DataChunk> chunk(DataChunk::create(str.size()));
	char* dst = chunk->getDataAddr();
	char const* src = str.begin();
	char const* const src_end = str.end();
	for (; src != src_end; ++src, ++dst) {
		*dst = toupper(uint8_t(*src));
	}
	return BString(chunk);
}

bool
StringUtils::startsWith(const std::string& str1, const std::string& str2)
{
	return startsWith(str1.data(), str1.data()+str1.length(), str2.data(), str2.data()+str2.length());
}

bool
StringUtils::endsWith(const std::string& str1, const std::string& str2)
{
	return endsWith(str1.data(), str1.data()+str1.length(), str2.data(), str2.data()+str2.length());
}

bool
StringUtils::ciStartsWith(const std::string& str1, const std::string& str2)
{
	return ciStartsWith(str1.data(), str1.data()+str1.length(), str2.data(), str2.data()+str2.length());
}

bool
StringUtils::ciEndsWith(const std::string& str1, const std::string& str2)
{
	return ciEndsWith(str1.data(), str1.data()+str1.length(), str2.data(), str2.data()+str2.length());
}

std::string
StringUtils::replace(const std::string& search, const std::string& replace, const std::string& subject)
{
	std::string res;
	size_t pos = 0, pos1, len = subject.length();
	res.reserve(len);
	while (pos < len) {
		pos1 = subject.find(search, pos);
		if (pos1 == std::string::npos) {
			break;
		} else {
			res.append(subject, pos, pos1 - pos);
			res.append(replace);
			pos = pos1 + search.length();
		}
	}
	res.append(subject, pos, std::string::npos);
	return res;
}

void*
StringUtils::findCharPtr(const void* begin, const void* end, char ch)
{
	void* p = memchr(begin, ch, static_cast<const char*>(end) - static_cast<const char*>(begin));
	return (p ? p : const_cast<void*>(end));
}

bool
StringUtils::startsWithCharPtr(const void* begin1, const void* end1, const void* begin2, const void* end2)
{
	size_t len1 = reinterpret_cast<const char*>(end1) - reinterpret_cast<const char*>(begin1);
	size_t len2 = reinterpret_cast<const char*>(end2) - reinterpret_cast<const char*>(begin2);
	if (len1 < len2) {
		return false;
	}
	return std::memcmp(begin1, begin2, len2) == 0;
}

bool
StringUtils::endsWithCharPtr(const void* begin1, const void* end1, const void* begin2, const void* end2)
{
	size_t len1 = reinterpret_cast<const char*>(end1) - reinterpret_cast<const char*>(begin1);
	size_t len2 = reinterpret_cast<const char*>(end2) - reinterpret_cast<const char*>(begin2);
	if (len1 < len2) {
		return false;
	}
	return std::memcmp(reinterpret_cast<const char*>(end1) - len2, begin2, len2) == 0;
}

std::string
StringUtils::trimStrCharPtr(const void* begin, const void* end)
{
	const char* b = reinterpret_cast<const char*>(begin);
	const char* e = reinterpret_cast<const char*>(end);
	b = ltrim(b, e);
	e = rtrim(b, e);
	return std::string(b, e);
}

