/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "AnalyzeResponseFilter.h"
#include "HttpResponseMetadata.h"
#include "AbstractContentIdentifier.h"
#include "GifContentIdentifier.h"
#include "JpegContentIdentifier.h"
#include "PngContentIdentifier.h"
#include "FlashContentIdentifier.h"
#include "HtmlResponseFilter.h"
#include "FilterTryList.h"
#include "ServiceContext.h"
#include "HttpRequestMetadata.h"
#include "HttpHeader.h"
#include "HttpHeadersCollection.h"
#include "AdSuspect.h"
#include "BString.h"
#include "URI.h"
#include <string>

using namespace std;

AnalyzeResponseFilter::AnalyzeResponseFilter(
	ResponseFilterChain& chain, bool ignore_size)
:	ResponseFilterBase(chain),
	m_state(IDENTIFYING_CONTENT),
	m_ignoreSize(ignore_size)
{
	m_candidates.push_back(IdentifierPtr(new GifContentIdentifier));
	m_candidates.push_back(IdentifierPtr(new JpegContentIdentifier));
	m_candidates.push_back(IdentifierPtr(new PngContentIdentifier));
	m_candidates.push_back(IdentifierPtr(new FlashContentIdentifier));
}

AnalyzeResponseFilter::~AnalyzeResponseFilter()
{
}

void
AnalyzeResponseFilter::processMetadata(auto_ptr<HttpResponseMetadata> metadata)
{
	m_ptrMetadata = metadata;
	if (m_ptrMetadata->statusLine().isRedirect()) {
		BString const location_str("Location");
		BString location_hdr = m_ptrMetadata->headers().getHeader(location_str).getValue();
		if (!location_hdr.empty()) {
			// We need to mark the Location header with either
			// bf-analyze or bf-analyze-is, so that this filter
			// would be applied again when fetching that request.
			URI location_url(getRequest().requestLine().getURI(), URI(location_hdr));
			BString new_location = AdSuspect::getAnalyzeURL(location_url, m_ignoreSize);
			m_ptrMetadata->headers().setHeader(HttpHeader(location_str, new_location));
		}
	}
}

void
AnalyzeResponseFilter::processBodyData(SplittableBuffer& data, bool eof)
{
	switch (m_state) {
		case DELEGATING: {
			m_ptrDelegate->processBodyData(data, eof);
			break;
		}
		case FORWARDING: {
			outputBodyData(data, eof);
			break;
		}
		case IDENTIFYING_CONTENT: {
			m_data.appendDestructive(data);
			list<IdentifierPtr>::iterator it = m_candidates.begin();
			list<IdentifierPtr>::iterator const end = m_candidates.end();
			while (it != end) {
				typedef AbstractContentIdentifier CI;
				CI::Status st = (*it)->identify(m_data);
				if (st == CI::NO_MATCH) {
					m_candidates.erase(it++);
					continue;
				} else if (st == CI::MATCH) {
					m_ptrDelegate = (*it)->getFilter(
						getFilterChain(), m_ignoreSize
					);
					break;
				}
				++it;
			}
			if (m_ptrDelegate.get()) {
				startDelegating(eof);
			} else if (m_candidates.empty() || eof) {
				FilterTryList::ContentType ctype = FilterTryList::CTYPE_HTML;
				if (FilterTryList::isHtmlFilterApplicable(getFilterChain(), *m_ptrMetadata, &ctype)) {
					m_ptrDelegate.reset(new HtmlResponseFilter(
						getFilterChain(),
						getFilterChain().getContext().htmlProcessor(),
						getFilterChain().getRequestPtr(),
						ctype == FilterTryList::CTYPE_XHTML
					));
					startDelegating(eof);
				} else {
					startForwarding(eof);
				}
			}
			break;
		}
	}
}

void
AnalyzeResponseFilter::startDelegating(bool eof)
{
	m_state = DELEGATING;
	m_ptrDelegate->processMetadata(m_ptrMetadata);
	m_ptrDelegate->processBodyData(m_data, eof);
}

void
AnalyzeResponseFilter::startForwarding(bool eof)
{
	m_state = FORWARDING;
	outputMetadata(m_ptrMetadata);
	outputBodyData(m_data, eof);
}
