/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "FilteringServer.h"
#include "URI.h"
#include "HttpRequestMetadata.h"
#include "HttpRequestLine.h"
#include "AbstractRequestHandler.h"
#include "AbstractResponseHandler.h"
#include "FilterTryList.h"
#include "CraftedResponse.h"
#include "CraftedResponseGenerator.h"
#include "ErrorResponse.h"
#include "ReplacementImage.h"
#include "ReplacementFlash.h"
#include "ReplacementHtml.h"
#include "ReplacementJs.h"
#include "RegexFilterDescriptor.h"
#include "SubstitutionRequestParser.h"
#include "GlobalState.h"
#include "UrlPatterns.h"
#include "ContentFilters.h"
#include "BString.h"
#include <string>

using namespace std;

class FilteringServer::ImageSubstitutionParser : public SubstitutionRequestParser
{
public:
	ImageSubstitutionParser(bool is_head_response)
	: SubstitutionRequestParser(is_head_response) {}
	
	CraftedResponseGenerator& responseGenerator() { return m_responseGenerator; }
private:
	virtual void handleResult(
		unsigned int width, unsigned int height,
		URI const& url, BString const& orig_path);
	
	CraftedResponseGenerator m_responseGenerator;
};


class FilteringServer::FlashSubstitutionParser : public SubstitutionRequestParser
{
public:
	FlashSubstitutionParser(bool is_head_response)
	: SubstitutionRequestParser(is_head_response) {}
	
	CraftedResponseGenerator& responseGenerator() { return m_responseGenerator; }
private:
	virtual void handleResult(
		unsigned int width, unsigned int height,
		URI const& url, BString const& orig_path);
	
	CraftedResponseGenerator m_responseGenerator;
};


FilteringServer::FilteringServer(ServiceContext& context, ServerTimeouts const& timeouts)
:	Server(context, timeouts)
{
}

FilteringServer::~FilteringServer()
{
}

IntrusivePtr<AbstractRequestHandler>
FilteringServer::submitRequest(
	ConstRequestPtr const& request_metadata,
	IntrusivePtr<AbstractResponseHandler> const& handler)
{
	IntrusivePtr<AbstractRequestHandler> res;
	if ((res = processSubstitutionRequest(request_metadata, handler))) {
		// do nothing
	} else if ((res = processUrlPatterns(request_metadata, handler))) {
		// do nothing
	} else if ((res = processAnalyzeRequest(request_metadata, handler))) {
		// do nothing
	} else {
		std::auto_ptr<FilterTryList> filters(new FilterTryList);
		{
			GlobalState::ReadAccessor global_state; // hold the lock
			if (global_state->isFilteringEnabled()) {
				applyStandardFilters(
					*filters,
					request_metadata->requestLine().getURI()
				);
			}
			if (global_state->config().isClientCompressionEnabled()) {
				filters->append(FilterTryList::FilterConstructorPtr(
					new FilterTryList::FilterConstructor(
						sigc::ptr_fun(&FilterTryList::tryCompressorFilter)
					)
				));
			}
		}
		res = Server::submitRequest(request_metadata, handler, filters);
	}
	return res;
}

IntrusivePtr<AbstractRequestHandler>
FilteringServer::processSubstitutionRequest(
	ConstRequestPtr const& request_metadata,
	IntrusivePtr<AbstractResponseHandler> const& handler)
{
	BString const si_prefix("bf-si-");
	BString const sf_prefix("bf-sf-");
	URI const& uri = request_metadata->requestLine().getURI();
	bool const is_head_request = (request_metadata->requestLine().getMethod() == BString("HEAD"));
	
	ImageSubstitutionParser image_parser(is_head_request);
	if (image_parser.parse(uri, si_prefix)) {
		return Server::submitRequest(
			request_metadata, handler,
			image_parser.responseGenerator()
		);
	}
	
	FlashSubstitutionParser flash_parser(is_head_request);
	if (flash_parser.parse(uri, sf_prefix)) {
		return Server::submitRequest(
			request_metadata, handler,
			flash_parser.responseGenerator()
		);
	}
	
	return IntrusivePtr<AbstractRequestHandler>();
}

IntrusivePtr<AbstractRequestHandler>
FilteringServer::processUrlPatterns(
	ConstRequestPtr const& request_metadata,
	IntrusivePtr<AbstractResponseHandler> const& handler)
{
	URI const& url = request_metadata->requestLine().getURI();
	GlobalState::ReadAccessor global_state;
	if (global_state->urlPatterns().isForbidden(url)) {
		return Server::submitRequest(
			request_metadata, handler,
			sigc::bind(
				sigc::ptr_fun(&ErrorResponse::errUrlForbidden),
				url.toString()
			)
		);
	}
	
	bool const is_head_request = request_metadata->requestLine().getMethod() == BString("HEAD");
	UrlPatterns::Substitution subst = global_state->urlPatterns().getSubstitutionFor(url);
	switch (subst) {
		case UrlPatterns::EMPTY_IMAGE: {
			return Server::submitRequest(
				request_metadata, handler,
				sigc::bind(
					sigc::ptr_fun(&ReplacementImage::createHttpResponse),
					is_head_request,
					1, 1 // width, height
				)
			);
		}
		case UrlPatterns::EMPTY_FLASH: {
			return Server::submitRequest(
				request_metadata, handler,
				sigc::bind(
					sigc::ptr_fun(&ReplacementFlash::createHttpResponse),
					is_head_request,
					1, 1, // width, height
					url.toString(), ReplacementFlash::DEFAULT_FRAMERATE 
				)
			);
		}
		case UrlPatterns::EMPTY_HTML: {
			return Server::submitRequest(
				request_metadata, handler,
				sigc::bind(
					sigc::ptr_fun(&ReplacementHtml::createHttpResponse),
					is_head_request
				)
			);
		}
		case UrlPatterns::EMPTY_JS: {
			return Server::submitRequest(
				request_metadata, handler,
				sigc::bind(
					sigc::ptr_fun(&ReplacementJs::createHttpResponse),
					is_head_request
				)
			);
		}
		default: break;
	}
	
	return IntrusivePtr<AbstractRequestHandler>();
}

IntrusivePtr<AbstractRequestHandler>
FilteringServer::processAnalyzeRequest(
	ConstRequestPtr const& request_metadata,
	IntrusivePtr<AbstractResponseHandler> const& handler)
{
	IntrusivePtr<AbstractRequestHandler> res;
	URI const& uri = request_metadata->requestLine().getURI();
	BString const prefix("bf-analyze");
	BString const is_str("-is");
	BString path = uri.getRawPath();
	bool ignore_size = false;
	if (StringUtils::startsWith(path.begin(), path.end(),
	                            prefix.begin(), prefix.end())) {
		path.trimFront(prefix.size());
		if (StringUtils::startsWith(path.begin(), path.end(),
		                            is_str.begin(), is_str.end())) {
			ignore_size = true;
			path.trimFront(is_str.size());
		}
		if (!path.empty()) {
			if (path[0] == '/') {
				path.trimFront(1);
			} else {
				return res;
			}
		}
		URI new_uri(uri);
		new_uri.setAbsoluteRawPath(path);
		RequestPtr req(new HttpRequestMetadata(*request_metadata));
		req->requestLine().setURI(new_uri);
		std::auto_ptr<FilterTryList> filter_try_list(new FilterTryList);
		if (GlobalState::ReadAccessor()->isFilteringEnabled()) {
			filter_try_list->append(FilterTryList::FilterConstructorPtr(
				new FilterTryList::FilterConstructor(
					sigc::bind(
						sigc::ptr_fun(&FilterTryList::tryAnalyzeFilter),
						ignore_size
					)
				)
			));
		}
		res = Server::submitRequest(req, handler, filter_try_list);
	}
	
	return res;
}

void
FilteringServer::applyStandardFilters(FilterTryList& filters, URI const& request_url)
{
	BString url_str(request_url.toBString());
	
	GlobalState::ReadAccessor state; // hold the read lock
	
	typedef ContentFilters::FilterList FilterList;
	FilterList const& avail_filters = state->contentFilters().filters();
	FilterList::const_iterator it(avail_filters.begin());
	FilterList::const_iterator const end(avail_filters.end());
	for (; it != end && (*it)->order() < 0; ++it) {
		maybeApplyRegexFilter(filters, url_str, *it);
	}
	filters.append(FilterTryList::FilterConstructorPtr(
		new FilterTryList::FilterConstructor(
			sigc::ptr_fun(&FilterTryList::tryHtmlFilter)
		)
	));
	for (; it != end; ++it) {
		maybeApplyRegexFilter(filters, url_str, *it);
	}
}

void
FilteringServer::maybeApplyRegexFilter(
	FilterTryList& filters, BString const& request_url,
	IntrusivePtr<RegexFilterDescriptor const> const& flt)
{
	if (flt->isValid() && flt->urlMatches(request_url)) {
		filters.append(FilterTryList::FilterConstructorPtr(
			new FilterTryList::FilterConstructor(
				sigc::bind(
					sigc::ptr_fun(&FilterTryList::tryRegexFilter),
					flt
				)
			)
		));
	}
}


void
FilteringServer::ImageSubstitutionParser::handleResult(
	unsigned int width, unsigned int height,
	URI const& url, BString const& orig_path)
{
	m_responseGenerator = sigc::bind(
		sigc::ptr_fun(&ReplacementImage::createHttpResponse),
		isHeadResponse(),
		width, height
	);
}

void
FilteringServer::FlashSubstitutionParser::handleResult(
	unsigned int width, unsigned int height,
	URI const& url, BString const& orig_path)
{
	URI orig_url(url);
	orig_url.setAbsoluteRawPath(orig_path);
	m_responseGenerator = sigc::bind(
		sigc::ptr_fun(&ReplacementFlash::createHttpResponse),
		isHeadResponse(), width, height,
		orig_url.toString(), ReplacementFlash::DEFAULT_FRAMERATE
	);
}
